/**
 * 
 */
package edu.jhu.ece.iacl.algorithms.icp;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Vector;

import javax.vecmath.Matrix3d;
import javax.vecmath.Point3d;
import javax.vecmath.Point3f;

import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.SurfaceIntersector;
import edu.jhu.ece.iacl.algorithms.graphics.utilities.PrincipalComponentAnalysisFloat;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;

import Jama.*;

/**
 * Iterative closest point registration using a trimmed least-squares cost
 * function. The algorithm is initialized with PCA. This method works well as
 * long as the angular disagreement is less than 90 degrees.
 * 
 * @author Blake Lucas (bclucas@jhu.edu)
 */
public class IterativeClosestPointRegistration extends AbstractCalculation {
	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.6 $");
	}

	public enum DegreesOfFreedom {
		TRANSLATION, RIGID, RIGID_AND_GLOBAL_SCALE, AFFINE
	};

	private SurfaceIntersector intersector;
	private Matrix TransMat;

	public IterativeClosestPointRegistration() {
		super();
		setLabel("Iterative Closest Point Registration");
	}

	/**
	 * Corresponding points for sorting
	 * 
	 * @author Blake Lucas
	 * 
	 */
	protected class CorrespondencePoints implements
			Comparable<CorrespondencePoints> {
		Point3f source;
		Point3f target;
		double distance;

		public CorrespondencePoints(Point3f source, Point3f target, double dist) {
			this.source = source;
			this.target = target;
			this.distance = dist;
		}

		public int compareTo(CorrespondencePoints cp) {
			return (int) Math.signum(distance - cp.distance);
		}
	}

	/**
	 * Get transformation matrix
	 * 
	 * @return transformation matrix
	 */
	public Matrix getTransformationMatrix() {
		return TransMat;
	}

	/**
	 * Solve iterative closest point registration
	 * 
	 * @param source
	 *            source surface
	 * @param target
	 *            target surface
	 * @param dof
	 *            degrees of freedom
	 * @param levels
	 *            resolution levels
	 * @param retainmentAmount
	 *            amount of correspondences to use in surface distance
	 *            comparison [0,1]
	 * @param errorThresh
	 *            error threshold for termination
	 * @param maxIters
	 *            maximum iterations
	 * @param initWithPCA
	 *            initialize with PCA
	 * @return
	 */
	public EmbeddedSurface solve(EmbeddedSurface source,
			EmbeddedSurface target, DegreesOfFreedom dof, int levels,
			double retainmentAmount, double errorThresh, int maxIters,
			boolean initWithPCA) {
		EmbeddedSurface sourceClone = source.clone();
		intersector = new SurfaceIntersector(this, 16, target);
		int vertCount = source.getVertexCount();
		Point3f[] sourcePts = sourceClone.getVertexCopy();
		Point3f[] targetPts = target.getVertexCopy();
		ArrayList<Integer> randomIndexes = new ArrayList(vertCount);
		for (int i = 0; i < vertCount; i++)
			randomIndexes.add(i);
		Collections.shuffle(randomIndexes);
		int stSampleSize = (int) Math.exp(Math.log(vertCount) - (levels - 1)
				* Math.log(2));
		int sampleSize = 0;
		int N = 0;
		int retainCount;
		Matrix R = Matrix.identity(3, 3);
		Matrix TransLevelMat;
		Point3d transPt = new Point3d();
		double error = 0;
		System.out.println("Vertex Count " + vertCount);
		setTotalUnits(levels);
		if (initWithPCA) {
			PrincipalComponentAnalysisFloat sourcePCA = new PrincipalComponentAnalysisFloat(
					sourcePts);
			PrincipalComponentAnalysisFloat targetPCA = new PrincipalComponentAnalysisFloat(
					targetPts);
			Matrix sourceT;
			Matrix targetT;
			switch (dof) {
			case TRANSLATION:
				sourceT = sourcePCA.getTranslationTransformationMatrix();
				targetT = targetPCA.getTranslationTransformationMatrix();
				break;
			case RIGID:
				sourceT = sourcePCA.getRigidTransformationMatrix();
				targetT = targetPCA.getRigidTransformationMatrix(sourcePCA
						.getLastV());
				break;
			case RIGID_AND_GLOBAL_SCALE:
				sourceT = sourcePCA.getRigidGlobalScaleTransformationMatrix();
				targetT = targetPCA
						.getRigidGlobalScaleTransformationMatrix(sourcePCA
								.getLastV());
				break;
			case AFFINE:
				sourceT = sourcePCA.getAffineTransformationMatrix();
				targetT = targetPCA.getAffineTransformationMatrix(sourcePCA
						.getLastV());
				// targetT = targetPCA.getAffineTransformationMatrix();
				break;
			default:
				sourceT = sourcePCA.getAffineTransformationMatrix();
				targetT = targetPCA.getAffineTransformationMatrix();
			}
			TransMat = targetT.times(sourceT.inverse());
			System.out.println("PCA Alignment Matrix: ");
			printMatrix(TransMat);
			applyTransform(sourcePts, TransMat);
		} else {
			TransMat = Matrix.identity(4, 4);
		}
		double lastError = 1E30;
		double relativeError = 0;
		for (int l = 1; l <= levels; l++) {
			sampleSize = Math.min(vertCount, Math.max(4, stSampleSize));
			retainCount = (int) (sampleSize * retainmentAmount);
			N = Math.min(retainCount, sampleSize);
			System.out.println("Level " + l + ") Sample Size " + sampleSize
					+ " Trimmed Sample Size " + N);

			TransLevelMat = Matrix.identity(4, 4);

			for (int i = 0; i < maxIters; i++) {

				// ///////////////////////////////////////////////////////
				System.out.println("iter: " + i);
				// ///////////////////////////////////////////////////////

				Vector<CorrespondencePoints> cps = findClosestPoints(sourcePts,
						randomIndexes, sampleSize);		
				
				switch (dof) {
				case TRANSLATION:
					error = translationEstimate(cps, R, transPt, N);
					break;
				case RIGID:

					
					// error = rigidEstimate(cps, R, transPt, N, false);
					error = rigidEstimate(cps, R, transPt, N, false, sourcePts);
					
					break;
				case RIGID_AND_GLOBAL_SCALE:
					// error = rigidEstimate(cps, R, transPt, N, true, );
					error = rigidEstimate(cps, R, transPt, N, true, sourcePts);
					break;
				case AFFINE:
					error = affineEstimate(cps, R, transPt, N);
					break;
				}

				TransLevelMat = createTransMatrix(R, transPt).times(
						TransLevelMat);
				
				applyTransform(sourcePts, createTransMatrix(R, transPt));

				relativeError = (lastError > 0) ? Math.abs(error - lastError)
						/ lastError : 0;
				System.out.println((i + 1) + ") MSE Error: " + error
						+ " Relative Error " + relativeError);
				if (relativeError < errorThresh)
					break;
				lastError = error;
			}
			
			// /////////////////////////////////////////////////////
			// markCompleted("Level " + l + " MSE Error " + error
			// + " Relative Error " + relativeError);
			TransMat = TransLevelMat.times(TransMat);
			// /////////////////////////////////////////////////////
			// sourcePts = sourceClone.getVertexCopy();
			// applyTransform(sourcePts, TransMat);
			System.out.println("Level " + l + ") Transformation Matrix: ");
			printMatrix(TransMat);
			incrementCompletedUnits();
			stSampleSize *= 2;
		}
		sourceClone.transform(TransMat);
		markCompleted();
		return sourceClone;
	}

	/**
	 * print matrix
	 * 
	 * @param M
	 *            matrix
	 */
	protected void printMatrix(Matrix M) {
		int rows = M.getRowDimension();
		int cols = M.getColumnDimension();
		for (int m = 0; m < rows; m++) {
			for (int n = 0; n < cols; n++) {
				System.out.printf("%8.4f ", M.get(m, n));
			}
			System.out.println("");
		}
	}

	/**
	 * create transformation matrix
	 * 
	 * @param R
	 *            rotation matrix
	 * @param pt
	 *            offset point
	 * @return
	 */
	protected Matrix createTransMatrix(Matrix R, Point3d pt) {
		Matrix Tmat = new Matrix(4, 4);
		for (int m = 0; m < 3; m++) {
			for (int n = 0; n < 3; n++) {
				Tmat.set(m, n, R.get(m, n));
			}
		}
		Tmat.set(0, 3, pt.x);
		Tmat.set(1, 3, pt.y);
		Tmat.set(2, 3, pt.z);
		Tmat.set(3, 3, 1);
		Tmat.set(3, 0, 0);
		Tmat.set(3, 1, 0);
		Tmat.set(3, 2, 0);
		return Tmat;
	}

	/**
	 * apply transformation
	 * 
	 * @param source
	 *            source surface
	 * @param R
	 *            rotation matrix
	 * @param pt
	 *            translation
	 */
	protected void applyTransform(EmbeddedSurface source, Matrix R, Point3d pt) {
		int vertCount = source.getVertexCount();
		double[][] T = R.getArray();
		for (int i = 0; i < vertCount; i++) {
			Point3f tmp = new Point3f();
			Point3f sp = source.getVertex(i);
			tmp.x = (float) (T[0][0] * sp.x + T[0][1] * sp.y + T[0][2] * sp.z + pt.x);
			tmp.y = (float) (T[1][0] * sp.x + T[1][1] * sp.y + T[1][2] * sp.z + pt.y);
			tmp.z = (float) (T[2][0] * sp.x + T[2][1] * sp.y + T[2][2] * sp.z + pt.z);
			source.setVertex(i, tmp);
		}
	}

	/**
	 * Apply transformation to points
	 * 
	 * @param sourcePts
	 *            source points
	 * @param Tmat
	 *            transformation matrix
	 */
	protected void applyTransform(Point3f[] sourcePts, Matrix Tmat) {
		int vertCount = sourcePts.length;
		double[][] T = Tmat.getArray();
		for (int i = 0; i < vertCount; i++) {
			Point3f tmp = new Point3f();
			Point3f sp = sourcePts[i];
			tmp.x = (float) (T[0][0] * sp.x + T[0][1] * sp.y + T[0][2] * sp.z + T[0][3]);
			tmp.y = (float) (T[1][0] * sp.x + T[1][1] * sp.y + T[1][2] * sp.z + T[1][3]);
			tmp.z = (float) (T[2][0] * sp.x + T[2][1] * sp.y + T[2][2] * sp.z + T[2][3]);
			sourcePts[i] = tmp;
		}
	}

	/**
	 * Apply transformation to points
	 * 
	 * @param sourcePts
	 *            source points
	 * @param R
	 *            rotation matrix
	 * @param pt
	 *            translation
	 */
	protected void applyTransform(Point3f[] sourcePts, Matrix R, Point3d pt) {
		int vertCount = sourcePts.length;
		double[][] T = R.getArray();
		for (int i = 0; i < vertCount; i++) {
			Point3f tmp = new Point3f();
			Point3f sp = sourcePts[i];
			tmp.x = (float) (T[0][0] * sp.x + T[0][1] * sp.y + T[0][2] * sp.z + pt.x);
			tmp.y = (float) (T[1][0] * sp.x + T[1][1] * sp.y + T[1][2] * sp.z + pt.y);
			tmp.z = (float) (T[2][0] * sp.x + T[2][1] * sp.y + T[2][2] * sp.z + pt.z);
			sourcePts[i] = tmp;
		}
	}

	/**
	 * Estimate affine transformation from correspondence points
	 * 
	 * @param cps
	 *            correspondence points
	 * @param TRmat
	 *            rotation matrix
	 * @param pt
	 *            translation matrix
	 * @param N
	 *            number of correspondence points to use in estimation
	 * @return error
	 */
	protected double affineEstimate(Vector<CorrespondencePoints> cps,
			Matrix TRmat, Point3d pt, int N) {
		double[][] A = new double[4][4];
		double[][] B = new double[4][4];
		for (int n = 0; n < N; n++) {
			CorrespondencePoints cp = cps.get(n);
			A[0][0] += cp.source.x * cp.source.x;
			A[0][1] += cp.source.x * cp.source.y;
			A[0][2] += cp.source.x * cp.source.z;
			A[0][3] += cp.source.x;
			A[1][0] += cp.source.y * cp.source.x;
			A[1][1] += cp.source.y * cp.source.y;
			A[1][2] += cp.source.y * cp.source.z;
			A[1][3] += cp.source.y;
			A[2][0] += cp.source.z * cp.source.x;
			A[2][1] += cp.source.z * cp.source.y;
			A[2][2] += cp.source.z * cp.source.z;
			A[2][3] += cp.source.z;
			A[3][0] += cp.source.x;
			A[3][1] += cp.source.y;
			A[3][2] += cp.source.z;
			A[3][3] += 1;
			B[0][0] += cp.target.x * cp.source.x;
			B[0][1] += cp.target.x * cp.source.y;
			B[0][2] += cp.target.x * cp.source.z;
			B[0][3] += cp.target.x;
			B[1][0] += cp.target.y * cp.source.x;
			B[1][1] += cp.target.y * cp.source.y;
			B[1][2] += cp.target.y * cp.source.z;
			B[1][3] += cp.target.y;
			B[2][0] += cp.target.z * cp.source.x;
			B[2][1] += cp.target.z * cp.source.y;
			B[2][2] += cp.target.z * cp.source.z;
			B[2][3] += cp.target.z;
			B[3][0] += cp.source.x;
			B[3][1] += cp.source.y;
			B[3][2] += cp.source.z;
			B[3][3] += 1;
		}
		Matrix Amat = new Matrix(A);
		Matrix Bmat = new Matrix(B);
		double[][] T = Bmat.times(Amat.inverse()).getArray();
		double error = 0;
		for (CorrespondencePoints cp : cps) {
			Point3f tmp = new Point3f();
			tmp.x = (float) (T[0][0] * cp.source.x + T[0][1] * cp.source.y
					+ T[0][2] * cp.source.z + T[0][3]);
			tmp.y = (float) (T[1][0] * cp.source.x + T[1][1] * cp.source.y
					+ T[1][2] * cp.source.z + T[1][3]);
			tmp.z = (float) (T[2][0] * cp.source.x + T[2][1] * cp.source.y
					+ T[2][2] * cp.source.z + T[2][3]);
			cp.source.x = tmp.x;
			cp.source.y = tmp.y;
			cp.source.z = tmp.z;
			error += ((cp.source.x - cp.target.x) * (cp.source.x - cp.target.x)
					+ (cp.source.y - cp.target.y) * (cp.source.y - cp.target.y) + (cp.source.z - cp.target.z)
					* (cp.source.z - cp.target.z));
		}
		error = Math.sqrt(error / cps.size());
		TRmat.set(0, 0, T[0][0]);
		TRmat.set(0, 1, T[0][1]);
		TRmat.set(0, 2, T[0][2]);
		TRmat.set(1, 0, T[1][0]);
		TRmat.set(1, 1, T[1][1]);
		TRmat.set(1, 2, T[1][2]);
		TRmat.set(2, 0, T[2][0]);
		TRmat.set(2, 1, T[2][1]);
		TRmat.set(2, 2, T[2][2]);
		pt.x = T[0][3];
		pt.y = T[1][3];
		pt.z = T[2][3];
		return error;
	}

	/**
	 * Estimate translation
	 * 
	 * @param cps
	 *            correspondence points
	 * @param TRmat
	 *            rotation matrix
	 * @param pt
	 *            translation
	 * @param N
	 *            number of points to use in estimation
	 * @return error
	 */
	protected double translationEstimate(Vector<CorrespondencePoints> cps,
			Matrix TRmat, Point3d pt, int N) {
		int i = 0;
		Point3d sourceMean = new Point3d();
		Point3d targetMean = new Point3d();
		for (CorrespondencePoints p : cps) {
			if (i >= N)
				break;
			sourceMean.x += p.source.x;
			sourceMean.y += p.source.y;
			sourceMean.z += p.source.z;
			targetMean.x += p.target.x;
			targetMean.y += p.target.y;
			targetMean.z += p.target.z;
			i++;
		}
		sourceMean.scale(1 / (double) N);
		targetMean.scale(1 / (double) N);
		System.out.println("Source Mean: " + sourceMean);
		System.out.println("Target Mean: " + targetMean);
		pt.x = targetMean.x - sourceMean.x;
		pt.y = targetMean.y - sourceMean.y;
		pt.z = targetMean.z - sourceMean.z;
		double error = 0;
		for (CorrespondencePoints cp : cps) {
			cp.source.x += pt.x;
			cp.source.y += pt.y;
			cp.source.z += pt.z;
			error += (cp.source.x - cp.target.x) * (cp.source.x - cp.target.x)
					+ (cp.source.y - cp.target.y) * (cp.source.y - cp.target.y)
					+ (cp.source.z - cp.target.z) * (cp.source.z - cp.target.z);
		}
		error = Math.sqrt(error / cps.size());
		return error;
	}

	/**
	 * Estimate rigid transformation
	 * 
	 * @param cps
	 *            correspondence points
	 * @param TRmat
	 *            rotation matrix
	 * @param pt
	 *            translation
	 * @param N
	 *            number of points to use in estimation
	 * @param globalScale
	 *            true if global scale should be estimated too
	 * @return
	 */
	protected double rigidEstimate(Vector<CorrespondencePoints> cps,
			Matrix TRmat, Point3d pt, int N, boolean globalScale, Point3f[] sourcePts) {
				
		Point3d centroid1a = new Point3d(), centroid2a = new Point3d();
		double Sxx, Sxy, Sxz, Syx, Syy, Syz, Szx, Szy, Szz;
		double[][] R = new double[3][3];
		double[][] M = new double[4][4];
		double x, y, z;
		double scale1, scale2;
		double error = 0;
		int k, l;
		centroid1a.x = 0;
		centroid1a.y = 0;
		centroid1a.z = 0;
		centroid2a.x = 0;
		centroid2a.y = 0;
		centroid2a.z = 0;

		for (k = 0; k < N; k++) {		

			CorrespondencePoints cp = cps.get(k);

			centroid1a.x += cp.source.x;
			centroid1a.y += cp.source.y;
			centroid1a.z += cp.source.z;
			centroid2a.x += cp.target.x;
			centroid2a.y += cp.target.y;
			centroid2a.z += cp.target.z;

		}
		centroid1a.x /= (double) N;
		centroid1a.y /= (double) N;
		centroid1a.z /= (double) N;
		centroid2a.x /= (double) N;
		centroid2a.y /= (double) N;
		centroid2a.z /= (double) N;
		


		Sxx = 0;
		Sxy = 0;
		Sxz = 0;
		Syx = 0;
		Syy = 0;
		Syz = 0;
		Szx = 0;
		Szy = 0;
		Szz = 0;
		scale1 = 0;
		scale2 = 0;
		
		
		for (CorrespondencePoints cp : cps) {
			cp.source.x -= centroid1a.x;
			cp.source.y -= centroid1a.y;
			cp.source.z -= centroid1a.z;
			cp.target.x -= centroid2a.x;
			cp.target.y -= centroid2a.y;
			cp.target.z -= centroid2a.z;
		}
		
		
		
		for (k = 0; k < N; k++) {
			CorrespondencePoints cp = cps.get(k);
			if (globalScale) {
				scale1 += (cp.source.x * cp.source.x + cp.source.y
						* cp.source.y + cp.source.z * cp.source.z);
			}
			Sxx += cp.source.x * cp.target.x;
			Sxy += cp.source.x * cp.target.y;
			Sxz += cp.source.x * cp.target.z;
			Syx += cp.source.y * cp.target.x;
			Syy += cp.source.y * cp.target.y;
			Syz += cp.source.y * cp.target.z;
			Szx += cp.source.z * cp.target.x;
			Szy += cp.source.z * cp.target.y;
			Szz += cp.source.z * cp.target.z;
		}
		


		M[0][0] = Sxx + Syy + Szz;
		M[0][1] = Syz - Szy;
		M[0][2] = Szx - Sxz;
		M[0][3] = Sxy - Syx;
		M[1][0] = Syz - Szy;
		M[1][1] = Sxx - Syy - Szz;
		M[1][2] = Sxy + Syx;
		M[1][3] = Sxz + Szx;
		M[2][0] = Szx - Sxz;
		M[2][1] = Sxy + Sxy;
		M[2][2] = -Sxx + Syy - Szz;
		M[2][3] = Syz + Szy;
		M[3][0] = Sxy - Syx;
		M[3][1] = Sxz + Szx;
		M[3][2] = Szy + Syz;
		M[3][3] = -Sxx - Syy + Szz;
		for (k = 0; k < 4; k++) {
			for (l = 0; l < 4; l++) {
				M[k][l] /= (double) (N);
			}
		}


		Matrix Mmat = new Matrix(M);
		EigenvalueDecomposition ed = new EigenvalueDecomposition(Mmat);
		double[][] d = ed.getD().getArray();
		int maxEigIdx = 0;
		double maxEig = 0;
		for (int i = 0; i < 4; i++) {
			if (d[i][i] > maxEig) {
				maxEigIdx = i;
				maxEig = d[i][i];
			}
		}
		double[][] v = ed.getV().getArray();
		


		R[0][0] = v[0][maxEigIdx] * v[0][maxEigIdx] + v[1][maxEigIdx]
				* v[1][maxEigIdx] - v[2][maxEigIdx] * v[2][maxEigIdx]
				- v[3][maxEigIdx] * v[3][maxEigIdx];
		R[0][1] = 2 * (v[1][maxEigIdx] * v[2][maxEigIdx] - v[0][maxEigIdx]
				* v[3][maxEigIdx]);
		R[0][2] = 2 * (v[1][maxEigIdx] * v[3][maxEigIdx] + v[0][maxEigIdx]
				* v[2][maxEigIdx]);
		R[1][0] = 2 * (v[1][maxEigIdx] * v[2][maxEigIdx] + v[0][maxEigIdx]
				* v[3][maxEigIdx]);
		R[1][1] = v[0][maxEigIdx] * v[0][maxEigIdx] - v[1][maxEigIdx]
				* v[1][maxEigIdx] + v[2][maxEigIdx] * v[2][maxEigIdx]
				- v[3][maxEigIdx] * v[3][maxEigIdx];
		R[1][2] = 2 * (v[2][maxEigIdx] * v[3][maxEigIdx] - v[0][maxEigIdx]
				* v[1][maxEigIdx]);
		R[2][0] = 2 * (v[1][maxEigIdx] * v[3][maxEigIdx] - v[0][maxEigIdx]
				* v[2][maxEigIdx]);
		R[2][1] = 2 * (v[2][maxEigIdx] * v[3][maxEigIdx] + v[0][maxEigIdx]
				* v[1][maxEigIdx]);
		R[2][2] = v[0][maxEigIdx] * v[0][maxEigIdx] - v[1][maxEigIdx]
				* v[1][maxEigIdx] - v[2][maxEigIdx] * v[2][maxEigIdx]
				+ v[3][maxEigIdx] * v[3][maxEigIdx];
		for (CorrespondencePoints cp : cps) {
			x = R[0][0] * cp.source.x + R[1][0] * cp.source.y + R[2][0]
					* cp.source.z;
			y = R[0][1] * cp.source.x + R[1][1] * cp.source.y + R[2][1]
					* cp.source.z;
			z = R[0][2] * cp.source.x + R[1][2] * cp.source.y + R[2][2]
					* cp.source.z;
			cp.source.x = (float) x;
			cp.source.y = (float) y;
			cp.source.z = (float) z;
			if (globalScale) {
				scale2 += (cp.source.x * cp.target.x + cp.source.y
						* cp.target.y + cp.source.z * cp.target.z);
			}
		}
		


//		///////////////////////////////////////////////////////
//		System.out.println("R[][]:");
//		System.out.println(R[0][0] + "  " + R[0][1] + "  " + R[0][2]);
//		System.out.println(R[1][0] + "  " + R[1][1] + "  " + R[1][2]);
//		System.out.println(R[2][0] + "  " + R[2][1] + "  " + R[2][2]);
//		///////////////////////////////////////////////////////

		if (globalScale) {
			scale1 = scale2 / scale1;
		}
		for (CorrespondencePoints cp : cps) {
			if (globalScale) {
				cp.source.x *= scale1;
				cp.source.y *= scale1;
				cp.source.z *= scale1;
			}
			error += ((cp.source.x - cp.target.x) * (cp.source.x - cp.target.x)
					+ (cp.source.y - cp.target.y) * (cp.source.y - cp.target.y) + (cp.source.z - cp.target.z)
					* (cp.source.z - cp.target.z));
			cp.source.x += centroid2a.x;
			cp.source.y += centroid2a.y;
			cp.source.z += centroid2a.z;
		}
		
		
		double[][] TR = TRmat.getArray();
		if (globalScale) {
			TR[0][0] = scale1 * R[0][0];
			TR[0][1] = scale1 * R[0][1];
			TR[0][2] = scale1 * R[0][2];
			TR[1][0] = scale1 * R[1][0];
			TR[1][1] = scale1 * R[1][1];
			TR[1][2] = scale1 * R[1][2];
			TR[2][0] = scale1 * R[2][0];
			TR[2][1] = scale1 * R[2][1];
			TR[2][2] = scale1 * R[2][2];

			pt.x = scale1
					* (R[0][0] * (-centroid1a.x) + R[0][1] * (-centroid1a.y) + R[0][2]
							* (-centroid1a.z)) + centroid2a.x;
			pt.y = scale1
					* (R[1][0] * (-centroid1a.x) + R[1][1] * (-centroid1a.y) + R[1][2]
							* (-centroid1a.z)) + centroid2a.y;
			pt.z = scale1
					* (R[2][0] * (-centroid1a.x) + R[2][1] * (-centroid1a.y) + R[2][2]
							* (-centroid1a.z)) + centroid2a.z;

		} else {
			TR[0][0] = R[0][0];
			TR[0][1] = R[0][1];
			TR[0][2] = R[0][2];
			TR[1][0] = R[1][0];
			TR[1][1] = R[1][1];
			TR[1][2] = R[1][2];
			TR[2][0] = R[2][0];
			TR[2][1] = R[2][1];
			TR[2][2] = R[2][2];

			/*
			 * pt.x = (R[0][0] * (-centroid1a.x) + R[1][0] * (-centroid1a.y) +
			 * R[2][0] (-centroid1a.z)) + centroid2a.x; pt.y = (R[0][1] *
			 * (-centroid1a.x) + R[1][1] * (-centroid1a.y) + R[2][1]
			 * (-centroid1a.z)) + centroid2a.y; pt.z = (R[0][2] *
			 * (-centroid1a.x) + R[1][2] * (-centroid1a.y) + R[2][2]
			 * (-centroid1a.z)) + centroid2a.z;
			 */

			pt.x = (R[0][0] * (-centroid1a.x) + R[0][1] * (-centroid1a.y) + R[0][2]
					* (-centroid1a.z))
					+ centroid2a.x;
			pt.y = (R[1][0] * (-centroid1a.x) + R[1][1] * (-centroid1a.y) + R[1][2]
					* (-centroid1a.z))
					+ centroid2a.y;
			pt.z = (R[2][0] * (-centroid1a.x) + R[2][1] * (-centroid1a.y) + R[2][2]
					* (-centroid1a.z))
					+ centroid2a.z;
		}

//		///////////////////////////////////////////////////////
//		System.out.println("t[]: " + pt.x + "  " + pt.y + "  " + pt.z);
//		///////////////////////////////////////////////////////


		error = Math.sqrt(error / cps.size());
		return error;
	}

	/**
	 * Find closest points
	 * 
	 * @param sourcePts
	 *            source points
	 * @param randomIndexes
	 *            random ordering of indices to use for finding closest points.
	 * @param sampleSize
	 *            sample size for point correspondences
	 * @return collection of correspondences
	 */
	public Vector<CorrespondencePoints> findClosestPoints(Point3f[] sourcePts,
			ArrayList<Integer> randomIndexes, int sampleSize) {
		Point3f targetPt, sourcePt;
		int i;
		double d;
		Vector<CorrespondencePoints> cpts = new Vector<CorrespondencePoints>();
		for (int n = 0; n < sampleSize; n++) {
			i = randomIndexes.get(n);
			sourcePt = sourcePts[i];
			d = intersector.distance(sourcePt);
			if (d >= 0) {
				targetPt = intersector.getLastIntersectionPoint();
				// cpts.add(new CorrespondencePoints(sourcePt, (Point3f) targetPt
				//		.clone(), d));
				cpts.add(new CorrespondencePoints((Point3f) sourcePt.clone(), 
						(Point3f) targetPt.clone(), d));
			}
		}
		Collections.sort(cpts);
		return cpts;
	}
}
