package edu.jhu.ece.iacl.algorithms.thickness;

import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.SurfaceIntersector;
import edu.jhu.ece.iacl.algorithms.graphics.isosurf.IsoSurfaceOnGrid;
import edu.jhu.ece.iacl.algorithms.thickness.grid.EmbeddedGrid;
import edu.jhu.ece.iacl.algorithms.thickness.grid.EmbeddedNode;
import edu.jhu.ece.iacl.algorithms.thickness.grid.GridNode;
import edu.jhu.ece.iacl.algorithms.thickness.laplace.LaplaceSolveOnGraph;
import edu.jhu.ece.iacl.algorithms.thickness.laplace.LaplaceSolveOnGraphDirichlet;
import edu.jhu.ece.iacl.algorithms.topology.ConnectivityRule;
import edu.jhu.ece.iacl.jist.structures.data.BinaryMinHeap;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.jist.structures.image.MaskVolume6;
/**
 * Thickness method that combines Euler-Lagrange with Lagrangian-Embedding.
 * @author Blake Lucas (bclucas@jhu.edu)
 *
 */
public class HybridLagrangianEmbedding extends ThicknessSolver {
	public HybridLagrangianEmbedding(double lambda, double isoVal, double lagrangeDeltaT) {
		super(lambda, isoVal, lagrangeDeltaT);
		neighborMask=new MaskVolume6();
		approxMethod=EULER;
	}
	protected EmbeddedGrid grid;
	protected GridNode[][][] gridMat;
	protected SurfaceIntersector surfaceLocator;
	protected int approxMethod;
	public static final int EULER=0;
	public static final int LAGRANGE=1;
	public static final int EULER_LAGRANGE=2; 
	SurfaceIntersector intersector;
	public void solve(ImageData inner, ImageData outer) {
		this.innerVol = inner;
		this.outerVol = outer;
		rows = innerVol.getRows();
		cols = innerVol.getCols();
		slices = innerVol.getSlices();
		grid = new EmbeddedGrid(this,rows, cols, slices,false,false);
		gridMat=grid.getGrid();
		if(ADJUST_BOUNDARY)adjustBoundary(innerVol, outerVol);
		IsoSurfaceOnGrid surfGen=new IsoSurfaceOnGrid();
		innerMesh=(ADJUST_BOUNDARY)?surfGen.solveNoMove(innerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false):surfGen.solveOriginal(innerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false);
		this.innerVol=surfGen.getAlteredLevelSet();
		outerMesh=(ADJUST_BOUNDARY)?surfGen.solveNoMove(outerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false):surfGen.solveOriginal(outerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false);
		this.outerVol=surfGen.getAlteredLevelSet();
		System.out.println("NUMBER OF HOLES IN INNER SURFACE "+EmbeddedSurface.getGenus(innerMesh));
		System.out.println("NUMBER OF HOLES IN OUTER SURFACE "+EmbeddedSurface.getGenus(outerMesh));
		thicknessVol = new ImageDataFloat(rows, cols, slices);
		thicknessMat = thicknessVol.toArray3d();
		thicknessVol.setName(innerVol.getName()+"_thick");
		grid.build(this.innerVol, this.outerVol, innerMesh,outerMesh,isoVal,EmbeddedGrid.Topology.SIMPLE);
		markCompleted();
		computeLaplace();


		markCompleted();
		computeThickness();
		updateLevelSet();
		updateVolumes();
		innerMesh=getInnerSurface();
		outerMesh=getOuterSurface();
		grid.updateMeshThickness(innerMesh, outerMesh);
		updateSurfStats(innerMesh);
		updateSurfStats(outerMesh);
		markCompleted();
	}
	/**
	 * Solve laplace's equation on graph
	 */
	protected void computeLaplace() {		
		LaplaceSolveOnGraph laplaceSolver=new LaplaceSolveOnGraphDirichlet(this,rows,cols,slices);
		tangetField=laplaceSolver.solveTanget(grid);
		tangetFieldMat=tangetField.toArray4d();
		tangetField.setName(innerVol.getName()+"_tanget");
	}
	protected void computeThickness() {
		//convertToQuaternions();

		heap = new BinaryMinHeap(grid.getGridNodes().size()
				+ grid.getBoundaryNodes().length, rows, cols, slices);

		boolean tooFar=false;
		double dist=0;
		double len;
		int total=0;
		int l;
		EmbeddedNode neighbor;
		setTotalUnits(2 * (grid.getValidNodeCount()));
		//setLabel("Computing Thickness");
		int count = 0;
		int insideCount=0;
		initializeBoundary(OUTSIDE_OUTER);
		intersector=new SurfaceIntersector(this,12,outerMesh);
		EmbeddedNode nd;
		while (heap.size() > 0) {
			EmbeddedNode node = (EmbeddedNode) heap.remove();
			node.setMarchingLabel(SOLVED);
			incrementCompletedUnits();
			total++;
			if(total%10000==0){
				System.out.println("COMPLETED "+total+" REMAINING "+heap.size()+" TRACED "+100*count/(float)(insideCount)+" %");
			}
			
			if(node.regionLabel==INSIDE){
				tooFar=false;
				insideCount++;
				Point3f corr=node.getCorrespodence();
				for (l = 0; l < 6; l++) {
					nd = node.nbhd[l];
					//System.out.println(node.getLabelString()+" CORR "+corr+" "+nd.getCorrespodence()+" "+nd.getMarchingLabel()+" "+nd.getLabelString());
					
					if(nd!=null&&(nd.getMarchingLabel()==VISITED||nd.getMarchingLabel()==SOLVED)&&(dist=corr.distance(nd.getCorrespodence()))>lambda){
						tooFar=true;
						break;
					}
				}
				if(tooFar){
					traceApproximate(node, true);
					count++;
				} 
			}
			
			for (l = 0; l < 6; l++) {
				neighbor = node.nbhd[l];
				if (neighbor != null) {
					if (neighbor.getMarchingLabel() == VISITED) {
						len=updateNode(neighbor,node);
						if(len<0){
							double tmpLen=neighbor.getLength();
							traceApproximate(neighbor, true);
							len=neighbor.getLength();
							neighbor.setLength(tmpLen);
						}
						heap.change(neighbor,len);
					} else if (neighbor.getMarchingLabel() == UNVISITED) {
						len=updateNode(neighbor,node);
						if(len<0){
							traceApproximate(neighbor,true);
						} else {
							neighbor.setLength(len);
						}
						neighbor.setMarchingLabel(VISITED);
						heap.add(neighbor);
					}
				}
			}
		}	
		
		markCompleted();
		System.gc();
		
		heap = new BinaryMinHeap(grid.getGridNodes().size()
				+ grid.getBoundaryNodes().length, rows, cols, slices);
		initializeBoundary(INSIDE_INNER);
		intersector=new SurfaceIntersector(this,12,innerMesh);
		
		while (heap.size() > 0) {
			EmbeddedNode node = (EmbeddedNode) heap.remove();
			total++;
			node.setMarchingLabel(SOLVED);
			incrementCompletedUnits();

			if(total%10000==0){
				System.out.println("COMPLETED "+total+" REMAINING "+heap.size()+" TRACED "+100*count/(float)(insideCount)+" %");
			}
			
			if(node.regionLabel==INSIDE){
				insideCount++;
				tooFar=false;
				Point3f corr=node.getCorrespodence();
				for (l = 0; l < 6; l++) {
					nd = node.nbhd[l];
					if(nd!=null&&(nd.getMarchingLabel()==VISITED||nd.getMarchingLabel()==SOLVED)&&(dist=corr.distance(nd.getCorrespodence()))>lambda){
						tooFar=true;
						break;
					}
				}
				if(tooFar){
					traceApproximate(node, false);
					count++;
				} 
			}
			
			for (l = 0; l < 6; l++) {
				neighbor = node.nbhd[l];

				if (neighbor != null) {

					if (neighbor.getMarchingLabel() == VISITED) {				
						len=updateNode(neighbor,node);
						if(len<0){
							double tmpLen=neighbor.getLength();
							traceApproximate(neighbor, false);
							len=neighbor.getLength();
							neighbor.setLength(tmpLen);
						}
						heap.change(neighbor,len);
					} else if (neighbor.getMarchingLabel() == UNVISITED) {
						len=updateNode(neighbor,node);
						if(len<0){
							traceApproximate(neighbor,false);
						} else {
							neighbor.setLength(len);
						}
						neighbor.setMarchingLabel(VISITED);
						heap.add(neighbor);
					}
				}
			}
		}
		//sanityCheck();
		markCompleted();
		//computeError(true);
		surfaceLocator=null;
		heap=null;
		grid.relist();
		System.gc();
		System.out.println("TRACED "+100*count/(float)(insideCount)+" %");
		 
	}
	/**
	 * Compute distance from node location
	 * @param cd node
	 * @param boundary true indicates distance from outer boundary, false indicates distance from inner boundary
	 */
	protected void traceApproximate(EmbeddedNode cd,boolean boundary){
		double len=0;
		Point3f p2;
		Point3f p3;
		Point3f p1=cd.getLocation();
		double step=(boundary)?lagrangeDeltaT:-lagrangeDeltaT;
		int count=0;
		double mag;
		final int maxLength=100;
		while(len<maxLength){
			Point3f tan=interpolateTanget(p1);
			mag=tan.x*tan.x+tan.y*tan.y+tan.z*tan.z;
			if(mag<0.01){
				p1.add(new Point3f((float)Math.random()*1E-4f,(float)Math.random()*1E-4f,(float)Math.random()*1E-4f));			
				System.err.println("TANGET VECTOR INVALID "+count+" "+tan+" "+Math.sqrt(mag)+" "+p1);
				continue;
			}
			p2=new Point3f((float)(p1.x+step*tan.x),(float)(p1.y+step*tan.y),(float)(p1.z+step*tan.z));
				double d=intersector.intersectSegmentDistance(p1, p2);
				p3=intersector.getLastIntersectionPoint();
				if(p3==null){
					if(approximateDistance(cd, p1, p2,len)){
						return;
					} else {
						//cannot approximate distanc e, continue to trace.
						len+=lagrangeDeltaT;
						//set start of next end point
						p1=p2;
					}
				} else {
					cd.setLength(len+d);
					cd.setCorrespodence(p3);
					return;
				}
			count++;
		}
		System.out.println("MAXED OUT "+p1+" "+interpolateTanget(p1));
		double d=intersector.distance(cd.getLocation());
		cd.setLength(d);
		//cd.setCorrespodence(p);
	}
	/**
	 * Update value for node
	 * @param node node
	 * @param parent parent node
	 * @return distance
	 */
	protected double updateNode(EmbeddedNode node,EmbeddedNode parent){
		if(EmbeddedNode.isCompareToInner()){
			return updateInnerNode(node, parent);
		} else {
			return updateOuterNode(node, parent);
		}
	}
	/**
	 * Update value for node measuring distance to inner boundary
	 * @param node node
	 * @param parent parent node
	 * @return distance
	 */
	protected double updateInnerNode(EmbeddedNode node, EmbeddedNode parent) {
		double tx, ty, tz;
		Vector3f tan = node.getTanget();
		double numer;
		double numx=0,numy=0,numz=0;
		double denom = 0;
		double len=0;
		int dirX, dirY, dirZ;
		if(node.regionLabel==OUTER_BOUNDARY){
			return guessUsingEuler(node,parent);
		}
		if (tan.x < 0) {
			tx = -tan.x;
			dirX = EmbeddedNode.EAST;
	
		} else {
			tx = tan.x;
			dirX = EmbeddedNode.WEST;

		}
		if (tan.y < 0) {
			ty = -tan.y;
			dirY = EmbeddedNode.NORTH;
		} else {
			ty = tan.y;
			dirY = EmbeddedNode.SOUTH;
		}
		if (tan.z < 0) {
			tz = -tan.z;
			dirZ = EmbeddedNode.UP;
		} else {
			tz = tan.z;
			dirZ = EmbeddedNode.DOWN;
		}
		Point3f loc = node.getLocation();
		EmbeddedNode neighborX = node.nbhd[dirX];
		double hx = (neighborX != null) ? loc.distance(neighborX.getLocation())
				: 1;
		EmbeddedNode neighborY = node.nbhd[dirY];
		double hy = (neighborY != null) ? loc.distance(neighborY.getLocation())
				: 1;
		EmbeddedNode neighborZ = node.nbhd[dirZ];
		double hz = (neighborZ != null) ? loc.distance(neighborZ.getLocation())
				: 1;
		numer = hx * hy * hz;
		int count=0;
		Point3f px=null,py=null,pz=null;
		if (neighborX != null && neighborX.getMarchingLabel() != UNVISITED&&GeometricUtilities.angle(tan, neighborX.getTanget())<ANGLE_THRESHOLD) {
			numer += tx * neighborX.getLength() * hy * hz;
			denom += tx * hy * hz;
			px=neighborX.getCorrespodence();
			numx+=tx*px.x* hy * hz;
			numy+=tx*px.y* hy * hz;
			numz+=tx*px.z* hy * hz;
			count++;
		}
		if (neighborY != null && neighborY.getMarchingLabel() != UNVISITED&&GeometricUtilities.angle(tan, neighborY.getTanget())<ANGLE_THRESHOLD) {
			numer += ty * neighborY.getLength() * hx * hz;
			denom += ty * hx * hz;
			py=neighborY.getCorrespodence();
			numx+=ty*py.x* hx * hz;
			numy+=ty*py.y* hx * hz;
			numz+=ty*py.z* hx * hz;
			count++;
		}
		if (neighborZ != null && neighborZ.getMarchingLabel() != UNVISITED&&GeometricUtilities.angle(tan, neighborZ.getTanget())<ANGLE_THRESHOLD) {
			numer += tz * neighborZ.getLength() * hx * hy;
			denom += tz * hx * hy;
			pz=neighborZ.getCorrespodence();
			numx+=tz*pz.x* hx * hy;
			numy+=tz*pz.y* hx * hy;
			numz+=tz*pz.z* hx * hy;
			count++;
		}
		
		len = numer / denom;
		Point3f p=new Point3f();
		p.x=(float)(numx/denom);
		p.y=(float)(numy/denom);
		p.z=(float)(numz/denom);
		if (count<3||Double.isNaN(len)||Double.isInfinite(len)||Math.abs(parent.getLength()-len)>MAX_LENGTH) {
			return -1;
			//return guessUsingEuler(node,parent);
		} else {
			node.setCorrespodence(p);
			return len;
		}
	}
	/**
	 * Guess distance boundary for nodes just inside boundary
	 * @param node node
	 * @param parent parent node
	 * @return distance
	 */
	protected double guessUsingEuler(EmbeddedNode node, EmbeddedNode parent){
		double len=0;
		Vector3f tan = node.getTanget();
		Vector3f v=node.getDirection(parent);
		float d=((!EmbeddedNode.isCompareToInner())?-1:1)*tan.dot(v);
		if(d>=0&&d<MAX_DIST){
			
			len=d+parent.getLength();
		} else {	
			
			len=-1;
		}
		node.setCorrespodence(parent.getCorrespodence());
		return len;		
	}
	/**
	 * Update value for node measuring distance to outer boundary
	 * @param node node
	 * @param parent parent node
	 * @return distance
	 */
	protected double updateOuterNode(EmbeddedNode node, EmbeddedNode parent) {
		double tx = 0, ty = 0, tz = 0;
		Vector3f tan = node.getTanget();
		double numer;
		double numx=0,numy=0,numz=0;
		double denom = 0;
		EmbeddedNode neighborX;
		EmbeddedNode neighborY;
		EmbeddedNode neighborZ;
		int dirX = -1, dirY = -1, dirZ = -1;
		double len;
		if(node.regionLabel==INNER_BOUNDARY){
			return guessUsingEuler(node,parent);
		}
		if (tan.x < 0) {
			tx = -tan.x;
			dirX = EmbeddedNode.WEST;

		} else {
			tx = tan.x;
			dirX = EmbeddedNode.EAST;

		}
		if (tan.y < 0) {
			ty = -tan.y;
			dirY = EmbeddedNode.SOUTH;

		} else {
			ty = tan.y;
			dirY = EmbeddedNode.NORTH;

		}
		if (tan.z < 0) {
			tz = -tan.z;
			dirZ = EmbeddedNode.DOWN;
		} else {
			tz = tan.z;
			dirZ = EmbeddedNode.UP;
		}
		Point3f loc = node.getLocation();
		neighborX = node.nbhd[dirX];
		double hx = (neighborX != null) ? loc.distance(neighborX.getLocation())
				: 1;
		neighborY = node.nbhd[dirY];
		double hy = (neighborY != null) ? loc.distance(neighborY.getLocation())
				: 1;
		neighborZ = node.nbhd[dirZ];
		double hz = (neighborZ != null) ? loc.distance(neighborZ.getLocation())
				: 1;
		numer = hx * hy * hz;
		Point3f px=null,py=null,pz=null;
		int count=0;
		if (neighborX != null && neighborX.getMarchingLabel() != UNVISITED&&GeometricUtilities.angle(tan, neighborX.getTanget())<ANGLE_THRESHOLD) {
			numer += tx * neighborX.getLength() * hy * hz;
			denom += tx * hy * hz;
			px=neighborX.getCorrespodence();
			numx+=tx*px.x* hy * hz;
			numy+=tx*px.y* hy * hz;
			numz+=tx*px.z* hy * hz;
			count++;
		}
		if (neighborY != null && neighborY.getMarchingLabel() != UNVISITED&&GeometricUtilities.angle(tan, neighborY.getTanget())<ANGLE_THRESHOLD) {
			numer += ty * neighborY.getLength() * hx * hz;
			denom += ty * hx * hz;
			py=neighborY.getCorrespodence();
			numx+=ty*py.x* hx * hz;
			numy+=ty*py.y* hx * hz;
			numz+=ty*py.z* hx * hz;
			count++;
		}
		if (neighborZ != null && neighborZ.getMarchingLabel() != UNVISITED&&GeometricUtilities.angle(tan, neighborZ.getTanget())<ANGLE_THRESHOLD) {
			numer += tz * neighborZ.getLength() * hx * hy;
			denom += tz * hx * hy;
			pz=neighborZ.getCorrespodence();
			numx+=tz*pz.x* hx * hy;
			numy+=tz*pz.y* hx * hy;
			numz+=tz*pz.z* hx * hy;
			count++;
		}
		
		len = numer / denom;
		Point3f p=new Point3f();
		p.x=(float)(numx/denom);
		p.y=(float)(numy/denom);
		p.z=(float)(numz/denom);
		if (count<3||Double.isNaN(len)||Double.isInfinite(len)||Math.abs(parent.getLength()-len)>MAX_LENGTH) {
			return -1;
			//return guessUsingEuler(node,parent);
		} else {
			node.setCorrespodence(p);
			return len;
		}
	}
	/**
	 * Initialize boundary
	 * 
	 * @param boundary {INNER_BOUNDARY,OUTER_BOUNDARY}
	 */
	protected void initializeBoundary(int boundary) {
		int i, j, k;
		EmbeddedNode.setCompareInnerLength((boundary == INSIDE_INNER));
		for (i = 0; i < rows; i++) {
			for (j = 0; j < cols; j++) {
				for (k = 0; k < slices; k++) {
					EmbeddedNode node = grid.getNode(i, j, k);
					if (node == null)
						continue;
					if (node.regionLabel == INSIDE||node.regionLabel == JUST_INSIDE) {
						node.setMarchingLabel(UNVISITED);
					} else {
						node.setMarchingLabel(SOLVED);
					}
				}
			}
		}
		for (EmbeddedNode node : grid.getBoundaryNodes()) {
			if ((boundary == INSIDE_INNER && node.regionLabel == INNER_BOUNDARY)
					|| (boundary == OUTSIDE_OUTER && node.regionLabel == OUTER_BOUNDARY)) {
				node.setLength(0);
				node.setCorrespodence(new Point3f(node.i,node.j,node.k));
				node.setMarchingLabel(SOLVED);
				heap.add(node);
			} else {
				node.setMarchingLabel(UNVISITED);
			}
		}
	}
	/**
	 * Update level set to be the distance from the inner boundary
	 */
	protected void updateLevelSet(){
		int i,j,k;
		levelSetVol=new ImageDataFloat(rows,cols,slices);
		levelSetVol.setName(innerVol.getName()+"_levelset");
		float[][][] levelSetMat=levelSetVol.toArray3d();
		byte[][][] labels=grid.getLabels();
		byte label;
		for (i = 0; i < rows; i++) {
			for (j = 0; j < cols; j++) {
				for (k = 0; k < slices; k++) {
					EmbeddedNode node = grid.getNode(i, j, k);
					label=labels[i][j][k];
					/*
					if(node!=null){
						levelSetMat[i][j][k]=node.getMarchingLabel();
					}
					*/
					if(label==JUST_INSIDE||label==INSIDE){
						levelSetMat[i][j][k]=(float)node.getNormalizedLength();//(float)(Math.atan(5*(l0-level*(l0+l1)))/Math.PI+0.5);
					} else {
						levelSetMat[i][j][k]=0;
					}
					
				}
			}
		}
	}
	/**
	 * Update volume to corresond to thickness measurements
	 */
	protected void updateVolumes() {
		int  i, j, k;
		double sqrs = 0;
		double sum = 0;
		double count = 0;
		maxThickness=0;
		minThickness=1E10;
		for (i = 0; i < rows; i++) {
			for (j = 0; j < cols; j++) {
				for (k = 0; k < slices; k++) {
					
					EmbeddedNode node = grid.getNode(i, j, k);
					thicknessMat[i][j][k] = -1;
					if (node == null)continue;
					if(node.regionLabel==JUST_INSIDE||node.regionLabel==INSIDE){
						float thick = (float) (node.getThickness());
						if(Float.isNaN(thick)){
							System.err.println("NAN "+node+"\n"+node.getCorrespodence());
						} else {
							maxThickness=Math.max(thick, maxThickness);
							minThickness=Math.min(thick, minThickness);
							thicknessMat[i][j][k] = thick;
							sqrs += (thick) * (thick);
							sum += thick;
							count++;
						}
					} 

				}
			}
		}
		meanThickness=(sum/count);
		stdevThickness=Math.sqrt((sqrs-sum*sum/count)/(count-1));

		double expected=Math.round(meanThickness);
		double error=0;
		for (i = 0; i < rows; i++) {
			for (j = 0; j < cols; j++) {
				for (k = 0; k < slices; k++) {				
					EmbeddedNode node = grid.getNode(i, j, k);
					if (node == null)continue;
					if(node.regionLabel==JUST_INSIDE||node.regionLabel==INSIDE){
						float thick=(float)(node.getThickness());
						error+=Math.pow(thick-expected,2);
					}
				}
			}
		}

		error=Math.sqrt(error/count);
	}
	protected double getLength(int i,int j,int k){
		GridNode node=grid.getNode(i, j, k);
		if(node!=null){
			return node.getLength();
		} else {
			return 0;
		}
	}
	/**
	 * Approximate distance to boundary from neighbors
	 * @param dat current node to compute distance
	 * @param ref last point on streamline
	 * @param p current point to find distance
	 * @param len cumulative length up to last point
	 * @return length could be successfully approximated
	 */
	protected boolean approximateDistance(EmbeddedNode dat,Point3f ref,Point3f p,double len){
		Point3f pint;	
		Point3f result=null;
		double dist=1E10;
		double tmp=0;
		
		double xmin=Math.floor(ref.x);
		double ymin=Math.floor(ref.y);
		double zmin=Math.floor(ref.z);
		
		double xmax=xmin+1;
		double ymax=ymin+1;
		double zmax=zmin+1;
		double avgLength=0;
		if(xmin==ref.x&&ymin==ref.y&&zmin==ref.z){
			//double actual=trace(ref, false);
			//System.out.println("BROKE "+ref+","+dat.label);
			//dat.setCorrespodence(interpolateCoorespondence(ref));
			return false;
		}
		int plane=0;
		if(p.x<xmin){
			result=new Point3f(
					(float)xmin,
					(float)(ref.y+(p.y-ref.y)*(xmin-ref.x)/(p.x-ref.x)),
					(float)(ref.z+(p.z-ref.z)*(xmin-ref.x)/(p.x-ref.x)));
			dist=p.distance(result);
			plane=0;
		} else if(p.x>xmax){
			result=new Point3f(
					(float)xmax,
					(float)(ref.y+(p.y-ref.y)*(xmax-ref.x)/(p.x-ref.x)),
					(float)(ref.z+(p.z-ref.z)*(xmax-ref.x)/(p.x-ref.x)));
			dist=p.distance(result);
			plane=1;
		}
		if(p.y<ymin){
			pint=new Point3f(
					(float)(ref.x+(p.x-ref.x)*(ymin-ref.y)/(p.y-ref.y)),
					(float)(ymin),
					(float)(ref.z+(p.z-ref.z)*(ymin-ref.y)/(p.y-ref.y)));
			tmp=p.distance(pint);
			if(tmp<dist){
				result=pint;
				dist=tmp;
				plane=2;
			}
		} else if(p.y>ymax){
			pint=new Point3f(
					(float)(ref.x+(p.x-ref.x)*(ymax-ref.y)/(p.y-ref.y)),
					(float)(ymax),
					(float)(ref.z+(p.z-ref.z)*(ymax-ref.y)/(p.y-ref.y)));
			tmp=p.distance(pint);
			if(tmp<dist){
				result=pint;
				dist=tmp;
				plane=3;
			}
		}
		if(p.z<zmin){
			pint=new Point3f(
					(float)(ref.x+(p.x-ref.x)*(zmin-ref.z)/(p.z-ref.z)),
					(float)(ref.y+(p.y-ref.y)*(zmin-ref.z)/(p.z-ref.z)),
					(float)zmin);
			tmp=p.distance(pint);
			if(tmp<dist){
				result=pint;
				dist=tmp;
				plane=4;
			}
		} else if(p.z>zmax){
			pint=new Point3f(
					(float)(ref.x+(p.x-ref.x)*(zmax-ref.z)/(p.z-ref.z)),
					(float)(ref.y+(p.y-ref.y)*(zmax-ref.z)/(p.z-ref.z)),
					(float)zmax);
			tmp=p.distance(pint);
			if(tmp<dist){
				result=pint;
				dist=tmp;
				plane=5;
			}
		}
		if(result==null)return false;
		GridNode[] cd=new GridNode[4];
		cd[0]=grid.getNode((int)xmin+planePoint1[plane].x,(int)ymin+planePoint1[plane].y,(int)zmin+planePoint1[plane].z);
		cd[1]=grid.getNode((int)xmin+planePoint2[plane].x,(int)ymin+planePoint2[plane].y,(int)zmin+planePoint2[plane].z);
		cd[2]=grid.getNode((int)xmin+planePoint3[plane].x,(int)ymin+planePoint3[plane].y,(int)zmin+planePoint3[plane].z);
		cd[3]=grid.getNode((int)xmin+planePoint4[plane].x,(int)ymin+planePoint4[plane].y,(int)zmin+planePoint4[plane].z);
		int votes=0;
		boolean voters[]=new boolean[]{false,false,false,false};
		for(int i=0;i<4;i++){
			if(cd[i]!=null&&cd[i].getMarchingLabel()==SOLVED&&(cd[i].regionLabel==INSIDE||cd[i].regionLabel==JUST_INSIDE)){votes++;voters[i]=true;}
		}
		if(votes==4){
			avgLength=interpolateLength(result)+ref.distance(result);
			Point3f avgCorrespondence=interpolateCoorespondence(result);
			dat.setCorrespodence(avgCorrespondence);
			dat.setLength(len+avgLength);
			return true;
		} else {
			return false;
		}
	}
	/**
	 * Get correspondence point for particular grid node
	 */
	protected Point3f getCorrespondence(int i,int j,int k){
		if(i<0||j<0||k<0||i>=rows||j>=cols||k>=slices)return null;
		if(grid.getNode(i, j, k)==null)return new Point3f();
		Point3f corr=grid.getNode(i, j, k).getCorrespodence();
		if(corr==null)return new Point3f();
		return corr;
	}
}
