package edu.jhu.ece.iacl.algorithms.thickness;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import javax.vecmath.Point2d;
import javax.vecmath.Point3f;
import javax.vecmath.Vector3d;
import javax.vecmath.Vector3f;

import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.EmbeddedSurfaceIntersector;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.SurfaceIntersector;
import edu.jhu.ece.iacl.algorithms.graphics.isosurf.IsoSurfaceOnGrid;
import edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree.KdPoint3;
import edu.jhu.ece.iacl.algorithms.graphics.utilities.rbf.RadialBasisFunc4DWithDerivConst;
import edu.jhu.ece.iacl.algorithms.graphics.utilities.rbf.RadialFunctionType;
import edu.jhu.ece.iacl.algorithms.thickness.grid.EmbeddedGrid;
import edu.jhu.ece.iacl.algorithms.thickness.grid.EmbeddedNode;
import edu.jhu.ece.iacl.algorithms.thickness.grid.ExtendedOutsideNode;
import edu.jhu.ece.iacl.algorithms.thickness.grid.GridNode;
import edu.jhu.ece.iacl.algorithms.thickness.grid.SimpleInnerBoundaryNode;
import edu.jhu.ece.iacl.algorithms.thickness.grid.SimpleInsideNode;
import edu.jhu.ece.iacl.algorithms.thickness.grid.SimpleOuterBoundaryNode;
import edu.jhu.ece.iacl.algorithms.thickness.grid.SimpleOutsideNode;
import edu.jhu.ece.iacl.algorithms.thickness.laplace.LaplaceSolveOnGraph;
import edu.jhu.ece.iacl.algorithms.thickness.laplace.LaplaceSolveOnGraphDirichlet;
import edu.jhu.ece.iacl.algorithms.topology.ConnectivityRule;
import edu.jhu.ece.iacl.jist.structures.data.BinaryMinHeap;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.jist.structures.image.MaskVolume6;

import Jama.*;
/**
 * Simple Lagrangian-Embedding computes laplacian thickness using an euler method on a non-uniform grid.
 * The non-uniform grid incorporates the exact iso-surface boundary instead of using a voxelated boundary.
 * @author Blake Lucas (bclucas@jhu.edu)
 *
 */
public class SimpleLagrangianEmbedding extends ThicknessSolver {
	public SimpleLagrangianEmbedding(double lambda, double isoVal, double lagrangeDeltaT) {
		super(lambda, isoVal, lagrangeDeltaT);
		neighborMask=new MaskVolume6();
		approxMethod=EULER;
	}
	public SimpleLagrangianEmbedding() {
		super();
		neighborMask=new MaskVolume6();
		approxMethod=EULER;
	}
	protected boolean mapInnerToOuter=true;
	protected boolean mapOuterToInner=true;
	protected EmbeddedGrid grid;
	protected GridNode[][][] gridMat;
	protected SurfaceIntersector surfaceIntersector;
	protected int approxMethod;
	public static final int EULER=0;
	public static final int LAGRANGE=1;
	public static final int EULER_LAGRANGE=2; 
	protected double sorIters=25;
	protected double sorStep=1;
	protected double sorError=0.001;
	public void setMapToInner(boolean mapInner){
		this.mapInnerToOuter=mapInner;
	}
	public void setMapToOuter(boolean mapOuter){
		this.mapOuterToInner=mapOuter;
	}
	public void solve(ImageData inner, ImageData outer) {
		this.innerVol = inner;
		this.outerVol = outer;
		rows = innerVol.getRows();
		cols = innerVol.getCols();
		slices = innerVol.getSlices();
		grid = new EmbeddedGrid(this,rows, cols, slices,false,false);
		gridMat=grid.getGrid();
		if(ADJUST_BOUNDARY)adjustBoundary(innerVol, outerVol);
		//Compute inner and outer iso-surface mesh
		IsoSurfaceOnGrid surfGen=new IsoSurfaceOnGrid();
		innerMesh=(ADJUST_BOUNDARY)?surfGen.solveNoMove(innerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false):surfGen.solveOriginal(innerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false);
		this.innerVol=surfGen.getAlteredLevelSet();
		outerMesh=(ADJUST_BOUNDARY)?surfGen.solveNoMove(outerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false):surfGen.solveOriginal(outerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false);
		this.outerVol=surfGen.getAlteredLevelSet();
		if(EmbeddedSurface.getGenus(innerMesh)!=0||EmbeddedSurface.getGenus(outerMesh)!=0){
			System.err.println("Surfaces are not genus zero!");
			System.exit(-1);
		}
		thicknessVol = new ImageDataFloat(rows, cols, slices);
		thicknessMat = thicknessVol.toArray3d();
		thicknessVol.setName(innerVol.getName()+"_thick");
		//Build embedded grid
		grid.build(this.innerVol, this.outerVol, innerMesh,outerMesh,isoVal,EmbeddedGrid.Topology.SIMPLE);
		markCompleted();
		computeLaplace();
		markCompleted();
		computeThickness();
		innerMesh=getInnerSurface();
		outerMesh=getOuterSurface();
		grid.updateMeshThickness(innerMesh, outerMesh);
		updateSurfStats(innerMesh);
		updateSurfStats(outerMesh);
		markCompleted();
	}
	/**
	 * Compute solution to laplace's equation and compute tangential field
	 */
	protected void computeLaplace() {		
		LaplaceSolveOnGraph laplaceSolver=new LaplaceSolveOnGraphDirichlet(this,rows,cols,slices);
		tangetField=laplaceSolver.solveTanget(grid);
		tangetFieldMat=tangetField.toArray4d();
		tangetField.setName(innerVol.getName()+"_tanget");
	}
	/**
	 * Compute thickness via fast-marching
	 */
	protected void computeThickness() {
		heap = new BinaryMinHeap(grid.getGridNodes().size()
				+ grid.getBoundaryNodes().length, rows, cols, slices);
		int l;
		double len;
		int total=0;
		EmbeddedNode neighbor;
		setTotalUnits(2 * (grid.getValidNodeCount()));
		int count = 0;
		//Compute fast-marching from outer surface to inner surface
		if(mapOuterToInner){
			initializeBoundary(OUTSIDE_OUTER);
			//If lagrange or euler-lagrange is used to approximate distance to surface, then we'll need a surface intersector
			if(approxMethod==LAGRANGE||approxMethod==EULER_LAGRANGE){
				surfaceIntersector=new EmbeddedSurfaceIntersector(this,10,outerMesh);
			}
			while (heap.size() > 0) {
				EmbeddedNode node = (EmbeddedNode) heap.remove();
				node.setMarchingLabel(SOLVED);
				incrementCompletedUnits();
				total++;
				if(total%10000==0){
					System.out.println("COMPLETED "+total+" REMAINING "+heap.size());
				}
				//Examine 6-connected neighbors
				for (l = 0; l < 6; l++) {
					neighbor = node.nbhd[l];
					if (neighbor != null) {
						if (neighbor.getMarchingLabel() == VISITED) {
							//Update streamline length at node
							len=updateNode(neighbor,node);
							//update successful, change value in heap
							if(len>0)heap.change(neighbor,len);
						} else if (neighbor.getMarchingLabel() == UNVISITED) {
							//This is the first time the node has been updated
							len=updateNode(neighbor,node);
							if(len>0){
								neighbor.setLength(len);
								neighbor.setMarchingLabel(VISITED);
								heap.add(neighbor);
							} 
						}
					}
				}
			}	
			//updateNodesSOR();
			markCompleted();
			System.gc();
		}
		//Compute fast-marching from inner surface to outer surface
		if(mapInnerToOuter){
			heap = new BinaryMinHeap(grid.getGridNodes().size()
					+ grid.getBoundaryNodes().length, rows, cols, slices);
			initializeBoundary(INSIDE_INNER);
			if(approxMethod==LAGRANGE||approxMethod==EULER_LAGRANGE){
				surfaceIntersector=new EmbeddedSurfaceIntersector(this,10,innerMesh);
			}
			while (heap.size() > 0) {
				EmbeddedNode node = (EmbeddedNode) heap.remove();
				total++;
				node.setMarchingLabel(SOLVED);
				incrementCompletedUnits();
				if(total%10000==0){
					System.out.println("COMPLETED "+total+" REMAINING "+heap.size());
				}
				//Examine 6-connected neighbors
				for (l = 0; l < 6; l++) {
					neighbor = node.nbhd[l];
					if (neighbor != null) {
						if (neighbor.getMarchingLabel() == VISITED) {		
							//Update streamline length at node
							len=updateNode(neighbor,node);
							//update successful, change value in heap
							if(len>0)heap.change(neighbor,len);
						} else if (neighbor.getMarchingLabel() == UNVISITED) {
							//This is the first time the node has been updated
							len=updateNode(neighbor,node);
							if(len>0){
								neighbor.setLength(len);
								neighbor.setMarchingLabel(VISITED);
								heap.add(neighbor);
							} 
						}
					}
				}
			}
			//updateNodesSOR();
			markCompleted();
		}
		heap=null;
		System.gc();
		System.out.println("TRACED "+100*count/(float)total+" %");
	}
	/**
	 * Check that nodes have the correct connectivity
	 */
	protected void sanityCheck(){
		EmbeddedNode[] nodes=grid.getNodes();
		for(EmbeddedNode node:nodes){
			ArrayList<EmbeddedNode> nbhrs=node.getNeighbors();
			if(node instanceof GridNode){

				if(node instanceof SimpleInsideNode){
					if(nbhrs.size()!=6){
						System.out.println("INCORRECT # OF NEIGHBORS\n"+node);
					}
					if(node.regionLabel!=JUST_INSIDE&&node.regionLabel!=INSIDE){
						System.out.println("INSIDE LABEL ERROR "+node.getRegionLabelString()+" \n"+node);
					}
				} else if(node instanceof SimpleOutsideNode){
					if(node.regionLabel!=JUST_INSIDE_INNER_BOUNDARY&&node.regionLabel!=JUST_OUTSIDE_OUTER_BOUNDARY){
						System.out.println("OUTSIDE LABEL ERROR "+node.getRegionLabelString()+" \n"+node);
					}					
				} 
			} else if(node instanceof SimpleInnerBoundaryNode||node instanceof SimpleOuterBoundaryNode){
				if(nbhrs.size()!=2){
					System.out.println("INCORRECT # OF NEIGHBORS\n"+node);
				}
			}
		}
	}
	/**
	 * Trace distance from node to boundary
	 * @param node node
	 * @param boundary boundary f
	 */
	protected void trace(EmbeddedNode node,boolean boundary){
		double len=0;
		Point3f p2=null;
		int count=0;
		final int maxCount=700;
		Point3f p1=node.getLocation();
		double step=(boundary)?lagrangeDeltaT:-lagrangeDeltaT;
		while(count<maxCount){
			Point3f tan=interpolateTanget(p1);
			//if(p1.x*p1.x+p1.y*p1.y+p1.z*p1.z<0.25)return 0;
			p2=new KdPoint3(p1.x+step*tan.x,p1.y+step*tan.y,p1.z+step*tan.z);
			if(tan.x*tan.x+tan.y*tan.y+tan.z*tan.z<0.25){
				System.err.println("INVALID TANGET VECTOR "+count+" "+p1+" "+tan+" "+node.getLocation()+"\n"+node);
				tan.scale(1.0f/(float)Math.sqrt(tan.x*tan.x+tan.y*tan.y+tan.z*tan.z));
				System.exit(0);
			}
			/*
			if(p1.x==p2.x&&Math.floor(p1.x)==p1.x){
				p1.x+=0.0001*Math.random();
			}
			if(p1.y==p2.y&&Math.floor(p1.y)==p1.y){
				p1.y+=0.0001*Math.random();
			}
			if(p1.z==p2.z&&Math.floor(p1.z)==p1.z){
				p1.z+=0.0001*Math.random();
			}
			*/
			double dist=surfaceIntersector.intersectSegmentDistance(p1,p2);

			if(dist<0){
				len+=lagrangeDeltaT;
					//set start of next end point
				p1=p2;
			} else {
				node.setLength(len+dist);
				return;
			}
			count++;
		}
		System.err.println("MAXED OUT "+p1+" "+p2);
	}
	protected void traceApproximate(EmbeddedNode cd,boolean boundary){
		double len=0;
		Point3f p2;
		Point3f p3;
		Point3f p1=cd.getLocation();
		double step=(boundary)?lagrangeDeltaT:-lagrangeDeltaT;
		int count=0;
		final int maxCount=200;
		while(count<maxCount){
			Point3f tan=interpolateTanget(p1);
			p2=new KdPoint3(p1.x+step*tan.x,p1.y+step*tan.y,p1.z+step*tan.z);
			if(tan.x*tan.x+tan.y*tan.y+tan.z*tan.z<0.25){
				System.err.println("INVALID TANGET VECTOR "+p1+" "+tan);
				tan.scale(1.0f/(float)Math.sqrt(tan.x*tan.x+tan.y*tan.y+tan.z*tan.z));
				//return 0;
			}
			p2=new Point3f((float)(p1.x+step*tan.x),(float)(p1.y+step*tan.y),(float)(p1.z+step*tan.z));
			/*
			if(p1.x==p2.x&&Math.floor(p1.x)==p1.x){
				p1.x+=0.0001;
			}
			if(p1.y==p2.y&&Math.floor(p1.y)==p1.y){
				p1.y+=0.0001;
			}
			if(p1.z==p2.z&&Math.floor(p1.z)==p1.z){
				p1.z+=0.0001;
			}
			*/
				double dist=surfaceIntersector.intersectSegmentDistance(p1,p2);	
				p3=surfaceIntersector.getLastIntersectionPoint();
				//No intersection, add full length of segment
				if(p3==null){
					if(approximateDistance(cd, p1, p2,len)){
						return;
					} else {
						//cannot approximate distance, continue to trace.
						len+=lagrangeDeltaT;
						//set start of next end point
						p1=p2;
					}
				} else {
					cd.setLength(len+dist);
					return;
				}
			count++;
		}
		cd.setLength(0);
		//System.out.println("MAXED OUT "+p1+" "+interpolateTanget(p1)+" "+intersector.isInside(p1));
	}
	protected static final byte UPDATE=1;
	protected static final byte KEEP=0;
	protected static final byte NOUPDATE=2;
	/*
	protected void updateNodesSOR(){
		EmbeddedNode[] nodes=grid.getNodes();
		double[] currentLengths=new double[nodes.length];
		int i=0;
		double err;
		double len;
		double clen;
		//CONSTRAINED=false;

		boolean sgn=EmbeddedNode.isCompareToInner();
		for(EmbeddedNode n:nodes){
			if(n instanceof SimpleOutsideNode||(!sgn&&n.regionLabel==EmbeddedNode.OUTER_BOUNDARY)||(sgn&&n.regionLabel==EmbeddedNode.INNER_BOUNDARY)){	
				n.setMarchingLabel(NOUPDATE);
			} else {
				n.setMarchingLabel(UPDATE);
				currentLengths[i]=n.getLength();
			}
			i++;
		}
		int validCount=0;
		double delta;
		byte label;
		delta=sorStep;//*Math.exp(-count/sorIters);
		for(int count=0;count<sorIters;count++){
			err=0;
			i=0;
			validCount=0;
			for(EmbeddedNode n:nodes){
				if(n.getMarchingLabel()==UPDATE){
					len=updateNode(n, null);
					if(len>0){
						validCount++;
						clen=currentLengths[i];
						err+=(len-clen)*(len-clen);
						/
						if(Math.abs(len-clen)>=0.5*sorError){
							for(EmbeddedNode nbr:n.getNeighbors()){
								if(nbr.getMarchingLabel()==KEEP){
									nbr.setMarchingLabel(UPDATE);
								}
							}
							currentLengths[i]=clen+delta*(len-clen);
						} else {
							n.setMarchingLabel(KEEP);
						}
						/
						currentLengths[i]=clen+delta*(len-clen);
						
					}
				} 
				i++;
			}
			if(validCount>0){
				err/=validCount;
			}
			err=Math.sqrt(Math.abs(err));
			System.out.println((count+1)+") ERROR "+err+" UPDATE NODES "+validCount);
			if(err<sorError){
				break;
			}
			i=0;
			
			for(EmbeddedNode n:nodes){
				label=n.getMarchingLabel();
				if(label==UPDATE){
					n.setLength(currentLengths[i]);
				}
				i++;
			}
		}
		//CONSTRAINED=true;
	}
	*/
	/*
	protected double guessUsingRBF(EmbeddedNode node){
		List<EmbeddedNode> boundaryNodes=new LinkedList<EmbeddedNode>();
		Vector3f tan=new Vector3f(node.getTanget());
		Point3f pivot=node.getLocation();
		Vector3f diff=new Vector3f();
		boolean sgn=EmbeddedNode.isCompareToInner();
		if(!EmbeddedNode.isCompareToInner()){
			tan.negate();
		}
		for(EmbeddedNode nd:node.getNeighbors()){
			if(nd.getMarchingLabel()!= UNVISITED&&!(nd instanceof ExtendedOutsideNode)){
				diff.sub(pivot,nd.getLocation());
				if(tan.dot(diff)>0){
					boundaryNodes.add(nd);
				}
			}
		}
		Point3f[] pts=new Point3f[boundaryNodes.size()];
		Vector3f[] devs=new Vector3f[boundaryNodes.size()];
		double[] lens=new double[boundaryNodes.size()];
		int i=0;
		for(EmbeddedNode n:boundaryNodes){
			System.out.println("NODE "+n.getLocation()+" "+n.getRegionLabelString()+" "+(180*GeomUtil.angle(n.getGradient(),tan)/Math.PI));
			pts[i]=n.getLocation();
			devs[i]=n.getGradient();
			lens[i]=n.getLength();
			i++;
		}
		RadialBasisFunc4DWithDerivConst rbf=new RadialBasisFunc4DWithDerivConst(pts,devs,lens,RadialFunctionType.THIN_PLATE);
		i=0;
		double len=rbf.interpolate(node.getLocation());
		Vector3f grad=rbf.interpolateGradient(node.getLocation());
		System.out.println(node.getRegionLabelString()+" OLD LENGTH "+node.getLength()+" NEW LENGTH "+len);
		System.out.println("OLD GRADIENT "+node.getGradient()+" NEW GRADIENT "+grad+" ANGLE "+180*GeomUtil.angle(node.getGradient(),grad)/Math.PI);
		node.setGradient(grad);
		return len;
	}
	*/
	protected double guessUsingEuler(EmbeddedNode node, EmbeddedNode parent){
		double len=0;
		Vector3f tan = node.getTanget();
		Vector3f v=node.getDirection(parent);
		float d=((!EmbeddedNode.isCompareToInner())?-1:1)*tan.dot(v);

		if(d>=0&&d<MAX_DIST){
			len=d+parent.getLength();
		} else {			
			//System.out.println("BAD GUESS "+d+" "+tan+" "+v);
			len=-1;
		}
		node.setCorrespodence(parent.getCorrespodence());
		return len;		
	}
	protected double guessUsingEulerLagrange(EmbeddedNode node){
		traceApproximate(node,!EmbeddedNode.isCompareToInner());
		return node.getLength();		
	}	
	protected double guessUsingLagrange(EmbeddedNode node){
		trace(node,!EmbeddedNode.isCompareToInner());
		return node.getLength();		
	}
	/**
	 * Cannot compute length using fast-marching, guess correct streamline length using different method
	 * @param node
	 * @param parent
	 * @return
	 */
	protected double guess(EmbeddedNode node,EmbeddedNode parent){
		if(parent==null)return -1;
		switch(approxMethod){
			case  EULER:
				return guessUsingEuler(node,parent);
			case LAGRANGE:
				return guessUsingLagrange(node);
			case EULER_LAGRANGE:
				return guessUsingEulerLagrange(node);
			default:
				System.err.println("Approximation Method Not Specified");
				System.exit(1);
				return 0;
		}
	}
	/**
	 * Compute update for streamline passing through voxel
	 * @param node
	 * @param parent
	 * @return
	 */
	protected double updateNode(EmbeddedNode node,EmbeddedNode parent){
		if(EmbeddedNode.isCompareToInner()){
			return updateInnerNode(node, parent);
		} else {
			return updateOuterNode(node, parent);
		}
	}
	/**
	 * Compute update for streamline length to inner surface passing through voxel
	 * @param node
	 * @param parent
	 * @return
	 */
	protected double updateInnerNode(EmbeddedNode node, EmbeddedNode parent) {
		double tx, ty, tz;
		Vector3f tan = node.getTanget();
		double numer;
		double denom = 0;
		double len=0;
		int dirX, dirY, dirZ;
		if(node.regionLabel==OUTER_BOUNDARY){
			//System.out.println("OUTER BOUNDARY NODE");
			return guess(node,parent);
		}
		if (tan.x < 0) {
			tx = -tan.x;
			dirX = EmbeddedNode.EAST;
		} else {
			tx = tan.x;
			dirX = EmbeddedNode.WEST;
		}
		if (tan.y < 0) {
			ty = -tan.y;
			dirY = EmbeddedNode.NORTH;
		} else {
			ty = tan.y;
			dirY = EmbeddedNode.SOUTH;
		}
		if (tan.z < 0) {
			tz = -tan.z;
			dirZ = EmbeddedNode.UP;
		} else {
			tz = tan.z;
			dirZ = EmbeddedNode.DOWN;
		}
		Point3f loc = node.getLocation();
		EmbeddedNode neighborX = node.nbhd[dirX];
		double hx = (neighborX != null) ? loc.distance(neighborX.getLocation())
				: 1;
		EmbeddedNode neighborY = node.nbhd[dirY];
		double hy = (neighborY != null) ? loc.distance(neighborY.getLocation())
				: 1;
		EmbeddedNode neighborZ = node.nbhd[dirZ];
		double hz = (neighborZ != null) ? loc.distance(neighborZ.getLocation())
				: 1;
		numer = hx * hy * hz;
		int count=0;
		if (neighborX != null && neighborX.getMarchingLabel() != UNVISITED) {
			numer += tx * neighborX.getLength() * hy * hz;
			denom += tx * hy * hz;
			count++;
		}
		if (neighborY != null && neighborY.getMarchingLabel() != UNVISITED) {
			numer += ty * neighborY.getLength() * hx * hz;
			denom += ty * hx * hz;
			count++;
		}
		if (neighborZ != null && neighborZ.getMarchingLabel() != UNVISITED) {
			numer += tz * neighborZ.getLength() * hx * hy;
			denom += tz * hx * hy;
			count++;
		}
		
		len = numer / denom;
		if (Double.isNaN(len)||Double.isInfinite(len)) {			
			return guess(node,parent);
		} else {
			if(count<3)return len;
			Vector3d grad=new Vector3d((len-neighborX.getLength())/hx,(len-neighborY.getLength())/hy,(len-neighborZ.getLength())/hz);
			if(grad.length()>MAX_LENGTH){
				return guess(node,parent);
			} else {
				return len;
			}
		}
	}
	/**
	 * Compute update for streamline length to outer surface passing through voxel
	 * @param node
	 * @param parent
	 * @return
	 */
	protected double updateOuterNode(EmbeddedNode node, EmbeddedNode parent) {
		double tx = 0, ty = 0, tz = 0;
		Vector3f tan = node.getTanget();
		double numer;
		double denom = 0;
		EmbeddedNode neighborX;
		EmbeddedNode neighborY;
		EmbeddedNode neighborZ;
		int dirX = -1, dirY = -1, dirZ = -1;
		double len;
		if(node.regionLabel==INNER_BOUNDARY){
			return guess(node,parent);
		}
		if (tan.x < 0) {
			tx = -tan.x;
			dirX = EmbeddedNode.WEST;
		} else {
			tx = tan.x;
			dirX = EmbeddedNode.EAST;
		}
		if (tan.y < 0) {
			ty = -tan.y;
			dirY = EmbeddedNode.SOUTH;
		} else {
			ty = tan.y;
			dirY = EmbeddedNode.NORTH;
		}
		if (tan.z < 0) {
			tz = -tan.z;
			dirZ = EmbeddedNode.DOWN;
		} else {
			tz = tan.z;
			dirZ = EmbeddedNode.UP;
		}
		Point3f loc = node.getLocation();
		neighborX = node.nbhd[dirX];
		double hx = (neighborX != null) ? loc.distance(neighborX.getLocation())
				: 1;
		neighborY = node.nbhd[dirY];
		double hy = (neighborY != null) ? loc.distance(neighborY.getLocation())
				: 1;
		neighborZ = node.nbhd[dirZ];
		double hz = (neighborZ != null) ? loc.distance(neighborZ.getLocation())
				: 1;
		int count=0;
		numer = hx * hy * hz;
		
		if (neighborX != null && neighborX.getMarchingLabel() != UNVISITED) {
			numer += tx * neighborX.getLength() * hy * hz;
			denom += tx * hy * hz;
			count++;
		}
		if (neighborY != null && neighborY.getMarchingLabel() != UNVISITED) {
			numer += ty * neighborY.getLength() * hx * hz;
			denom += ty * hx * hz;
			count++;
		}
		if (neighborZ != null && neighborZ.getMarchingLabel() != UNVISITED) {
			numer += tz * neighborZ.getLength() * hx * hy;
			denom += tz * hx * hy;
			count++;
		}
		
		len = numer / denom;
		
		if (Double.isNaN(len)||Double.isInfinite(len)) {			
			return guess(node,parent);
		} else {
			if(count<3)return len;
			Vector3d grad=new Vector3d((len-neighborX.getLength())/hx,(len-neighborY.getLength())/hy,(len-neighborZ.getLength())/hz);
			if(grad.length()>MAX_LENGTH){
				return guess(node,parent);
			} else {
				return len;
			}
		}
	}
	/**
	 * Initialize boundary for fast-marching computation
	 * @param boundary
	 */
	protected void initializeBoundary(int boundary) {
		int i, j, k;
		EmbeddedNode.setCompareInnerLength((boundary == INSIDE_INNER));
		for (i = 0; i < rows; i++) {
			for (j = 0; j < cols; j++) {
				for (k = 0; k < slices; k++) {
					EmbeddedNode node = grid.getNode(i, j, k);
					if (node == null)
						continue;
					if (node.regionLabel == INSIDE||node.regionLabel == JUST_INSIDE) {
						node.setMarchingLabel(UNVISITED);
					} else {
						node.setMarchingLabel(SOLVED);
					}
				}
			}
		}
		for (EmbeddedNode node : grid.getBoundaryNodes()) {
			if ((boundary == INSIDE_INNER && node.regionLabel == INNER_BOUNDARY)
					|| (boundary == OUTSIDE_OUTER && node.regionLabel == OUTER_BOUNDARY)) {
				node.setLength(0);
				node.setMarchingLabel(SOLVED);
				heap.add(node);
			} else {
				node.setMarchingLabel(UNVISITED);
			}
		}
	}
	/*
	protected void updateLevelSet(){
		int i,j,k;
		//levelSetVol=new CubicVolumeFloat(rows,cols,slices);
		//levelSetVol.setName(innerVol.getName()+"_levelset");
		//float[][][] levelSetMat=levelSetVol.toArray3d();
		byte[][][] labels=grid.getLabels();
		byte label;
		for(EmbeddedNode n:grid.getBoundaryNodes()){
			n.setMarchingLabel((byte)0);
		}
		for (i = 0; i < rows; i++) {
			for (j = 0; j < cols; j++) {
				for (k = 0; k < slices; k++) {
					EmbeddedNode node = grid.getNode(i, j, k);
					label=labels[i][j][k];
					
					if(label==JUST_INSIDE||label==INSIDE){
						//node.setMarchingLabel((byte)levelSetMat[node.i][node.j][node.k]);
						//levelSetMat[i][j][k]=node.getMarchingLabel();
						Vector3f tan=new Vector3f(node.getTanget());
						//if(!EmbeddedNode.isCompareToInner()){
							//tan.negate();
						//}
						Point3f q=node.getCorrespodence();
						Point2d d=GeomUtil.toSpherical(q);
						tangetFieldMat[node.i][node.j][node.k][0]=(float)d.x;
						tangetFieldMat[node.i][node.j][node.k][1]=(float)d.y;
						tangetFieldMat[node.i][node.j][node.k][2]=(float)1;
						//if(levelSetMat[i][j][k]!=0){
							//levelSetMat[i][j][k]=(float)(GeomUtil.angle(tan, node.getGradient())*180.0/Math.PI);
						//}
						//levelSetMat[i][j][k]=(float)(GeomUtil.angle(tan, node.getGradient())*180.0/Math.PI);
					} else {
						//levelSetMat[i][j][k]=-1;
					}
					

					
				}
			}
		}
	}
	*/
	/*
	protected void updateTotalThickness() {
		int i, j, k;
		double sqrs = 0;
		double sum = 0;
		double count = 0;
		maxThickness=0;
		minThickness=1E10;
		for (i = 0; i < rows; i++) {
			for (j = 0; j < cols; j++) {
				for (k = 0; k < slices; k++) {
					
					EmbeddedNode node = grid.getNode(i, j, k);
					thicknessMat[i][j][k] = -1;
					tangetFieldMat[i][j][k][0]=0;
					tangetFieldMat[i][j][k][1]=0;
					tangetFieldMat[i][j][k][2]=0;
					if (node == null)continue;
					if(node.regionLabel==JUST_INSIDE||node.regionLabel==INSIDE){
						float thick = (float) (node.getThickness());
						Point3f pt=node.getCorrespodence();
						tangetFieldMat[i][j][k][0]=pt.x;
						tangetFieldMat[i][j][k][1]=pt.y;
						tangetFieldMat[i][j][k][2]=pt.z;
						//EmbeddedNode.setCompareInnerLength(true);
						//if(node.getLength()==0){
						//	System.out.println("INNER "+node.getLocation()+" "+node.getRegionLabelString()+" "+node.getMarchingLabel()+" "+node.getLength()+" "+node.getThickness());
						//}
						//EmbeddedNode.setCompareInnerLength(false);
						//if(node.getLength()==0){
						//	System.out.println("OUTER "+node.getLocation()+" "+node.getRegionLabelString()+" "+node.getMarchingLabel()+" "+node.getLength()+" "+node.getThickness());
						//}
						if(Float.isNaN(thick)){
							System.err.println("NAN "+node+"\n"+node.getCorrespodence());
						} else {
							maxThickness=Math.max(thick, maxThickness);
							minThickness=Math.min(thick, minThickness);
							thicknessMat[i][j][k] = thick;
							sqrs += (thick) * (thick);
							sum += thick;
							count++;
						}
					} 
				}
			}
		}
		meanThickness=(sum/count);
		stdevThickness=Math.sqrt((sqrs-sum*sum/count)/(count-1));

		double expected=Math.round(meanThickness);
		double error=0;
		for (i = 0; i < rows; i++) {
			for (j = 0; j < cols; j++) {
				for (k = 0; k < slices; k++) {				
					EmbeddedNode node = grid.getNode(i, j, k);
					if (node == null)continue;
					if(node.regionLabel==JUST_INSIDE||node.regionLabel==INSIDE){
						float thick=(float)(node.getThickness());
						error+=Math.pow(thick-expected,2);
					}
				}
			}
		}

		error=Math.sqrt(error/count);
		System.out.println(expected+" "+meanThickness+" "+stdevThickness+" "+error+" "+minThickness+" "+maxThickness+" "+count);
	}
	*/
	/*
	protected Point2d getCorrespondence(Matrix A,int i,int j,int k){
		GridNode node=grid.getNode(i, j, k);
		if(node!=null){
			return GeomUtil.toUnitSpherical(GeomUtil.multMatrix(A, node.getCorrespodence()));
		} else {
			return new Point2d();
		}
	}
	*/
	/*
	protected Point3f interpolateCoorespondence(Point3f p,Matrix A){
		int i0, j0, k0, i1, j1, k1;
		double dx, dy, dz, hx, hy, hz;
		double x=p.x,y=p.y,z=p.z;
		j0 = (int) Math.floor(x);
		i0 = (int) Math.floor(y);
		k0 = (int) Math.floor(z);
		j1 = j0+1;
		i1 = i0+1;
		k1 = k0+1;
		if (j0 < 0 || j1 > (rows - 1) || i0 < 0 || i1 > (cols - 1) || k0 < 0
				|| k1 > (slices - 1)) {
			System.out.println("INVALID "+x+" "+y+" "+z+" "+rows+" "+cols+" "+slices);
			return grid.getNode(j0,i0,k0).getCorrespodence();
		} else {
			dx = x - j0;
			dy = y - i0;
			dz = z - k0;
			// Introduce more variables to reduce computation 
			hx = 1.0 - dx;
			hy = 1.0 - dy;
			hz = 1.0 - dz;
			Point2d result=new Point2d();
			Point2d p1=getCorrespondence(A,j0,i0,k0);
			Point2d p2=getCorrespondence(A,j1,i0,k0);
			Point2d p3=getCorrespondence(A,j0,i1,k0);
			Point2d p4=getCorrespondence(A,j1,i1,k0);
			Point2d p5=getCorrespondence(A,j0,i0,k1);
			Point2d p6=getCorrespondence(A,j1,i0,k1);
			Point2d p7=getCorrespondence(A,j0,i1,k1);
			Point2d p8=getCorrespondence(A,j1,i1,k1);
			result.x=(((p1.x * hx + 
					p2.x* dx)* hy + 
					(p3.x* hx + 
							p4.x* dx)* dy)* hz+ 
					((p5.x * hx + 
							p6.x* dx)* hy + 
					(p7.x * hx + 
							p8.x* dx)
							* dy) * dz);
			result.y=(((p1.y * hx + 
					p2.y* dx)* hy + 
					(p3.y* hx + 
							p4.y* dx)* dy)* hz+ 
					((p5.y * hx + 
							p6.y* dx)* hy + 
					(p7.y * hx + 
							p8.y* dx)
							* dy) * dz);

			return GeomUtil.multMatrix(A.inverse(),GeomUtil.toCartesian(result));
		}
}		
	*/
	/**
	 * Interpolate streamline length to surface a specified point
	 */
	protected double interpolateLength(Point3f p){
		int i0, j0, k0, i1, j1, k1;
		double dx, dy, dz, hx, hy, hz;
		double x=p.x,y=p.y,z=p.z;
		j0 = (int) Math.floor(x);
		i0 = (int) Math.floor(y);
		k0 = (int) Math.floor(z);
		j1 = j0+1;
		i1 = i0+1;
		k1 = k0+1;
		//System.out.println("("+j1+","+i1+","+k1+") ("+j0+","+i0+","+k0+") "+p+getLength(j0,i0,k0));
		if(j0==x&&i0==y&&k0==z){
			return getLength(j0,k0,i0);
		}
		if (j0 < 0 || j1 > (rows - 1) || i0 < 0 || i1 > (cols - 1) || k0 < 0
				|| k1 > (slices - 1)) {
			System.out.println("INVALID "+x+" "+y+" "+z+" "+rows+" "+cols+" "+slices);
			return 0;
		} else {
			dx = x - j0;
			dy = y - i0;
			dz = z - k0;
			/* Introduce more variables to reduce computation */
			hx = 1.0 - dx;
			hy = 1.0 - dy;
			hz = 1.0 - dz;
			return (((getLength(j0,i0,k0) * hx + 
					getLength(j1,i0,k0)* dx)* hy + 
					(getLength(j0,i1,k0)* hx + 
					getLength(j1,i1,k0)* dx)* dy)* hz+ 
					((getLength(j0,i0,k1)* hx + 
					getLength(j1,i0,k1)* dx)* hy + 
					(getLength(j0,i1,k1)* hx + 
					getLength(j1,i1,k1)* dx)
							* dy) * dz);
		}
	}
	/**
	 * Get streamline length to surface
	 */
	protected double getLength(int i,int j,int k){
		GridNode node=grid.getNode(i, j, k);
		if(node!=null){
			return node.getLength();
		} else {
			return 0;
		}
	}
	/**
	 * Approximate distance to surface using lagrange method
	 * @param dat
	 * @param ref
	 * @param p
	 * @param len
	 * @return
	 */
	protected boolean approximateDistance(EmbeddedNode dat,Point3f ref,Point3f p,double len){
		Point3f pint;	
		Point3f result=null;
		double dist=1E10;
		double tmp=0;
		
		double xmin=Math.floor(ref.x);
		double ymin=Math.floor(ref.y);
		double zmin=Math.floor(ref.z);
		
		double xmax=xmin+1;
		double ymax=ymin+1;
		double zmax=zmin+1;
		double avgLength=0;
		if(xmin==ref.x&&ymin==ref.y&&zmin==ref.z){
			//double actual=trace(ref, false);
			//System.out.println("BROKE "+ref+","+dat.label);
			//dat.setCorrespodence(interpolateCoorespondence(ref));
			return false;
		}
		int plane=0;
		if(p.x<xmin){
			result=new Point3f(
					(float)xmin,
					(float)(ref.y+(p.y-ref.y)*(xmin-ref.x)/(p.x-ref.x)),
					(float)(ref.z+(p.z-ref.z)*(xmin-ref.x)/(p.x-ref.x)));
			dist=p.distance(result);
			plane=0;
		} else if(p.x>xmax){
			result=new Point3f(
					(float)xmax,
					(float)(ref.y+(p.y-ref.y)*(xmax-ref.x)/(p.x-ref.x)),
					(float)(ref.z+(p.z-ref.z)*(xmax-ref.x)/(p.x-ref.x)));
			dist=p.distance(result);
			plane=1;
		}
		if(p.y<ymin){
			pint=new Point3f(
					(float)(ref.x+(p.x-ref.x)*(ymin-ref.y)/(p.y-ref.y)),
					(float)(ymin),
					(float)(ref.z+(p.z-ref.z)*(ymin-ref.y)/(p.y-ref.y)));
			tmp=p.distance(pint);
			if(tmp<dist){
				result=pint;
				dist=tmp;
				plane=2;
			}
		} else if(p.y>ymax){
			pint=new Point3f(
					(float)(ref.x+(p.x-ref.x)*(ymax-ref.y)/(p.y-ref.y)),
					(float)(ymax),
					(float)(ref.z+(p.z-ref.z)*(ymax-ref.y)/(p.y-ref.y)));
			tmp=p.distance(pint);
			if(tmp<dist){
				result=pint;
				dist=tmp;
				plane=3;
			}
		}
		if(p.z<zmin){
			pint=new Point3f(
					(float)(ref.x+(p.x-ref.x)*(zmin-ref.z)/(p.z-ref.z)),
					(float)(ref.y+(p.y-ref.y)*(zmin-ref.z)/(p.z-ref.z)),
					(float)zmin);
			tmp=p.distance(pint);
			if(tmp<dist){
				result=pint;
				dist=tmp;
				plane=4;
			}
		} else if(p.z>zmax){
			pint=new Point3f(
					(float)(ref.x+(p.x-ref.x)*(zmax-ref.z)/(p.z-ref.z)),
					(float)(ref.y+(p.y-ref.y)*(zmax-ref.z)/(p.z-ref.z)),
					(float)zmax);
			tmp=p.distance(pint);
			if(tmp<dist){
				result=pint;
				dist=tmp;
				plane=5;
			}
		}
		if(result==null)return false;
		GridNode[] cd=new GridNode[4];
		cd[0]=grid.getNode((int)xmin+planePoint1[plane].x,(int)ymin+planePoint1[plane].y,(int)zmin+planePoint1[plane].z);
		cd[1]=grid.getNode((int)xmin+planePoint2[plane].x,(int)ymin+planePoint2[plane].y,(int)zmin+planePoint2[plane].z);
		cd[2]=grid.getNode((int)xmin+planePoint3[plane].x,(int)ymin+planePoint3[plane].y,(int)zmin+planePoint3[plane].z);
		cd[3]=grid.getNode((int)xmin+planePoint4[plane].x,(int)ymin+planePoint4[plane].y,(int)zmin+planePoint4[plane].z);
		int votes=0;
		boolean voters[]=new boolean[]{false,false,false,false};
		for(int i=0;i<4;i++){
			if(cd[i]!=null&&cd[i].getMarchingLabel()==SOLVED&&(cd[i].regionLabel==INSIDE||cd[i].regionLabel==JUST_INSIDE)){votes++;voters[i]=true;}
		}
		if(votes==4){
			avgLength=interpolateLength(result)+ref.distance(result);
			dat.setLength(len+avgLength);
			return true;
		} else {
			return false;
		}
	}
	

}
