package edu.jhu.ece.iacl.plugins.measure.statistics;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Arrays;

import edu.jhu.ece.iacl.algorithms.PrinceGroupAuthors;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.EmbeddedSurfaceIntersector;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.EmbeddedTriangle;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.SurfaceIntersector;
import edu.jhu.ece.iacl.jist.io.*;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.*;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;

public class MedicAlgorithmSurfaceStatistics extends ProcessingAlgorithm{
	
	private ParamSurface surfParam;
	private ParamDouble mean;
	private ParamDouble median;
	private ParamDouble stdev;
	private ParamDouble firstQuant;
	private ParamDouble thirdQuant;
	private ParamDouble maxVal;
	private ParamDouble minVal;
	private ParamInteger offset;
	private ParamInteger vertexCount;
	private ParamInteger faceCount;
	private ParamObject<double[][]> vertexData;
	private ParamString statFile;
	private	ParamString omit;
	private File resultFile;
	
	
	

	private static final String cvsversion = "$Revision: 1.7 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	
	protected void createInputParameters(ParamCollection inputParams) {
		inputParams.add(surfParam=new ParamSurface("Embedded Surface"));
		inputParams.add(offset=new ParamInteger("Offset",0,100000,0));
		inputParams.add(statFile = new ParamString("Statisitics File Name"));
		statFile.setMandatory(false);
		inputParams.add(omit = new ParamString("Truncating character", "_"));
		omit.setDescription("Omit the name of the file in statitic report after first occurence of this character");
		inputParams.setName("surf_stats");
		inputParams.setLabel("Surface Statistics");

		inputParams.setPackage("IACL");
		inputParams.setCategory("Measurement.Surface");

		AlgorithmInformation info = getAlgorithmInformation();
		info.add(PrinceGroupAuthors.blakeLucas);
		info.add(PrinceGroupAuthors.navidShiee);
		info.setAffiliation("Johns Hopkins University, Departments of Electrical and Computer Engineering");
		info.setDescription("Computes statistics on embedded vertex data and outputs a file with all vertex data along with statistics on a specified vertex component.");
		info.setLongDescription("This module can be used in combination with closest point distance to generate global statistics for a surface. The statistics are computed on triangle face statistics by taking the average over each face.");
		
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.ALPHA);
	}

	@Override
	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.add(vertexData=new ParamObject<double[][]>("Vertex Data",new ArrayDoubleReaderWriter()));
		outputParams.add(minVal=new ParamDouble("Min"));
		outputParams.add(maxVal=new ParamDouble("Max"));
		outputParams.add(mean=new ParamDouble("Mean"));
		outputParams.add(median=new ParamDouble("Median"));
		outputParams.add(firstQuant=new ParamDouble("First Quantile"));
		outputParams.add(thirdQuant=new ParamDouble("Third Quantile"));
		outputParams.add(stdev=new ParamDouble("Std. Dev."));
		outputParams.add(vertexCount=new ParamInteger("Vertex Count"));
		outputParams.add(faceCount=new ParamInteger("Face Count"));
	}
	private class SurfStats extends AbstractCalculation{
		public SurfStats(){
			super();
			setLabel("Surface Statistics");
		}
		public void solve(EmbeddedSurface surf,int off){

			double sum=0;
			double sqrs=0;
			double wsum=0;
			double w;
			double v1,v2,v3,v;
			double min=1E30;
			double max=-1E30;
			int indexCount=surf.getIndexCount();
			faceCount.setValue(indexCount/3);
			setTotalUnits(indexCount/3);
			for(int i=0;i<indexCount;i+=3){
				v1=surf.getVertexDataAtOffset(surf.getCoordinateIndex(i), off);
				v2=surf.getVertexDataAtOffset(surf.getCoordinateIndex(i+1), off);
				v3=surf.getVertexDataAtOffset(surf.getCoordinateIndex(i+2), off);
				v=(v1+v2+v3)*0.3333333;
				if(v>=35){
					continue;
				}
				if(!Double.isNaN(v)){
					min=Math.min(v, min);
					max=Math.max(v, max);
					w=surf.getFaceArea(i/3);
					sum+=v*w;
					sqrs+=v*v*w;
					wsum+=w;
				}
				incrementCompletedUnits();
			}
			int vertCount=surf.getVertexCount();
			vertexCount.setValue(vertCount);
			double[] dat=new double[vertCount];
			for(int i=0;i<vertCount;i++){
				dat[i]=surf.getVertexDataAtOffset(i, off);
			}
			Arrays.sort(dat);
			if(vertCount%2==0){
				median.setValue(0.5*(dat[vertCount/2]+dat[(vertCount)/2-1]));
			} else {
				median.setValue(dat[(vertCount-1)/2]);
			}
			if (vertCount%4==0){
				firstQuant.setValue(0.5*(dat[vertCount/4]+dat[vertCount/4-1]));
				thirdQuant.setValue(0.5*(dat[3*vertCount/4]+dat[3*vertCount/4-1]));
			}else if (vertCount%2==0){
				firstQuant.setValue(0.5*(dat[(vertCount+2)/4]+dat[(vertCount+2)/4-1]));
				thirdQuant.setValue(0.5*(dat[3*(vertCount+2)/4]+dat[3*(vertCount+2)/4-1]));
			}else if ((vertCount-1)%4==0){
				firstQuant.setValue(dat[(vertCount-1)/4]);
				thirdQuant.setValue(dat[3*(vertCount-1)/4]);
			}else{
				firstQuant.setValue(dat[(vertCount+1)/4]);
				thirdQuant.setValue(dat[3*(vertCount+1)/4]);
			}
			mean.setValue(sum/wsum);
			stdev.setValue(Math.sqrt((sqrs-sum*sum/wsum)/wsum));
			minVal.setValue(min);
			maxVal.setValue(max);
			writeStattoFile();
			markCompleted();
		}
	}
	@Override
	protected void execute(CalculationMonitor monitor) {
		EmbeddedSurface surf=surfParam.getSurface();
		resultFile = null;
		File destdir = null;
		if (statFile.getValue() == null){
			 try{
				destdir = new File(this.getOutputDirectory().getCanonicalFile()+File.separator+this.getAlgorithmName());
				if(!destdir.isDirectory()){
					(new File(destdir.getCanonicalPath())).mkdir();
				}
			}catch(IOException e){
				e.printStackTrace();
			}
			resultFile = new File(destdir+File.separator+"SurfaceStatistics.csv");
		}else
			resultFile = new File(statFile.getValue());
		SurfStats ss=new SurfStats();
		monitor.observe(ss);
		ss.solve(surf,offset.getInt());
		vertexData.setObject(surf.getVertexData());
		vertexData.setFileName(surf.getName());

	}
	
	private void writeStattoFile(){
				
		String id = surfParam.getSurface().getName();
		if (!omit.getValue().equals(""))
			id = id.substring(0,id.indexOf(omit.getValue()));
		String output = id + "," + minVal + "," +maxVal + ","+mean+ ","+median+ ","+stdev+ ","+firstQuant+","+thirdQuant+"\n";
		addStatisticsToFile(output);
	
	}

	private final void addStatisticsToFile(String output) {
		// check if the statistics file exists
		
		// open the file
		ArrayList<String> 	previous = loadStatisticsFile();
		// merge the output
		previous.add(output);
		// save the result
		writeStatisticsFile(previous);
	}
	
	
	
	private final ArrayList<String> loadStatisticsFile() {
		ArrayList<String> list = new ArrayList();
		try {
            System.out.println("reading previous statistic file: "+resultFile.getName());
            File f = new File(resultFile.getAbsolutePath());
            FileReader fr = new FileReader(f);
            BufferedReader br = new BufferedReader(fr);
            String line = br.readLine();
			
			// Exact corresponding template for first line ?
           	if (!line.startsWith("Subject,Min,Max,Mean,Median,StdDev,1stQuant,3rdQuant")) {
                System.out.println("not a proper CRUISE Surface statistics file");
                br.close();
                fr.close();
                return null;
            }
           	line = br.readLine();
           	while (line!=null) {
				list.add(line+"\n");
				line = br.readLine();
				//System.out.println(line);
			}
			br.close();
            fr.close();
        }
        catch (FileNotFoundException e) {
            System.out.println(e.getMessage());
        }
        catch (IOException e) {
            System.out.println(e.getMessage());
        } 
		catch (OutOfMemoryError e){
			System.out.println(e.getMessage());
		}
		catch (Exception e) {
			System.out.println(e.getMessage());
        }
		return list;	
	}

	private final void writeStatisticsFile(ArrayList<String> list) {
		try {
            File f = new File(resultFile.getAbsolutePath());
            FileWriter fw = new FileWriter(f);
            PrintWriter pw = new PrintWriter( fw );
			pw.write("Subject,Min,Max,Mean,Median,StdDev,1stQuant,3rdQuant\n");
            for (int n=0;n<list.size();n++) {
				pw.write(list.get(n));
				//System.out.print(list.get(n));
			}
			pw.close();
            fw.close();
       }
        catch (FileNotFoundException e) {
            System.out.println(e.getMessage());
        }
        catch (IOException e) {
            System.out.println(e.getMessage());
        } 
		catch (OutOfMemoryError e){
			System.out.println(e.getMessage());
		}
		catch (Exception e) {
			System.out.println(e.getMessage());
        }
	}
}
