package edu.jhu.ece.iacl.plugins.measure.statistics;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;

import edu.jhmi.rad.medic.algorithms.AlgorithmVolumeStatistics;
import edu.jhu.ece.iacl.jist.io.ArrayIntegerTxtReaderWriter;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamOption;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import gov.nih.mipav.model.structures.ModelImage;

/**
 * The "Output Statistics" file will contain the values of the requested statistics for the UNION
 *  of labels found in the "Segmentation" and "Reference" images (including zero).  If a label in the Reference is not found
 *	in the Segmentation, then appropriate default values will be returned.\n\n  Object groupings can be specified by giving a text file in which
 *	each line contains a number of space separated integers that specifies a group.  For example,
 *
 *	1 2 3
 *	1 2 3 4
 *
 *	specifies two groups, the first is the union of three labels and the second is a union of four.  Statistics computed
 * 	for groups will be appended as additional columns in the "Output Statistics" file.
 *	
 *  Finally, a confusion matrix can be computed.  The output text file will be a square matrix such that the i-j element is the count
 *  of voxels for which the reference image had label i and the segmentation image had label j.
 * 
 * @author Pilou Bazin
 * @author Navid Shiee
 * @author Aaron Carass
 * @author John Bogovic
 *
 */
public class MedicAlgorithmVolumeStatistic extends ProcessingAlgorithm {

	public ParamBoolean pvox, pvol, psurf, prvol, pec, pop, poc, pminint, pmaxint, pmeanint, pstdint,
	pdice, pjacc, ptpr, pfpr, pfnr, ptsr, pfsr, pcisir, pciris,  pavgsurfdist, pavgsurfdiff, pavgsqrsurfdist, phausdist;

	
	private ParamVolume srcImage;
	private ParamVolume intImage;
	private ParamVolume refImage;
	private ParamBoolean checkVoxels;
	private ParamBoolean checkVolume;
	private ParamBoolean checkSurface;
	private ParamBoolean checkEuler;
	private ParamBoolean checkParts;
	private ParamBoolean checkCavities ;
	private ParamBoolean checkSurfaceRipples;
	private ParamBoolean checkMinIntensity;
	private ParamBoolean checkMaxIntensity;
	private ParamBoolean checkMeanIntensity;
	private ParamBoolean checkStdevIntensity;
	private ParamBoolean checkSumIntensity; 
	private ParamBoolean checkMedianIntensity;
	private ParamBoolean checkDice;
	private ParamBoolean checkJaccard;
	private ParamBoolean checkTruePositive;
	private ParamBoolean checkFalsePositive;
	private ParamBoolean checkFalseNegative;
	private ParamBoolean checkTrueSegmented;
	private ParamBoolean checkFalseSegmented;
	private ParamBoolean checkMountford;
	private ParamBoolean checkAinB;
	private ParamBoolean checkBinA;
	private ParamBoolean checkVolumeDiff;
	private ParamBoolean checkAvSurfDist;
	private ParamBoolean checkAvSurfDiff;
	private ParamBoolean checkAvSqSurfDist;
	private ParamBoolean checkHausdorff;
	private ParamBoolean checkConfusionMatrix;
	
	private ArrayList<String> needRefStats;
	private ArrayList<String> needIntStats;
	
	public ParamOption pconn;
	
	public ParamFile specOutputFile;
	public ParamFile groupFile;
	
	public ParamFile outputFile;
	public ParamFile cmFile;
	
	
	private static final ArrayIntegerTxtReaderWriter intArrayRW = ArrayIntegerTxtReaderWriter.getInstance();
	
	private static final String cvsversion = "$Revision: 1.3 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "Computes overlap measures between a segmentation and a reference and statistics of " +
			"an intensity image over labels in a segmentation";
	
	private static final String longDescription = "The \"Output Statistics\" file will contain the values of the requested statistics for the UNION" +
			"of labels found in the \"Segmentation\" and \"Reference\" images (including zero).  If a label in the Reference is not found" +
			"in the Segmentation, then appropriate default values will be returned.\n\n  Object groupings can be specified by giving a text file in which" +
			"each line contains a number of space separated integers that specifies a group.  For example,\n\n" +
			"1 2 3\n1 2 3 4\n\n specifies two groups, the first is the union of three labels and the second is a union of four.  Statistics computed" +
			"for groups will be appended as additional columns in the \"Output Statistics\" file.\n\n" +
			"Finally, a confusion matrix can be computed.  The output text file will be a square matrix such that the i-j element is the count" +
			"of voxels for which the reference image had label i and the segmentation image had label j ";

	protected void createInputParameters(ParamCollection inputParams) {
		
		inputParams.add(srcImage = new ParamVolume("Segmentation Image"));
		
		inputParams.add(intImage = new ParamVolume("Intensity Image"));
		intImage.setMandatory(false);
		
		inputParams.add(refImage = new ParamVolume("Reference Image"));
		refImage.setMandatory(false);
		
		inputParams.add(specOutputFile = new ParamFile("File to write/append"));
		specOutputFile.setMandatory(false);
		
		inputParams.add(groupFile = new ParamFile("Label Grouping"));
		groupFile.setMandatory(false);
		
		inputParams.add(pconn = new ParamOption("Connectivity",new String[]{"6/26","6/18","18/6","26/6"}));
		
		inputParams.add(checkVoxels = new ParamBoolean("Voxels",false));
		inputParams.add(checkVolume = new ParamBoolean("Volume",false));
		inputParams.add(checkSurface = new ParamBoolean("Surface",false));
		inputParams.add(checkEuler = new ParamBoolean("Euler_characteristic",false));
		inputParams.add(checkParts = new ParamBoolean("Object_parts",false));
		inputParams.add(checkCavities = new ParamBoolean("Object_cavities",false));
		inputParams.add(checkSurfaceRipples = new ParamBoolean("Surface_ripples",false));
		inputParams.add(checkMinIntensity = new ParamBoolean("Min_intensity",false));
		inputParams.add(checkMaxIntensity = new ParamBoolean("Max_intensity",false));
		inputParams.add(checkMeanIntensity = new ParamBoolean("Mean_intensity",false));
		inputParams.add(checkStdevIntensity = new ParamBoolean("Stdev_intensity",false));
		inputParams.add(checkMedianIntensity = new ParamBoolean("Median_intensity",false));
		inputParams.add(checkSumIntensity = new ParamBoolean("Sum_intensity",false));
		inputParams.add(checkDice = new ParamBoolean("Dice_overlap",false));
		inputParams.add(checkJaccard = new ParamBoolean("Jaccard_overlap",false));
		inputParams.add(checkTruePositive = new ParamBoolean("True_positive_ratio",false));
		inputParams.add(checkFalsePositive = new ParamBoolean("False_positive_ratio",false));
		inputParams.add(checkFalseNegative = new ParamBoolean("False_negative_ratio",false));
		inputParams.add(checkTrueSegmented = new ParamBoolean("True_segmented_ratio",false));
		inputParams.add(checkFalseSegmented = new ParamBoolean("False_segmented_ratio",false));
		inputParams.add(checkMountford = new ParamBoolean("Mountford_overlap",false));
		inputParams.add(checkAinB = new ParamBoolean("Containment_index(seg_in_ref)",false));
		inputParams.add(checkBinA = new ParamBoolean("Containment_index(ref_in_seg)",false));
		inputParams.add(checkVolumeDiff = new ParamBoolean("Volume_difference",false));
		inputParams.add(checkAvSurfDist= new ParamBoolean("Average_surface_distance",false));
		inputParams.add(checkAvSurfDiff= new ParamBoolean("Average_surface_difference",false));
		inputParams.add(checkAvSqSurfDist= new ParamBoolean("Average_squared_surface_distance",false));
		inputParams.add(checkHausdorff= new ParamBoolean("Hausdorff_distance",false));
		inputParams.add(checkConfusionMatrix= new ParamBoolean("Confusion_Matrix",false));
		
		inputParams.setName("Volume_Statistics");
		inputParams.setLabel("Volume_Statistics");

		inputParams.setPackage("IACL");
		inputParams.setCategory("Measurement.Statistics");
		
		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("http://www.iacl.ece.jhu.edu/");
		info.setAffiliation("Johns Hopkins University, Departments of Electrical and Computer Engineering");
		info.setDescription(shortDescription);
		info.setLongDescription(shortDescription + "\n\n" + longDescription);	
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.BETA);
	}

	@Override
	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.add(outputFile = new ParamFile("Output Statistics"));
		outputFile.setMandatory(true);
		
		outputParams.add(cmFile = new ParamFile("Confusion Matrix File"));
		cmFile.setMandatory(false);
	}

	@Override
	protected void execute(CalculationMonitor arg0)
			throws AlgorithmRuntimeException {
		File destdir = null;
		try{
			destdir = new File(this.getOutputDirectory().getCanonicalFile()+File.separator+this.getAlgorithmName());
			//			File destdir = new File(outputdir.getCanonicalFile().toString());
			if(!destdir.isDirectory()){
				(new File(destdir.getCanonicalPath())).mkdir();
			}
		}catch(IOException e){
			e.printStackTrace();
		}
		
		ModelImage segimg = srcImage.getImageData().getModelImageCopy();
		ModelImage intimg = null;
		ModelImage refimg = null;
		String segname = srcImage.getImageData().getName();
		
		
		
		// AlgorithmVolumeStatistics is unhappy if any of the volume inputs are null
		// if no inputs are given for the intensity and reference volumes, set them equal to the seg
		// (they won't be used)
		if(intImage.getValue()!=null){
			intimg = intImage.getImageData().getModelImageCopy();
		}else{
			intimg = segimg;
		}
		if(refImage.getValue()!=null){
			refimg = refImage.getImageData().getModelImageCopy();
		}else{
			refimg = segimg;
		}
		JistLogger.logOutput(1, "Segmentation image: " + segimg);
		JistLogger.logOutput(1, "Intensity image: " + intimg);
		JistLogger.logOutput(1, "Reference image: " + refimg);
		
		
		int[][] groups = null;
		if(groupFile.getValue()!=null){
			groups = intArrayRW.read(groupFile.getValue());
		}
		
		ArrayList<String> statistics = new ArrayList<String>();
		
		needRefStats = new ArrayList<String>();
		needIntStats = new ArrayList<String>();
		
		if (checkVoxels.getValue()) statistics.add(checkVoxels.getName()); 
		if (checkVolume.getValue()) statistics.add(checkVolume.getName()); 
		if (checkSurface.getValue()) statistics.add(checkSurface.getName());
		
		if (checkEuler.getValue()) statistics.add(checkEuler.getName());
		if (checkParts.getValue()) statistics.add(checkParts.getName());
		if (checkCavities.getValue()) statistics.add(checkCavities.getName());
		
		if (checkSurfaceRipples.getValue()) statistics.add(checkSurfaceRipples.getName());
		
		if (checkMeanIntensity.getValue()){ statistics.add(checkMeanIntensity.getName()); 		needIntStats.add(checkMeanIntensity.getName());}
		if (checkMinIntensity.getValue()){ statistics.add(checkMinIntensity.getName()); 		needIntStats.add(checkMinIntensity.getName());}
		if (checkMaxIntensity.getValue()){ statistics.add(checkMaxIntensity.getName()); 		needIntStats.add(checkMaxIntensity.getName());}
		if (checkStdevIntensity.getValue()){ statistics.add(checkStdevIntensity.getName()); 	needIntStats.add(checkStdevIntensity.getName());}
		if (checkSumIntensity.getValue()){ statistics.add(checkSumIntensity.getName()); 		needIntStats.add(checkSumIntensity.getName());}
		if (checkMedianIntensity.getValue()){ statistics.add(checkMedianIntensity.getName()); 	needIntStats.add(checkMedianIntensity.getName());}
		
		if (checkDice.getValue()){ 			statistics.add(checkDice.getName());			needRefStats.add(checkDice.getName());  }
		if (checkJaccard.getValue()){ 		statistics.add(checkJaccard.getName());			needRefStats.add(checkJaccard.getName());  }
		if (checkMountford.getValue()){ 	statistics.add(checkMountford.getName());		needRefStats.add(checkMountford.getName());  }
		if (checkTruePositive.getValue()){ 	statistics.add(checkTruePositive.getName());	needRefStats.add(checkTruePositive.getName());  }
		if (checkFalsePositive.getValue()){ statistics.add(checkFalsePositive.getName());	needRefStats.add(checkFalsePositive.getName());  }
		if (checkFalseNegative.getValue()){ statistics.add(checkFalseNegative.getName());	needRefStats.add(checkFalseNegative.getName());  }
		if (checkTrueSegmented.getValue()){ statistics.add(checkTrueSegmented.getName());	needRefStats.add(checkTrueSegmented.getName());  }
		if (checkFalseSegmented.getValue()){statistics.add(checkFalseSegmented.getName());	needRefStats.add(checkFalseSegmented.getName());  }
		if (checkVolumeDiff.getValue()){ 	statistics.add(checkVolumeDiff.getName());		needRefStats.add(checkVolumeDiff.getName());  }
		if (checkAinB.getValue()){ 			statistics.add(checkAinB.getName());			needRefStats.add(checkAinB.getName());  }
		if (checkBinA.getValue()){ 			statistics.add(checkBinA.getName());			needRefStats.add(checkBinA.getName());  }
		if (checkAvSurfDist.getValue()){ 	statistics.add(checkAvSurfDist.getName());		needRefStats.add(checkAvSurfDist.getName());  }
		if (checkAvSurfDiff.getValue()){ 	statistics.add(checkAvSurfDiff.getName());		needRefStats.add(checkAvSurfDiff.getName());  }
		if (checkAvSqSurfDist.getValue()){ 	statistics.add(checkAvSqSurfDist.getName());	needRefStats.add(checkAvSqSurfDist.getName());  }
		if (checkHausdorff.getValue()){ 	statistics.add(checkHausdorff.getName());		needRefStats.add(checkHausdorff.getName());  }
		if (checkConfusionMatrix.getValue()){ 	statistics.add(checkConfusionMatrix.getName());	needRefStats.add(checkConfusionMatrix.getName());  }
		
		boolean areInputsValid = validateInputs();
		if(!areInputsValid){
			System.err.println("INVALID INPUTS");
			System.out.println("INVALID INPUTS");
//			this.set
			return;
		}

		System.out.println(statistics);
		File outfile=null;
		if(specOutputFile.getValue()!=null){
			outfile=specOutputFile.getValue();
		}else{
			outfile = new File(destdir+File.separator+segname+"_VolStats.txt");
		}
		String	delimSymbol = " ";
		
		AlgorithmVolumeStatistics vsalg;
		vsalg = new AlgorithmVolumeStatistics(segimg, intimg, refimg, statistics, false, pconn.getValue(), true, outfile.getAbsolutePath(), delimSymbol);
		vsalg.setGroups(groups);
		
		vsalg.run();
		
		if (checkConfusionMatrix.getValue()){ 
			File cmout = new File(destdir+File.separator+segname+"_ConfusionMtx.txt");
			File cmwritten = ArrayIntegerTxtReaderWriter.getInstance().write(vsalg.getConfusionMatrix(), cmout);
			cmFile.setValue(cmwritten);
		}
		
		outputFile.setValue(outfile);
		
	}
	
	
	
	// ensure that the reference and intensity volumes are given when 
	// statistics that need them are asked for
	private boolean validateInputs(){
		
		boolean needRef = false;
		boolean needInt = false;
		
		if (checkDice.getValue()) needRef = true;
		if (checkJaccard.getValue()) needRef = true;
		if (checkMountford.getValue()) needRef = true;
		if (checkTruePositive.getValue()) needRef = true;
		if (checkFalsePositive.getValue()) needRef = true;
		if (checkFalseNegative.getValue()) needRef = true;
		if (checkTrueSegmented.getValue()) needRef = true;
		if (checkFalseSegmented.getValue()) needRef = true;
		if (checkVolumeDiff.getValue()) needRef = true;
		if (checkAinB.getValue()) needRef = true;
		if (checkBinA.getValue()) needRef = true;
		if (checkAvSurfDist.getValue()) needRef = true;
		if (checkAvSurfDiff.getValue()) needRef = true;
		if (checkAvSqSurfDist.getValue()) needRef = true;
		if (checkHausdorff.getValue()) needRef = true;
		if (checkConfusionMatrix.getValue()) needRef = true;
		
		if (checkMeanIntensity.getValue()) needInt = true;
		if (checkMinIntensity.getValue()) needInt = true;
		if (checkMaxIntensity.getValue()) needInt = true;
		if (checkStdevIntensity.getValue()) needInt = true;
		if (checkSumIntensity.getValue()) needInt = true;
		if (checkMedianIntensity.getValue()) needInt = true;
		
		if(needRef && refImage.getImageData()==null){
			System.err.println("THE INPUTS:\n" + needRefStats);
			System.err.println("REQUIRE A REFERENCE IMAGE. ");
			System.out.println("THE INPUTS:\n" + needRefStats);
			System.out.println("REQUIRE A REFERENCE IMAGE. ");
			return false;
		}
		
		if(needInt && intImage.getImageData()==null){
			System.err.println("THE INPUTS:\n" + needIntStats);
			System.err.println("REQUIRE A INTENSITY IMAGE. ");
			System.out.println("THE INPUTS:\n" + needIntStats);
			System.out.println("REQUIRE A INTENSITY IMAGE. ");
			return false;
		}
			
		return true;
			
		
	}

}
