package edu.jhu.ece.iacl.plugins.registration;

import java.io.File;
import java.util.List;

import javax.vecmath.Point3f;
import javax.vecmath.Point3i;

import Jama.Matrix;

import edu.jhu.ece.iacl.algorithms.gvf.Interpolate3dBSpline;

import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamMatrix;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamPointFloat;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamPointInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolumeCollection;

import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataMipav;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataMipavWrapper;
import edu.jhu.ece.iacl.jist.structures.image.ImageHeader;


import gov.nih.mipav.model.structures.ModelImage;


/**
 * @author Philippe Thevenaz
 * @author Aaron Carass
 *
 * Based in LARGE part on affine transformation code distributed by
 * Philippe Thevenaz, Ecole Polytechnique Federale de Lausanne.
 *
 */
public class MedicAlgorithmTransformVolume2 extends ProcessingAlgorithm {
	public ParamVolumeCollection inputVolumeCollection;
	public ParamVolumeCollection outputVolumeCollection;
	public ParamMatrix matrix;
	public ParamPointInteger dims;
	public ParamPointFloat res;
	public ParamVolume matchToMe;
	public ParamInteger paramSplineDegree;

	private ImageHeader hdr, newHdr;
	private String name;

	private static final String cvsversion = "$Revision: 1.2 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");

	private static final String shortDescription = "Applies a given transformation matrix to a volumetric image using the specified interpolation method.\nThe interpolation methods are \"Nearest Neighbours\", \"Linear Interpolation\", \"Quadratic\",  and the B-Spline methods: \"Cubic\", \"Quartic\", \"Quintic\", \"Sextic\", and \"Septic\".\nBased in LARGE part on code distributed by Philippe Thevenaz.";
	private static final String longDescription = "";

	protected void createInputParameters(ParamCollection inputParams) {
		inputParams.add(inputVolumeCollection = new ParamVolumeCollection("Volumes"));
		inputParams.add(paramSplineDegree = new ParamInteger("B-Spline Degree", 1, 5, 3));
		inputParams.add(matrix = new ParamMatrix("Transformation Matrix", 4, 4));
		for (int i = 0; i < 4; i++){
			for (int j = 0; j < 4; j++){
				matrix.setValue(i, j, 0);
			}
			matrix.setValue(i, i, 1);
		}
		
		inputParams.add(res = new ParamPointFloat("Final Resolutions", new Point3f(1, 1, 1)));
		inputParams.add(dims = new ParamPointInteger("Final Dimensions", new Point3i(256, 256, 198)));
		inputParams.add(matchToMe = new ParamVolume("Match Final Dimensions & Resolution to Volume", null,-1,-1,-1,-1));
		matchToMe.setLoadAndSaveOnValidate(false);
		matchToMe.setMandatory(false);


		inputParams.setPackage("IACL");
		inputParams.setCategory("Registration");
		inputParams.setLabel("Transform Volume 2");
		inputParams.setName("Transform_Volume_2");


		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("http://www.iacl.ece.jhu.edu/");
		info.add(new AlgorithmAuthor("Aaron Carass", "aaron_carass@jhu.edu", "http://www.iacl.ece.jhu.edu/"));
		info.add(new AlgorithmAuthor("Philippe Thevenaz", "", ""));
		info.setDescription(shortDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.Release);
	}


	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.add(outputVolumeCollection = new ParamVolumeCollection("Transformed Volumes"));
		outputVolumeCollection.setLoadAndSaveOnValidate(false);
	}


	protected void execute(CalculationMonitor monitor) throws AlgorithmRuntimeException {
		ExecuteWrapper wrapper = new ExecuteWrapper();
		monitor.observe(wrapper);
		wrapper.execute(this);
	}


	protected class ExecuteWrapper extends AbstractCalculation {
		public void execute(ProcessingAlgorithm alg){
			List<File> filelist = inputVolumeCollection.getFileList();
			int N = filelist.size();
			int n = 0, i = 0, j = 0, k = 0;
			//float rxOld = -1, ryOld = -1, rzOld = -1;
			float[] rOld = new float[3];
			float[] rNew = new float[3];
			//int nxOld = -1, nyOld = -1, nzOld = -1;
			//int nxNew = -1, nyNew = -1, nzNew = -1;
			int[] nOld = new int[3]; 
			int[] nNew = new int[3];
			
			float[] oldO, newO;
			int splineDegree = paramSplineDegree.getInt();


			Matrix m = matrix.getValue();

			if(matchToMe.getImageData() != null){
				System.out.format("Reading Resolution and Dimension from Volume to match.");
				ImageData matchid =  matchToMe.getImageData();
				ModelImage matchmi = matchid.getModelImageCopy();
				nNew = matchmi.getExtents();
				rNew =  matchmi.getResolutions(0);
				
				
				matchmi.disposeLocal();
				matchmi = null;
				matchid.dispose();
				matchid = null;
			}else{
				rNew[0] = res.getValue().x;
				rNew[1] = res.getValue().y;
				rNew[2] = res.getValue().z;				
				nNew[0] = dims.getValue().x;
				nNew[1] = dims.getValue().y;
				nNew[2] = dims.getValue().z;
			}
			

			for (n = 0; n < N; n++){
				ImageData v = inputVolumeCollection.getParamVolume(n).getImageData();
				hdr = v.getHeader();
				newHdr = hdr.clone();
				name = v.getName();

				float inputMat[][][] = (new ImageDataFloat(v)).toArray3d();
				double tmpDouble;


				/*
				 * Old and New Origin.
				 */
				oldO = hdr.getOrigin();
				newO = new float[3];


				nOld[0] = v.getRows();
				nOld[1] = v.getCols();
				nOld[2] = v.getSlices();
				v.dispose();


				rOld[0] = hdr.getDimResolutions()[0];
				rOld[1] = hdr.getDimResolutions()[1];
				rOld[2] = hdr.getDimResolutions()[2];

			
				System.out.format("\n\nm =\n");
				for (int ii = 0; ii < 4 ; ii++) {
					System.out.format("%2.2g %2.2g %2.2g %2.2g\n", m.get(ii,0),m.get(ii,1),m.get(ii,2),m.get(ii,3));
				}
								
				m = m.inverse();//need to use inverse, because we want to pull back to target space.
				
				System.out.format("\n\nm_inverse =\n");
				for (int ii = 0; ii < 4 ; ii++) {
					System.out.format("%2.2g %2.2g %2.2g %2.2g\n", m.get(ii,0),m.get(ii,1),m.get(ii,2),m.get(ii,3));
				}
				
				for (i = 0; i < 3 ; i++) {
					newO[i] = oldO[0] * ((float) m.get(0, i)) + oldO[1] * ((float) m.get(1, i)) + oldO[2] * ((float) m.get(2, i));
				}


				newHdr.setDimResolutions(rNew);
				newHdr.setOrigin(newO);


				System.out.format("\nOld Origin: %g %g %g\n", oldO[0], oldO[1], oldO[2]);
				System.out.format("New Origin: %g %g %g\n", newO[0], newO[1], newO[2]);

				System.out.format("\nOld Dimension: %d %d %d\n", nOld[0], nOld[1], nOld[2]);
				System.out.format("New Dimension: %d %d %d\n\n", nNew[0], nNew[1], nNew[2]);

				System.out.format("\nOld Resolution: %g %g %g\n", rOld[0], rOld[1], rOld[2]);
				System.out.format("New Resolution: %g %g %g\n\n", rNew[0], rNew[1], rNew[2]);

				

				double workingMat[][][] = new double[nOld[0]][nOld[1]][nOld[2]];
				float outputMat[][][] = new float[nNew[0]][nNew[1]][nNew[2]];
				float maxFloat = inputMat[0][0][0];
				float minFloat = inputMat[0][0][0];
				double x, y, z, t;
				int Error = 0;


					
				for (i = 0; i < nOld[0]; i++){
					for (j = 0; j < nOld[1]; j++){
						for (k = 0; k < nOld[2]; k++){
							
							workingMat[i][j][k] = inputMat[i][j][k];

							if (maxFloat < inputMat[i][j][k]) {
								maxFloat = inputMat[i][j][k];
							}


							if (minFloat > inputMat[i][j][k]) {
								minFloat = inputMat[i][j][k];
							}
						}
					}
				}


				inputMat = null;

				Interpolate3dBSpline interpolater = new Interpolate3dBSpline();


				Error = interpolater.SamplesToCoefficients(workingMat, nOld[0], nOld[1], nOld[2], splineDegree);
				if (Error == 1) {
					System.out.print("\n\nChange of basis failed\n\n");
				}

				
				float i_mm,j_mm,k_mm;//index locations in mm;
				
				for (i = 0; i < nNew[0]; i++){
					for (j = 0; j < nNew[1]; j++){
							for (k = 0; k < nNew[2]; k++){
							
							//convert target space to physical space
							i_mm = i * rNew[0];	
							j_mm = j * rNew[1];
							k_mm = k * rNew[2];
								
							//transform in physical space and move to subject space
							x = (m.get(0, 0)*i_mm + m.get(0, 1)*j_mm + m.get(0, 2)*k_mm + m.get(0, 3))/rOld[0];
							y = (m.get(1, 0)*i_mm + m.get(1, 1)*j_mm + m.get(1, 2)*k_mm + m.get(1, 3))/rOld[1];
							z = (m.get(2, 0)*i_mm + m.get(2, 1)*j_mm + m.get(2, 2)*k_mm + m.get(2, 3))/rOld[2];
							t = (m.get(3, 0)*i_mm + m.get(3, 1)*j_mm + m.get(3, 2)*k_mm + m.get(3, 3));

							x /= t;
							y /= t;
							z /= t;


							//interpolate value in subject space (out of bounds is set as zero)
							if ((x < -0.5 || (nOld[0] - 0.5) < x)
								|| (y < -0.5 || (nOld[1] - 0.5) < y)
								|| (z < -0.5 || (nOld[2] - 0.5) < z)){
								tmpDouble = 0.0;
							} else {
								tmpDouble = interpolater.interpolateBSpline(workingMat, nOld[0], nOld[1], nOld[2], x, y, z, splineDegree);
							}

							/**
							 * Why was this in the old code ?
							 *
							 * tmpDouble = Math.floor(tmpDouble + 0.5);
							 */


							//set interpolated value
							outputMat[i][j][k] = (float) tmpDouble;


							if (outputMat[i][j][k] > maxFloat) {
								outputMat[i][j][k] = maxFloat;
							}


							if (outputMat[i][j][k] < minFloat) {
								outputMat[i][j][k] = minFloat;
							}
						}
					}
				}


				ImageDataFloat imgOut = new ImageDataFloat(outputMat);
				imgOut.setHeader(newHdr);
				imgOut.setName(name + "_interpolated");
				outputVolumeCollection.add(imgOut);
				outputVolumeCollection.writeAndFreeNow(alg);
			}

			/*
			 * For other modules to acces this one, these lines need to
			 * go.
			 *
			inputVolumeCollection.dispose();
			inputVolumeCollection = null;
			*/
		}
	}
}
