/*
 * 
 */
package edu.jhu.ece.iacl.plugins.utilities.volume;


import Jama.Matrix;
import edu.jhmi.rad.medic.libraries.Morphology;
import edu.jhmi.rad.medic.libraries.ObjectProcessing;
import edu.jhu.ece.iacl.algorithms.volume.IsotropicResample;
import edu.jhu.ece.iacl.algorithms.volume.IsotropicResample.InterpolationMethod;
import edu.jhu.ece.iacl.jist.io.ImageDataReaderWriter;
import edu.jhu.ece.iacl.jist.io.ModelImageReaderWriter;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.jist.pipeline.parameter.*;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataInt;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataMipav;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataUByte;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.jist.structures.image.ImageHeader;
import edu.jhu.ece.iacl.jist.structures.image.ImageHeader.AxisOrientation;
import edu.jhu.ece.iacl.jist.structures.image.VoxelType;
import gov.nih.mipav.model.algorithms.AlgorithmCostFunctions;
import gov.nih.mipav.model.structures.ModelImage;
import gov.nih.mipav.model.structures.ModelStorageBase;
import gov.nih.mipav.view.ViewOpenFileUI;
import gov.nih.mipav.view.ViewUserInterface;
import gov.nih.mipav.view.dialogs.JDialogBase;


/*
 * @author Aaron Carass <aaron_carass@jhu.edu>
 */
public class MedicAlgorithmSimpleNoiseEstimation extends ProcessingAlgorithm {
	private ParamVolume origVol;
	private ParamVolume resultVol;

	private static final String[] methods = { "Otsu", "Gaussian Variation" };
	private ParamOption comboMethod;

	// Boolean to decide if we close after applying the threshold.
	private ParamBoolean morphClosing;
	private ParamInteger morphClosingSizeParam;

	// Boolean to decide if we do a hole filling.
	private ParamBoolean holeFilling;

	// Fraction of histogram to use in computing the threshold.
	private ParamDouble histFractionParam;


	private static final String revnum = "$Revision: 1.5 $".replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = 
		"Simple Noise Estimation tools.\n"
		+ "################################################\n"
		+ "Algorithm Version: "
		+ revnum
		+ "\n";
	private static final String longDescription =
		"Implementation of Otsu method:\n"
		+ "N. Otsu, \"A Threshold Selection Method from Gray-Level Histograms\", IEEE Trans. Systems, Man, and Cybernetics, 9(1):62-66, January 1979.\n";


	protected void createInputParameters(ParamCollection inputParams) {
		inputParams.add(origVol = new ParamVolume("Volume"));

		inputParams.add(comboMethod = new ParamOption("Methods", methods));
		comboMethod.setValue(0);

		inputParams.add(morphClosing = new ParamBoolean("Morphologically close the image", true));
		inputParams.add(morphClosingSizeParam = new ParamInteger("Morphologically closing Radius", 1, 10, 1));

		inputParams.add(holeFilling = new ParamBoolean("Fill holes in the foreground", true));
		holeFilling.setDescription("Boolean to decide if we do a hole filling.");
		inputParams.add(histFractionParam = new ParamDouble("Fraction of Histogram to use", 0.00, 1.00, 0.45));
		histFractionParam.setDescription("Fraction of histogram to use in computing the threshold.");


		inputParams.setPackage("IACL");
		inputParams.setCategory("Utilities.Volume");
		inputParams.setLabel("Simple Noise Estimation");
		inputParams.setName("Simple_Noise_Estimation");


		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("http://www.iacl.ece.jhu.edu/");
		info.add(new AlgorithmAuthor("Aaron Carass", "aaron_carass@jhu.edu", "http://www.iacl.ece.jhu.edu/"));
		info.setAffiliation("Johns Hopkins University, Departments of Electrical and Computer Engineering");
		info.setDescription(shortDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.RC);
	}


	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.add(resultVol = new ParamVolume("Noise Removed Result"));
	}


	protected void execute(CalculationMonitor monitor)
			throws AlgorithmRuntimeException {
		SimpleNoiseEstimationWrapper algorithm = new SimpleNoiseEstimationWrapper();
		monitor.observe(algorithm);
		algorithm.execute();
	}


	protected class SimpleNoiseEstimationWrapper extends AbstractCalculation {
		public SimpleNoiseEstimationWrapper() {
		}


		public void execute() {
			System.out.println("\n\nSimpleNoiseEstimator");
			ImageDataMipav image =new ImageDataMipav(origVol.getImageData());
			String name = image.getName();
			name = name.replaceAll("\\.", "_");

			int XN = image.getRows();
			int YN = image.getCols();
			int ZN = image.getSlices();
			int x = 0, y = 0, z = 0;
			int value_i = 0;
			int threshold_level = -1;

			int[][][] int_image = new int[XN][YN][ZN];

			for (x = 0; x < XN; x++) {
				System.out.println(((x + 1.0) * 100.0)/XN + "% done");
				for (y = 0; y < YN; y++) {
					for (z = 0; z < ZN; z++) {
						value_i = image.getInt(x, y, z);
						if (image.getType() == VoxelType.UBYTE) {
							if (value_i < 0) {
								value_i += 256;
							}
						}
						int_image[x][y][z] = value_i;
					}
				}
			}


			ImageDataInt intVol = new ImageDataInt(int_image);


			switch (comboMethod.getIndex()) {
				case 0:
					threshold_level = otsuThreshold(intVol, 0, histFractionParam.getDouble());
					break;
				case 1:
					threshold_level = gaussianThreshold(intVol, 0, histFractionParam.getDouble());
					break;
				default:
					System.out.println("\n\nERROR\n\n");
					break;
			}


			System.out.println("threshold_level = " + threshold_level);


			boolean[][][] boolean_tmp = new boolean[XN][YN][ZN];


			for (x = 0; x < XN; x++)
				for (y = 0; y < YN; y++)
					for (z = 0; z < ZN; z++) {
						value_i = int_image[x][y][z];
						if (value_i > threshold_level) {
							boolean_tmp[x][y][z] = true;
						} else {
							boolean_tmp[x][y][z] = false;
						}
					}


			if (morphClosing.getValue()) {	
				System.out.println("Morphological Closing");
				boolean_tmp = Morphology.erodeObject(boolean_tmp, XN, YN, ZN, morphClosingSizeParam.getInt(), morphClosingSizeParam.getInt(), morphClosingSizeParam.getInt());
				boolean_tmp = Morphology.dilateObject(boolean_tmp, XN, YN, ZN, morphClosingSizeParam.getInt(), morphClosingSizeParam.getInt(), morphClosingSizeParam.getInt());
				// boolean_tmp = ObjectProcessing.largest6Object(boolean_tmp, XN, YN, ZN);
			} else {
				System.out.println("SKIPPING Morphological Closing");
			}


			if (holeFilling.getValue()) {	
				System.out.println("Hole Filling");
				boolean_tmp = ObjectProcessing.removeHoles6(boolean_tmp, XN, YN, ZN);
			} else {
				System.out.println("SKIPPING Hole Filling");
			}


			for (x = 0; x < XN; x++)
				for (y = 0; y < YN; y++)
					for (z = 0; z < ZN; z++) {
						if (!boolean_tmp[x][y][z]) {
							image.set(x, y, z, 0.0);
						}
					}


			image.setName(name + "_sne");
			resultVol.setValue(image);
		}


		final public int otsuThreshold(ImageDataInt input, int lb, double fraction) {
			System.out.println("otsuThreshold started");
			int magic_number = -1; 
			int XN = input.getRows();
			int YN = input.getCols();
			int ZN = input.getSlices();

			int x = 0, y = 0, z = 0, n = 0, m = 0;
			int max_int = -1000, min_int = 1000;
			int bins = -1;
			int value_i = 0;
			int total = 0, counter = 0;
			int h_max = 0;

			int k = 0;

			double bin_ratio = 1.0, i_tmp = 0.0;

			double omega_b_cur = 0.0, omega_b_max = 0.0;
			double mu_t = 0.0, mu_0 = 0.0, mu_1 = 0.0;
			double weight_0 = 0.0, weight_1 = 0.0;


			for (x = 0; x < XN; x++) {
				System.out.println(((x + 1.0) * 100.0)/XN + "% done");
				for (y = 0; y < YN; y++) {
					for (z = 0; z < ZN; z++) {
						value_i = input.getInt(x, y, z); 

						if (value_i <= lb) {
							continue;
						}

						if (value_i > max_int) {
							max_int = value_i;
						} 

						if (value_i < min_int) {
							min_int = value_i;
						} 
					}
				}
			}

			bins = max_int - min_int + 1;

			System.out.println("Number of bins: " + bins);

			if (bins > 50000) {
				bins = 50000;
				System.out.println("Resetting bins: " + bins);
				bin_ratio = (max_int - min_int + 1) / bins;
			}


			int[] h = new int[bins];
			double[] p = new double[bins];

			for (n = 0; n < bins; n++) {
				h[n] = 0;
				p[n] = 0;
			}


			/*
			 * Build a histogram.
			 */
			for (x = 0; x < XN; x++)
				for (y = 0; y < YN; y++)
					for (z = 0; z < ZN; z++) {
						value_i = input.getInt(x, y, z);

						if (value_i <= lb) {
							continue;
						}

						i_tmp = value_i/bin_ratio;
						value_i = (int) i_tmp;

						if (value_i >= bins) {
							value_i = bins - 1;
						}

						h[value_i]++;
						total++;
					}


			/*
			 * Trim the histogram based on the required 
			 * fraction of the histogram area.
			 */
			for (n = 0; n < bins; n++) {
				counter += h[n];
				if (counter >= fraction * total) {
					h[n] = 0;
				} else {
					h_max = n;
				}
			}


			/*
			 * Convert histogram to probability.
			 */
			for (n = 0; n < bins; n++) {
				p[n] = ((double) h[n]/ (double) total);
				mu_t += n * p[n];
			}


			/*
			 *  Maximize the covariance between the two levels
			 *  of the image.
			 */
			for (n = 0; n < bins; n++) {
				mu_0 = 0.0; weight_0 = 0.0;
				mu_1 = 0.0; weight_1 = 0.0;

				for (m = 0; m < bins; m++) {
					if (m < n) {
						mu_0 += m * p[m];
						weight_0 += p[m];
					} else { 
						mu_1 += m * p[m];
						weight_1 += p[m];
					}
				}

				omega_b_cur = weight_0 * (mu_0 - mu_t) * (mu_0 - mu_t)
					+ weight_1 * (mu_1 - mu_t) * (mu_1 - mu_t);

				if (omega_b_cur > omega_b_max) {
					omega_b_max = omega_b_cur;
					magic_number = n;
				}
			}

			magic_number *= bin_ratio;

			System.out.println("otsuThreshold finished");

			return magic_number;
		}


		final public int gaussianThreshold(ImageDataInt input, int lb, double fraction) {
			System.out.println("gaussianThreshold");
			int magic_number = -1;
			int XN = input.getRows();
			int YN = input.getCols();
			int ZN = input.getSlices();

			int x = 0, y = 0, z = 0, n = 0, m = 0;
			int max_int = -1000, min_int = 1000;
			int bins = -1;
			int value_i = 0;
			int total = 0, counter = 0;
			int h_max = 0;

			int k = 0;

			double current_diff = 0.0, max_diff = 0.0;
			double mean = 0.0, sd = 0.0;
			double s0 = 0.0, s1 = 0.0, s2 = 0.0;
			double alpha = 0.0, gz = 0.0;


			for (x = 0; x < XN; x++)
				for (y = 0; y < YN; y++)
					for (z = 0; z < ZN; z++) {
						value_i = input.getInt(x, y, z); 

						if (value_i <= lb) {
							continue;
						}

						if (value_i > max_int) {
							max_int = value_i;
						} 

						if (value_i < min_int) {
							min_int = value_i;
						} 
					}
			bins = max_int - min_int + 1;


			int[] h = new int[bins];
			double[] p = new double[bins];

			for (n = 0; n < bins; n++) {
				h[n] = 0;
				p[n] = 0;
			}


			/*
			 * Build a histogram.
			 */
			for (x = 0; x < XN; x++)
				for (y = 0; y < YN; y++)
					for (z = 0; z < ZN; z++) {
						value_i = input.getInt(x, y, z);

						if (value_i <= lb) {
							continue;
						}

						if (value_i >= bins) {
							value_i = bins - 1;
						} 

						h[value_i]++;
						total++;
					}


			/*
			 * Trim the histogram based on the required 
			 * fraction of the histogram area.
			 */
			for (n = 0; n < bins; n++) {
				counter += h[n];
				if (counter >= fraction * total) {
					h[n] = 0;
				} else {
					h_max = n;
				}
			}


			/*
			 * One pass calculation of the mean and standard 
			 * deviation based on the new histogram.
			 */
			for (x = 0; x < XN; x++)
				for (y = 0; y < YN; y++)
					for (z = 0; z < ZN; z++) {
						value_i = input.getInt(x, y, z); 

						if (value_i <= lb) {
							continue;
						} 

						if (value_i <= h_max) {
							s0 += 1;
							s1 += value_i;
							s2 += value_i * value_i;
						}
					}
			mean = (s1/s0);
			sd = ((1.0/s0) * Math.sqrt(s0 * s2 - (s1*s1)));


			/*
			 * Compute weighting for function.
			 */
			for (n = 0; n < h_max; n++) {
				alpha += h[n];
				gz += g_x(n, mean, sd);
			}


			/*
			 * Estimate the noise level based on the maximum 
			 * between the new histogram and a gaussian model
			 * of the data.
			 */
			for (n = 1; n < h_max; n++) {
				current_diff = (int) ((alpha/gz) * (g_x(n, mean, sd)) - h[n]);

				if (current_diff > max_diff) {
					max_diff = current_diff;
					magic_number = n;
				}
			}


			return magic_number;
		}


		final public double g_x(double x, double mean, double sd) {
			double value = 0.0;
			value = (1.0/Math.sqrt(2 * sd * sd * Math.PI)) * Math.exp((-1.0 * ((x - mean)*(x - mean))) / (2 * sd * sd)); 
			return value;
		}
	}
}
