package edu.vanderbilt.masi.plugins.CRUISE.tgdm.GAC;

import java.awt.Dimension;
import java.io.File;
import java.io.IOException;

import javax.vecmath.Point3f;
import javax.vecmath.Point3i;

import edu.jhmi.rad.medic.utilities.CropParameters;
import edu.jhmi.rad.medic.utilities.CubicVolumeCropper;
import edu.jhu.ece.iacl.algorithms.PrinceGroupAuthors;
import edu.jhu.ece.iacl.algorithms.ReferencedPapers;
import edu.jhu.ece.iacl.algorithms.ace.AnisotropicDiffusion;
import edu.jhu.ece.iacl.algorithms.gvf.Gradient3d;
import edu.jhu.ece.iacl.algorithms.tgdm.ProfileTGDM.Tracking;
import edu.jhu.ece.iacl.algorithms.topology.ConnectivityRule;
import edu.jhu.ece.iacl.algorithms.volume.DistanceField;
import edu.jhu.ece.iacl.jist.io.ImageDataReaderWriter;
import edu.jhu.ece.iacl.jist.io.SurfaceFreeSurferReaderWriter;
import edu.jhu.ece.iacl.jist.io.SurfaceVtkReaderWriter;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamDouble;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamOption;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamSurface;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.jist.structures.image.ImageHeader;
import edu.jhu.ece.iacl.jist.structures.image.VoxelType;

/**
 * @author Yuankai Huo
 * @email yuankai.huo@vanderbilt.edu
 * @version 1.0
 *
 * This TGDM requires the original GM file to get 
 * the crop information as used in LongCRUiSE
 * It is used to test the TGDM step seperate from the LongCRUISE
 *
 */


public class MaTGDMcropGAC extends ProcessingAlgorithm {
    private ParamVolume gvfVol;
    private ParamVolume gmAceVol;
    private ParamVolume gmVol;
    private ParamVolume wmVol;
    private ParamVolume cropVol;
    private ParamVolume T1Vol;
    private ParamVolume initVol;
    private ParamVolume skelVol;
    private ParamVolume innerVol;
    private ParamVolume centralVol;
    private ParamVolume outerVol;
    private ParamVolume sphereInnerVol;
    private ParamVolume sphereCentralVol;
    private ParamVolume sphereOuterVol;
    private ParamVolume sgnVol;
    private ParamVolume brainMaskVol;
    private ParamBoolean runInner, runCentral, runOuter;

    private ParamCollection dataCollection;
    private ParamCollection paramCollection;
    private ParamDouble initCurvatureForce;
    private ParamDouble innerCurvatureForce, innerExternalForce,
            innerPressureForce;
    private ParamDouble centralCurvatureForce, centralExternalForce,
            centralPressureForce;
    private ParamDouble outerCurvatureForce, outerExternalForce,
            outerPressureForce;
    private ParamInteger initIterations;
    private ParamInteger innerIterations;
    private ParamInteger centralIterations;
    private ParamInteger outerIterations;
    private ParamBoolean gensurfs;
    private ParamSurface embeddedInnerSurface;
    private ParamSurface innerSurf;
    private ParamSurface centralSurf;
    private ParamSurface outerSurf;
    private ParamOption connectivity;
    private ParamOption tracking;
    private ParamDouble initIsoLevel;

    private ParamBoolean isMembership;
    private ParamDouble isoWm, isoGm;
    
    private static final ImageDataReaderWriter vrw = ImageDataReaderWriter.getInstance();
    private static final SurfaceVtkReaderWriter srw = SurfaceVtkReaderWriter.getInstance();
    private static final SurfaceFreeSurferReaderWriter frw = SurfaceFreeSurferReaderWriter.getInstance();
    
    private static final String cvsversion = "$Revision: 1.1 $";
    private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");

    protected void createInputParameters(ParamCollection inputParams) {

        dataCollection = new ParamCollection("Data");
        dataCollection.add(initVol = new ParamVolume("Initial Level Set",
                VoxelType.FLOAT));
        dataCollection.add(isMembership = new ParamBoolean(
                "Initial Volume is Membership [0,255]", true));
        dataCollection.add(wmVol = new ParamVolume("White Matter Membership",
                VoxelType.FLOAT));
        dataCollection.add(gmVol = new ParamVolume("Gray Matter Membership",
                VoxelType.FLOAT));
        dataCollection.add(gmAceVol = new ParamVolume(
                "ACE Gray Matter Membership", VoxelType.FLOAT));
        dataCollection.add(skelVol = new ParamVolume("Thinned ACE Skeleton",
                VoxelType.UBYTE));
        dataCollection.add(gvfVol = new ParamVolume("Gradient Vector Field",
                VoxelType.FLOAT, -1, -1, -1, 3));
        dataCollection.add(cropVol = new ParamVolume("Crop GM Image",
                VoxelType.FLOAT));
        dataCollection.add(T1Vol = new ParamVolume("N3T1 image",
                VoxelType.FLOAT));
        
        dataCollection.add(brainMaskVol = new ParamVolume(
                "Brain Mask Level Set", VoxelType.FLOAT));
        brainMaskVol.setMandatory(false);
        dataCollection.add(embeddedInnerSurface = new ParamSurface(
                "Embedded Inner Surface"));
        initVol.setMandatory(false);
        wmVol.setMandatory(false);
        gmVol.setMandatory(false);
        gmAceVol.setMandatory(false);
        skelVol.setMandatory(false);
        gvfVol.setMandatory(false);
        T1Vol.setMandatory(false);
        
        embeddedInnerSurface.setMandatory(false);
        dataCollection.add(tracking = new ParamOption("Tracking", new String[] {
                "Off", "Sphere", "Point", "Shell" }));
        dataCollection.add(runInner = new ParamBoolean(
                "Generate Inner Surface", true));
        dataCollection.add(runCentral = new ParamBoolean(
                "Generate Central Surface", true));
        dataCollection.add(runOuter = new ParamBoolean(
                "Generate Outer Surface", true));
        inputParams.add(dataCollection);
        paramCollection = new ParamCollection("Parameters");
        paramCollection.add(initIsoLevel = new ParamDouble("Initial Iso Level",
                126.9));
        paramCollection.add(isoWm = new ParamDouble("White Matter Iso Level",
                0, 1, 0.6)); 
        paramCollection.add(isoGm = new ParamDouble("Gray Matter Iso Level", 0,
                1, 0.4));
        paramCollection.add(initCurvatureForce = new ParamDouble(
                "Initial Smoothing Curvature Force", 0, 10,1));

        paramCollection.add(initIterations = new ParamInteger(
                "Iterations for Initial Smoothing", 0, 1000, 6));
        paramCollection.add(innerCurvatureForce = new ParamDouble(
                "Inner Surface Curvature Force", 0, 10, 0.2));
        paramCollection.add(innerPressureForce = new ParamDouble(
                "Inner Surface Pressure Force", 0, 10, 1));
        paramCollection.add(innerExternalForce = new ParamDouble(
                "Inner Surface External Force", 0, 10, 0));
        paramCollection.add(innerIterations = new ParamInteger(
                "Iterations for Inner Surface", 0, 1000, 4));
        paramCollection.add(centralCurvatureForce = new ParamDouble(
                "Central Surface Curvature Force", 0, 10, 0.15));
        paramCollection.add(centralPressureForce = new ParamDouble(
                "Central Surface Pressure Force", 0, 10, 1.5));
        paramCollection.add(centralExternalForce = new ParamDouble(
                "Central Surface External Force", 0, 10, 1));
        paramCollection.add(centralIterations = new ParamInteger(
                "Iterations for Central Surface", 0, 1000, 7));
        paramCollection.add(outerCurvatureForce = new ParamDouble(
                "Outer Surface Curvature Force", 0, 10, 1.0));
        paramCollection.add(outerPressureForce = new ParamDouble(
                "Outer Surface Pressure Force", 0, 10, 1));
        paramCollection.add(outerExternalForce = new ParamDouble(
                "Outer Surface External Force", 0, 10, 0));
        paramCollection.add(outerIterations = new ParamInteger(
                "Iterations for Outer Surface", 0, 1000, 7));
        paramCollection.add(connectivity = new ParamOption(
                "Connectivity (Foreground,Background)", new String[] {
                        "(18,6)", "(6,18)", "(26,6)", "(6,26)" }));
        connectivity.setValue(0);
        paramCollection.add(gensurfs = new ParamBoolean(
                "Generate Iso-Surfaces", true));
        inputParams.add(paramCollection);

        this.setPreferredSize(new Dimension(350, 500));

        inputParams.setName("nested_tgdm");
        inputParams.setLabel("Nested TGDM");

        inputParams.setPackage("IACL");
        inputParams.setCategory("Segmentation.Geometric_Deformable_Model");

        AlgorithmInformation info = getAlgorithmInformation();
        info.setWebsite("http://www.iacl.ece.jhu.edu/");
        info.setVersion(revnum);
        info.setEditable(false);
        info.setDescription("Geometric deformable model to find inner, central, and outer coritcal surfaces.");
        info.setDescription("There are slight numerical differences between this implementation and the original implementation, but those differences are far less than the precision of the model (0.1 voxel).");
        info.setStatus(DevelopmentStatus.Release);
        info.add(ReferencedPapers.tgdm);
        info.add(PrinceGroupAuthors.xiaoHan);
        info.add(PrinceGroupAuthors.blakeLucas);
        info.setAdditionalDocURL("html/edu/jhu/ece/iacl/plugins/segmentation/gdm/MedicAlgorithmNestedTGDM/index.html");
    }

    protected void createOutputParameters(ParamCollection outputParams) {
        outputParams.add(innerVol = new ParamVolume("Inner Level Set",
                VoxelType.FLOAT));
        outputParams.add(centralVol = new ParamVolume("Central Level Set",
                VoxelType.FLOAT));
        outputParams.add(outerVol = new ParamVolume("Outer Level Set",
                VoxelType.FLOAT));
        outputParams.add(sgnVol = new ParamVolume("Signed Volume",
                VoxelType.UBYTE));
//        outputParams.add(sphereInnerVol = new ParamVolume("Sphere Inner Map",
//                VoxelType.FLOAT, -1, -1, -1, 3));
//        outputParams.add(sphereCentralVol = new ParamVolume(
//                "Sphere Central Map", VoxelType.FLOAT, -1, -1, -1, 3));
//        outputParams.add(sphereOuterVol = new ParamVolume("Sphere Outer Map",
//                VoxelType.FLOAT, -1, -1, -1, 3));
        outputParams.add(innerSurf = new ParamSurface("Inner Surface"));
        outputParams.add(centralSurf = new ParamSurface("Central Surface"));
        outputParams.add(outerSurf = new ParamSurface("Outer Surface"));
        innerVol.setMandatory(false);
        centralVol.setMandatory(false);
        outerVol.setMandatory(false);
        sgnVol.setMandatory(false);
        innerSurf.setMandatory(false);
        centralSurf.setMandatory(false);
        outerSurf.setMandatory(false);
//        sphereInnerVol.setMandatory(false);
//        sphereCentralVol.setMandatory(false);
//        sphereOuterVol.setMandatory(false);
        outputParams.setName("nested_tgdm");
        outputParams.setLabel("Nested TGDM");
    }

    protected void execute(CalculationMonitor monitor) {
        int conn = 0;
        switch (connectivity.getIndex()) {
        case 0:
            conn = ConnectivityRule.CONNECT_18_6;
            break;
        case 1:
            conn = ConnectivityRule.CONNECT_6_18;
            break;
        case 2:
            conn = ConnectivityRule.CONNECT_26_6;
            break;
        case 3:
            conn = ConnectivityRule.CONNECT_6_26;
            break;
        default:
            break;
        }
        
        System.out.println("run MedicAlgorithmTGDM_testing");// yk add debug
//        System.out.println(embeddedInnerSurface.getSurface());// yk add debug
//        System.out.println(brainMaskVol.getImageData());// yk add debug
        
		File testdir = new File(this.getOutputDirectory()+File.separator+"GACintermediate");
		try{
			if(!testdir.isDirectory()){
				(new File(testdir.getCanonicalPath())).mkdir();
			}
		}catch(IOException e){ e.printStackTrace(); }


        
        //yk add debug
        ImageData gmcrop = cropVol.getImageData();
        ImageHeader gmhdr = gmcrop.getHeader();
        float[] gmorigin = gmhdr.getOrigin();
        
        int gmrows=gmcrop.getRows();
        int gmcols=gmcrop.getCols();
        int gmslices=gmcrop.getSlices();
        CubicVolumeCropper cropper = new CubicVolumeCropper();    
        gmcrop = cropper.crop(gmcrop, 0, 5); //enlarge (crop) the image by 5 voxels
        CropParameters crops = cropper.getLastCropParams();
        Point3f offset = new Point3f(crops.xmin, crops.ymin, crops.zmin);
//        Point3f freesurferoffset = new Point3f((-gmorigin[0]), (-gmorigin[1]), (gmorigin[2]));
        Point3i d = new Point3i(crops.getCroppedRows(),crops.getCroppedCols(),crops.getCroppedSlices());
        Point3i off = new Point3i(crops.xmin,crops.ymin,crops.zmin);
        CropParameters uncrop = new CropParameters(gmrows,gmcols,gmslices,1,off.x,d.x+off.x-1,off.y,d.y+off.y-1,off.z,d.z+off.z-1,20);

        ImageDataFloat T1, T1blur,T1g;
        T1 = new ImageDataFloat(T1Vol.getImageData());
        T1blur = AnisotropicDiffusion.GaussianBlur(T1, 1.8f);
        T1g = Gradient3d.doSolve(T1blur);
        
        T1.setName("01_T1");
        T1.setHeader(gmVol.getImageData().getHeader());
		vrw.write(T1,testdir);	
		
		T1blur.setName("02_T1_blur");
		T1blur.setHeader(gmVol.getImageData().getHeader());
		vrw.write(T1blur,testdir);	
		
		T1g.setName("03_T1_gaussian");
		T1g.setHeader(gmVol.getImageData().getHeader());
		vrw.write(T1g,testdir);	
        
        
        File dir = new File(this.getOutputDirectory()+File.separator+"uncroped");
        try{
            if(!dir.isDirectory()){
                (new File(dir.getCanonicalPath())).mkdir();
            }
        }catch(IOException e){ e.printStackTrace(); }
        
        System.out.println(dir);// yk add debug
        ImageDataFloat brainMask = (brainMaskVol.getImageData() != null) ? new ImageDataFloat(
                brainMaskVol.getImageData())
                : null;
        if(brainMask!=null)System.out.println("Use brain mask "+brainMask.getName());
        Tracking track = Tracking.values()[tracking.getIndex()];
        // String name = initVol.getImageData().getName();
        String name = gmVol.getImageData().getName();
        // MedicUtil.displayMessage("Before anything in NestedTGDM: name="+name+"\n");
        MaGenericTGDMGAC tgdm = new MaGenericTGDMGAC(conn);

        monitor.observe(tgdm);
        ImageDataFloat inner, central, outer;
        float isomem = isoWm.getFloat();
        if (runInner.getValue()) {
            ImageDataFloat phi;
            if (isMembership.getValue()) {
                ImageData VpMImg = initVol.getImageData();
                int rows = VpMImg.getRows();
                int cols = VpMImg.getCols();
                int slices = VpMImg.getSlices();
                DistanceField df = new DistanceField();
                phi = new ImageDataFloat(VpMImg);
                float[][][] Phi = phi.toArray3d();
                for (int x = 0; x < rows; x++)
                    for (int y = 0; y < cols; y++)
                        for (int z = 0; z < slices; z++) {
                            Phi[x][y][z] = initIsoLevel.getFloat()
                                    - Phi[x][y][z];
                        }
                System.out.println("Fast Marching...");
                phi = df.solve(phi, 8);
                phi = tgdm.solveSmoothSurface(phi, initCurvatureForce
                        .getFloat(), initIterations.getInt());
            } else {
                phi = new ImageDataFloat(initVol.getImageData());
            }
            inner = tgdm.solveInnerSurfaceFromOuter(phi, brainMask, wmVol
                    .getImageData(), null, innerCurvatureForce.getFloat(),
                    innerExternalForce.getFloat(), innerPressureForce
                            .getFloat(), initIsoLevel.getFloat(), isomem,
                    innerIterations.getInt());
        }else {
            inner = new ImageDataFloat(initVol.getImageData());
        }
        ImageData GM = gmAceVol.getImageData();
        ImageData WM = wmVol.getImageData();
        ImageDataFloat signVol = new ImageDataFloat(GM.getRows(), GM.getCols(),
                GM.getSlices());
        float sign, gm, wm, csf;
        float isowm = isoWm.getFloat(), isogm = isoGm.getFloat();
        for (int i = 0; i < signVol.getRows(); i++) {
            for (int j = 0; j < signVol.getCols(); j++) {
                for (int k = 0; k < signVol.getSlices(); k++) {
                    gm = (float) GM.getFloat(i, j, k) / 255.0f;
                    wm = (float) WM.getFloat(i, j, k) / 255.0f;
                    csf = 1.0f - wm - gm;
                    if (wm > csf) {
                        sign = (wm - isowm) / (1.0f - isowm);
                        if (sign < 0)
                            sign = 0;
                    } else {
                        sign = (1 - csf - isogm) / isogm;
                        if (sign > 0)
                            sign = 0;
                    }
                    sign = (int) (sign * 127 + 127);
                    signVol.set(i, j, k, (int) ((sign < 254) ? sign : 254));
                }
            }
        }
        signVol.setName(name + "_sign");
        sgnVol.setValue(signVol);
        //sgnVol.getImageData().getModelImageCopy().copyFileTypeInfo(gmVol.getImageData().getModelImageCopy());
        sgnVol.getImageData().setHeader(gmVol.getImageData().getHeader());
        if (embeddedInnerSurface.getSurface() != null) {
            EmbeddedSurface sphericalMap = embeddedInnerSurface.getSurface();

        }
        if (runCentral.getValue()) {
            central = tgdm.solveCentralSurface(inner, brainMask, signVol,
                    gvfVol.getImageData(), centralCurvatureForce.getFloat(),
                    centralExternalForce.getFloat(), centralPressureForce
                            .getFloat(), centralIterations.getInt());
            central.setName(name + "_centrallevelset");
            centralVol.setValue(central);
            //centralVol.getImageData().getModelImageCopy().copyFileTypeInfo(gmVol.getImageData().getModelImageCopy());
            centralVol.getImageData().setHeader(gmVol.getImageData().getHeader());
            if (gensurfs.getValue()) {
                EmbeddedSurface surf = tgdm.getFinalSurface();
                surf.setName(name + "_CentralSurf");
                centralSurf.setValue(surf);
                surf = tgdm.getInitialSurface();
                surf.setName(name + "_InnerSurf");
                innerSurf.setValue(surf);
            }

        } else {
            central = new ImageDataFloat(initVol.getImageData());
        }
       
      
        //recompute inner surface starting from central
        if (runInner.getValue()){
            ///push central surface inward
            ImageDataFloat central_shrink = tgdm.solveInnerSurfaceFromCentral(central, null, null
                    , null, centralCurvatureForce.getFloat(),
                    0, -1.0f, initIsoLevel.getFloat(), isomem,1);
            
		
			central_shrink.setName("04_CentralShrink");
			central_shrink.setHeader(central.getHeader());
			vrw.write(central_shrink,testdir);	
			
			System.out.println("centralCurvatureForce centralCurvatureForce is"+centralCurvatureForce.getFloat());// yk add debug
			System.out.println("initIsoLevel initIsoLevel"+initIsoLevel.getFloat());// yk add debug
			System.out.println("isomem isomem is"+isomem);// yk add debug
            
            //recompute inner surface
            inner = tgdm.solveInnerSurfaceFromCentral(central_shrink, null, wmVol
                .getImageData(), null, innerCurvatureForce.getFloat(),
                innerExternalForce.getFloat(), -innerPressureForce
                        .getFloat(), initIsoLevel.getFloat(), isomem,
                innerIterations.getInt());
//            
//            inner.setName("05_inner_init");
//            inner.setHeader(central.getHeader());
//			vrw.write(inner,testdir);	
			
			ImageData WMtest = wmVol.getImageData();
			WMtest.setName("05_WM");
			WMtest.setHeader(central.getHeader());
			vrw.write(WMtest,testdir);	
            
    		//recompute inner surface using Geodesic Active Contour
//    		inner = tgdm.solveInnerSurfaceFromCentralGAC(central_shrink, null, wmVol
//    				.getImageData(), null, innerCurvatureForce.getFloat(),
//    				innerExternalForce.getFloat(), -innerPressureForce
//    				.getFloat(), initIsoLevel.getFloat(), isomem,
//    				innerIterations.getInt(),T1g);
            
            inner.setName("06_inner_final");
            inner.setHeader(central.getHeader());
			vrw.write(inner,testdir);
            
            inner.setName(name + "_innerlevelset");

//            inner.setHeader(gmVol.getImageData().getHeader());
//			vrw.write(inner,testdir);
            
            
            innerVol.setValue(inner);  //yk modified from central_shrink 05/29/2015 
            innerVol.getImageData().setHeader(gmVol.getImageData().getHeader());
            if (gensurfs.getValue()) {
                EmbeddedSurface surf = tgdm.getFinalSurface();
                // MedicUtil.displayMessage("Center of mass = "+surf.getCenterOfMass()+", name="+name+"_InnerSurf\n");
                surf.setName(name + "_InnerSurf");
                innerSurf.setValue(surf);
            }

        } else {
            inner = new ImageDataFloat(initVol.getImageData());
        }
        
        
        
        if (runOuter.getValue()) {
            outer = tgdm.solveOuterSurface(central, brainMask, gmVol
                    .getImageData(), null, skelVol.getImageData(),
                    outerCurvatureForce.getFloat(), outerExternalForce
                            .getFloat(), outerPressureForce.getFloat(), isogm,
                    outerIterations.getInt());
            outer.setName(name + "_outerlevelset");
            outerVol.setValue(outer);
            //outerVol.getImageData().getModelImageCopy().copyFileTypeInfo(gmVol.getImageData().getModelImageCopy());
            outerVol.getImageData().setHeader(gmVol.getImageData().getHeader());
            
            if (gensurfs.getValue()) {
                EmbeddedSurface surf = tgdm.getFinalSurface();
                surf.setName(name + "_OuterSurf");
                outerSurf.setValue(surf);
                surf = tgdm.getInitialSurface();

                centralSurf.setValue(surf);
                centralSurf.getSurface().setName(name + "_CentralSurf");
            }

        } else {
            outer = new ImageDataFloat(initVol.getImageData());
        }
        

        //yk debug
//        CropParameters uncrop2=new CropParameters(rows,cols,slices,1,off.x,d.x+off.x-1,off.y,d.y+off.y-1,off.z,d.z+off.z-1,0);
//        CropParameters uncrop3=new CropParameters(rows,cols,slices,3,off.x,d.x+off.x-1,off.y,d.y+off.y-1,off.z,d.z+off.z-1,0);
//
//        
        
        ImageData A;
        ParamFile ykdebug = new ParamVolume("yk debug");
        A = innerVol.getImageData();
        A = cropper.uncrop(A,uncrop);
        A.setName(name+"_innerVol");
        ykdebug.setValue(vrw.write(A, dir));
//        
        A = centralVol.getImageData();
        A = cropper.uncrop(A,uncrop);
        A.setName(name+"_centralVol");
        ykdebug.setValue(vrw.write(A, dir));
//        
        A = outerVol.getImageData();
        A = cropper.uncrop(A,uncrop);
        A.setName(name+"_outerVol");
        ykdebug.setValue(vrw.write(A, dir));
//        
        A = sgnVol.getImageData();
        A = cropper.uncrop(A,uncrop);
        A.setName(name+"_sgnVol");
        ykdebug.setValue(vrw.write(A, dir));
        
//        A = sphereInnerVol.getImageData();
//        A.setName(name+"_sphereInnerVol");
//        ykdebug.setValue(vrw.write(A, dir));
////        
//        A = sphereCentralVol.getImageData();
//        A.setName(name+"_sphereCentralVol");
//        ykdebug.setValue(vrw.write(A, dir));
////        
//        A = sphereOuterVol.getImageData();
//        A.setName(name+"_sphereOuterVol");
//        ykdebug.setValue(vrw.write(A, dir));
        
        
        EmbeddedSurface surfyk;
        surfyk = innerSurf.getSurface();
        surfyk.translate(offset);
        surfyk.setName(name+"_innerSurf");
        ykdebug.setValue(srw.write(surfyk, dir));
        //set freesurfer surface
//        surfyk.translate(freesurferoffset);
//        surfyk.setName(name+"_innerSurf_freesurfer");
//        ykdebug.setValue(frw.write(surfyk, dir));
        
        surfyk = centralSurf.getSurface();
        surfyk.setName(name+"_centralSurf");
        surfyk.translate(offset);
        ykdebug.setValue(srw.write(surfyk, dir));
//        surfyk.translate(freesurferoffset);
//        surfyk.setName(name+"_centralSurf_freesurfer");
//        ykdebug.setValue(frw.write(surfyk, dir));
        
        surfyk = outerSurf.getSurface();
        surfyk.setName(name+"_outerSurf");
        surfyk.translate(offset);
        ykdebug.setValue(srw.write(surfyk, dir));
//        surfyk.translate(freesurferoffset);
//        surfyk.setName(name+"_outerSurf_freesurfer");
//        ykdebug.setValue(frw.write(surfyk, dir));
    }    
}