package edu.vanderbilt.masi.plugins.CRUISE.utilities;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import edu.jhu.ece.iacl.algorithms.topology.ConnectivityRule;
import edu.jhu.ece.iacl.algorithms.volume.DistanceField;
import edu.jhu.ece.iacl.jist.io.ImageDataReaderWriter;
import edu.jhu.ece.iacl.jist.io.SurfaceFreeSurferReaderWriter;
import edu.jhu.ece.iacl.jist.io.SurfaceVtkReaderWriter;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamDouble;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamOption;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamString;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataInt;
import edu.jhu.ece.iacl.jist.structures.image.VoxelType;
import edu.vanderbilt.masi.plugins.CRUISE.tgdm.MaGenericTGDM;
import edu.vanderbilt.masi.plugins.CRUISE.tgdm.GAC.MaGenericTGDMGAC;
import edu.vanderbilt.masi.plugins.CRUISE.tgdm.GAC.MaTGDMcropGAC;

/**
 * @author Yuankai Huo
 * @email yuankai.huo@vanderbilt.edu
 * @version 1.0
 *
 *transfer the toads cruise space image to original space
 *
 */

public class SkullLevelSet extends ProcessingAlgorithm{

	private ParamVolume InitLevelset;
	private ParamVolume Skull;
	
	private ParamVolume gvfVol;
	
	private ParamVolume OutputVol;
	
	private ParamBoolean DebugFlag;

	//init levelset
	private ParamDouble initIsoLevel;
	private ParamDouble initCurvatureForce;
	
	private ParamInteger initIterations;

	//inner levelset
	private ParamDouble isoWm;
	private ParamDouble CurvatureForce;
	private ParamDouble PressureForce;
	private ParamDouble ExternalForce;
	private ParamInteger levelIteration;

	private static final ImageDataReaderWriter vrw = ImageDataReaderWriter.getInstance();
	private static final SurfaceVtkReaderWriter srw = SurfaceVtkReaderWriter.getInstance();
	private static final SurfaceFreeSurferReaderWriter frw = SurfaceFreeSurferReaderWriter.getInstance();

	ParamFile ykdebug = new ParamVolume("yk debug");

	private static final String cvsversion = "$Revision: 1.4 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "Conduct levelset for skull" + revnum + "\n";

	@Override
	protected void createInputParameters(ParamCollection inputParams) {
		inputParams.setPackage("Vanderbilt");
		inputParams.setCategory("format");
		inputParams.setLabel("Skull Levelset");
		inputParams.setName("Skull_Levelset");

		// input init levelset
		InitLevelset=new ParamVolume("InitLevelset");
		Skull = new ParamVolume("InitSkull");
		gvfVol = new ParamVolume("Gradient Vector Field", VoxelType.FLOAT, -1, -1, -1, 3);
		
		// add parameters
		inputParams.add(InitLevelset);
		inputParams.add(Skull);
		inputParams.add(gvfVol);
		

		// init ISO level
		inputParams.add(initIsoLevel = new ParamDouble
				("Initial Iso Level",126.9));
		inputParams.add(initCurvatureForce = new ParamDouble
				("Initial Smoothing Curvature Force",0, 10,1));
		inputParams.add(initIterations = new ParamInteger(
				"Iterations for Initial Smoothing", 0, 1000, 6));

		// 
		inputParams.add(isoWm = new ParamDouble("White Matter Iso Level",
				0, 1, 0.6)); 
		inputParams.add(CurvatureForce = new ParamDouble(
				"Curvature Force", 0, 10, 0.2));
		inputParams.add(PressureForce = new ParamDouble(
				"Pressure Force", 0, 10, 1.5));
		inputParams.add(ExternalForce = new ParamDouble(
				"External Force", 0, 10, 0));
		inputParams.add(levelIteration = new ParamInteger(
				"Iterations for Surface", 0, 1000, 4));
		

		DebugFlag=new ParamBoolean("Generate intermediate files",false);
		inputParams.add(DebugFlag);


	}

	@Override
	protected void createOutputParameters(ParamCollection outputParams) {
//		 TODO Auto-generated method stub
		OutputVol = new ParamVolume("Output volume");
		outputParams.add(OutputVol);
	}

	@Override
	protected void execute(CalculationMonitor monitor)
			throws AlgorithmRuntimeException {
		// TODO Auto-generated method stub

		//set the output directory and create it if it doesn't exist
		File dir = new File(this.getOutputDirectory()+File.separator+edu.jhu.ece.iacl.jist.utility.FileUtil.forceSafeFilename(this.getAlgorithmName()));
		try{
			if(!dir.isDirectory()){
				(new File(dir.getCanonicalPath())).mkdir();
			}
		}catch(IOException e){ e.printStackTrace(); }

		ImageData initlevelset = InitLevelset.getImageData();
		ImageData skull = Skull.getImageData();
		ImageDataFloat levelvol;

		int rows = initlevelset.getRows();
		int cols = initlevelset.getCols();
		int slices = initlevelset.getSlices();

		int conn = ConnectivityRule.CONNECT_18_6;

		MaGenericTGDMGAC tgdm = new MaGenericTGDMGAC(conn);
		monitor.observe(tgdm);

		//get init levelset
		ImageDataFloat phi;
		if(DebugFlag.getValue()&&FilesExist(dir,"InitPhi","initphi")){
			phi = new ImageDataFloat(load_file_from_disk(dir,"InitPhi","initphi"));
			System.out.println("Exist, skip step initphi");// yk add debug
		}else{


			DistanceField df = new DistanceField();
			phi = new ImageDataFloat(initlevelset);
			float[][][] Phi = phi.toArray3d();
			for (int x = 0; x < rows; x++)
				for (int y = 0; y < cols; y++)
					for (int z = 0; z < slices; z++) {
						Phi[x][y][z] = initIsoLevel.getFloat()
								- Phi[x][y][z];
					}
			System.out.println("Fast Marching...");
			phi = df.solve(phi, 8);
			phi = tgdm.solveSmoothSurface(phi, initCurvatureForce
					.getFloat(), initIterations.getInt());

			save_intermfile(dir,"InitPhi",phi,"initphi");	
		}

		//get init levelset
		float isomem = isoWm.getFloat();
		if(DebugFlag.getValue()&&FilesExist(dir,"OutputLevel","OutputLevel")){
			levelvol = new ImageDataFloat(load_file_from_disk(dir,"OutputLevel","OutputLevel"));
			System.out.println("Exist, skip step initphi");// yk add debug
		}else{

			//		levelvol = tgdm.solveOuterSurface(phi, null, initlevelset, null,
			//				skull, CurvatureForce.getFloat(),
			//				ExternalForce.getFloat(), PressureForce
			//				.getFloat(), isomem,
			//				levelIteration.getInt());
			levelvol = tgdm.solveCentralSurface(phi, null, initlevelset, gvfVol.getImageData(),
					CurvatureForce.getFloat(),
					ExternalForce.getFloat(), PressureForce
					.getFloat(), levelIteration.getInt());

			save_intermfile(dir,"OutputLevel",levelvol,"OutputLevel");
		}


		String outputname =  "OutputVol";
		OutputVol.setName(outputname);
		OutputVol.setValue(levelvol);
		OutputVol.getImageData().setHeader(initlevelset.getHeader());


	}

	public ImageData load_file_from_disk(File dir,String IntermediateFolderName,String NewOutputName){
		String file_str = dir + "/"+IntermediateFolderName+"/"+NewOutputName+".nii.gz";
		File file_file = new File(file_str);
		ImageData Data = vrw.read(file_file);
		return Data;
	}

	public boolean FilesExist(File dir,String IntermediateFolderName,String NewOutputName){	
		String file_output_str = dir + "/"+IntermediateFolderName+"/"+NewOutputName+".nii.gz";
		File f = new File(file_output_str);
		return f.exists();
	}


	public void save_intermfile(File dir,String IntermediateFolderName,
			ImageData OutputImg,String NewOutputName){
		//cortical seperation folder
		String dir_output_str = dir + "/"+IntermediateFolderName+"/";
		File dir_output= new File(dir_output_str);
		if(DebugFlag.getValue()){
			try{
				if(!dir_output.isDirectory()){
					(new File(dir_output.getCanonicalPath())).mkdir();
				}
			}catch(IOException e){ e.printStackTrace(); }
		}
		if(DebugFlag.getValue()){		
			System.out.println("NewOutputName: "+NewOutputName);// yk add debug
			OutputImg.setName(NewOutputName);
			ykdebug.setValue(vrw.write(OutputImg, dir_output));
		}
	}

	public void save_intermfile(File dir,String IntermediateFolderName,
			EmbeddedSurface OutputSurf,String NewOutputName){
		//cortical seperation folder
		String dir_output_str = dir + "/"+IntermediateFolderName+"/";
		File dir_output= new File(dir_output_str);
		if(DebugFlag.getValue()){
			try{
				if(!dir_output.isDirectory()){
					(new File(dir_output.getCanonicalPath())).mkdir();
				}
			}catch(IOException e){ e.printStackTrace(); }
		}
		if(DebugFlag.getValue()){		
			System.out.println("NewOutputName: "+NewOutputName);// yk add debug
			OutputSurf.setName(NewOutputName);
			ykdebug.setValue(srw.write(OutputSurf, dir_output));
		}
	}

}
