package edu.jhmi.rad.medic.algorithms;

import edu.jhmi.rad.medic.dialogs.*;
import edu.jhmi.rad.medic.methods.*;
import gov.nih.mipav.model.algorithms.*;
import edu.jhmi.rad.medic.utilities.*;
import gov.nih.mipav.view.dialogs.*;
import gov.nih.mipav.model.structures.*;
import gov.nih.mipav.view.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.lang.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;

 /**
 *
 *   FANTASM: Fuzzy And Noise Tolerant Adaptive Segmentation Method.
 *   <p>
 *	 FANTASM is an improvement of the regular
 *   Fuzzy C-mean (FCM) algorithm for segmenting 2D and 3D images. It incorporates a
 *   spatial constraint that requires neighboring pixels to be similar and reduce
 *   the noise effect obtained with FCM, as well as inhomogeneity correction and edge detection
 *	 options.
 *   
 *   The smoothing parameter controls the amount of regularity required, other 
 *   parameters are similar to those of the FCM algorithm in Mipav.
 *   @see AlgorithmFuzzyCMeans
 *   <p>		
 *   References:
 *	This code is a ported subset of C-code rewritten from the original implementation
 *  by Dzung Pham.
 *	1.)	Dzung L. Pham, "Spatial Models for Fuzzy Clustering", Computer Vision and 
 *  Image Understanding, vol. 84, pp. 285-297, 2001.
 *
 *
 *	@version    March 2004
 *	@author     Pilou Bazin
 *  @see 		JDialogFantasm
 *  @see 		JDialogFantasms
 *	@see		SegmentationFCM
 *		
 *
*/
public class AlgorithmFantasm extends AlgorithmBase {

    // Fuzzy images require 1 image for each class
    // Hard images 1 image with assigned clusters
    private ModelImage          destImage[];
    private ModelImage          srcImage;
    private int                 destNum=0;
    private ViewUserInterface   userInterface;
    private float				Imin;
    private float				Imax;
    
     // information on the parts of the image to process
    private     float       backThreshold;
    private     boolean     cropBackground;
	private		boolean		useRelative;
    private		boolean		adaptiveSmoothing = true;
    private     int         nxmin,nymin,nzmin,nxmax,nymax,nzmax;
    private     int         x0,xN,y0,yN,z0,zN;
    private     int         X0,XN,Y0,YN,Z0,ZN;
    private     int         mx,my,mz;

    // image size
	private int         clusters, classes; // classes include the outliers, clusters does not
	private int 		nx,ny,nz,dim;
	private float		rx,ry,rz;
	
    // algorithm parameters
	private		String		initMode;
	private		String		outputType;
	
	private 	float 		smoothing;
	private     int 		iterations;
	private     float   	maxDistance;
	private		float		fuzziness;
	
	private		boolean		addOutliers;
	private		float		outlier;        
	
	private		int			correctInhomogeneity;
	private static final int	NONE = 1;
	private static final int	IMAGE = 2;
	private static final int	CENTROIDS = 3;
	private static final int	SEPARATE_IMG = 4;
	private static final int	SEPARATE_CENT = 5;
	private		int			fieldDegree;
	
	private		boolean		useEdges;
	private		float		edgeSmoothness, edgeContrast, edgePrior;

	private		String		distanceMode;
	
	private     boolean		verbose = true;
	private     boolean		debug = false;
	
    /**
    *	Constructor for 3D images in which changes are placed in a predetermined destination image.
    *   @param destImg_      Image model where result image is to stored.
    *   @param srcImg_       Source image model.
    */
	public AlgorithmFantasm(ModelImage[] destImg_, ModelImage srcImg_, int destNum_,
							String init_, String output_,
							int nClasses_, int nIterMax_, float distMax_, float smooth_,
							float fuzzy_,
                            boolean addOut_, float outVal_, 
							String correct_, int poly_,
							boolean edges_, float eSmooth_, float eContr_, float ePr_,
                            boolean cropBack_, float cropVal_, boolean useRel_, boolean adaptSmooth_) {
	
		this(destImg_, srcImg_, destNum_, init_, output_, nClasses_, nIterMax_, distMax_, smooth_,
			fuzzy_, addOut_, outVal_, correct_, poly_, edges_, eSmooth_, eContr_, ePr_, cropBack_, cropVal_, 
			useRel_, adaptSmooth_, "Gaussian");
	}
	
	public AlgorithmFantasm(ModelImage[] destImg_, ModelImage srcImg_, int destNum_,
							String init_, String output_,
							int nClasses_, int nIterMax_, float distMax_, float smooth_,
							float fuzzy_,
                            boolean addOut_, float outVal_, 
							String correct_, int poly_,
							boolean edges_, float eSmooth_, float eContr_, float ePr_,
                            boolean cropBack_, float cropVal_, boolean useRel_, 
							boolean adaptSmooth_, String dist_) {
								
        super(null, srcImg_);
        srcImage = srcImg_;
        destImage = destImg_;
        userInterface = ViewUserInterface.getReference();
		destNum = destNum_;
		
		initMode = init_;
        outputType = output_;
        
        cropBackground = cropBack_;
		backThreshold = cropVal_;
        useRelative = useRel_;
		adaptiveSmoothing = adaptSmooth_;
         
		addOutliers = addOut_;
		outlier = outVal_;
		        
		clusters = nClasses_;
		classes = nClasses_;
        if (addOutliers) classes++;
		
		iterations = nIterMax_;
        maxDistance = distMax_;
		smoothing = smooth_;
		fuzziness = fuzzy_;
        
		if (correct_.equals("image")) 
			correctInhomogeneity = IMAGE;
		else if (correct_.equals("centroids")) 
			correctInhomogeneity = CENTROIDS;
		else if (correct_.equals("separate")) 
			correctInhomogeneity = SEPARATE_IMG;
		else 
			correctInhomogeneity = NONE;
		
		fieldDegree = poly_;
		
		useEdges = edges_;
		edgeSmoothness = eSmooth_;
		edgeContrast = eContr_;
		edgePrior = ePr_;
    
		distanceMode = dist_;
		
		if (debug) {
			MedicUtilPublic.displayMessage("image: \n");
			MedicUtilPublic.displayMessage(""+srcImage.getImageName()+"\n");
		}
	}

    /**
    *	Prepares this class for destruction.
    */
	public void finalize(){
	    destImage   = null;
	    System.gc();
        super.finalize();
	}

    /**
    *   Starts the algorithm.
    */
	public void runAlgorithm() {

        if (srcImage  == null) {
            displayError("Source Image is null");
            //notifyListeners(this);
            return;
        }
        if (destImage  == null) {
            displayError("Destination Image is null");
            //notifyListeners(this);
            return;
        }

        // start the timer to compute the elapsed time
        setStartTime();

        if (destImage != null){     // if there exists a destination image
            //constructLog(true);
			
			float[] buffer;
            if (srcImage.getNDims() == 2) {
			   try {
					// image length is length in 2 dims
					int length = srcImage.getExtents()[0] 
                                * srcImage.getExtents()[1];
                    
                    // retrieve all data: for each volume
                    buffer = new float[length];
                    srcImage.exportData(0,length, buffer); // locks and releases lock
					fireProgressStateChanged(srcImage.getImageName(), "Processing image ...");
				} catch (IOException error) {
					buffer = null;
					error.printStackTrace();
					errorCleanUp("Algorithm: source image locked", true);
					return;
				} catch (OutOfMemoryError e){
					buffer = null;
					e.printStackTrace();
					errorCleanUp("Algorithm: Out of memory creating process buffer", true);
					return;
				}
		
				// init dimensions
				nx = srcImage.getExtents()[0];
				ny = srcImage.getExtents()[1];
				nz = 1;
				rx = srcImage.getFileInfo()[0].getResolutions()[0];
				ry = srcImage.getFileInfo()[0].getResolutions()[1];
				rz = 1.0f;
				dim = 2;
				
                // main algorithm
				calcSegmentation(buffer);
				
			} else if (srcImage.getNDims() == 3) {
			   try {
					// image length is length in 3 dims
					int length = srcImage.getExtents()[0] 
                                * srcImage.getExtents()[1] * srcImage.getExtents()[2];
                    
                    // retrieve all data: for each volume
                    buffer = new float[length];
                    srcImage.exportData(0,length, buffer); // locks and releases lock
					fireProgressStateChanged(srcImage.getImageName(), "Processing image ...");
				} catch (IOException error) {
					buffer = null;
					error.printStackTrace();
					errorCleanUp("Algorithm: source image locked", true);
					return;
				} catch (OutOfMemoryError e){
					buffer = null;
					e.printStackTrace();
					errorCleanUp("Algorithm: Out of memory creating process buffer", true);
					return;
				}
		
				// init dimensions
				nx = srcImage.getExtents()[0];
				ny = srcImage.getExtents()[1];
				nz = srcImage.getExtents()[2];
				rx = srcImage.getFileInfo()[0].getResolutions()[0];
				ry = srcImage.getFileInfo()[0].getResolutions()[1];
				rz = srcImage.getFileInfo()[0].getResolutions()[2];
				dim = 3;
				
                // main algorithm
				calcSegmentation(buffer);
				
			}
        }

        // compute the elapsed time
        computeElapsedTime();

    } // end run()

    /**
    *	produces a RFCM fuzzy segmentation of the input images
    */
    private void calcSegmentation(float img[]){
		boolean[][][]			objectMask;
		float[][][]				image;
		int                     x,y,z,t,k,m,l;
		float                   dist;
		int                     indx;
        float[]                 buffer;
        byte[]                 	bytebuffer;
		int[]					id;
        SegmentationFCM			segmentation;
		InhomogeneityCorrection2	correction;
		EdgeAdaptation			edge;
		ClusterSearch			search;

		int mod;
		int progress;

		boolean stop,stopReg;
		int n,Nt;
		float distance;
		
        if (verbose) MedicUtilPublic.displayMessage("\n -- FANTASM Segmentation --\n");
		
		// increase the dimension to make boundaries
		nx = nx+2; ny = ny+2; nz = nz+2;
		// pre-processing : expand boundaries, so that we don't have to worry for them
        try {
            image = expandImage(img);
            img = null;
        } catch (OutOfMemoryError e) {
            img = null; image = null;
            e.printStackTrace();
            errorCleanUp("Algorithm: Out of memory creating process buffer", true);
            setCompleted(false);
            return;
        }

		srcImage.calcMinMax();
		Imin = (float)srcImage.getMin();
		Imax = (float)srcImage.getMax();
		
		// parameter normalization
		if (useRelative) {
			//if (!adaptiveSmoothing) smoothing = smoothing*(Imax-Imin)*smoothing*(Imax-Imin);
			if (!adaptiveSmoothing) smoothing = smoothing*(Imax-Imin)*(Imax-Imin);
			backThreshold = Imin + backThreshold*(Imax-Imin);
			outlier = outlier*(Imax-Imin);
			// shouldn't this be scale-free ?? not related to image intensities..
			edgeSmoothness = edgeSmoothness*(Imax-Imin)*(Imax-Imin);
			edgeContrast= edgeContrast*(Imax-Imin)*(Imax-Imin);
			edgePrior = edgePrior*(Imax-Imin)*(Imax-Imin);
		} else {
			//if (!adaptiveSmoothing) smoothing = smoothing*smoothing;
			//edgeSmoothness = edgeSmoothness*edgeSmoothness;
			//edgeContrast= edgeContrast*edgeContrast;
			//edgePrior = edgePrior*edgePrior;
		}
		
		// if use mask, process only parts where intensity > min (mask stripped areas)
        objectMask = createObjectMask(image, backThreshold);
		computeProcessingBoundaries(objectMask);
		mx = xN-x0+1+2;
        my = yN-y0+1+2;
        mz = zN-z0+1+2;
        // rescale images
		if (debug) MedicUtilPublic.displayMessage("shrink images..\n");
		image = reduceImageSize(image); 
        objectMask = reduceImageSize(objectMask); 
        if (debug) MedicUtilPublic.displayMessage("new dimensions: "+mx+"x"+my+"x"+mz+"\n");

        //***************************************//
		//*          MAIN ALGORITHM             *//
		//***************************************//
 
		// record time
		long start_time = System.currentTimeMillis();
		long inner_loop_time;

		// create the appropriate segmentation algorithm
		segmentation = new SegmentationFCM(image, objectMask, mx, my, mz,
											classes, clusters, 
											smoothing, fuzziness, outlier,
											distanceMode,
											userInterface, getProgressChangeListener());

		// create auxiliary algorithms
		if (correctInhomogeneity!=NONE) {
			/*
			// create the inhomogeneity correction algorithm
			correction = new InhomogeneityCorrection(fieldDegree, 
													InhomogeneityCorrection.IMAGE,
													image, objectMask, 
													segmentation.getMemberships(),
													segmentation.getCentroids(),
													mx,my,mz,1.0f,1.0f,1.0f,
													classes,clusters,dim,
													getProgressChangeListener());
			*/
			
			// create the inhomogeneity correction algorithm
			
			switch (correctInhomogeneity) {
				case IMAGE:	correction = new InhomogeneityCorrection2(image, 
													segmentation.getMemberships(),
													segmentation.getCentroids(),
													classes,clusters,
													objectMask,
													fieldDegree, 0.0f,
													InhomogeneityCorrection2.IMAGE,
													InhomogeneityCorrection2.GLOBAL,
													InhomogeneityCorrection2.CHEBYSHEV,
													InhomogeneityCorrection2.FIXED,
													InhomogeneityCorrection2.EQUAL_VARIANCE,
													fuzziness, null, null,
													mx,my,mz,rx,ry,rz,
													getProgressChangeListener()); break;
				case CENTROIDS:	correction = new InhomogeneityCorrection2(image, 
													segmentation.getMemberships(),
													segmentation.getCentroids(),
													classes,clusters,
													objectMask,
													fieldDegree, 0.0f,
													InhomogeneityCorrection2.CENTROIDS,
													InhomogeneityCorrection2.GLOBAL,
													InhomogeneityCorrection2.CHEBYSHEV,
													InhomogeneityCorrection2.FIXED,
													InhomogeneityCorrection2.EQUAL_VARIANCE,
													fuzziness, null, null,
													mx,my,mz,rx,ry,rz,
													getProgressChangeListener()); break;
				case SEPARATE_IMG: correction = new InhomogeneityCorrection2(image, 
													segmentation.getMemberships(),
													segmentation.getCentroids(),
													classes,clusters,
													objectMask,
													fieldDegree, 0.0f,
													InhomogeneityCorrection2.IMAGE,
													InhomogeneityCorrection2.SEPARATE,
													InhomogeneityCorrection2.CHEBYSHEV,
													InhomogeneityCorrection2.FIXED,
													InhomogeneityCorrection2.EQUAL_VARIANCE,
													fuzziness, null, null,
													mx,my,mz,rx,ry,rz,
													getProgressChangeListener()); break;
				default:	correction = new InhomogeneityCorrection2(image, 
													segmentation.getMemberships(),
													segmentation.getCentroids(),
													classes,clusters,
													objectMask,
													fieldDegree, 0.0f,
													InhomogeneityCorrection2.IMAGE,
													InhomogeneityCorrection2.GLOBAL,
													InhomogeneityCorrection2.CHEBYSHEV,
													InhomogeneityCorrection2.FIXED,
													InhomogeneityCorrection2.EQUAL_VARIANCE,
													fuzziness, null, null,
													mx,my,mz,rx,ry,rz,
													getProgressChangeListener()); break;
			}
			
			// integrate it into the segmentation
			if (correctInhomogeneity==SEPARATE_IMG) {
				segmentation.addSeparateInhomogeneityCorrection(correction.getFields());
			} else {
				segmentation.addInhomogeneityCorrection(correction.getField(),correctInhomogeneity);
			}
		} else correction = null;
		
		if (useEdges) {
			// create the edge map algorithm
			edge = new EdgeAdaptation(segmentation.getMemberships(),
											mx,my,mz,classes,clusters,
											edgeSmoothness,edgeContrast,edgePrior,
											userInterface, getProgressChangeListener());
			// integrate it into the segmentation
			segmentation.addEdgeMap(edge.getEdges());
		} else edge = null;
		
        // initial guess for centroids
		if (initMode.equals("modes")) {
			search = new ClusterSearch(clusters);
			search.computeHistogram(image,objectMask,mx,my,mz);
			search.findCentroids();
		
			segmentation.importCentroids(search.exportCentroids());

			search.finalize();
			search = null;	
		} else if (initMode.equals("range")) {
			float[] cent = new float[clusters];
			cent[0] = Imin + 0.5f*(Imax-Imin)/(float)clusters;
			for (k=1;k<clusters;k++)
				cent[k] = cent[k-1] + (Imax-Imin)/(float)clusters;
						
			segmentation.importCentroids(cent);
		} else if (initMode.equals("manual")) {
			float[] cent = new float[clusters];
			
			JDialogInitialCentroids dialogInitialCentroids = new JDialogInitialCentroids(srcImage.getParentFrame(), clusters, (float)srcImage.getMin(), (float)srcImage.getMax());
			// if the dialog is not used: revert to 'range' initialization
			if (dialogInitialCentroids.isCancelled()) {
				cent[0] = Imin + 0.5f*(Imax-Imin)/(float)clusters;
				for (k=1;k<clusters;k++)
					cent[k] = cent[k-1] + (Imax-Imin)/(float)clusters;
			} else {
				for (k=0;k<clusters;k++) {
					cent[k] = dialogInitialCentroids.getCentroids()[k];
				}
            }			
			segmentation.importCentroids(cent);
        }
       
        // initialize the segmentation
		segmentation.setMRF(0.0f);
		segmentation.computeMemberships();
		if (adaptiveSmoothing) {
			segmentation.setMRF(segmentation.computeEnergyFactor(smoothing));
			if (verbose) MedicUtilPublic.displayMessage("smoothing "+(Math.sqrt(segmentation.getMRF())/(Imax-Imin))+"\n");
		} else {
			segmentation.setMRF(smoothing);
		}
		// main iterations: compute the classes on the image
		
		// with inhomogeneity correction, two steps: first iterates until mild convergence without inhomogeneity correction
		float maxDist0 = (float)Math.sqrt(maxDistance);
		int iter0 = (int)Math.sqrt(iterations);
		distance = 0.0f;
		Nt = 0;
		n = 0;
		int Niterations = 1;
		if (correctInhomogeneity!=NONE) {
			for (int d=0;(d<=fieldDegree) && (Niterations<=iterations);d++) {
				stop = false;
				n = 0;
				// for stability sake: low smoothing at first
				if (!adaptiveSmoothing) segmentation.setMRF(smoothing*d/(float)fieldDegree);
				while ((!stop) && (!threadStopped)) {
					fireProgressStateChanged("iteration " + Niterations + " (max: " + distance + ")"); 
					fireProgressStateChanged(Math.round( (float)n/(float)iterations)*100);
					
					if (verbose) MedicUtilPublic.displayMessage("iteration " + Niterations + " (max: " + distance + ")\n");
					
					// update external fields
					if (correctInhomogeneity!=NONE) {
						// fixed degree
						correction.computeCorrectionField(d);
						// debug: 
						//if (d>0) return;
					}
					if (useEdges) {
						edge.computeEdgeMap();
					}
					
					// update centroids
					segmentation.computeCentroids();
					
					// update membership
					distance = segmentation.computeMemberships();
					
					if (adaptiveSmoothing) {
						segmentation.setMRF(segmentation.computeEnergyFactor(smoothing));
						if (verbose) MedicUtilPublic.displayMessage("smoothing "+(Math.sqrt(segmentation.getMRF())/(Imax-Imin))+"\n");
					}
					// check for segmentation convergence 
					n++;
					Niterations++;
					if (n >= iter0) stop = true;
					if (Niterations >= iterations) stop = true;
					if (distance < maxDist0) stop = true;            
				}
				if (d<fieldDegree) Nt += n;
			}
		}
		
 		// last round: iterate toward the fine limit
		stop = false;
		if (Niterations >= iterations) stop = true;
		while ((!stop) && (!threadStopped)) {
			fireProgressStateChanged("iteration " + Niterations + " (max: " + distance + ")");
			fireProgressStateChanged(Math.round( (float)n/(float)iterations)*100);

			if (verbose) MedicUtilPublic.displayMessage("iteration " + Niterations + " (max: " + distance + ")\n");
			
			// update external fields
			if (correctInhomogeneity!=NONE) {
				// fixed degree
				correction.computeCorrectionField(fieldDegree);
			}
			if (useEdges) {
				edge.computeEdgeMap();
			}
			
			// update centroids
			segmentation.computeCentroids();
			
			// update membership
			distance = segmentation.computeMemberships();
			
			if (adaptiveSmoothing) {
				segmentation.setMRF(segmentation.computeEnergyFactor(smoothing));
				if (verbose) MedicUtilPublic.displayMessage("smoothing "+(Math.sqrt(segmentation.getMRF())/(Imax-Imin))+"\n");
			}
			// check for segmentation convergence 
			n++;
			Niterations++;
			if (Niterations > iterations) stop = true;
			if (distance < maxDistance) stop = true;            
		}
		Nt += n;

		// order the classes in increasing order
		id = segmentation.computeCentroidOrder();
        
        // compute times
		if (verbose) MedicUtilPublic.displayMessage("total iterations: "+Nt+", total time: (milliseconds): " + (System.currentTimeMillis()-start_time)); 
		
        // extract results (segmentation and/or classes) and store them in destImage[]
		fireProgressStateChanged("creating result images...");
		
        int Ndest = 0;
        try {            
			if (!outputType.equals("hard_segmentation")) {
				fireProgressStateChanged("memberships...");
				for (k=0;k<classes;k++) {
					buffer = bufferFromImage(segmentation.exportMemberships()[k],0.0f);
					destImage[id[k+1]-1].importData(0, buffer, true);
					Ndest++;
				}
			}
			if (!outputType.equals("fuzzy_segmentation")) {
				fireProgressStateChanged("final classification...");
				bytebuffer = orderedBufferFromImage(segmentation.exportHardClassification(), id);
                destImage[Ndest].importData(0, bytebuffer, true);
				Ndest++;
			}
			segmentation.finalize();
			segmentation = null;
			if (outputType.equals("all_result_images")) {
				if ( (correctInhomogeneity!=NONE)
					&& (correctInhomogeneity!=SEPARATE_IMG) ) {
					fireProgressStateChanged("gain field...");
					buffer = bufferFromImage(correction.exportField(),1.0f);
					destImage[Ndest].importData(0, buffer, true);
					Ndest++;
					correction.finalize();
					correction = null;
				} else if (correctInhomogeneity==SEPARATE_IMG) {
					fireProgressStateChanged("gain field...");
					for (k=0;k<clusters;k++) {
						buffer = bufferFromImage(correction.exportFields(k),1.0f);
						destImage[Ndest].importData(0, buffer, true);
						Ndest++;
					}
					correction.finalize();
					correction = null;
				}
				if (useEdges) {
					fireProgressStateChanged("edge maps...");
					buffer = bufferFromImage(edge.exportEdges()[0],0.0f);
					destImage[Ndest].importData(0, buffer, true);
					Ndest++;
					buffer = bufferFromImage(edge.exportEdges()[1],0.0f);
					destImage[Ndest].importData(0, buffer, true);
					Ndest++;
					buffer = bufferFromImage(edge.exportEdges()[2],0.0f);
					destImage[Ndest].importData(0, buffer, true);
					Ndest++;
					edge.finalize();
					edge = null;
				}
			}
			bytebuffer = null;
			buffer = null;
		} catch (OutOfMemoryError e) {
            bytebuffer = null;
			buffer = null;
			e.printStackTrace();
            errorCleanUp("Algorithm: Out of memory creating hard classification", true);
            finalize();
            setCompleted(false);
            return;
        } catch (IOException error) {
        	error.printStackTrace();
            errorCleanUp("Algorithm: export problem to destImage[]", true);
            finalize();
			setCompleted(false);
            return;
        }
        
        threadStopped = false;
		setCompleted(true);
    } // calcSegmentation
	
	/** brings image in correct array */
	private float[][][] expandImage(float[] image) {
		int 		x,y,z;
		float[][][] 	tmp;
		
		tmp = new float[nx][ny][nz];
		for (x=1;x<nx-1;x++)
			for (y=1;y<ny-1;y++)
				for (z=1;z<nz-1;z++)
					tmp[x][y][z] = image[ (x-1) + (nx-2)*(y-1) + (nx-2)*(ny-2)*(z-1) ];
		
		return tmp;
	}
	
	/** creates a mask for unused data */
	private boolean[][][] createObjectMask(float[][][] image, float val) {
		int 		x,y,z;
		boolean[][][]  	objMask;
        boolean useWholeImage = true;

		// uses only values over the threshold, if mask used
		objMask = new boolean[nx][ny][nz];
		for (x=1;x<nx-1;x++)
			for (y=1;y<ny-1;y++)
				for (z=1;z<nz-1;z++) {
                    if (useWholeImage) {
                        if ( (cropBackground) && (image[x][y][z] <= val) )
                            objMask[x][y][z] = false;
                        else
                            objMask[x][y][z] = true;
                    } else if (mask.get((x-1)+(nx-2)*(y-1)+(nx-2)*(ny-2)*(z-1)) ) {
                        if ( (cropBackground) && (image[x][y][z] <= val) )
                            objMask[x][y][z] = false;
                        else
                            objMask[x][y][z] = true;
                    }
                }
		// remove the boundary from the computations
		for (x=0;x<nx;x++)
			for (y=0;y<ny;y++) {
				objMask[x][y][0] = false;
				objMask[x][y][nz-1] = false;
			}
		for (y=0;y<ny;y++)
			for (z=0;z<nz;z++) {
				objMask[0][y][z] = false;
				objMask[nx-1][y][z] = false;
			}
		for (z=0;z<nz;z++)
			for (x=0;x<nx;x++) {
				objMask[x][0][z] = false;
				objMask[x][ny-1][z] = false;
			}

		return objMask;
	} // createObjectMask
    
    /** sets the processing lower and upper boundaries */
    private void computeProcessingBoundaries(boolean[][][] objMask) {
		int 		x,y,z;
        
        x0 = nx;
        xN = 0;
        y0 = ny;
        yN = 0;
        z0 = nz;
        zN = 0;
        for (x=0;x<nx;x++)
			for (y=0;y<ny;y++)
				for (z=0;z<nz;z++) {
                    if (objMask[x][y][z]) {
                        if (x < x0) x0 = x;
                        if (x > xN) xN = x;
                        if (y < y0) y0 = y;
                        if (y > yN) yN = y;
                        if (z < z0) z0 = z;
                        if (z > zN) zN = z;
                    }
                }
				
        // debug
        System.out.print("boundaries: ["+x0+","+xN+"] ["+y0+","+yN+"] ["+z0+","+zN+"]\n");
        
        return;
    }

    /** create smaller image (for saving memory) */
	private float[][][] reduceImageSize(float[][][] image) {
		float[][][] smaller = new float[mx][my][mz];

		for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
			smaller[x][y][z] = 0.0f;
		}
		for (int x=x0;x<=xN;x++) {
            for (int y=y0;y<=yN;y++) {
                for (int z=z0;z<=zN;z++) {
					smaller[x-x0+1][y-y0+1][z-z0+1] = image[x][y][z];
				}
			}
		}
		return smaller;
	}
	private boolean[][][] reduceImageSize(boolean[][][] image) {
		boolean[][][] smaller = new boolean[mx][my][mz];

		for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
			smaller[x][y][z] = false;
		}
		for (int x=x0;x<=xN;x++) {
            for (int y=y0;y<=yN;y++) {
                for (int z=z0;z<=zN;z++) {
					smaller[x-x0+1][y-y0+1][z-z0+1] = image[x][y][z];
				}
			}
		}
		return smaller;
	}

	/** retrieve original size from smaller image */
	private float[] bufferFromImage(float[][][] image, float bg) {
		float[] larger = new float[(nx-2)*(ny-2)*(nz-2)];

		for (int n=0;n<(nx-2)*(ny-2)*(nz-2);n++)
			larger[n] = bg;
		
		for (int x=x0;x<=xN;x++) {
            for (int y=y0;y<=yN;y++) {
                for (int z=z0;z<=zN;z++) {
					larger[(x-1)+(nx-2)*(y-1)+(nx-2)*(ny-2)*(z-1)] = image[x-x0+1][y-y0+1][z-z0+1];
				}
			}
		}
		return larger;
	}

	/** retrieve original size from smaller image */
	private byte[] orderedBufferFromImage(byte[][][] image, int[] id) {
		byte[] larger = new byte[(nx-2)*(ny-2)*(nz-2)];

		for (int n=0;n<(nx-2)*(ny-2)*(nz-2);n++)
			larger[n] = 0;
		
		for (int x=x0;x<=xN;x++) {
            for (int y=y0;y<=yN;y++) {
                for (int z=z0;z<=zN;z++) {
					larger[(x-1)+(nx-2)*(y-1)+(nx-2)*(ny-2)*(z-1)] = 0;
					for (int k=0;k<classes+1;k++)
						if (image[x-x0+1][y-y0+1][z-z0+1]==id[k]) 
							larger[(x-1)+(nx-2)*(y-1)+(nx-2)*(ny-2)*(z-1)] = (byte)k;
				}
			}
		}
		return larger;
	}

}