package edu.jhmi.rad.medic.dialogs;

import edu.jhmi.rad.medic.algorithms.*;

import gov.nih.mipav.plugins.*;     
import gov.nih.mipav.view.*;
import gov.nih.mipav.view.dialogs.*;
import gov.nih.mipav.model.structures.*;
import gov.nih.mipav.model.file.*;
import gov.nih.mipav.model.algorithms.*;
import gov.nih.mipav.model.scripting.*;
import gov.nih.mipav.model.scripting.parameters.*;

import java.awt.event.*;
import java.awt.*;
import java.util.*;
import java.io.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

/** 
*   
*   Dialog class for the segmentation plug-in using K-means, EM, or FCM
*
*
*	@version    November 2006
*	@author     Pilou Bazin
*   @see        AlgorithmFantasm
*   @see        AlgorithmKmeansSegmentation
*   @see        AlgorithmEMSegmentation
*   @see        AlgorithmEMvSegmentation
*   @see        AlgorithmEMpvSegmentation
*
*
*/  
public class JDialogTissueSegmentation extends JDialogScriptableBase implements AlgorithmInterface, DialogDefaultsInterface {

    
    private     AlgorithmFantasm			algo = null;
    private     AlgorithmKmeansSegmentation		algo2 = null;
    private     AlgorithmEMSegmentation			algo3 = null;
    private 	AlgorithmEMSegmentation		        algo4 = null;
    private 	AlgorithmEMSegmentation		        algo5 = null;
    private     ModelImage              image;                // reference image
    private     ModelImage              images[];             // source images
    private     ModelImage              resultImage[] = null; // result images
    private     int                     resultNumber;
    private     int                     imageNumber;
    private     int                     presentNumber;
    private     int                     destExtents[];
    private 	ViewUserInterface       userInterface;
    private     String                  titles[];
	private		int[]					order;
	private		int						nOrder;
	private		int						borders=0;

	// parameters
	// classification
	private 	int 		nClasses		= 3;
	private 	int 		nIterations		= 20;
	private 	float 		maxDiff			= 0.01f;
	private 	float 		smoothing		= 0.01f;
	private		float		fuzziness		= 2.0f;
    private		boolean		addOutliers		= false;
	private		float		outlierRatio	= 0.5f;
    private     String      algoType = "FANTASM";
	private     String[]    algoTypes = {"FANTASM","K-means","EM","EMv","EMpv"};
    private     String      initMode = "range";
	private     String[]    initModes = {"range","modes","manual"};
    private     String      maskingMode = "one";
	private     String[]    maskingModes = {"one","all"};
	private     String      dataType = "single_image";
	private     String[]    dataTypes = {"single_image","multichannel"};
	
	// temporal / multi channel
	private 	float 		temporal		= 0.05f;

	// registration
	private		boolean		useRegistration = false;
	private		int			topIter	= 10;
	private		int			firstIter	= 2;
	private		int			lastIter	= 3;
	private		int			levels		= 3;
	
	// inhomogeneity
	private     String      correctType = "centroids";
	private     String[]    correctTypes = {"image","centroids","separate"};
    private		int			polynomialDegree		= 3;
	private		String[]	polynomialDegrees		= {"1","2","3","4"};
	
	// edges
	private		boolean		useEdges = false;
	private		float		edgeSmoothness = 0.05f;
	private		float		edgeContrast = 0.05f;
	private		float		edgePrior = 0.001f;
	
	// general
    private     float       backThreshold	= 0.0f;
    private     boolean     cropBackground	= true;
	private		boolean		useRelative		= true;
    private     String      outputType = "hard_segmentation";
	private     String[]    outputTypes = {"all_result_images", "hard_segmentation", "fuzzy_segmentation", "both_fuzzy_&_hard"};
	        
    // dialog elements
	private		JTabbedPane	tabPane;
	private		JPanel		mainPanel;
	private		JPanel		imagePanel;
	private		JPanel		parameterPanel;
    
	private 	JPanel  	classifPanel;
	private 	JTextField  textNClasses;
	private 	JLabel  	labelNClasses;
	private		JComboBox	comboType;
	private		JLabel		labelType;
	
	private		JPanel		classifParamPanel;
	private 	JTextField  textMaxIter;
	private 	JLabel  	labelMaxIter;
	private 	JTextField  textMaxDiff;
	private 	JLabel  	labelMaxDiff;
	private 	JTextField  textSmoothing;
	private 	JLabel  	labelSmoothing;
	private 	JTextField  textFuzzy;
	private 	JLabel  	labelFuzzy;
	private 	JComboBox   comboInit;
	private 	JLabel  	labelInit;
	private 	JComboBox   comboMasking;
	private 	JLabel  	labelMasking;
	
    private		JPanel		generalPanel;
	private     JCheckBox   checkCrop;
	private		JComboBox	comboResult;
	private		JLabel		labelResult;
    
	private		JPanel		generalParamPanel;
	private 	JTextField  textSignal;
	private 	JLabel  	labelSignal;
    private     JCheckBox   checkRelative;
	
	private		JPanel		optionPanel;
	private     JCheckBox   checkInhomog;
	
	private		JPanel		optionParamPanel;
	private		JComboBox	comboPolyDeg;
	private		JLabel		labelPolyDeg;
    private 	JComboBox   comboCorrection;
	private 	JLabel  	labelCorrection;
    
	// label list
	private		JPanel			imageListPanel;
	private		JCheckBox[]		checkImage;
	private		JLabel[]		labelOrder;
	private		int				nImage = 0;
	private		JLabel			labelImage;
	private		JPanel			viewPanel;
	private		JScrollPane		scrollImage;

    /**
    *  Creates dialog for plugin.
    *  @param theParentFrame          Parent frame.
    *  @param im              Source image.
    */
    public JDialogTissueSegmentation(Frame theParentFrame, ModelImage im) {
		super(theParentFrame, true);
		if (im.getType() == ModelImage.BOOLEAN || im.isColorImage()) {
            MipavUtil.displayError("Source Image must NOT be Boolean or Color"); 
            dispose();
            return;
        }
        image = im;
		userInterface = ((ViewJFrameBase)(parentFrame)).getUserInterface();	    
		loadDefaults();
        init();
        setVisible(true);
	}
	
    /**
    *	Used primarily for the script to store variables and run the algorithm.  No
    *	actual dialog will appear but the set up info and result image will be stored here.
    *	@param UI   The user interface, needed to create the image frame.
    *	@param im	Source image.
    */
    public JDialogTissueSegmentation(ViewUserInterface UI, ModelImage im) {
        super();
    	userInterface = UI;
    	image = im;
        parentFrame = image.getParentFrame();
    }
    
    /**
     * Empty constructor needed for dynamic instantiation.
     */
    public JDialogTissueSegmentation() {};

	/**
     * Construct a delimited string that contains the parameters to this algorithm.
     * @param delim  the parameter delimiter (defaults to " " if empty)
     * @return       the parameter string
     */
	public String getParameterString( String delim ) {
        if ( delim.equals( "" ) ) {
            delim = " ";
        }

        String str = new String();
        str += nClasses + delim;
        str += nIterations + delim;
        str += maxDiff + delim;
        str += smoothing + delim;
        str += fuzziness + delim;
        str += addOutliers + delim;
        str += outlierRatio + delim;
        str += algoType + delim;
        str += initMode + delim;
        str += maskingMode + delim;
        str += temporal + delim;
        str += useRegistration + delim;
        str += topIter + delim;
        str += firstIter + delim;
        str += lastIter + delim;
        str += levels + delim;
        str += correctType + delim;
        str += polynomialDegree + delim;
        str += useEdges + delim;
        str += edgeSmoothness + delim;
        str += edgeContrast + delim;
        str += edgePrior + delim;
        str += backThreshold + delim;
        str += cropBackground + delim;
        str += useRelative + delim;
        str += outputType;

        return str;
    }
	

    /**
     *  Loads the default settings from Preferences to set up the dialog
     */
	public void loadDefaults() {
        String defaultsString = Preferences.getDialogDefaults(getDialogName());

        if (defaultsString != null) {

            try {
                System.out.println(defaultsString);
                StringTokenizer st = new StringTokenizer(defaultsString, ",");
				nClasses = MipavUtil.getInt(st);
				nIterations = MipavUtil.getInt(st);
				maxDiff = MipavUtil.getFloat(st);
				smoothing = MipavUtil.getFloat(st);
				fuzziness = MipavUtil.getFloat(st);
				addOutliers = MipavUtil.getBoolean(st);
				outlierRatio = MipavUtil.getFloat(st);
				algoType = st.nextToken();
				initMode = st.nextToken();
				maskingMode = st.nextToken();
				temporal = MipavUtil.getFloat(st);
				useRegistration = MipavUtil.getBoolean(st);
				topIter = MipavUtil.getInt(st);
				firstIter = MipavUtil.getInt(st);
				lastIter = MipavUtil.getInt(st);
				levels = MipavUtil.getInt(st);
				correctType = st.nextToken();
				polynomialDegree = MipavUtil.getInt(st);
				useEdges = MipavUtil.getBoolean(st);
				edgeSmoothness = MipavUtil.getFloat(st);
				edgeContrast = MipavUtil.getFloat(st);
				edgePrior = MipavUtil.getFloat(st);
				backThreshold = MipavUtil.getFloat(st);
				cropBackground = MipavUtil.getBoolean(st);
				useRelative = MipavUtil.getBoolean(st);
				outputType = st.nextToken();
            }
            catch (Exception ex) {
                // since there was a problem parsing the defaults string, start over with the original defaults
                System.out.println( "Resetting defaults for dialog: " + getDialogName() );
                Preferences.removeProperty( getDialogName() );
            }
        } else {
			System.out.println( "no saved dialogs for "+getDialogName() );
		}
    }
		
    /**
     * Saves the default settings into the Preferences file
     */
	
    public void saveDefaults() {
        String defaultsString = new String( getParameterString(",") );
        System.out.println(defaultsString);
        Preferences.saveDialogDefaults(getDialogName(),defaultsString);
    }
	 

    /**
    *	Sets up the GUI (panels, buttons, etc) and displays it on the screen.
    */
	private void init(){
        setForeground(Color.black);
		setTitle("Tissue Segmentation");
				
        labelNClasses = new JLabel("Number of classes ");
		labelNClasses.setForeground(Color.black);
		labelNClasses.setFont(serif12);

		textNClasses = new JTextField(5);
		textNClasses.setText(String.valueOf(nClasses));
		textNClasses.setFont(serif12);
		        		
 		labelType = new JLabel("Algorithm ");
		labelType.setForeground(Color.black);
		labelType.setFont(serif12);

		comboType = new JComboBox(algoTypes);
		comboType.setFont(serif12);
        comboType.setBackground(Color.white);
		comboType.addActionListener(this);
		comboType.setActionCommand("algoType");
		comboType.setSelectedItem(algoType);
		
		labelMaxIter = new JLabel("Maximum iterations ");
		labelMaxIter.setForeground(Color.black);
		labelMaxIter.setFont(serif12);
	
		textMaxIter = new JTextField(5);
		textMaxIter.setText(String.valueOf(nIterations));
		textMaxIter.setFont(serif12);
	
		labelMaxDiff = new JLabel("Maximum difference ");
		labelMaxDiff.setForeground(Color.black);
		labelMaxDiff.setFont(serif12);
	
		textMaxDiff = new JTextField(5);
		textMaxDiff.setText(String.valueOf(maxDiff));
		textMaxDiff.setFont(serif12);
	
		labelSmoothing = new JLabel("Smoothing parameter ");
		labelSmoothing.setForeground(Color.black);
		labelSmoothing.setFont(serif12);
	
		textSmoothing = new JTextField(5);
		textSmoothing.setText(String.valueOf(smoothing));
		textSmoothing.setFont(serif12);
	
        labelFuzzy = new JLabel("Fuzziness coefficient ");
		labelFuzzy.setForeground(Color.black);
		labelFuzzy.setFont(serif12);
	
		textFuzzy = new JTextField(5);
		textFuzzy.setText(String.valueOf(fuzziness));
		textFuzzy.setFont(serif12);
	
        labelInit = new JLabel("Initialization ");
		labelInit.setForeground(Color.black);
		labelInit.setFont(serif12);
	
		comboInit = new JComboBox(initModes);
		comboInit.setFont(serif12);
        comboInit.setBackground(Color.white);
		comboInit.setSelectedItem(initMode);
	
        labelMasking = new JLabel("Masking mode ");
		labelMasking.setForeground(Color.black);
		labelMasking.setFont(serif12);
	
		comboMasking = new JComboBox(maskingModes);
		comboMasking.setFont(serif12);
        comboMasking.setBackground(Color.white);
		comboMasking.setSelectedItem(maskingMode);
	
		checkInhomog = new JCheckBox("Correct inhomogeneity");
        checkInhomog.setFont(serif12);
		if (correctType.equals("none"))
			checkInhomog.setSelected(false);
		else
			checkInhomog.setSelected(true);
                
		labelPolyDeg = new JLabel("Inhomogeneity field degree ");
		labelPolyDeg.setForeground(Color.black);
		labelPolyDeg.setFont(serif12);
	
		comboPolyDeg = new JComboBox(polynomialDegrees);
		comboPolyDeg.setFont(serif12);
        comboPolyDeg.setBackground(Color.white);
		comboPolyDeg.setSelectedItem(""+polynomialDegree);
						
        labelCorrection = new JLabel("Inhomogeneity mode ");
		labelCorrection.setForeground(Color.black);
		labelCorrection.setFont(serif12);
	
		comboCorrection = new JComboBox(correctTypes);
		comboCorrection.setFont(serif12);
        comboCorrection.setBackground(Color.white);
		comboCorrection.setSelectedItem(correctType);
	
		checkCrop = new JCheckBox("Background cropping");
        checkCrop.setFont(serif12);
        checkCrop.setSelected(cropBackground);
        
		labelResult = new JLabel("Output ");
		labelResult.setForeground(Color.black);
		labelResult.setFont(serif12);
	
		comboResult = new JComboBox(outputTypes);
		comboResult.setSelectedItem(outputType);
		comboResult.setFont(serif12);
        comboResult.setBackground(Color.white);
				
        labelSignal = new JLabel("Background threshold ");
        labelSignal.setForeground(Color.black);
        labelSignal.setFont(serif12);

        textSignal = new JTextField(5);
        textSignal.setText(String.valueOf(backThreshold));
        textSignal.setFont(serif12);
        
        checkRelative = new JCheckBox("Use relative values");
        checkRelative.setFont(serif12);
        checkRelative.setSelected(useRelative);

        scrollImage = new JScrollPane();
		scrollImage.setMaximumSize(new Dimension(200,200));
		scrollImage.setPreferredSize(new Dimension(100,100));
		scrollImage.setMinimumSize(new Dimension(30,30));
		displayImageList();

		mainPanel = new JPanel(new GridBagLayout());
		imagePanel = new JPanel(new GridBagLayout());
		parameterPanel = new JPanel(new GridBagLayout());
        
		// put all together
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridwidth = 1;
        gbc.gridheight = 1;
        gbc.anchor = gbc.WEST;
        gbc.insets = new Insets(0, 0, 0, 0);

		// classification panel
        gbc.gridx = 0; 
		gbc.gridy = 0;
        gbc.weightx = 1; 
		gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        mainPanel.add(labelNClasses, gbc);
        gbc.gridx = 1; 
		gbc.gridy = 0;
        gbc.weightx = 0; 
		gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        mainPanel.add(textNClasses, gbc);
        gbc.gridx = 0; 
		gbc.gridy = 1;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        mainPanel.add(labelType, gbc);
        gbc.gridx = 1;
        gbc.gridy = 1;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        mainPanel.add(comboType, gbc);
		gbc.gridx = 1;
        gbc.gridy = 2;
        gbc.gridwidth = 2;
        gbc.anchor = gbc.WEST;
        gbc.fill = gbc.NONE;
        mainPanel.add(checkInhomog, gbc);
        gbc.gridx = 1;
        gbc.gridy = 3;
        gbc.gridwidth = 2;
        gbc.anchor = gbc.WEST;
        gbc.fill = gbc.NONE;
        mainPanel.add(checkCrop, gbc);
        gbc.gridx = 0;
        gbc.gridy = 4;
        gbc.gridwidth = 1;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        mainPanel.add(labelResult, gbc);
        gbc.gridx = 1;
        gbc.gridy = 4;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        mainPanel.add(comboResult, gbc);
		
		// classif params
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        parameterPanel.add(labelSmoothing, gbc);
        gbc.gridx = 1;
        gbc.gridy = 0;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        parameterPanel.add(textSmoothing, gbc);
        gbc.gridx = 0;
        gbc.gridy = 2;
        gbc.weightx = 1;
        gbc.anchor = gbc.WEST;
        gbc.fill = gbc.HORIZONTAL;
        parameterPanel.add(labelMaxDiff, gbc);
        gbc.gridx = 1;
        gbc.gridy = 2;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        parameterPanel.add(textMaxDiff, gbc);
        gbc.gridx = 0;
        gbc.gridy = 3;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        gbc.anchor = gbc.EAST;
        parameterPanel.add(labelMaxIter, gbc);
        gbc.gridx = 1;
        gbc.gridy = 3;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        parameterPanel.add(textMaxIter, gbc);
        gbc.gridx = 0;
        gbc.gridy = 4;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        parameterPanel.add(labelInit, gbc);
        gbc.gridx = 1;
        gbc.gridy = 4;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        parameterPanel.add(comboInit, gbc);
        gbc.gridx = 0;
        gbc.gridy = 5;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        parameterPanel.add(labelMasking, gbc);
        gbc.gridx = 1;
        gbc.gridy = 5;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        parameterPanel.add(comboMasking, gbc);
		gbc.gridx = 0;
        gbc.gridy = 6;
        gbc.gridwidth = 1;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        parameterPanel.add(labelPolyDeg, gbc);
        gbc.gridx = 1;
        gbc.gridy = 6;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        parameterPanel.add(comboPolyDeg, gbc);
		gbc.gridx = 0;
        gbc.gridy = 7;
        gbc.gridwidth = 1;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        parameterPanel.add(labelCorrection, gbc);
        gbc.gridx = 1;
        gbc.gridy = 7;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        parameterPanel.add(comboCorrection, gbc);
		gbc.gridx = 0;
        gbc.gridy = 8;
        gbc.gridwidth = 1;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        parameterPanel.add(labelSignal, gbc);
        gbc.gridx = 1;
        gbc.gridy = 8;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        parameterPanel.add(textSignal, gbc);
		gbc.gridx = 0;
        gbc.gridy = 9;
        gbc.gridwidth = 1;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        parameterPanel.add(labelFuzzy, gbc);
        gbc.gridx = 1;
        gbc.gridy = 9;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        parameterPanel.add(textFuzzy, gbc);
		
		// image panel
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.gridwidth = 1;
        gbc.weightx = 1;
        gbc.fill = gbc.BOTH;
        imagePanel.add(scrollImage, gbc);
		
		// assemble everything
        tabPane = new JTabbedPane();
        tabPane.setForeground(Color.black);
		
        tabPane.add("main",mainPanel);
        //tabPane.add("images",imagePanel);
        tabPane.add("parameters",parameterPanel);
        
        getContentPane().add(tabPane);
        getContentPane().add(buildButtons(), BorderLayout.SOUTH);

        pack();
        //setVisible(true); 
		//setResizable(false);
    	System.gc();
		
	} // end init()
	
	void displayImageList() {
		int j;
        
        Enumeration names = userInterface.getRegisteredImageNames();

        // Add images from user interface that have the same exact dimensionality
        // ! note: if the images have different dimensions, problems occur in Algorithms
        
		// 1. count the images
		nImage = 0;
		while (names.hasMoreElements()) {
            String name = (String) names.nextElement();
            ModelImage img = userInterface.getRegisteredImageByName(name);
			if (userInterface.getFrameContainingImage(img) != null) {
				if ( (image.getNDims() == img.getNDims()) && (!img.isColorImage()) ) {
					nImage++;
				}
            }
        }
		// 2. create the checkboxes
		checkImage = new JCheckBox[nImage];
		labelOrder = new JLabel[nImage];
		order = new int[nImage];
		for (int n=0;n<nImage;n++) order[n] = -1;
		int n = 0;
		names = userInterface.getRegisteredImageNames();
		while (names.hasMoreElements()) {
            String name = (String) names.nextElement();
            ModelImage img = userInterface.getRegisteredImageByName(name);
			if (userInterface.getFrameContainingImage(img) != null) {
				if ( (image.getNDims() == img.getNDims()) && (!img.isColorImage()) ) {
					checkImage[n] = new JCheckBox(name);
					checkImage[n].setFont(serif12);
					checkImage[n].setSelected(false);
					checkImage[n].addActionListener(this);
					checkImage[n].setActionCommand("Image "+n);
					
					labelOrder[n] = new JLabel(".");
					labelOrder[n].setFont(serif12);
					labelOrder[n].setBackground(Color.white);
					
					if (dataType.equals("single_image")) {
						checkImage[n].setEnabled(false);
						labelOrder[n].setEnabled(false);
					} else {
						checkImage[n].setEnabled(true);
						labelOrder[n].setEnabled(true);
					}
					n++;
				}
            }
        }
		// display
		viewPanel = new JPanel(new GridBagLayout());
		GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridwidth = 1;
        
		for (n=0;n<nImage;n++) {
			gbc.gridy = n;
			gbc.gridx = 0;
			gbc.fill = gbc.NONE;
			gbc.weightx = 0;
			viewPanel.add(labelOrder[n], gbc);
			gbc.gridx = 1;
			gbc.fill = gbc.HORIZONTAL;
			gbc.weightx = 1;
			viewPanel.add(checkImage[n], gbc);
		}
		scrollImage.setViewportView(viewPanel);
		scrollImage.revalidate();
		pack();
    }

	
    /**
    *  Accessor that returns the image.
    *  @return          The result image.
    */
    public ModelImage[] getResultImage() {
        return resultImage;
    }

    /**
    *	Accessor that sets the parameters (empty)
    */
    public void setParameters() {
    }
	
    //************************************************************************
    //************************** Event Processing ****************************
    //************************************************************************

	/**
	*  Closes dialog box when the OK button is pressed and calls the algorithm.
	*  @param event       Event that triggers function.
	*/
	public void actionPerformed(ActionEvent event) {
		String command = event.getActionCommand();
	
		if (command.equals("OK")) {
			if (setVariables()) { 
				callAlgorithm();
			}    
		} else if (command.equals("Cancel")) {
			dispose();
		} else if (command.equals("Help")) {
            MipavUtil.showHelp("10027");
        } else if (command.equals("algoType")) {
			algoType = (String)comboType.getSelectedItem();
			if (dataType.equals("single_image")) {
				for (int n=0;n<nImage;n++) {
					checkImage[n].setEnabled(false);
					labelOrder[n].setEnabled(false);
				}
			} else {
				for (int n=0;n<nImage;n++) {
					checkImage[n].setEnabled(true);
					labelOrder[n].setEnabled(true);
				}
			}
		} else {
			for (int n=0;n<nImage;n++) {
				if (command.equals("Image "+n)) {
					if (checkImage[n].isSelected()) {
						order[n] = nOrder;
						labelOrder[n].setText(""+(nOrder+1));
						nOrder++;
					} else {
						for (int m=0;m<nImage;m++) {
							if (order[m]>order[n]) {
								order[m]--;
								labelOrder[m].setText(""+(order[m]+1));
							}
						}
						order[n] = -1;
						labelOrder[n].setText(".");
						nOrder--;
					}
				}
			}
		}
		return;		
    }

    //************************************************************************
    //************************** Algorithm Events ****************************
    //************************************************************************
    
    /** 
    *	This method is required if the AlgorithmPerformed interface is implemented. 
    *   It is called by the algorithm when it has completed or failed to to complete, 
    *   so that the dialog can be display the result image and/or clean up.
    *   @param algorithm   Algorithm that caused the event.
    */
    public void algorithmPerformed(AlgorithmBase algorithm) {
        
       if (Preferences.isPreference(Preferences.PREF_SAVE_DEFAULTS) && this.getOwner() != null && !isScriptRunning()) {
          saveDefaults();
       }
	   
	   int i;
	   int Nimg;
        
		if (dataType.equals("longitudinal")) Nimg = imageNumber;
		else Nimg = 1;
        ViewJFrameImage imageFrame[] = new ViewJFrameImage[resultNumber*imageNumber];
        if ((algorithm instanceof AlgorithmFantasm) || (algorithm instanceof AlgorithmKmeansSegmentation) || 
	(algorithm instanceof AlgorithmEMSegmentation) || (algorithm instanceof AlgorithmEMvSegmentation) || 
	(algorithm instanceof AlgorithmEMpvSegmentation)) {
            image.clearMask();
			if(algorithm.isCompleted() == true && resultImage != null) {
                //The algorithm has completed and produced a new image to be displayed.
                for (i = 0; i < resultNumber*Nimg; i++) {
                    updateFileInfo(image, resultImage[i]);
                    resultImage[i].clearMask();

                    try {
                        imageFrame[i] = new ViewJFrameImage(resultImage[i], null,
                                           new Dimension(610, 200 + i * 20));
                    } catch (OutOfMemoryError error) {
                        System.gc();
                        JOptionPane.showMessageDialog(null, 
                                                "Out of memory: unable to open new frame",
                                                "Error", JOptionPane.ERROR_MESSAGE);
                    }
                 }
            } else if (resultImage != null) {
                //algorithm failed but result image still has garbage
                for (i = 0; i < resultNumber*Nimg; i++) {
                    if (resultImage[i] != null) {
                        resultImage[i].disposeLocal(); // Clean up memory of result image
                        resultImage[i] = null;
                    }
                }
                resultImage = null;
                System.gc();
			}
			if (algorithm.isCompleted()) {
				insertScriptLine();
			}
       }
       algorithm.finalize();
       algorithm = null;
       dispose();
    }  // end AlgorithmPerformed()
    
  
    /**
    *	Use the GUI results to set up the variables needed to run the algorithm.
    *	@return		<code>true</code> if parameters set successfully, <code>false</code> otherwise.
    */
    /**
    *	Use the GUI results to set up the variables needed to run the algorithm.
    *	@return		<code>true</code> if parameters set successfully, <code>false</code> otherwise.
    */
    private boolean setVariables() {
    	String tmpStr;
             	        
       	tmpStr = textNClasses.getText();
        if ( testParameter(tmpStr, 1, 20) ){
            nClasses = Integer.valueOf(tmpStr).intValue();
        } else {
            textNClasses.requestFocus();
            textNClasses.selectAll();
            return false;
        }
       	tmpStr = textMaxIter.getText();
        if ( testParameter(tmpStr, 0, 1000) ){
            nIterations = Integer.valueOf(tmpStr).intValue();
        } else {
            textMaxIter.requestFocus();
            textMaxIter.selectAll();
            return false;
        }
       	tmpStr = textMaxDiff.getText();
        if ( testParameter(tmpStr, 0.0, 10.0) ){
            maxDiff = Float.valueOf(tmpStr).floatValue();
        } else {
            textMaxDiff.requestFocus();
            textMaxDiff.selectAll();
            return false;
        }
       	tmpStr = textSmoothing.getText();
        if ( testParameter(tmpStr, 0.0, 10000.0) ){
            smoothing = Float.valueOf(tmpStr).floatValue();
        } else {
            textSmoothing.requestFocus();
            textSmoothing.selectAll();
            return false;
        }
		tmpStr = textFuzzy.getText();
        if ( testParameter(tmpStr, 1.001, 10.0) ){
            fuzziness = Float.valueOf(tmpStr).floatValue();
        } else {
            textFuzzy.requestFocus();
            textFuzzy.selectAll();
            return false;
        }
		if (checkCrop.isSelected()) {
			cropBackground = true;
		} else {
			cropBackground = false;
		}
        tmpStr = textSignal.getText();
        if (testParameter(tmpStr, -100000.0, 100000.0)) {
          backThreshold = Float.valueOf(tmpStr).floatValue();
        } else {
          textSignal.requestFocus();
          textSignal.selectAll();
          return false;
        }
        if (checkRelative.isSelected()) {
			useRelative = true;
		} else {
			useRelative = false;
		}
        outputType = (String)comboResult.getSelectedItem();
		initMode = (String)comboInit.getSelectedItem();
		if (checkInhomog.isSelected()) {
			correctType = (String)comboCorrection.getSelectedItem();
		} else {
			correctType = "none";
		}
        polynomialDegree = Integer.valueOf((String)comboPolyDeg.getSelectedItem()).intValue();
		
		
		algoType = (String)comboType.getSelectedItem();
		
		if (dataType.equals("single_image")) {
			imageNumber = 1;
			images = new ModelImage[imageNumber];
			images[0] = image;
		} else {
			imageNumber = 0;
			for (int i=0;i<nImage;i++) 
				if (checkImage[i].isSelected() ) imageNumber++;
        
			if (imageNumber==0) {
				tabPane.setSelectedComponent(imagePanel);
				imageListPanel.requestFocus();
				return false;
			}
			images = new ModelImage[imageNumber];
			// put images in proper order
			for (int i=0;i<nImage;i++) if (checkImage[i].isSelected() ) {
				images[order[i]] = userInterface.getRegisteredImageByName(checkImage[i].getText());
			}
		}
    	return true;  	
    }   // end setVariables()
    
    /**
    *	Once all the necessary variables are set, call the Gaussian Blur
    *	algorithm based on what type of image this is and whether or not there
    *	is a separate destination image.
    */
    protected void callAlgorithm() {
        int i, Nimg;
        String name = (image.getImageName()+ "");
		AlgorithmBase algorithm = null;
		String suffix = "";
        
        // Calculate the number of result images.
        resultNumber = 0;
        if (outputType.equals("hard_segmentation")) resultNumber=1; 
		else if (outputType.equals("fuzzy_segmentation")) {
			resultNumber = nClasses;
			if (addOutliers) resultNumber++;
        } else {
			resultNumber = nClasses+1;
			if (addOutliers) resultNumber++;
			if (outputType.equals("all_result_images")) {
				if (correctType.equals("image") || correctType.equals("centroids") ) {
					if (dataType.equals("multichannel")) resultNumber+=imageNumber;
					else resultNumber++;
				} else if (correctType.equals("separate")) {
					if (dataType.equals("multichannel")) resultNumber+=imageNumber*nClasses;
					else resultNumber+=nClasses;
				}
				if (useEdges) resultNumber += 3;
			}
		}
		if (dataType.equals("longitudinal")) Nimg = imageNumber;
		else Nimg = 1;
		
		// add suffix for algorithm type
		if (algoType.equals("FANTASM")) suffix = "_fcm";
		else if (algoType.equals("EM")) suffix = "_em";
		else if (algoType.equals("EMv")) suffix = "_emv";
		else if (algoType.equals("EMpv")) suffix = "_empv";
		else if (algoType.equals("K-means")) suffix = "_km";
		
		// add suffix for data type
		if (dataType.equals("longitudinal")) suffix = "_long";
		else if (dataType.equals("multichannel")) suffix = "_multi";
		
        if (image.getNDims() == 2) { // source image is 2D
            destExtents = new int[2];
            destExtents[0] = image.getExtents()[0]; // X dim
            destExtents[1] = image.getExtents()[1]; // Y dim
        } else { // source image is 3D or more (?)
            destExtents = new int[3];
            destExtents[0] = image.getExtents()[0];
            destExtents[1] = image.getExtents()[1];
            destExtents[2] = image.getExtents()[2];
        }

        try {
            resultImage = new ModelImage[resultNumber*Nimg];
            presentNumber = 0;
            
			for (int n=0;n<Nimg;n++) {
				if (!outputType.equals("hard_segmentation")) {
					for (i = 0; i < nClasses; i++) {
						resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents,
															(images[n].getImageName()+ suffix+"_class" + (i + 1)));
					}
					if (addOutliers) {
						resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents, 
															(images[n].getImageName()+ suffix+"_out"));
					}
				}
				if (!outputType.equals("fuzzy_segmentation")) {
					resultImage[presentNumber++] = new ModelImage(ModelStorageBase.UBYTE, destExtents,
															(images[n].getImageName()+ suffix+"_seg"));
				}
				if (outputType.equals("all_result_images")) {
					if (correctType.equals("image") || correctType.equals("centroids")) {
						if (dataType.equals("multichannel")) {
							for (int t=0;t<imageNumber;t++) {
								resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents,
																	(images[n].getImageName()+ suffix+"_field"));
							}
						} else {
							resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents,
																(images[n].getImageName()+suffix+"_field"));
						}
					} else if (correctType.equals("separate")) {
						if (dataType.equals("multichannel")) {
							for (int t=0;t<imageNumber;t++) {
								for (int k=0;k<nClasses;k++)
									resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents,
																	(images[t].getImageName()+ suffix+"_field"+(k+1)));
							}
						} else {
							for (int k=0;k<nClasses;k++)
								resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents,
																(images[n].getImageName()+ suffix+"_field"+(k+1)));
						}
					}					
					if (useEdges) {
						resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents,
															(images[n].getImageName()+ suffix+"_edgeX"));
						resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents,
															(images[n].getImageName()+ suffix+"_edgeY"));
						resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents,
															(images[n].getImageName()+ suffix+"_edgeZ"));
					}
				}
			}
            // Create algorithm
			if (algoType.equals("FANTASM")) {
				algo = new AlgorithmFantasm(resultImage, images[0], resultNumber,
											initMode, outputType,
											nClasses, nIterations,
											maxDiff, smoothing, fuzziness,
											addOutliers, outlierRatio, 
											correctType, polynomialDegree,
											useEdges, edgeSmoothness, edgeContrast, edgePrior,
											cropBackground, backThreshold, useRelative, false);
				algorithm = algo;
			} else if (algoType.equals("K-means")) {
				algo2 = new AlgorithmKmeansSegmentation(resultImage, images[0], resultNumber,
												initMode, outputType,
												nClasses, nIterations, maxDiff, smoothing, 
												addOutliers, outlierRatio, 
												correctType, polynomialDegree,
												useEdges, edgeSmoothness, edgeContrast, edgePrior,
												cropBackground, backThreshold, useRelative, false);
				algorithm = algo2;
			} else if (algoType.equals("EM")) {
			    boolean eqPriors=true;
			    boolean eqVars=true;
			    algo3 = new AlgorithmEMSegmentation(resultImage, images[0], resultNumber,
								initMode, outputType,
								nClasses, nIterations, maxDiff, smoothing, 
								addOutliers, outlierRatio, 
								correctType, polynomialDegree,
								useEdges, edgeSmoothness, edgeContrast, edgePrior,
								cropBackground, backThreshold, useRelative, false,
								eqPriors,eqVars);
			    algorithm = algo3;
			} else if (algoType.equals("EMv")) {
			    boolean eqPriors=true;
			    boolean eqVars=false;
			    algo4 = new AlgorithmEMSegmentation(resultImage, images[0], resultNumber,
								initMode, outputType,
								nClasses, nIterations, maxDiff, smoothing, 
								addOutliers, outlierRatio, 
								correctType, polynomialDegree,
								useEdges, edgeSmoothness, edgeContrast, edgePrior,
								cropBackground, backThreshold, useRelative, false,
								eqPriors,eqVars);
			    algorithm = algo4;
			} else if (algoType.equals("EMpv")) {
			    boolean eqPriors=false;
			    boolean eqVars=false;
			    algo5 = new AlgorithmEMSegmentation(resultImage, images[0], resultNumber,
								initMode, outputType,
								nClasses, nIterations, maxDiff, smoothing, 
								addOutliers, outlierRatio, 
								correctType, polynomialDegree,
								useEdges, edgeSmoothness, edgeContrast, edgePrior,
								cropBackground, backThreshold, useRelative, false,
								eqPriors,eqVars);
			    algorithm = algo5;
			}
            
            // This is very important. Adding this object as a listener allows the algorithm to
            // notify this object when it has completed or failed. See algorithm performed event.
            // This is made possible by implementing AlgorithmedPerformed interface
            algorithm.addListener(this);
                 
            setVisible(false);  // Hide dialog

			createProgressBar(image.getImageName(), algorithm);

			// bug fix: check if the linux version crashes in this case
			//setSeparateThread(false);
			
            if (runInSeparateThread) {
                // Start the thread as a low priority because we wish to still have user interface work fast.
                if (algorithm.startMethod(Thread.MIN_PRIORITY) == false) {
                    MipavUtil.displayError("A thread is already running on this object");
                }
            } else {
                //algorithm.setActiveImage(isActiveImage);
                algorithm.run();
            }
        } catch (OutOfMemoryError x) {

            if (resultImage != null) {
                for (i = 0; i < resultNumber*Nimg; i++) {
                    if (resultImage[i] != null) {
                        resultImage[i].disposeLocal(); // Clean up memory of result image
                        resultImage[i] = null;
                    }
                }
                resultImage = null;
            }
            System.gc();
            MipavUtil.displayError( "Dialog Tissue Segmentation: unable to allocate enough memory");
            return;
        }
    } // end callAlgorithm()
        
    /**
     * Perform any actions required after the running of the algorithm is complete.
     */
    protected void doPostAlgorithmActions() {
		for (int n=0;n<resultNumber;n++)
			AlgorithmParameters.storeImageInRunner(getResultImage()[n]);
    }

    /**
     * Set up the dialog GUI based on the parameters before running the algorithm as part of a script.
     */
    protected void setGUIFromParams() {
		imageNumber = scriptParameters.getParams().getInt("input_images");
		images = new ModelImage[imageNumber];
		for (int n=0;n<imageNumber;n++) {
			images[n] = scriptParameters.retrieveInputImage(n+1);
		}
		image = images[0];
        userInterface = ViewUserInterface.getReference();
        parentFrame = image.getParentFrame();
		
		nClasses = scriptParameters.getParams().getInt("classes");
		nIterations = scriptParameters.getParams().getInt("iterations");
		maxDiff = scriptParameters.getParams().getFloat("max_difference");
		smoothing = scriptParameters.getParams().getFloat("smoothing");
		fuzziness = scriptParameters.getParams().getFloat("fuzziness");
		
		algoType = scriptParameters.getParams().getString("algorithm_type");
		initMode = scriptParameters.getParams().getString("initialization");
		maskingMode = scriptParameters.getParams().getString("masking_mode");
		
		correctType = scriptParameters.getParams().getString("inhomogeneity_correction");
		polynomialDegree = scriptParameters.getParams().getInt("field_degree");
		
		cropBackground = scriptParameters.getParams().getBoolean("crop_background");
		backThreshold = scriptParameters.getParams().getFloat("background_threshold");
		outputType = scriptParameters.getParams().getString("output_type");
    }

	/**
     * Store the parameters from the dialog to record the execution of this algorithm.
     * 
     * @throws  ParserException  If there is a problem creating one of the new parameters.
     */
    protected void storeParamsFromGUI() throws ParserException {
        scriptParameters.getParams().put(ParameterFactory.newParameter("input_images", imageNumber));
		for (int n=0;n<imageNumber;n++) {
			scriptParameters.storeInputImage(images[n]);
		}
        scriptParameters.getParams().put(ParameterFactory.newParameter("output_images", resultNumber));
		for (int n=0;n<resultNumber;n++) {
			scriptParameters.storeImageInRecorder(resultImage[n]);  // false means computation in place
		}
		
		scriptParameters.getParams().put(ParameterFactory.newParameter("classes", nClasses));
		scriptParameters.getParams().put(ParameterFactory.newParameter("iterations", nIterations));
		scriptParameters.getParams().put(ParameterFactory.newParameter("max_difference", maxDiff));
		scriptParameters.getParams().put(ParameterFactory.newParameter("smoothing", smoothing));
		scriptParameters.getParams().put(ParameterFactory.newParameter("fuzziness", fuzziness));
		
		scriptParameters.getParams().put(ParameterFactory.newParameter("algorithm_type", algoType));
		scriptParameters.getParams().put(ParameterFactory.newParameter("initialization", initMode));
		scriptParameters.getParams().put(ParameterFactory.newParameter("masking_mode", maskingMode));
		
		scriptParameters.getParams().put(ParameterFactory.newParameter("inhomogeneity_correction", correctType));
		scriptParameters.getParams().put(ParameterFactory.newParameter("field_degree", polynomialDegree));
		
		scriptParameters.getParams().put(ParameterFactory.newParameter("crop_background", cropBackground));
		scriptParameters.getParams().put(ParameterFactory.newParameter("background_threshold", backThreshold));
		scriptParameters.getParams().put(ParameterFactory.newParameter("output_type", outputType));
	}
}
