package edu.jhmi.rad.medic.dialogs;

import edu.jhmi.rad.medic.algorithms.*;
import edu.jhmi.rad.medic.utilities.*;

import gov.nih.mipav.model.algorithms.*;
import gov.nih.mipav.model.scripting.*;
import gov.nih.mipav.model.scripting.parameters.*;
import gov.nih.mipav.model.structures.*;
import gov.nih.mipav.view.*;
import gov.nih.mipav.view.dialogs.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;



/**
 * The dialog window for the Volume Statistics tool.
 * 
 * @author Pilou Bazin
 * @version May 2007
 */
public class JDialogVolumeStatistics2 extends JDialogScriptableBase implements
		AlgorithmInterface, DialogDefaultsInterface {

	/** Reference to the MIPAV interface.*/
	private ViewUserInterface userInterface;
	
	private     AlgorithmVolumeStatistics2 		algorithm = null;
    
	/** The source image for the algorithm. */
	private ModelImage image = null;
	
	/** Additional images for complex measurements */
	private ModelImage intensity = null;
	/** Additional images for complex measurements */
	private ModelImage reference = null;
	
	/** The resulting image, if any	 */
	private ModelImage resultImage = null;
	
	/** parameters */
	private	ArrayList<String>	statistics;
										   
	private boolean		imgVOI = false;

	private	String[]	connectivityTypes = {"6/26","6/18","18/6","26/6"};
	private String		connectivityType = "6/26";
	
	private	boolean		toFile;
	private	String		filename = null;
	private	String[]	delims = {"space","tab","comma","semi-colon"};
	private String		delim = "space";
	private String		delimSymbol = " ";
	
	/** GUI components */
	private	boolean		useGui = false;
	
	private	JButton		refreshImageListsButton;
	private	JButton		computeButton;
	
	private JLabel		imageLabel;
	private JComboBox 	imageCombo;	
	private	JCheckBox	useImageVOIsCheck;
	
	private JLabel		intensityLabel;
	private JComboBox 	intensityImageCombo;	
	private JLabel		referenceLabel;
	private JComboBox 	referenceImageCombo;
	
	private JScrollPane		statisticsPane;
	
	private JCheckBox		checkVoxels;
	private JCheckBox		checkVolume;
	private JCheckBox		checkSurface;
	
	private JCheckBox		checkEuler;
	private JCheckBox		checkCavities;
	private JCheckBox		checkParts;
	
	private JCheckBox		checkSurfaceRipples;

	private JCheckBox		checkDice;
	private JCheckBox		checkJaccard;
	private JCheckBox		checkMountford;
	private JCheckBox		checkTruePositive;
	private JCheckBox		checkFalsePositive;
	private JCheckBox		checkFalseNegative;
	private JCheckBox		checkTrueSegmented;
	private JCheckBox		checkFalseSegmented;
	private JCheckBox		checkVolumeDiff;
	private JCheckBox		checkAinB;
	private JCheckBox		checkBinA;
	
	private JCheckBox		checkHausdorff;
	private JCheckBox		checkAvSurfDist;
	private JCheckBox		checkAvSurfDiff;
	private JCheckBox		checkAvSqSurfDist;
	
	private JCheckBox		checkMinIntensity;
	private JCheckBox		checkMaxIntensity;
	private JCheckBox		checkMeanIntensity;
	private JCheckBox		checkStdevIntensity;
	
	private JLabel			connectivityLabel;
	private	JComboBox		connectivityCombo;
	
	private JCheckBox		checkWriteToFile;
	private JButton			fileButton;
	private JTextField		fileText;
	private	JFileChooser	fileDialog;
	private JLabel			delimLabel;
	private JComboBox 		delimCombo;	
	
	private JTable			outputTable;
	private JScrollPane		outputScroll;
	
	/**
	 * Empty constructor required for dynamic instantiation.
	 */
	public JDialogVolumeStatistics2() {
	}
	
	
	/**
	 * Creates a dialog for the Volume Statistics plugin.
	 * 
	 * @param parentFrame		the parent <code>Frame</code>.
	 * @param image			the source <code>ModelImage</code> image.
	 */
	public JDialogVolumeStatistics2(ViewUserInterface ui) {
		super();
		userInterface = ui;
        useGui = true;
		
        loadDefaults();
        init();
	}
	

	/**
	 * Set up the GUI components and display them on the screen.
	 */
	private void init() {
        setForeground(Color.BLACK);
		setTitle("Volume Statistics");
		
		// the main panel to add other panels to
		JPanel mainPanel = new JPanel(new GridBagLayout());
		mainPanel.setName("Main");
		
		// GUI elements
		imageLabel = new JLabel("Segmentation");
		imageLabel.setForeground(Color.BLACK);
		imageLabel.setFont(serif12);
		imageLabel.setToolTipText("Segmentation image to process");
		
		imageCombo = new JComboBox();
        imageCombo.setFont(serif12);
        imageCombo.setBackground(Color.white);
		buildImageComboBox(imageCombo);
		
		useImageVOIsCheck = new JCheckBox("use VOIs");
        useImageVOIsCheck.setFont(serif12);
        useImageVOIsCheck.setForeground(Color.black);
		useImageVOIsCheck.setSelected(imgVOI);
        useImageVOIsCheck.setToolTipText("Uses the image VOIs");
		
		// additional image for intensity		
		intensityLabel = new JLabel("Intensity");
		intensityLabel.setForeground(Color.BLACK);
		intensityLabel.setFont(serif12);
		intensityLabel.setToolTipText("Intensity image for intensity analysis");
		
		intensityImageCombo = new JComboBox();
        intensityImageCombo.setFont(serif12);
        intensityImageCombo.setBackground(Color.white);
		buildImageComboBox(intensityImageCombo);
		
		// additional image for comparison		
		referenceLabel = new JLabel("Reference");
		referenceLabel.setForeground(Color.BLACK);
		referenceLabel.setFont(serif12);
		referenceLabel.setToolTipText("Reference image for comparisons");
		
		referenceImageCombo = new JComboBox();
        referenceImageCombo.setFont(serif12);
        referenceImageCombo.setBackground(Color.white);
		buildImageComboBox(referenceImageCombo);
		
		statisticsPane = new JScrollPane();
		statisticsPane.setMaximumSize(new Dimension(50,100));
		statisticsPane.setPreferredSize(new Dimension(50,100));
		statisticsPane.setMinimumSize(new Dimension(50,100));
		
		checkVoxels = new JCheckBox("Voxels");
        checkVoxels.setFont(serif12);
        checkVoxels.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkVoxels.getText())) checkVoxels.setSelected(true);
		
		checkVolume = new JCheckBox("Volume");
        checkVolume.setFont(serif12);
        checkVolume.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkVolume.getText())) checkVolume.setSelected(true);
		
		checkSurface = new JCheckBox("Surface");
        checkSurface.setFont(serif12);
        checkSurface.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkSurface.getText())) checkSurface.setSelected(true);
		
		checkEuler = new JCheckBox("Euler_characteristic");
        checkEuler.setFont(serif12);
        checkEuler.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkEuler.getText())) checkEuler.setSelected(true);
		
		checkParts = new JCheckBox("Object_parts");
        checkParts.setFont(serif12);
        checkParts.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkParts.getText())) checkParts.setSelected(true);
		
		checkCavities = new JCheckBox("Object_cavities");
        checkCavities.setFont(serif12);
        checkCavities.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkCavities.getText())) checkCavities.setSelected(true);
		
		checkSurfaceRipples = new JCheckBox("Surface_ripples");
        checkSurfaceRipples.setFont(serif12);
        checkSurfaceRipples.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkSurfaceRipples.getText())) checkSurfaceRipples.setSelected(true);
		
		checkMinIntensity = new JCheckBox("Min_intensity");
        checkMinIntensity.setFont(serif12);
        checkMinIntensity.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkMinIntensity.getText())) checkMinIntensity.setSelected(true);
		
		checkMaxIntensity = new JCheckBox("Max_intensity");
        checkMaxIntensity.setFont(serif12);
        checkMaxIntensity.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkMaxIntensity.getText())) checkMaxIntensity.setSelected(true);
		
		checkMeanIntensity = new JCheckBox("Mean_intensity");
        checkMeanIntensity.setFont(serif12);
        checkMeanIntensity.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkMeanIntensity.getText())) checkMeanIntensity.setSelected(true);
		
		checkStdevIntensity = new JCheckBox("Stdev_intensity");
        checkStdevIntensity.setFont(serif12);
        checkStdevIntensity.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkStdevIntensity.getText())) checkStdevIntensity.setSelected(true);
		
		checkDice = new JCheckBox("Dice_overlap");
        checkDice.setFont(serif12);
        checkDice.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkDice.getText())) checkDice.setSelected(true);
		
		checkJaccard = new JCheckBox("Jaccard_overlap");
        checkJaccard.setFont(serif12);
        checkJaccard.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkJaccard.getText())) checkJaccard.setSelected(true);
		
		checkTruePositive = new JCheckBox("True_positive_ratio");
        checkTruePositive.setFont(serif12);
        checkTruePositive.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkTruePositive.getText())) checkTruePositive.setSelected(true);
		
		checkFalsePositive = new JCheckBox("False_positive_ratio");
        checkFalsePositive.setFont(serif12);
        checkFalsePositive.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkFalsePositive.getText())) checkFalsePositive.setSelected(true);
		
		checkFalseNegative = new JCheckBox("False_negative_ratio");
        checkFalseNegative.setFont(serif12);
        checkFalseNegative.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkFalseNegative.getText())) checkFalseNegative.setSelected(true);
		
		checkTrueSegmented = new JCheckBox("True_segmented_ratio");
        checkTrueSegmented.setFont(serif12);
        checkTrueSegmented.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkTrueSegmented.getText())) checkTrueSegmented.setSelected(true);
		
		checkFalseSegmented = new JCheckBox("False_segmented_ratio");
        checkFalseSegmented.setFont(serif12);
        checkFalseSegmented.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkFalseSegmented.getText())) checkFalseSegmented.setSelected(true);
		
		checkMountford = new JCheckBox("Mountford_overlap");
        checkMountford.setFont(serif12);
        checkMountford.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkMountford.getText())) checkMountford.setSelected(true);
		
		checkAinB = new JCheckBox("Containment_index(seg_in_ref)");
        checkAinB.setFont(serif12);
        checkAinB.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkAinB.getText())) checkAinB.setSelected(true);
		
		checkBinA = new JCheckBox("Containment_index(ref_in_seg)");
        checkBinA.setFont(serif12);
        checkBinA.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkBinA.getText())) checkBinA.setSelected(true);
		
		checkVolumeDiff = new JCheckBox("Volume_difference");
        checkVolumeDiff.setFont(serif12);
        checkVolumeDiff.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkVolumeDiff.getText())) checkVolumeDiff.setSelected(true);
		
		checkAvSurfDist= new JCheckBox("Average_surface_distance");
        checkAvSurfDist.setFont(serif12);
        checkAvSurfDist.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkAvSurfDist.getText())) checkAvSurfDist.setSelected(true);
		
		checkAvSurfDiff= new JCheckBox("Average_surface_difference");
        checkAvSurfDiff.setFont(serif12);
        checkAvSurfDiff.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkAvSurfDiff.getText())) checkAvSurfDiff.setSelected(true);
		
		checkAvSqSurfDist= new JCheckBox("Average_squared_surface_distance");
        checkAvSqSurfDist.setFont(serif12);
        checkAvSqSurfDist.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkAvSqSurfDist.getText())) checkAvSqSurfDist.setSelected(true);
		
		checkHausdorff= new JCheckBox("Hausdorff_distance");
        checkHausdorff.setFont(serif12);
        checkHausdorff.setForeground(Color.black);
		for (int n=0;n<statistics.size();n++) if (statistics.get(n).equals(checkHausdorff.getText())) checkHausdorff.setSelected(true);
		
		connectivityLabel = new JLabel("connectivity: ");
        connectivityLabel.setFont(serif12);
        connectivityLabel.setBackground(Color.white);
		connectivityLabel.setToolTipText("Selects the proper object connectivity");
        
		connectivityCombo = new JComboBox(connectivityTypes);
        connectivityCombo.setFont(serif12);
        connectivityCombo.setBackground(Color.white);
		connectivityCombo.setSelectedItem(connectivityType);
        
		refreshImageListsButton = new JButton(MipavUtil.getIcon("refresh.gif"));
		refreshImageListsButton.setForeground(Color.black);
        refreshImageListsButton.setFont(serif12);
        refreshImageListsButton.addActionListener(this);
        refreshImageListsButton.setActionCommand("Refresh");
        refreshImageListsButton.setBorderPainted( false );
        refreshImageListsButton.setRolloverEnabled( true );
        refreshImageListsButton.setRolloverIcon( MipavUtil.getIcon( "refreshroll.gif" ) );
        refreshImageListsButton.setFocusPainted( false );
        refreshImageListsButton.setToolTipText("Refresh image list");

		computeButton = new JButton("Compute");
		computeButton.setForeground(Color.black);
        computeButton.setFont(serif12);
        computeButton.addActionListener(this);
        computeButton.setActionCommand("Compute");
        computeButton.setToolTipText("Compute the statistics");

		checkWriteToFile = new JCheckBox("write to file");
        checkWriteToFile.setFont(serif12);
        checkWriteToFile.setForeground(Color.black);
		checkWriteToFile.addActionListener(this);
        checkWriteToFile.setActionCommand("WriteToFile");
        checkWriteToFile.setSelected(toFile);
        checkWriteToFile.setToolTipText("Writes the statistics to a text file");
		
		fileText = new JTextField(10);
		fileText.setText(filename);
		fileText.setFont(serif12);
		fileText.setEnabled(toFile);
		
		fileButton = new JButton("Browse");
		fileButton.setForeground(Color.black);
        fileButton.setFont(serif12);
        fileButton.addActionListener(this);
        fileButton.setActionCommand("ChooseFile");
		fileButton.setToolTipText("Text file to save the statistics");
        fileButton.setEnabled(toFile);
		
		delimLabel = new JLabel("field delimitation: ");
        delimLabel.setFont(serif12);
        delimLabel.setBackground(Color.white);
		delimLabel.setToolTipText("Chooses the appropriate field delimitation symbol for the file");
        
		delimCombo = new JComboBox(delims);
        delimCombo.setFont(serif12);
        delimCombo.setBackground(Color.white);
		delimCombo.setSelectedItem(delim);
        
		outputScroll = new JScrollPane(ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED,
										ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);
		outputScroll.setMaximumSize(new Dimension(800,100));
		outputScroll.setPreferredSize(new Dimension(800,100));
		outputScroll.setMinimumSize(new Dimension(800,100));
		
		JPanel imagePanel = new JPanel(new GridBagLayout());
		JPanel optionPanel = new JPanel(new GridBagLayout());
		JPanel filePanel = new JPanel(new GridBagLayout());
		JPanel outputPanel = new JPanel(new GridBagLayout());
		
		GridBagConstraints gbc = new GridBagConstraints();
		gbc.weightx = 1.0;
		gbc.gridwidth = 1;
        gbc.fill = GridBagConstraints.HORIZONTAL;
		
		gbc.gridx = 0;
		gbc.gridy = 0;
		imagePanel.add(imageLabel, gbc);
        gbc.gridx = 0;
		gbc.gridy = 1;
		imagePanel.add(intensityLabel, gbc);
		gbc.gridx = 0;
		gbc.gridy = 2;
		imagePanel.add(referenceLabel, gbc);
		gbc.gridx = 1;
		gbc.gridy = 0;
		imagePanel.add(imageCombo, gbc);
		gbc.gridx = 1;
		gbc.gridy = 1;
		imagePanel.add(intensityImageCombo, gbc);
		gbc.gridx = 1;
		gbc.gridy = 2;
		imagePanel.add(referenceImageCombo, gbc);
		gbc.gridx = 2;
		gbc.gridy = 0;
		imagePanel.add(useImageVOIsCheck, gbc);
		gbc.gridx = 2;
		gbc.gridy = 2;
		imagePanel.add(refreshImageListsButton, gbc);
		
		gbc.gridx = 0;
		gbc.gridy = 0;
		optionPanel.add(connectivityLabel, gbc);
		gbc.gridx = 1;
		gbc.gridy = 0;
		optionPanel.add(connectivityCombo, gbc);
		
		JPanel statisticsPanel = new JPanel(new GridBagLayout());
		gbc.weightx = 1;
		gbc.weighty = 0;
		gbc.fill = gbc.HORIZONTAL;
		gbc.gridwidth=1;
		gbc.gridheight=1;
		gbc.gridx = 0;
		
		gbc.gridy = 0;
		statisticsPanel.add(checkVoxels, gbc);
		gbc.gridy++;
		statisticsPanel.add(checkVolume, gbc);
		gbc.gridy++;
		statisticsPanel.add(checkSurface, gbc);
		
		gbc.gridy = 0;
		gbc.gridx++;
		statisticsPanel.add(checkEuler, gbc);
		gbc.gridy++;
		statisticsPanel.add(checkParts, gbc);
		gbc.gridy++;
		statisticsPanel.add(checkCavities, gbc);
		//gbc.gridy++;
		//statisticsPanel.add(checkSurfaceRipples, gbc);
		
		gbc.gridy = 0;
		gbc.gridx++;
		statisticsPanel.add(checkMinIntensity, gbc);
		gbc.gridy++;
		statisticsPanel.add(checkMaxIntensity, gbc);
		gbc.gridy++;
		statisticsPanel.add(checkMeanIntensity, gbc);
		gbc.gridy++;
		statisticsPanel.add(checkStdevIntensity, gbc);
		
		gbc.gridy = 0;
		gbc.gridx++;
		statisticsPanel.add(checkDice, gbc);
		gbc.gridy++;
		statisticsPanel.add(checkJaccard, gbc);
		gbc.gridy++;
		statisticsPanel.add(checkTruePositive, gbc);
		gbc.gridy++;
		statisticsPanel.add(checkFalsePositive, gbc);
		gbc.gridy++;
		statisticsPanel.add(checkFalseNegative, gbc);
		gbc.gridy++;
		statisticsPanel.add(checkTrueSegmented, gbc);
		gbc.gridy++;
		statisticsPanel.add(checkFalseSegmented, gbc);
		//gbc.gridy++;
		//statisticsPanel.add(checkMountford, gbc);
		gbc.gridy++;
		statisticsPanel.add(checkAinB, gbc);
		gbc.gridy++;
		statisticsPanel.add(checkBinA, gbc);
		
		gbc.gridy = 0;
		gbc.gridx++;
		statisticsPanel.add(checkAvSurfDist, gbc);
		gbc.gridy++;
		statisticsPanel.add(checkAvSurfDiff, gbc);
		gbc.gridy++;
		statisticsPanel.add(checkAvSqSurfDist, gbc);
		gbc.gridy++;
		statisticsPanel.add(checkHausdorff, gbc);
		
		statisticsPane.setViewportView(statisticsPanel);
		statisticsPane.revalidate();
		
		gbc.gridx = 0;
		gbc.gridy = 0;
		filePanel.add(checkWriteToFile, gbc);
		gbc.gridx = 1;
		gbc.gridy = 0;
		filePanel.add(fileText, gbc);
		gbc.gridx = 2;
		gbc.gridy = 0;
		filePanel.add(fileButton, gbc);
		gbc.gridx = 1;
		gbc.gridy = 1;
		filePanel.add(delimLabel, gbc);
		gbc.gridx = 2;
		gbc.gridy = 1;
		filePanel.add(delimCombo, gbc);	
		
		gbc.gridx = 0;
		gbc.gridy = 0;
		outputPanel.add(outputScroll, gbc);
		
		gbc.gridx = 0;
		gbc.gridy = 0;
		mainPanel.add(imagePanel, gbc);
		gbc.gridx = 0;
		gbc.gridy = 1;
		mainPanel.add(statisticsPane, gbc);
		gbc.gridx = 0;
		gbc.gridy = 2;
		mainPanel.add(optionPanel, gbc);
		gbc.gridx = 0;
		gbc.gridy = 3;
		mainPanel.add(filePanel, gbc);
		gbc.gridx = 0;
		gbc.gridy = 4;
		mainPanel.add(computeButton, gbc);
		gbc.gridx = 0;
		gbc.gridy = 5;
		mainPanel.add(outputPanel, gbc);
		
		// Add the tabs and bottom Panel to the dialog
		getContentPane().add(mainPanel);
		getContentPane().add(buildButtons(), BorderLayout.SOUTH);

		pack();
		setResizable(true);
        setVisible(true);
	}
	

	/** create a combo box with all open images */
    private void buildImageComboBox(JComboBox combo) {
        int j;
        
		Enumeration names = userInterface.getRegisteredImageNames();

		// Add images from user interface that have the same exact dimensionality
        while (names.hasMoreElements()) {
            String name = (String) names.nextElement();
			boolean found = false;
			for (int i=0;i<combo.getItemCount() && !found;i++) {
				if (name.equals((String)combo.getItemAt(i))) found = true;
			}
			if (!found) {
				ModelImage img = userInterface.getRegisteredImageByName(name);
				if (userInterface.getFrameContainingImage(img) != null) {
					combo.addItem(name);
				}
			}
		}
		return;
    }//buildImageComboBox
	
	
	/** put the statistics into a table for display */
	private final void writeToTable(ArrayList<String> stats) {
		// first line: labels
		String[] headings = stats.get(0).split(delimSymbol);
		headings[0] = "Statistic";
		headings[1] = "Image";
		headings[2] = "Reference";
		headings[3] = "Intensity";
		
		// rest is data
		String[][] data = new String[stats.size()-1][];
		for (int n=1;n<stats.size();n++) {
			data[n-1] = stats.get(n).split(delimSymbol);
		}
		if (useGui) {
			outputTable = new JTable(data,headings);
			outputTable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
			outputScroll.setViewportView(outputTable);
		}
	}
	
	/**
	 * Handles a given event.
	 * 
	 * @param event	the <code>ActionEvent</code> to handle.
	 */
	public void actionPerformed(ActionEvent event) {
		String command = event.getActionCommand();
		
        System.err.println("ActionPerformed:command=" + command);
        
		if (command.equals("OK")) {
			if (setVariables()) { 
				callAlgorithm();
			}
			dispose();
		} else if (command.equals("Compute")) {
			if (setVariables()) { 
				callAlgorithm();
			}    
		} else if (command.equals("Cancel")) {
			dispose();
		} else if (command.equals("Help")) {
			MedicUtilPublic.displayMessage("Help pressed\n");
			
		} else if (command.equals("Refresh")) {
			buildImageComboBox(imageCombo);
			buildImageComboBox(referenceImageCombo);
			buildImageComboBox(intensityImageCombo);
			pack();
		} else if (command.equals("WriteToFile")) {
			if (checkWriteToFile.isSelected()) {
				fileButton.setEnabled(true);
				fileText.setEnabled(true);
			} else {
				fileButton.setEnabled(false);
				fileText.setEnabled(false);
			}
		} else if (command.equals("ChooseFile")) {
			buildFileDialog();
            fileDialog.setSize(500,326);
		}
	}

	private void buildFileDialog() {
        fileDialog = new javax.swing.JFileChooser();
        fileDialog.setDialogTitle("Write to statistics file");
        fileDialog.setDialogType(JFileChooser.SAVE_DIALOG);
        fileDialog.setMaximumSize(new java.awt.Dimension(500, 326));
        fileDialog.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                saveFileActionPerformed(evt);
            }
        });
        
		if (userInterface.getDefaultDirectory() != null) {
			fileDialog.setCurrentDirectory(new File(userInterface.getDefaultDirectory()));
        } else {
			fileDialog.setCurrentDirectory(new File(System.getProperties().getProperty("user.dir")));
        }

		fileDialog.setFileSelectionMode(JFileChooser.FILES_ONLY);
		fileDialog.showSaveDialog(this);
	}

 	public void saveFileActionPerformed(ActionEvent evt) {
		if ( JFileChooser.APPROVE_SELECTION.equals(evt.getActionCommand()) ) {
			String filename = fileDialog.getSelectedFile().getAbsolutePath();
			MedicUtilPublic.displayMessage("statistics will be saved in "+filename+"\n");
			userInterface.setDefaultDirectory(fileDialog.getCurrentDirectory().getAbsolutePath());
			fileText.setText(filename);
		}
	}
	
	/**
	 * This method is called by the algorithm when it has completed or failed to finish, so that 
	 * the dialog can either display the correct, resulting image, and/or clean up.
	 * 
	 * @param	the algorithm that caused the event.
	 */
	public void algorithmPerformed(AlgorithmBase algorithm) {
		
		if (Preferences.isPreference(Preferences.PREF_SAVE_DEFAULTS) && this.getOwner() != null && !isScriptRunning()) {
			saveDefaults();
		}
		if (algorithm instanceof AlgorithmVolumeStatistics2) {
			image.clearMask();
			if (algorithm.isCompleted()) {
				// write result to the table
				writeToTable(((AlgorithmVolumeStatistics2)algorithm).getOutput());
				
				insertScriptLine();
			}
		}
		algorithm.finalize();
		algorithm = null;
 	}
	
	/**
	 * Sets the all the user input variables for calling the SegmentationPainter algorithm and makes 
	 * sure that these values are valid.
	 * 
	 * @return	true if all the variables were set, false otherwise.
	 */
	private boolean setVariables() {
		
		imgVOI = useImageVOIsCheck.isSelected();
		
		image = userInterface.getRegisteredImageByName((String)imageCombo.getSelectedItem());
		reference = userInterface.getRegisteredImageByName((String)referenceImageCombo.getSelectedItem());
		intensity = userInterface.getRegisteredImageByName((String)intensityImageCombo.getSelectedItem());
			
		connectivityType = (String)connectivityCombo.getSelectedItem();
		
		statistics = new ArrayList();
		if (checkVoxels.isSelected()) statistics.add(checkVoxels.getText());
		if (checkVolume.isSelected()) statistics.add(checkVolume.getText());
		if (checkSurface.isSelected()) statistics.add(checkSurface.getText());
		
		if (checkEuler.isSelected()) statistics.add(checkEuler.getText());
		if (checkParts.isSelected()) statistics.add(checkParts.getText());
		if (checkCavities.isSelected()) statistics.add(checkCavities.getText());
		
		if (checkSurfaceRipples.isSelected()) statistics.add(checkSurfaceRipples.getText());
		
		if (checkMeanIntensity.isSelected()) statistics.add(checkMeanIntensity.getText());
		if (checkMinIntensity.isSelected()) statistics.add(checkMinIntensity.getText());
		if (checkMaxIntensity.isSelected()) statistics.add(checkMaxIntensity.getText());
		if (checkStdevIntensity.isSelected()) statistics.add(checkStdevIntensity.getText());
		
		if (checkDice.isSelected()) statistics.add(checkDice.getText());
		if (checkJaccard.isSelected()) statistics.add(checkJaccard.getText());
		if (checkMountford.isSelected()) statistics.add(checkMountford.getText());
		if (checkTruePositive.isSelected()) statistics.add(checkTruePositive.getText());
		if (checkFalsePositive.isSelected()) statistics.add(checkFalsePositive.getText());
		if (checkFalseNegative.isSelected()) statistics.add(checkFalseNegative.getText());
		if (checkTrueSegmented.isSelected()) statistics.add(checkTrueSegmented.getText());
		if (checkFalseSegmented.isSelected()) statistics.add(checkFalseSegmented.getText());
		if (checkVolumeDiff.isSelected()) statistics.add(checkVolumeDiff.getText());
		if (checkAinB.isSelected()) statistics.add(checkAinB.getText());
		if (checkBinA.isSelected()) statistics.add(checkBinA.getText());
		
		if (checkAvSurfDist.isSelected()) statistics.add(checkAvSurfDist.getText());
		if (checkAvSurfDiff.isSelected()) statistics.add(checkAvSurfDiff.getText());
		if (checkAvSqSurfDist.isSelected()) statistics.add(checkAvSqSurfDist.getText());
		if (checkHausdorff.isSelected()) statistics.add(checkHausdorff.getText());
		
		toFile = checkWriteToFile.isSelected();
		if (toFile) filename = fileText.getText();
		
		delim = (String)delimCombo.getSelectedItem();
			 if (delim.equals("space")) 	 delimSymbol = " ";
		else if (delim.equals("tab")) 	 	 delimSymbol = "	";
		else if (delim.equals("comma")) 	 delimSymbol = ",";
		else if (delim.equals("semi-colon")) delimSymbol = ";";
		else 								 delimSymbol = " ";
		
		return true;
	}
	
   /**
    *	Once all the necessary variables are set, call the Gaussian Blur
    *	algorithm based on what type of image this is and whether or not there
    *	is a separate destination image.
    */
    protected void callAlgorithm() {
        int i;
        
        try {			
            // Create algorithm
			algorithm = new AlgorithmVolumeStatistics2(image, intensity, reference, statistics, imgVOI, connectivityType, toFile, filename, delimSymbol);
			
            // This is very important. Adding this object as a listener allows the algorithm to
            // notify this object when it has completed or failed. See algorithm performed event.
            // This is made possible by implementing AlgorithmedPerformed interface
            algorithm.addListener(this);
                
            // for linux debugs: no threading
			// setSeparateThread(false);

            if (runInSeparateThread) {
                // Start the thread as a low priority because we wish to still have user interface work fast.
                if (algorithm.startMethod(Thread.MIN_PRIORITY) == false) {
                    MipavUtil.displayError("A thread is already running on this object");
                }
            } else {
                algorithm.run();
            }
        } catch (OutOfMemoryError x) {
			System.gc();
            MipavUtil.displayError( "Dialog: unable to allocate enough memory");
            return;
        }
    } // end callAlgorithm()
	

	/** method to laod and save defaults */
	public void loadDefaults() {
        String defaultsString = Preferences.getDialogDefaults(getDialogName());

        statistics = new ArrayList();
		if (defaultsString != null) {

            try {
                System.out.println(defaultsString);
                StringTokenizer st = new StringTokenizer(defaultsString, ",");
				int Nstat = MipavUtil.getInt(st);
				for (int n=0;n<Nstat;n++)
					statistics.add(st.nextToken() );
				imgVOI = MipavUtil.getBoolean(st);
				connectivityType = st.nextToken();
				toFile = MipavUtil.getBoolean(st);
				filename = st.nextToken();
				delim = st.nextToken();
            }
            catch (Exception ex) {
                // since there was a problem parsing the defaults string, start over with the original defaults
                System.out.println( "Resetting defaults for dialog: " + getDialogName() );
                Preferences.removeProperty( getDialogName() );
            }
        } else {
			System.out.println( "no saved dialogs for "+getDialogName() );
		}
	}

	public void saveDefaults() {
        String defaultsString = new String( getParameterString(",") );
        System.out.println(defaultsString);
        Preferences.saveDialogDefaults(getDialogName(),defaultsString);
	}

    /**
     * Construct a delimited string that contains the parameters to this algorithm.
     * @param delim  the parameter delimiter (defaults to " " if empty)
     * @return       the parameter string
     */
	public String getParameterString( String delimin ) {
        if ( delimin.equals( "" ) ) {
            delimin = " ";
        }

        String str = new String();
		str += statistics.size() + delimin;
		for (int n=0;n<statistics.size();n++)
			str += statistics.get(n) + delimin;
        str += imgVOI + delimin;
        str += connectivityType + delimin;
        str += toFile + delimin;
		str += filename + delimin;
		str += delim;
		
        return str;
    }
	
    /**
     * Perform any actions required after the running of the algorithm is complete.
     */
    protected void doPostAlgorithmActions() {
		//AlgorithmParameters.storeImageInRunner(resultImage);
    }

    /**
     * Set up the dialog GUI based on the parameters before running the algorithm as part of a script.
     */
    protected void setGUIFromParams() {
		image = scriptParameters.retrieveImage("image");
		userInterface = ViewUserInterface.getReference();
        parentFrame = image.getParentFrame();
		
		intensity = scriptParameters.retrieveImage("intensity");
		reference = scriptParameters.retrieveImage("reference");
		
		statistics = new ArrayList();
		int Nstat = scriptParameters.getParams().getInt("stat_number");
		for (int n=0;n<Nstat;n++)
			statistics.add( scriptParameters.getParams().getString("statistics_"+(n+1)) );
		
		imgVOI = scriptParameters.getParams().getBoolean("use_voi");
		connectivityType = scriptParameters.getParams().getString("connectivity_type");
		
		toFile = scriptParameters.getParams().getBoolean("to_file");
		filename = scriptParameters.getParams().getString("filename");
		
		delim = scriptParameters.getParams().getString("delimitator");
			 if (delim.equals("space")) 	 delimSymbol = " ";
		else if (delim.equals("tab")) 	 	 delimSymbol = "	";
		else if (delim.equals("comma")) 	 delimSymbol = ",";
		else if (delim.equals("semi-colon")) delimSymbol = ";";
		else 								 delimSymbol = " ";
    }

	/**
     * Store the parameters from the dialog to record the execution of this algorithm.
     * 
     * @throws  ParserException  If there is a problem creating one of the new parameters.
     */
    protected void storeParamsFromGUI() throws ParserException {
        scriptParameters.storeImage(image, "image");
		scriptParameters.storeImage(intensity, "intensity");
		scriptParameters.storeImage(reference, "reference");
		
		scriptParameters.getParams().put(ParameterFactory.newParameter("stat_number", statistics.size()));
		for (int n=0;n<statistics.size();n++)
			scriptParameters.getParams().put(ParameterFactory.newParameter("statistics_"+(n+1), statistics.get(n)));
		
		scriptParameters.getParams().put(ParameterFactory.newParameter("use_voi", imgVOI));
		scriptParameters.getParams().put(ParameterFactory.newParameter("connectivity_type", connectivityType));
		scriptParameters.getParams().put(ParameterFactory.newParameter("to_file", toFile));
		scriptParameters.getParams().put(ParameterFactory.newParameter("filename", filename));
		scriptParameters.getParams().put(ParameterFactory.newParameter("delimitator", delim));
	}
	
}