package edu.jhmi.rad.medic.methods;


import java.io.*;
import java.util.*;
import java.lang.*;


/**
 *
 *   This is the algorithm for initializing segmentation clusters
 *	(adapted from Dzung Pham's method)
 *
 *
 *	@version    March 2004
 *	@author     Dzung L. Pham
 *	@author     Pierre-Louis Bazin
 *  @see SegmentationFCM
 *		
 *
*/
public class ClusterSearch {
	
	private static final    int 	HCRIT_LOW = 0;
	private static final    int 	HCRIT_HIGH = 100;
	private static final    int 	MAX_MODES = 200;
	private static final    float   MODE_TOLERANCE = 0.01f;
	private static final    float   PI = 3.141592654f;

	private static final	int		ITMAX = 100;
	private static final	float	EPS = 3.0e-8f;

	private static final	boolean	DEBUG = false;

	public      int[]       Histogram;
	public  	int         min, max, N;
	public  	double[]    Centroid;
	private		int			clusters;
	
	public  	boolean isCompleted = true;
	public  	String  errorLog;

	private 	double  prior, var;
	
	public ClusterSearch(int clusters_) {
		clusters = clusters_;
		Centroid = new double[clusters];
	}
	public final void finalize() {
		Centroid = null;
		Histogram = null;
	}
	public final float[] exportCentroids() {
		float[] cent = new float[clusters];
		for (int k=0;k<clusters;k++) cent[k] = (float)Centroid[k];
		return cent;
	}
	
	public void computeHistogram(float[][][] img, boolean[][][] objMask, int nx, int ny, int nz) {

		int temp;
		
		min =  100000;
		max = -100000;
		for (int i=0;i<nx;i++) {
			for (int j=0;j<ny;j++) {
				for (int k=0;k<nz;k++) {
                    if (objMask[i][j][k] ) {
                        if (img[i][j][k]<min) min = Math.round(img[i][j][k]);
                        if (img[i][j][k]>max) max = Math.round(img[i][j][k]);
                    }
                }
            }
        }
	
	
		Histogram = new int[max-min+1];
		N = 0;
		for (int i=min;i<=max;i++)
			Histogram[i-min] = 0;
	
		for (int i=0;i<nx;i++) {
			for (int j=0;j<ny;j++) {
				for (int k=0;k<nz;k++) {
					if (objMask[i][j][k] ) {
						temp = (int)img[i][j][k];
						if (temp >= min) {
							Histogram[temp-min] += 1;
							N += 1;
						}	
					}
				}
			}
		}
		return;		
	}

	public void findCentroids() {   
		double change;         /* Total change in hcrit */
		double guess;          /* Current guess of htol */
		double [] modes;       /* Vector of mode locations */
		int cnt=0;             /* Number of modes found by findmode, we can initialize to zero */
		int i;                 /* Counter variable */
		int maxmodes;
		double modetol;
		double low,high;

		low = HCRIT_LOW;
		high = HCRIT_HIGH;
		modetol = MODE_TOLERANCE;
		maxmodes = MAX_MODES;

		change = high - low;
		prior = 1.0 /(double) N;
		modes = new double[maxmodes];
		while (cnt != clusters) {
			guess = low + change/2;
			var = guess * guess;
			if (DEBUG) System.out.print("\n--- Current guess for hcrit is:"+guess+" ---\n");
			cnt = findmode(maxmodes,modetol,modes);
    
			if (DEBUG) {
			  System.out.print("Found "+cnt+" modes at: ");
			  for (i=1;i<=cnt;i++) {
			if (i>1) System.out.print(", ");
			System.out.print(modes[i]);
			  }
			  System.out.print(".\n");
			}
			
			if (cnt <= clusters)
				high = guess;
			else if (cnt >= (clusters+1))
				low = guess;
				change = high - low;
		}

		/* Copy modes to centroids */
		for (i=0;i<clusters;i++)
			Centroid[i] = modes[i+1];

		return;
	} // getInitCent

	/*--------------------------------------------------------------
	:   This function attempts to find the modes in a mixture model
	:   starting at xleft going to xright with a maximum number of
	:   nmax.  The function returns the number of counts and array
	:   of the position of the modes.
	:
	:   Written by Dzung Pham, Oct. 24, 1995
	:--------------------------------------------------------------*/
	/* int nmax;                    Maximum number of modes possible */
	/* double tol;                  Precision with which to find mode */
	/* double[] m;                  Vector of mode locations */
	
	int findmode(int nmax, double tol,double []m) {
		int cnt;                /* Number of modes found */
		double xleft, xright;
		int nb;                 /* Number of brackets found by zbrak */
		double[] xb1;
		double[] xb2;     /* Brackets found by zbrak */
		int i;                  /* Counter variable */
		double root;            /* Zero of the derivative */


		xb1 = new double [nmax];
		xb2 = new double [nmax];
		nb = nmax;
  
    	xleft = (double)min;
		xright = (double)max;
    
		/* First determine brackets */
		zbrak(xleft, xright, nmax, xb1, xb2, nb);
		if (DEBUG) System.out.print("# of brackets found: "+nb+"\n"); /**/
		if (nb == 0) {
            isCompleted = false;
 			errorLog = "Error: No modes found !";
			return 0;
		}

		/* Now find zeros */
		cnt = 0;
		for (i=0;i<nb;i++) {
			if (DEBUG) System.out.print("Using bracket: "+xb1[i]+", "+xb2[i]); /**/


			/* Find zero of first derivative */
			root = zbrent(xb1[i],xb2[i],tol); /**/
			/* Actually, zbrent's not necesarry since we don't need an exact
			value.  But since iterations are so low, it doesn' make much
			difference.  For a faster alternative, you can take the center
			value (such as below) or simply take one of the bracket values */
			/*    root = (xb2[i]-xb1[i])/2.0 + xb1[i]; */
			if (DEBUG) System.out.print("Root found at "+root+". Deriv value is "+fast_mixderiv(root)+".\n");/**/
			
			/* Now check second derivative to make sure it's a maximum */
			/*    printf("second derivative:%f\n",mixsecderiv(root)); */
			if ((slope(xb1[i],xb2[i]) < 0.0) && (root != 0.0)) {
				if (DEBUG) System.out.print("Found mode at "+m[cnt]+".\n"); /**/
				m[++cnt] = root;
			}
		}
		if (cnt == 0) {
            isCompleted = false;
 			errorLog = "Error: No modes found !";
			return 0;
		}

		return cnt;
	}

	/*-----------------------------------------------------------
	:   Computes first derivative at specific value of 
	:   kernel estimate. Relies solely on the histogram of the 
	:   data rather than on the data itself so is therefore
	:   "fast".
	:-----------------------------------------------------------*/


	double fast_mixderiv(double x)   { /* x :Position at which to compute derivative */
		int i;
		double y;
		double mu;

		y = 0.0;
		for (i=min;i<=max;i++) {
			mu = (double) i;
			y += (double)Histogram[i-min] * (-prior/(Math.sqrt(2*PI*var)) *(x-mu)/var
											* Math.exp(-0.5 * (x-mu) * (x-mu) / var));
		}
		return y;
	}

	/* Checks the brackets to see if slope was positive or negative */

	int slope(double x,  double y)   {
		if (fast_mixderiv(x) < fast_mixderiv(y)) return 1;
		else if (fast_mixderiv(x) > fast_mixderiv(y)) return -1;
		else return 0;
	}


	/*--------------------------------------------------------------
	:   File: zbrak.c
	:   This file takes an interval from x1 to x2, subdivides the 
	:   interval into n subsections, and finds brackets for zero 
	:   crossings of a function.  Taken from Numerical Recipes in
	:   C, modified for double precision by Dzung Pham, 10/28/95
	:   Used for finding the modes in the modal analysis program.
	:   Together with zbrent.c, finds the zeros of the first 
	:   derivative of the smoothed histogram.
	--------------------------------------------------------------*/
	void zbrak(double x1,double x2,int n,double xb1[],double xb2[],int nb){
		int nbb,i;
		double x,fp,fc,dx;

		nbb=0;
		dx=(x2-x1)/n;
		x = x1;
		fp=fast_mixderiv(x1);
		for (i=0;i<n;i++) {
			x += dx;
			fc=fast_mixderiv(x);
			if (fc*fp < 0.0) {
				xb1[++nbb]=x-dx;
				xb2[nbb]=x;
			if (nb == nbb) 
				return;
			}
			fp=fc;
		}
		nb = nbb;
	}

	/*-----------------------------------------------------------
	:   File: zbrent.c
	:   This function uses Brent's method to find the root of a 
	:   function known to lie between x1 and x2.  Taken from
	:   Numerical Recipes in C and modified
	:   for double precision by Dzung Pham, 10/28/95.
	:   Used in the modal analysis program, together with zbrak,
	:   to find the zeros of the first derivative of the smoothed
	:   histogram.
	:-----------------------------------------------------------*/
	double zbrent(double x1,double x2,double tol) {
		int iter;
		double a=x1,b=x2,c=0.0,d=0.0,e=0.0,min1,min2;
		double fa=fast_mixderiv(a);
		double fb=fast_mixderiv(b);
		double fc,p,q,r,s,tol1,xm;

		if ((fa > 0.0 && fb > 0.0) || (fa < 0.0 && fb < 0.0)) {
           isCompleted = false;
 			errorLog = "Error: Root must be bracketed in ZBRENT";
			return 0;
		}
		fc=fb;
		for (iter=1;iter<=ITMAX;iter++) {
			if ((fb > 0.0 && fc > 0.0) || (fb < 0.0 && fc < 0.0)) {
				c=a;
				fc=fa;
				e=d=b-a;
			}
			if (Math.abs(fc) < Math.abs(fb)) {
				a=b;
				b=c;
				c=a;
				fa=fb;
				fb=fc;
				fc=fa;
			}
			tol1=2.0*EPS*Math.abs(b)+0.5*tol;
			xm=0.5*(c-b);
			if (Math.abs(xm) <= tol1 || fb == 0.0) {
				return b;
			}
			if (Math.abs(e) >= tol1 && Math.abs(fa) > Math.abs(fb)) {
				s=fb/fa;
				if (a == c) {
					p=2.0*xm*s;
					q=1.0-s;
				} else {
					q=fa/fc;
					r=fb/fc;
					p=s*(2.0*xm*q*(q-r)-(b-a)*(r-1.0));
					q=(q-1.0)*(r-1.0)*(s-1.0);
				}
				if (p > 0.0)  q = -q;
				p=Math.abs(p);
				min1=3.0*xm*q-Math.abs(tol1*q);
				min2=Math.abs(e*q);
				if (2.0*p < (min1 < min2 ? min1 : min2)) {
					e=d;
					d=p/q;
				} else {
					d=xm;
					e=d;
				}
			} else {
				d=xm;
				e=d;
			}
			a=b;
			fa=fb;
			if (Math.abs(d) > tol1)
				b += d;
				else
					b += (xm > 0.0 ? Math.abs(tol1) : -Math.abs(tol1));
			fb=fast_mixderiv(b);
		}
		isCompleted = false;
		errorLog = "Error: Maximum number of iterations exceeded in ZBRENT";
		return 0;
	}
}
