package edu.jhmi.rad.medic.methods;

import java.io.*;
import java.util.*;
import gov.nih.mipav.view.*;
import gov.nih.mipav.model.structures.jama.*;

import edu.jhmi.rad.medic.libraries.*;
import edu.jhmi.rad.medic.structures.*;
import edu.jhmi.rad.medic.utilities.*;
 
/**
 *
 *  This algorithm handles registration algorithms
 *	of images with the Demons algorithm
 *	and a multiscale / multigrid technique
 *	(original Demons method, not robust)
 *
 *
 *	@version    November 2004
 *	@author     Pierre-Louis Bazin
 *		
 *
 */
 
public class DemonsScaledRegistration {
		
	// numerical quantities
	private static final	float   INF=1e30f;
	private static final	float   ZERO=1e-30f;
	
	// convenience tags
	private static final	int   X=0;
	private static final	int   Y=1;
	private static final	int   Z=2;
	private static final	int   T=3;
	
	// flag labels
	public static final	int   COMPOSITIVE 		= 101;
	public static final	int   DIFFEOMORPHIC 	= 102;
	
	public static final	int   FIXED 			= 201;
	public static final	int   MOVING 			= 202;
	public static final	int   SYMMETRIC 		= 203;
	
	public static final	int   GAUSS_FLUID 		= 301;
	public static final	int   GAUSS_DIFFUSION 	= 302;
	public static final	int   GAUSS_MIXED	 	= 303;
	public static final	int   DIV_FLUID 		= 304;
	public static final	int   DIV_DIFFUSION 	= 305;
	public static final	int   GAUSS_DIV_FLUID 		= 306;
	public static final	int   GAUSS_DIV_DIFFUSION 	= 307;
	
	public static final int	  RAW				= 401;
	public static final int	  NORMALIZED		= 402;
	public static final int	  SCALED			= 403;
	public static final int	  SEGMENTATION		= 404;
	public static final int	  FCM				= 405;
	public static final int	  MULTICHANNEL		= 406;
	
	// data buffers
	private 	float[][]		image;  		// source image
	private 	float[][]	    target;			// target image: best memberships
	private		float[][]		c;				// added transform from target space to original space (multiscale)
	private		float[][]		s;				// added transform from target space to original space (multiscale)
	private		float[][]		u;				// added transform update from target space to original space (multiscale)
	private static	int		nix,niy,niz;   	// image dimensions 
	private static	int		ntx,nty,ntz;   	// target dimensions
	private static	int		nsx,nsy,nsz;   	// deformation dimensions
	private static	float 	scale;
	private static	float	rix,riy,riz;   	// image resolutions (no pyramid)
	private static	float	rtx,rty,rtz;   	// target resolutions (no pyramid)
	private static 	int		nc;
	private		float		imin, imax;
	private		float[]		lb;
	private 		float[][]	transform;		// prior transform matrax (for instance from prior alignment)		
	
	// parameters
	private		float		smoothingKernel; 	// smoothing kernel size
	private		float		spatialScale; 		// scale of transformation
	private		boolean		useMask;
	private		float  		maskingThreshold;			// background threshold
	private		int		regType;
	private		int		forceType;
	private		int		fieldType;
	private		int		preType;
	private		int		Niter;
	
	// computation variables
	private		float			sigma2; 		// scale of transformation
	

	private		float[][]		gaussKernel;
	private		int				gx,gy,gz;
    
	private		float[][]		divKernelX, divKernelY, divKernelZ;

	// computation flags
	private 	boolean 		isWorking;
	private 	boolean 		isCompleted;
	
	// for debug and display
	static final boolean		debug=true;
	static final boolean		verbose=true;
    
	/**
	 *  constructor
     *  note: the membership function mems_ must have sum = 0 on the masked areas
	 */
	public DemonsScaledRegistration(float[] image_, float[] target_,
									int nix_, int niy_, int niz_,
									float rix_, float riy_, float riz_,
									int ntx_, int nty_, int ntz_,
									float rtx_, float rty_, float rtz_,
									float smoothing_,
									float spScale_,
									boolean useMask_,
									float maskVal_,
									float scale_, int Ni_, int Nt_,
									int reg_, int force_, int field_,
									int pre_,
									float[][] trans_) {
	
		scale = scale_;
		
		smoothingKernel = smoothing_;
		spatialScale = spScale_;
		sigma2 = spatialScale*spatialScale;
		
		useMask = useMask_;
		maskingThreshold = maskVal_;
		Niter = Ni_;

		transform = trans_;
		
		nix = nix_;
		niy = niy_;
		niz = niz_;
		
		rix = rix_;
		riy = riy_;
		riz = riz_;
		
		ntx = ntx_;
		nty = nty_;
		ntz = ntz_;
		
		rtx = rtx_;
		rty = rty_;
		rtz = rtz_;
			
		nsx = Numerics.ceil(ntx/scale);
		nsy = Numerics.ceil(nty/scale);
		nsz = Numerics.ceil(ntz/scale);
		
		regType = reg_;
		forceType = force_;
		fieldType = field_;
		preType = pre_;
		
		//initialize the smoothing kernel
		gaussKernel = ImageFunctions.separableGaussianKernel(smoothingKernel/rtx,smoothingKernel/rty,smoothingKernel/rtz);
		gx = (gaussKernel[X].length-1)/2;
		gy = (gaussKernel[Y].length-1)/2;
		gz = (gaussKernel[Z].length-1)/2;
		
		divKernelX = new float[3][];
		divKernelX[X] = new float[3];
		divKernelX[Y] = new float[1];
		divKernelX[Z] = new float[1];
		divKernelX[X][0] = 1.0f/4.0f;
		divKernelX[X][1] = 1.0f/2.0f;
		divKernelX[X][2] = 1.0f/4.0f;
		divKernelX[Y][0] = 1.0f;
		divKernelX[Z][0] = 1.0f;
				
		divKernelY = new float[3][];
		divKernelY[X] = new float[1];
		divKernelY[Y] = new float[3];
		divKernelY[Z] = new float[1];
		divKernelY[X][0] = 1.0f;
		divKernelY[Y][0] = 1.0f/4.0f;
		divKernelY[Y][1] = 1.0f/2.0f;
		divKernelY[Y][2] = 1.0f/4.0f;
		divKernelY[Z][0] = 1.0f;
				
		divKernelZ = new float[3][];
		divKernelZ[X] = new float[1];
		divKernelZ[Y] = new float[1];
		divKernelZ[Z] = new float[3];
		divKernelZ[X][0] = 1.0f;
		divKernelZ[Y][0] = 1.0f;
		divKernelZ[Z][0] = 1.0f/4.0f;
		divKernelZ[Z][1] = 1.0f/2.0f;
		divKernelZ[Z][2] = 1.0f/4.0f;

		
		try {			
			preprocessInputImages(image_, target_);
		} catch (OutOfMemoryError e){
			isWorking = false;
            finalize();
			System.out.println(e.getMessage());
			return;
		}
		isWorking = true;
		if (debug) MedicUtilPublic.displayMessage("Demons:initialisation\n");
	}

	final public void finalize() {
		image = null;
		target = null;
		s = null; u = null; c = null;
		System.gc();
	}
    
   public final float[][] getCurrentTransform() {
	   return s;
   }
   public final float[][] getCurrentUpdate() {
		return u;
	}
    
	public final boolean isWorking() { return isWorking; }
	public final boolean isCompleted() { return isCompleted; }
    
	
	private final void preprocessInputImages(float[] img, float[] trg) {
		if (preType==RAW) {
			nc = 1;
			image = new float[nc][];
			target = new float[nc][];
			image[0] = img;
			target[0] = trg;
		} else if (preType==NORMALIZED) {
			// find min, max
			imin = ImageFunctions.robustMinimum(img, 0.01f, 2, nix, niy, niz);
			imax = ImageFunctions.robustMaximum(img, 0.01f, 2, nix, niy, niz);
			
			float tmin = ImageFunctions.robustMinimum(trg, 0.01f, 2, ntx, nty, ntz);
			float tmax = ImageFunctions.robustMaximum(trg, 0.01f, 2, ntx, nty, ntz);

			if (debug) MedicUtilPublic.displayMessage("Image min / max: "+imin+" / "+imax+"\n");
			if (debug) MedicUtilPublic.displayMessage("Target min / max: "+tmin+" / "+tmax+"\n");
			
			// allocate
			nc = 1;
			image = new float[nc+1][];
			target = new float[nc][];
			image[0] = ImageFunctions.normalize(img, nix,niy,niz, imin, imax);
			target[0] = ImageFunctions.normalize(trg, ntx,nty,ntz, tmin, tmax);
			// keep original image for results
			image[1] = img;
		} else if (preType==SCALED) {
			// find min, max
			imin = ImageFunctions.robustMinimum(img, 0.01f, 2, nix, niy, niz);
			imax = ImageFunctions.robustMaximum(img, 0.01f, 2, nix, niy, niz);
			
			float tmin = ImageFunctions.robustMinimum(trg, 0.01f, 2, ntx, nty, ntz);
			float tmax = ImageFunctions.robustMaximum(trg, 0.01f, 2, ntx, nty, ntz);
			
			if (debug) MedicUtilPublic.displayMessage("Image min / max: "+imin+" / "+imax+"\n");
			if (debug) MedicUtilPublic.displayMessage("Target min / max: "+tmin+" / "+tmax+"\n");
			
			ImageFunctions.rescale(img, nix,niy,niz, imin, imax);
			ImageFunctions.rescale(trg, ntx,nty,ntz, tmin, tmax);
			
			// allocate
			nc = 1;
			image = new float[nc][];
			target = new float[nc][];
			image[0] = img;
			target[0] = trg;
			
		} else if (preType==SEGMENTATION) {
			// find max number of objects and list them; assume the target has proper labeling
			lb = ObjectProcessing.listLabels(trg,ntx,nty,ntz);
			nc = lb.length;
			if (useMask) nc--;
			
			image = new float[nc][];
			target = new float[nc][];
			
			int c=0;
			for (int n=0;n<lb.length;n++) {
				if ( (!useMask) || (lb[n]!=maskingThreshold) ) {
					image[c] = ObjectProcessing.floatObjectFromImage(img,nix,niy,niz,lb[n],ObjectProcessing.EQUAL);
					target[c] = ObjectProcessing.floatObjectFromImage(trg,nix,niy,niz,lb[n],ObjectProcessing.EQUAL);
					c++;
				}			
			}
		} else if (preType==FCM) {
			// perform basic FCM segmentation
			nc = 3;
			//...
		} else if (preType==MULTICHANNEL) {
			nc = Numerics.round(img.length/(nix*niy*niz));
			image = new float[nc][nix*niy*niz];
			target = new float[nc][ntx*nty*ntz];
			for (int n=0;n<nc;n++) for (int xyz=0;xyz<nix*niy*niz;xyz++) {
				image[n][xyz] = img[xyz + n*nix*niy*niz];
			}
			for (int n=0;n<nc;n++) for (int xyz=0;xyz<ntx*nty*ntz;xyz++) {
				target[n][xyz] = trg[xyz + n*ntx*nty*ntz];
			}
			img = null;
			trg = null;
		} else {
			nc = 1;
			image = new float[nc][];
			target = new float[nc][];
			image[0] = img;
			target[0] = trg;
		}	
	}
	
    /** initialize the transform from previous estimate, if exists */
    public final void initializeTransform() {
				
		// initialization: start from prior transform or zero
		s = new float[3][nsx*nsy*nsz];
		if (transform==null) {
			for (int x=0;x<nsx;x++) for (int y=0;y<nsy;y++) for (int z=0;z<nsz;z++) {
				int xyz = x+nsx*y+nsx*nsy*z;
				s[X][xyz] = x*scale*rtx/rix;
				s[Y][xyz] = y*scale*rty/riy;
				s[Z][xyz] = z*scale*rtz/riz;
			}
		} else {
			for (int x=0;x<nsx;x++) for (int y=0;y<nsy;y++) for (int z=0;z<nsz;z++) {
				int xyz = x+nsx*y+nsx*nsy*z;
				s[X][xyz] = transform[X][X]*x*scale + transform[X][Y]*y*scale + transform[X][Z]*z*scale + transform[X][T];
				s[Y][xyz] = transform[Y][X]*x*scale + transform[Y][Y]*y*scale + transform[Y][Z]*z*scale + transform[Y][T];
				s[Z][xyz] = transform[Z][X]*x*scale + transform[Z][Y]*y*scale + transform[Z][Z]*z*scale + transform[Z][T];
			}
		}
				
		// update always zero
		u = new float[3][nsx*nsy*nsz];
		for (int xyz=0;xyz<nsx*nsy*nsz;xyz++) {
			u[X][xyz] = 0.0f;
			u[Y][xyz] = 0.0f;
			u[Z][xyz] = 0.0f;
		}
		
		// composite update always zero
		c = new float[3][nsx*nsy*nsz];
		for (int xyz=0;xyz<nsx*nsy*nsz;xyz++) {
			c[X][xyz] = 0.0f;
			c[Y][xyz] = 0.0f;
			c[Z][xyz] = 0.0f;
		}
		
		return;
    }
    
	
    /**
	 * compute the image position given the target
	 * for a given level l
     * performs only one iteration
	 */
    final public float registerImageToTarget() {
		
		if (debug) MedicUtilPublic.displayMessage("update: ");
		
		float meanDiff = 0.0f;
		for (int x=1;x<nsx-1;x++) for (int y=1;y<nsy-1;y++) for (int z=1;z<nsz-1;z++) {
			int xyz = x+nsx*y+nsx*nsy*z;
		
			// compute the update field
			float xs = s[X][xyz];
			float ys = s[Y][xyz];
			float zs = s[Z][xyz];
		
			float xsmx = s[X][xyz-1];
			float ysmx = s[Y][xyz-1];
			float zsmx = s[Z][xyz-1];
	
			float xspx = s[X][xyz+1];
			float yspx = s[Y][xyz+1];
			float zspx = s[Z][xyz+1];
	
			float xsmy = s[X][xyz-nsx];
			float ysmy = s[Y][xyz-nsx];
			float zsmy = s[Z][xyz-nsx];
	
			float xspy = s[X][xyz+nsx];
			float yspy = s[Y][xyz+nsx];
			float zspy = s[Z][xyz+nsx];
	
			float xsmz = s[X][xyz-nsx*nsy];
			float ysmz = s[Y][xyz-nsx*nsy];
			float zsmz = s[Z][xyz-nsx*nsy];
	
			float xspz = s[X][xyz+nsx*nsy];
			float yspz = s[Y][xyz+nsx*nsy];
			float zspz = s[Z][xyz+nsx*nsy];
			
			float xt = x*scale;
			float yt = y*scale;
			float zt = z*scale;
			
			u[X][xyz] = 0.0f;
			u[Y][xyz] = 0.0f;
			u[Z][xyz] = 0.0f;
			float den = 0.0f;
			for (int n=0;n<nc;n++) {
			
				float diff = (ImageFunctions.linearClosestInterpolation(target[n],xt,yt,zt,ntx,nty,ntz) 
							- ImageFunctions.linearClosestInterpolation(image[n],xs,ys,zs,nix,niy,niz));				
				
				float Jx, Jy, Jz;
				if (forceType==FIXED) {
					Jx = 0.5f/rtx*(ImageFunctions.linearClosestInterpolation(target[n],xt+scale,yt,zt,ntx,nty,ntz) - ImageFunctions.linearClosestInterpolation(target[n],xt-scale,yt,zt,ntx,nty,ntz));
					Jy = 0.5f/rty*(ImageFunctions.linearClosestInterpolation(target[n],xt,yt+scale,zt,ntx,nty,ntz) - ImageFunctions.linearClosestInterpolation(target[n],xt,yt-scale,zt,ntx,nty,ntz));
					Jz = 0.5f/rtz*(ImageFunctions.linearClosestInterpolation(target[n],xt,yt,zt+scale,ntx,nty,ntz) - ImageFunctions.linearClosestInterpolation(target[n],xt,yt,zt-scale,ntx,nty,ntz));
				} else if (forceType==MOVING) {
					Jx = 0.5f/rix*(ImageFunctions.linearClosestInterpolation(image[n],xspx,yspx,zspx,nix,niy,niz)-ImageFunctions.linearClosestInterpolation(image[n],xsmx,ysmx,zsmx,nix,niy,niz));
					Jy = 0.5f/riy*(ImageFunctions.linearClosestInterpolation(image[n],xspy,yspy,zspy,nix,niy,niz)-ImageFunctions.linearClosestInterpolation(image[n],xsmy,ysmy,zsmy,nix,niy,niz));
					Jz = 0.5f/riz*(ImageFunctions.linearClosestInterpolation(image[n],xspz,yspz,zspz,nix,niy,niz)-ImageFunctions.linearClosestInterpolation(image[n],xsmz,ysmz,zsmz,nix,niy,niz));
				} else {
					Jx = 0.25f/rtx*(ImageFunctions.linearClosestInterpolation(target[n],xt+scale,yt,zt,ntx,nty,ntz) - ImageFunctions.linearClosestInterpolation(target[n],xt-scale,yt,zt,ntx,nty,ntz));
					Jy = 0.25f/rty*(ImageFunctions.linearClosestInterpolation(target[n],xt,yt+scale,zt,ntx,nty,ntz) - ImageFunctions.linearClosestInterpolation(target[n],xt,yt-scale,zt,ntx,nty,ntz));
					Jz = 0.25f/rtz*(ImageFunctions.linearClosestInterpolation(target[n],xt,yt,zt+scale,ntx,nty,ntz) - ImageFunctions.linearClosestInterpolation(target[n],xt,yt,zt-scale,ntx,nty,ntz));

					Jx += 0.25f/rix*(ImageFunctions.linearClosestInterpolation(image[n],xspx,yspx,zspx,nix,niy,niz)-ImageFunctions.linearClosestInterpolation(image[n],xsmx,ysmx,zsmx,nix,niy,niz));
					Jy += 0.25f/riy*(ImageFunctions.linearClosestInterpolation(image[n],xspy,yspy,zspy,nix,niy,niz)-ImageFunctions.linearClosestInterpolation(image[n],xsmy,ysmy,zsmy,nix,niy,niz));
					Jz += 0.25f/riz*(ImageFunctions.linearClosestInterpolation(image[n],xspz,yspz,zspz,nix,niy,niz)-ImageFunctions.linearClosestInterpolation(image[n],xsmz,ysmz,zsmz,nix,niy,niz));
				}
			
				float J2 = Jx*Jx+Jy*Jy+Jz*Jz;
				den += diff*diff/sigma2 + J2;
				
				u[X][xyz] += diff*Jx;
				u[Y][xyz] += diff*Jy;
				u[Z][xyz] += diff*Jz;
				
				meanDiff += Numerics.abs(diff);
			}
			u[X][xyz] /= Numerics.max(ZERO,den);
			u[Y][xyz] /= Numerics.max(ZERO,den);
			u[Z][xyz] /= Numerics.max(ZERO,den);			
		}
		meanDiff /= (ntx*nty*ntz);
		
		if (regType==GAUSS_FLUID || regType==GAUSS_MIXED || regType==GAUSS_DIV_FLUID) {
			if (debug) MedicUtilPublic.displayMessage("GAUSS_FLUID regularization \n");
		
			// smooth the result with a gaussian kernel
			u[X] = ImageFunctions.separableConvolution(u[X],nsx,nsy,nsz,gaussKernel,gx,gy,gz);
			u[Y] = ImageFunctions.separableConvolution(u[Y],nsx,nsy,nsz,gaussKernel,gx,gy,gz);
			u[Z] = ImageFunctions.separableConvolution(u[Z],nsx,nsy,nsz,gaussKernel,gx,gy,gz);
		} else if (regType==DIV_FLUID || regType==GAUSS_DIV_FLUID) {
		
			// smooth the result with a gaussian kernel
			u[X] = ImageFunctions.separableConvolution(u[X],nsx,nsy,nsz,divKernelX,1,0,0);
			u[Y] = ImageFunctions.separableConvolution(u[Y],nsx,nsy,nsz,divKernelY,0,1,0);
			u[Z] = ImageFunctions.separableConvolution(u[Z],nsx,nsy,nsz,divKernelZ,0,0,1);
		}
		
		// compose the transformations
		if (fieldType==COMPOSITIVE) {
			if (debug) MedicUtilPublic.displayMessage("compose with current transform \n");
							
			for (int x=0;x<nsx;x++) for (int y=0;y<nsy;y++) for (int z=0;z<nsz;z++) {
				int xyz = x+nsx*y+nsx*nsy*z;
				
				float xu = x+u[X][xyz];
				float yu = y+u[Y][xyz];
				float zu = z+u[Z][xyz];
				
				// note: if outside, extrapolate as X+u
				c[X][xyz] = ImageFunctions.linearClosestInterpolation(s[X],xu,yu,zu,nsx,nsy,nsz) - x*scale;
				c[Y][xyz] = ImageFunctions.linearClosestInterpolation(s[Y],xu,yu,zu,nsx,nsy,nsz) - y*scale;
				c[Z][xyz] = ImageFunctions.linearClosestInterpolation(s[Z],xu,yu,zu,nsx,nsy,nsz) - z*scale;
			}
		}
		
		if (regType==GAUSS_DIFFUSION || regType==GAUSS_MIXED || regType==GAUSS_DIV_DIFFUSION) {
			if (debug) MedicUtilPublic.displayMessage("GAUSS_DIFFUSION regularization \n");
			
			// smooth the result with a gaussian kernel
			c[X] = ImageFunctions.separableConvolution(c[X],nsx,nsy,nsz,gaussKernel,gx,gy,gz);
			c[Y] = ImageFunctions.separableConvolution(c[Y],nsx,nsy,nsz,gaussKernel,gx,gy,gz);
			c[Z] = ImageFunctions.separableConvolution(c[Z],nsx,nsy,nsz,gaussKernel,gx,gy,gz);
		} else if (regType==DIV_DIFFUSION || regType==GAUSS_DIV_DIFFUSION) {
			if (debug) MedicUtilPublic.displayMessage("GAUSS_DIFFUSION regularization \n");
			
			// smooth the result with a gaussian kernel
			c[X] = ImageFunctions.separableConvolution(c[X],nsx,nsy,nsz,divKernelX,1,0,0);
			c[Y] = ImageFunctions.separableConvolution(c[Y],nsx,nsy,nsz,divKernelY,0,1,0);
			c[Z] = ImageFunctions.separableConvolution(c[Z],nsx,nsy,nsz,divKernelZ,0,0,1);
		}
					
		float meanC = 0.0f;
		for (int x=0;x<nsx;x++) for (int y=0;y<nsy;y++) for (int z=0;z<nsz;z++) {
			int xyz = x+nsx*y+nsx*nsy*z;
			
			s[X][xyz] = x*scale + c[X][xyz];
			s[Y][xyz] = y*scale + c[Y][xyz];
			s[Z][xyz] = z*scale + c[Z][xyz];
			
			meanC += c[X][xyz]*c[X][xyz]+c[Y][xyz]*c[Y][xyz]+c[Z][xyz]*c[Z][xyz];
		}
		meanC /= (nsx*nsy*nsz);
		
		if (debug) MedicUtilPublic.displayMessage("convergence "+meanC+" -> "+meanDiff+"\n");

        return meanDiff;
    } // 
    
    /** 
	 *	runs the algorithm
	 */
	public final void runRegistration() {        
        // top level
        if (debug) MedicUtilPublic.displayMessage("initialize all parameters \n");
        initializeTransform();
		
		float dist=0, dist0=0, prec=0;
		for (int t=0;t<Niter;t++) {
			if (debug) MedicUtilPublic.displayMessage("iteration "+(t+1)+"\n");
			prec = dist;
			dist = registerImageToTarget();
			if (t==0) dist0 = dist;
			else {
				MedicUtilPublic.displayMessage("score: "+(dist/dist0)+", variation: "+( (dist-prec)/dist0)+"\n");
			}
		}
    }   
	
	/** 
	 *	returns the transformed coordinates
	 */
	public final float[] getCurrentMapping(int x, int y, int z) {
		float xs = x/scale;
		float ys = y/scale;
		float zs = z/scale;
		
		float[] Xs = new float[3];
		Xs[X] = ImageFunctions.linearClosestInterpolation(s[X],xs,ys,zs,nsx,nsy,nsz);
		Xs[Y] = ImageFunctions.linearClosestInterpolation(s[Y],xs,ys,zs,nsx,nsy,nsz);
		Xs[Z] = ImageFunctions.linearClosestInterpolation(s[Z],xs,ys,zs,nsx,nsy,nsz);
		return Xs;
	}// getCurrentMapping

    /** 
	 *	returns the transformed image
	 */
	public final float[][] exportTransformedImage() {
		float 	xs,ys,zs;
		float[][]	img;
		
		if (preType==MULTICHANNEL) img = new float[nc][ntx*nty*ntz];
		else img = new float[1][ntx*nty*ntz];
		
		if (preType==SCALED) ImageFunctions.unscale(image[0],nix,niy,niz,imin,imax);
		
        for (int x=0;x<ntx;x++) for (int y=0;y<nty;y++) for (int z=0;z<ntz;z++) {
			int xyz = x+ntx*y+ntx*nty*z;
			xs = ImageFunctions.linearClosestInterpolation(s[X],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			ys = ImageFunctions.linearClosestInterpolation(s[Y],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			zs = ImageFunctions.linearClosestInterpolation(s[Z],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			
				// compute interpolated values
			if (preType==RAW) {
				img[0][xyz] = ImageFunctions.linearClosestInterpolation(image[0],xs,ys,zs,nix,niy,niz);
			} else if (preType==NORMALIZED) {
				img[0][xyz] = ImageFunctions.linearClosestInterpolation(image[1],xs,ys,zs,nix,niy,niz);
			} else if (preType==SCALED) {
				img[0][xyz] = ImageFunctions.linearClosestInterpolation(image[0],xs,ys,zs,nix,niy,niz);
			} else if (preType==SEGMENTATION) {
				float nbest = maskingThreshold;
				float best = 0.0f;
				int c = 0;
				for (int n=0;n<lb.length;n++) if ( (!useMask) || (lb[n]!=maskingThreshold) ) {
					float score = ImageFunctions.linearClosestInterpolation(image[c],xs,ys,zs,nix,niy,niz);
					if (score>best) {
						best = score;
						nbest = lb[n];
					}
					c++;
				}
				img[0][xyz] = nbest;
			} else if (preType==MULTICHANNEL) {
				for (int n=0;n<nc;n++) {
					img[n][xyz] = ImageFunctions.linearClosestInterpolation(image[n],xs,ys,zs,nix,niy,niz);
				}
			}
		}
		return img;
	} // exportTransformedImage

    /** 
	 *	returns the transform field v defined as X' = X+v (=s)
	 */
	public final float[][] exportTransformField() {
		float 	xs,ys,zs;
		float[][]	vec = new float[3][ntx*nty*ntz];
		
        for (int x=0;x<ntx;x++) for (int y=0;y<nty;y++) for (int z=0;z<ntz;z++) {
			int xyz = x+ntx*y+ntx*nty*z;
			
			xs = ImageFunctions.linearClosestInterpolation(s[X],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			ys = ImageFunctions.linearClosestInterpolation(s[Y],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			zs = ImageFunctions.linearClosestInterpolation(s[Z],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			
			vec[X][xyz] = xs-x;
			vec[Y][xyz] = ys-y;
			vec[Z][xyz] = zs-z;			
 		}
		return vec;
	} // exportTransformedImage

	/** 
	 *	returns the image residuals
	 */
	public final float[][] exportResiduals() {
		float 	xs,ys,zs;
		float[][]	img;
		
		if (preType==MULTICHANNEL) img = new float[nc][ntx*nty*ntz];
		else img = new float[1][ntx*nty*ntz];
		
		for (int x=0;x<ntx;x++) for (int y=0;y<nty;y++) for (int z=0;z<ntz;z++) {
			int xyz = x+ntx*y+ntx*nty*z;
    		xs = ImageFunctions.linearClosestInterpolation(s[X],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			ys = ImageFunctions.linearClosestInterpolation(s[Y],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			zs = ImageFunctions.linearClosestInterpolation(s[Z],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			
				// compute interpolated values
			if (preType==RAW || preType==NORMALIZED || preType==SCALED) {
				img[0][xyz] = Numerics.square(target[0][xyz] - ImageFunctions.linearClosestInterpolation(image[0],xs,ys,zs,nix,niy,niz));
			} else if (preType==SEGMENTATION) {
				img[0][xyz] = 0.0f;
				for (int n=0;n<nc;n++) {
					img[0][xyz] += Numerics.square(target[n][xyz]-ImageFunctions.linearClosestInterpolation(image[n],xs,ys,zs,nix,niy,niz));
				}
			} else if (preType==MULTICHANNEL) {
				for (int n=0;n<nc;n++) {
					img[n][xyz] = Numerics.square(target[n][xyz]-ImageFunctions.linearClosestInterpolation(image[n],xs,ys,zs,nix,niy,niz));
				}
			}
		}
		return img;
	} // exportResiduals

	
}
