package edu.jhmi.rad.medic.visualization.models;

import javax.media.opengl.*;
import javax.media.opengl.glu.*;
import java.util.Vector;
import edu.jhmi.rad.medic.visualization.primitives.*;
import edu.jhmi.rad.medic.visualization.*;

import javax.vecmath.Vector3f;
import javax.vecmath.Point3f;

public class TriangleMesh implements Drawable {
	private String m_Name;								//Name of the surface
	private Vector<Vertex> m_Vertices;					//List of Vertices in the surface
	private Vector<Vector<Integer>> m_Polygons;			//List of polygons in the surface. Each polygon is a further list of vertice indices.
	private int m_FileType;								//Type of file, Ascii or Binary	
	private int m_DisplayListIdSurface;					//OpenGL List Id for surface
	private int m_DisplayListIdEdges;					//OpenGL List Id for edges		
	private boolean m_UseCallList;						//Use Call Lists or not
	private BoundingBox bBox;							//Surface Bounding Box
	private boolean drawEdges;							//If true draw surface as edges, instead of polygons
	
	//Constants
	public static final int FILE_ASCII	= 1;			
	public static final int FILE_BINARY	= 2;
	
	/*
	 * ctor
	 */
	public TriangleMesh() {
		m_UseCallList = false;
		drawEdges = false;
		bBox = new BoundingBox();
	}
	
	/*
	 * dtor
	 */
	protected void finalize() {		
		dispose();
	}
	
	public void dispose() {
		System.out.println("Surface `" + getName() + "` freeing up memory ... ");		
		
		int polynum = getPolygonCount();
		int vnum = getVertexCount();
		
		// Free polygon list
		for(int i = 0; i < polynum; ++i) {
			m_Polygons.elementAt(i).clear();
			m_Polygons.set(i,null);
		}
		m_Polygons.clear();
		m_Polygons = null;
		
		// Free vertex list
		for(int i = 0; i < vnum; ++i) {
			m_Vertices.elementAt(i).point	= null;
			m_Vertices.elementAt(i).normal	= null;
			m_Vertices.elementAt(i).color	= null;
			m_Vertices.set(i,null);
		}
		m_Vertices.clear();
		m_Vertices = null;
		
		// Free openGL handles
		if(m_UseCallList) {				
			// @BUG Unable to free call list
			System.out.println("Freeing GL Call Lists @BUG");
			//m_GL.glDeleteLists(m_DisplayListIdSurface, 2);			
		}		
				
		// Garbage collect
		System.gc();
	}
	
	/*
	 * Sets the draw parameters for the surface, either to draw polygons
	 * or to draw edges.
	 *	@param	drawParams		The drawing parameter Edges
	 */
	public void setDrawParam(DrawParam drawParams) {
		if(drawParams.getName().compareTo("Edges") == 0)
			drawEdges = Boolean.parseBoolean(drawParams.getValue());
	}
	
	/*
	 * Returns the bounding box for this object
	 */
	public BoundingBox getBoundingBox() {
		return bBox;
	}
	
	/*
	 * Called to do some precomputation
	 */
	public void preCompute() {}
	
	/*
	 * Sets up the call list for this surface. Lists for both edges and polygons are generated
	 *	@param	drawable	Object containing the active openGL reference	 
	 */
	public boolean setupGLList(GLAutoDrawable glAutoDrawable) {
		GL gl = glAutoDrawable.getGL();		
		m_DisplayListIdSurface	= gl.glGenLists(2);
		m_DisplayListIdEdges	= m_DisplayListIdSurface + 1;
		gl.glNewList(m_DisplayListIdSurface, GL.GL_COMPILE);
		drawSurface(glAutoDrawable, false);
    	gl.glEndList();
    	gl.glNewList(m_DisplayListIdEdges, GL.GL_COMPILE);
		drawSurface(glAutoDrawable, true);
    	gl.glEndList();    	
    	m_UseCallList = true;
    	return m_UseCallList;
	}
	
	
	/*
	 * Draws the actual surface and associated edges.
	 *	@param	drawable	Object containing the active openGL reference
	 *	@param	drawAsEdges	Draw surface as edges instead of polygons
	 */
	private void drawSurface(GLAutoDrawable drawable, boolean drawAsEdges) {
		GL gl = drawable.getGL();		
		int polynum = getPolygonCount();
		Vertex v1, v2, v3;		
		for(int i = 0; i < polynum; ++i) {
			v1 = m_Vertices.get(m_Polygons.get(i).get(0));
			v2 = m_Vertices.get(m_Polygons.get(i).get(1));
			v3 = m_Vertices.get(m_Polygons.get(i).get(2));
			if(!drawAsEdges)
				gl.glBegin(gl.GL_TRIANGLES);
			else
				gl.glBegin(gl.GL_LINE_LOOP);
			gl.glNormal3f(v1.normal.x, v1.normal.y, v1.normal.z);
			gl.glVertex3f(v1.point.x , v1.point.y , v1.point.z );
				
			gl.glNormal3f(v2.normal.x, v2.normal.y, v2.normal.z);
			gl.glVertex3f(v2.point.x , v2.point.y , v2.point.z );
			
			gl.glNormal3f(v3.normal.x, v3.normal.y, v3.normal.z);
			gl.glVertex3f(v3.point.x , v3.point.y , v3.point.z );
			gl.glEnd(); 
		}
	}
	
	/*
	 * Draws the surface. This function should be called by
	 *	@param	drawable	Object containing the active openGL reference
	 *	@param	drawAsEdges	Draw surface as edges instead of polygons
	 */
	public void drawGL(GLAutoDrawable glAutoDrawable) {
		GL gl = glAutoDrawable.getGL();				
			
		if(!drawEdges) {		
			if(m_UseCallList) 
				gl.glCallList(m_DisplayListIdSurface);
			else
				drawSurface(glAutoDrawable, false);
		}
		else {		
			if(m_UseCallList)		
				gl.glCallList(m_DisplayListIdEdges);
			else
				drawSurface(glAutoDrawable, true);  
		}
	}
		
	/*
	 * Computes the bounding box for this surface.
	 */
	public void computeBoundingBox() {		
		int vnum = getVertexCount();				
		Point3f temp =  new Point3f(m_Vertices.get(0).point.x, m_Vertices.get(0).point.y, m_Vertices.get(0).point.z);
		Point3f m_BoxMin, m_BoxMax;
		
		m_BoxMin = new Point3f(temp.x, temp.y, temp.z);
		m_BoxMax = new Point3f(temp.x, temp.y, temp.z);
		
		for(int i = 1; i  < vnum; ++i) {
			temp.set(m_Vertices.get(i).point.x, m_Vertices.get(i).point.y, m_Vertices.get(i).point.z);			
			
			if(m_BoxMin.x > temp.x)
				m_BoxMin.x = temp.x;
			if(m_BoxMin.y > temp.y)
				m_BoxMin.y = temp.y;
			if(m_BoxMin.z > temp.z)
				m_BoxMin.z = temp.z;
				
			if(m_BoxMax.x < temp.x)
				m_BoxMax.x = temp.x;
			if(m_BoxMax.y < temp.y)
				m_BoxMax.y = temp.y;
			if(m_BoxMax.z < temp.z)
				m_BoxMax.z = temp.z;
				
		}		
			
		bBox.set(m_BoxMin, m_BoxMax);
		System.out.println("Surface : " + getName());
		System.out.println("Min [" + m_BoxMin.x + ", " + m_BoxMin.y + ", " + m_BoxMin.z + "]");
		System.out.println("Max [" + m_BoxMax.x + ", " + m_BoxMax.y + ", " + m_BoxMax.z + "]");
		
		temp = m_BoxMin = m_BoxMax = null;
	}
	
	/*
	 * Computes the per-vertex normals. Simple averaging of face normals is used.
	 */
	public void computeNormals() {
		int vnum = getVertexCount();
		int pnum = getPolygonCount();
		float length = 0;
		Vector3f avec, bvec, normal;
		
		avec	= new Vector3f(0,0,0);
		bvec	= new Vector3f(0,0,0);
		normal	= new Vector3f(0,0,0);
		
		System.out.println("Computing Per-Vertex Normals ... ");
		
		for(int i = 0; i < pnum; ++i) {
			avec.sub(m_Vertices.get(m_Polygons.get(i).get(1)).point , m_Vertices.get(m_Polygons.get(i).get(0)).point);
			bvec.sub(m_Vertices.get(m_Polygons.get(i).get(2)).point , m_Vertices.get(m_Polygons.get(i).get(0)).point);
			normal.cross(avec, bvec);
			
			m_Vertices.elementAt(m_Polygons.get(i).get(0)).normal.add(normal);
			m_Vertices.elementAt(m_Polygons.get(i).get(1)).normal.add(normal);
			m_Vertices.elementAt(m_Polygons.get(i).get(2)).normal.add(normal);
		}
		
		for(int i = 0; i < vnum; ++i)
			m_Vertices.elementAt(i).normal.normalize();
			
		avec = bvec = normal = null;		
	}
	
	/*
	 * Intersects the surface with a plane and returns the lines of intersection for each polygon.
	 *	@param	point	A point on the plane
	 *	@param 	normal	Normal of the plane
	 *	@return			A vector contain the lines formed by intersecting a plane with a polygon
	 */
	public Vector<Line> intersectPlane(Vector3f point, Vector3f normal) {
		Vector<Line> lines = new Vector<Line>();
		int polynum = getPolygonCount();
		Vector3f p1, i1, i2;
		p1 = new Vector3f(0,0,0);
		i1 = new Vector3f(0,0,0);		
		i2 = new Vector3f(0,0,0);		
		float d = point.dot(normal);		
		float u, denom;
		int intersects = 0;
		Line l;
		for(int i = 0; i < polynum; ++i) {
			intersects = 0;			
			i1.set(0,0,0);	
			i2.set(0,0,0);
			
			// Side A
			p1.set(0,0,0);				
			p1.sub(m_Vertices.get(m_Polygons.get(i).get(1)).point , m_Vertices.get(m_Polygons.get(i).get(0)).point);
			u = computePlaneIntersection(normal, m_Vertices.get(m_Polygons.get(i).get(0)).point, p1, d);
			if(u >= 0 && u <= 1) {
				i1.set(m_Vertices.get(m_Polygons.get(i).get(0)).point.x, m_Vertices.get(m_Polygons.get(i).get(0)).point.y, m_Vertices.get(m_Polygons.get(i).get(0)).point.z);
				p1.scale(u);
				i1.add(p1);				
				intersects++;				
			}
			
			// Side B
			p1.set(0,0,0);			
			p1.sub(m_Vertices.get(m_Polygons.get(i).get(2)).point , m_Vertices.get(m_Polygons.get(i).get(0)).point);
			u = computePlaneIntersection(normal, m_Vertices.get(m_Polygons.get(i).get(0)).point, p1, d);
			if(u >= 0 && u <= 1) {				
				p1.scale(u);
				if(intersects == 0) {				
					i1.set(m_Vertices.get(m_Polygons.get(i).get(0)).point.x, m_Vertices.get(m_Polygons.get(i).get(0)).point.y, m_Vertices.get(m_Polygons.get(i).get(0)).point.z);
					i1.add(p1);
				}
				else {			
					i2.set(m_Vertices.get(m_Polygons.get(i).get(0)).point.x, m_Vertices.get(m_Polygons.get(i).get(0)).point.y, m_Vertices.get(m_Polygons.get(i).get(0)).point.z);	
					i2.add(p1);
				}
				intersects++;					
			}
			
			// Side C
			if(intersects != 2) {
				p1.set(0,0,0);			
				p1.sub(m_Vertices.get(m_Polygons.get(i).get(2)).point , m_Vertices.get(m_Polygons.get(i).get(1)).point);
				u = computePlaneIntersection(normal, m_Vertices.get(m_Polygons.get(i).get(0)).point, p1, d);
				if(u >= 0 && u <= 1) {
					i2.set(m_Vertices.get(m_Polygons.get(i).get(1)).point.x, m_Vertices.get(m_Polygons.get(i).get(1)).point.y, m_Vertices.get(m_Polygons.get(i).get(1)).point.z);
					p1.scale(u);
					i2.add(p1);
					intersects++;										
				}
			}
			
			if(intersects == 2) {
				l = new Line();
				l.p1.set(i1.x, i1.y, i1.z);
				l.p2.set(i2.x, i2.y, i2.z);
				lines.addElement(l);
			}
			
		}
		return lines;
	}
	
	/*
	 * Computes a plane-ray intersection.
	 *	@param	normal	Normal of the plane
	 *	@param	start	Vector indicating starting position of the ray
	 *	@param	line	Vector indicating direction
	 *	@param	d		Distance from origin of the plane
	 *	@return			Scalar value. Point of intersection = start + line * scalar
	 */
	private float computePlaneIntersection(Vector3f normal, Vector3f start, Vector3f line, float d) {
		float denom = 0, u = -1;
		denom = line.dot(normal);
		if(denom == 0)
			return -1;
		u = d - start.dot(normal);
		return u / denom;
	}
	
	
	public float computeDistance(Vector3f point) {
		float distance = Float.POSITIVE_INFINITY;
		float pd = 0.0f;
		float d  = 0.0f;
		float area1, area2, area3;
		float u, v, w;		
		int polynum = getPolygonCount();
		Vector3f normal, isectPoint;
		Vector3f vecA, vecB, vecC;		
		area1 = area2 = area3 = u = v = w = 0.0f;					
		normal 		= new Vector3f(0,0,0);
		vecA   		= new Vector3f(0,0,0);
		vecB		= new Vector3f(0,0,0);
		vecC		= new Vector3f(0,0,0);
		isectPoint	= new Vector3f(0,0,0);		
		
		System.out.println("Point is :  "  + point.x + "," + point.y + "," + point.z);
		
		for(int i = 0 ; i < polynum; ++i) {
			// Compute triangle normal
			vecA.sub(m_Vertices.get(m_Polygons.get(i).get(1)).point , m_Vertices.get(m_Polygons.get(i).get(0)).point);
			vecB.sub(m_Vertices.get(m_Polygons.get(i).get(2)).point , m_Vertices.get(m_Polygons.get(i).get(0)).point);
			normal.cross(vecA, vecB);		
			
			// Compute Area of Triangle	
			vecC.normalize(normal);
			area1 = vecC.dot(normal);	
				
			// Normalize normal		
			normal.normalize();
			pd = m_Vertices.get(m_Polygons.get(i).get(1)).point.dot(normal);
			
			// Compute distance to plane represented by triangle
			vecC.sub(point, m_Vertices.get(m_Polygons.get(i).get(1)).point);
			d = normal.dot(vecC);
			
			if(d < 0.0) 				
				continue;			
					
			// Compute point of intersection
			isectPoint.set(point.x, point.y, point.z);
			
			vecC.set(-normal.x, -normal.y, -normal.z);
			vecC.scale(d);
			isectPoint.add(vecC);
			
			// Compute if point is within triangle
			// Compute u
			vecA.sub(m_Vertices.get(m_Polygons.get(i).get(1)).point , isectPoint);
			vecB.sub(m_Vertices.get(m_Polygons.get(i).get(2)).point , isectPoint);
			vecC.cross(vecA, vecB);
			area2 = normal.dot(vecC);
			u = area2 / area1;
			
			// Compute V
			vecA.sub(m_Vertices.get(m_Polygons.get(i).get(2)).point , isectPoint);
			vecB.sub(m_Vertices.get(m_Polygons.get(i).get(0)).point , isectPoint);
			vecC.cross(vecA, vecB);
			area3 = normal.dot(vecC);
			v = area3 / area1;

			// Compute W
			w = 1.0f - u - v;
			
			if(u >= 0.0f && u <= 1.0f && v >= 0 && v <= 1.0f && w >= 1.0f && w <=1.0f) {						
				if(d < distance)
					distance = d;			
			}
			else {
				// Find the closest distance to vertices of the triangle
				for(int l = 0; l < 3; ++l) {
					vecC.sub(m_Vertices.get(m_Polygons.get(i).get(l)).point, point);
					d = vecC.length();
					if(d < distance)
						distance = d;	
				}
			}
			
		}
		return distance;
	}
	
	/*
	 * Sets if openGL Call Lists should be used while drawing
	 *	@param	value	true for using Call Lists else false
	 */
	public void setUseCallList(boolean value) {
		m_UseCallList = value;
	}
	
	/*
	 * Gets the number of vertices in the surface
	 *	@return		Number of vertices
	 */
	public int getVertexCount() {
		return m_Vertices.size();
	}
	
	/*
	 * Gets the name of the surface
	 *	@return		Surface name
	 */ 
	public String getName() {
		return m_Name;
	}
	
	/*
	 * Gets the number of polygons in the surface
	 *	@return		Number of polygons.
	 */
	public int getPolygonCount() {
		return m_Polygons.size();
	}
	
	/*
	 * Sets the name of the mesh
	 *	@param	name	The name of the mesh
	 */
	public void setName(String name) {
		m_Name = name;
	}
	
	/*
	 * Returns the type of the file	 
	 */
	public int getFileType() {
		return m_FileType;	
	}
	
	/*
	 * Sets the type of file, ascii or binary
	 *	@param	type	Integer indicating the type of file.
	 */
	public void setFileType(int type) {
		m_FileType = type;
	}
	
	/*
	 * Allocates a new vertices vector.
	 *	@param	num		The number of vertices the vector must hold.
	 */
	public void allocateVertices(int num) {
		m_Vertices = new Vector<Vertex>(num);		
	}
	
	/*
	 * Adds a vertex to the surface
	 *	@param	v	Vertex object
	 */
	public void addVertex(Vertex v) {
		m_Vertices.addElement(v);
	}
	
	/*
	 * Allocates a new vector to hold polygons
	 *	@param	num		The number of polygons the vector must hold.
	 */
	public void allocatePolygons(int num) {
		m_Polygons = new Vector<Vector<Integer>>(num);
	}
	
	/*
	 * Adds a polygon to the surface
	 *	@param	poly	Vector holding polygon indices
	 */
	public void addPolygon(Vector<Integer> poly) {
		m_Polygons.addElement(poly);
	}
	
	public boolean useGLList() {
		return m_UseCallList;
	}
}