package edu.jhu.ece.iacl.algorithms.graphics.intersector;

import java.util.*;

import javax.vecmath.Point2d;
import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree.BBox;
import edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree.KdTree;
import edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree.KdTriangle;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;

/**
 * A surface intersection data structure intended for detecting where line
 * segments and rays intersect a surface. Mesh triangles are stored as a KD-Tree
 * for faster lookup.
 * 
 * @author Blake Lucas
 * 
 */
public class SurfaceIntersector extends AbstractCalculation {
	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.1 $");
	}

	protected EmbeddedSurface mesh;
	protected ArrayList<KdTriangle> triangles;
	protected KdTree tree;
	protected KdTriangle lastTriangle;
	protected Point3f lastPoint;

	/**
	 * Constructor
	 * 
	 * @param parent
	 *            parent calculation
	 */
	public SurfaceIntersector(AbstractCalculation parent) {
		super(parent);
		this.mesh = null;
	}

	/**
	 * Constructor
	 */
	public SurfaceIntersector() {
		super();
		this.mesh = null;
	}

	/**
	 * Constructor.
	 * 
	 * @param parent
	 *            parent calculation
	 * @param depth
	 *            maximum KD tree depth
	 * @param mesh
	 *            surface
	 */
	public SurfaceIntersector(AbstractCalculation parent, int depth,
			EmbeddedSurface mesh) {
		super(parent);
		init(mesh, depth);
	}

	/**
	 * Constructor
	 * 
	 * @param depth
	 *            maximum KD tree depth
	 * @param mesh
	 *            surface
	 */
	public SurfaceIntersector(int depth, EmbeddedSurface mesh) {
		init(mesh, depth);
	}

	/**
	 * Initialize surface with specified KD tree depth
	 * 
	 * @param mesh
	 *            surface
	 * @param depth
	 *            maximum KD tree depth
	 */
	public void init(EmbeddedSurface mesh, int depth) {
		this.mesh = mesh;
		setLabel("Surface Intersector");
		buildTriangles();
		tree = new KdTree(this, triangles, depth);
		markCompleted();
	}

	/**
	 * Get last intersection point
	 * 
	 * @return last intersection point
	 */
	public Point3f getLastIntersectionPoint() {
		return lastPoint;
	}

	/**
	 * Get last intersection triangle
	 * 
	 * @return last intersection triangle
	 */
	public KdTriangle getLastIntersectionTriangle() {
		return lastTriangle;
	}

	/**
	 * Get minimum distance along ray that intersects a surface. Returns -1 if
	 * ray does not intersect surface.
	 * 
	 * @param p1
	 *            origin
	 * @param v
	 *            ray direction
	 * @return distance along ray
	 */
	public double intersectRayDistance(Point3f p1, Vector3f v) {
		if (mesh == null)
			throw new RuntimeException(
					"Surface Locator has not been initialized");
		LinkedList<BBox> boxes = new LinkedList<BBox>();
		boxes.add(tree.getRoot());
		BBox box = null;
		double d;
		double mind = 1E30;
		KdTriangle resultTriangle = null;
		Point3f resultIntersect = null;
		KdTriangle tri;
		int countIntersects = 0;
		while (!boxes.isEmpty()) {
			box = boxes.remove();
			if (box.intersects(p1, v)) {
				if (box instanceof KdTriangle) {
					countIntersects++;
					tri = (KdTriangle) box;
					Point3f intersect = tri.intersectionPoint(p1, v);
					if (intersect != null) {
						d = p1.distance(intersect);
						if (d < mind) {
							mind = d;
							resultTriangle = tri;
							resultIntersect = intersect;
						}
					}
				}
				boxes.addAll(box.getChildren());
			}
		}

		lastTriangle = resultTriangle;
		lastPoint = resultIntersect;
		if (resultTriangle == null) {
			return -1;
		} else {
			return mind;
		}
	}

	/**
	 * Get distance along line segment that intersects surface. Returns -1 if
	 * segment does not intersect surface.
	 * 
	 * @param p1
	 *            point 1
	 * @param p2
	 *            point 2
	 * @return distance from p1 in direction of (p2-p1)
	 */
	public double intersectSegmentDistance(Point3f p1, Point3f p2) {
		if (mesh == null)
			throw new RuntimeException(
					"Surface Locator has not been initialized");
		LinkedList<BBox> boxes = new LinkedList<BBox>();
		boxes.add(tree.getRoot());
		BBox box = null;
		double d;
		double mind = 1E30;
		KdTriangle resultTriangle = null;
		Point3f resultIntersect = null;
		KdTriangle tri;
		lastPoint = null;
		lastTriangle = null;
		int countIntersects = 0;
		while (!boxes.isEmpty()) {
			box = boxes.remove();
			if (box.intersects(p1, p2)) {
				if (box instanceof KdTriangle) {

					tri = (KdTriangle) box;
					Point3f intersect = tri.intersectionPoint(p1, p2);
					if (intersect != null) {
						d = p1.distance(intersect);
						if (d < mind) {
							mind = d;
							resultTriangle = tri;
							resultIntersect = intersect;
						}
					}
				}
				boxes.addAll(box.getChildren());
			}
		}
		lastTriangle = resultTriangle;
		lastPoint = resultIntersect;
		if (resultTriangle == null) {
			return -1;
		} else {
			return mind;
		}
	}

	/**
	 * Return signed distance indicating whether point is inside or outside a
	 * surface.
	 * 
	 * This method is not correct! We should determine if point is inside object
	 * by counting the number of intersections with a ray cast in any arbitrary
	 * direction.
	 * 
	 * @param r
	 *            reference point
	 * @return signed distance
	 */
	public double signedDistance(Point3f r) {
		double d = distance(r);
		if (d >= 0) {
			Vector3f norm = lastTriangle.getNormal();
			Point3f center = lastTriangle.getCentroid();
			Vector3f diff = new Vector3f();
			diff.sub(r, center);
			return Math.signum(diff.dot(norm)) * d;
		} else {
			return Double.NaN;
		}
	}

	/**
	 * Bounding box class used for sorting bounding boxes along specified
	 * direction.
	 * 
	 * @author Blake Lucas
	 * 
	 */
	private static class BBoxDistance implements Comparable<BBoxDistance> {
		public BBox box;
		private double dist;

		public BBoxDistance(Point3f ref, BBox tri) {
			dist = tri.distanceToBox(ref);
			this.box = tri;
		}

		public int compareTo(BBoxDistance box) {
			return (int) Math.signum(this.dist - box.dist);
		}

	}

	/**
	 * Compute distance to specified point.
	 * 
	 * @param pt
	 *            point
	 * @return distance from point.
	 */
	public double distance(Point3f pt) {
		if (mesh == null || pt == null)
			throw new RuntimeException(
					"Surface Locator has not been initialized");
		double d;
		double mind = 1E30;
		PriorityQueue<BBoxDistance> queue = new PriorityQueue<BBoxDistance>();
		queue.add(new BBoxDistance(pt, tree.getRoot()));
		lastTriangle = null;
		lastPoint = null;
		while (!queue.isEmpty()) {
			BBoxDistance boxd = queue.remove();
			if ((d = boxd.dist) < mind) {
				if (boxd.box instanceof KdTriangle) {
					d = ((KdTriangle) boxd.box).distance(pt);
					boxd.dist = d;
					if (d < mind) {
						mind = d;
						lastTriangle = (KdTriangle) boxd.box;
						lastPoint = KdTriangle.getLastIntersectionPoint();
					}
				} else {
					for (BBox child : boxd.box.getChildren()) {
						queue.add(new BBoxDistance(pt, child));
					}
				}
			}
		}
		if (lastTriangle == null) {
			return -1;
		} else {
			return mind;
		}
	}

	/**
	 * Compute minimum distance for all points on one side of plane
	 */
	public double distance(Point3f r, Vector3f v) {
		if (mesh == null || r == null)
			throw new RuntimeException(
					"Surface Locator has not been initialized");
		double d;
		double mind = 1E30;
		PriorityQueue<BBoxDistance> queue = new PriorityQueue<BBoxDistance>();
		queue.add(new BBoxDistance(r, tree.getRoot()));
		lastTriangle = null;
		lastPoint = null;
		Point3f pt;
		Vector3f testv = new Vector3f();
		while (!queue.isEmpty()) {
			BBoxDistance boxd = queue.remove();
			if ((d = boxd.dist) < mind) {
				if (boxd.box instanceof KdTriangle) {
					d = ((KdTriangle) boxd.box).distance(r);
					boxd.dist = d;
					if (d < mind) {
						pt = KdTriangle.getLastIntersectionPoint();
						testv.sub(pt, r);
						// Test if point is on one side of plane
						if (testv.dot(v) >= 0) {
							mind = d;
							lastTriangle = (KdTriangle) boxd.box;
							lastPoint = pt;
						}
					}
				} else {
					for (BBox child : boxd.box.getChildren()) {
						queue.add(new BBoxDistance(r, child));
					}
				}
			}
		}
		if (lastTriangle == null) {
			return -1;
		} else {
			return mind;
		}
	}

	/**
	 * Build collection of triangles to be used by surface
	 */
	protected void buildTriangles() {
		int indexCount = mesh.getIndexCount();
		int id1, id2, id3;
		IntersectorTriangle t;
		triangles = new ArrayList<KdTriangle>();
		for (int i = 0; i < indexCount; i += 3) {
			id1 = mesh.getCoordinateIndex(i);
			id2 = mesh.getCoordinateIndex(i + 1);
			id3 = mesh.getCoordinateIndex(i + 2);
			t = new IntersectorTriangle(id1, id2, id3, i / 3, mesh);
			triangles.add(t);
		}
	}
}
