package edu.jhu.ece.iacl.algorithms.graphics.surf;

import java.util.Arrays;
import java.util.LinkedList;

import javax.vecmath.Point3f;

import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.SelfIntersectionTriangle;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.SurfaceSelfIntersector;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;

/**
 * An extension of the progressive surface class that rejects surface collapse
 * or edge swap operations if that operation will create a self-intersection of
 * the surface.
 * 
 * @author Blake Lucas
 * 
 */
public class SelfIntersectingPreventionProgressiveSurface extends
		ProgressiveSurface {
	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.1 $");
	}
	public SurfaceSelfIntersector intersector;

	public SelfIntersectingPreventionProgressiveSurface(EmbeddedSurface surf) {
		super(surf);
		intersector = new SurfaceSelfIntersector(surf, 150);
		intersector.init();
		intersector.labelIntersections();
	}

	public SelfIntersectingPreventionProgressiveSurface(
			AbstractCalculation parent, EmbeddedSurface surf) {
		super(parent);
		init(surf, false);
		intersector = new SurfaceSelfIntersector(this.surf, 150);
		intersector.init();
		intersector.labelIntersections();
	}

	public SelfIntersectingPreventionProgressiveSurface(
			AbstractCalculation parent) {
		super(parent);
	}

	public SelfIntersectingPreventionProgressiveSurface() {
		super();
	}

	/**
	 * Initialize surface
	 * 
	 * @param surf
	 *            surface
	 */
	public void init(EmbeddedSurface surf) {
		init(surf, false);
		intersector = new SurfaceSelfIntersector(this.surf, 150);
		intersector.init();
	}

	/**
	 * Create edge swap operation
	 */
	public EdgeSwap createEdgeSwap() {
		return (intersector == null) ? new EdgeSwap()
				: new SelfIntersectingEdgeSwap();
	}

	/**
	 * Create ege collapse operation
	 */
	public EdgeCollapse createEdgeCollapse() {
		return (intersector == null) ? new EdgeCollapse()
				: new SelfIntersectingEdgeCollapse();
	}

	/**
	 * Self-intersection edge swap that rejects edge swaps that cause
	 * self-intersections
	 * 
	 * @author Blake Lucas
	 * 
	 */
	public class SelfIntersectingEdgeSwap extends EdgeSwap {

		public boolean apply(int v3, int v4) {

			int nbrIndex1 = find(v3, v4);
			int[] nbrs1 = neighborVertexVertexTable[v3];
			this.v1 = nbrs1[(nbrIndex1 + 1) % nbrs1.length];
			this.v2 = nbrs1[(nbrIndex1 + nbrs1.length - 1) % nbrs1.length];
			if (find(v1, v2) != -1)
				return false;
			if (surf.getVertex(v1).distance(surf.getVertex(v2)) >= surf
					.getVertex(v3).distance(surf.getVertex(v4))) {
				return false;
			}
			if (!intersector.swapEdge(v1, v2, v3, v4)) {
				// System.out.println("Could not swap edge because of self-intersection");
				return false;
			}

			connectAfter(v1, v3, v2);
			connectAfter(v2, v4, v1);
			disconnect(v3, v4);
			disconnect(v4, v3);
			return true;
		}

	}

	protected boolean validate = true;

	public void setValidate(boolean val) {
		this.validate = val;
	}

	/**
	 * Edge collapse operation that rejects edge collapses that would cause
	 * self-intersections.
	 * 
	 * @author Blake
	 * 
	 */
	public class SelfIntersectingEdgeCollapse extends EdgeCollapse {

		public int isCollapsable(int v2) {
			int[] nbrs = getSortedNeighbors(v2);
			nbrs = getSortedNeighbors(v2);
			int v1;
			Point3f v2old = new Point3f(surf.getVertex(v2));
			for (int i = 0; i < nbrs.length; i++) {
				// If we cannot remove the shortest edge because of topology,
				// then try other edges in order of length until an edge is
				// removed
				v1 = nbrs[i];
				if (!topologyCheck(v1, v2)) {
					continue;
				}
				Point3f p = new Point3f(surf.getVertex(v1));
				if (!intersector.updateVertex(v2, p)) {
					continue;
				}
				if (!intersector.updateVertex(v2, v2old)) {
					surf.setVertex(v2, p);
				}
				return v1;
			}
			return -1;
		}

		public boolean apply(int vert1, int vert2) {
			this.v1 = vert1;
			this.v2 = vert2;
			if (!topologyCheck(v1, v2))
				return false;
			Point3f p = new Point3f(surf.getVertex(v1));
			if (!intersector.updateVertex(v2, p)) {
				// System.out.println("Could not collapse edge because of self-intersection");

				return false;
			}
			int nbrIndex1 = find(v1, v2);
			int nbrIndex2 = find(v2, v1);
			int[] nbrs1 = neighborVertexVertexTable[v1];
			int[] nbrs2 = neighborVertexVertexTable[v2];
			int[] newNbrs1 = new int[nbrs1.length + nbrs2.length - 4];
			int v3, e;
			nbr1 = nbrs1[(nbrIndex1 + 1) % nbrs1.length];
			nbr2 = nbrs1[(nbrIndex1 + nbrs1.length - 1) % nbrs1.length];
			// Update indexes for triangles to replace v2 with v1

			for (int i = 2; i < nbrs2.length - 1; i++) {
				// vertex to add to ring around v1
				v3 = nbrs2[(i + nbrIndex2) % nbrs2.length];
				// add v2 neighbor to v1 neighbors
				newNbrs1[i - 2] = v3;
				// replace connection from v3 to v2 with v3 to v1
				e = find(v3, v2);
				neighborVertexVertexTable[v3][e] = v1;
			}
			// Disconnect first and last point from ring around v2

			// Add old vertexes to ring around v1
			int j = nbrIndex1 + 1;
			for (int i = nbrs2.length - 3; i < newNbrs1.length; i++) {
				newNbrs1[i] = nbrs1[j % nbrs1.length];
				j++;
			}
			neighborVertexVertexTable[v2] = new int[0];
			neighborVertexVertexTable[v1] = newNbrs1;

			disconnect(v3 = nbrs2[(nbrIndex2 + 1) % nbrs2.length], v2);
			disconnect(
					v3 = nbrs2[(nbrIndex2 + nbrs2.length - 1) % nbrs2.length],
					v2);
			intersector.removeFace(intersector.findFace(v2, v1, nbr1));
			intersector.removeFace(intersector.findFace(v1, v2, nbr2));
			LinkedList<SelfIntersectionTriangle> faceList = intersector
					.findFaces(v2);
			for (SelfIntersectionTriangle t : faceList) {
				if (t.ids[0] == v2) {
					t.ids[0] = v1;
				} else if (t.ids[1] == v2) {
					t.ids[1] = v1;
				} else if (t.ids[2] == v2) {
					t.ids[2] = v1;
				}
			}
			// surf.setVertex(v1, p);
			return true;

		}

		@Override
		public int[] getChangedVerts() {
			int[] newList = Arrays.copyOf(neighborVertexVertexTable[v1],
					neighborVertexVertexTable[v1].length + 1);
			newList[newList.length - 1] = v1;
			return newList;
		}
	}
}
