package edu.jhu.ece.iacl.algorithms.graphics.topo;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Stack;

import javax.vecmath.Point3f;

import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface.Direction;

/**
 * Compute connected surface components for a surface and return a list of
 * surface components sorted by vertex count.
 * 
 * @author Blake Lucas
 * 
 */
public class ConnectedSurfaceComponents {
	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.1 $");
	}

	private int[][] neighborVertexVertexTable;
	private EmbeddedSurface surf;

	public ConnectedSurfaceComponents(EmbeddedSurface surf) {
		neighborVertexVertexTable = EmbeddedSurface
				.buildNeighborVertexVertexTable(surf, Direction.CLOCKWISE);
		this.surf = surf;
	}
	/**
	 * Vertex label
	 * @author Blake Lucas
	 *
	 */
	protected class VertexLabel {
		public int vid;
		public int label;

		public VertexLabel(int vid, int label) {
			this.vid = vid;
			this.label = label;
		}
	}	
	/**
	 * Get connected components for surface
	 * @return connected components
	 */
	public ArrayList<EmbeddedSurface> solve() {
		int vertCount = surf.getVertexCount();
		Stack<VertexLabel> stack = new Stack<VertexLabel>();
		VertexLabel[] labels = new VertexLabel[vertCount];
		for (int id = 0; id < vertCount; id++) {
			labels[id] = new VertexLabel(id, -1);
		}
		int labelCount = 0;
		int label;
		ArrayList<EmbeddedSurface> surfgroup = new ArrayList<EmbeddedSurface>();
		while (true) {
			VertexLabel first = null;
			for (int id = 0; id < vertCount; id++) {
				if (labels[id].label == -1) {
					first = labels[id];
					break;
				}
			}
			if (first == null)
				break;
			first.label = label = labelCount++;
			stack.push(first);
			while (!stack.isEmpty()) {
				VertexLabel top = stack.pop();
				int[] nbrs = neighborVertexVertexTable[top.vid];
				for (int nbr : nbrs) {
					if (labels[nbr].label == -1) {
						labels[nbr].label = label;
						stack.push(labels[nbr]);
					}
				}
			}
			surfgroup.add(createSurface(labels, label));
		}
		Collections.sort(surfgroup);
		Collections.reverse(surfgroup);
		return surfgroup;
	}
	/**
	 * Create surface for particular label
	 * @param labels vertex labels
	 * @param label label
	 * @return sub-component surface
	 */
	protected EmbeddedSurface createSurface(VertexLabel[] labels, int label) {
		int vertCount = surf.getVertexCount();
		int vertIDs[] = new int[vertCount];
		int currentId = 0, n1, n2;

		for (int i = 0; i < vertCount; i++) {
			int len = neighborVertexVertexTable[i].length;
			if (len == 0 || labels[i].label != label) {
				vertIDs[i] = -1;
			} else {
				vertIDs[i] = currentId++;
			}
		}
		int ptsCount = currentId;
		ArrayList<Integer> indexes = new ArrayList<Integer>();
		for (int i = 0; i < vertCount; i++) {
			int[] nbrs = neighborVertexVertexTable[i];
			currentId = vertIDs[i];
			for (int j = 0; j < nbrs.length; j++) {

				n1 = vertIDs[nbrs[j]];
				n2 = vertIDs[nbrs[(j + 1) % nbrs.length]];
				// Only create new triangles if they include vertex not seen
				// before
				if (n1 > currentId && n2 > currentId) {
					// Add new triangle
					indexes.add(currentId);
					indexes.add(n1);
					indexes.add(n2);
				}
			}
		}
		int[] indexArray = new int[indexes.size()];
		Point3f[] pts = new Point3f[ptsCount];
		// Copy indexes to array
		for (int i = 0; i < indexArray.length; i++) {
			indexArray[i] = indexes.get(i);
		}
		// Copy subset of points to array
		for (int i = 0; i < vertCount; i++) {
			if (vertIDs[i] != -1) {
				pts[vertIDs[i]] = surf.getVertex(i);
			}
		}
		EmbeddedSurface newSurf = new EmbeddedSurface(pts, indexArray);
		newSurf.setName(surf.getName() + "_comp" + label);
		return newSurf;
	}
}
