package edu.jhu.ece.iacl.algorithms.graphics.topo;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Stack;

import javax.vecmath.Point3f;

import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface.Direction;

/**
 * Compute connected components for labeled regions for labels stored as embedded
 * vertex data.
 * 
 * @author Blake Lucas
 * 
 */
public class ConnectedSurfaceLabels {
	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.1 $");
	}

	private int[][] neighborVertexVertexTable;
	private EmbeddedSurface surf;
	private int offset;
	protected VertexLabel[] labels;

	public ConnectedSurfaceLabels(EmbeddedSurface surf, int offfset) {
		neighborVertexVertexTable = EmbeddedSurface
				.buildNeighborVertexVertexTable(surf, Direction.CLOCKWISE);
		this.surf = surf;
		this.offset = offset;
	}
	/**
	 * Connected label component
	 * @author Blake Lucas
	 *
	 */
	protected class ConnectedComponent extends ArrayList<VertexLabel> implements
			Comparable<ConnectedComponent> {
		public int vlabel;

		public ConnectedComponent(int label) {
			this.vlabel = label;
		}

		// Sort by label and then by size
		public int compareTo(ConnectedComponent c) {
			if (vlabel == c.vlabel) {
				return (int) Math.signum(c.size() - this.size());
			} else {
				return (int) Math.signum(vlabel - c.vlabel);
			}
		};
	}
	/**
	 * Vertex label
	 * @author Blake Lucas
	 *
	 */
	public class VertexLabel {
		public int vid;
		public int clabel;
		public int vlabel;

		public VertexLabel(int vid) {
			this.vid = vid;
			this.vlabel = (int) surf.getVertexDataAtOffset(vid, offset);
			this.clabel = -1;
		}
	}
	/**
	 * Label surface using largest connected components
	 * @return labeled surface
	 */
	public EmbeddedSurface labelSurface() {
		ArrayList<ConnectedComponent> comps = getLabeledComponents();
		Collections.sort(comps);
		int lastLabel = -1;
		for (ConnectedComponent c : comps) {
			if (c.vlabel != lastLabel) {
				lastLabel = c.vlabel;
			} else {
				if (c.size() < 200) {
					// mark as unlabeled
					for (VertexLabel l : c) {
						l.vlabel = -1;
					}
				}
			}
		}
		for (int i = 0; i < labels.length; i++) {
			surf.setVertexData(i, offset, labels[i].vlabel);
		}
		return surf;
	}
	/**
	 * Get labeled components
	 * @return labeled connected components
	 */
	public ArrayList<ConnectedComponent> getLabeledComponents() {
		int vertCount = surf.getVertexCount();
		Stack<VertexLabel> stack = new Stack<VertexLabel>();
		labels = new VertexLabel[vertCount];
		for (int id = 0; id < vertCount; id++) {
			labels[id] = new VertexLabel(id);
		}
		int labelCount = 0;
		int clabel;
		int id = 0;
		ArrayList<ConnectedComponent> comps = new ArrayList<ConnectedComponent>();
		while (true) {
			VertexLabel first = null;
			while (id < vertCount) {
				if (labels[id].clabel == -1) {
					first = labels[id];
					break;
				}
				id++;
			}
			if (first == null)
				break;
			ConnectedComponent comp = new ConnectedComponent(first.vlabel);
			comps.add(comp);
			comp.add(first);
			first.clabel = clabel = labelCount++;
			stack.push(first);
			while (!stack.isEmpty()) {
				VertexLabel top = stack.pop();
				int[] nbrs = neighborVertexVertexTable[top.vid];
				for (int nbr : nbrs) {
					if (labels[nbr].clabel == -1
							&& labels[nbr].vlabel == first.vlabel) {
						labels[nbr].clabel = clabel;
						comp.add(labels[nbr]);
						stack.push(labels[nbr]);
					}
				}
			}
		}
		return comps;
	}
}
