package edu.jhu.ece.iacl.algorithms.graphics.topo;

import javax.vecmath.Matrix3d;
import javax.vecmath.Point2d;
import javax.vecmath.Point3d;
import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.algorithms.graphics.surf.ProgressiveSurface;
import edu.jhu.ece.iacl.algorithms.graphics.surf.SphericalArc;
import edu.jhu.ece.iacl.algorithms.graphics.surf.ProgressiveSurface.SphericalEdgeCollapse;

/**
 * Compute 1-ring unfolding around a vertex on a sphere. This was intended to be
 * used for local simplex optimization, but that line of work was abandoned.
 * 
 * @author Blake Lucas
 * 
 */
public class OneRingUnfolding {
	protected ProgressiveSurface ps;

	public OneRingUnfolding(ProgressiveSurface ps) {
		this.ps = ps;
	}

	protected Point2d[] ptsCart2D;
	protected Point2d[] ptsPolar2D;
	protected Vector3f[] pts3D;
	protected static final Point2d origin2D = new Point2d(0, 0);
	protected Vector3f pivot;
	protected int vid;

	public void unfold(int id) {
		int[] nbrs = ps.neighborVertexVertexTable[id];
		ptsCart2D = new Point2d[nbrs.length];
		ptsPolar2D = new Point2d[nbrs.length];
		pts3D = new Vector3f[nbrs.length];
		double totalAngle = 0;
		double l;
		vid = id;
		pivot = new Vector3f(ps.surf.getVertex(id));
		Vector3f plast = new Vector3f(ps.surf.getVertex(nbrs[0]));
		l = GeometricUtilities.angle(pivot, plast);
		ptsCart2D[0] = new Point2d();
		ptsPolar2D[0] = new Point2d();
		ptsCart2D[0].x = l;
		ptsCart2D[0].y = 0;
		ptsPolar2D[0].x = l;
		ptsPolar2D[0].y = 0;// (r,theta)
		pts3D[0] = plast;
		double area = 0;
		for (int i = 1; i < nbrs.length; i++) {
			Vector3f pcurr = new Vector3f(ps.surf.getVertex(nbrs[i]));
			pts3D[i] = pcurr;
			totalAngle += GeometricUtilities
					.sphericalAngle(pivot, plast, pcurr);
			l = GeometricUtilities.angle(pivot, pcurr);
			Point2d ptC = new Point2d();
			ptC.x = l * Math.cos(totalAngle);
			ptC.y = l * Math.sin(totalAngle);
			ptsCart2D[i] = ptC;
			area += GeometricUtilities.triangleArea(origin2D, ptsCart2D[i - 1],
					ptsCart2D[i]);
			Point2d ptP = new Point2d();
			ptP.x = l;
			ptP.y = totalAngle;
			ptsPolar2D[i] = ptP;
			plast = pcurr;
		}
		double scaleFactor = 1;
		if (area > 0) {
			// Scale ring so it has area of a unit circle
			scaleFactor = Math.sqrt(Math.PI / area);
		}
		for (int i = 0; i < nbrs.length; i++) {
			Point2d pt = ptsCart2D[i];
			pt.x *= scaleFactor;
			pt.y *= scaleFactor;
			ptsPolar2D[i].x *= scaleFactor;

		}
	}

	public Vector3f mapToSphere(Point2d pt) {
		double l = GeometricUtilities.length(pt);
		if (l == 0) {
			return pivot;
		}
		double theta = Math.atan2(pt.y / l, pt.x / l);
		int lastIndex = 0;
		int nextIndex = 0;
		for (nextIndex = 1; nextIndex < ptsPolar2D.length; nextIndex++) {
			if (theta >= ptsPolar2D[lastIndex].y
					&& theta < ptsPolar2D[nextIndex].y) {
				Vector3f interp = GeometricUtilities.interpolateVectorFromBary(
						GeometricUtilities.getBaryCoords(pt, origin2D,
								ptsCart2D[lastIndex], ptsCart2D[nextIndex]),
						pivot, pts3D[lastIndex], pts3D[nextIndex]);
				GeometricUtilities.normalize(interp);
				return interp;
			}
			lastIndex = nextIndex;
		}
		lastIndex = ptsPolar2D.length - 1;
		nextIndex = 0;
		Vector3f interp = GeometricUtilities.interpolateVectorFromBary(
				GeometricUtilities.getBaryCoords(pt, origin2D,
						ptsCart2D[lastIndex], ptsCart2D[nextIndex]), pivot,
				pts3D[lastIndex], pts3D[nextIndex]);
		GeometricUtilities.normalize(interp);
		return interp;
	}

	public double[] interpolateVectorOnSphere(Point2d pt) {
		double l = GeometricUtilities.length(pt);
		if (l == 0) {
			return ps.surf.getVertexData(vid);
		}
		double theta = Math.atan2(pt.y / l, pt.x / l);
		int lastIndex = 0;
		int[] nbrs = ps.neighborVertexVertexTable[vid];
		for (int i = 0; i < ptsPolar2D.length; i++) {
			lastIndex = (i + ptsPolar2D.length - 1) % ptsPolar2D.length;
			if (theta >= ptsPolar2D[lastIndex].y && theta < ptsPolar2D[i].y) {
				return GeometricUtilities.interpolateVectorFromBary(
						GeometricUtilities.getBaryCoords(pt, origin2D,
								ptsCart2D[lastIndex], ptsCart2D[i]), ps.surf
								.getVertexData(vid), ps.surf
								.getVertexData(nbrs[lastIndex]), ps.surf
								.getVertexData(nbrs[i]));
			}
		}
		return ps.surf.getVertexData(vid);
	}

	public double interpolateQuantityOnSphere(Point2d pt, int index) {
		double l = GeometricUtilities.length(pt);
		if (l == 0) {
			return ps.surf.getVertexDataAtOffset(vid, index);
		}
		double theta = Math.atan2(pt.y / l, pt.x / l);
		int lastIndex = 0;
		int[] nbrs = ps.neighborVertexVertexTable[vid];
		for (int i = 0; i < ptsPolar2D.length; i++) {
			lastIndex = (i + ptsPolar2D.length - 1) % ptsPolar2D.length;
			if (theta >= ptsPolar2D[lastIndex].y && theta < ptsPolar2D[i].y) {
				return GeometricUtilities.interpolateQuantityFromBary(
						GeometricUtilities.getBaryCoords(pt, origin2D,
								ptsCart2D[lastIndex], ptsCart2D[i]), ps.surf
								.getVertexDataAtOffset(vid, index), ps.surf
								.getVertexDataAtOffset(nbrs[lastIndex], index),
						ps.surf.getVertexDataAtOffset(nbrs[i], index));
			}
		}
		return ps.surf.getVertexDataAtOffset(vid, index);
	}

}
