package edu.jhu.ece.iacl.algorithms.graphics.utilities;

import java.util.List;

import Jama.*;

import javax.vecmath.Matrix3d;
import javax.vecmath.Matrix3f;
import javax.vecmath.Point3d;
import javax.vecmath.Point3d;
import javax.vecmath.Point3f;
import javax.vecmath.Vector3d;

import edu.jhu.ece.iacl.algorithms.VersionUtil;

/**
 * PCA analysis on a set of 3D points. This method computes the 3D
 * transformation matrix needed to align the minimum Eigen vector with the Z
 * axis. All computations are performed in double floating point.
 * 
 * @author Blake Lucas
 * 
 */
public class PrincipalComponentAnalysisDouble {

	protected Matrix convariance;
	protected Matrix3d correctRot;
	protected Matrix3d returnRot;
	protected Point3d centroid;
	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.1 $");
	}
	public PrincipalComponentAnalysisDouble(Point3d[] points) {
		correctRot = computeRotation(convariance = computeCovariance(points));
		returnRot = new Matrix3d();
		returnRot.invert(correctRot);
	}

	public Matrix3d getForwardRotation() {
		return correctRot;
	}

	public Matrix3d getBackgroundRotation() {
		return returnRot;
	}

	public Matrix getTransformationMatrix() {
		Matrix Tmat = new Matrix(4, 4);
		EigenvalueDecomposition ed = new EigenvalueDecomposition(convariance);
		Matrix R = ed.getV();
		Matrix D = ed.getD();
		for (int m = 0; m < 3; m++) {
			for (int n = 0; n < 3; n++) {
				Tmat.set(m, n, Math.signum(D.get(n, n)) * R.get(m, n));
			}
		}
		Tmat.set(0, 3, centroid.x);
		Tmat.set(1, 3, centroid.y);
		Tmat.set(2, 3, centroid.z);
		Tmat.set(3, 3, 1);
		Tmat.set(3, 0, 0);
		Tmat.set(3, 1, 0);
		Tmat.set(3, 2, 0);
		return Tmat;
	}

	public Point3d getCentroid() {
		return centroid;
	}

	public PrincipalComponentAnalysisDouble(List points) {
		convariance = computeCovariance(points);
	}

	public void translateAndRotatePoint(Point3d p) {
		p.sub(centroid);
		correctRot.transform(p);
	}

	public void translateAndRotatePoints(Point3d[] points) {
		for (Point3d p : points) {
			p.sub(centroid);
			correctRot.transform(p);
		}
	}

	public void translateAndRotateBackPoint(Point3d p) {
		returnRot.transform(p);
		p.add(centroid);
	}

	public void translateAndRotateBackPoints(Point3d[] points) {
		for (Point3d p : points) {
			returnRot.transform(p);
			p.add(centroid);
		}
	}

	public void rotateVector(Vector3d p) {
		correctRot.transform(p);
	}

	public void rotateVectors(Vector3d[] points) {
		for (Vector3d p : points) {

			correctRot.transform(p);
		}
	}

	public void rotateBackVector(Vector3d p) {
		returnRot.transform(p);
	}

	public void rotateBackVectors(Vector3d[] points) {
		for (Vector3d p : points) {
			returnRot.transform(p);
		}
	}

	protected Matrix computeCovariance(Point3d[] points) {
		centroid = new Point3d();
		Point3d p;
		Matrix b = new Matrix(points.length, 3);
		for (int i = 0; i < points.length; i++) {
			p = points[i];
			centroid.x += p.x;
			centroid.y += p.y;
			centroid.z += p.z;
		}
		centroid.x /= points.length;
		centroid.y /= points.length;
		centroid.z /= points.length;
		for (int i = 0; i < points.length; i++) {
			p = points[i];
			b.set(i, 0, p.x - centroid.x);
			b.set(i, 1, p.y - centroid.y);
			b.set(i, 2, p.z - centroid.z);
		}
		Matrix A = b.transpose().times(b);
		return A;
	}

	protected Matrix computeCovariance(List points) {
		centroid = new Point3d();
		int sz = points.size();
		Matrix b = new Matrix(sz, 3);
		for (Object obj : points) {
			Point3d p = (Point3d) obj;
			centroid.x += p.x;
			centroid.y += p.y;
			centroid.z += p.z;
		}
		centroid.x /= sz;
		centroid.y /= sz;
		centroid.z /= sz;
		int i = 0;
		for (Object obj : points) {
			Point3d p = (Point3d) obj;
			b.set(i, 0, p.x - centroid.x);
			b.set(i, 1, p.y - centroid.y);
			b.set(i, 2, p.z - centroid.z);
			i++;
		}
		Matrix A = b.transpose().times(b);
		return A;
	}

	public Vector3d getPrinicpalEigenVector() {
		EigenvalueDecomposition ed = new EigenvalueDecomposition(convariance);
		Matrix rot = ed.getV();
		Matrix cov = ed.getD();
		Vector3d v = new Vector3d();
		Matrix3d correctRot = new Matrix3d();
		// Align max principal axis with the z axis
		if (cov.get(0, 0) > cov.get(1, 1)) {
			if (cov.get(0, 0) > cov.get(2, 2)) {
				// X is max principal axis
				v.x = rot.get(0, 0);
				v.y = rot.get(1, 0);
				v.z = rot.get(2, 0);
			} else {
				// Z is max principal axis
				v.x = rot.get(0, 2);
				v.y = rot.get(1, 2);
				v.z = rot.get(2, 2);
			}
		} else {
			if (cov.get(1, 1) > cov.get(2, 2)) {
				// Y is max principal axis
				v.x = rot.get(0, 1);
				v.y = rot.get(1, 1);
				v.z = rot.get(2, 1);
			} else {
				// Z is max principal axis
				v.x = rot.get(0, 2);
				v.y = rot.get(1, 2);
				v.z = rot.get(2, 2);
			}
		}
		return v;
	}

	protected Matrix3d computeRotation(Matrix A) {
		EigenvalueDecomposition ed = new EigenvalueDecomposition(A);
		Matrix rot = ed.getV();
		Matrix cov = ed.getD();
		Vector3d v = new Vector3d();
		Matrix3d correctRot = new Matrix3d();
		// Align min principal axis with the z axis
		if (cov.get(0, 0) < cov.get(1, 1)) {
			if (cov.get(0, 0) < cov.get(2, 2)) {
				// X is min principal axis
				correctRot.rotY((float) Math.PI * 0.5f);
			} else {
				// Z is min principal axis
				correctRot.rotX(0);
			}
		} else {
			if (cov.get(1, 1) < cov.get(2, 2)) {
				// Y is min principal axis
				correctRot.rotX((float) Math.PI * 0.5f);
			} else {
				// Z is min principal axis
				correctRot.rotX(0);
			}
		}

		Matrix Ainv = rot.inverse();
		Matrix3d B = new Matrix3d();
		// System.out.println("Covariance "+" "+A.getRowDimension()+" "+A.getColumnDimension());
		for (int i = 0; i < 3; i++) {
			for (int j = 0; j < 3; j++) {
				B.setElement(i, j, (float) Ainv.get(i, j));
				// System.out.print(B.getElement(i, j)+" ");
			}
			// System.out.println();
		}
		correctRot.mul(B);
		return correctRot;
	}
}
