package edu.jhu.ece.iacl.algorithms.graphics.utilities;

import javax.vecmath.Point3f;

import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.SurfaceIntersector;
import edu.jhu.ece.iacl.algorithms.volume.DistanceField;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;

/**
 * Convert a surface to mask by using a scan line rasterization method. There
 * may still be problems with how the line/surface intersector handles cracks in
 * the surface.
 * 
 * @author Blake Lucas
 * 
 */
public class SurfaceToMask extends AbstractCalculation {

	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.1 $");
	}

	public SurfaceToMask() {
		super();
		setLabel("Surface to Level Set");
	}

	public SurfaceToMask(AbstractCalculation parent) {
		super(parent);
		setLabel("Surface to Level Set");
	}

	public ImageDataFloat solve(ImageData orVol, EmbeddedSurface surf) {
		ImageDataFloat vol = new ImageDataFloat(orVol);
		int rows = vol.getRows();
		int cols = vol.getCols();
		int slices = vol.getSlices();
		float[][][] resultMat = new float[rows][cols][slices];
		boolean[][][] maskMat = new boolean[rows][cols][slices];
		ImageDataFloat resultVol = new ImageDataFloat(resultMat);
		SurfaceIntersector locator = new SurfaceIntersector(this, 5, surf);
		setTotalUnits(rows);
		Point3f st = new Point3f(), end = new Point3f();
		double dist;
		boolean inside;
		for (int i = 0; i < rows; i++) {
			for (int j = 0; j < cols; j++) {
				st.x = i;
				st.y = j;
				st.z = 0;
				end.x = i;
				end.y = j;
				end.z = slices;
				inside = false;
				while ((dist = locator.intersectSegmentDistance(st, end)) != -1) {

					if (inside) {
						for (int k = (int) st.z; k < Math.min(slices,
								(int) end.z); k++) {
							maskMat[i][j][k] = true;
						}
					}
					st.z = st.z + (float) Math.ceil(dist + 1E-6f);
					inside = !inside;
				}
				for (int k = (int) st.z; k < Math.min(slices, (int) end.z); k++) {
					maskMat[i][j][k] = false;
				}
			}
			incrementCompletedUnits();
		}
		DistanceField df = new DistanceField();
		for (int i = 0; i < rows; i++) {
			for (int j = 0; j < cols; j++) {
				for (int k = 0; k < slices; k++) {
					resultMat[i][j][k] = (maskMat[i][j][k]) ? -1 : 1;
				}
			}
		}
		resultVol = df.solve(resultVol, 5);
		resultMat = resultVol.toArray3d();
		resultVol.setName(orVol.getName() + "_mask");
		return resultVol;
	}

	public ImageDataFloat solve(ImageData orVol, EmbeddedSurface surf,
			double fuzz, double damt) {
		ImageDataFloat vol = (new ImageDataFloat(orVol));
		int rows = vol.getRows();
		int cols = vol.getCols();
		int slices = vol.getSlices();
		float[][][] resultMat = new float[rows][cols][slices];
		boolean[][][] maskMat = new boolean[rows][cols][slices];
		ImageDataFloat resultVol = new ImageDataFloat(resultMat);
		SurfaceIntersector locator = new SurfaceIntersector(this, 16, surf);
		setTotalUnits(rows);
		Point3f st = new Point3f(), end = new Point3f();
		double dist;
		boolean inside;
		for (int i = 0; i < rows; i++) {
			for (int j = 0; j < cols; j++) {
				st.x = i;
				st.y = j;
				st.z = 0;
				end.x = i;
				end.y = j;
				end.z = slices;
				inside = false;
				while ((dist = locator.intersectSegmentDistance(st, end)) != -1) {
					if (inside) {
						for (int k = (int) st.z; k < Math.min(slices,
								(int) end.z); k++) {
							maskMat[i][j][k] = true;
						}
					}
					st.z = st.z + (float) Math.ceil(dist + 1E-6f);
					inside = !inside;
				}
				for (int k = (int) st.z; k < Math.min(slices, (int) end.z); k++) {
					maskMat[i][j][k] = false;
				}
			}
			incrementCompletedUnits();
		}
		DistanceField df = new DistanceField();
		for (int i = 0; i < rows; i++) {
			for (int j = 0; j < cols; j++) {
				for (int k = 0; k < slices; k++) {
					resultMat[i][j][k] = (maskMat[i][j][k]) ? -1 : 1;
				}
			}
		}
		resultVol = df.solve(resultVol, 5);
		resultMat = resultVol.toArray3d();
		for (int i = 0; i < rows; i++) {
			for (int j = 0; j < cols; j++) {
				for (int k = 0; k < slices; k++) {
					resultMat[i][j][k] = (float) (1.0 / (1 + Math.exp(fuzz
							* (resultMat[i][j][k] - damt))));
				}
			}
		}
		resultVol.setName(orVol.getName() + "_mask");
		return resultVol;
	}
}
