package edu.jhu.ece.iacl.algorithms.thickness;

import javax.vecmath.Point3f;

import edu.jhu.ece.iacl.algorithms.graphics.intersector.SurfaceIntersector;
import edu.jhu.ece.iacl.algorithms.graphics.isosurf.IsoSurfaceOnGrid;
import edu.jhu.ece.iacl.algorithms.thickness.grid.SimpleThicknessNode;
import edu.jhu.ece.iacl.algorithms.topology.ConnectivityRule;
import edu.jhu.ece.iacl.jist.structures.data.BinaryMinHeap;
/**
 * Compute thickness using Rocha, Yezzi, Prince method.
 * @author Blake Lucas (bclucas@jhu.edu)
 *
 */
public class EulerLagrangeThickness extends ThicknessSolver{
	SurfaceIntersector intersector;
	public EulerLagrangeThickness(double lambda,double isoVal,double lagrangeDeltaT){
		super(lambda,isoVal,lagrangeDeltaT);
	}
	/**
	 * Determine which octant a point lies inside
	 * @param p point
	 * @param xmin x min
	 * @param xmax x max
	 * @param ymin y min
	 * @param ymax y max
	 * @param zmin z min
	 * @param zmax z max
	 * @return bit code representing the point's octant
	 */
	protected int octant(Point3f p,double xmin,double xmax,double ymin,double ymax,double zmin,double zmax){
		int oct=0;
		if(p.x<xmin){
			oct|=ilow;
		} else if(p.x>xmax){
			oct|=ihigh;
		}
		if(p.y<ymin){
			oct|=jlow;
		} else if(p.y>ymax){
			oct|=jhigh;
		}
		if(p.z<zmin){
			oct|=klow;
		} else if(p.z>zmax){
			oct|=khigh;
		}
		return oct;
	}
	/**
	 * Approximate distance to boundary from neighbors
	 * @param dat current node to compute distance
	 * @param ref last point on streamline
	 * @param p current point to find distance
	 * @param len cumulative length up to last point
	 * @return length could be successfully approximated
	 */
	protected boolean approximateDistance(SimpleThicknessNode dat,Point3f ref,Point3f p,double len){
		Point3f pint;	
		Point3f result=null;
		double dist=1E10;
		double tmp=0;
		
		double xmin=Math.floor(ref.x);
		double ymin=Math.floor(ref.y);
		double zmin=Math.floor(ref.z);
		
		double xmax=xmin+1;
		double ymax=ymin+1;
		double zmax=zmin+1;
		double avgLength=0;
		if(xmin==ref.x&&ymin==ref.y&&zmin==ref.z){
			//double actual=trace(ref, false);
			//System.out.println("BROKE "+ref+","+dat.label);
			//dat.setCorrespodence(interpolateCoorespondence(ref));
			return false;
		}
		int plane=0;
		if(p.x<xmin){
			result=new Point3f(
					(float)xmin,
					(float)(ref.y+(p.y-ref.y)*(xmin-ref.x)/(p.x-ref.x)),
					(float)(ref.z+(p.z-ref.z)*(xmin-ref.x)/(p.x-ref.x)));
			dist=p.distance(result);
			plane=0;
		} else if(p.x>xmax){
			result=new Point3f(
					(float)xmax,
					(float)(ref.y+(p.y-ref.y)*(xmax-ref.x)/(p.x-ref.x)),
					(float)(ref.z+(p.z-ref.z)*(xmax-ref.x)/(p.x-ref.x)));
			dist=p.distance(result);
			plane=1;
		}
		if(p.y<ymin){
			pint=new Point3f(
					(float)(ref.x+(p.x-ref.x)*(ymin-ref.y)/(p.y-ref.y)),
					(float)ymin,
					(float)(ref.z+(p.z-ref.z)*(ymin-ref.y)/(p.y-ref.y)));
			tmp=p.distance(pint);
			if(tmp<dist){
				result=pint;
				dist=tmp;
				plane=2;
			}
		} else if(p.y>ymax){
			pint=new Point3f(
					(float)(ref.x+(p.x-ref.x)*(ymax-ref.y)/(p.y-ref.y)),
					(float)ymax,
					(float)(ref.z+(p.z-ref.z)*(ymax-ref.y)/(p.y-ref.y)));
			tmp=p.distance(pint);
			if(tmp<dist){
				result=pint;
				dist=tmp;
				plane=3;
			}
		}
		if(p.z<zmin){
			pint=new Point3f(
					(float)(ref.x+(p.x-ref.x)*(zmin-ref.z)/(p.z-ref.z)),
					(float)(ref.y+(p.y-ref.y)*(zmin-ref.z)/(p.z-ref.z)),
				(float)zmin);
			tmp=p.distance(pint);
			if(tmp<dist){
				result=pint;
				dist=tmp;
				plane=4;
			}
		} else if(p.z>zmax){
			pint=new Point3f(
					(float)(ref.x+(p.x-ref.x)*(zmax-ref.z)/(p.z-ref.z)),
					(float)(ref.y+(p.y-ref.y)*(zmax-ref.z)/(p.z-ref.z)),
				(float)zmax);
			tmp=p.distance(pint);
			if(tmp<dist){
				result=pint;
				dist=tmp;
				plane=5;
			}
		}
		if(result==null)return false;
		SimpleThicknessNode[] cd=new SimpleThicknessNode[4];
		cd[0]=getData((int)xmin+planePoint1[plane].x,(int)ymin+planePoint1[plane].y,(int)zmin+planePoint1[plane].z);
		cd[1]=getData((int)xmin+planePoint2[plane].x,(int)ymin+planePoint2[plane].y,(int)zmin+planePoint2[plane].z);
		cd[2]=getData((int)xmin+planePoint3[plane].x,(int)ymin+planePoint3[plane].y,(int)zmin+planePoint3[plane].z);
		cd[3]=getData((int)xmin+planePoint4[plane].x,(int)ymin+planePoint4[plane].y,(int)zmin+planePoint4[plane].z);
		int votes=0;
		boolean voters[]=new boolean[]{false,false,false,false};
		for(int i=0;i<4;i++){
			if(cd[i]!=null&&cd[i].label==SOLVED){votes++;voters[i]=true;}
		}
		if(votes==4){
			/*
			for(int i=0;i<4;i++){
				System.out.println("CORNER "+cd[i].getLocation());
			}
			System.out.println("APPROXIMATE "+result);
			*/
			Point3f avgCorrespondence;
			avgLength=interpolateLength(result)+ref.distance(result);
			avgCorrespondence=interpolateCoorespondence(result);

			dat.setCorrespodence(avgCorrespondence);
			
			dat.setLength(len+avgLength);
			return true;
		} else {
			return false;
		}
	}
	/**
	 * Trace streamline to boundary using lagrangian method
	 * @param cd current node to compute distance
	 * @param boundary true indicates measure distance from outer boundary, false indicates distance from inner boundary
	 */
	protected void traceApproximate(SimpleThicknessNode cd,boolean boundary){
		double len=0;
		Point3f p2;
		Point3f p3;
		Point3f p1=cd.getLocation();
		double step=(boundary)?lagrangeDeltaT:-lagrangeDeltaT;
		int count=0;
		double mag;
		final int maxLength=100;
		while(len<maxLength){
			Point3f tan=interpolateTanget(p1);
			mag=tan.x*tan.x+tan.y*tan.y+tan.z*tan.z;
			if(mag<0.01){
				p1.add(new Point3f((float)Math.random()*1E-4f,(float)Math.random()*1E-4f,(float)Math.random()*1E-4f));			
				System.err.println("TANGET VECTOR INVALID "+count+" "+tan+" "+Math.sqrt(mag)+" "+p1);
				continue;
			}
			p2=new Point3f((float)(p1.x+step*tan.x),(float)(p1.y+step*tan.y),(float)(p1.z+step*tan.z));
				double d=intersector.intersectSegmentDistance(p1, p2);
				p3=intersector.getLastIntersectionPoint();
				if(p3==null){
					if(approximateDistance(cd, p1, p2,len)){
						return;
					} else {
						//cannot approximate distance, continue to trace.
						len+=lagrangeDeltaT;
						//set start of next end point
						p1=p2;
					}
				} else {
					cd.setLength(len+d);
					cd.setCorrespodence(p3);
					return;
				}
			count++;
		}
		System.out.println("MAXED OUT "+p1+" "+interpolateTanget(p1));
		double d=intersector.distance(cd.getLocation());
		cd.setLength(d);
		//cd.setCorrespodence(p);
	}
	/**
	 * Compute distance from outer boundary
	 * @param cd node to compute distance
	 * @param i i
	 * @param j j
	 * @param k k
	 * @param parent parent node
	 * @return distance from outer boundary
	 */
	protected double solveOuterLength(SimpleThicknessNode cd,int i,int j,int k,SimpleThicknessNode parent){
		int ai,aj,ak;
		double tx,ty, tz;
		float[] tan=tangetFieldMat[i][j][k];
		double numer=1;
		double numx=0;
		double numy=0;
		double numz=0;
		double denom=0;
		if(tan[0]<0){
			tx=-tan[0];ai=i-1;
		} else {
			tx=tan[0];ai=i+1;
		}
		if(tan[1]<0){
			ty=-tan[1];aj=j-1;
		} else {
			ty=tan[1];aj=j+1;
		}
		if(tan[2]<0){
			tz=-tan[2];ak=k-1;
		} else {
			tz=tan[2];ak=k+1;
		}
		if(getLabel(ai,j,k)!=UNVISITED){
			numer+=tx*getLength(ai,j,k);
			denom+=tx;
			Point3f px=getCorrespondence(ai,j, k);
			numx+=tx*px.x;
			numy+=tx*px.y;
			numz+=tx*px.z;
		}
		if(getLabel(i,aj,k)!=UNVISITED){
			numer+=ty*getLength(i,aj,k);
			denom+=ty;
			Point3f py=getCorrespondence(i,aj, k);
			numx+=ty*py.x;
			numy+=ty*py.y;
			numz+=ty*py.z;
		}
		if(getLabel(i,j,ak)!=UNVISITED){
			numer+=tz*getLength(i,j,ak);
			denom+=tz;
			Point3f pz=getCorrespondence(i,j, ak);
			numx+=tz*pz.x;
			numy+=tz*pz.y;
			numz+=tz*pz.z;
		}
		double len=numer/denom;
		Point3f p=new Point3f();
		p.x=(float)(numx/denom);
		p.y=(float)(numy/denom);
		p.z=(float)(numz/denom);
		if(Double.isNaN(len)||Double.isInfinite(len)||Math.abs(len-parent.getLength())>MAX_LENGTH ||len<parent.getLength()){
			
			/*
			 Vector3f t =new Vector3f(tan);
			t.add(new Vector3f(tangetFieldMat[parent.i][parent.j][parent.k]));
			t.scale(0.5f);
			Vector3f v=new Vector3f(parent.i-i,parent.j-j,parent.k-k);
			float d=t.dot(v);
			if(d>=0){
				t.scale(d);
				v.sub(t);
				p=new Point3f(parent.getCorrespodence());
				p.add(v);
				cd.setCorrespodence(p);
				return (d+parent.getLength());
			} else {
				
				cd.setCorrespodence(parent.getCorrespodence());
				return (parent.getLength()+v.length());
			}
			 */
			return -1;
		} else {
			cd.setCorrespodence(p);
			return len;
		}
	}
	/**
	 * Compute distance from inner boundary
	 * @param cd node to compute distance
	 * @param i i
	 * @param j j
	 * @param k k
	 * @param parent parent node
	 * @return distance from inner boundary
	 */
	protected double solveInnerLength(SimpleThicknessNode cd,int i,int j,int k,SimpleThicknessNode parent){
		int ai,aj,ak;
		double tx,ty, tz;
		float[] tan=tangetFieldMat[i][j][k];
		double numer=1;
		double numx=0;
		double numy=0;
		double numz=0;
		double denom=0;
		if(tan[0]<0){
			tx=-tan[0];ai=i+1;
		} else {
			tx=tan[0];ai=i-1;
		}
		if(tan[1]<0){
			ty=-tan[1];aj=j+1;
		} else {
			ty=tan[1];aj=j-1;
		}
		if(tan[2]<0){
			tz=-tan[2];ak=k+1;
		} else {
			tz=tan[2];ak=k-1;
		}
		if(getLabel(ai,j,k)!=UNVISITED){
			numer+=tx*getLength(ai,j,k);
			denom+=tx;
			Point3f px=getCorrespondence(ai,j, k);
			numx+=tx*px.x;
			numy+=tx*px.y;
			numz+=tx*px.z;
		}
		if(getLabel(i,aj,k)!=UNVISITED){
			numer+=ty*getLength(i,aj,k);
			denom+=ty;
			Point3f py=getCorrespondence(i,aj, k);
			numx+=ty*py.x;
			numy+=ty*py.y;
			numz+=ty*py.z;
		}
		if(getLabel(i,j,ak)!=UNVISITED){
			numer+=tz*getLength(i,j,ak);
			denom+=tz;
			Point3f pz=getCorrespondence(i,j, ak);
			numx+=tz*pz.x;
			numy+=tz*pz.y;
			numz+=tz*pz.z;
		}
		double len=numer/denom;		
		Point3f p=new Point3f();
		p.x=(float)(numx/denom);
		p.y=(float)(numy/denom);
		p.z=(float)(numz/denom);

		if(Double.isNaN(len)||Double.isInfinite(len)||Math.abs(len-parent.getLength())>MAX_LENGTH||len<parent.getLength()){
			/*
			Vector3f t =new Vector3f(tan);
			t.add(new Vector3f(tangetFieldMat[parent.i][parent.j][parent.k]));
			t.scale(0.5f);
			Vector3f v=new Vector3f(i-parent.i,j-parent.j,k-parent.k);
			float d=t.dot(v);
			if(d>=0){
				
				t.scale(d);
				v.sub(t);
				p=new Point3f(parent.getCorrespodence());
				p.add(v);
				cd.setCorrespodence(p);
				return d+parent.getLength();
			} else {
				cd.setCorrespodence(parent.getCorrespodence());
				return parent.getLength()+v.length();
			}
			*/
			return -1;
		} else {
			
			cd.setCorrespodence(p);
			return len;
		}	

	}
	/**
	 * Initialize boundary 
	 * @param boundary {INSIDE_TAG,OUTSIDE_TAG}
	 */
	protected void initializeBoundary(int boundary){
		int n;
		int ni,nj,nk;
		int i,j,k;
		
		byte[] nbhdX=neighborMask.getNeighborsX();
		byte[] nbhdY=neighborMask.getNeighborsY();
		byte[] nbhdZ=neighborMask.getNeighborsZ();
		SimpleThicknessNode cd;
		heap=new BinaryMinHeap(corticalData.length,rows,cols,slices);

		SimpleThicknessNode.setCompareInnerLength((boundary==INSIDE_TAG));
		for(n=0;n<corticalData.length;n++){
			cd=corticalData[n];
			cd.label=UNVISITED;
		}
		for(n=0;n<corticalData.length;n++){
			cd=corticalData[n];
			
			if(cd.label==UNVISITED){
				i=cd.i;j=cd.j;k=cd.k;
				for(int l=0;l<nbhdX.length;l++){
					ni=i+nbhdX[l];
					nj=j+nbhdY[l];
					nk=k+nbhdZ[l];
					if(ni<0||nj<0||nk<0||ni>=rows||nj>=cols||nk>=slices)continue;
					if(tags[ni][nj][nk]==boundary){
						cd.setCorrespodence(new Point3f(cd.i,cd.j,cd.k));
						//System.out.println("TRACE "+cd.getLocation());
						traceApproximate(cd,(boundary==OUTSIDE_TAG));
						//System.out.println(n+"/"+corticalData.length+" : ("+i+","+j+","+k+") "+cd.innerLength);
						cd.label=SOLVED;
						break;
					}
				}
			}
		}
		
		for(n=0;n<corticalData.length;n++){
			cd=corticalData[n];
			if(cd.label==UNVISITED){
				i=cd.i;j=cd.j;k=cd.k;
				for(int l=0;l<nbhdX.length;l++){
					ni=i+nbhdX[l];
					nj=j+nbhdY[l];
					nk=k+nbhdZ[l];
					if(ni<0||nj<0||nk<0||ni>=rows||nj>=cols||nk>=slices)continue;
					SimpleThicknessNode neighbor=getData(ni,nj,nk);
					if(getLabel(ni,nj,nk)==SOLVED&&neighbor!=null){
						if(boundary==INSIDE_TAG)cd.setLength(solveInnerLength(cd, i, j, k,neighbor)); else cd.setLength(solveOuterLength(cd, i, j, k,neighbor));
						
						//System.out.println("DISTNACE "+neighbor.getCorrespodence()+" "+cd.getCorrespodence()+" "+neighbor.getCorrespodence().distance(cd.getCorrespodence()));
						//System.out.println(n+"/"+corticalData.length+" : ("+i+","+j+","+k+") "+cd.innerLength);
						cd.label=VISITED;
						heap.add(cd);
						break;
					}
				}
			}
		}
		
	}
	/**
	 * Compute thickness for all grid nodes
	 */
	protected void computeThickness() {
		//convertToQuaternions();

		byte[] nbhdX=neighborMask.getNeighborsX();
		byte[] nbhdY=neighborMask.getNeighborsY();
		byte[] nbhdZ=neighborMask.getNeighborsZ();
		int i,j,k,l,ni,nj,nk;
		SimpleThicknessNode cd;
		double len;
		
		int count=0;
		

		boolean tooFar=false;
		double dist=0;
		count=0;
		int totalCount=0;
		IsoSurfaceOnGrid surfGen=new IsoSurfaceOnGrid();
		setLabel("Building Bounding Box Tree");
		setTotalUnits(1);
		innerMesh=(ADJUST_BOUNDARY)?surfGen.solveNoMove(innerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false):surfGen.solveOriginal(innerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false);
		intersector=new SurfaceIntersector(this,12,innerMesh);
		innerMesh=null;
		System.gc();
		this.setLabel("Computing Inner Length");
		
		setTotalUnits(corticalData.length);
		initializeBoundary(INSIDE_TAG);
		
		while(heap.size()>0){
			SimpleThicknessNode data=(SimpleThicknessNode)heap.remove();
			//System.out.println("OUT "+data.getLength()+" "+data.getLocation());
			totalCount++;
			data.label=SOLVED;
			incrementCompletedUnits();
			i=data.i;
			j=data.j;
			k=data.k;
			Point3f corr=data.getCorrespodence();
			tooFar=false;
			for(l=0;l<nbhdX.length;l++){
				ni=i+nbhdX[l];
				nj=j+nbhdY[l];
				nk=k+nbhdZ[l];
				cd=getData(ni,nj,nk);
				if(cd!=null&&(cd.label==VISITED||cd.label==SOLVED)&&(dist=corr.distance(cd.getCorrespodence()))>lambda){
					//System.out.println("TOO FAR "+dist+" "+corr+" "+cd.getCorrespodence());
					tooFar=true;
					break;
				}
			}
			if(tooFar){
				traceApproximate(data, false);
				count++;
			} 
				for(l=0;l<nbhdX.length;l++){
					ni=i+nbhdX[l];
					nj=j+nbhdY[l];
					nk=k+nbhdZ[l];
					cd=getData(ni,nj,nk);
					if(cd!=null){
						if(cd.label==VISITED){
							len=solveInnerLength(cd,ni,nj,nk,data);
							if(len<0){
								double tmpLen=cd.getLength();
								traceApproximate(cd, false);
								len=cd.getLength();
								cd.setLength(tmpLen);
							}
							heap.change(cd,len);
						} else if(cd.label==UNVISITED){
							len=solveInnerLength(cd,ni,nj,nk,data);
							if(len<0){
								traceApproximate(cd,false);
							} else {
								cd.setLength(len);
							}
							cd.label=VISITED;
							heap.add(cd);
						}
					}
				}
		}
		
		markCompleted();
		intersector=null;	
		
		surfGen=new IsoSurfaceOnGrid();
		setLabel("Building Bounding Box Tree");
		setTotalUnits(1);
		outerMesh=(ADJUST_BOUNDARY)?surfGen.solveNoMove(outerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false):surfGen.solveOriginal(outerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false);

		intersector=new SurfaceIntersector(this,12,outerMesh);
		outerMesh=null;
		System.gc();
		this.setLabel("Computing  Outer Length");
		setTotalUnits(corticalData.length);
		
		initializeBoundary(OUTSIDE_TAG);
		//count=0;
		
		while(heap.size()>0){
			SimpleThicknessNode data=(SimpleThicknessNode)heap.remove();
			//System.out.println("OUT "+data.getLength()+" "+data.getLocation());
			totalCount++;
			data.label=SOLVED;
			incrementCompletedUnits();
			i=data.i;
			j=data.j;
			k=data.k;
			Point3f corr=data.getCorrespodence();
			tooFar=false;
			for(l=0;l<nbhdX.length;l++){
				ni=i+nbhdX[l];
				nj=j+nbhdY[l];
				nk=k+nbhdZ[l];
				cd=getData(ni,nj,nk);
				if(cd!=null&&(cd.label==VISITED||cd.label==SOLVED)&&(dist=corr.distance(cd.getCorrespodence()))>lambda){
					//System.out.println("TOO FAR "+dist+" "+corr+" "+cd.getCorrespodence());
					tooFar=true;
					break;
				}
			}
			//System.out.println("DISTANCE "+dist);
			if(tooFar){
				traceApproximate(data, true);
				count++;
			} 
				for(l=0;l<nbhdX.length;l++){
					ni=i+nbhdX[l];
					nj=j+nbhdY[l];
					nk=k+nbhdZ[l];
					cd=getData(ni,nj,nk);
					if(cd!=null){
						if(cd.label==VISITED){
							len=solveOuterLength(cd,ni,nj,nk,data);
							if(len<0){
								double tmpLen=cd.getLength();
								traceApproximate(cd, true);
								len=cd.getLength();
								cd.setLength(tmpLen);
							}
							heap.change(cd,len);
						} else if(cd.label==UNVISITED){
							len=solveOuterLength(cd,ni,nj,nk,data);
							if(len<0){
								traceApproximate(cd,true);
							} else {
								cd.setLength(len);
							}
							cd.label=VISITED;
							heap.add(cd);
						}
					}
				}
		}
		
		markCompleted();
		System.out.println("TRACED "+(100*count/(double)totalCount)+"%");
	}
}
