package edu.jhu.ece.iacl.algorithms.thickness;

import java.util.ArrayList;
import java.util.Arrays;

import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.IntersectorTriangle;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.SurfaceIntersector;
import edu.jhu.ece.iacl.algorithms.graphics.isosurf.IsoSurfaceOnGrid;
import edu.jhu.ece.iacl.algorithms.graphics.utilities.rbf.RadialBasisFuncFloat4D;
import edu.jhu.ece.iacl.algorithms.graphics.utilities.rbf.RadialFunctionType;
import edu.jhu.ece.iacl.algorithms.thickness.grid.EmbeddedGrid;
import edu.jhu.ece.iacl.algorithms.thickness.grid.EmbeddedNode;
import edu.jhu.ece.iacl.algorithms.thickness.grid.GridNode;
import edu.jhu.ece.iacl.algorithms.thickness.grid.SimpleThicknessNode;
import edu.jhu.ece.iacl.algorithms.thickness.laplace.LaplaceSolveOnGraph;
import edu.jhu.ece.iacl.algorithms.thickness.laplace.LaplaceSolveOnGraphDirichlet;
import edu.jhu.ece.iacl.algorithms.topology.ConnectivityRule;
import edu.jhu.ece.iacl.jist.structures.geom.CurveCollection;
import edu.jhu.ece.iacl.jist.structures.geom.CurvePath;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;
/**
 * Robust Lagrangian Thickness uses a runge-kutta scheme for tracing streamlines and interpolates tangential vectors
 * near the boundary using radial basis functions, which produces smoother streamlines that are more accurate near
 * the shell boundary.
 * @author Blake Lucas (bclucas@jhu.edu)
 *
 */
public class RobustLagrangeThickness extends ThicknessSolver {
	public RobustLagrangeThickness(double lambda,double isoVal,double lagrangeDeltaT){
		super(lambda,isoVal,lagrangeDeltaT);
		setLabel("Robust Lagrange Thickness Solver");
		streamlines=new CurveCollection();
	}
	EmbeddedGrid grid;
	protected GridNode[][][] gridMat;
	int[] innerMask,outerMask;
	SurfaceIntersector intersector;
	public void solve(ImageData inner, ImageData outer) {
		this.innerVol = inner;
		this.outerVol = outer;
		rows = innerVol.getRows();
		cols = innerVol.getCols();
		slices = innerVol.getSlices();
		grid = new EmbeddedGrid(this,rows, cols, slices,false,false);
		gridMat=grid.getGrid();
		if(ADJUST_BOUNDARY)adjustBoundary(innerVol, outerVol);
		IsoSurfaceOnGrid surfGen=new IsoSurfaceOnGrid();
		innerMesh=(ADJUST_BOUNDARY)?surfGen.solveNoMove(innerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false):surfGen.solveOriginal(innerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false);
		this.innerVol=surfGen.getAlteredLevelSet();
		outerMesh=(ADJUST_BOUNDARY)?surfGen.solveNoMove(outerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false):surfGen.solveOriginal(outerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false);
		this.outerVol=surfGen.getAlteredLevelSet();
		thicknessVol = new ImageDataFloat(rows, cols, slices);
		thicknessMat = thicknessVol.toArray3d();
		thicknessVol.setName(innerVol.getName()+"_thick");
		grid.build(this.innerVol, this.outerVol, innerMesh,outerMesh,isoVal,EmbeddedGrid.Topology.EXTENDED);
		markCompleted();
		computeLaplace();
		markCompleted();
		computeThickness();
		updateLevelSet();
		updateVolumes();
		innerMesh=getInnerSurface();
		outerMesh=getOuterSurface();
		
		grid.updateMeshThickness(innerMesh, outerMesh);
		updateSurfStats(innerMesh);
		updateSurfStats(outerMesh);
		markCompleted();
	}
	/**
	 * Solve laplace's equation
	 */
	protected void computeLaplace() {		
		LaplaceSolveOnGraph laplaceSolver=new LaplaceSolveOnGraphDirichlet(this,rows,cols,slices);
		tangetField=laplaceSolver.solveTanget(grid);
		tangetFieldMat=tangetField.toArray4d();
		tangetField.setName(innerVol.getName()+"_tanget");
	}
	protected CurveCollection streamlines; 
	protected double streamCount=15000;
	protected void computeThickness() {
			int n;
			innerMask=new int[innerMesh.getVertexCount()];
			outerMask=new int[outerMesh.getVertexCount()];
			EmbeddedNode[] nodes=grid.getBoundaryNodes();
			setTotalUnits(nodes.length);
			IsoSurfaceOnGrid surfGen=new IsoSurfaceOnGrid();
			
			System.out.println("HOLES IN OUTER SURFACE "+outerMesh.getNumberOfHoles());
			SurfaceIntersector outerIntersect,innerIntersect;
			outerIntersect=intersector=new SurfaceIntersector(this,12,outerMesh);
			System.gc();
			this.setLabel("Computing Outer Length");
			SimpleThicknessNode.setCompareInnerLength(false);
			ThickIndex[] indexes=new ThickIndex[nodes.length];
			//For each node, compute the streamline length to the inner surface measured from that location
			int sz=innerMesh.getVertexCount();
			for(n=0;n<sz;n++){
				EmbeddedNode node=nodes[n];
				if(node.regionLabel!=INNER_BOUNDARY)continue;
				incrementCompletedUnits();
				node.setLength(trace(node.getLocation(),true));
				indexes[n]=new ThickIndex(node,n,true);
				
			}
			markCompleted();
			
			System.out.println("HOLES IN INNER SURFACE "+innerMesh.getNumberOfHoles());
			innerIntersect=intersector=new SurfaceIntersector(this,12,innerMesh);
			System.gc();
			this.setLabel("Computing Inner Length");
			SimpleThicknessNode.setCompareInnerLength(true);
			//For each nod, compute the streamline length to the outer surface measured from that location
			for(;n<nodes.length;n++){
				EmbeddedNode node=nodes[n];
				if(node.regionLabel!=OUTER_BOUNDARY)continue;
				incrementCompletedUnits();
				node.setLength(trace(node.getLocation(),false));
				indexes[n]=new ThickIndex(node,n,false);
			}
			//Sort streamlines by length
			Arrays.sort(indexes);
			for(int i=0;i<streamCount&&i<indexes.length;i++){
				if(indexes[i].boundary){
					intersector=outerIntersect;
					innerMask[indexes[i].index]=1;
				} else {
					intersector=innerIntersect;
					outerMask[indexes[i].index-innerMask.length]=1;
				}
				//Remember path of longest streamlines for visualization
				createStreamline(indexes[i].node.getLocation(), indexes[i].boundary);
				System.out.println("Streamline Length "+indexes[i].node.getThickness());
			}

			intersector=null;	
			
			markCompleted();
		
	}
	protected class ThickIndex implements Comparable<ThickIndex>{
		boolean boundary;
		EmbeddedNode node;
		int index;
		public ThickIndex(EmbeddedNode n,int index,boolean boundary){
			this.node=n;
			this.index=index;
			this.boundary=boundary;
		}
		public int compareTo(ThickIndex obj) {
			return (int)Math.signum(obj.node.getThickness()-node.getThickness());
		}
	}
	/**
	 * Update level set used for testing
	 */
	protected void updateLevelSet(){
		int i,j,k;
		levelSetVol=new ImageDataFloat(rows,cols,slices);
		levelSetVol.setName(innerVol.getName()+"_levelset");
		float[][][] levelSetMat=levelSetVol.toArray3d();
		byte[][][] labels=grid.getLabels();
		byte label;
		for(EmbeddedNode n:grid.getBoundaryNodes()){
			n.setMarchingLabel((byte)0);
		}
		for (i = 0; i < rows; i++) {
			for (j = 0; j < cols; j++) {
				for (k = 0; k < slices; k++) {
					EmbeddedNode node = grid.getNode(i, j, k);
					label=labels[i][j][k];
					
					if(label==JUST_INSIDE||label==INSIDE){
						node.setMarchingLabel((byte)levelSetMat[node.i][node.j][node.k]);
						//levelSetMat[i][j][k]=node.getMarchingLabel();
						Vector3f tan=new Vector3f(node.getTanget());
						if(!EmbeddedNode.isCompareToInner()){
							tan.negate();
						}
						//if(levelSetMat[i][j][k]!=0){
							//levelSetMat[i][j][k]=(float)(GeomUtil.angle(tan, node.getGradient())*180.0/Math.PI);
						//}
						//levelSetMat[i][j][k]=(float)(GeomUtil.angle(tan, node.getGradient())*180.0/Math.PI);
					} else {
						levelSetMat[i][j][k]=-1;
					}
					
					/*
					if(label==JUST_INSIDE||label==INSIDE){
						levelSetMat[i][j][k]=(float)node.getNormalizedLength();//(float)(Math.atan(5*(l0-level*(l0+l1)))/Math.PI+0.5);
					} else {
						levelSetMat[i][j][k]=0;
					}
					*/
					
				}
			}
		}
	}
	/**
	 * Update volumes and thickness statistics
	 */
	protected void updateVolumes() {
		double sqrs = 0;
		double sum = 0;
		double count = 0;
		maxThickness=0;
		minThickness=1E10;
		for (EmbeddedNode n:grid.getBoundaryNodes()) {
			double thick=n.getThickness();
			if(thick>=MAX_LENGTH)continue;
							maxThickness=Math.max(thick, maxThickness);
							minThickness=Math.min(thick, minThickness);
							sqrs += (thick) * (thick);
							sum += thick;
							count++;
		}
		meanThickness=(sum/count);
		stdevThickness=Math.sqrt((sqrs-sum*sum/count)/(count-1));

		double expected=Math.round(meanThickness);

		System.out.println(expected+" "+meanThickness+" "+stdevThickness+" "+minThickness+" "+maxThickness+" "+count);
	}

	public CurveCollection getStreamlines(){
		streamlines.setName(innerVol.getName()+"_streams");
		return streamlines;
	}
	protected static final int MAX_COUNT=1000;
	protected static final float MAX_LENGTH=40;
	//Create streamline geometry for particular path
	protected void createStreamline(Point3f p1,boolean boundary){
		CurvePath path=new CurvePath();
		double len=0;
		Point3f p2=null;
		int count=0;
		double step=(boundary)?lagrangeDeltaT:-lagrangeDeltaT;
		double mag;
		//Point3f start=p1;
		//double d;
		path.add(p1);
		Point3f tan_1=null;
		Point3f tan_2=null;
		Point3f tan_3=null;
		Point3f tan;
		while(len<MAX_LENGTH&&count<MAX_COUNT){
			tan_1=tan_2;
			tan_2=tan_3;
			tan_3=interpolateTanget(p1);
			if(tan_2!=null){
				if(tan_1!=null){
					tan=new Point3f(0.166666f*tan_3.x+0.666668f*tan_2.x+0.166666f*tan_1.x,
									0.166666f*tan_3.y+0.666668f*tan_2.y+0.166666f*tan_1.y,
									0.166666f*tan_3.z+0.666668f*tan_2.z+0.166666f*tan_1.z);
				} else {
					tan=new Point3f(0.5f*(tan_3.x+tan_2.x),0.5f*(tan_3.y+tan_2.y),0.5f*(tan_3.z+tan_2.z));
				}
			} else {
				tan=tan_3;
			}
			mag=Math.sqrt(tan.x*tan.x+tan.y*tan.y+tan.z*tan.z);
			if(mag==0){
				System.out.println("Zero Magnitude! "+tan+" "+count+" "+tan_3);
				break;
			}
			p2=new Point3f((float)(p1.x+step*tan.x),(float)(p1.y+step*tan.y),(float)(p1.z+step*tan.z));
			double dist=intersector.intersectSegmentDistance(p1,p2);
			
				if(dist<0){
					len+=lagrangeDeltaT*mag;
					if(!Float.isNaN(p2.x)&&!Float.isNaN(p2.y)&&!Float.isNaN(p2.z)){
						path.add(p2);
					} else {
						streamlines.add(path);
						return;						
					}
					p1=p2;
				} else {
					Point3f pt=intersector.getLastIntersectionPoint();
					if(!Float.isNaN(pt.x)&&!Float.isNaN(pt.y)&&!Float.isNaN(pt.z)){
						path.add(pt);
					} else {
						streamlines.add(path);
						return;						
					}
					IntersectorTriangle tri=(IntersectorTriangle)intersector.getLastIntersectionTriangle();
					if(boundary){
						outerMask[tri.id1]=1;
						outerMask[tri.id2]=1;
						outerMask[tri.id3]=1;
					} else {
						innerMask[tri.id1]=1;
						innerMask[tri.id2]=1;
						innerMask[tri.id3]=1;
					}
					streamlines.add(path);
					
					return;
				}
			count++;
		}

		streamlines.add(path);	
	}
	/**
	 * Trace distance from point
	 * @param p1 point
	 * @param boundary
	 *            false indicates trace distance to inner boundary, true
	 *            indicates trace distance to outer boundary
	 * @return
	 */
	protected double trace(Point3f p1,boolean boundary){
		double len=0;
		Point3f p2=null;
		int count=0;
		double step=(boundary)?lagrangeDeltaT:-lagrangeDeltaT;
		double mag;
		//Point3f start=p1;
		//double d;
		Point3f tan_1=null;
		Point3f tan_2=null;
		Point3f tan_3=null;
		Point3f tan;
		//Use Runge-Kutta 4th order scheme
		while(len<MAX_LENGTH&&count<MAX_COUNT){
			tan_1=tan_2;
			tan_2=tan_3;
			tan_3=interpolateTanget(p1);
			if(tan_2!=null){
				if(tan_1!=null){
					tan=new Point3f(0.166666f*tan_3.x+0.666668f*tan_2.x+0.166666f*tan_1.x,
									0.166666f*tan_3.y+0.666668f*tan_2.y+0.166666f*tan_1.y,
									0.166666f*tan_3.z+0.666668f*tan_2.z+0.166666f*tan_1.z);
				} else {
					tan=new Point3f(0.5f*(tan_3.x+tan_2.x),0.5f*(tan_3.y+tan_2.y),0.5f*(tan_3.z+tan_2.z));
				}
			} else {
				tan=tan_3;
			}
			mag=Math.sqrt(tan.x*tan.x+tan.y*tan.y+tan.z*tan.z);
			if(mag==0){
				System.out.println("Zero Magnitude! "+tan+" "+count+" "+tan_3);
				break;
			}
			p2=new Point3f((float)(p1.x+step*tan.x),(float)(p1.y+step*tan.y),(float)(p1.z+step*tan.z));
			double dist=intersector.intersectSegmentDistance(p1,p2);
				if(dist<0){
					len+=lagrangeDeltaT*mag;
					p1=p2;
				} else {
					return len+dist;
				}
			count++;
		}
		//d=intersector.distance(start);
		System.out.println("Count or Length Max "+count+" "+len);
		return len;
	}
	/**
	 * Interpolate tangential field using either linear interpolation or radial basis functions
	 */
	protected Point3f interpolateTanget(Point3f p){
		
		int i=(int)Math.floor(p.x);
		int j=(int)Math.floor(p.y);
		int k=(int)Math.floor(p.z);
		if(i==p.x&&j==p.y&&k==p.z){
			return new Point3f(interpolateTanget(p.x,p.y,p.z));
		}
		
		for(int ni=0;ni<2;ni++){
			for(int nj=0;nj<2;nj++){
				for(int nk=0;nk<2;nk++){
					EmbeddedNode n=grid.getNode(i+ni, j+nj, k+nk);
					if(n==null||n.isOutside()){
						//Point3f vp1=new Point3f(interpolateTanget(p.x,p.y,p.z));
						Point3f vp2=interpolateTangetRBF(p);
						//System.out.println("ANGLE "+(180/Math.PI)*GeomUtil.angle(vp1, vp2)+" "+vp2+" "+vp1);
						return vp2;
					}
				}
			}
		}
		
		return new Point3f(interpolateTanget(p.x,p.y,p.z));
	}
	/**
	 * interpolate vector field using radial basis functions
	 * @param p location
	 * @return interpolated vector
	 */
	protected Point3f interpolateTangetRBF(Point3f p){
		ArrayList<EmbeddedNode> nbrs=grid.getNeighbors(p);
		double tanx[]=new double[nbrs.size()];
		double tany[]=new double[nbrs.size()];
		double tanz[]=new double[nbrs.size()];
		Point3f locs[]=new Point3f[nbrs.size()];
		for(int i=0;i<nbrs.size();i++){
			EmbeddedNode n=nbrs.get(i);
			locs[i]=n.getLocation();
			Vector3f tan=n.getTanget();
			tanx[i]=tan.x;
			tany[i]=tan.y;
			tanz[i]=tan.z;
		}
		//Use thin plate spline to interpolate
		RadialBasisFuncFloat4D rbfx=new RadialBasisFuncFloat4D(locs,tanx ,RadialFunctionType.THIN_PLATE);
		RadialBasisFuncFloat4D rbfy=new RadialBasisFuncFloat4D(locs,tany,RadialFunctionType.THIN_PLATE);
		RadialBasisFuncFloat4D rbfz=new RadialBasisFuncFloat4D(locs,tanz,RadialFunctionType.THIN_PLATE);
		Point3f tan=new Point3f(rbfx.interpolate(p),rbfy.interpolate(p),rbfz.interpolate(p));
		GeometricUtilities.normalize(tan);
		return tan;
	}
}
