package edu.jhu.ece.iacl.algorithms.thickness.laplace;

import javax.vecmath.Vector3f;

import edu.jhu.ece.iacl.algorithms.thickness.grid.BoundaryNode;
import edu.jhu.ece.iacl.algorithms.thickness.grid.EmbeddedGrid;
import edu.jhu.ece.iacl.algorithms.thickness.grid.EmbeddedNode;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataDouble;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;

/**
 * Solve laplace's equation for rectilinear graph.
 * 
 * @author Blake Lucas
 * 
 */
public abstract class LaplaceSolveOnGraph extends AbstractCalculation {
	protected static final double SMOOTH_ERROR = 1E-20;
	protected static final int MAX_ITERS = 200;
	protected static final float ETA = 0.999f;
	protected static final byte INSIDE_NARROW_BAND = EmbeddedNode.INSIDE_NARROW_BAND;
	protected static final byte OUTSIDE_NARROW_BAND = EmbeddedNode.OUTSIDE_NARROW_BAND;
	protected static final byte OUTSIDE_OUTER = EmbeddedNode.OUTSIDE_OUTER;
	protected static final byte INSIDE_INNER = EmbeddedNode.INSIDE_INNER;
	protected static final byte JUST_OUTSIDE_OUTER_BOUNDARY = EmbeddedNode.JUST_OUTSIDE_OUTER_BOUNDARY;
	protected static final byte OUTER_BOUNDARY = EmbeddedNode.OUTER_BOUNDARY;
	protected static final byte JUST_INSIDE = EmbeddedNode.JUST_INSIDE;
	protected static final byte INSIDE = EmbeddedNode.INSIDE;
	protected static final byte JUST_INSIDE_INNER_BOUNDARY = EmbeddedNode.JUST_INSIDE_INNER_BOUNDARY;
	protected static final byte INNER_BOUNDARY = EmbeddedNode.INNER_BOUNDARY;

	protected static final byte EAST = EmbeddedNode.EAST;
	protected static final byte WEST = EmbeddedNode.WEST;
	protected static final byte NORTH = EmbeddedNode.NORTH;

	protected static final byte SOUTH = EmbeddedNode.SOUTH;
	protected static final byte UP = EmbeddedNode.UP;
	protected static final byte DOWN = EmbeddedNode.DOWN;
	public ImageDataDouble laplaceVol;
	protected ImageDataFloat tangetField;
	protected int rows, cols, slices;
	protected EmbeddedNode[][][] gridMat;
	protected double[][][] laplaceMat;
	protected float[][][][] tangetFieldMat;
	protected EmbeddedGrid grid;

	public LaplaceSolveOnGraph(AbstractCalculation parent, int rows, int cols,
			int slices) {
		super(parent);
		this.setLabel("Solving Laplace");
		this.setTotalUnits(MAX_ITERS);
		this.rows = rows;
		this.cols = cols;
		this.slices = slices;
		laplaceVol = new ImageDataDouble(rows, cols, slices);
		laplaceMat = laplaceVol.toArray3d();
		tangetField = new ImageDataFloat(rows, cols, slices, 3);
		tangetFieldMat = tangetField.toArray4d();
	}

	public ImageDataFloat solveTanget(EmbeddedGrid grid) {
		solve(grid);
		computeGradient(true);
		return tangetField;
	}

	public ImageDataFloat solveGradient(EmbeddedGrid grid) {
		solve(grid);
		computeGradient(false);
		return tangetField;
	}

	public ImageDataFloat solve(EmbeddedGrid grid) {
		this.grid = grid;
		gridMat = grid.getGrid();
		double err = 0;
		byte[][][] labels = grid.getLabels();
		for (BoundaryNode node : grid.getBoundaryNodes()) {
			node.getTanget().scale(0.001f);
		}
		for (int i = 0; i < rows; i++) {
			for (int j = 0; j < cols; j++) {
				for (int k = 0; k < slices; k++) {
					if (labels[i][j][k] == OUTSIDE_NARROW_BAND) {
						laplaceMat[i][j][k] = 2;
					} else if (labels[i][j][k] == INSIDE_NARROW_BAND) {
						laplaceMat[i][j][k] = -1;
					} else if (labels[i][j][k] == OUTSIDE_OUTER
							|| labels[i][j][k] == JUST_OUTSIDE_OUTER_BOUNDARY) {
						laplaceMat[i][j][k] = 1.5f;
					} else if (labels[i][j][k] == INSIDE_INNER
							|| labels[i][j][k] == JUST_INSIDE_INNER_BOUNDARY) {
						laplaceMat[i][j][k] = -0.5f;
					} else {
						laplaceMat[i][j][k] = 0.5;
					}
				}
			}
		}
		int i = 0;
		for (i = 0; i < MAX_ITERS; i++) {
			err = 0;
			err += octant(0, 0, 0);
			err += octant(0, 1, 1);
			err += octant(1, 0, 1);
			err += octant(1, 1, 0);
			err += octant(1, 0, 0);
			err += octant(0, 0, 1);
			err += octant(0, 1, 0);
			err += octant(1, 1, 1);
			err /= 8;
			// System.out.println("Iteration "+i+" Error "+err);
			incrementCompletedUnits();
			if (err < SMOOTH_ERROR)
				break;
		}

		System.out.println("Laplace Solver Iterations " + i + " Error " + err);

		markCompleted();
		return tangetField;
	}

	protected double octant(int ic, int jc, int kc) {
		int x, y, z;
		double err = 0;
		int count = 0;
		double e = 0;
		byte[][][] labels = grid.getLabels();
		for (x = kc; x < rows; x += 2) {
			for (y = ic; y < cols; y += 2) {
				for (z = jc; z < slices; z += 2) {
					if (labels[x][y][z] != INSIDE_NARROW_BAND
							&& labels[x][y][z] != OUTSIDE_NARROW_BAND) {
						e = computeLaplace(x, y, z);
						laplaceMat[x][y][z] += (float) (ETA * e);
						err += e * e;
						count++;
					}
				}
			}
		}
		if (count > 0) {
			return err / count;
		} else {
			return 0;
		}
	}

	public ImageDataDouble getLaplaceVolume() {
		return laplaceVol;
	}

	protected abstract double computeLaplace(int i, int j, int k);

	protected abstract void computeGradient(int i, int j, int k);

	protected void computeGradient(boolean normalizeData) {
		setTotalUnits(rows);
		int i, j, k;
		for (i = 0; i < rows; i++) {
			for (j = 0; j < cols; j++) {
				for (k = 0; k < slices; k++) {
					if (grid.getNode(i, j, k) != null) {
						grid.getNode(i, j, k).setLength(laplaceMat[i][j][k]);
					}
				}
			}
		}
		for (EmbeddedNode node : grid.getBoundaryNodes()) {
			if (node.isInnerBoundary()) {
				node.setLength((EmbeddedNode.isCompareToInner()) ? 0 : 1);
			}
			if (node.isOuterBoundary()) {
				node.setLength((EmbeddedNode.isCompareToInner()) ? 1 : 0);
			}
		}
		for (i = 0; i < rows; i++) {
			for (j = 0; j < cols; j++) {
				for (k = 0; k < slices; k++) {
					computeGradient(i, j, k);
				}
			}
			incrementCompletedUnits();
		}

		for (EmbeddedNode node : grid.getGridNodes()) {
			i = node.i;
			j = node.j;
			k = node.k;
			// System.out.printf("(%6.3f,%6.3f,%6.3f)\n",
			// tangetFieldMat[i][j][k][0],tangetFieldMat[i][j][k][1],tangetFieldMat[i][j][k][2]);

			Vector3f v = new Vector3f(tangetFieldMat[i][j][k][0],
					tangetFieldMat[i][j][k][1], tangetFieldMat[i][j][k][2]);
			if (normalizeData) {
				if (v.length() > 0) {
					v.normalize();
					tangetFieldMat[i][j][k][0] = v.x;
					tangetFieldMat[i][j][k][1] = v.y;
					tangetFieldMat[i][j][k][2] = v.z;
				} else {
					if (node.isInterior()) {
						System.out.println("ZERO LENGTH (" + i + "," + j + ","
								+ k + ") " + v + " "
								+ node.getRegionLabelString() + " "
								+ node.getLength() + " " + laplaceMat[i][j][k]);
					}
				}
			}
			node.setTanget(v);
		}
		for (EmbeddedNode node : grid.getBoundaryNodes()) {
			if (normalizeData) {
				node.getTanget().normalize();
			} else {
				node.setGradient(node.getTanget());
			}
		}

	}
}
