/**
 * 
 */
package edu.jhu.ece.iacl.plugins.segmentation.skull_strip;

import edu.jhu.ece.iacl.algorithms.PrinceGroupAuthors;
import edu.jhu.ece.iacl.algorithms.ReferencedPapers;
import edu.jhu.ece.iacl.algorithms.skull_strip.SmoothBrainMask;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamDouble;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamSurface;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;


/**
 * @author Blake Lucas (bclucas@jhu.edu)
 *
 */
public class MedicAlgorithmSmoothBrainMask extends ProcessingAlgorithm{
	ParamVolume brainmask;
	ParamVolume origVol;
	ParamVolume newBrainMask;
	ParamVolume maskedBrain;
	ParamDouble levelSetVal;
	ParamDouble maxCurvature;
	ParamDouble translation;
	ParamSurface shrunkSurf;

	private static final String revnum = SmoothBrainMask.get_version();

	protected void createInputParameters(ParamCollection inputParams) {
		inputParams.add(origVol=new ParamVolume("Original Image"));
		origVol.setDescription("Original image before skull stripping");
		inputParams.add(brainmask=new ParamVolume("Brain Mask"));
		brainmask.setDescription("Brain mask volume {0,1}");
		inputParams.add(levelSetVal=new ParamDouble("Level Set Value",0,10000,0.5));
		inputParams.add(maxCurvature=new ParamDouble("Max Curvature",0,20,0.2));
		inputParams.add(translation=new ParamDouble("Translation",-1E6,1E6,0));

		inputParams.setName("smooth_brainmask");
		inputParams.setLabel("Smooth Brain Mask");

		inputParams.setPackage("IACL");
		inputParams.setCategory("Segmentation.Skull_Stripping");

		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("http://www.iacl.ece.jhu.edu/");
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.RC);
		info.add(PrinceGroupAuthors.blakeLucas);
		info.add(ReferencedPapers.implicitFairing);
		info.add(ReferencedPapers.progressiveMesh);
		info.add(ReferencedPapers.topoCorrect);
		info.setLongDescription("This method improves the smoothness of the brain mask generated by SPECTRE since the brain mask is likely to have sharp indentations or cusps which are clearly not indicative of brain geometry.");
		info.setDescription("Smooths a skull strip mask using the following procedure:\n1) Perform topology correction on brain mask.\n2) Generate iso-surface for brain mask.\n3) Identify the largest connected surface component as the brain.\n4) Create a progressive mesh decomposition of the surface.\n5) Implicitly smooth the surface at multiple resolutions using curvature weighted laplacian flow.\n6) Apply a scan-line rasterization to transform the volume back into a mask.\n7) Dilate brain mask.\n8) Apply brain mask to original image.");
	}

	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.add(shrunkSurf=new ParamSurface("Smoothed Brain Mask Surface"));
		outputParams.add(newBrainMask=new ParamVolume("Smooth Brain Mask Volume"));
		outputParams.add(maskedBrain=new ParamVolume("Masked Brain Volume"));
	}

	protected void execute(CalculationMonitor monitor) throws AlgorithmRuntimeException {
		SmoothBrainMask smoothBrainMask=new SmoothBrainMask();
		monitor.observe(smoothBrainMask);
		
		EmbeddedSurface surf=smoothBrainMask.solve(brainmask.getImageData(),origVol.getImageData(),levelSetVal.getFloat(),maxCurvature.getFloat(),translation.getFloat());
		shrunkSurf.setValue(surf);
		maskedBrain.setValue(smoothBrainMask.getMaskedBrain());
		newBrainMask.setValue(smoothBrainMask.getBrainMask());
	}
}
