package edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree;

import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

/**
 * A line segment used for fast intersection testing against a surface stored as
 * a Kd-Tree.
 * 
 * @author Blake Lucas
 * 
 */
public class KdSegment {
	public float extent;
	public Vector3f direction;
	public Point3f origin;
	protected Point3f lastIntersect;
	public static final double ZERO_TOLERANCE = 1E-6;

	/**
	 * Constructor
	 * 
	 * @param p1
	 *            end point 1
	 * @param p2
	 *            end point 2
	 */
	public KdSegment(Point3f p1, Point3f p2) {
		origin = p1;
		direction = new Vector3f();
		direction.sub(p2, p1);
		extent = direction.length();
		direction.scale(1.0f / extent);
	}

	/**
	 * Constructor
	 * 
	 * @param p1
	 *            end point 1
	 * @param v
	 *            direction
	 */
	public KdSegment(Point3f p1, Vector3f v) {
		origin = p1;
		direction = v;
		extent = 1E10f;
	}

	/**
	 * Compute distance between two segments. This is a port of the algorithm
	 * from WildMagic.
	 * 
	 * @param seg
	 *            segment
	 * @return distance between segments.
	 */
	public double distance(KdSegment seg) {
		Vector3f kDiff = new Vector3f();
		kDiff.sub(this.origin, seg.origin);
		double fA01 = -this.direction.dot(seg.direction);
		double fB0 = kDiff.dot(this.direction);
		double fB1 = -kDiff.dot(seg.direction);
		double fC = kDiff.lengthSquared();
		double fDet = Math.abs((double) 1.0 - fA01 * fA01);
		double fS0, fS1, fSqrDist, fExtDet0, fExtDet1, fTmpS0, fTmpS1;

		if (fDet >= ZERO_TOLERANCE) {
			// segments are not parallel
			fS0 = fA01 * fB1 - fB0;
			fS1 = fA01 * fB0 - fB1;
			fExtDet0 = this.extent * fDet;
			fExtDet1 = seg.extent * fDet;

			if (fS0 >= -fExtDet0) {
				if (fS0 <= fExtDet0) {
					if (fS1 >= -fExtDet1) {
						if (fS1 <= fExtDet1) // region 0 (interior)
						{
							// minimum at two interior points of 3D lines
							double fInvDet = ((double) 1.0) / fDet;
							fS0 *= fInvDet;
							fS1 *= fInvDet;
							fSqrDist = fS0
									* (fS0 + fA01 * fS1 + ((double) 2.0) * fB0)
									+ fS1
									* (fA01 * fS0 + fS1 + ((double) 2.0) * fB1)
									+ fC;
						} else // region 3 (side)
						{
							fS1 = seg.extent;
							fTmpS0 = -(fA01 * fS1 + fB0);
							if (fTmpS0 < -this.extent) {
								fS0 = -this.extent;
								fSqrDist = fS0
										* (fS0 - ((double) 2.0) * fTmpS0) + fS1
										* (fS1 + ((double) 2.0) * fB1) + fC;
							} else if (fTmpS0 <= this.extent) {
								fS0 = fTmpS0;
								fSqrDist = -fS0 * fS0 + fS1
										* (fS1 + ((double) 2.0) * fB1) + fC;
							} else {
								fS0 = this.extent;
								fSqrDist = fS0
										* (fS0 - ((double) 2.0) * fTmpS0) + fS1
										* (fS1 + ((double) 2.0) * fB1) + fC;
							}
						}
					} else // region 7 (side)
					{
						fS1 = -seg.extent;
						fTmpS0 = -(fA01 * fS1 + fB0);
						if (fTmpS0 < -this.extent) {
							fS0 = -this.extent;
							fSqrDist = fS0 * (fS0 - ((double) 2.0) * fTmpS0)
									+ fS1 * (fS1 + ((double) 2.0) * fB1) + fC;
						} else if (fTmpS0 <= this.extent) {
							fS0 = fTmpS0;
							fSqrDist = -fS0 * fS0 + fS1
									* (fS1 + ((double) 2.0) * fB1) + fC;
						} else {
							fS0 = this.extent;
							fSqrDist = fS0 * (fS0 - ((double) 2.0) * fTmpS0)
									+ fS1 * (fS1 + ((double) 2.0) * fB1) + fC;
						}
					}
				} else {
					if (fS1 >= -fExtDet1) {
						if (fS1 <= fExtDet1) // region 1 (side)
						{
							fS0 = this.extent;
							fTmpS1 = -(fA01 * fS0 + fB1);
							if (fTmpS1 < -seg.extent) {
								fS1 = -seg.extent;
								fSqrDist = fS1
										* (fS1 - ((double) 2.0) * fTmpS1) + fS0
										* (fS0 + ((double) 2.0) * fB0) + fC;
							} else if (fTmpS1 <= seg.extent) {
								fS1 = fTmpS1;
								fSqrDist = -fS1 * fS1 + fS0
										* (fS0 + ((double) 2.0) * fB0) + fC;
							} else {
								fS1 = seg.extent;
								fSqrDist = fS1
										* (fS1 - ((double) 2.0) * fTmpS1) + fS0
										* (fS0 + ((double) 2.0) * fB0) + fC;
							}
						} else // region 2 (corner)
						{
							fS1 = seg.extent;
							fTmpS0 = -(fA01 * fS1 + fB0);
							if (fTmpS0 < -this.extent) {
								fS0 = -this.extent;
								fSqrDist = fS0
										* (fS0 - ((double) 2.0) * fTmpS0) + fS1
										* (fS1 + ((double) 2.0) * fB1) + fC;
							} else if (fTmpS0 <= this.extent) {
								fS0 = fTmpS0;
								fSqrDist = -fS0 * fS0 + fS1
										* (fS1 + ((double) 2.0) * fB1) + fC;
							} else {
								fS0 = this.extent;
								fTmpS1 = -(fA01 * fS0 + fB1);
								if (fTmpS1 < -seg.extent) {
									fS1 = -seg.extent;
									fSqrDist = fS1
											* (fS1 - ((double) 2.0) * fTmpS1)
											+ fS0
											* (fS0 + ((double) 2.0) * fB0) + fC;
								} else if (fTmpS1 <= seg.extent) {
									fS1 = fTmpS1;
									fSqrDist = -fS1 * fS1 + fS0
											* (fS0 + ((double) 2.0) * fB0) + fC;
								} else {
									fS1 = seg.extent;
									fSqrDist = fS1
											* (fS1 - ((double) 2.0) * fTmpS1)
											+ fS0
											* (fS0 + ((double) 2.0) * fB0) + fC;
								}
							}
						}
					} else // region 8 (corner)
					{
						fS1 = -seg.extent;
						fTmpS0 = -(fA01 * fS1 + fB0);
						if (fTmpS0 < -this.extent) {
							fS0 = -this.extent;
							fSqrDist = fS0 * (fS0 - ((double) 2.0) * fTmpS0)
									+ fS1 * (fS1 + ((double) 2.0) * fB1) + fC;
						} else if (fTmpS0 <= this.extent) {
							fS0 = fTmpS0;
							fSqrDist = -fS0 * fS0 + fS1
									* (fS1 + ((double) 2.0) * fB1) + fC;
						} else {
							fS0 = this.extent;
							fTmpS1 = -(fA01 * fS0 + fB1);
							if (fTmpS1 > seg.extent) {
								fS1 = seg.extent;
								fSqrDist = fS1
										* (fS1 - ((double) 2.0) * fTmpS1) + fS0
										* (fS0 + ((double) 2.0) * fB0) + fC;
							} else if (fTmpS1 >= -seg.extent) {
								fS1 = fTmpS1;
								fSqrDist = -fS1 * fS1 + fS0
										* (fS0 + ((double) 2.0) * fB0) + fC;
							} else {
								fS1 = -seg.extent;
								fSqrDist = fS1
										* (fS1 - ((double) 2.0) * fTmpS1) + fS0
										* (fS0 + ((double) 2.0) * fB0) + fC;
							}
						}
					}
				}
			} else {
				if (fS1 >= -fExtDet1) {
					if (fS1 <= fExtDet1) // region 5 (side)
					{
						fS0 = -this.extent;
						fTmpS1 = -(fA01 * fS0 + fB1);
						if (fTmpS1 < -seg.extent) {
							fS1 = -seg.extent;
							fSqrDist = fS1 * (fS1 - ((double) 2.0) * fTmpS1)
									+ fS0 * (fS0 + ((double) 2.0) * fB0) + fC;
						} else if (fTmpS1 <= seg.extent) {
							fS1 = fTmpS1;
							fSqrDist = -fS1 * fS1 + fS0
									* (fS0 + ((double) 2.0) * fB0) + fC;
						} else {
							fS1 = seg.extent;
							fSqrDist = fS1 * (fS1 - ((double) 2.0) * fTmpS1)
									+ fS0 * (fS0 + ((double) 2.0) * fB0) + fC;
						}
					} else // region 4 (corner)
					{
						fS1 = seg.extent;
						fTmpS0 = -(fA01 * fS1 + fB0);
						if (fTmpS0 > this.extent) {
							fS0 = this.extent;
							fSqrDist = fS0 * (fS0 - ((double) 2.0) * fTmpS0)
									+ fS1 * (fS1 + ((double) 2.0) * fB1) + fC;
						} else if (fTmpS0 >= -this.extent) {
							fS0 = fTmpS0;
							fSqrDist = -fS0 * fS0 + fS1
									* (fS1 + ((double) 2.0) * fB1) + fC;
						} else {
							fS0 = -this.extent;
							fTmpS1 = -(fA01 * fS0 + fB1);
							if (fTmpS1 < -seg.extent) {
								fS1 = -seg.extent;
								fSqrDist = fS1
										* (fS1 - ((double) 2.0) * fTmpS1) + fS0
										* (fS0 + ((double) 2.0) * fB0) + fC;
							} else if (fTmpS1 <= seg.extent) {
								fS1 = fTmpS1;
								fSqrDist = -fS1 * fS1 + fS0
										* (fS0 + ((double) 2.0) * fB0) + fC;
							} else {
								fS1 = seg.extent;
								fSqrDist = fS1
										* (fS1 - ((double) 2.0) * fTmpS1) + fS0
										* (fS0 + ((double) 2.0) * fB0) + fC;
							}
						}
					}
				} else // region 6 (corner)
				{
					fS1 = -seg.extent;
					fTmpS0 = -(fA01 * fS1 + fB0);
					if (fTmpS0 > this.extent) {
						fS0 = this.extent;
						fSqrDist = fS0 * (fS0 - ((double) 2.0) * fTmpS0) + fS1
								* (fS1 + ((double) 2.0) * fB1) + fC;
					} else if (fTmpS0 >= -this.extent) {
						fS0 = fTmpS0;
						fSqrDist = -fS0 * fS0 + fS1
								* (fS1 + ((double) 2.0) * fB1) + fC;
					} else {
						fS0 = -this.extent;
						fTmpS1 = -(fA01 * fS0 + fB1);
						if (fTmpS1 < -seg.extent) {
							fS1 = -seg.extent;
							fSqrDist = fS1 * (fS1 - ((double) 2.0) * fTmpS1)
									+ fS0 * (fS0 + ((double) 2.0) * fB0) + fC;
						} else if (fTmpS1 <= seg.extent) {
							fS1 = fTmpS1;
							fSqrDist = -fS1 * fS1 + fS0
									* (fS0 + ((double) 2.0) * fB0) + fC;
						} else {
							fS1 = seg.extent;
							fSqrDist = fS1 * (fS1 - ((double) 2.0) * fTmpS1)
									+ fS0 * (fS0 + ((double) 2.0) * fB0) + fC;
						}
					}
				}
			}
		} else {
			// The segments are parallel. The average b0 term is designed to
			// ensure symmetry of the function. That is, dist(seg0,seg1) and
			// dist(seg1,seg0) should produce the same number.
			double fE0pE1 = this.extent + seg.extent;
			double fSign = (fA01 > (double) 0.0 ? (double) -1.0 : (double) 1.0);
			double fB0Avr = ((double) 0.5) * (fB0 - fSign * fB1);
			double fLambda = -fB0Avr;
			if (fLambda < -fE0pE1) {
				fLambda = -fE0pE1;
			} else if (fLambda > fE0pE1) {
				fLambda = fE0pE1;
			}

			fS1 = -fSign * fLambda * seg.extent / fE0pE1;
			fS0 = fLambda + fSign * fS1;
			fSqrDist = fLambda * (fLambda + ((double) 2.0) * fB0Avr) + fC;
		}

		Point3f m_kClosestPoint0 = new Point3f();
		Point3f m_kClosestPoint1 = new Point3f();

		Vector3f tmpDir = new Vector3f(this.direction);
		tmpDir.scale((float) fS0);
		m_kClosestPoint0.add(this.origin, tmpDir);
		this.setLastIntersect(m_kClosestPoint1);

		tmpDir = new Vector3f(seg.direction);
		tmpDir.scale((float) fS1);
		m_kClosestPoint1.add(seg.origin, tmpDir);
		seg.setLastIntersect(m_kClosestPoint1);
		return Math.sqrt(Math.abs(fSqrDist));
	}
	/**
	 * Determine if point intersects segment. 
	 * @param test point
	 * @return true if point intersects segment.
	 */
	public boolean intersectsPoint(Point3f test) {
		Vector3f v1 = new Vector3f();
		v1.sub(test, origin);
		Vector3f v2 = (Vector3f) direction.clone();
		float dot = v1.dot(direction);
		if (dot <= KdTriangle.EPS)
			return false;
		v2.scale(dot);
		v1.sub(v2);
		if (v1.length() > KdTriangle.EPS)
			return false;
		if (dot < extent - KdTriangle.EPS)
			return true;
		return false;
	}
	/**
	 * Determine if segment intersects plane.
	 * @param center point on plane
	 * @param kNormal plane normal
	 * @return true if segment intersects plane.
	 */
	public Point3f intersectsPlane(Point3f center, Vector3f kNormal) {
		float A = kNormal.x;
		float B = kNormal.y;
		float C = kNormal.z;
		float numer = A * (center.x - origin.x) + B * (center.y - origin.y) + C
				* (center.z - origin.z);
		float denom = (A * direction.x + B * direction.y + C * direction.z);
		if (denom != 0) {
			float u = numer / denom;
			if (u > KdTriangle.EPS && u < extent - KdTriangle.EPS) {
				Point3f p = (Point3f) origin.clone();
				Vector3f dir = (Vector3f) direction.clone();
				dir.scale(u);
				p.add(dir);
				return p;
			} else {
				return null;
			}
		} else {
			return null;
		}
	}
	/**
	 * Set last intersection point
	 * @param p point
	 */
	public void setLastIntersect(Point3f p) {
		this.lastIntersect = p;
	}
	/**
	 * Get last intersection point
	 * @return point
	 */
	public Point3f getLastIntersect() {
		return this.lastIntersect;
	}

	public String toString() {
		return origin + " " + direction + " " + extent;
	}
}
