package edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.Vector;

import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;

/**
 * A Kd-Tree construction algorithm used for fast line/ray surface intersection
 * testing.
 * 
 * M. Pharr, and G. Humphreys, Physically Based Rendering: From Theory to
 * Implementation: Morgan Kaufmann, 2004.
 * 
 * @author Blake
 * 
 */
public class KdTree extends AbstractCalculation {
	private BBox root;
	private static final double intersectCost = 80;
	private static final double traversalCost = 1;
	private static final double emptyBonus = 0.2;

	/**
	 * Get KD tree root
	 * 
	 * @return
	 */
	public BBox getRoot() {
		return root;
	}

	/**
	 * Get split position for list of children
	 * 
	 * @param children
	 *            children
	 * @param edges
	 *            corresponding edges
	 * @param box
	 *            bounding box
	 * @return position to split children
	 */
	protected int splitPosition(Vector<BBox> children, BBoxEdge[] edges,
			BBox box) {
		int nAbove = 0;
		int nBelow = 0;
		double pAbove = 0;
		double pBelow = 0;
		double cost = 0;
		int dim = 0;
		int bestSplit = -1;
		double sum = 0;
		double prod = 0;
		double voxel;
		double val;
		double minCost;
		int sz;
		BBoxEdge e1 = new BBoxEdge(), e2 = new BBoxEdge();
		Vector3f v = new Vector3f();
		v.sub(box.getMax(), box.getMin());
		if (v.x > v.y && v.x > v.z)
			dim = 0;
		else if (v.y > v.z)
			dim = 1;
		else
			dim = 2;
		int retry = 3;
		voxel = 1.0 / (v.x + v.y + v.z);
		do {
			bestSplit = -1;
			switch (dim) {
			case 0:
				sum = v.y + v.z;
				prod = v.y * v.z;
				break;
			case 1:
				sum = v.x + v.z;
				prod = v.x * v.z;
				break;
			case 2:
				sum = v.x + v.y;
				prod = v.x * v.y;
				break;
			}
			sz = children.size() * 2;
			e1.set(box, dim, true);
			e2.set(box, dim, false);
			for (int i = 0; i < sz; i++) {
				BBox child = children.get(i / 2);
				edges[i].set(child, dim, (i % 2 == 1) ? true : false);
			}
			Arrays.sort(edges, 0, sz);
			nBelow = 0;
			nAbove = children.size();
			// Do not assume prior minimum cost because the spherical
			// orientation
			// is in some sense optimal when rays are cast from outside the
			// sphere.
			// If minimum cost is assumed, then no branches in kd-tree will be
			// generated
			// minCost=intersectCost*nAbove;
			minCost = 1E30;
			for (int i = 0; i < sz; i++) {
				if (!edges[i].isMin())
					nAbove--;
				val = edges[i].getValue();
				if (val > e1.getValue() && e2.getValue() > val) {
					pBelow = (sum * (val - e1.getValue()) + prod) * voxel;
					pAbove = (sum * (e2.getValue() - val) + prod) * voxel;
					cost = traversalCost
							+ intersectCost
							* (1 - ((nAbove == 0 || nBelow == 0) ? emptyBonus
									: 0)) * (pBelow * nBelow + pAbove * nAbove);
					if (cost < minCost) {
						minCost = cost;
						bestSplit = i;
					}
				}
				if (edges[i].isMin())
					nBelow++;
			}
			dim = (dim + 1) % 3;
			retry--;
		} while (bestSplit == -1 && retry > 0);
		return bestSplit;
	}

	/**
	 * Constructor
	 * 
	 * @param parent
	 *            parent calculation
	 * @param triangles
	 *            triangles to insert into tree
	 * @param maxDepth
	 *            maximum tree depth
	 */
	public KdTree(AbstractCalculation parent, ArrayList<KdTriangle> triangles,
			int maxDepth) {
		super(parent);
		setLabel("Kd-Tree");
		root = new BBox();
		for (KdTriangle t : triangles) {
			t.setDepth(1);
			root.children.add(t);
		}
		root.update();
		buildTree(root, maxDepth);
	}

	/**
	 * Build KD tree
	 * 
	 * @param box
	 *            root boundling box
	 * @param maxDepth
	 *            maximum tree depth
	 */
	protected void buildTree(BBox box, int maxDepth) {
		LinkedList<BBox> boxes = new LinkedList<BBox>();
		boxes.add(box);
		int sz;
		int splitPos;
		BBoxEdge[] edges;
		BBox leftChild;
		BBox rightChild;
		int initSz = box.getChildren().size() * 2;
		edges = new BBoxEdge[initSz];
		setTotalUnits(Math.min(maxDepth, (int) Math.floor(Math.log(initSz / 2)
				/ Math.log(2))));
		// System.out.println("TOTAL UNITS "+Math.ceil(Math.log(initSz/2)/Math.log(2)));
		int depthCount = 0;
		for (int i = 0; i < edges.length; i++)
			edges[i] = new BBoxEdge();
		while (!boxes.isEmpty()) {
			box = boxes.remove();
			if (box.getDepth() > maxDepth)
				continue;
			if (box.getDepth() > depthCount) {
				depthCount = box.getDepth();
				incrementCompletedUnits();
			}
			Vector<BBox> children = box.getChildren();
			if (children.size() < 2)
				continue;
			sz = children.size() * 2;
			splitPos = splitPosition(children, edges, box);
			if (splitPos <= 0 || splitPos == sz - 1) {
				continue;
			}
			leftChild = new BBox(box.getDepth() + 1);
			rightChild = new BBox(box.getDepth() + 1);
			Vector<BBox> lchildren = leftChild.getChildren();
			Vector<BBox> rchildren = rightChild.getChildren();

			for (int i = 0; i < splitPos; i++) {
				if (edges[i].isMin()) {
					edges[i].getBox().setDepth(leftChild.getDepth() + 1);
					lchildren.add(edges[i].getBox());
				}
			}
			for (int i = splitPos + 1; i < sz; i++) {
				if (!edges[i].isMin()) {
					edges[i].getBox().setDepth(rightChild.getDepth() + 1);
					rchildren.add(edges[i].getBox());
				}
			}
			if (lchildren.size() >= children.size()
					|| rchildren.size() >= children.size()) {
				continue;
			}
			leftChild.update();
			rightChild.update();
			children.clear();
			children.add(leftChild);
			children.add(rightChild);
			box.update();
			boxes.add(leftChild);
			boxes.add(rightChild);
		}
		markCompleted();
	}

}
