package edu.jhu.ece.iacl.algorithms.graphics.smooth;

import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

import Jama.Matrix;
import Jama.SingularValueDecomposition;

import java.util.*;

import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.SurfaceSelfIntersector;
import edu.jhu.ece.iacl.algorithms.graphics.surf.ProgressiveSurface;
import edu.jhu.ece.iacl.algorithms.graphics.surf.SelfIntersectingPreventionProgressiveSurface;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface.Edge;
/**
 * Smooth and regularize surface using a number of different approaches. See 
 * Y. Ohtake, A. Belyaev, and I. Bogaevski, "Polyhedral surface smoothing with simultaneous mesh regularization." pp. 229-237.
 * for details.
 * 
 * @author Blake Lucas
 *
 */
public class SmoothAndRegularize extends
		SelfIntersectingPreventionProgressiveSurface {
	public static enum Method {
		TAUBIN, LAPLACIAN, WEIGHTED_LAPLACIAN, BILAPALCIAN, MEAN_CURVATURE, SMOOTH_AND_REGULARIZE
	};
	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.3 $");
	}

	public SmoothAndRegularize() {
		super();
		setLabel("Smooth and Regularize Surface");
	}


	public SmoothAndRegularize(AbstractCalculation parent) {
		super(parent);
		setLabel("Smooth and Regularize Surface");
	}

	protected float lambda = 0.5f;
	protected float mu = 1.0f / (1.0f / lambda - 0.1f);
	protected float chi = 0.8f;
	protected float fuzz = 0.2f;
	protected float radialForce = 0;
	protected int updateStep = 10;
	protected int shuffleInterval = 10;
	protected int maxIterations = 100;
	protected static final float ERROR_TOLERANCE = 1E-6f;
	protected double curvThreshold = 10;
	protected int resolutions = 4;
	protected double maxDecimation = 0.5;
	protected boolean preventSelfIntersection = false;
	protected boolean regularizeEdges = false;

	public boolean useShapeMetrics() {
		return shapeMetrics;
	}

	public void setShapeMetrics(boolean shapeMetrics) {
		this.shapeMetrics = shapeMetrics;
	}

	public int getShapeMetricsResolutions() {
		return shapeMetricsResolutions;
	}

	public void setShapeMetricsResolutions(int shapeMetricsResolutions) {
		this.shapeMetricsResolutions = shapeMetricsResolutions;
	}

	protected boolean shapeMetrics = false;
	protected int shapeMetricsResolutions = 5;

	public void setRegularizeEdges(boolean regularize) {
		this.regularizeEdges = regularize;
	}

	public void setPreventSelfIntersection(boolean preventSelfIntersection) {
		this.preventSelfIntersection = preventSelfIntersection;
	}

	protected boolean useLorentzian = true;

	public void setUpdateStep(int updateStep) {
		this.updateStep = updateStep;
	}

	public void setResolutions(int resolutions) {
		this.resolutions = resolutions;
	}

	public void setMaxDecimation(double maxDecimation) {
		this.maxDecimation = maxDecimation;
	}

	public void setUseLorentzian(boolean useLorentzian) {
		this.useLorentzian = useLorentzian;
	}

	public void setCurvatureThreshold(double curvThreshold) {
		this.curvThreshold = curvThreshold;
	}

	protected class HeapElement implements Comparable<HeapElement> {
		double val;
		int vid;

		public HeapElement(int vid, double val) {
			this.vid = vid;
			this.val = val;
		}

		public int compareTo(HeapElement elem) {
			return (int) Math.signum(elem.val - this.val);
		}

	}
	protected VertexMetric decimationMetric=VertexMetric.SURFACE_DISTANCE;
	public void setDecimationMetric(VertexMetric vertexMetric){
		this.decimationMetric=vertexMetric;
	}
	public int[][] getNeighborVertexVertexTable() {
		return neighborVertexVertexTable;
	}

	public void setRadialForce(float speed) {
		this.radialForce = speed;
	}

	public void setFuzziness(float fuzz) {
		this.fuzz = fuzz;
	}
	/**
	 * Smooth surface using specified metric
	 * @param origSurf surface
	 * @param method smoothing metric
	 * @return smoothed surface
	 */
	public EmbeddedSurface smooth(EmbeddedSurface origSurf, Method method) {
		if(shapeMetrics)preventSelfIntersection=false;
		if (preventSelfIntersection) {
			init(origSurf);
			resolutions = 1;
		} else {
			init(origSurf, shapeMetrics);
		}
		surf.setName(origSurf.getName() + "_smooth");
		int vertCount = surf.getVertexCount();
		Point3f p;
		Vector3f v;
		double max_disp = 0;
		double len;
		Point3f center = null;
		double radius = 0;
		double[][] vertData = null;
		int shapeMetricResolution = 0;
		if (shapeMetrics) {
			vertData = new double[vertCount][2*shapeMetricsResolutions];
			resolutions = 1;
		}
		if (radialForce != 0) {
			center = surf.getCenterOfMass();
			radius = Math.sqrt(surf.area() / 4 * Math.PI);
			System.out.println("RADIAL FORCE " + radialForce + " CENTER "
					+ center + " RADIUS " + radius);
		}
		int MR_N = 1; // number of multi-resolution steps
		float MR_step = 2; // difference on the L2 norm of mean curvature
							// between different multiresolution levels
		setTotalUnits(100);
		int Vnum = XUGetVertexTableVN();
		double areaO = 0, value;
		int i;
		int[] Mask = new int[Vnum];
		double hCurv;
		double origVolume = surf.volume();
		Vector3f V;
		double areaN, scale;
		setMetric(decimationMetric);
		setCollapseMethod(CollapseMethod.EDGE);
		if (!preventSelfIntersection && !shapeMetrics) {
			decimate(-1, maxDecimation);
		}

		double targetCurvature;
		if (shapeMetrics) {
			updateShapeMetrics(vertData, 0);
			shapeMetricResolution = 1;
		}
		for (int res = 0; res < resolutions; res++) {
			System.out.println("Resolution: " + (res + 1) + " of "
					+ resolutions);
			if (!preventSelfIntersection && !shapeMetrics) {
				tessellate(maxDecimation / (double) resolutions);
			}
			areaO = surf.area();
			float scaleL = 0;
			if (useLorentzian) {
				scaleL = (float) updateScaleL(Mask, areaO, Vnum);
				System.out.printf("Lorentzian scale factor = %g\n", scaleL);
				value = Math.sqrt(Math.log(1 + 0.5 / (scaleL * scaleL)));
				targetCurvature = curvThreshold * value;
			} else {
				targetCurvature = curvThreshold;
			}
			System.out.println("Target Curvature " + targetCurvature);

			hCurv = meanCurv(Mask, areaO, Vnum, useLorentzian, scaleL);
			double hCurvFirst = -1;
			for (int iter = 0; iter < maxIterations; iter++) {
				for (int vid = 0; vid < vertCount; vid++) {
					if (neighborVertexVertexTable[vid].length == 0)
						continue;
					p = surf.getVertex(vid);
					switch (method) {
					case LAPLACIAN:
						v = getUniformUmbrella(vid);
						v.scale((1 - radialForce) * lambda);
						p.add(v);
						max_disp = Math.max(v.length(), max_disp);
						break;
					case WEIGHTED_LAPLACIAN:
						v = getWeightedUmbrella(vid);
						v.scale((1 - radialForce) * lambda);
						p.add(v);
						max_disp = Math.max(v.length(), max_disp);
						break;
					case TAUBIN:
						v = getUniformUmbrella(vid);
						v.scale(mu - lambda);
						double l1 = v.length();
						p.sub(v);
						v = getBilaplacianUmbrella(vid);
						v.scale(mu * (1 - radialForce) * lambda);
						p.sub(v);
						max_disp = Math.max(v.length() + l1, max_disp);
						break;
					case BILAPALCIAN:
						v = getBilaplacianUmbrella(vid);
						v.scale(-(1 - radialForce) * lambda);
						p.add(v);
						break;
					case MEAN_CURVATURE:
						v = getMeanCurvature(vid);
						len = v.length();
						if (len > 0) {
							v.scale((float) ((1 - radialForce) * (lambda
									* Math.log(len + 1) / len)));
							max_disp = Math.max(max_disp, v.length());
							p.add(v);
						}
						break;
					case SMOOTH_AND_REGULARIZE:
						Vector3f norm = getCalculatedNormal(vid);
						Vector3f laplace = getUniformUmbrella(vid);
						norm.scale(laplace.dot(norm));
						laplace.sub(norm);
						laplace.scale(chi);
						v = getMeanCurvature(vid);
						len = v.length();
						if (len > 0) {
							v.scale((float) (Math.log(len + 1) / len));
						}
						v.add(laplace);
						v.scale(lambda * (1 - radialForce));
						max_disp = Math.max(v.length(), max_disp);
						p.add(v);

						break;
					}
					if (center != null) {
						float vel = (float) (2 * 1.0 / (1 + Math.exp(fuzz
								* (p.distance(center) / radius - 1)))) - 1;
						v = getCalculatedNormal(vid);
						Vector3f r = new Vector3f();
						r.sub(p, center);
						GeometricUtilities.normalize(r);
						v.scale(radialForce * vel * Math.max(0, v.dot(r)));

						p.add(v);
					}
					if (preventSelfIntersection) {
						intersector.updateVertex(vid, p);
					} else {
						surf.setVertex(vid, p);
					}
				}
				if (iter % updateStep == 0) {
					// Rescale vertices to preserve volume
					if (!preventSelfIntersection) {
						scale = Math.pow((origVolume / surf.volume()),
								0.3333333333);
						Point3f centroid = surf.getCenterOfMass();
						for (i = 0; i < Vnum; ++i) {
							V = XUGetVertex(i);
							V.x = (float) ((V.x - centroid.x) * scale + centroid.x);
							V.y = (float) ((V.y - centroid.y) * scale + centroid.y);
							V.z = (float) ((V.z - centroid.z) * scale + centroid.z);
							surf.setVertex(i, V);
						}
					}
					areaN = surf.area();

					/*** curvature calculation ***/
					// Calculate Total Mean Curvature
					hCurv = meanCurv(Mask, areaN, Vnum, useLorentzian, scaleL);
					if (hCurvFirst == -1)
						hCurvFirst = hCurv;
					setCompletedUnits(1 - (hCurv - targetCurvature)
							/ (hCurvFirst - targetCurvature));
					System.out.println("Itearation " + iter
							+ ": Mean Curvature " + hCurv);
					if (hCurv <= targetCurvature)
						break;
					if (shapeMetrics&& shapeMetricResolution/(double)(shapeMetricsResolutions - 1)<1-(hCurv - targetCurvature)/ (hCurvFirst - targetCurvature)) {
						System.out.println("Compute Shape Metrics at Curvature "+hCurv);
						updateShapeMetrics(vertData, shapeMetricResolution);
						shapeMetricResolution++;
					}
				}
			}
		}
		if (shapeMetrics){
			updateShapeMetrics(vertData, shapeMetricsResolutions - 1);
			surf.setVertexData(vertData);
		}
		if (!preventSelfIntersection && !shapeMetrics) {
			tessellate(1.0);
		}
		markCompleted();
		return surf;
	}
	/**
	 * Update shape metrics for data
	 * @param data vertex data
	 * @param index offset into vertex data to store metrics
	 */
	protected void updateShapeMetrics(double[][] data, int index) {
		float[] k1 = new float[data.length];
		float[] k2 = new float[data.length];
		double value = 0;
		PrincipleCurvature(6, k1, k2, data.length);
		for (int i = 0; i < data.length; ++i) {
			data[i][2 * index] = (float) Math.sqrt((k1[i] * k1[i] + k2[i]
					* k2[i]) / 2);
			if (k1[i] != k2[i]) {
				value = 2 * Math.atan((k1[i] + k2[i]) / (k1[i] - k2[i]))
						/ Math.PI;
				data[i][2 * index + 1] = (float) value;
			} else if (k1[i] == 0)
				data[i][2 * index + 1] = -2;
			else if (k1[i] < 0)
				data[i][2 * index + 1] = -1;
			else if (k1[i] > 0)
				data[i][2 * index + 1] = 1;
		}
	}
	/**
	 * Decompose surface into multiple resolutions
	 * @param origSurf original surface
	 * @param method smoothing metric
	 * @return multiple resolution surfaces
	 */
	public EmbeddedSurface[] decompose(EmbeddedSurface origSurf, Method method) {
		if (preventSelfIntersection) {
			init(origSurf);
		} else {
			init(origSurf, false);
		}
		EmbeddedSurface[] multiResSurfs = new EmbeddedSurface[resolutions];
		surf.setName(origSurf.getName() + "_smooth");
		int vertCount = surf.getVertexCount();
		Point3f p;
		Vector3f v;
		double max_disp = 0;
		double len;
		Point3f center = null;
		double radius = 0;

		if (radialForce != 0) {
			center = surf.getCenterOfMass();
			radius = Math.sqrt(surf.area() / 4 * Math.PI);
			System.out.println("RADIAL FORCE " + radialForce + " CENTER "
					+ center + " RADIUS " + radius);
		}
		double[][] vertData = new double[vertCount][1];
		for (int i = 0; i < vertCount; i++) {
			vertData[i][0] = i;
		}
		surf.setVertexData(vertData);
		int MR_N = 1; // number of multi-resolution steps
		float MR_step = 2; // difference on the L2 norm of mean curvature
							// between different multiresolution levels
		setTotalUnits(100);
		int Vnum = XUGetVertexTableVN();
		double areaO = 0, value;
		int i;
		int[] Mask = new int[Vnum];
		double hCurv;
		double origVolume = surf.volume();
		Vector3f V;
		double areaN, scale;
		setMetric(decimationMetric);
		setCollapseMethod(CollapseMethod.EDGE);
		double targetCurvature;
		for (int res = 0; res < resolutions; res++) {
			System.out.println("Resolution: " + (res + 1) + " of "
					+ resolutions);
			decimate(-1, maxDecimation * (res + 1) / resolutions);
			if (regularizeEdges)
				regularize(vertStack);
			areaO = surf.area();
			float scaleL = 0;
			if (useLorentzian) {
				scaleL = (float) updateScaleL(Mask, areaO, Vnum);
				System.out.printf("Lorentzian scale factor = %g\n", scaleL);
				value = Math.sqrt(Math.log(1 + 0.5 / (scaleL * scaleL)));
				targetCurvature = curvThreshold * value;
			} else {
				targetCurvature = curvThreshold;
			}
			System.out.println("Target Curvature " + targetCurvature);

			hCurv = meanCurv(Mask, areaO, Vnum, useLorentzian, scaleL);
			double hCurvFirst = -1;
			for (int iter = 0; iter < maxIterations; iter++) {
				for (int vid = 0; vid < vertCount; vid++) {
					if (neighborVertexVertexTable[vid].length == 0)
						continue;
					p = surf.getVertex(vid);
					switch (method) {
					case LAPLACIAN:
						v = getUniformUmbrella(vid);
						v.scale((1 - radialForce) * lambda);
						p.add(v);
						max_disp = Math.max(v.length(), max_disp);
						break;
					case WEIGHTED_LAPLACIAN:
						v = getWeightedUmbrella(vid);
						v.scale((1 - radialForce) * lambda);
						p.add(v);
						max_disp = Math.max(v.length(), max_disp);
						break;
					case TAUBIN:
						v = getUniformUmbrella(vid);
						v.scale(mu - lambda);
						double l1 = v.length();
						p.sub(v);
						v = getBilaplacianUmbrella(vid);
						v.scale(mu * (1 - radialForce) * lambda);
						p.sub(v);
						max_disp = Math.max(v.length() + l1, max_disp);
						break;
					case BILAPALCIAN:
						v = getBilaplacianUmbrella(vid);
						v.scale(-(1 - radialForce) * lambda);
						p.add(v);
						break;
					case MEAN_CURVATURE:
						v = getMeanCurvature(vid);
						len = v.length();
						if (len > 0) {
							v.scale((float) ((1 - radialForce) * (lambda
									* Math.log(len + 1) / len)));
							max_disp = Math.max(max_disp, v.length());
							p.add(v);
						}
						break;
					case SMOOTH_AND_REGULARIZE:
						Vector3f norm = getCalculatedNormal(vid);
						Vector3f laplace = getUniformUmbrella(vid);
						norm.scale(laplace.dot(norm));
						laplace.sub(norm);
						laplace.scale(chi);
						v = getMeanCurvature(vid);
						len = v.length();
						if (len > 0) {
							v.scale((float) (Math.log(len + 1) / len));
						}
						v.add(laplace);
						v.scale(lambda * (1 - radialForce));
						max_disp = Math.max(v.length(), max_disp);
						p.add(v);

						break;
					}
					if (center != null) {
						float vel = (float) (2 * 1.0 / (1 + Math.exp(fuzz
								* (p.distance(center) / radius - 1)))) - 1;
						v = getCalculatedNormal(vid);
						Vector3f r = new Vector3f();
						r.sub(p, center);
						GeometricUtilities.normalize(r);
						v.scale(radialForce * vel * Math.max(0, v.dot(r)));

						p.add(v);
					}
					if (preventSelfIntersection) {
						intersector.updateVertex(vid, p);
					} else {
						surf.setVertex(vid, p);
					}
				}
				if (iter % updateStep == 0) {
					// Rescale vertices to preserve volume
					if (!preventSelfIntersection) {
						scale = Math.pow((origVolume / surf.volume()),
								0.3333333333);
						Point3f centroid = surf.getCenterOfMass();
						for (i = 0; i < Vnum; ++i) {
							V = XUGetVertex(i);
							V.x = (float) ((V.x - centroid.x) * scale + centroid.x);
							V.y = (float) ((V.y - centroid.y) * scale + centroid.y);
							V.z = (float) ((V.z - centroid.z) * scale + centroid.z);
							surf.setVertex(i, V);
						}
					}
					areaN = surf.area();

					/*** curvature calculation ***/
					// Calculate Total Mean Curvature
					hCurv = meanCurv(Mask, areaN, Vnum, useLorentzian, scaleL);
					if (hCurvFirst == -1)
						hCurvFirst = hCurv;
					setCompletedUnits(1 - (hCurv - targetCurvature)
							/ (hCurvFirst - targetCurvature));
					System.out.println("Itearation " + iter
							+ ": Mean Curvature " + hCurv);
					if (hCurv <= targetCurvature)
						break;
				}
			}
			multiResSurfs[resolutions - 1 - res] = createSurface();
			multiResSurfs[resolutions - 1 - res].setName(surf.getName()
					+ "_res" + (resolutions - res));
		}
		markCompleted();
		return multiResSurfs;
	}

	protected class Vertex {
		Vector3f v;
		int N;
		int[] nbd;
	}
	/**
	 * Get mean curvature normal
	 * @param id vertex id
	 * @return
	 */
	protected double getMeanCurvatureNorm(int id) {
		return getMeanCurvature(id).length();
	}
	/**
	 * Get unified umbrella vector
	 * @param id vertex id
	 * @return umbrella vector
	 */
	protected Vector3f getUniformUmbrella(int id) {
		Point3f pivot = surf.getVertex(id);
		int len = neighborVertexVertexTable[id].length;
		Vector3f umb = new Vector3f();
		for (int i = 0; i < len; i++) {
			umb.add(surf.getVertex(neighborVertexVertexTable[id][i]));
		}
		umb.scale(1.0f / len);
		umb.sub(pivot);
		return umb;
	}
	/**
	 * Get weighted umbrella vector
	 * @param id vertex id
	 * @return weighted umbrella vector
	 */
	protected Vector3f getWeightedUmbrella(int id) {
		Point3f pivot = surf.getVertex(id);
		int[] nbrs = neighborVertexVertexTable[id];
		int len = nbrs.length;
		Vector3f umb = new Vector3f();
		Point3f last = surf.getVertex(nbrs[len - 1]);
		double areaTotal = 0;
		double area;
		for (int i = 0; i < len; i++) {
			Point3f cur = surf.getVertex(nbrs[i]);
			area = GeometricUtilities.triangleArea(pivot, cur, last);
			Point3f center = GeometricUtilities
					.triangleCenter(pivot, cur, last);
			last = cur;
			center.scale((float) area);
			umb.add(center);
			areaTotal += area;
		}
		if (areaTotal > 1E-10) {
			umb.scale(1.0f / (float) areaTotal);
			umb.sub(pivot);
		} else {
			umb = new Vector3f();
		}
		return umb;
	}
	/**
	 * Get centroid for 1-ring kernel
	 * @param id vertex id
	 * @return
	 */
	protected Point3f getCentroid(int id) {
		int len = neighborVertexVertexTable[id].length;
		Point3f umb = new Point3f();
		for (int i = 0; i < len; i++) {
			umb.add(surf.getVertex(neighborVertexVertexTable[id][i]));
		}
		umb.scale(1.0f / len);
		return umb;
	}
	/**
	 * Get bilaplacian umbrella 
	 * @param id vertex id
	 * @return vector
	 */
	protected Vector3f getBilaplacianUmbrella(int id) {
		int len = neighborVertexVertexTable[id].length;
		Vector3f umb = new Vector3f();
		for (int i = 0; i < len; i++) {
			umb.add(getUniformUmbrella(neighborVertexVertexTable[id][i]));
		}
		umb.scale(1.0f / len);
		umb.sub(getUniformUmbrella(id));
		return umb;
	}
	/**
	 * Get inverse distance umbrella operator
	 * @param id vertex id
	 * @return
	 */
	protected Vector3f getInverseDistUmbrellaOperator(int id) {
		Point3f pivot = surf.getVertex(id);
		int len = neighborVertexVertexTable[id].length;
		Vector3f umb = new Vector3f();
		Point3f p;
		float weight, d, sumw = 0;
		for (int i = 0; i < len; i++) {
			p = surf.getVertex(neighborVertexVertexTable[id][i]);
			d = pivot.distance(p);
			if (d == 0)
				continue;
			weight = 1 / d;
			sumw += weight;
			p.scale(weight);
			umb.add(p);
		}
		umb.scale(1.0f / sumw);
		umb.sub(pivot);
		return umb;
	}
	/**
	 * Get calculated normal
	 */
	protected Vector3f getCalculatedNormal(int id) {
		Vector3f norm = new Vector3f();
		Vector3f edge1 = new Vector3f();
		Vector3f edge2 = new Vector3f();
		Vector3f cross = new Vector3f();
		Point3f pivot = surf.getVertex(id);
		int len = neighborVertexVertexTable[id].length;
		for (int i = 0; i < len; i++) {
			edge1.sub(surf.getVertex(neighborVertexVertexTable[id][i]), pivot);
			edge2.sub(surf.getVertex(neighborVertexVertexTable[id][(i + 1)
					% len]), pivot);
			cross.cross(edge1, edge2);
			norm.add(cross);
		}
		norm.normalize();
		return norm;
	}

	public float getChi() {
		return chi;
	}

	public void setChi(float chi) {
		this.chi = chi;
	}

	public float getLambda() {
		return lambda;
	}

	public void setLambda(float lambda) {
		this.lambda = lambda;
		// this is a semi-optimal choice of mu according to the paper
		this.mu = 1.0f / (1.0f / lambda - 0.1f);
	}

	public int getMaxIterations() {
		return maxIterations;
	}

	public void setMaxIterations(int maxIterations) {
		this.maxIterations = maxIterations;
	}

	public int getShuffleInterval() {
		return shuffleInterval;
	}

	public void setShuffleInterval(int shuffleInterval) {
		this.shuffleInterval = shuffleInterval;
	}

	/**
	 * Get mean curvature
	 * @param mask 
	 * @param area
	 * @param VN
	 * @param useLorentzian
	 * @param scaleL
	 * @return
	 */
	protected double meanCurv(int[] mask, double area, int VN,
			boolean useLorentzian, float scaleL) {
		int i, j;
		Vector3f v0, v1, v2;
		Vector3f v, w, u;
		int vid, EN;
		double curv;
		Vector3f Curv = new Vector3f();
		double alpha, temp = 0, Narea;
		double kcurv;

		curv = 0;
		for (i = 0; i < VN; ++i) {
			if (mask[i] == 0) {
				v0 = XUGetVertex(i);
				Curv.x = 0;
				Curv.y = 0;
				Curv.z = 0;
				EN = XUGetNeighborEN(i);
				if (EN == 0)
					continue;
				vid = XUGetVertexNeighborVId(i, EN - 1);
				v1 = XUGetVertex(vid);
				Narea = 0.0;
				for (j = 0; j < EN; ++j) {
					vid = XUGetVertexNeighborVId(i, j);
					v2 = XUGetVertex(vid);
					Narea += GeometricUtilities.triangleArea(v0, v1, v2);
					v = XUVecSub(v0, v2);
					w = XUVecSub(v1, v2);
					alpha = Math.acos(Math.max(-1, Math.min(1, XUVecDot(v, w)
							/ (XUVecMag(v) * XUVecMag(w)))));
					if (alpha != 0) {
						temp = 1.0f / Math.tan(alpha);
					} else {
						temp = 0;
						System.err.println("Alpha is " + alpha + " " + v + " "
								+ w);
					}
					vid = XUGetVertexNeighborVId(i, ((EN + j - 2) % EN));
					v2 = XUGetVertex(vid);

					v = XUVecSub(v0, v2);
					w = XUVecSub(v1, v2);
					alpha = Math.acos(Math.max(-1, Math.min(1, XUVecDot(v, w)
							/ (XUVecMag(v) * XUVecMag(w)))));
					if (alpha != 0) {
						temp += 1 / Math.tan(alpha);
					} else {
						System.err.println("Alpha is " + alpha + " " + v + " "
								+ w);
					}
					if (Double.isNaN(temp) || Double.isInfinite(temp)) {
						temp = 0;
					}
					v = XUVecSub(v0, v1);

					Curv.x += v.x * temp * 0.5f;
					Curv.y += v.y * temp * 0.5f;
					Curv.z += v.z * temp * 0.5f;
					vid = XUGetVertexNeighborVId(i, j);
					v1 = XUGetVertex(vid);
				}
				Narea /= 3.0;
				kcurv = (Narea > 0) ? Curv.length() / Narea : 0;
				if (useLorentzian)
					curv += Math.log(1 + 0.5 * kcurv * kcurv
							/ (scaleL * scaleL))
							* Narea;
				else
					curv += kcurv * kcurv * Narea;
			}
		}
		// curv /= area;
		curv *= curvNorm;
		curv = Math.sqrt(curv);
		return (curv);
	}

	private static final float curvNorm = (float) (1.0f / (4 * Math.acos(-1)));

	protected double updateScaleL(int[] mask, double area, int VN) {
		int i, j;
		Vector3f v0, v1, v2;
		Vector3f v, w;
		int vid, EN;
		Vector3f Curv = new Vector3f();
		double alpha, temp, Narea;
		float scaleL = 0;
		float median;
		int nonMask = 0;
		float[] kcurv = new float[VN];
		;
		for (i = 0; i < VN; ++i) {
			if (mask[i] == 0) {
				v0 = XUGetVertex(i);
				Curv.x = 0;
				Curv.y = 0;
				Curv.z = 0;
				EN = XUGetNeighborEN(i);
				if (EN == 0)
					continue;
				vid = XUGetVertexNeighborVId(i, EN - 1);
				v1 = XUGetVertex(vid);
				Narea = 0.0;
				for (j = 0; j < EN; ++j) {
					vid = XUGetVertexNeighborVId(i, j);
					v2 = XUGetVertex(vid);
					Narea += GeometricUtilities.triangleArea(v0, v1, v2);
					v = XUVecSub(v0, v2);
					w = XUVecSub(v1, v2);
					alpha = Math.acos(XUVecDot(v, w)
							/ (XUVecMag(v) * XUVecMag(w)));
					temp = 1 / Math.tan(alpha);
					vid = XUGetVertexNeighborVId(i, ((EN + j - 2) % EN));
					v2 = XUGetVertex(vid);
					v = XUVecSub(v0, v2);
					w = XUVecSub(v1, v2);
					alpha = Math.acos(XUVecDot(v, w)
							/ (XUVecMag(v) * XUVecMag(w)));
					if (alpha != 0) {
						temp += 1 / Math.tan(alpha);
					}
					if (Double.isNaN(temp) || Double.isInfinite(temp)) {
						temp = 0;
					}
					v = XUVecSub(v0, v1);
					Curv.x += v.x * temp / 2;
					Curv.y += v.y * temp / 2;
					Curv.z += v.z * temp / 2;
					vid = XUGetVertexNeighborVId(i, j);
					v1 = XUGetVertex(vid);
				}
				Narea /= 3.0f;
				kcurv[nonMask++] = (Narea > 0) ? (float) (Curv.length() / Narea)
						: 0;
			}
		}
		hpsort(nonMask, kcurv);
		median = (kcurv[(int) Math.floor(nonMask / 2)] + kcurv[(int) Math
				.ceil(nonMask / 2)]) / 2;
		// printf("median = %f ---> ",median);
		for (i = 0; i < nonMask; ++i)
			kcurv[i] = Math.abs(kcurv[i] - median);
		hpsort(nonMask, kcurv);
		median = (kcurv[(int) Math.floor(nonMask / 2)] + kcurv[(int) Math
				.ceil(nonMask / 2)]) / 2;
		// printf("%f\n",median);
		// if (median > 1)
		// Magic number for Median Absolute Deviation MAD
		scaleL = MedianAbsoluteDeviation * median;
		// else
		// scaleL = 1.4826;
		// printf("scaleL = %f\n",scaleL);
		return (scaleL);
	}

	public static final float MedianAbsoluteDeviation = 1.4826f;

	void hpsort(int n, float[] r) {
		int i, ir, j, l;
		float rra;
		int k;

		float[] ra;
		int[] x;
		int xx;

		x = new int[n + 1];
		ra = new float[n + 1];
		for (k = 1; k <= n; ++k) {
			x[k] = k - 1;
			ra[k] = r[k - 1];
		}

		if (n < 2) {
			System.out.printf("size of the array is < 2! \n");
			System.exit(1);
		}
		l = (n >> 1) + 1;
		ir = n;

		/*
		 * * The index l will be decremented from its initial value down to 1
		 * during the "hiring"* (heap creation) phase. Once it reaches 1, the
		 * index ir will be decremented from its* initial value down to 1 during
		 * the "retirement-and-promotion" (heap selection) phase.
		 */

		for (;;) {
			if (l > 1) { /* still in hiring phase */
				k = --l;
				xx = x[k];
				rra = ra[k];
			} else { /* In retirment-and-promotion phase. */
				rra = ra[ir]; /* Clear a space at the end of array. */
				xx = x[ir];
				ra[ir] = ra[1]; /* Retire the top of the array. */
				x[ir] = x[1];
				if (--ir == 0) { /* Done with the last promotion. */
					ra[1] = rra; /* The last competent worker of all! */
					break;
				}
			}
			i = l; /* Whether in hiring phase or promotion phase, */
			j = l + l; /*
						 * here set up to shift down element rra to its proper
						 * level.
						 */
			while (j <= ir) {
				if (j < ir && ra[j] < ra[j + 1])
					j++; /* Compare to the better underling. */
				if (rra < ra[j]) { /* Demote rra. */
					ra[i] = ra[j];
					x[i] = x[j];
					i = j;
					j <<= 1;
				} else
					break; /* Found rra's level. Terminate the sift-down. */
			}
			ra[i] = rra; /* Put rra into slot. */
			x[i] = xx;
		}

		for (k = 0; k < n; ++k) {
			r[k] = ra[k + 1];
		}
	}

	int NBR_MAX_NUM = 5000;

	protected void PrincipleCurvature(int NBRLayer, float[] k1, float[] k2,
			int Vnum) {

		int i, j, k, tenthVnum;
		int[] nbrVtxNum = new int[1];
		int[] nbrPolyNum = new int[1];
		Vector3f dummy = new Vector3f();
		float[] curv = new float[3];
		float[] h = new float[5];
		float error;
		int[] nbrVtxID = new int[NBR_MAX_NUM];
		int[] nbrPolyID = new int[NBR_MAX_NUM];
		Vector3f[] vtx = new Vector3f[NBR_MAX_NUM];
		Vector3f n = new Vector3f();
		for (i = 0; i < vtx.length; i++) {
			vtx[i] = new Vector3f();
		}
		tenthVnum = Vnum / 10;
		/*
		 * Calculating the differential geometric data and store them in dx
		 * format in file whose name is specified by argv[2]
		 */
		for (k = 0; k < Vnum; k++) {
			GetNbr(k, nbrVtxID, nbrPolyID, nbrVtxNum, nbrPolyNum, NBRLayer);
			for (j = 0; j < nbrVtxNum[0]; j++) {
				dummy = XUGetVertex(nbrVtxID[j]);
				vtx[j].x = dummy.x;
				vtx[j].y = dummy.y;
				vtx[j].z = dummy.z;
			}
			findnorm(n, nbrPolyID, nbrPolyNum[0]);
			error = lsqsfit(vtx, nbrVtxNum[0], vtx[0], n, h, 5);
			diffgeom(h, curv);
			k1[k] = curv[0];
			k2[k] = curv[1];
		}
	}

	void diffgeom(float[] h, float[] k) {
		float K; /* Gaussian curvatures */
		float H; /* Mean curvatures */
		float h1, h2, h11, h12, h22;
		float temp;
		float kk1, kk2;
		float TOL = 1.0e-6f;

		h1 = h[0];
		h2 = h[1];
		h11 = h[2];
		h12 = h[3];
		h22 = h[4];

		temp = 1.0f + h1 * h1 + h2 * h2;
		k[2] = temp;

		K = (h11 * h22 - h12 * h12) / (temp * temp);
		H = (float) (((1.0f + h1 * h1) * h22 + (1.0f + h2 * h2) * h11 - 2.0f
				* h1 * h2 * h12) / (2.0f * Math.sqrt(temp) * temp));

		temp = (H * H - K);
		if (temp < TOL)
			temp = 0;

		kk1 = (float) (H + Math.sqrt(temp));
		kk2 = (float) (H - Math.sqrt(temp));

		k[0] = (float) Math.max(kk1, kk2);
		k[1] = (float) Math.min(kk1, kk2);
	}

	void findnorm(Vector3f n, int[] p, int nbrPolyNum) {
		Vector3f[] tri = new Vector3f[3];
		Vector3f dummy;
		Vector3f v1 = new Vector3f(), v2 = new Vector3f(), v = new Vector3f();
		double meg;
		int k, j;
		int M, K, n1, n2, n0;

		n.x = 0.0f;
		n.y = 0.0f;
		n.z = 0.0f;

		M = nbrPolyNum;

		for (k = 0; k < M; k++) {

			/*
			 * find the right order of the vertics which form polygon with ID
			 * p[k]
			 */
			if (p[k] == edges[faces[p[k]].edges[0].id].f1) {
				n0 = edges[faces[p[k]].edges[0].id].v1;
				n1 = edges[faces[p[k]].edges[0].id].v2;
			} else {
				n0 = edges[faces[p[k]].edges[0].id].v2;
				n1 = edges[faces[p[k]].edges[0].id].v1;
			}

			n2 = edges[faces[p[k]].edges[1].id].v1;
			if (n2 == n0 || n2 == n1)
				n2 = edges[faces[p[k]].edges[1].id].v2;

			tri[0] = XUGetVertex(n0);
			tri[1] = XUGetVertex(n1);
			tri[2] = XUGetVertex(n2);

			v1.x = tri[1].x - tri[0].x;
			v1.y = tri[1].y - tri[0].y;
			v1.z = tri[1].z - tri[0].z;
			v2.x = tri[2].x - tri[1].x;
			v2.y = tri[2].y - tri[1].y;
			v2.z = tri[2].z - tri[1].z;

			v.x = v1.y * v2.z - v1.z * v2.y;
			v.y = v1.z * v2.x - v1.x * v2.z;
			v.z = v1.x * v2.y - v1.y * v2.x;

			meg = Math.sqrt(v.x * v.x + v.y * v.y + v.z * v.z);
			if (meg > 1.0e-10) {
				n.x += v.x / meg;
				n.y += v.y / meg;
				n.z += v.z / meg;
			}
		}
		meg = Math.sqrt(n.x * n.x + n.y * n.y + n.z * n.z);
		n.x /= meg;
		n.y /= meg;
		n.z /= meg;
	}

	public void svdcmp(float[][] a, float[] w, float[][] v) {
		Matrix A = new Matrix(a.length, a[0].length);
		for (int i = 0; i < A.getRowDimension(); i++) {
			for (int j = 0; j < A.getColumnDimension(); j++) {
				A.set(i, j, a[i][j]);
			}
		}
		SingularValueDecomposition svd = new SingularValueDecomposition(A);
		double[] vals = svd.getSingularValues();
		for (int i = 0; i < vals.length; i++) {
			w[i] = (float) vals[i];
		}
		Matrix leftVecs = svd.getU();
		Matrix rightVecs = svd.getV();
		for (int i = 0; i < leftVecs.getRowDimension(); i++) {
			for (int j = 0; j < leftVecs.getColumnDimension(); j++) {
				a[i][j] = (float) leftVecs.get(i, j);
			}
		}
		for (int i = 0; i < rightVecs.getRowDimension(); i++) {
			for (int j = 0; j < rightVecs.getColumnDimension(); j++) {
				v[i][j] = (float) rightVecs.get(i, j);
			}
		}
	}

	void svbksb(float[][] u, float w[], float[][] v, int m, int n, float b[],
			float x[]) {
		int jj, j, i;
		float s;
		float[] tmp = new float[n];

		for (j = 0; j < n; j++) {
			s = 0.0f;
			if (w[j] != 0) {
				for (i = 0; i < m; i++)
					s += u[i][j] * b[i];
				s /= w[j];
			}
			tmp[j] = s;
		}
		for (j = 0; j < n; j++) {
			s = 0.0f;
			for (jj = 0; jj < n; jj++)
				s += v[j][jj] * tmp[jj];
			x[j] = s;
		}
	}

	protected float lsqsfit(Vector3f[] v, int vn, Vector3f o, Vector3f n,
			float[] h, int ma) {
		/* Vectornv; */
		Vector3f[] nv;
		float vx, vy, vz;
		float a, b, c, d;
		float[][] U;
		float[][] V;
		float[] W; /* workspaces for the singular value decompostion */
		float[] w; /* the data vector */
		int i;
		float err;
		float Wmax, thresh;
		float TOL = 1.0e-5f;

		err = 0.0f;
		/* allocate memories */
		/* nv = (Vector)fvector3d(1,vn); */
		nv = new Vector3f[vn];
		U = new float[vn][ma];
		V = new float[ma][ma];
		W = new float[ma];
		w = new float[vn];

		/* translate v to the new coordinate system with origin o */
		for (i = 0; i < vn; i++) {
			nv[i] = new Vector3f();
			nv[i].x = v[i].x - o.x;
			nv[i].y = v[i].y - o.y;
			nv[i].z = v[i].z - o.z;
		}

		/*
		 * rotate the translated v such that n is in the z direction at the new
		 * coordinate system
		 * 
		 * Let a = n.x, b = n.y, c = n.z
		 * 
		 * rotation around the x directions and bring the nv to the xz plane / 1
		 * 0 0 \ Rx = | 0 c/d -b/d | where d = sqrt(bb+cc) \ 0 b/d c/d /
		 * 
		 * rotation around the y direction and bring the nv to the z axis / d 0
		 * -a \ Ry = | 0 1 0 | \ a 0 d /
		 * 
		 * the composite rotation / d -a/d b -a/d c \ Rc = RyRx = | 0 c/d -b/d |
		 * \ a b c / therefore
		 * 
		 * v' = Rcv = / dvx - a/d (bvy + cvz) \ | (cvy - bvz)/d | \ avx + bvy +
		 * cvz /
		 */
		a = n.x;
		b = n.y;
		c = n.z;
		d = (float) Math.sqrt(b * b + c * c);

		/*
		 * if c==1 , then the vector n is already in the z direction, so no
		 * rotation should be applied if d==0, then n is in the x direction, no
		 * rotation along x is necessary hence Rc = Ry =[[0 0 -1], [0 1 0], [1 0
		 * 0]]
		 */
		if (c != 1)
			if (d == 0)
				for (i = 0; i < vn; i++) {
					vx = nv[i].x;
					nv[i].x = -nv[i].z;
					/* nv[i].y unchanged */
					nv[i].z = vx;
					;
				}
			else
				for (i = 0; i < vn; i++) {
					vx = nv[i].x;
					vy = nv[i].y;
					vz = nv[i].z;
					nv[i].x = d * vx - a / d * (b * vy + c * vz);
					nv[i].y = (c * vy - b * vz) / d;
					nv[i].z = a * vx + b * vy + c * vz;
				}

		/* constructing the data matrix and vectors */

		/* constructing the data matrix */
		for (i = 0; i < vn; i++) {
			vx = nv[i].x;
			vy = nv[i].y;
			vz = nv[i].z;

			U[i][0] = vx;
			U[i][1] = vy;
			U[i][2] = vx * vx / 2.0f;
			U[i][3] = vx * vy;
			U[i][4] = vy * vy / 2.0f;

			w[i] = vz;
		}

		svdcmp(U, W, V); /* singular value decomposition */

		Wmax = 0.0f; /* Edit the singular values, given TOL */
		for (i = 0; i < ma; i++)
			if (Math.abs(W[i]) > Wmax)
				Wmax = Math.abs(W[i]);
		thresh = TOL * Wmax;
		for (i = 0; i < ma; i++)
			if (Math.abs(W[i]) < thresh)
				W[i] = 0.0f;

		svbksb(U, W, V, vn, ma, w, h); /* solve for parameter vector */

		for (i = 0; i < vn; i++)
			err += Math.abs(nv[i].z - h[0] * nv[i].x - h[1] * nv[i].y - h[2]
					* nv[i].x * nv[i].x / 2.0 - h[3] * nv[i].x * nv[i].y - h[4]
					* nv[i].y * nv[i].y / 2.0);

		err = err / (float) vn;

		/* frees the memory allocated for the data */

		return err;
	}

	/*
	 * This routine is used to find the nearest Layer layers of neighbor of
	 * vertex CurrentVtxID. The neighboring vertex IDs are stored in VtxID and
	 * the neighboring polygon IDs are in PolyID. The VtxID is not ordered
	 * except the immidiate neighbors. Neither is PolyID.
	 */
	public void GetNbr(int CurrentVtxID, int[] VtxID, int[] PolyID, int[] VNum,
			int[] PNum, int Layer) {

		int i, j, k;
		int Vtx0, Poly0;
		int VtxNum;
		int VtxAdded;
		int v0, v1, p0, p1;
		Edge[] nb;

		Vtx0 = 0;
		Poly0 = 0;
		VtxAdded = 0;
		VtxID[Vtx0] = CurrentVtxID;
		VtxNum = 1;
		for (i = 0; i < Layer; i++) {
			VtxAdded = 0;
			for (j = Vtx0; j < VtxNum; j++) {
				nb = XUGetNeighbor(VtxID[j]);
				for (k = 0; k < nb.length; k++) {
					v0 = edges[nb[k].id].v1;
					v1 = edges[nb[k].id].v2;
					p0 = edges[nb[k].id].f1;
					p1 = edges[nb[k].id].f2;

					if (!IsInSet(VtxID, VtxNum + VtxAdded, v0)) {
						VtxID[VtxNum + VtxAdded] = v0;
						VtxAdded++;
					}
					if (!IsInSet(VtxID, VtxNum + VtxAdded, v1)) {
						VtxID[VtxNum + VtxAdded] = v1;
						VtxAdded++;
					}

					if (!IsInSet(PolyID, Poly0, p0))
						PolyID[Poly0++] = p0;

					if (!IsInSet(PolyID, Poly0, p1))
						PolyID[Poly0++] = p1;

				}
			}
			Vtx0 = VtxNum;
			VtxNum += VtxAdded;
		}
		VNum[0] = VtxNum;
		PNum[0] = Poly0;
		return;
	}

	protected boolean IsInSet(int[] set, int N, int ID) {
		int k;

		if (N == 0)
			return false;

		for (k = 0; k < N; k++) {
			if (ID == set[k])
				return true;
		}
		return false;
	}

	protected static float SIGN(float a, float b) {
		return ((b) >= 0.0 ? Math.abs(a) : -Math.abs(a));
	}

	static float at, bt, ct;

	protected static float PYTHAG(float a, float b) {
		if ((at = Math.abs(a)) > (bt = Math.abs(b))) {
			ct = bt / at;
			return (float) (at * Math.sqrt(1.0 + ct * ct));
		} else {
			if (bt != 0) {
				ct = at / bt;
				return (float) (bt * Math.sqrt(1.0 + ct * ct));
			} else
				return 0;
		}
	}

}
