package edu.jhu.ece.iacl.algorithms.graphics.surf;

import java.util.ArrayList;

import javax.vecmath.Point3f;

import edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree.KdPoint3;
import edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree.KdTriangle;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;

/**
 * Hierarchical surface allows insertion of vertices into the mesh. Each
 * insertion is treated as adding a node to a tree with its root at each
 * triangle face. A surface containing existing and inserted vertices can then
 * be reconstructed from the hierarchical representation.
 * 
 * @author Blake Lucas
 * 
 */
public class HierarchicalSurface {
	private int vertCount, faceCount;
	private EmbeddedSurface surf;

	public HierarchicalSurface(EmbeddedSurface surf) {
		this.surf = surf;
		init();

	}

	NestedTriangle[] tris;
	ArrayList<VertexData> newPoints;
	/**
	 * Initialize surface
	 */
	protected void init() {
		newPoints = new ArrayList<VertexData>();
		vertCount = surf.getVertexCount();
		faceCount = surf.getFaceCount();
		tris = new NestedTriangle[faceCount];
		for (int id = 0; id < faceCount * 3; id += 3) {
			tris[id / 3] = new NestedTriangle(surf.getCoordinateIndex(id), surf
					.getCoordinateIndex(id + 1), surf
					.getCoordinateIndex(id + 2));
		}
	}
	/**
	 * Insert point into triangle face
	 * @param p point
	 * @param faceId face id
	 */
	public void insert(Point3f p, int faceId) {
		newPoints.add(new VertexData(p, null));
		int vid = vertCount++;
		tris[faceId].insert(p, vid);
	}
	/**
	 * Insert point into triangle face 
	 * @param p point
	 * @param data associated vertex data
	 * @param faceId face id
	 */
	public void insert(Point3f p, double[] data, int faceId) {
		newPoints.add(new VertexData(p, data));
		int vid = vertCount++;
		tris[faceId].insert(p, vid);
	}
	/**
	 * Construct surface from hierarchial representation
	 * @param sourceFirst list vertices from original surface first
	 * @return constructed surface
	 */
	public EmbeddedSurface getSurface(boolean sourceFirst) {
		Point3f[] pts = new Point3f[vertCount];
		double[][] oldData = surf.getVertexData();
		if (oldData != null) {
			double[][] data = new double[vertCount][oldData[0].length];
			int indexes[] = new int[faceCount * 3];
			int oldVertCount = surf.getVertexCount();
			if (sourceFirst) {
				for (int id = 0; id < vertCount; id++) {
					if (id < oldVertCount) {
						pts[id] = surf.getVertex(id);
						data[id] = surf.getVertexData(id);
					} else {
						VertexData vd = newPoints.get(id - oldVertCount);
						pts[id] = vd.p;
					}
				}
			} else {
				for (int id = 0; id < vertCount; id++) {
					if (id < oldVertCount) {
						pts[id + (vertCount - oldVertCount)] = surf
								.getVertex(id);
						data[id + (vertCount - oldVertCount)] = surf
								.getVertexData(id);
					} else {
						VertexData vd = newPoints.get(id - oldVertCount);
						pts[id - oldVertCount] = vd.p;
						data[id - oldVertCount] = vd.data;
					}
				}
			}
			int[] in = new int[] { 0 };
			for (NestedTriangle tri : tris) {
				tri.addTriangles(indexes, in);
			}
			if (!sourceFirst) {
				for (int i = 0; i < indexes.length; i++) {
					if (indexes[i] >= oldVertCount) {
						indexes[i] -= oldVertCount;
					} else {
						indexes[i] += (vertCount - oldVertCount);
					}
				}
			}
			return new EmbeddedSurface(pts, indexes, data);
		} else {
			int indexes[] = new int[faceCount * 3];
			int oldVertCount = surf.getVertexCount();
			if (sourceFirst) {
				for (int id = 0; id < vertCount; id++) {
					if (id < oldVertCount) {
						pts[id] = surf.getVertex(id);
					} else {
						VertexData vd = newPoints.get(id - oldVertCount);
						pts[id] = vd.p;
					}
				}
			} else {
				for (int id = 0; id < vertCount; id++) {
					if (id < oldVertCount) {
						pts[id + (vertCount - oldVertCount)] = surf
								.getVertex(id);
					} else {
						VertexData vd = newPoints.get(id - oldVertCount);
						pts[id - oldVertCount] = vd.p;
					}
				}
			}
			int[] in = new int[] { 0 };
			for (NestedTriangle tri : tris) {
				tri.addTriangles(indexes, in);
			}
			if (!sourceFirst) {
				for (int i = 0; i < in.length; i++) {
					if (in[i] >= oldVertCount) {
						in[i] -= oldVertCount;
					} else {
						in[i] += (vertCount - oldVertCount);
					}
				}
			}
			return new EmbeddedSurface(pts, indexes);
		}
	}
	/**
	 * Vertex data class
	 * @author Blake Lucas
	 *
	 */
	protected class VertexData {
		public Point3f p;
		public double[] data;

		public VertexData(Point3f p, double[] data) {
			this.p = p;
			this.data = data;
		}
	}
	/**
	 * Point class
	 * @param v vertex id
	 * @return point 3d
	 */
	protected KdPoint3 getPoint(int v) {
		if (v < surf.getVertexCount()) {
			return new KdPoint3(surf.getVertex(v));
		} else {
			return new KdPoint3(newPoints.get(v - surf.getVertexCount()).p);
		}
	}
	/**
	 * Nested triangle 
	 * @author Blake Lucas
	 *
	 */
	protected class NestedTriangle extends KdTriangle {
		public int v1, v2, v3;

		public NestedTriangle[] children = null;

		public NestedTriangle(int v1, int v2, int v3) {
			super();
			this.v1 = v1;
			this.v2 = v2;
			this.v3 = v3;
			this.pts = new KdPoint3[] { getPoint(v1), getPoint(v2),
					getPoint(v3) };
		}

		public void addTriangles(int[] indexes, int[] in) {
			if (children == null) {
				indexes[in[0]++] = v1;
				indexes[in[0]++] = v2;
				indexes[in[0]++] = v3;
			} else {
				children[0].addTriangles(indexes, in);
				children[1].addTriangles(indexes, in);
				children[2].addTriangles(indexes, in);
			}
		}

		public void insert(Point3f p, int vid) {
			if (children == null) {
				children = new NestedTriangle[] {
						new NestedTriangle(v1, v2, vid),
						new NestedTriangle(v2, v3, vid),
						new NestedTriangle(v3, v1, vid) };
				faceCount += 2;
			} else {
				double d1 = children[0].distance(p);
				double d2 = children[1].distance(p);
				double d3 = children[2].distance(p);
				if (d1 < d2) {
					if (d1 < d3) {
						children[0].insert(p, vid);
					} else {
						children[2].insert(p, vid);
					}
				} else {
					if (d2 < d3) {
						children[1].insert(p, vid);
					} else {
						children[2].insert(p, vid);
					}
				}
			}
		}

		public Point3f[] getPoints() {
			return pts;
		}

		public void update() {
		}
	}
}
