package edu.jhu.ece.iacl.algorithms.graphics.surf;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Stack;

import javax.vecmath.Matrix3d;
import javax.vecmath.Point2d;
import javax.vecmath.Point3f;
import javax.vecmath.Vector2d;
import javax.vecmath.Vector3d;
import javax.vecmath.Vector3f;

import com.sun.j3d.utils.geometry.GeometryInfo;
import com.sun.j3d.utils.geometry.Triangulator;

import edu.jhu.ece.iacl.jist.algorithms.graphics.GeomUtil;
import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.algorithms.graphics.XuGraphicsWrapper;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.data.BinaryMinHeap;
import edu.jhu.ece.iacl.jist.structures.geom.EdgeIndexed;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.geom.VertexIndexed;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface.Direction;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface.Edge;

/**
 * Progressive mesh implementation for multi-resolution decomposition of surface
 * meshes. The mesh topology can be changed through a series of edge collapse or
 * edge swap operations, and then restored by applying inverse operations. The
 * metric used to decide which edges should be collapse and in what order can be
 * modified.
 * 
 * 
 * @author Blake Lucas
 * 
 */
public class ProgressiveSurface extends XuGraphicsWrapper {
	protected float MIN_EDGE_LENGTH = 0;
	protected int maxNonHarmonic = 0;
	protected HeapVertexMetric vertexMetric;
	protected HeapEdgeMetric edgeMetric;
	protected VertexInsertionMetric vertInsertMetric;
	protected VertexInsertionOptimizationMethod vertInsertOptMethod;
	protected CollapseMethod collapseMethod = CollapseMethod.EDGE;
	protected Direction direction = Direction.COUNTER_CLOCKWISE;
	protected Stack<MeshTopologyOperation> vertStack;
	protected static final double TAU = 0.5 * (1 + Math.sqrt(5));
	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.3 $");
	}
	public static enum VertexMetric {
		CURVATURE, SURFACE_DISTANCE, SPHERE_CURVATURE, DISTANCE_XY
	};

	public static enum EdgeMetric {
		DISTANCE, ARC_LENGTH, ANGLE, DISTANCE_XY
	};

	public static enum CollapseMethod {
		EDGE, HALF_EDGE, SPHERICAL_EDGE
	};

	public ProgressiveSurface() {
		super();
		setTotalUnits(100);
		vertexMetric = new CurvatureVertexMetric();
		edgeMetric = new EdgeLengthMetric();
		vertInsertMetric = new MinOfMaxEdgeLengthMetric();
		vertInsertOptMethod = new VertexRadialSearchMethod();
		vertStack = new Stack<MeshTopologyOperation>();
	}

	public ProgressiveSurface(AbstractCalculation parent) {
		super(parent);
		setTotalUnits(100);
		vertexMetric = new CurvatureVertexMetric();
		edgeMetric = new EdgeLengthMetric();
		vertInsertMetric = new MinOfMaxEdgeLengthMetric();
		vertInsertOptMethod = new VertexRadialSearchMethod();
		vertStack = new Stack<MeshTopologyOperation>();
	}

	/**
	 * Set vertex metric used to decimate surface
	 * 
	 * @param m
	 */
	public void setMetric(VertexMetric m) {
		switch (m) {
		case CURVATURE:
			vertexMetric = new CurvatureVertexMetric();
			break;
		case SURFACE_DISTANCE:
			vertexMetric = new SurfaceDistanceVertexMetric();
			break;
		case SPHERE_CURVATURE:
			vertexMetric = new SphericalCurvatureVertexMetric();
			break;
		case DISTANCE_XY:
			vertexMetric = new DistanceXYVertexMetric();
			break;
		}
	}

	/**
	 * Set method to collapse edges in mesh
	 * 
	 * @param collapseMethod
	 */
	public void setCollapseMethod(CollapseMethod collapseMethod) {
		this.collapseMethod = collapseMethod;
	}

	/**
	 * Set edge metric for mesh regularization
	 * 
	 * @param m
	 *            metric
	 */
	public void setMetric(EdgeMetric m) {
		switch (m) {
		case DISTANCE:
			edgeMetric = new EdgeLengthMetric();
			break;
		case DISTANCE_XY:
			edgeMetric = new EdgeLength2DMetric();
			break;
		case ARC_LENGTH:
			edgeMetric = new ArcLengthEdgeMetric();
			break;
		case ANGLE:
			edgeMetric = new AngleEdgeMetric();
			break;
		}
	}

	/**
	 * Get reference surface surface
	 * 
	 * @return
	 */
	public EmbeddedSurface getSurface() {
		return surf;
	}

	/**
	 * Get vertex heap metric
	 * 
	 * @param id
	 *            vertex id
	 * @return metric value
	 */
	public double heapMetric(int id) {
		return vertexMetric.evaluate(id);
	}

	public ProgressiveSurface(EmbeddedSurface surf) {
		super();
		vertexMetric = new CurvatureVertexMetric();
		edgeMetric = new EdgeLengthMetric();
		vertStack = new Stack<MeshTopologyOperation>();
		setTotalUnits(100);
		init(surf, false);
	}

	/**
	 * Set winding direction
	 * 
	 * @param d
	 *            direction
	 */
	public void setDirection(Direction d) {
		this.direction = d;
	}

	/**
	 * Create edge swap operation
	 * 
	 * @return edge swap
	 */
	public EdgeSwap createEdgeSwap() {
		return new EdgeSwap();
	}

	/**
	 * Create edge collapse operation
	 * 
	 * @return edge collapse
	 */
	public EdgeCollapse createEdgeCollapse() {
		switch (collapseMethod) {
		case EDGE:
			return new EdgeCollapse();
		case SPHERICAL_EDGE:
			return new SphericalEdgeCollapse();
		case HALF_EDGE:
			return new HalfEdgeCollapse();
		default:
			return null;
		}
	}

	/**
	 * Initialize progressive surface
	 * 
	 * @param origSurf
	 *            reference surface
	 * @param initAllTables
	 *            initialize all lookup tables. Only the vertex-vertex table is
	 *            need for progressive surfaces, but generating other tables can
	 *            be useful for other types of operations.
	 */
	public void init(EmbeddedSurface origSurf, boolean initAllTables) {
		if (surf == null) {
			this.surf = origSurf.clone();
			initTables(initAllTables);
		}

	}

	/**
	 * Vertex heap metric.
	 * 
	 * @author Blake Lucas
	 * 
	 */
	public interface HeapVertexMetric {
		public double evaluate(int id);
	}

	/**
	 * Vertex insertion metric.
	 * 
	 * @author Blake Lucas
	 * 
	 */
	public interface VertexInsertionMetric {
		public double evaluate(int id);
	}

	/**
	 * Edge heap metric.
	 * 
	 * @author Blake Lucas
	 * 
	 */
	public interface HeapEdgeMetric {
		public double evaluate(int vid1, int vid2);
	}

	/**
	 * Vertex insertion method.
	 * 
	 * @author Blake Lucas
	 * 
	 */
	public interface VertexInsertionOptimizationMethod {
		public void insert(SphericalEdgeCollapse vs);
	}

	/**
	 * Edge length metric
	 * 
	 * @author Blake Lucas
	 * 
	 */
	public class EdgeLengthMetric implements HeapEdgeMetric {
		public double evaluate(int vid1, int vid2) {
			return -(surf.getVertex(vid1).distance(surf.getVertex(vid2)));
		}
	}

	/**
	 * Edge length 2D metric
	 * 
	 * @author Blake Lucas
	 * 
	 */
	public class EdgeLength2DMetric implements HeapEdgeMetric {
		public double evaluate(int vid1, int vid2) {
			Point3f pt1 = surf.getVertex(vid1);
			Point3f pt2 = surf.getVertex(vid2);
			return -(Math.sqrt((pt1.x - pt2.x) * (pt1.x - pt2.x)
					+ (pt1.y - pt2.y) * (pt1.y - pt2.y)));
		}
	}

	/**
	 * Max anlge metric
	 * 
	 * @author Blake Lucas
	 * 
	 */
	public class AngleEdgeMetric implements HeapEdgeMetric {
		public double evaluate(int vid1, int vid2) {
			if (find(vid1, vid2) == -1)
				return 0;
			int nbrIndex1 = find(vid1, vid2);
			int[] nbrs1 = neighborVertexVertexTable[vid1];
			int vid3 = nbrs1[(nbrIndex1 + 1) % nbrs1.length];
			int vid4 = nbrs1[(nbrIndex1 + nbrs1.length - 1) % nbrs1.length];
			Point3f pt1 = surf.getVertex(vid1);
			Point3f pt2 = surf.getVertex(vid2);
			return -Math.max(GeometricUtilities.angle(pt1, pt2, surf
					.getVertex(vid3)), GeometricUtilities.angle(pt2, pt1, surf
					.getVertex(vid4)));
		}
	}

	/**
	 * Spherical arc length metric
	 * 
	 * @author Blake Lucas
	 * 
	 */
	public class ArcLengthEdgeMetric implements HeapEdgeMetric {
		public double evaluate(int vid1, int vid2) {
			return -GeometricUtilities.angle(surf.getVertex(vid1), surf
					.getVertex(vid2));
		}
	}

	/**
	 * Surface distance metric
	 * 
	 * @author Blake Lucas
	 * 
	 */
	public class SurfaceDistanceVertexMetric implements HeapVertexMetric {
		public double evaluate(int id) {
			int[] nbrs = neighborVertexVertexTable[id];
			Vector3f norm = getCalculatedNormal(id);
			// If this a degenerate point, promote to top of queue
			// Degenerate points should be removed first
			double d = 0;
			for (int i = 0; i < nbrs.length; i++) {
				d += GeometricUtilities.dot(norm, surf.getVertex(nbrs[i]));
			}
			d = (nbrs.length > 0) ? (GeometricUtilities.dot(norm, surf
					.getVertex(id)) - d / nbrs.length) : 0;
			return Math.abs(d);
		}
	}

	/**
	 * Minimum neighbor distance vertex metric
	 * 
	 * @author Blake Lucas
	 * 
	 */
	public class DistanceXYVertexMetric implements HeapVertexMetric {
		public double evaluate(int id) {
			int[] nbrs = neighborVertexVertexTable[id];
			double mind = 1E30;
			Point3f pt1 = surf.getVertex(id);
			for (int i = 0; i < nbrs.length; i++) {
				Point3f pt2 = surf.getVertex(nbrs[i]);
				mind = Math
						.min(mind, (Math.sqrt((pt1.x - pt2.x) * (pt1.x - pt2.x)
								+ (pt1.y - pt2.y) * (pt1.y - pt2.y))));
			}
			return mind;
		}
	}

	protected static final double ANGLE_THRESH = Math.PI / 6;
	protected static double MIN_HEAP_VAL = -1E30;

	/**
	 * Surface curvature metric
	 * 
	 * @author Blake Lucas
	 * 
	 */
	public class CurvatureVertexMetric implements HeapVertexMetric {
		public double evaluate(int id) {
			int[] nbrs = neighborVertexVertexTable[id];

			Point3f pivot = surf.getVertex(id);
			// If this a degenerate point, promote to top of queue
			// Degenerate points should be removed first
			for (int i = 0; i < nbrs.length; i++) {
				if (pivot.distance(surf.getVertex(nbrs[i])) <= MIN_EDGE_LENGTH) {
					return MIN_HEAP_VAL;
				}
			}
			Vector3f c = getMeanCurvature(id);
			return -c.length();
		}

	}

	/**
	 * Edge heap metric
	 * 
	 * @param vid1
	 *            vertex id 1
	 * @param vid2
	 *            vertex id 2
	 * @return
	 */
	protected double heapMetric(int vid1, int vid2) {
		return edgeMetric.evaluate(vid1, vid2);
	}

	public int getMaxNonHarmonic() {
		return maxNonHarmonic;
	}

	/**
	 * Get number of non-harmonic vertices.
	 * 
	 * @return number of vertices
	 */
	protected int nonHarmonicPoints() {
		int nonHarmonicCount = 0;
		int vertCount = surf.getVertexCount();

		for (int id = 0; id < vertCount; id++) {
			nonHarmonicCount += (isWoundCorrectly(id)) ? 0 : 1;
		}
		maxNonHarmonic = Math.max(maxNonHarmonic, nonHarmonicCount);
		System.out.println("Non-Harmonic Points " + nonHarmonicCount);
		setCompletedUnits(1 - ((maxNonHarmonic == 0) ? 0 : nonHarmonicCount
				/ (float) maxNonHarmonic));
		return nonHarmonicCount;
	}

	/**
	 * Is spherical map bijective embedding?
	 * 
	 * @return true if bijective
	 */
	public boolean isHarmonic() {
		return (nonHarmonicPoints() == 0);
	}

	/**
	 * Simple indexed vertex for sorting
	 * 
	 * @author Blake Lucas
	 * 
	 */
	public class SimpleVertexIndexed implements Comparable<SimpleVertexIndexed> {
		public double d;
		public int v;

		public SimpleVertexIndexed(int v, double d) {
			this.v = v;
			this.d = d;
		}

		public int compareTo(SimpleVertexIndexed o) {
			return (int) Math.signum(this.d - o.d);
		}
	}

	/**
	 * Get sorted 1-ring neighbors
	 * 
	 * @param v1
	 *            vertex id
	 * @return list of vertex ids
	 */
	protected int[] getSortedNeighbors(int v1) {
		double d;
		int[] nbrs = neighborVertexVertexTable[v1];
		Point3f ref = surf.getVertex(v1);
		SimpleVertexIndexed[] vd = new SimpleVertexIndexed[nbrs.length];
		for (int i = 0; i < nbrs.length; i++) {
			d = ref.distance(surf.getVertex(nbrs[i]));
			vd[i] = new SimpleVertexIndexed(nbrs[i], d);
		}
		Arrays.sort(vd);
		int newNbrs[] = new int[nbrs.length];
		for (int i = 0; i < newNbrs.length; i++) {
			newNbrs[i] = vd[i].v;
		}
		return newNbrs;
	}

	/**
	 * Check if removal of edge will create a topology change
	 * 
	 * @param v1
	 *            vertex id 1
	 * @param v2
	 *            vertex id 2
	 * @return true if operation will not create topology change
	 */
	public boolean topologyCheck(int v1, int v2) {
		int[] nbrs1 = neighborVertexVertexTable[v1];
		int[] nbrs2 = neighborVertexVertexTable[v2];
		int nbrIndex1 = find(v1, v2);
		int nbrIndex2 = find(v2, v1);
		int v3;
		// Check that nodes do not have more than three vertexes
		for (int i = 2; i < nbrs1.length - 1; i++) {
			v3 = nbrs1[(i + nbrIndex1) % nbrs1.length];
			for (int j = 2; j < nbrs2.length - 1; j++) {
				if (v3 == nbrs2[(j + nbrIndex2) % nbrs2.length]) {
					return false;
				}
			}
		}
		return true;
	}

	/**
	 * Check if removal of vertex will create topology change
	 * 
	 * @param v1
	 *            vertex id
	 * @return true if operation will not create topology change
	 */
	protected boolean topologyCheckVertex(int v1) {
		int[] nbrs1 = neighborVertexVertexTable[v1];
		int[] nbrs2;
		int count = 0;
		for (int v2 : nbrs1) {
			nbrs2 = neighborVertexVertexTable[v2];
			count = 0;
			for (int nbr2 : nbrs2) {
				for (int nbr1 : nbrs1) {
					if (nbr1 == nbr2) {
						count++;
					}
					if (count > 2) {
						return false;
					}
				}
			}

		}
		return true;
	}

	/**
	 * Mesh topology operation
	 * 
	 * @author Blake Lucas
	 * 
	 */
	public static abstract class MeshTopologyOperation {
		public int v1;

		public boolean apply(int vert) {
			return false;
		}

		public boolean apply(int v1, int v2) {
			return false;
		}

		public abstract void restore();

		public int getRemovedVertexId() {
			return v1;
		}

		public abstract int[] getChangedVerts();
	}

	/**
	 * Edge swap operation
	 * 
	 * @author Blake Lucas
	 * 
	 */
	public class EdgeSwap extends MeshTopologyOperation {
		public int v2;
		public int v3;
		public int v4;

		public boolean apply(int v3, int v4) {

			int nbrIndex1 = find(v3, v4);
			int[] nbrs1 = neighborVertexVertexTable[v3];
			this.v1 = nbrs1[(nbrIndex1 + 1) % nbrs1.length];
			this.v2 = nbrs1[(nbrIndex1 + nbrs1.length - 1) % nbrs1.length];
			if (find(v1, v2) != -1)
				return false;

			if (surf.getVertex(v1).distance(surf.getVertex(v2)) >= surf
					.getVertex(v3).distance(surf.getVertex(v4))) {
				return false;
			}

			connectAfter(v1, v3, v2);
			connectAfter(v2, v4, v1);
			disconnect(v3, v4);
			disconnect(v4, v3);
			/*
			 * System.out.println("AFTER SWAP"); printNeighbors(v1);
			 * printNeighbors(v2); printNeighbors(v3); printNeighbors(v4);
			 */
			return true;
		}

		public boolean apply(int vert) {
			int[] nbrs = getSortedNeighbors(vert);
			for (int i = nbrs.length - 1; i >= 0; i--) {
				// If we cannot remove the shortest edge because of topology,
				// then try other edges in order of length until an edge is
				// removed
				if (apply(nbrs[i], vert)) {
					return true;
				}
			}
			return false;
		}

		public int[] getChangedVerts() {
			return new int[] { v1, v2, v3, v4 };
		}

		public int getRemovedVertexId() {
			return -1;
		}

		public void restore() {
			int nbrIndex1 = find(v1, v2);
			int[] nbrs1 = neighborVertexVertexTable[v1];
			int v3 = nbrs1[(nbrIndex1 + 1) % nbrs1.length];
			int v4 = nbrs1[(nbrIndex1 + nbrs1.length - 1) % nbrs1.length];
			connectAfter(v3, v1, v4);
			connectAfter(v4, v2, v3);
			disconnect(v1, v2);
			disconnect(v2, v1);
		}

	}

	/**
	 * Spherical edge collapse operation
	 * 
	 * @author Blake Lucas
	 * 
	 */
	public class SphericalEdgeCollapse extends EdgeCollapse {
		public int v2;
		public int nbr1;
		public int nbr2;

		public int getRemovedVertexId() {
			return v2;
		}

		public int[] getChangedVerts() {
			return neighborVertexVertexTable[v1];
		}

		public boolean apply(int vert) {
			int[] nbrs = getSortedNeighbors(vert);
			nbrs = getSortedNeighbors(vert);
			for (int i = 0; i < nbrs.length; i++) {
				// If we cannot remove the shortest edge because of topology,
				// then try other edges in order of length until an edge is
				// removed
				if (apply(nbrs[i], vert)) {
					return true;
				}
			}
			return false;
		}

		public boolean apply(int v1, int v2) {
			this.v1 = v1;
			this.v2 = v2;
			if (!topologyCheck(v1, v2))
				return false;
			int nbrIndex1 = find(v1, v2);
			int nbrIndex2 = find(v2, v1);
			int[] nbrs1 = neighborVertexVertexTable[v1];
			int[] nbrs2 = neighborVertexVertexTable[v2];
			int[] newNbrs1 = new int[nbrs1.length + nbrs2.length - 4];
			int v3, e;

			this.nbr1 = nbrs1[(nbrIndex1 + 1) % nbrs1.length];
			this.nbr2 = nbrs1[(nbrIndex1 + nbrs1.length - 1) % nbrs1.length];

			for (int i = 2; i < nbrs2.length - 1; i++) {
				// vertex to add to ring around v1
				v3 = nbrs2[(i + nbrIndex2) % nbrs2.length];
				// add v2 neighbor to v1 neighbors
				newNbrs1[i - 2] = v3;
				// replace connection from v3 to v2 with v3 to v1
				e = find(v3, v2);
				neighborVertexVertexTable[v3][e] = v1;
			}
			// Disconnect first and last point from ring around v2

			// Add old vertexes to ring around v1
			int j = nbrIndex1 + 1;
			for (int i = nbrs2.length - 3; i < newNbrs1.length; i++) {
				newNbrs1[i] = nbrs1[j % nbrs1.length];
				j++;
			}
			neighborVertexVertexTable[v2] = new int[0];
			neighborVertexVertexTable[v1] = newNbrs1;

			disconnect(v3 = nbrs2[(nbrIndex2 + 1) % nbrs2.length], v2);
			disconnect(
					v3 = nbrs2[(nbrIndex2 + nbrs2.length - 1) % nbrs2.length],
					v2);

			Vector3f p = GeometricUtilities.slerp(surf.getVertex(v1), surf
					.getVertex(v2), 0.5);
			surf.setVertex(v1, p);
			return true;
		}

		@Override
		public void restore() {
			insertSphericalEdge(this);
		}
	}

	/**
	 * Edge collapse operation
	 * 
	 * @author Blake Lucas
	 * 
	 */
	public class EdgeCollapse extends MeshTopologyOperation {
		public int v1;
		public int v2;
		public int nbr1;
		public int nbr2;

		public int getRemovedVertexId() {
			return v2;
		}

		public EdgeCollapse() {
		}

		public int isCollapsable(int v2) {
			int[] nbrs = getSortedNeighbors(v2);
			nbrs = getSortedNeighbors(v2);
			int v1;
			for (int i = 0; i < nbrs.length; i++) {
				// If we cannot remove the shortest edge because of topology,
				// then try other edges in order of length until an edge is
				// removed
				v1 = nbrs[i];
				if (!topologyCheck(v1, v2)) {
					continue;
				}
				return v1;
			}
			return -1;
		}

		public boolean apply(int vert) {
			int[] nbrs = getSortedNeighbors(vert);
			nbrs = getSortedNeighbors(vert);
			for (int i = 0; i < nbrs.length; i++) {
				// If we cannot remove the shortest edge because of topology,
				// then try other edges in order of length until an edge is
				// removed
				if (apply(nbrs[i], vert)) {
					return true;
				}
			}
			return false;
		}

		public boolean apply(int vert1, int vert2) {
			this.v1 = vert1;
			this.v2 = vert2;
			if (!topologyCheck(v1, v2))
				return false;
			int nbrIndex1 = find(v1, v2);
			int nbrIndex2 = find(v2, v1);
			int[] nbrs1 = neighborVertexVertexTable[v1];
			int[] nbrs2 = neighborVertexVertexTable[v2];
			int[] newNbrs1 = new int[nbrs1.length + nbrs2.length - 4];
			int v3, e;

			nbr1 = nbrs1[(nbrIndex1 + 1) % nbrs1.length];
			nbr2 = nbrs1[(nbrIndex1 + nbrs1.length - 1) % nbrs1.length];

			for (int i = 2; i < nbrs2.length - 1; i++) {
				// vertex to add to ring around v1
				v3 = nbrs2[(i + nbrIndex2) % nbrs2.length];
				// add v2 neighbor to v1 neighbors
				newNbrs1[i - 2] = v3;
				// replace connection from v3 to v2 with v3 to v1
				e = find(v3, v2);
				neighborVertexVertexTable[v3][e] = v1;
			}
			// Disconnect first and last point from ring around v2

			// Add old vertexes to ring around v1
			int j = nbrIndex1 + 1;
			for (int i = nbrs2.length - 3; i < newNbrs1.length; i++) {
				newNbrs1[i] = nbrs1[j % nbrs1.length];
				j++;
			}
			neighborVertexVertexTable[v2] = new int[0];
			neighborVertexVertexTable[v1] = newNbrs1;

			disconnect(v3 = nbrs2[(nbrIndex2 + 1) % nbrs2.length], v2);
			disconnect(
					v3 = nbrs2[(nbrIndex2 + nbrs2.length - 1) % nbrs2.length],
					v2);

			Point3f p = new Point3f(surf.getVertex(v1));
			p.add(surf.getVertex(v2));
			p.scale(0.5f);
			surf.setVertex(v1, p);
			return true;
		}

		@Override
		public void restore() {
			insertSurfEdge(this);
		}

		@Override
		public int[] getChangedVerts() {
			int[] newList = Arrays.copyOf(neighborVertexVertexTable[v1],
					neighborVertexVertexTable[v1].length + 1);
			newList[newList.length - 1] = v1;
			return newList;
		}
	}

	/**
	 * Half edge collapse operation
	 * 
	 * @author Blake Lucas
	 * 
	 */
	public class HalfEdgeCollapse extends EdgeCollapse {
		public int v1;
		public int v2;
		public int nbr1;
		public int nbr2;

		public HalfEdgeCollapse() {
		}

		public boolean apply(int vert1, int vert2) {
			this.v1 = vert1;
			this.v2 = vert2;
			if (!topologyCheck(v1, v2))
				return false;
			int nbrIndex1 = find(v1, v2);
			int nbrIndex2 = find(v2, v1);
			int[] nbrs1 = neighborVertexVertexTable[v1];
			int[] nbrs2 = neighborVertexVertexTable[v2];
			int[] newNbrs1 = new int[nbrs1.length + nbrs2.length - 4];
			int v3, e;

			nbr1 = nbrs1[(nbrIndex1 + 1) % nbrs1.length];
			nbr2 = nbrs1[(nbrIndex1 + nbrs1.length - 1) % nbrs1.length];

			for (int i = 2; i < nbrs2.length - 1; i++) {
				// vertex to add to ring around v1
				v3 = nbrs2[(i + nbrIndex2) % nbrs2.length];
				// add v2 neighbor to v1 neighbors
				newNbrs1[i - 2] = v3;
				// replace connection from v3 to v2 with v3 to v1
				e = find(v3, v2);
				neighborVertexVertexTable[v3][e] = v1;
			}
			// Disconnect first and last point from ring around v2

			// Add old vertexes to ring around v1
			int j = nbrIndex1 + 1;
			for (int i = nbrs2.length - 3; i < newNbrs1.length; i++) {
				newNbrs1[i] = nbrs1[j % nbrs1.length];
				j++;
			}
			neighborVertexVertexTable[v2] = new int[0];
			neighborVertexVertexTable[v1] = newNbrs1;

			disconnect(v3 = nbrs2[(nbrIndex2 + 1) % nbrs2.length], v2);
			disconnect(
					v3 = nbrs2[(nbrIndex2 + nbrs2.length - 1) % nbrs2.length],
					v2);
			return true;
		}

		@Override
		public void restore() {
			insertSurfEdge(this);
		}

		@Override
		public int[] getChangedVerts() {
			int[] newList = Arrays.copyOf(neighborVertexVertexTable[v1],
					neighborVertexVertexTable[v1].length + 1);
			newList[newList.length - 1] = v1;
			return newList;
		}
	}

	/**
	 * Vertex collapse operation
	 * 
	 * @author Blake Lucas
	 * 
	 */
	public class VertexCollapse extends MeshTopologyOperation {
		public int[][] nbrs;
		public int[] ring;

		@Override
		public boolean apply(int vert) {
			if (!topologyCheckVertex(vert)) {
				return false;
			}
			int[] nbrs1 = neighborVertexVertexTable[vert];
			this.ring = Arrays.copyOf(nbrs1, nbrs1.length);
			this.v1 = vert;
			int[] indexArray = new int[nbrs1.length];
			Point3f[] points = new Point3f[nbrs1.length];
			int[] polySizeArray = new int[] { nbrs1.length };
			GeometryInfo gi = new GeometryInfo(GeometryInfo.POLYGON_ARRAY);
			Point3f[] vertCopy = new Point3f[surf.getVertexCount()];
			for (int nbr : nbrs1) {
				vertCopy[nbr] = surf.getVertex(nbr);
			}
			gi.setCoordinates(vertCopy);
			gi.setStripCounts(polySizeArray);
			gi.setCoordinateIndices(nbrs1);
			gi.setContourCounts(new int[] { polySizeArray.length });
			Triangulator tr = new Triangulator();
			tr.triangulate(gi);
			int[] newIndexes = gi.getCoordinateIndices();
			int[][] polyNbrs = buildPolyNeighborList(nbrs1, newIndexes);
			// Disconnect vertex from all neighbors
			for (int nbr : nbrs1) {
				disconnect(nbr, vert);
				// System.out.print("CURRENT NEIGHBORS OF ");
				// printNeighbors(nbr);
			}
			this.nbrs = new int[nbrs1.length][];
			for (int i = 0; i < nbrs1.length; i++) {
				int current = nbrs1[i];
				int next = nbrs1[(i + 1) % nbrs1.length];
				// int last=nbrs1[(i+nbrs1.length-1)%nbrs1.length];
				int nextIndex = find(current, next);
				// int lastIndex=find(current,last);
				int[] vertPolyNbrs = polyNbrs[current];
				nbrs[i] = vertPolyNbrs;
				int[] currentNbrs = neighborVertexVertexTable[current];
				if (nextIndex == -1) {
					System.out.println(current + " BEFORE INSERT NEXT " + next
							+ " NEXT INDEX " + nextIndex);
					printNeighbors(current);
					System.exit(1);
				}
				for (int offset = vertPolyNbrs.length - 2; offset > 0; offset--) {
					int newNbr = vertPolyNbrs[offset % vertPolyNbrs.length];
					connect(current, newNbr, (nextIndex + 1)
							% currentNbrs.length);
				}
				// System.out.print("NEW NEIGHBORS OF ");
				// printNeighbors(current);
				// System.out.println("AFTER INSERT");
				// printNeighbors(current);

			}

			neighborVertexVertexTable[vert] = new int[0];

			/*
			 * for(int nbr:nbrs1){ if(!sanityCheck(nbr)){
			 * System.out.println("SANITY CHECK FAILED ON "+nbr);
			 * System.exit(0); } }
			 */
			return true;
		}

		public int[] getChangedVerts() {
			return neighborVertexVertexTable[v1];
		}

		@Override
		public void restore() {
			// Disconnect triangulated region
			Point3f avgPoint = new Point3f();
			for (int i = 0; i < ring.length; i++) {

				int[] nbrnbrs = nbrs[i];
				int nbr = ring[i];

				Point3f p = surf.getVertex(nbr);
				avgPoint.add(p);
				for (int j = 1; j < nbrnbrs.length - 1; j++) {
					disconnect(nbr, nbrnbrs[j]);
				}
			}
			avgPoint.scale(1 / (float) ring.length);
			// Reconnect vertex
			for (int i = 0; i < ring.length; i++) {
				connectAfter(ring[i], ring[(i + 1) % ring.length], v1);
			}
			neighborVertexVertexTable[v1] = ring;
			surf.setVertex(v1, avgPoint);
		}

		protected int[][] buildPolyNeighborList(int[] verts, int indexes[]) {
			int vertCount = surf.getVertexCount();
			ArrayList<Integer> tmpTable[] = new ArrayList[vertCount];
			int v1, v2, v3;
			for (int i = 0; i < indexes.length; i += 3) {
				v1 = indexes[i];
				v2 = indexes[i + 1];
				v3 = indexes[i + 2];
				if (tmpTable[v1] == null)
					tmpTable[v1] = new ArrayList<Integer>();
				if (tmpTable[v2] == null)
					tmpTable[v2] = new ArrayList<Integer>();
				if (tmpTable[v3] == null)
					tmpTable[v3] = new ArrayList<Integer>();
				tmpTable[v1].add(v2);
				tmpTable[v1].add(v3);
				tmpTable[v2].add(v3);
				tmpTable[v2].add(v1);
				tmpTable[v3].add(v1);
				tmpTable[v3].add(v2);
			}
			int pivot;
			int count = 0;
			ArrayList<Integer> neighbors;
			boolean found;

			int[][] neighborTable = new int[vertCount][0];

			for (int i = 0; i < verts.length; i++) {
				ArrayList<Integer> tmpNbrs = new ArrayList<Integer>();
				int v = verts[i];
				neighbors = tmpTable[v];

				if (neighbors == null || neighbors.size() == 0)
					continue;
				count = 0;
				int m, n;
				int firstIndex = -1;
				int next = verts[(i + 1) % verts.length];
				for (int j = 0; j < neighbors.size(); j++) {
					if (neighbors.get(j) == next) {
						firstIndex = j;
						break;
					}
				}
				if (firstIndex == -1) {
					System.err.println("CURRENT NEIGHBORS OF " + verts[i]
							+ " NEXT " + next + " " + verts.length);
					for (int k = 0; k < neighbors.size(); k++) {
						System.err.print(neighbors.get(k) + " ");
					}
					System.err.println();
					System.exit(1);
				}
				tmpNbrs.add(neighbors.remove(firstIndex));
				tmpNbrs.add(pivot = neighbors.remove(firstIndex
						% neighbors.size()));

				while (neighbors.size() > 0) {
					found = false;
					for (int k = 0; k < neighbors.size(); k += 2) {
						if (neighbors.get(k) == pivot) {
							neighbors.remove(k);
							tmpNbrs.add(pivot = neighbors.remove(k
									% neighbors.size()));
							found = true;
							break;
						}
					}
					if (!found) {
						System.err.println("NOT FOUND " + pivot);
						System.exit(1);
					}
				}
				neighborTable[v] = new int[tmpNbrs.size()];

				// System.out.print("POLY NEIGHBORS OF "+v+") ");
				for (int tmp : tmpNbrs) {
					// System.out.print(tmp+" ");
					neighborTable[v][count++] = tmp;
				}
				// System.out.println();
			}
			return neighborTable;
		}
	}

	/**
	 * Insert edge into surface
	 * 
	 * @param vs
	 *            edge collapse operation to reverse
	 */
	protected void insertSurfEdge(EdgeCollapse vs) {
		int v1 = vs.v1;
		int v2 = vs.v2;
		int nbr1 = vs.nbr1;
		int nbr2 = vs.nbr2;
		int[] nbrs1 = neighborVertexVertexTable[v1];
		int nbrIndex1 = find(v1, nbr1);
		int nbrIndex2 = find(v1, nbr2);
		int[] newNbrs1 = new int[((nbrIndex2 > nbrIndex1) ? (nbrIndex2 - nbrIndex1)
				: (nbrIndex2 + nbrs1.length - nbrIndex1)) + 2];
		int[] newNbrs2 = new int[nbrs1.length - newNbrs1.length + 4];
		newNbrs1[newNbrs1.length - 1] = v2;
		for (int i = 0; i < newNbrs1.length - 1; i++) {
			newNbrs1[i] = nbrs1[(nbrIndex1 + i) % nbrs1.length];
		}
		newNbrs2[0] = v1;
		int v3;
		for (int i = 0; i < newNbrs2.length - 1; i++) {
			v3 = newNbrs2[i + 1] = nbrs1[(nbrIndex2 + i) % nbrs1.length];
			if (i == 0) {
				connect(v3, v2, find(v3, v1));
			} else if (i == newNbrs2.length - 2) {
				connect(v3, v2, (find(v3, v1) + 1)
						% neighborVertexVertexTable[v3].length);
			} else {
				int e = find(v3, v1);
				neighborVertexVertexTable[v3][e] = v2;
			}
		}
		neighborVertexVertexTable[v1] = newNbrs1;
		neighborVertexVertexTable[v2] = newNbrs2;

		insertSurfPoint(vs);
	}

	/**
	 * Insert half-edge into surface
	 * 
	 * @param vs
	 *            edge collapse to reverse
	 */
	protected void insertHalfSurfEdge(EdgeCollapse vs) {
		int v1 = vs.v1;
		int v2 = vs.v2;
		int nbr1 = vs.nbr1;
		int nbr2 = vs.nbr2;
		int[] nbrs1 = neighborVertexVertexTable[v1];
		int nbrIndex1 = find(v1, nbr1);
		int nbrIndex2 = find(v1, nbr2);
		int[] newNbrs1 = new int[((nbrIndex2 > nbrIndex1) ? (nbrIndex2 - nbrIndex1)
				: (nbrIndex2 + nbrs1.length - nbrIndex1)) + 2];
		int[] newNbrs2 = new int[nbrs1.length - newNbrs1.length + 4];
		newNbrs1[newNbrs1.length - 1] = v2;
		for (int i = 0; i < newNbrs1.length - 1; i++) {
			newNbrs1[i] = nbrs1[(nbrIndex1 + i) % nbrs1.length];
		}
		newNbrs2[0] = v1;
		int v3;
		for (int i = 0; i < newNbrs2.length - 1; i++) {
			v3 = newNbrs2[i + 1] = nbrs1[(nbrIndex2 + i) % nbrs1.length];
			if (i == 0) {
				connect(v3, v2, find(v3, v1));
			} else if (i == newNbrs2.length - 2) {
				connect(v3, v2, (find(v3, v1) + 1)
						% neighborVertexVertexTable[v3].length);
			} else {
				int e = find(v3, v1);
				neighborVertexVertexTable[v3][e] = v2;
			}
		}
		neighborVertexVertexTable[v1] = newNbrs1;
		neighborVertexVertexTable[v2] = newNbrs2;

		insertHalfSurfPoint(vs);
	}

	/**
	 * Insert edge into surface
	 * 
	 * @param vs
	 *            edge collapse operation to reverse
	 */
	protected void insertSurfPoint(EdgeCollapse vs) {
		int[] nbrs2 = neighborVertexVertexTable[vs.v2];
		int[] nbrs1 = neighborVertexVertexTable[vs.v1];
		surf.setVertex(vs.v2, surf.getVertex(vs.v1));
		Point3f centroid1 = new Point3f();
		Point3f centroid2 = new Point3f();
		for (int i = 0; i < nbrs1.length; i++) {
			centroid1.add(surf.getVertex(nbrs1[i]));
		}
		centroid1.scale(1 / (float) nbrs1.length);
		for (int i = 0; i < nbrs2.length; i++) {
			centroid2.add(surf.getVertex(nbrs2[i]));
		}
		centroid2.scale(1 / (float) nbrs2.length);
		surf.setVertex(vs.v1, centroid1);
		surf.setVertex(vs.v2, centroid2);
	}

	/**
	 * Insert half-edge into surface
	 * 
	 * @param vs
	 *            edge collapse operation to reverse
	 */
	protected void insertHalfSurfPoint(EdgeCollapse vs) {
		int[] nbrs2 = neighborVertexVertexTable[vs.v2];
		surf.setVertex(vs.v2, surf.getVertex(vs.v1));
		Point3f centroid2 = new Point3f();
		for (int i = 0; i < nbrs2.length; i++) {
			centroid2.add(surf.getVertex(nbrs2[i]));
		}
		centroid2.scale(1 / (float) nbrs2.length);
		surf.setVertex(vs.v2, centroid2);
	}

	protected boolean sanityCheck(int vid) {
		int[] nbrs = neighborVertexVertexTable[vid];
		for (int nbr : nbrs) {
			if (find(nbr, vid) == -1) {
				return false;
			}
			int count = 0;
			for (int nbr2 : nbrs) {
				if (nbr == nbr2) {
					count++;
				}
			}
			if (count > 1)
				return false;
			if (nbr == vid)
				return false;
		}

		return true;
	}

	/**
	 * Print neighbors for each vertex
	 * 
	 * @param v1
	 *            vertex id
	 */
	protected void printNeighbors(int v1) {
		int[] nbrs1 = neighborVertexVertexTable[v1];

		System.out.print(v1 + "-" + nbrs1.length + ") ");
		for (int i = 0; i < nbrs1.length; i++) {
			System.out.print(nbrs1[i] + " ");
		}
		System.out.print("\n");
	}

	/**
	 * Find neighbor index of vertex id
	 * 
	 * @param v1
	 *            vertex id
	 * @param v2
	 *            neighbor vertex id
	 * @return neighbor index
	 */
	public int find(int v1, int v2) {
		return find(v1, v2, neighborVertexVertexTable);
	}

	/**
	 * Find neighbor index of vertex id
	 * 
	 * @param v1
	 *            vertex id
	 * @param v2
	 *            neighbor vertex id
	 * @param vertexTable
	 *            vertex-vertex table
	 * @return neighbor index
	 */
	protected int find(int v1, int v2, int vertexTable[][]) {
		int[] nbrs = vertexTable[v1];
		for (int i = 0; i < nbrs.length; i++) {
			if (nbrs[i] == v2) {
				return i;
			}
		}
		return -1;
	}

	/**
	 * Connect vertex v3 after vertex v2 around v1
	 * 
	 * @param v1
	 *            vertex v1
	 * @param v2
	 *            vertex v2
	 * @param v3
	 *            vertex v3
	 * @return
	 */
	protected boolean connectAfter(int v1, int v2, int v3) {
		return connectAfter(v1, v2, v3, neighborVertexVertexTable);
	}

	/**
	 * Connect vertex v3 after vertex v2 around v1
	 * 
	 * @param v1
	 *            vertex v1
	 * @param v2
	 *            vertex v2
	 * @param v3
	 *            vertex v3
	 * @param vertexTable
	 *            vertex-vertex table
	 * @return vertex
	 */
	protected boolean connectAfter(int v1, int v2, int v3, int vertexTable[][]) {
		int[] nbrs = vertexTable[v1];
		for (int i = 0; i < nbrs.length; i++) {
			if (nbrs[i] == v2) {
				connect(v1, v3, (i + 1) % nbrs.length);
				return true;
			}
		}

		return false;
	}

	/**
	 * Insert vertex v2 at the index position around vertex v1
	 * 
	 * @param v1
	 *            vertex v1
	 * @param v2
	 *            vertex v2
	 * @param in1
	 *            index position
	 */
	protected void connect(int v1, int v2, int in1) {
		int[] nbrs1 = neighborVertexVertexTable[v1];
		int[] newNbrs1 = new int[nbrs1.length + 1];
		int count = 0;
		for (int i = 0; i < newNbrs1.length; i++) {
			newNbrs1[i] = (i == in1) ? v2 : nbrs1[count++];
		}
		neighborVertexVertexTable[v1] = newNbrs1;
	}

	/**
	 * Disconnect v1 from v2
	 * 
	 * @param v1
	 *            vertex v1
	 * @param v2
	 *            vertex v2
	 */
	public void disconnect(int v1, int v2) {
		int[] nbrs1 = neighborVertexVertexTable[v1];
		int[] newNbrs1 = new int[nbrs1.length - 1];

		int count = 0;
		for (int i = 0; i < nbrs1.length; i++) {
			// Remove only first occurence
			newNbrs1[count] = (nbrs1[i] == v2 && count == i) ? nbrs1[++i]
					: nbrs1[i];
			count++;
			if (count == nbrs1.length - 1)
				break;
		}

		neighborVertexVertexTable[v1] = newNbrs1;
	}

	/**
	 * Create surface from current progressive surface representation
	 * 
	 * @return surface
	 */
	public EmbeddedSurface createSurface() {
		return createSurface(surf);
	}

	/**
	 * Create surface from current progressive surface representation
	 * 
	 * @param vertSurf
	 *            surface to use vertex locations and vertex data from
	 * @return surface
	 */
	public EmbeddedSurface createSurface(EmbeddedSurface vertSurf) {
		int vertCount = surf.getVertexCount();
		int vertIDs[] = new int[vertCount];
		int currentId = 0, n1, n2;

		for (int i = 0; i < vertCount; i++) {
			int len = neighborVertexVertexTable[i].length;
			if (len == 0) {
				vertIDs[i] = -1;
			} else {
				vertIDs[i] = currentId++;
			}
		}
		int ptsCount = currentId;
		ArrayList<Integer> indexes = new ArrayList<Integer>();
		for (int i = 0; i < vertCount; i++) {
			int[] nbrs = neighborVertexVertexTable[i];
			currentId = vertIDs[i];
			for (int j = 0; j < nbrs.length; j++) {

				n1 = vertIDs[nbrs[j]];
				n2 = vertIDs[nbrs[(j + 1) % nbrs.length]];
				// Only create new triangles if they include vertex not seen
				// before
				if (n1 > currentId && n2 > currentId) {
					// Add new triangle
					indexes.add(currentId);
					indexes.add(n2);
					indexes.add(n1);

				}
			}
		}
		int[] indexArray = new int[indexes.size()];
		Point3f[] pts = new Point3f[ptsCount];
		// Copy indexes to array
		for (int i = 0; i < indexArray.length; i++) {
			indexArray[i] = indexes.get(i);
		}
		double[][] oldVertData = vertSurf.getVertexData();
		double[][] newVertData = new double[ptsCount][0];
		// Copy subset of points to array
		for (int i = 0; i < vertCount; i++) {
			if (vertIDs[i] != -1) {
				pts[vertIDs[i]] = vertSurf.getVertex(i);
				if (oldVertData != null) {
					newVertData[vertIDs[i]] = oldVertData[i];
				}
			}
		}
		EmbeddedSurface newSurf = new EmbeddedSurface(pts, indexArray);
		newSurf.setVertexData(newVertData);
		newSurf.setName(surf.getName());
		return newSurf;
	}

	/**
	 * Get maximum mean curvature for surface
	 * 
	 * @return
	 */
	protected double getMaxCurvature() {
		int vertCount = surf.getVertexCount();
		double maxCurvature = 0;
		double curv;
		for (int id = 0; id < vertCount; id++) {
			Vector3f c = getMeanCurvature(id);
			curv = c.length();
			maxCurvature = Math.max(curv, maxCurvature);
		}
		return maxCurvature;
	}

	/**
	 * Get mean curvature for surface
	 * 
	 * @return
	 */
	protected double getMeanCurvature() {
		int vertCount = surf.getVertexCount();
		double curv = 0;
		for (int id = 0; id < vertCount; id++) {
			Vector3f c = getMeanCurvature(id);
			curv += c.length();
		}
		curv /= vertCount;
		return curv;
	}

	/**
	 * Get mean curvature for vertex
	 * 
	 * @param id
	 *            vertex id
	 * @return curvature vector
	 */
	public Vector3f getMeanCurvature(int id) {
		Vector3f meanCurv = new Vector3f();
		int len = neighborVertexVertexTable[id].length;
		if (len == 0)
			return new Vector3f(0, 0, 0);
		Point3f pivot = surf.getVertex(id);
		Vector3f edge1 = new Vector3f();
		Vector3f edge2 = new Vector3f();
		Vector3f edge3 = new Vector3f();
		double areaSum = 0, area, weight;
		double cota, cotb;
		Point3f pnext, pcurr, plast;
		for (int i = 0; i < len; i++) {
			pnext = surf
					.getVertex(neighborVertexVertexTable[id][(i + 1) % len]);
			pcurr = surf.getVertex(neighborVertexVertexTable[id][i]);
			plast = surf.getVertex(neighborVertexVertexTable[id][(i + len - 1)
					% len]);
			cota = GeometricUtilities.cotAngle(pivot, pcurr, pnext);
			cotb = GeometricUtilities.cotAngle(pivot, pcurr, plast);
			weight = 0.5 * (cota + cotb);
			area = GeometricUtilities.triangleArea(pivot, pcurr, pnext);
			edge1.sub(pcurr, pivot);
			edge2.sub(pnext, pcurr);
			edge3.sub(pivot, pnext);
			if (edge3.dot(edge2) < 0) {
				area /= 2;
			}
			if (edge1.dot(edge2) < 0 || edge1.dot(edge3) < 0) {
				area /= 2;
			}
			areaSum += area;

			pcurr.sub(pivot);
			pcurr.scale((float) weight);
			meanCurv.add(pcurr);

		}
		if (areaSum > 0) {
			meanCurv.scale((float) (0.5 / areaSum));
		}

		return meanCurv;
	}

	/**
	 * Get calculated normal
	 * 
	 * @param id
	 *            vertex id
	 * @return calcualted normal
	 */
	protected Vector3f getCalculatedNormal(int id) {
		Vector3f norm = new Vector3f();
		Vector3f edge1 = new Vector3f();
		Vector3f edge2 = new Vector3f();
		Vector3f cross = new Vector3f();
		Point3f pivot = surf.getVertex(id);
		int len = neighborVertexVertexTable[id].length;
		for (int i = 0; i < len; i++) {
			edge1.sub(surf.getVertex(neighborVertexVertexTable[id][i]), pivot);
			edge2.sub(surf.getVertex(neighborVertexVertexTable[id][(i + 1)
					% len]), pivot);
			cross.cross(edge1, edge2);
			norm.add(cross);
		}
		norm.normalize();
		return norm;
	}

	/**
	 * Find optimal location for vertex pair using radial line search.
	 * 
	 * @author Blake Lucas
	 * 
	 */
	protected class VertexRadialSearchMethod implements
			VertexInsertionOptimizationMethod {
		protected int angleSteps = 180;

		public void setAngleSteps(int angleSteps) {
			this.angleSteps = angleSteps;
		}

		public VertexRadialSearchMethod() {

		}

		public void insert(SphericalEdgeCollapse vs) {
			Point3f p = surf.getVertex(vs.v1);
			Point3f oldp = new Point3f(p);
			Vector3f pivot = new Vector3f(0, 0, 1);
			// Neighborhood vids
			int[] nbrs2 = neighborVertexVertexTable[vs.v2];
			int[] nbrs1 = neighborVertexVertexTable[vs.v1];
			// Neighborhood points
			Vector3f[] pnbrs2 = new Vector3f[nbrs2.length];
			Vector3f[] pnbrs1 = new Vector3f[nbrs1.length];
			// Rotate pivot point to (0,0,1)
			if (Math.abs(GeometricUtilities.length(p) - 1) > 1E-3) {
				System.out.println("Vertex " + vs.v1 + " " + p);
			}
			Matrix3d R = GeometricUtilities.rotateInverseMatrix3d(Math.atan2(
					p.y, p.x), Math.acos(p.z) + Math.PI * 0.5);
			for (int i = 0; i < nbrs1.length; i++) {
				if (nbrs1[i] == vs.v2) {
					pnbrs1[i] = null;
					continue;
				}
				p = surf.getVertex(nbrs1[i]);
				p = GeometricUtilities.multMatrix(R, p);
				// Edge length is zero, cannot insert point.
				if (oldp.distance(p) <= MIN_EDGE_LENGTH) {
					return;
				}
				pnbrs1[i] = new Vector3f(p);
			}
			for (int i = 0; i < nbrs2.length; i++) {
				if (nbrs2[i] == vs.v1) {
					pnbrs2[i] = null;
					continue;
				}
				p = surf.getVertex(nbrs2[i]);
				p = GeometricUtilities.multMatrix(R, p);
				// Edge length is zero, cannot insert point.
				if (oldp.distance(p) <= MIN_EDGE_LENGTH) {
					return;
				}
				pnbrs2[i] = new Vector3f(p);
			}
			// Store the best metric value
			double bestVal1 = -1E30;
			double bestVal2 = -1E30;
			int bestl = -1, bestn = -1;
			// Store nbhd position for next and last point
			int stV2 = find(vs.v2, vs.nbr2);
			int endV2 = find(vs.v2, vs.nbr1);
			int stV1 = find(vs.v1, vs.nbr1);
			int endV1 = find(vs.v1, vs.nbr2);

			Vector3f p1 = new Vector3f(pnbrs1[stV1]);
			p1.z = 0;
			p1.normalize();
			Vector3f p2 = new Vector3f(pnbrs1[endV1]);
			p2.z = 0;
			p2.normalize();
			Vector3f p0 = new Vector3f();
			double theta;
			float cos, sin;
			double t, tmin;
			Vector3f next, last, pt;
			// Invert rotation matrix to transform points back to original space
			R.invert();
			Vector3f[] candidates = new Vector3f[angleSteps];
			int nextId;
			// Total sweep angle between triangle
			double totalAngle = GeometricUtilities
					.sphericalAngle(pivot, p1, p2);
			if (totalAngle < Math.PI) {
				for (int l = 0; l < angleSteps; l++) {
					theta = totalAngle
							* (0.001 + 0.998 * l / (float) (angleSteps - 1));
					cos = (float) Math.cos(theta);
					sin = (float) Math.sin(theta);
					p0 = new Vector3f(p1.x * cos + p1.y * sin, -p1.x * sin
							+ p1.y * cos, 0);
					last = pnbrs1[stV1];
					tmin = 1E30;
					for (int i = 1; i < pnbrs1.length; i++) {
						nextId = (i + stV1) % pnbrs1.length;
						next = pnbrs1[nextId];
						t = GeometricUtilities.intersectionAngle(pivot, p0,
								last, next);
						last = next;
						if (t >= 0) {
							tmin = Math.min(t, tmin);
						}
						if (nextId == endV1)
							break;
					}
					if (tmin == 1E30)
						continue;
					if (tmin > Math.PI * 0.25) {
						System.err.println("COULD NOT OPTIMIZE VERTEX " + tmin
								+ " " + theta + " " + totalAngle);
					}

					SphericalArc arc = new SphericalArc(pivot, p0);
					double lower = tmin * 0.001;
					double upper = tmin * 0.999;
					double middle = lower + (upper - lower) / (1 + TAU);
					double ext;
					double maxValMiddle = 0;
					pt = arc.interpolate(middle);
					pt = GeometricUtilities.multMatrix(R, pt);
					surf.setVertex(vs.v1, pt);
					if (isNbhdWoundCorrectly(vs.v1)) {
						maxValMiddle = -vertInsertMetric.evaluate(vs.v1);
						if (maxValMiddle >= bestVal1) {
							bestl = l;
							candidates[l] = pt;
							bestVal1 = maxValMiddle;
						}
					} else {
						lower = upper;
					}
					// System.out.println("V1 INIT LOWER "+lower+" UPPER "+upper+" MAX "+maxValMiddle);
					while ((upper - lower) > 1E-5) {
						if (upper - middle > middle - lower) {
							ext = middle + (upper - middle) / (1 + TAU);
						} else {
							ext = lower + (middle - lower) / (1 + TAU);
						}
						pt = arc.interpolate(ext);
						pt = GeometricUtilities.multMatrix(R, pt);
						surf.setVertex(vs.v1, pt);
						if (isNbhdWoundCorrectly(vs.v1)) {
							double maxVal = -vertInsertMetric.evaluate(vs.v1);
							if (ext > middle) {
								if (maxVal > maxValMiddle) {
									lower = middle;
									middle = ext;
									maxValMiddle = maxVal;
								} else {
									upper = ext;
								}
							} else {
								if (maxVal > maxValMiddle) {
									upper = middle;
									middle = ext;
									maxValMiddle = maxVal;
								} else {
									lower = ext;
								}
							}
							if (maxVal >= bestVal1) {
								bestl = l;
								candidates[l] = pt;
								bestVal1 = maxVal;
							}
						} else {
							break;
						}
					}
					// System.out.println("V1 LOWER "+lower+" UPPER "+upper+" MAX "+maxValMiddle);
				}
				if (bestl == -1 || bestn == -1) {
					surf.setVertex(vs.v1, oldp);
				} else {
					surf.setVertex(vs.v1, candidates[bestl]);
				}
				// System.out.println("INCORRECT WINDINGS V1 "+count);
				// count = 0;
				bestl = -1;
				bestn = -1;
				totalAngle = 2 * Math.PI - totalAngle;
				p1 = p2;
				for (int l = 0; l < angleSteps; l++) {
					// Linear line and angular search
					theta = totalAngle
							* (0.001 + 0.998 * l / (float) (angleSteps - 1));
					cos = (float) Math.cos(theta);
					sin = (float) Math.sin(theta);
					p0 = new Vector3f(p1.x * cos + p1.y * sin, -p1.x * sin
							+ p1.y * cos, 0);
					last = pnbrs2[stV2];
					tmin = 1E30;
					for (int i = 1; i < pnbrs2.length; i++) {
						nextId = (i + stV2) % pnbrs2.length;
						next = pnbrs2[nextId];
						t = GeometricUtilities.intersectionAngle(pivot, p0,
								last, next);
						last = next;
						if (t >= 0) {
							tmin = Math.min(t, tmin);
						}
						if (nextId == endV2)
							break;
					}
					if (tmin == 1E30)
						continue;
					if (tmin > Math.PI * 0.25) {
						System.out.println("COULD NOT OPTIMIZE VERTEX " + tmin
								+ " " + theta + " " + totalAngle);
					}
					SphericalArc arc = new SphericalArc(pivot, p0);
					double lower = tmin * 0.001;
					double upper = tmin * 0.999;
					double middle = lower + (upper - lower) / (1 + TAU);
					double ext;
					double maxValMiddle = 0;
					pt = arc.interpolate(middle);
					pt = GeometricUtilities.multMatrix(R, pt);
					surf.setVertex(vs.v2, pt);
					if (isNbhdWoundCorrectly(vs.v2)) {
						maxValMiddle = -vertInsertMetric.evaluate(vs.v2);
						if (maxValMiddle >= bestVal2) {
							bestl = l;
							candidates[l] = pt;
							bestVal2 = maxValMiddle;
						}
					} else {
						lower = upper;
					}
					// System.out.println("V2 INIT LOWER "+lower+" UPPER "+upper+" MAX "+maxValMiddle);
					while ((upper - lower) > 1E-5) {
						if (upper - middle > middle - lower) {
							ext = middle + (upper - middle) / (1 + TAU);
						} else {
							ext = lower + (middle - lower) / (1 + TAU);
						}
						pt = arc.interpolate(ext);
						pt = GeometricUtilities.multMatrix(R, pt);
						surf.setVertex(vs.v2, pt);
						if (isNbhdWoundCorrectly(vs.v2)) {
							double maxVal = -vertInsertMetric.evaluate(vs.v2);
							if (ext > middle) {
								if (maxVal > maxValMiddle) {
									lower = middle;
									middle = ext;
									maxValMiddle = maxVal;
								} else {
									upper = ext;
								}
							} else {
								if (maxVal > maxValMiddle) {
									upper = middle;
									middle = ext;
									maxValMiddle = maxVal;
								} else {
									lower = ext;
								}
							}
							if (maxVal >= bestVal2) {
								bestl = l;
								candidates[l] = pt;
								bestVal2 = maxVal;
							}
						} else {
							break;
						}
					}
					// System.out.println("V2 LOWER "+lower+" UPPER "+upper+" MAX "+maxValMiddle);
				}
				if (bestl == -1 || bestn == -1) {
					surf.setVertex(vs.v2, oldp);
				} else {
					surf.setVertex(vs.v2, candidates[bestl]);
				}
				// System.out.println("INCORRECT WINDINGS V2 "+count);
			} else {
				totalAngle = 2 * Math.PI - totalAngle;
				Vector3f ptmp = p1;
				p1 = p2;
				p2 = ptmp;
				for (int l = 0; l < angleSteps; l++) {
					theta = totalAngle
							* (0.001 + 0.998 * l / (float) (angleSteps - 1));
					cos = (float) Math.cos(theta);
					sin = (float) Math.sin(theta);
					p0 = new Vector3f(p1.x * cos + p1.y * sin, -p1.x * sin
							+ p1.y * cos, 0);
					last = pnbrs2[stV2];
					tmin = 1E30;
					for (int i = 1; i < pnbrs2.length; i++) {
						nextId = (i + stV2) % pnbrs2.length;
						next = pnbrs2[nextId];
						t = GeometricUtilities.intersectionAngle(pivot, p0,
								last, next);
						last = next;
						if (t >= 0) {
							tmin = Math.min(t, tmin);
						}
						if (nextId == endV2)
							break;
					}
					if (tmin == 1E30)
						continue;
					SphericalArc arc = new SphericalArc(pivot, p0);
					double lower = tmin * 0.001;
					double upper = tmin * 0.999;
					double middle = lower + (upper - lower) / (1 + TAU);
					double ext;
					double maxValMiddle = 0;
					pt = arc.interpolate(middle);
					pt = GeometricUtilities.multMatrix(R, pt);
					surf.setVertex(vs.v2, pt);
					if (isNbhdWoundCorrectly(vs.v2)) {
						maxValMiddle = -vertInsertMetric.evaluate(vs.v2);
						if (maxValMiddle >= bestVal2) {
							bestl = l;
							candidates[l] = pt;
							bestVal2 = maxValMiddle;
						}
					} else {
						lower = upper;
					}
					// System.out.println("V2 INIT LOWER "+lower+" UPPER "+upper+" MAX "+maxValMiddle);
					while ((upper - lower) > 1E-5) {
						if (upper - middle > middle - lower) {
							ext = middle + (upper - middle) / (1 + TAU);
						} else {
							ext = lower + (middle - lower) / (1 + TAU);
						}
						pt = arc.interpolate(ext);
						pt = GeometricUtilities.multMatrix(R, pt);
						surf.setVertex(vs.v2, pt);
						if (isNbhdWoundCorrectly(vs.v2)) {
							double maxVal = -vertInsertMetric.evaluate(vs.v2);
							if (ext > middle) {
								if (maxVal > maxValMiddle) {
									lower = middle;
									middle = ext;
									maxValMiddle = maxVal;
								} else {
									upper = ext;
								}
							} else {
								if (maxVal > maxValMiddle) {
									upper = middle;
									middle = ext;
									maxValMiddle = maxVal;
								} else {
									lower = ext;
								}
							}
							if (maxVal >= bestVal2) {
								bestl = l;
								candidates[l] = pt;
								bestVal2 = maxVal;
							}
						} else {
							break;
						}
					}
					// System.out.println("V2 LOWER "+lower+" UPPER "+upper+" MAX "+maxValMiddle);
				}
				if (bestl == -1 || bestn == -1) {
					surf.setVertex(vs.v2, oldp);
				} else {
					surf.setVertex(vs.v2, candidates[bestl]);
				}
				// System.out.println("INCORRECT WINDINGS V2 "+count);
				// count = 0;
				totalAngle = 2 * Math.PI - totalAngle;
				p1 = p2;
				bestl = -1;
				bestn = -1;
				for (int l = 0; l < angleSteps; l++) {
					theta = totalAngle
							* (0.001 + 0.998 * l / (float) (angleSteps - 1));
					cos = (float) Math.cos(theta);
					sin = (float) Math.sin(theta);
					p0 = new Vector3f(p1.x * cos + p1.y * sin, -p1.x * sin
							+ p1.y * cos, 0);
					last = pnbrs1[stV1];
					tmin = 1E30;
					for (int i = 1; i < pnbrs1.length; i++) {
						nextId = (i + stV1) % pnbrs1.length;
						next = pnbrs1[nextId];
						t = GeometricUtilities.intersectionAngle(pivot, p0,
								last, next);
						last = next;
						if (t >= 0) {
							tmin = Math.min(t, tmin);
						}
						if (nextId == endV1)
							break;
					}
					if (tmin == 1E30)
						continue;
					SphericalArc arc = new SphericalArc(pivot, p0);
					double lower = tmin * 0.001;
					double upper = tmin * 0.999;
					double middle = lower + (upper - lower) / (1 + TAU);
					double ext;
					double maxValMiddle = 0;
					pt = arc.interpolate(middle);
					pt = GeometricUtilities.multMatrix(R, pt);
					surf.setVertex(vs.v1, pt);
					if (isNbhdWoundCorrectly(vs.v1)) {
						maxValMiddle = -vertInsertMetric.evaluate(vs.v1);
						if (maxValMiddle >= bestVal1) {
							bestl = l;
							candidates[l] = pt;
							bestVal1 = maxValMiddle;
						}
					} else {
						lower = upper;
					}
					// System.out.println("V1 INIT LOWER "+lower+" UPPER "+upper+" MAX "+maxValMiddle);
					while ((upper - lower) > 1E-5) {
						if (upper - middle > middle - lower) {
							ext = middle + (upper - middle) / (1 + TAU);
						} else {
							ext = lower + (middle - lower) / (1 + TAU);
						}
						pt = arc.interpolate(ext);
						pt = GeometricUtilities.multMatrix(R, pt);
						surf.setVertex(vs.v1, pt);
						if (isNbhdWoundCorrectly(vs.v1)) {
							double maxVal = -vertInsertMetric.evaluate(vs.v1);
							if (ext > middle) {
								if (maxVal > maxValMiddle) {
									lower = middle;
									middle = ext;
									maxValMiddle = maxVal;
								} else {
									upper = ext;
								}
							} else {
								if (maxVal > maxValMiddle) {
									upper = middle;
									middle = ext;
									maxValMiddle = maxVal;
								} else {
									lower = ext;
								}
							}
							if (maxVal >= bestVal1) {
								bestl = l;
								candidates[l] = pt;
								bestVal1 = maxVal;
							}
						} else {
							break;
						}
					}
					// System.out.println("V1 LOWER "+lower+" UPPER "+upper+" MAX "+maxValMiddle);
				}
				if (bestl == -1 || bestn == -1) {
					surf.setVertex(vs.v1, oldp);
				} else {
					surf.setVertex(vs.v1, candidates[bestl]);
				}
				// System.out.println("INCORRECT WINDINGS V1 "+count);
			}
		}
	}

	/**
	 * Maximum intersection time for ray and and polyline. The ray's origin is
	 * assumed to be at the origin.
	 * 
	 * @param vec
	 *            vector
	 * @param pnbrs
	 *            points for polyline
	 * @return
	 */
	protected double maxIntersectionTime(Vector2d vec, Point2d[] pnbrs) {
		double tmin = 1E30, t;
		Point2d next, org2, vec2 = new Point2d();
		Point2d org = new Point2d(0, 0);
		for (int i = 0; i < pnbrs.length; i++) {
			org2 = pnbrs[i];
			next = pnbrs[(i + 1) % pnbrs.length];
			if (org2 == null || next == null)
				continue;
			vec2.sub(next, org2);
			t = GeometricUtilities.intersectionTime(org, vec, org2, vec2);
			if (t >= 0) {
				tmin = Math.min(t, tmin);
			}
		}
		return tmin;
	}

	/**
	 * Calculate maximum intersection time between spherical ray and polyline on
	 * sphere. The ray's origin is assumed to be at the origin.
	 * 
	 * @param vec
	 *            vector direction for arc
	 * @param pnbrs
	 *            polyline on sphere
	 * @return
	 */
	protected double maxIntersectionSphericalTime(Vector3f vec, Vector3f[] pnbrs) {
		double tmin = 1E30, t;
		Vector3f next, org2;
		Vector3f org = new Vector3f(0, 0, 1);
		for (int i = 0; i < pnbrs.length; i++) {
			org2 = pnbrs[i];
			next = pnbrs[(i + 1) % pnbrs.length];
			if (org2 == null || next == null)
				continue;
			t = GeometricUtilities.intersectionAngle(org, vec, org2, next);
			if (t >= 0) {
				tmin = Math.min(t, tmin);
			}
		}
		return tmin;
	}

	/**
	 * Calculate min of max edge lengths
	 * 
	 * @author Blake Lucas
	 * 
	 */
	public class MinOfMaxEdgeLengthMetric implements VertexInsertionMetric {
		public double evaluate(int id) {
			Point3f oldp = surf.getVertex(id);
			Point3f p;
			int[] nbrs1 = neighborVertexVertexTable[id];
			double maxEdge = 0;
			for (int i = 0; i < nbrs1.length; i++) {
				p = surf.getVertex(nbrs1[i]);
				maxEdge = Math.max(maxEdge, p.distance(oldp));
			}
			return -maxEdge;
		}
	}

	/**
	 * Get minimum angle between neighboring vertices
	 * 
	 * @param id
	 *            vertex id
	 * @return minimum angle
	 */
	protected double minAngle(int id) {
		Point3f p = surf.getVertex(id);
		Point3f oldp = p;
		Vector3d pivot = new Vector3d(oldp);
		int[] nbrs1 = neighborVertexVertexTable[id];
		if (nbrs1.length == 0)
			return 1;
		Vector3d last = null, next = null;
		int st = 0;
		double minAngle = 7;
		for (int i = 0; i < nbrs1.length; i++) {
			p = surf.getVertex(nbrs1[i]);
			if (p.distance(oldp) > MIN_EDGE_LENGTH) {
				last = new Vector3d(p);
				st = i + 1;
				break;
			}
		}
		if (last == null) {
			return 1;
		}
		for (int i = 0; i < nbrs1.length; i++) {
			p = surf.getVertex(nbrs1[(i + st) % nbrs1.length]);
			if (p.distance(oldp) <= MIN_EDGE_LENGTH
					|| GeometricUtilities.distance(p, last) <= MIN_EDGE_LENGTH) {
				continue;
			}
			next = new Vector3d(p);
			minAngle = Math.min(minAngle, GeometricUtilities.sphericalAngle(
					pivot, last, next));
			last = next;
		}
		// System.out.println("MAX EDGE "+maxEdge+" MIN ANGLE "+minAngle);
		return minAngle;
	}

	/**
	 * Get winding number for a vertex. If a vertex neighbor has the same
	 * position as the pivot vertex, then that vertex is ignored when computing
	 * the winding number.
	 * 
	 * @param id
	 *            pivot vertex id
	 * @return winding number
	 */
	protected double windingNumber(int id) {
		Point3f p = surf.getVertex(id);
		Point3f oldp = p;
		Vector3d pivot = new Vector3d(p);
		int[] nbrs1 = neighborVertexVertexTable[id];
		if (nbrs1.length == 0)
			return 1;
		double angle = 0;
		Vector3d last = null, next = null;
		int st = 0;
		// int count=0;
		for (int i = 0; i < nbrs1.length; i++) {
			p = surf.getVertex(nbrs1[i]);
			if (GeometricUtilities.distance(p, pivot) > MIN_EDGE_LENGTH) {
				last = new Vector3d(p);
				st = i + 1;
				break;
			}
		}
		if (last == null) {
			return 1;
		}
		for (int i = 0; i < nbrs1.length; i++) {
			p = surf.getVertex(nbrs1[(i + st) % nbrs1.length]);
			if (GeometricUtilities.distance(p, oldp) <= MIN_EDGE_LENGTH
					|| GeometricUtilities.distance(p, last) <= MIN_EDGE_LENGTH) {
				continue;
			}
			next = new Vector3d(p);
			angle += GeometricUtilities.sphericalAngle(pivot, last, next);
			// count++;
			last = next;
		}
		double wind = angle / (2 * Math.PI);
		/*
		 * if(Math.abs(wind)<1E-5){ System.out.println("WINDING NUMBER "+wind+"
		 * "+count); }
		 */
		// Only one non-degenerate edge. Winding number should be 1
		return wind;
	}

	/**
	 * Return true if neighborhood is wound correctly
	 * 
	 * @param id
	 *            pivot vertex
	 * @return true if wounding correctly
	 */
	protected boolean isNbhdWoundCorrectly(int id) {
		// Check whether winding criterion is satsifed for this vertex and its
		// neighbors
		if (!isWoundCorrectly(id))
			return false;
		int[] nbrs = neighborVertexVertexTable[id];
		for (int i = 0; i < nbrs.length; i++) {
			if (!isWoundCorrectly(nbrs[i]))
				return false;
		}
		return true;
	}

	/**
	 * A vertex is wound correctly if its winding number is 0 or 1
	 * 
	 * @param id
	 *            pivot vertex
	 * @return true if wound correctly
	 */
	protected boolean isWoundCorrectly(int id) {
		// check whether the winding number of the vertexes is 1
		double wind = windingNumber(id);
		return (Math.abs(wind - 1) < 1E-4 || Math.abs(wind) < 1E-4);
	}

	/**
	 * Insert spherical edge into surface
	 * 
	 * @param vs
	 *            spherical edge collapse operation to reverse
	 */
	protected void insertSphericalEdge(SphericalEdgeCollapse vs) {
		int v1 = vs.v1;
		int v2 = vs.v2;
		int nbr1 = vs.nbr1;
		int nbr2 = vs.nbr2;
		int[] nbrs1 = neighborVertexVertexTable[v1];
		int nbrIndex1 = find(v1, nbr1);
		int nbrIndex2 = find(v1, nbr2);
		int[] newNbrs1 = new int[((nbrIndex2 > nbrIndex1) ? (nbrIndex2 - nbrIndex1)
				: (nbrIndex2 + nbrs1.length - nbrIndex1)) + 2];
		int[] newNbrs2 = new int[nbrs1.length - newNbrs1.length + 4];
		newNbrs1[newNbrs1.length - 1] = v2;
		for (int i = 0; i < newNbrs1.length - 1; i++) {
			newNbrs1[i] = nbrs1[(nbrIndex1 + i) % nbrs1.length];
		}
		newNbrs2[0] = v1;
		int v3;
		for (int i = 0; i < newNbrs2.length - 1; i++) {
			v3 = newNbrs2[i + 1] = nbrs1[(nbrIndex2 + i) % nbrs1.length];
			if (i == 0) {
				connect(v3, v2, find(v3, v1));
			} else if (i == newNbrs2.length - 2) {
				connect(v3, v2, (find(v3, v1) + 1)
						% neighborVertexVertexTable[v3].length);
			} else {
				int e = find(v3, v1);
				neighborVertexVertexTable[v3][e] = v2;
			}
		}
		neighborVertexVertexTable[v1] = newNbrs1;
		neighborVertexVertexTable[v2] = newNbrs2;
		surf.setVertex(vs.v2, surf.getVertex(vs.v1));
		if (isWoundCorrectly(vs.v1) && isWoundCorrectly(vs.v2)) {
			vertInsertOptMethod.insert(vs);
		}
	}

	/**
	 * Spherical curvature metric
	 * 
	 * @author Blake Lucas
	 * 
	 */
	protected class SphericalCurvatureVertexMetric implements HeapVertexMetric {
		public double evaluate(int id) {
			int[] nbrs = neighborVertexVertexTable[id];
			Point3f pivot = surf.getVertex(id);
			// If this a degenerate point, promote to top of queue
			// Degenerate points should be removed first
			double l = GeometricUtilities.length(pivot);
			if (l > 1.01 || l < 0.99) {
				// Point is not on sphere!
				System.err.println("POINT NOT ON SPHERE " + id + " " + pivot
						+ " " + l);
				return 2 * MIN_HEAP_VAL;
			}
			for (int i = 0; i < nbrs.length; i++) {
				if (pivot.distance(surf.getVertex(nbrs[i])) <= MIN_EDGE_LENGTH) {
					return MIN_HEAP_VAL;
				}
			}
			Vector3f c = getMeanCurvature(id);
			return -c.length();
		}

	}

	/**
	 * Regularize surface
	 * 
	 * @param vertStack
	 *            vertex stack to append regularization operations
	 */
	public void regularize(Stack<MeshTopologyOperation> vertStack) {
		int vertCount = neighborVertexVertexTable.length;
		ArrayList<Edge> edges = new ArrayList<Edge>(surf.getIndexCount() / 2);
		int eId = 0;
		for (int i = 0; i < vertCount; i++) {
			int[] nbrs = neighborVertexVertexTable[i];
			for (int nbr : nbrs) {
				if (i < nbr) {
					edges.add(new Edge(i, nbr, eId));
					eId++;
				}
			}
		}
		int edgeCount = eId;
		BinaryMinHeap heap = new BinaryMinHeap(edgeCount, edgeCount, 1, 1);
		int count = vertCount * 3;
		EdgeIndexed eox;
		EdgeSwap vs;
		boolean applied;
		int v1, v2;
		double lastMaxEdgeLength = 1E10;
		double currentMinHeapValue = 0;
		setLabel("Regularize");
		boolean updated = false;
		int lastCount;
		int maxCount = 0;
		setTotalUnits(100);
		do {
			lastCount = count;
			count = 0;
			updated = false;
			for (int i = 0; i < edgeCount; i++) {
				Edge e = edges.get(i);
				EdgeIndexed vox = new EdgeIndexed(heapMetric(e.v1, e.v2));
				vox.setPosition(i, e.v1, e.v2);
				heap.add(vox);
			}
			currentMinHeapValue = -((EdgeIndexed) heap.peek()).getValue();
			/*
			 * if (currentMaxEdgeLength >= lastMaxEdgeLength) break;
			 */
			System.out.println("HEAP METRIC " + currentMinHeapValue);
			// lastMaxEdgeLength = currentMaxEdgeLength;
			while (!heap.isEmpty()) {
				eox = (EdgeIndexed) heap.remove();
				eId = eox.getEdgeIndex();
				v1 = eox.getV1();
				v2 = eox.getV2();
				vs = createEdgeSwap();
				applied = vs.apply(v1, v2);
				if (applied) {
					vertStack.push(vs);
					Edge e = edges.get(eId);
					e.v1 = vs.v1;
					e.v2 = vs.v2;
					count++;
				}
			}
			System.out.println("EDGES SWAPED " + count);
			maxCount = Math.max(count, maxCount);
			setCompletedUnits((maxCount - count) / (double) maxCount);

		} while (count < lastCount);
		markCompleted();
	}

	/**
	 * Tessellate surface to specified amount
	 * 
	 * @param tessAmount
	 *            tesselation amount
	 */
	public void tessellate(double tessAmount) {
		System.out.println("TESSELLATE " + Math.round(100 * tessAmount) + " %");
		int vertCount = surf.getVertexCount();
		int tessPtsCount = (int) Math.ceil(tessAmount * vertCount);
		int count = 0;
		while (!vertStack.isEmpty() && count < tessPtsCount) {
			MeshTopologyOperation vs = vertStack.pop();
			vs.restore();
			count++;
			int id = vs.getRemovedVertexId();
		}
	}

	/**
	 * Decimate surface to specified metric threshold
	 * 
	 * @param metricThreshold
	 *            metric threshold
	 * @param maxPercent
	 *            maximum amount of decimation [0,1]
	 * @return number of decimated points
	 */
	protected int decimate(double metricThreshold, double maxPercent) {
		return decimate(metricThreshold, maxPercent, false, null);
	}

	/**
	 * Decimate surface to specified metric threshold
	 * 
	 * @param metricThreshold
	 *            metric threshold
	 * @param maxPercent
	 *            maximum amount of decimation [0,1]
	 * @param stopIfHarmonic
	 *            stop decimation if spherical parameterization is harmonic
	 * @param mask
	 *            mask to indicate vertices to ignore for decimation
	 * @return
	 */
	protected int decimate(double metricThreshold, double maxPercent,
			boolean stopIfHarmonic, int[] mask) {
		int vertCount = surf.getVertexCount();
		BinaryMinHeap heap = new BinaryMinHeap(vertCount, vertCount, 1, 1);
		int count = 0;
		for (int i = 0; i < vertCount; i++) {
			if (neighborVertexVertexTable[i].length > 0) {
				// Insert un-decimated points into heap
				VertexIndexed vox = new VertexIndexed(heapMetric(i));
				vox.setPosition(i);
				heap.add(vox);
			} else {
				// Count removed points
				count++;
			}
		}
		int v2;
		int step = 0;
		boolean removed = false;
		EdgeCollapse vs = null;
		VertexIndexed vox;
		int[] nbrs;
		double currentMetric = 0;
		int updateInterval = (int) (maxPercent * vertCount / 100);
		while (!heap.isEmpty() && (vertCount - count) > 4
				&& count < maxPercent * vertCount) {
			if (stopIfHarmonic && (count % updateInterval == 0) && isHarmonic())
				break;
			vox = (VertexIndexed) heap.remove();
			currentMetric = -vox.getValue();

			if (currentMetric < metricThreshold) {
				break;
			}
			v2 = vox.getRow();
			if (neighborVertexVertexTable[v2].length == 0) {
				// V2 has already been removed
				continue;
			}
			if (mask != null && mask[v2] == 0)
				continue;
			vs = createEdgeCollapse();
			if (mask != null) {
				if (mask[v2] == 0)
					continue;
				boolean cont = true;
				for (int nbr : neighborVertexVertexTable[v2]) {
					if (mask[nbr] == 0) {
						cont = false;
						break;
					}
				}
				if (!cont)
					continue;
			}
			removed = vs.apply(v2);

			if (removed) {

				vertStack.push(vs);
				// Recompute curvatures
				nbrs = vs.getChangedVerts();
				// Removing edge may change curvature of neighboring vertexes,
				// so update them as well
				for (int i = 0; i < nbrs.length; i++) {
					v2 = nbrs[i];
					vox = new VertexIndexed(heapMetric(v2));
					vox.setPosition(v2);
					heap.change(v2, 0, 0, vox);
				}
				step++;
				count++;
			} else {
				// Haven't hit this statement yet, but it maybe possible
				System.err
						.println("COULD NOT REMOVE VERTEX BECAUSE OF TOPOLOGY "
								+ v2 + " " + vox.getValue());
				printNeighbors(v2);
			}
		}
		System.out.println("DECIMATION " + 100 * vertStack.size()
				/ (float) vertCount + " % Max Heap Value " + currentMetric);
		heap.makeEmpty();
		return vertCount - count;
	}
}
