package edu.jhu.ece.iacl.algorithms.thickness;

import java.util.ArrayList;

import javax.vecmath.Matrix3d;
import javax.vecmath.Matrix3f;
import javax.vecmath.Point2d;
import javax.vecmath.Point3f;
import javax.vecmath.Point3i;
import javax.vecmath.Quat4f;

import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.SurfaceIntersector;
import edu.jhu.ece.iacl.algorithms.graphics.isosurf.IsoSurfaceOnGrid;
import edu.jhu.ece.iacl.algorithms.graphics.map.MapLabelsToSurface;
import edu.jhu.ece.iacl.algorithms.gvf.Gradient3d;
import edu.jhu.ece.iacl.algorithms.thickness.grid.EmbeddedNode;
import edu.jhu.ece.iacl.algorithms.thickness.grid.SimpleThicknessNode;
import edu.jhu.ece.iacl.algorithms.thickness.laplace.LaplaceSolveOnGrid;
import edu.jhu.ece.iacl.algorithms.topology.ConnectivityRule;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.data.BinaryMinHeap;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.jist.structures.image.MaskVolume;
import edu.jhu.ece.iacl.jist.structures.image.MaskVolume6;

/**
 * Super class for all thickness solver methods
 * 
 * @author Blake Lucas (bclucas@jhu.edu)
 * 
 */
public abstract class ThicknessSolver extends AbstractCalculation {
	protected ImageDataFloat thicknessVol;
	protected ImageDataFloat tangetField;
	protected ImageData innerVol;
	protected ImageData outerVol;
	protected ImageDataFloat levelSetVol;
	protected EmbeddedSurface innerMesh;
	protected EmbeddedSurface centralMesh;
	protected EmbeddedSurface outerMesh;
	protected BinaryMinHeap heap;
	protected double isoVal = 0;
	protected double lambda = 0.25;
	protected static double ANGLE_THRESHOLD = Math.PI / 3;
	public static final byte UNVISITED = 0;
	public static final byte VISITED = 1;
	public static final byte SOLVED = 2;
	public static final byte OUTSIDE_TAG = -1;
	public static final byte INSIDE_TAG = -2;
	public static final byte INSIDE_NARROW_BAND = EmbeddedNode.INSIDE_NARROW_BAND;
	public static final byte OUTSIDE_NARROW_BAND = EmbeddedNode.OUTSIDE_NARROW_BAND;
	public static final byte OUTSIDE_OUTER = EmbeddedNode.OUTSIDE_OUTER;
	public static final byte JUST_OUTSIDE_OUTER_BOUNDARY = EmbeddedNode.JUST_OUTSIDE_OUTER_BOUNDARY;
	public static final byte OUTER_BOUNDARY = EmbeddedNode.OUTER_BOUNDARY;
	public static final byte JUST_INSIDE = EmbeddedNode.JUST_INSIDE;
	public static final byte INSIDE = EmbeddedNode.INSIDE;
	public static final byte JUST_INSIDE_INNER_BOUNDARY = EmbeddedNode.JUST_INSIDE_INNER_BOUNDARY;
	public static final byte INNER_BOUNDARY = EmbeddedNode.INNER_BOUNDARY;
	public static final byte INSIDE_INNER = EmbeddedNode.INSIDE_INNER;
	protected double lagrangeDeltaT = 0.1f;
	protected int rows, cols, slices;
	protected float[][][][] tangetFieldMat;
	protected float[][][] thicknessMat;
	protected int[][][] tags;
	protected static final float BOUNDARY_TOL = 0.0f;
	protected static final double TANGENT_SUM_TOL = 1E-7;
	protected static final double MAX_LENGTH = 3;
	protected static final double MAX_DIST = 200;
	protected SimpleThicknessNode[] corticalData;
	protected MaskVolume neighborMask;
	protected static final int ilow = 0x000001;
	protected static final int ihigh = 0x000010;
	protected static final int jlow = 0x000100;
	protected static final int jhigh = 0x001000;
	protected static final int klow = 0x010000;
	protected static final int khigh = 0x100000;
	protected boolean ADJUST_BOUNDARY = false;
	protected static final Point3i[] planePoint1 = new Point3i[] {
			new Point3i(0, 0, 0), new Point3i(1, 0, 0), new Point3i(0, 0, 0),
			new Point3i(0, 1, 0), new Point3i(0, 0, 0), new Point3i(0, 0, 1) };
	protected static final Point3i[] planePoint2 = new Point3i[] {
			new Point3i(0, 1, 0), new Point3i(1, 1, 0), new Point3i(0, 0, 1),
			new Point3i(0, 1, 1), new Point3i(1, 0, 0), new Point3i(1, 0, 1) };
	protected static final Point3i[] planePoint3 = new Point3i[] {
			new Point3i(0, 1, 1), new Point3i(1, 1, 1), new Point3i(1, 0, 1),
			new Point3i(1, 1, 1), new Point3i(1, 1, 0), new Point3i(1, 1, 1) };
	protected static final Point3i[] planePoint4 = new Point3i[] {
			new Point3i(0, 0, 1), new Point3i(1, 0, 1), new Point3i(1, 0, 0),
			new Point3i(1, 1, 0), new Point3i(0, 1, 0), new Point3i(0, 1, 1) };

	/**
	 * Constructor
	 * 
	 * @param lambda
	 *            lambda factor to determine when to switch between euler and
	 *            lagrangian methods
	 * @param isoVal
	 *            iso-level for surface
	 * @param lagrangeDeltaT
	 *            step size for streamline tracing
	 */
	public ThicknessSolver(double lambda, double isoVal, double lagrangeDeltaT) {
		this.isoVal = isoVal;
		this.lambda = lambda;
		this.lagrangeDeltaT = lagrangeDeltaT;
		setLabel("Thickness Solver");
	}

	public ThicknessSolver() {
		setLabel("Thickness Solver");
	}

	public EmbeddedSurface getInnerSurface() {
		return innerMesh;
	}

	public EmbeddedSurface getOuterSurface() {
		return outerMesh;
	}

	public EmbeddedSurface getCentralSurface() {
		return centralMesh;
	}

	public EmbeddedSurface getQuarterSurface() {
		return null;
	}

	public EmbeddedSurface getThreeQuarterSurface() {
		return null;
	}

	/**
	 * Adjust levelset boundary so that computations are relative to augment
	 * level set, which is what the iso-surface algorithm uses.
	 * 
	 * @param innerVol inner level set
	 * @param outerVol outer level set
	 */
	public void adjustBoundary(ImageData innerVol, ImageData outerVol) {
		// If either surface is close to the zero level set,
		// move it away from the level set in a way that preserves the nesting
		// property
		int rows = innerVol.getRows();
		int cols = innerVol.getCols();
		int slices = innerVol.getSlices();
		int i, j, k;
		float DIST = 0.01f;
		for (i = 0; i < rows; i++) {
			for (j = 0; j < cols; j++) {
				for (k = 0; k < slices; k++) {
					double val1 = innerVol.getDouble(i, j, k) - isoVal;
					double val2 = outerVol.getDouble(i, j, k) - isoVal;
					/* The following is for TGDM purpose */
					if (val1 > -DIST && val1 <= 0)
						val1 = -DIST;
					else if (val1 < DIST && val1 > 0)
						val1 = DIST;
					//Case where val1==0 is already handled?
					if (val1 == 0) {
						//Always choose non-inclusive case
						innerVol.set(i, j, k, -DIST);
					} else {
						innerVol.set(i, j, k, val1);
					}
					//Case where val2==0 is already handled?
					if (val2 > -DIST && val2 <= 0) {
						val2 = -DIST;
					} else if (val2 < DIST && val2 > 0) {
						val2 = DIST;
					}
					if (val2 == 0) {
						//Always choose inclusive case for outer surface
						outerVol.set(i, j, k, DIST);
					} else {
						outerVol.set(i, j, k, val2);
					}
				}
			}
		}

	}
	/**
	 * Compute thickness for specified inner and outer level set
	 * @param innerVol
	 * @param outerVol
	 */
	public void solve(ImageData innerVol, ImageData outerVol) {
		int i, j, k;
		int ni, nj, nk;
		rows = innerVol.getRows();
		cols = innerVol.getCols();
		slices = innerVol.getSlices();
		this.innerVol = innerVol;
		this.outerVol = outerVol;
		// Adjust level set so that it is far enough away from the zero level
		// set
		if (ADJUST_BOUNDARY)
			adjustBoundary(innerVol, outerVol);
		neighborMask = new MaskVolume6();
		byte[] nbhdX = neighborMask.getNeighborsX();
		byte[] nbhdY = neighborMask.getNeighborsY();
		byte[] nbhdZ = neighborMask.getNeighborsZ();
		// Volume to store laplace's solution
		ImageDataFloat u = new ImageDataFloat(rows, cols, slices);
		float[][][] umat = u.toArray3d();
		// Volume to store thickness measurements
		thicknessVol = new ImageDataFloat(rows, cols, slices);
		thicknessMat = thicknessVol.toArray3d();

		tags = new int[rows][cols][slices];
		ArrayList<SimpleThicknessNode> data = new ArrayList<SimpleThicknessNode>();
		setTotalUnits(4);
		setLabel("Thickness Solver");
		for (i = 0; i < rows; i++) {
			for (j = 0; j < cols; j++) {
				for (k = 0; k < slices; k++) {
					double val1 = outerVol.getDouble(i, j, k);
					double val2 = innerVol.getDouble(i, j, k);
					// Initialize U volume for dirichlet boundary conditions
					if (val1 >= isoVal) {
						// outside outer boundary
						umat[i][j][k] = 1.0f;
					} else if (val2 < isoVal) {
						// Inside inner boundary
						umat[i][j][k] = 0;
					} else {
						// inside shell
						umat[i][j][k] = 0.5f;
					}
					if (val1 >= isoVal) {
						tags[i][j][k] = OUTSIDE_TAG;
					} else if (val2 < isoVal) {
						tags[i][j][k] = INSIDE_TAG;
					} else {
						tags[i][j][k] = data.size();
						data.add(new SimpleThicknessNode(i, j, k));
					}
					thicknessMat[i][j][k] = -1;
				}
			}
		}
		boolean nearBoundary = false;
		for (i = 0; i < rows; i++) {
			for (j = 0; j < cols; j++) {
				for (k = 0; k < slices; k++) {
					// Check if voxel is not in shell
					if (umat[i][j][k] == 1 || umat[i][j][k] == 0) {
						nearBoundary = false;
						for (int l = 0; l < nbhdX.length; l++) {
							ni = i + nbhdX[l];
							nj = j + nbhdY[l];
							nk = k + nbhdZ[l];
							if (ni < 0 || nj < 0 || nk < 0 || ni >= rows
									|| nj >= cols || nk >= slices)
								continue;
							// neighboring voxel is inside, this voxel must be
							// near the shell boundary
							if (umat[ni][nj][nk] == 0.5) {
								nearBoundary = true;
								break;
							}
						}
						if (!nearBoundary) {
							// if voxel is not near boundary,
							// then use the signed distance to initialize the
							// laplace solution
							if (umat[i][j][k] == 1) {
								umat[i][j][k] = outerVol.getFloat(i, j, k) + 1;
								if (umat[i][j][k] > 3) {
									umat[i][j][k] = 3;
								}
							} else {
								umat[i][j][k] = innerVol.getFloat(i, j, k);
								if (umat[i][j][k] < -2) {
									umat[i][j][k] = -2;
								}
							}
						}
					}
				}
			}
		}

		incrementCompletedUnits();
		corticalData = new SimpleThicknessNode[data.size()];
		data.toArray(corticalData);
		data = null;
		markCompleted();
		// Solve laplace's equations
		u = LaplaceSolveOnGrid.doSolve(u);
		incrementCompletedUnits();
		// Solve gradient of tangent field
		tangetField = Gradient3d.doSolve18(u);
		tangetField.normalize();
		incrementCompletedUnits();
		thicknessVol.setName(innerVol.getName() + "_thick");
		tangetField.setName(innerVol.getName() + "_tanget");
		tangetFieldMat = tangetField.toArray4d();
		umat = null;
		u = null;
		System.gc();
		markCompleted();
		computeThickness();
		updateVolumes();
		updateSurfaces();
		markCompleted();
	}

	private double meanInnerThickness;
	private double stdevOuterThickness;
	private double stdevInnerThickness;
	private double meanOuterThickness;

	/**
	 * Update surface statistics
	 * 
	 * @param mesh
	 */
	protected void updateSurfStats(EmbeddedSurface mesh) {
		int vertCount = mesh.getVertexCount();
		int count = 0;
		double sum = 0;
		double sqrs = 0;
		double val;
		for (int n = 0; n < vertCount; n++) {
			val = mesh.getThickness(n);
			if (Double.isInfinite(val) || Double.isNaN(val)) {
				System.err.println("INVALID THICKNESS " + val);
				continue;
			}
			count++;
			sum += val;
			sqrs += val * val;
		}
		double mean = (sum / count);
		double stdev = Math.sqrt((sqrs - sum * sum / count) / (count - 1));
		// Store statistics with appropriate mesh
		if (mesh == innerMesh) {
			meanInnerThickness = mean;
			stdevInnerThickness = stdev;
		} else {
			meanOuterThickness = mean;
			stdevOuterThickness = stdev;
		}
	}

	/**
	 * Map thicknesses to surface and update surface statistics
	 */
	protected void updateSurfaces() {
		IsoSurfaceOnGrid surfGen = new IsoSurfaceOnGrid();
		if (ADJUST_BOUNDARY)
			adjustBoundary(innerVol, outerVol);
		if (innerMesh == null) {
			innerMesh = (ADJUST_BOUNDARY) ? surfGen.solveNoMove(innerVol,
					ConnectivityRule.CONNECT_18_6, (float) isoVal, false)
					: surfGen.solveOriginal(innerVol,
							ConnectivityRule.CONNECT_18_6, (float) isoVal,
							false);
		}
		if (outerMesh == null) {
			outerMesh = (ADJUST_BOUNDARY) ? surfGen.solveNoMove(outerVol,
					ConnectivityRule.CONNECT_18_6, (float) isoVal, false)
					: surfGen.solveOriginal(outerVol,
							ConnectivityRule.CONNECT_18_6, (float) isoVal,
							false);
		}
		mapThickness(innerMesh);
		updateSurfStats(innerMesh);
		mapThickness(outerMesh);
		updateSurfStats(outerMesh);

		innerMesh.setName(innerVol.getName());
		outerMesh.setName(outerVol.getName());
	}

	protected void mapThickness(EmbeddedSurface mesh) {
		MapLabelsToSurface map = new MapLabelsToSurface(this);
		map.solve(mesh, thicknessVol);
	}

	protected abstract void computeThickness();

	/**
	 * Interpolate tangent direction
	 * 
	 * @param p
	 * @return
	 */
	protected Point3f interpolateTanget(Point3f p) {
		return new Point3f(interpolateTanget(p.x, p.y, p.z));
	}

	/**
	 * Interpolate length of streamline to surface from point
	 * 
	 * @param p
	 * @return
	 */
	protected double interpolateLength(Point3f p) {
		int i0, j0, k0, i1, j1, k1;
		double dx, dy, dz, hx, hy, hz;
		double x = p.x, y = p.y, z = p.z;
		j0 = (int) Math.floor(x);
		i0 = (int) Math.floor(y);
		k0 = (int) Math.floor(z);
		j1 = j0 + 1;
		i1 = i0 + 1;
		k1 = k0 + 1;
		if (j0 == x && i0 == y && k0 == z) {
			return getLength(j0, k0, i0);
		}
		if (j0 < 0 || j1 > (rows - 1) || i0 < 0 || i1 > (cols - 1) || k0 < 0
				|| k1 > (slices - 1)) {
			System.out.println("INVALID " + x + " " + y + " " + z + " " + rows
					+ " " + cols + " " + slices);
			return 0;
		} else {
			dx = x - j0;
			dy = y - i0;
			dz = z - k0;
			/* Introduce more variables to reduce computation */
			hx = 1.0 - dx;
			hy = 1.0 - dy;
			hz = 1.0 - dz;
			return (((getLength(j0, i0, k0) * hx + getLength(j1, i0, k0) * dx)
					* hy + (getLength(j0, i1, k0) * hx + getLength(j1, i1, k0)
					* dx)
					* dy)
					* hz + ((getLength(j0, i0, k1) * hx + getLength(j1, i0, k1)
					* dx)
					* hy + (getLength(j0, i1, k1) * hx + getLength(j1, i1, k1)
					* dx)
					* dy)
					* dz);
		}
	}

	/**
	 * Interpolate inner level set value
	 * 
	 * @param p
	 * @return
	 */
	protected double interpolateInnerLevelSet(Point3f p) {
		int i0, j0, k0, i1, j1, k1;
		double dx, dy, dz, hx, hy, hz;
		double x = p.x, y = p.y, z = p.z;
		j0 = (int) Math.floor(x);
		i0 = (int) Math.floor(y);
		k0 = (int) Math.floor(z);
		j1 = j0 + 1;
		i1 = i0 + 1;
		k1 = k0 + 1;
		if (j0 == x && i0 == y && k0 == z) {
			return innerVol.getFloat(j0, k0, i0);
		}
		if (j0 < 0 || j1 > (rows - 1) || i0 < 0 || i1 > (cols - 1) || k0 < 0
				|| k1 > (slices - 1)) {
			System.out.println("INVALID " + x + " " + y + " " + z + " " + rows
					+ " " + cols + " " + slices);
			return 0;
		} else {
			dx = x - j0;
			dy = y - i0;
			dz = z - k0;
			/* Introduce more variables to reduce computation */
			hx = 1.0 - dx;
			hy = 1.0 - dy;
			hz = 1.0 - dz;
			return (((innerVol.getFloat(j0, i0, k0) * hx + innerVol.getFloat(
					j1, i0, k0)
					* dx)
					* hy + (innerVol.getFloat(j0, i1, k0) * hx + innerVol
					.getFloat(j1, i1, k0)
					* dx)
					* dy)
					* hz + ((innerVol.getFloat(j0, i0, k1) * hx + innerVol
					.getFloat(j1, i0, k1)
					* dx)
					* hy + (innerVol.getFloat(j0, i1, k1) * hx + innerVol
					.getFloat(j1, i1, k1)
					* dx)
					* dy)
					* dz);
		}
	}
	/**
	 * Trilinear interpolation of level set
	 * @param x x position
	 * @param y y position 
	 * @param z z position
	 * @param phi volume to interpolate
	 * @return
	 */
	protected double interpolateLevelSet(double x, double y, double z,
			float[][][] phi) {
		int i0, j0, k0, i1, j1, k1;
		double dx, dy, dz, hx, hy, hz;
		j0 = (int) Math.floor(x);
		i0 = (int) Math.floor(y);
		k0 = (int) Math.floor(z);
		j1 = j0 + 1;
		i1 = i0 + 1;
		k1 = k0 + 1;

		if (j0 < 0 || j1 > (rows - 1) || i0 < 0 || i1 > (cols - 1) || k0 < 0
				|| k1 > (slices - 1)) {
			System.out.println("INVALID " + x + " " + y + " " + z + " " + rows
					+ " " + cols + " " + slices);
			return 0;
		} else {
			dx = x - j0;
			dy = y - i0;
			dz = z - k0;
			/* Introduce more variables to reduce computation */
			hx = 1.0 - dx;
			hy = 1.0 - dy;
			hz = 1.0 - dz;
			return (((phi[j0][i0][k0] * hx + phi[j1][i0][k0] * dx) * hy + (phi[j0][i1][k0]
					* hx + phi[j1][i1][k0] * dx)
					* dy)
					* hz + ((phi[j0][i0][k1] * hx + phi[j1][i0][k1] * dx) * hy + (phi[j0][i1][k1]
					* hx + phi[j1][i1][k1] * dx)
					* dy)
					* dz);
		}
	}

	/**
	 * Interpolate outer level set
	 * 
	 * @param p
	 * @return
	 */
	protected double interpolateOuterLevelSet(Point3f p) {
		int i0, j0, k0, i1, j1, k1;
		double dx, dy, dz, hx, hy, hz;
		double x = p.x, y = p.y, z = p.z;
		j0 = (int) Math.floor(x);
		i0 = (int) Math.floor(y);
		k0 = (int) Math.floor(z);
		j1 = j0 + 1;
		i1 = i0 + 1;
		k1 = k0 + 1;
		if (j0 == x && i0 == y && k0 == z) {
			return outerVol.getFloat(j0, k0, i0);
		}
		if (j0 < 0 || j1 > (rows - 1) || i0 < 0 || i1 > (cols - 1) || k0 < 0
				|| k1 > (slices - 1)) {
			System.out.println("INVALID " + x + " " + y + " " + z + " " + rows
					+ " " + cols + " " + slices);
			return 0;
		} else {
			dx = x - j0;
			dy = y - i0;
			dz = z - k0;
			/* Introduce more variables to reduce computation */
			hx = 1.0 - dx;
			hy = 1.0 - dy;
			hz = 1.0 - dz;
			return (((outerVol.getFloat(j0, i0, k0) * hx + outerVol.getFloat(
					j1, i0, k0)
					* dx)
					* hy + (outerVol.getFloat(j0, i1, k0) * hx + outerVol
					.getFloat(j1, i1, k0)
					* dx)
					* dy)
					* hz + ((outerVol.getFloat(j0, i0, k1) * hx + outerVol
					.getFloat(j1, i0, k1)
					* dx)
					* hy + (outerVol.getFloat(j0, i1, k1) * hx + outerVol
					.getFloat(j1, i1, k1)
					* dx)
					* dy)
					* dz);
		}
	}

	/**
	 * Interpolate correspondence
	 * 
	 * @param p
	 * @return
	 */
	public Point3f interpolateCoorespondence(Point3f p) {
		int i0, j0, k0, i1, j1, k1;
		double dx, dy, dz, hx, hy, hz;
		double x = p.x, y = p.y, z = p.z;

		j0 = (int) Math.floor(x);
		i0 = (int) Math.floor(y);
		k0 = (int) Math.floor(z);
		j1 = j0 + 1;
		i1 = i0 + 1;
		k1 = k0 + 1;
		if (j0 < 0 || j1 > (rows - 1) || i0 < 0 || i1 > (cols - 1) || k0 < 0
				|| k1 > (slices - 1)) {
			System.out.println("INVALID " + x + " " + y + " " + z + " " + rows
					+ " " + cols + " " + slices);
			return getData(j0, i0, k0).getCorrespodence();
		} else {
			dx = x - j0;
			dy = y - i0;
			dz = z - k0;
			/* Introduce more variables to reduce computation */
			hx = 1.0 - dx;
			hy = 1.0 - dy;
			hz = 1.0 - dz;
			Point3f result = new Point3f();
			Point3f p1 = getCorrespondence(j0, i0, k0);
			Point3f p2 = getCorrespondence(j1, i0, k0);
			Point3f p3 = getCorrespondence(j0, i1, k0);
			Point3f p4 = getCorrespondence(j1, i1, k0);
			Point3f p5 = getCorrespondence(j0, i0, k1);
			Point3f p6 = getCorrespondence(j1, i0, k1);
			Point3f p7 = getCorrespondence(j0, i1, k1);
			Point3f p8 = getCorrespondence(j1, i1, k1);
			result.x = (float) (((p1.x * hx + p2.x * dx) * hy + (p3.x * hx + p4.x
					* dx)
					* dy)
					* hz + ((p5.x * hx + p6.x * dx) * hy + (p7.x * hx + p8.x
					* dx)
					* dy)
					* dz);
			result.y = (float) (((p1.y * hx + p2.y * dx) * hy + (p3.y * hx + p4.y
					* dx)
					* dy)
					* hz + ((p5.y * hx + p6.y * dx) * hy + (p7.y * hx + p8.y
					* dx)
					* dy)
					* dz);
			result.z = (float) (((p1.z * hx + p2.z * dx) * hy + (p3.z * hx + p4.z
					* dx)
					* dy)
					* hz + ((p5.z * hx + p6.z * dx) * hy + (p7.z * hx + p8.z
					* dx)
					* dy)
					* dz);
			return result;
		}
	}

	/**
	 * Interpolate thickness at point
	 * 
	 * @param x
	 * @param y
	 * @param z
	 * @return
	 */
	public double interpolateThickness(double x, double y, double z) {
		int i0, j0, k0, i1, j1, k1;
		double dx, dy, dz, hx, hy, hz;
		j0 = (int) Math.floor(x);
		i0 = (int) Math.floor(y);
		k0 = (int) Math.floor(z);
		j1 = j0 + 1;
		i1 = i0 + 1;
		k1 = k0 + 1;

		if (j0 < 0 || j1 > (rows - 1) || i0 < 0 || i1 > (cols - 1) || k0 < 0
				|| k1 > (slices - 1)) {
			System.out.println("INVALID " + x + " " + y + " " + z + " " + rows
					+ " " + cols + " " + slices);
			return 0;
		} else {
			dx = x - j0;
			dy = y - i0;
			dz = z - k0;
			/* Introduce more variables to reduce computation */
			hx = 1.0 - dx;
			hy = 1.0 - dy;
			hz = 1.0 - dz;
			return (((thicknessMat[j0][i0][k0] * hx + thicknessMat[j1][i0][k0]
					* dx)
					* hy + (thicknessMat[j0][i1][k0] * hx + thicknessMat[j1][i1][k0]
					* dx)
					* dy)
					* hz + ((thicknessMat[j0][i0][k1] * hx + thicknessMat[j1][i0][k1]
					* dx)
					* hy + (thicknessMat[j0][i1][k1] * hx + thicknessMat[j1][i1][k1]
					* dx)
					* dy)
					* dz);
		}
	}

	/**
	 * Convert tangential vectors to quaternions
	 */
	protected Quat4f[][][] quaternions = null;

	public void convertToQuaternions() {
		if (quaternions != null)
			return;
		quaternions = new Quat4f[rows][cols][slices];
		for (int i = 0; i < rows; i++) {
			for (int j = 0; j < cols; j++) {
				for (int k = 0; k < slices; k++) {
					Point3f pt = new Point3f(tangetFieldMat[i][j][k]);
					if (pt.x * pt.x + pt.y * pt.y + pt.z * pt.z != 0) {
						Point2d r = GeometricUtilities.toUnitSpherical(pt);
						Matrix3d m = GeometricUtilities
								.rotateMatrix3d(r.x, r.y);
						Quat4f rot = new Quat4f();
						rot.set(m);
						quaternions[i][j][k] = rot;
					}
				}
			}
		}
	}

	/**
	 * Interpolate tangential field using quaternions
	 * 
	 * @param p
	 * @return
	 */
	protected Point3f interpolateTangetQuat(Point3f p) {
		int i0, j0, k0, i1, j1, k1;
		float dx, dy, dz;
		j0 = (int) Math.floor(p.x);
		i0 = (int) Math.floor(p.y);
		k0 = (int) Math.floor(p.z);
		j1 = j0 + 1;
		i1 = i0 + 1;
		k1 = k0 + 1;
		dx = p.x - j0;
		dy = p.y - i0;
		dz = p.z - k0;
		Quat4f q12 = new Quat4f();
		Quat4f q34 = new Quat4f();
		Quat4f q1234 = new Quat4f();

		Quat4f q56 = new Quat4f();
		Quat4f q78 = new Quat4f();
		Quat4f q5678 = new Quat4f();
		Quat4f q = new Quat4f();

		Quat4f q1 = quaternions[j0][i0][k0];
		Quat4f q2 = quaternions[j0][i1][k0];

		Quat4f q3 = quaternions[j1][i0][k0];
		Quat4f q4 = quaternions[j1][i1][k0];

		Quat4f q5 = quaternions[j0][i0][k1];
		Quat4f q6 = quaternions[j0][i1][k1];

		Quat4f q7 = quaternions[j1][i0][k1];
		Quat4f q8 = quaternions[j1][i1][k1];

		q12.interpolate(q1, q2, dy);
		q34.interpolate(q3, q4, dy);
		q1234.interpolate(q12, q34, dx);

		q56.interpolate(q5, q6, dy);
		q78.interpolate(q7, q8, dy);
		q5678.interpolate(q56, q78, dx);

		q.interpolate(q1234, q5678, dz);

		Matrix3f m = new Matrix3f();
		m.set(q);
		Point3f result = new Point3f();
		m.transform(new Point3f(1, 0, 0), result);
		return result;
	}

	/**
	 * Interpolate tangential field using linear interpolation
	 * 
	 * @param x
	 * @param y
	 * @param z
	 * @return
	 */
	public float[] interpolateTanget(double x, double y, double z) {
		int i0, j0, k0, i1, j1, k1;
		double dx, dy, dz, hx, hy, hz;
		j0 = (int) Math.floor(x);
		i0 = (int) Math.floor(y);
		k0 = (int) Math.floor(z);
		j1 = j0 + 1;
		i1 = i0 + 1;
		k1 = k0 + 1;
		if (j0 < 0 || j1 > (rows - 1) || i0 < 0 || i1 > (cols - 1) || k0 < 0
				|| k1 > (slices - 1)) {
			System.out.println("INVALID " + x + " " + y + " " + z + " " + rows
					+ " " + cols + " " + slices);
			j0 = (int) Math.min(Math.max(j0, 0), rows - 1);
			i0 = (int) Math.min(Math.max(i0, 0), cols - 1);
			k0 = (int) Math.min(Math.max(k0, 0), slices - 1);

			float[] vec = new float[3];
			vec[0] = tangetFieldMat[j0][i0][k0][0];
			vec[1] = tangetFieldMat[j0][i0][k0][1];
			vec[2] = tangetFieldMat[j0][i0][k0][2];
			return vec;
		} else {
			dx = x - j0;
			dy = y - i0;
			dz = z - k0;
			/* Introduce more variables to reduce computation */
			hx = 1.0 - dx;
			hy = 1.0 - dy;
			hz = 1.0 - dz;
			float[] vec = new float[3];
			vec[0] = (float) (((tangetFieldMat[j0][i0][k0][0] * hx + tangetFieldMat[j1][i0][k0][0]
					* dx)
					* hy + (tangetFieldMat[j0][i1][k0][0] * hx + tangetFieldMat[j1][i1][k0][0]
					* dx)
					* dy)
					* hz + ((tangetFieldMat[j0][i0][k1][0] * hx + tangetFieldMat[j1][i0][k1][0]
					* dx)
					* hy + (tangetFieldMat[j0][i1][k1][0] * hx + tangetFieldMat[j1][i1][k1][0]
					* dx)
					* dy)
					* dz);
			vec[1] = (float) (((tangetFieldMat[j0][i0][k0][1] * hx + tangetFieldMat[j1][i0][k0][1]
					* dx)
					* hy + (tangetFieldMat[j0][i1][k0][1] * hx + tangetFieldMat[j1][i1][k0][1]
					* dx)
					* dy)
					* hz + ((tangetFieldMat[j0][i0][k1][1] * hx + tangetFieldMat[j1][i0][k1][1]
					* dx)
					* hy + (tangetFieldMat[j0][i1][k1][1] * hx + tangetFieldMat[j1][i1][k1][1]
					* dx)
					* dy)
					* dz);
			vec[2] = (float) (((tangetFieldMat[j0][i0][k0][2] * hx + tangetFieldMat[j1][i0][k0][2]
					* dx)
					* hy + (tangetFieldMat[j0][i1][k0][2] * hx + tangetFieldMat[j1][i1][k0][2]
					* dx)
					* dy)
					* hz + ((tangetFieldMat[j0][i0][k1][2] * hx + tangetFieldMat[j1][i0][k1][2]
					* dx)
					* hy + (tangetFieldMat[j0][i1][k1][2] * hx + tangetFieldMat[j1][i1][k1][2]
					* dx)
					* dy)
					* dz);

			double mag = Math.sqrt(vec[0] * vec[0] + vec[1] * vec[1] + vec[2]
					* vec[2]);
			if (mag > 0) {
				vec[0] /= mag;
				vec[1] /= mag;
				vec[2] /= mag;
			} else {
				System.out.println("MAGNITUDE ZERO " + x + "," + y + "," + z);
			}

			return vec;
		}
	}

	/**
	 * Get marching label for voxel
	 * 
	 * @param i
	 * @param j
	 * @param k
	 * @return
	 */
	protected byte getLabel(int i, int j, int k) {
		if (i < 0 || j < 0 || k < 0 || i >= rows || j >= cols || k >= slices)
			return SOLVED;
		int t = tags[i][j][k];
		if (t < 0)
			return SOLVED;
		else
			return corticalData[t].label;
	}

	/**
	 * Get streamline length for voxel
	 * 
	 * @param i
	 * @param j
	 * @param k
	 * @return
	 */
	protected double getLength(int i, int j, int k) {
		if (i < 0 || j < 0 || k < 0 || i >= rows || j >= cols || k >= slices)
			return 0;
		int t = tags[i][j][k];
		if (t < 0) {
			return 0;
		} else
			return corticalData[t].getLength();
	}

	/**
	 * Get correspondence point at voxel
	 * 
	 * @param i
	 * @param j
	 * @param k
	 * @return
	 */
	protected Point3f getCorrespondence(int i, int j, int k) {
		if (i < 0 || j < 0 || k < 0 || i >= rows || j >= cols || k >= slices)
			return null;
		int t = tags[i][j][k];
		if (t < 0)
			return new Point3f(i, j, k);
		else
			return corticalData[t].getCorrespodence();
	}

	/**
	 * Get data storage node for voxel
	 * 
	 * @param i
	 * @param j
	 * @param k
	 * @return
	 */
	protected SimpleThicknessNode getData(int i, int j, int k) {
		if (i < 0 || j < 0 || k < 0 || i >= rows || j >= cols || k >= slices)
			return null;
		int t = tags[i][j][k];
		if (t < 0) {
			return null;
		} else {
			return corticalData[t];
		}
	}

	/**
	 * Set marching label for voxel
	 * 
	 * @param i
	 * @param j
	 * @param k
	 * @param label
	 */
	protected void setLabel(int i, int j, int k, byte label) {
		int t = tags[i][j][k];
		if (t < 0)
			System.err.println("CANNOT SET LABEL " + i + "," + j + "," + k
					+ " : " + label);
		else {
			corticalData[t].label = label;
		}
	}

	/**
	 * Set data node for voxel
	 * 
	 * @param i
	 * @param j
	 * @param k
	 * @param dat
	 */
	protected void setData(int i, int j, int k, SimpleThicknessNode dat) {
		int t = tags[i][j][k];
		if (t < 0)
			System.err.println("CANNOT SET DATA " + i + "," + j + "," + k
					+ " : " + dat.label);
		else {
			corticalData[t] = dat;
		}
	}

	protected void setAllLabels(byte label) {
		for (SimpleThicknessNode cd : corticalData) {
			cd.label = label;
		}
	}

	protected double minThickness;
	protected double maxThickness;
	protected double meanThickness;
	protected double stdevThickness;

	/**
	 * Update thickness statistics, level set values, and tangential field
	 * values
	 */
	protected void updateVolumes() {
		SimpleThicknessNode cd;
		int n;
		double sqrs = 0;
		double sum = 0;
		double count = 0;
		minThickness = 1E30;
		maxThickness = -1E30;
		levelSetVol = new ImageDataFloat(rows, cols, slices);
		levelSetVol.setName(innerVol.getName() + "_levelset");
		float[][][] levelSetMat = levelSetVol.toArray3d();
		for (int i = 0; i < rows; i++) {
			for (int j = 0; j < cols; j++) {
				for (int k = 0; k < slices; k++) {
					// float[] v=tangetFieldMat[i][j][k];
					// float
					// mag=(float)Math.sqrt(v[0]*v[0]+v[1]*v[1]+v[2]*v[2]);
					// tangetFieldMat[i][j][k]=new
					// float[]{mag,(float)Math.atan2(v[1],v[0]),(float)Math.acos(v[2]/mag)};
					thicknessMat[i][j][k] = -1;
				}
			}
		}
		for (n = 0; n < corticalData.length; n++) {
			cd = corticalData[n];
			float thick = (float) (cd.getThickness());
			if (Float.isInfinite(thick) || Float.isNaN(thick)) {
				System.err.println("INVALID THICKNESS " + thick);
				continue;
			}
			// Point3f p=cd.getCorrespodence();
			// tangetFieldMat[cd.i][cd.j][cd.k]=new float[]{p.x,p.y,p.z};
			thicknessMat[cd.i][cd.j][cd.k] = thick;
			levelSetMat[cd.i][cd.j][cd.k] = (float) cd.getNormalizedLength();
			sqrs += (thick) * (thick);
			sum += thick;
			minThickness = Math.min(minThickness, thick);
			maxThickness = Math.max(maxThickness, thick);
			count++;
		}
		meanThickness = (sum / count);
		stdevThickness = Math.sqrt((sqrs - sum * sum / count) / (count - 1));
		double expected = Math.round(meanThickness);
		double error = 0;
		for (n = 0; n < corticalData.length; n++) {
			cd = corticalData[n];
			float thick = (float) (cd.getThickness());
			error += Math.pow(thick - expected, 2);
		}
		error = Math.sqrt(error / count);
	}

	public ImageData getThickness() {
		return thicknessVol;
	}

	public ImageData getTangetField() {
		return tangetField;
	}

	public ImageData getLevelSet() {
		return levelSetVol;
	}

	public double getMaxThickness() {
		return maxThickness;
	}

	public double getMeanThickness() {
		return meanThickness;
	}

	public double getMinThickness() {
		return minThickness;
	}

	public double getStdevThickness() {
		return stdevThickness;
	}

	public double getMeanInnerThickness() {
		return meanInnerThickness;
	}

	public double getMeanOuterThickness() {
		return meanOuterThickness;
	}

	public double getStdevInnerThickness() {
		return stdevInnerThickness;
	}

	public double getStdevOuterThickness() {
		return stdevOuterThickness;
	}

}
