package edu.jhmi.rad.medic.algorithms;


import edu.jhmi.rad.medic.methods.AtlasEMPreprocess;
import edu.jhmi.rad.medic.methods.AtlasEMSegmentation;
import edu.jhmi.rad.medic.methods.DemonToadDeformableAtlas;
import edu.jhmi.rad.medic.methods.InhomogeneityCorrection2;
import edu.jhmi.rad.medic.utilities.*;


import gov.nih.mipav.model.algorithms.AlgorithmBase;
import gov.nih.mipav.model.structures.*;
import gov.nih.mipav.model.file.FileInfoBase;
import gov.nih.mipav.view.*;
import gov.nih.mipav.view.dialogs.*;

import java.io.*;

public class AlgorithmAtlasEMSegmentation extends AlgorithmBase{ 
	

	
    private ModelImage  		destImage[];
    private ModelImage  		srcImage[];
    private int         		destNum=0;
    private String      		resultOutput;
    private ViewUserInterface   userInterface;
		
    // image size
	private int 		classes; 		// number of classes
	private int 		nx,ny,nz,nc;    // original image dimensions
	private int 		ntx,nty,ntz; 	// original template dimensions
	private	float		rx,ry,rz;		// image resolutions
	private	float		rtx,rty,rtz;	// template resolutions
	private int			orient;			// image orientation
	private int			orx,ory,orz;	// image axis orientation
	private	String[]	modality;		// image modality
	private	String		priorMode;		// variants of the algorithm
	
	
    // segmentation parameters
    private	String				atlasName;
    private	DemonToadDeformableAtlas		atlas=null;
    

	//private	boolean		segment = true;
	private float   	smoothing;
	private int 		iterations;
	private float   	maxDistance;
	private float		bgthreshold;
	
	private	String		centroidMode;
	private	float		centroidSmoothness = 0.5f;
	private	String		registrationMode = "rigid";
	
	private	boolean		register = true;
	private	int			levels;
	private	int			initRegIter;
	private	int			mainRegIter;
	private float		demonsSmoothing;
	private float		demonsScale;
	
	private	boolean		alignShapes = false;
	private boolean		equalPriors = false;
	private	boolean		outputAtlas = true;
	
	
    private	boolean		correctInhomogeneity = false;
	private	int			polynomialDegree;
	private	float		splineKernel;
	private	int			correctionMethod;
	private float   	lowEratio = 0.1f;
	
	private boolean     outputField = false;
	private boolean		verbose = true;
	private boolean     debug   = true;
	
	
	private static final String cvsversion = "$Revision: 1.2 $";
	public static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");

	public AlgorithmAtlasEMSegmentation() {
	}
	public String get_version() {
		return revnum;
	}
	/**
    *	Constructor for 3D images in which changes are placed in a predetermined destination image.
    *   @param destImg_      Image model where result image is to stored.
    *   @param srcImg_       Source image model.
    */
	public AlgorithmAtlasEMSegmentation(ModelImage[] srcImg_, int nInput_,String[] imgModal_,
								String aName_, DemonToadDeformableAtlas atlas_, String segOutput_, 
								float smooth_, int nIterMax_, float distMax_, float bgth_,
								String regMode_,
								boolean register_, 
								int lev_, int init_, int main_, 
								float dSmooth_, float dScale_,
								boolean correct_, boolean outputField_,
								String correctType_,
								int poly_, float kernel_,
								String prior_
								) {
        super(null, srcImg_[0]);
		userInterface = ViewUserInterface.getReference();
		srcImage = srcImg_;
        nc = nInput_;
        modality = imgModal_;
		atlasName = aName_;
		atlas = atlas_;
		resultOutput = segOutput_;
		smoothing = smooth_;
		iterations = Numerics.abs(nIterMax_);
        maxDistance = distMax_;
		lowEratio = (float)Math.sqrt(maxDistance);
		bgthreshold = bgth_;
		priorMode = prior_;
		registrationMode = regMode_;
		register = register_;
		levels = lev_;
		initRegIter = init_;
		mainRegIter = main_;
		demonsSmoothing = dSmooth_;
		demonsScale = dScale_;
		correctInhomogeneity = correct_;
		outputField = outputField_;
		if (correctInhomogeneity) {
			polynomialDegree = poly_;
			splineKernel = kernel_;
			if (correctType_.equals("Splines")) correctionMethod = InhomogeneityCorrection2.SPLINES;
			else correctionMethod = InhomogeneityCorrection2.CHEBYSHEV;
		} else {
			polynomialDegree = 0;
			splineKernel = 0.0f;
			correctionMethod = InhomogeneityCorrection2.CHEBYSHEV;
		}

	}


    /**
    *	Prepares this class for destruction.
    */
	public void finalize(){
	    destImage   = null;
	    srcImage    = null;
        super.finalize();
        System.gc();
	}

  
	
    /**
    *   Starts the segmentation.
    */
	public void runAlgorithm() {
   
		if (srcImage  == null) {
            displayError("Source Image is null");
            return;
        }
        
		// start the timer to compute the elapsed time
        setStartTime();

		//constructLog(true);
		
		float[][] buffer = null;
		if (srcImage[0].getNDims() == 2){
			try {
				// image length is length in 2 dims
				int length = srcImage[0].getExtents()[0]  * srcImage[0].getExtents()[1];
				 buffer = new float[nc][];
				for (int t=0;t<nc;t++) {
					buffer[t] = new float[length];
					srcImage[t].exportData(0,length, buffer[t]); // locks and releases lock
				}
			}
			catch (IOException error) {
				buffer = null;
				errorCleanUp("Segmentation reports: source image locked", true);
				return;
			}
			catch (OutOfMemoryError e){
				buffer = null;
				errorCleanUp("Segmentation reports: out of memory", true);
				return;
			}
			// init dimensions
			nx = srcImage[0].getExtents()[0];
			ny = srcImage[0].getExtents()[1];
			nz = 1;
			
			ntx = atlas.getImageDim()[0];
			nty = atlas.getImageDim()[1];
			ntz = 1;
			
			rx = srcImage[0].getFileInfo()[0].getResolutions()[0];
			ry = srcImage[0].getFileInfo()[0].getResolutions()[1];
			rz = 1.0f;
			
			// check for non-mm units
			if (srcImage[0].getFileInfo()[0].getUnitsOfMeasure()[0]==FileInfoBase.CENTIMETERS) rx = 10.0f*rx;
			if (srcImage[0].getFileInfo()[0].getUnitsOfMeasure()[1]==FileInfoBase.CENTIMETERS) ry = 10.0f*ry;
			
			rtx = atlas.getImageRes()[0];
			rty = atlas.getImageRes()[1];
			rtz = 1.0f;
			
			orient = srcImage[0].getImageOrientation();
			orx = srcImage[0].getFileInfo()[0].getAxisOrientation(0);
			ory = srcImage[0].getFileInfo()[0].getAxisOrientation(1);
			orz = FileInfoBase.UNKNOWN_ORIENT;
			
			// main segmentation
			calcSegmentation(buffer);
			
		}
		else if (srcImage[0].getNDims() > 2) {
		   try {
				// image length is length in 3 dims
				int length = srcImage[0].getExtents()[0] *
							 srcImage[0].getExtents()[1] *
							 srcImage[0].getExtents()[2];
									 buffer = new float[nc][];
				for (int t=0;t<nc;t++) {
					buffer[t] = new float[length];
					srcImage[t].exportData(0,length, buffer[t]); // locks and releases lock
				}
				fireProgressStateChanged(srcImage[0].getImageName(), "Processing image ...");
			}
			catch (IOException error) {
				buffer = null;
				errorCleanUp("Segmentation: source image locked", true);
				return;
			}
			catch (OutOfMemoryError e){
				buffer = null;
				errorCleanUp("Segmentation: Out of memory creating process buffer", true);
				return;
			}
			// init dimensions
			nx = srcImage[0].getExtents()[0];
			ny = srcImage[0].getExtents()[1];
			nz = srcImage[0].getExtents()[2];
			
			ntx = atlas.getImageDim()[0];
			nty = atlas.getImageDim()[1];
			ntz = atlas.getImageDim()[2];
			
			rx = srcImage[0].getFileInfo()[0].getResolutions()[0];
			ry = srcImage[0].getFileInfo()[0].getResolutions()[1];
			rz = srcImage[0].getFileInfo()[0].getResolutions()[2];
			
			// check for non-mm units
			if (srcImage[0].getFileInfo()[0].getUnitsOfMeasure()[0]==FileInfoBase.CENTIMETERS) rx = 10.0f*rx;
			if (srcImage[0].getFileInfo()[0].getUnitsOfMeasure()[1]==FileInfoBase.CENTIMETERS) ry = 10.0f*ry;
			if (srcImage[0].getFileInfo()[0].getUnitsOfMeasure()[2]==FileInfoBase.CENTIMETERS) rz = 10.0f*rz;
			
			rtx = atlas.getImageRes()[0];
			rty = atlas.getImageRes()[1];
			rtz = atlas.getImageRes()[2];
			
			orient = srcImage[0].getImageOrientation();
			orx = srcImage[0].getFileInfo()[0].getAxisOrientation(0);
			ory = srcImage[0].getFileInfo()[0].getAxisOrientation(1);
			orz = srcImage[0].getFileInfo()[0].getAxisOrientation(2);
			
			classes = atlas.getNumber();
				
			// main segmentation
			calcSegmentation(buffer);
			
        }

        // compute the elapsed time
        computeElapsedTime();
    } // end run()

    /**
    *	produces the segmentation of the input image
    */
    private void calcSegmentation(float img[][]) {
		byte[][][]   	classification = null; 
		float[][]		centroids = null;
		float[][][][] 	memberships = null;
        float[][][][] 	field = null;
    	float[]     buffer;
		byte[]		bytebuffer;
		AtlasEMSegmentation	segmentation=null;
		AtlasEMPreprocess		preprocess=null;
		InhomogeneityCorrection2[]	correction=null;
		float		 Eprev, Eratio;
		float[]     energy;
	

		boolean stop;
		
		if (verbose) {
			userInterface.setGlobalDataText("Adaptive EM\n");
			userInterface.setGlobalDataText(getParameterText());
		}
		
		fireProgressStateChanged(srcImage[0].getImageName(), "Initialization");
		
		
		//---------------------------------------/
		//-          MAIN SEGMENTATION          -/
		//---------------------------------------/
		
		// record time
		long start_time = System.currentTimeMillis();
		

        memberships = null;
        classification = null;
		
		// pre-processing
		preprocess = new AtlasEMPreprocess(img, nc, nx, ny, nz, rx, ry, rz,
												orient, orx, ory, orz,
												 ntx, nty, ntz, rtx, rty, rtz,
												classes, (byte)1, 0.0f, 10, userInterface);
		
		if (debug) System.out.println(preprocess.displayTransform());
		// put all intensities in [0,1]										
		preprocess.normalizeImages();
		// find unused regions
		preprocess.findCroppingBoundaries();
		// crop images and atlas
		preprocess.cropImages();
		if (debug) System.out.println("cropped: "+preprocess.displayTransform());
		int[] dim = preprocess.getCurrentImageDimensions();
		nx = dim[0]; ny = dim[1]; nz = dim[2];
		float[] res = preprocess.getSignedImageResolutions();
		rx = res[0]; ry = res[1]; rz = res[2];
		
		// first centroids, memberships : tissue types
		centroids = preprocess.initialCentroids(atlas.getIntensityPriors(modality,nc),classes);		
			
		// align the atlas
		memberships = new float[nx][ny][nz][classes];
		preprocess.initialMemberships(memberships, centroids, classes);
		atlas.setImageInfo(nx,ny,nz,rx,ry,rz,orient,orx,ory,orz);
		atlas.initShapeRegistration(memberships, "rigid", initRegIter, levels);
		atlas.setTransform(preprocess.getTransform());
		if (debug) System.out.println("pre-process: "+preprocess.displayTransform());
		//if (alignShapes) {
			if (verbose) MedicUtilPublic.displayMessage("Atlas alignment\n");


			atlas.updateShapeRegistration(memberships, null, modality, "rigid", initRegIter, levels, 1.0f, 1.0f);
			if (debug) System.out.println("atlas: "+atlas.displayTransform(atlas.getTransform()));

			fireProgressStateChanged(30);

			atlas.registerShapesPyramid();

			if (debug) System.out.println("atlas: "+atlas.displayTransform(atlas.getTransform()));

			fireProgressStateChanged(50);

			memberships = null;

			if (debug) System.out.println("aligned atlas: "+atlas.displayTransform(atlas.getTransform()));
			if (debug) System.out.println("aligned pre-process: "+preprocess.displayTransform());

			preprocess.uncropImages();
			preprocess.findCroppingBoundaries();
			preprocess.cropImages();

			if (debug) System.out.println("final pre-process: "+preprocess.displayTransform());
			if (debug) System.out.println("final aligned atlas: "+atlas.displayTransform(atlas.getTransform()));

			dim = preprocess.getCurrentImageDimensions();
			nx = dim[0]; ny = dim[1]; nz = dim[2];

			atlas.setImageInfo(nx,ny,nz,rx,ry,rz,orient,orx,ory,orz);
		//}
			
		
		
		
		// create the segmentation algorithm
		segmentation = new AtlasEMSegmentation(preprocess.getImages(), atlas, 
								nx, ny, nz, nc,Math.abs(rx), 
								smoothing, 
								priorMode,
								userInterface, getProgressChangeListener());
		
		
		segmentation.setCentroids(centroids);
		segmentation.setIntensityMax(preprocess.getIntensityMax());
		segmentation.setIntensityScale(preprocess.getIntensityScale());
		segmentation.initializePriors();
		userInterface.setGlobalDataText("initial centroids: \n"+segmentation.displayCentroids()); 

		//also use the initial memberships (re-computed)
		preprocess.initialMemberships(segmentation.getMemberships(), centroids, classes);
		
		// re-initialize the atlas alignment
		//if (alignShapes){
			atlas.updateShapeRegistration(segmentation.getMemberships(), segmentation.getSegmentation(), 
					modality, registrationMode, mainRegIter, 1, demonsSmoothing, demonsScale);
			atlas.registerShapes();
			if (verbose) System.out.println("final aligned atlas: "+atlas.displayTransform(atlas.getTransform()));
	//	}
			
		if (correctInhomogeneity) {
			fireProgressStateChanged("initialization (inhomogeneity)");
				
			correction = new InhomogeneityCorrection2[nc];
			for (int c=0;c<nc;c++) {
				// create the inhomogeneity correction segmentation
				correction[c] = new InhomogeneityCorrection2(preprocess.getImages()[c],
						segmentation.getMemberships(),
						segmentation.getCentroids(c),
						classes,
						segmentation.getSegmentation(),atlas.getTopology(),
						polynomialDegree, splineKernel,
						InhomogeneityCorrection2.IMAGE,
						InhomogeneityCorrection2.GLOBAL,
						correctionMethod,
						InhomogeneityCorrection2.FIXED,
						InhomogeneityCorrection2.EQUAL_VARIANCE,
						2.0f, null, null,
						nx,ny,nz,rx,ry,rz,
						getProgressChangeListener());

				// integrate it into the segmentation
				segmentation.addInhomogeneityCorrection(correction[c].getField(),c);
			}
		} else correction = null;
		
		if (debug) System.out.print("pre-processing time: " + (System.currentTimeMillis()-start_time)+"\n"); 
		
		
		
		
		//Initial Variance : A diagonal Covariance  
		
		float[][] stdDev = segmentation.computeDiagonalCovariances();
		segmentation.setDiagCovariance(stdDev);
		if (verbose) System.out.println(segmentation.displayCovariance());
		
				
		if (verbose) userInterface.setGlobalDataText(segmentation.displayCovariance());
		if (verbose && priorMode.equals("Class dependent")) System.out.println(segmentation.displayMixingCoefficients());	
		// memberships & energy
		segmentation.computeAtlas();
		energy = segmentation.computeMemberships();
		//userInterface.setGlobalDataText("initial energy : "+energy[0]+"\n");
			
		if (debug) System.out.print("membership initialization time: " + (System.currentTimeMillis()-start_time)+"\n"); 
			
		Eprev = segmentation.computeEnergy();
		Eratio = 1.0f;
			
		fireProgressStateChanged(99);
			
		if (debug) System.out.print("final initialization time: " + (System.currentTimeMillis()-start_time)+"\n"); 
			
		if (debug) System.out.print("Mixing Coefficients :  " +priorMode+"\n"); 
		
		       
        // 3. main loop
		//--------------------------------------------------------------------
		stop = false;
		int iter = 0;
		int deg = 0;
		if (iterations==0) stop=true;
		
		
		while ((!stop) && (!threadStopped) && segmentation.isWorking()) {
			long iterationTime = System.currentTimeMillis();
				
			iter++;
			if (verbose) System.out.println("iteration "+iter);
			if (verbose) userInterface.setGlobalDataText("iteration "+iter+"\n");
			
			// update parameter information
			//------------------------------------------------------------
			fireProgressStateChanged("iteration " + iter + " (parameters)");
			
			// inhomogeneity field
			if (correctInhomogeneity) {
				if ( (deg<polynomialDegree) && (Eratio < lowEratio) ) deg++;
				for (int c=0;c<nc;c++) correction[c].computeCorrectionField(deg);
							
				fireProgressStateChanged(60);

				if (debug) System.out.print("inhomogeneity time: " + (System.currentTimeMillis()-iterationTime)+"\n"); 
				iterationTime = System.currentTimeMillis();
			}
			
			segmentation.computeCentriods();
			segmentation.computeCovariance();
			//segmentation.computeAtlas();
			
			if (priorMode.equals("Class dependent")) segmentation.computePriors();
			if (verbose) System.out.println("EM parameters Computed!");
			if (verbose) userInterface.setGlobalDataText(segmentation.displayCentroids());
			if (verbose) userInterface.setGlobalDataText(segmentation.displayCovariance());
			if (verbose) userInterface.setGlobalDataText(segmentation.displayMixingCoefficients());
			fireProgressStateChanged(65);
			if (debug) System.out.print("centroid time: " + (System.currentTimeMillis()-iterationTime)+"\n"); 
			iterationTime = System.currentTimeMillis();
				
			// memberships & energy
			
			energy=segmentation.computeMemberships();
			
			fireProgressStateChanged(70);
			
			
			if (debug) System.out.print("membership time: " + (System.currentTimeMillis()-iterationTime)+"\n"); 
			iterationTime = System.currentTimeMillis();
				
			// shape alignment
			//if (alignShapes && (priorMode.equals("Atlas") ) ) atlas.registerShapes();
				
			//System.out.print("alignment time: " + (System.currentTimeMillis()-iterationTime)+"\n"); 
			iterationTime = System.currentTimeMillis();

			if ( energy[1] < maxDistance) stop = true;
			if (iter >= iterations) stop = true;
			fireProgressStateChanged(99);
			iterationTime = System.currentTimeMillis();
		
		}
		
		segmentation.updateClassificationFromMemberships();
		segmentation.cleanUp();
		
		 System.out.println("Segmentation clean up... Done!");
		
		fireProgressStateChanged("creating result images...");
		fireProgressStateChanged(30);
		
		
		classification = segmentation.getClassification();
		
		if (resultOutput.equals("class+member")) {
			memberships = segmentation.exportMemberships();
		} 
		
		if (resultOutput.equals("dura removal inputs"))
			memberships = segmentation.generateDuraRemovalOutputs();

		if (correctInhomogeneity && outputField) {
			field = new float[nc][][][];
			for (int c=0;c<nc;c++) field[c] = correction[c].exportField();
		}
		
		float[][][][] shapes = segmentation.exportAtlasPriors();
			
		fireProgressStateChanged(99);
		// clean-up the algorithms
	
		if (segmentation!=null) segmentation.finalize();
		segmentation = null;
		if (correction!=null) {
			for (int c=0;c<nc;c++) if (correction[c]!=null) correction[c].finalize();
			correction =  null;
		}
		
		
		fireProgressStateChanged("export results...");
		fireProgressStateChanged(20);
		
		try {
			// Calculate the number of result images.
			if (resultOutput.equals("class+member")) destNum = 2;   
			else if (resultOutput.equals("cruise inputs")||resultOutput.equals("dura removal inputs")) destNum =4;
			else destNum = 1; // classification
			
		
			if (outputField) destNum++; //for inhomogeneity field
			if (outputAtlas) destNum++;
			destImage = new ModelImage[destNum];
			
			// output dimensions
			int Nout=0;
            if (outputAtlas){
            	destImage[Nout] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(classes),
						JDialogBase.makeImageName(srcImage[0].getImageName(), "_AtlasEM_priors"));
            	for (int k=0;k<classes;k++) {
            		buffer = preprocess.uncropAndBuffer(shapes[k],0.0f);
            		destImage[Nout].importData(k*preprocess.getOriginalImageSize(), buffer, true);
            	}
            	Nout++;
            }
       	
        	
			if (resultOutput.equals("all_images")
				|| resultOutput.equals("class+member")
				) {
				
            	destImage[Nout] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(classes),
						JDialogBase.makeImageName(srcImage[0].getImageName(), "_AtlasEM_posteriors"));
            	for (int k=0;k<classes;k++) {
            		buffer = preprocess.uncropAndBuffer(memberships[k],0.0f);
            		destImage[Nout].importData(k*preprocess.getOriginalImageSize(), buffer, true);
            	}
            	memberships = null;
            	buffer = null;
            	Nout++;
			}
            
            fireProgressStateChanged(60);
			
         
				destImage[Nout] = new ModelImage(ModelStorageBase.UBYTE, preprocess.getOriginalImageDimensions(),
                                                        JDialogBase.makeImageName(srcImage[0].getImageName(), "_AtlasEM_class"));				

				bytebuffer = preprocess.uncropAndBuffer(classification,(byte)0);
				destImage[Nout].importData(0, bytebuffer, true);
				Nout++;
				
				
				buffer = null;
				bytebuffer = null;
				
				classification = null;
							

			fireProgressStateChanged(90);
			if ( (resultOutput.equals("cruise inputs")) || (resultOutput.equals("dura removal inputs"))) {
				destImage[Nout] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
						  JDialogBase.makeImageName(srcImage[0].getImageName(), "_AtlasEM_gm"));
				destImage[Nout+1] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
						  JDialogBase.makeImageName(srcImage[0].getImageName(), "_AtlasEM_wmFill"));
				destImage[Nout+2] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
						  JDialogBase.makeImageName(srcImage[0].getImageName(), "_AtlasEM_wmMask"));
				
				for (int i=0;i<3;i++){
					buffer = preprocess.uncropAndBuffer(memberships[i],0.0f);
					destImage[Nout+i].importData(0, buffer, true);
				}
				Nout += 3;
				buffer = null;
			}
			
           if (outputField && correctInhomogeneity){
            	if (nc>1) {
					destImage[Nout] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(nc),
														JDialogBase.makeImageName(srcImage[0].getImageName(), "_AtlasEM_fields"));
					for (int c=0;c<nc;c++) {
						buffer = preprocess.uncropAndBuffer(field[c],1.0f);
						destImage[Nout].importData(c*preprocess.getOriginalImageSize(), buffer, true);
					}
				} else {
					destImage[Nout] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
														JDialogBase.makeImageName(srcImage[0].getImageName(), "_AtlasEM_field"));
					buffer = preprocess.uncropAndBuffer(field[0],1.0f);
					destImage[Nout].importData(0, buffer, true);
				}
			}
            
            
            fireProgressStateChanged(99);
			
			buffer = null;
			
		} catch (OutOfMemoryError e) {
			buffer = null;
			errorCleanUp("Adaptive EM Segmentation : Out of memory creating hard classification", true);
			System.err.println(e.getMessage());
			finalize();
			
			setCompleted(false);
			return;
		} catch (IOException error) {
			errorCleanUp("Adaptive EM Segmentation : export problem to destImage[]", true);
			System.err.println(error.getMessage());
			finalize();
			
			setCompleted(false);
			return;
		}
        if (debug) MedicUtilPublic.displayMessage("output...\n");

		// clean-up the algorithms
		if (preprocess!=null) preprocess.finalize();
		preprocess = null;
		
		memberships = null;
		classification = null;
		//lesions = null;
		
		
		setCompleted(true);
		
		
    } // calcSegmentation
	
    public ModelImage[] getResultImages(){
    	return destImage;
    }
    
    
    /**
     * Construct a readable list of the parameters to this segmentation.
     * @return       the parameter string
     */
	public String getParameterText() {
		String delim = "\n";
        String str = new String();
		for (int c=0;c<nc;c++) {
			str += "image "+(c+1)+": "+  srcImage[c].getImageName() + " (modality: "+modality[c]+")"+ delim;
		}
        str += "atlas name: " + atlasName + delim;
        str += "results: "+ resultOutput + " ("+destNum+") "+delim;
		str += "smoothing: " + smoothing + delim;
        str += "max difference: "+ maxDistance + delim;
        str += "iterations: "+ iterations + delim;
        str += "background threshold: "+ bgthreshold + delim;
       	str += "centroid mode: "+ centroidMode + delim;
		str += "centroid smoothing: "+ centroidSmoothness + delim;
		str += "register: "+ register + delim;
		str += "registration iterations (init,seg): " + initRegIter +", "+ mainRegIter + delim;
        str += "registration levels: "+ levels + delim;
		str += "inhomogeneity correction: "+ correctInhomogeneity + delim;
		str += "output inhomogeneity field: " + outputField + delim;
		str += "correction method: "+ correctionMethod + delim;
        str += "degree: "+ polynomialDegree + delim;
        str += "kernel: "+ splineKernel + delim;
        
        return str;
    }

	/** access the atlas information */

	public final DemonToadDeformableAtlas getAtlas() { return atlas; }
}



