package edu.jhmi.rad.medic.algorithms;

import edu.jhmi.rad.medic.dialogs.*;
import edu.jhmi.rad.medic.methods.*;
import edu.jhmi.rad.medic.libraries.*;
import edu.jhmi.rad.medic.utilities.*;

import gov.nih.mipav.model.algorithms.AlgorithmBase;
import gov.nih.mipav.model.structures.*;
import gov.nih.mipav.view.dialogs.*;
import gov.nih.mipav.view.*;

import java.io.*;
import java.util.*;
import java.lang.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

 /**
 *
 *   Algorithm for npon-linear registration with the DEMONS algorithm.
 *
 *	@version    Feb 2005
 *	@author     Pierre-Louis Bazin
 *  @see 		JDialogDemons
 *		
 *
*/
public class AlgorithmDemons2 extends AlgorithmBase {

    // Fuzzy images require 1 image for each class
    // Hard images 1 image with assigned clusters
    private ModelImage  srcImage;
    private ModelImage  targetImage;
    private ModelImage[]	destImage;
	private String      output;
    private ViewUserInterface       userInterface;

    private boolean 		cropBackground = false;
    private float       	backThreshold=0.0f;
    
	private		int         	nix,niy,niz;
	private 	float     	rix,riy,riz;
    private		int    		ntx,nty,ntz;
	private 	float 		rtx,rty,rtz;
	private		int			nc;
    
    // algorithm parameters
	private DemonsRegistration2					demons2;
	private DemonsScaledRegistration			sdemons;
	private DemonsScaledConsistentRegistration	idemons;
	private DemonsLevelsetRegistration			ldemons;
	private static final int		DEMONS2 = 1;
	private static final int		SCALED = 2;
	private static final int		INVERSE_CONSISTENT = 3;
	private static final int		BEST_SCALED = 4;
	private static final int		BEST_SCALED_PYRAMID = 5;
	private static final int		LEVELSET_IMG = 6;
	private static final int		LEVELSET_IC = 7;
	private static final int		LEVELSET_MAXIC = 8;
	private float   	smoothing;
	private float   	scale;
	private int			levels;
	private int			Ni,Nt;
	private int			regType,forceType,fieldType;
	private int			preType;
	private int			algoType;
	
	private boolean		verbose = true;
	private boolean		debug = true;
	
    /**
    *	Constructor for 3D images in which changes are placed in a predetermined destination image.
    *   @param destImg      Image model where result image is to stored.
    *   @param srcImg       Source image model.
    */
	public AlgorithmDemons2(ModelImage srcImg, ModelImage targetImg, 
									int levels_, int Ni_, int Nt_, float smooth_, float scale_,
									boolean cropBack, float cropVal, 
									String reg_, String force_, String field_,
									String pre_, String algo_,
									String out_) {
			
		super(null, srcImg);
		userInterface = ViewUserInterface.getReference();
		srcImage = srcImg;
        targetImage = targetImg;  // Put results in destination image.
        output = out_;
        
       cropBackground = cropBack;
		backThreshold = cropVal;
        
		levels = levels_;
		Ni = Ni_; Nt = Nt_;
        smoothing = smooth_;
		scale = scale_;

		regType = DemonsRegistration2.GAUSS_DIFFUSION;
		if (reg_.equals("gauss_diffusion")) {
			regType = DemonsRegistration2.GAUSS_DIFFUSION;
			MedicUtilPublic.displayMessage("regularization: gaussian diffusion-like \n");
		} else if (reg_.equals("gauss_fluid")) {
			regType = DemonsRegistration2.GAUSS_FLUID;
			MedicUtilPublic.displayMessage("regularization: gaussian fluid-like \n");
		} else if (reg_.equals("gauss_mixed")) {
			regType = DemonsRegistration2.GAUSS_MIXED;
			MedicUtilPublic.displayMessage("regularization: gaussian fluid-like \n");
		} else if (reg_.equals("div_diffusion")) {
			regType = DemonsRegistration2.DIV_DIFFUSION;
			MedicUtilPublic.displayMessage("regularization: div=0 diffusion-like \n");
		} else if (reg_.equals("div_fluid")) {
			regType = DemonsRegistration2.DIV_FLUID;
			MedicUtilPublic.displayMessage("regularization: div=0 fluid-like \n");
		} else if (reg_.equals("gauss_div_diffusion")) {
			regType = DemonsRegistration2.GAUSS_DIV_DIFFUSION;
			MedicUtilPublic.displayMessage("regularization: div=0 diffusion-like \n");
		} else if (reg_.equals("gauss_div_fluid")) {
			regType = DemonsRegistration2.GAUSS_DIV_FLUID;
			MedicUtilPublic.displayMessage("regularization: div=0 fluid-like \n");
		}
		
		forceType = DemonsRegistration2.SYMMETRIC;
		if (force_.equals("fixed")) {
			forceType = DemonsRegistration2.FIXED;
			MedicUtilPublic.displayMessage("forces: fixed image \n");
		} else if (force_.equals("moving")) {
			forceType = DemonsRegistration2.MOVING;
			MedicUtilPublic.displayMessage("forces: moving image \n");
		} else if (force_.equals("symmetric")) {
			forceType = DemonsRegistration2.SYMMETRIC;
			MedicUtilPublic.displayMessage("forces: symmetric \n");
		}
		
		if (field_.equals("compositive")) {
			fieldType = DemonsRegistration2.COMPOSITIVE;
			MedicUtilPublic.displayMessage("field: compositive \n");
		} else if (field_.equals("mindiv")) {
			fieldType = DemonsLevelsetRegistration.MINDIV;
			MedicUtilPublic.displayMessage("field: mindiv \n");
		} else if (field_.equals("mindivsep")) {
			fieldType = DemonsLevelsetRegistration.MINDIVSEP;
			MedicUtilPublic.displayMessage("field: mindivsep \n");
		}
		
		preType = DemonsRegistration2.RAW;
		if (pre_.equals("raw")) {
			preType = DemonsRegistration2.RAW;
			MedicUtilPublic.displayMessage("input pre-processing: raw images \n");
		} else if (pre_.equals("normalized")) {
			preType = DemonsRegistration2.NORMALIZED;
			MedicUtilPublic.displayMessage("input pre-processing: normalize images \n");
		} else if (pre_.equals("scaled")) {
			preType = DemonsRegistration2.SCALED;
			MedicUtilPublic.displayMessage("input pre-processing: scale images \n");
		} else if (pre_.equals("segmentation")) {
			preType = DemonsRegistration2.SEGMENTATION;
			MedicUtilPublic.displayMessage("input pre-processing: segmentation images \n");
		} else if (pre_.equals("multichannel")) {
			preType = DemonsRegistration2.MULTICHANNEL;
			MedicUtilPublic.displayMessage("input pre-processing: multichannel images \n");
		} else if (pre_.startsWith("levelset")) {
			preType = DemonsRegistration2.LEVELSET;
			MedicUtilPublic.displayMessage("input pre-processing: levelset image \n");
		}				
		
		algoType = DEMONS2;
		if (algo_.equals("simple")) {
			algoType = DEMONS2;
			Nt = Ni;
			levels = 1;
			MedicUtilPublic.displayMessage("algorithm: simple\n");
		} else if (algo_.equals("pyramid"))  {
			algoType = DEMONS2;
			Nt = Ni;
			levels = 1;
			MedicUtilPublic.displayMessage("algorithm: pyramid\n");
		} else if (algo_.equals("scaled")) {
			algoType = SCALED;
			MedicUtilPublic.displayMessage("algorithm: scaled\n");
		} else if (algo_.equals("best_scaled")) {
			algoType = BEST_SCALED;
			MedicUtilPublic.displayMessage("algorithm: scaled\n");
		} else if (algo_.equals("scaled_pyramid")) {
			algoType = BEST_SCALED_PYRAMID;
			MedicUtilPublic.displayMessage("algorithm: scaled\n");
		}  else if (algo_.equals("invertible")) {
			algoType = INVERSE_CONSISTENT;
			MedicUtilPublic.displayMessage("algorithm: invertible\n");
		}  else if (algo_.equals("levelset_img")) {
			algoType = LEVELSET_IMG;
			MedicUtilPublic.displayMessage("algorithm: levelset_img\n");
		}  else if (algo_.equals("levelset_ic")) {
			algoType = LEVELSET_IC;
			MedicUtilPublic.displayMessage("algorithm: levelset_ic\n");
		}  else if (algo_.equals("levelset_maxic")) {
			algoType = LEVELSET_MAXIC;
			MedicUtilPublic.displayMessage("algorithm: levelset_maxic\n");
		}  
 	}

    /**
    *	Prepares this class for destruction.
    */
	public void finalize(){
        targetImage = null;
	    destImage   = null;
	    srcImage    = null;
        super.finalize();
        System.gc();
	}
	
	/** for output of results */
	public ModelImage[] getResultImages() { return destImage; }

    /**
    *   Starts the algorithm.
    */
	public void runAlgorithm() {

        if (srcImage  == null) {
            displayError("Source Image is null");
            return;
        }
        if (targetImage  == null) {
            displayError("Target Image is null");
            return;
        }
		// start the timer to compute the elapsed time
        setStartTime();

		fireProgressStateChanged(srcImage.getImageName(), "Extracting data ...");
		fireProgressStateChanged(10);
		
		// init dimensions
		nix = srcImage.getExtents()[0];
		niy = srcImage.getExtents()[1];
		niz = srcImage.getExtents()[2];

		ntx = targetImage.getExtents()[0];
		nty = targetImage.getExtents()[1];
		ntz = targetImage.getExtents()[2];

		// check for 4D images
		if (srcImage.getExtents().length==4 && targetImage.getExtents().length==4) {
			nc = Numerics.min(srcImage.getExtents()[3], targetImage.getExtents()[3]);
		} else {
			nc = 1;
		}
		
		float[] image; 
		float[] target;
		try {
			// retrieve all data: for each volume
			image = new float[nix*niy*niz*nc];
			srcImage.exportData(0, nix*niy*niz*nc, image); // locks and releases lock
			target = new float[ntx*nty*ntz*nc];
			targetImage.exportData(0, ntx*nty*ntz*nc, target); // locks and releases lock
		} catch (IOException error) {
			image = null;
			target = null;
			errorCleanUp("Algorithm: source image locked", true);
			return;
		} catch (OutOfMemoryError e){
			image = null;
			target = null;
			errorCleanUp("Algorithm: Out of memory creating process buffer", true);
			return;
		}

		// resolution
		rix = srcImage.getFileInfo()[0].getResolutions()[0];
		riy = srcImage.getFileInfo()[0].getResolutions()[1];
		riz = srcImage.getFileInfo()[0].getResolutions()[2];
		
		rtx = targetImage.getFileInfo()[0].getResolutions()[0];
		rty = targetImage.getFileInfo()[0].getResolutions()[1];
		rtz = targetImage.getFileInfo()[0].getResolutions()[2];
		
		// cropping
		ImageCropper icrop,tcrop;
		
		icrop = new ImageCropper(nix,niy,niz);
		tcrop = new ImageCropper(ntx,nty,ntz);
		
		icrop.setBorderSize(5*levels+5);
		tcrop.setBorderSize(5*levels+5);
				
		float imin = ImageFunctions.robustMinimum(image, 0.01f, 2, nix, niy, niz);
		float imax = ImageFunctions.robustMaximum(image, 0.01f, 2, nix, niy, niz);
		
		float tmin = ImageFunctions.robustMinimum(target, 0.01f, 2, ntx, nty, ntz);
		float tmax = ImageFunctions.robustMaximum(target, 0.01f, 2, ntx, nty, ntz);
		
		MedicUtilPublic.displayMessage("Image min / max: "+imin+" / "+imax+"\n");
		MedicUtilPublic.displayMessage("Target min / max: "+tmin+" / "+tmax+"\n");
			
		if (cropBackground) {
			fireProgressStateChanged(srcImage.getImageName(), "Cropping ...");
			fireProgressStateChanged(20);
			
			MedicUtilPublic.displayMessage("Cropping img / trg: "+(imin + backThreshold*(imax-imin))+" / "+(tmin + backThreshold*(tmax-tmin))+"\n");
			
			icrop.findCroppingBoundaries(image, imin + backThreshold*(imax-imin));
			tcrop.findCroppingBoundaries(target, tmin + backThreshold*(tmax-tmin));
		}
		
		// synchronize the image cropper
		image = icrop.cropImage(image,nc);
		target = tcrop.cropImage(target,nc);	
			
		nix = icrop.mx();
		niy = icrop.my();
		niz = icrop.mz();
		ntx = tcrop.mx();
		nty = tcrop.my();
		ntz = tcrop.mz();
		
		// main algorithm
		fireProgressStateChanged(srcImage.getImageName(), "Main computations ...");
		fireProgressStateChanged(40);

		if (algoType==DEMONS2) {
			demons2 = new DemonsRegistration2(image, target, 
												nix, niy, niz, rix, riy, riz, 
												ntx, nty, ntz, rtx, rty, rtz,
												smoothing, scale, cropBackground, backThreshold, levels, Ni, Nt, 
												regType, forceType, fieldType, preType, null);
			
			demons2.runGaussianPyramid();
		} else if (algoType==SCALED) {
			sdemons = new DemonsScaledRegistration(image, target, 
												nix, niy, niz, rix, riy, riz, 
												ntx, nty, ntz, rtx, rty, rtz,
												smoothing, 1.0f, cropBackground, backThreshold, scale, Ni, Nt, 
												regType, forceType, fieldType, preType, null);
			
			sdemons.runRegistration();
		} else if (algoType==BEST_SCALED) {
			sdemons = new DemonsScaledRegistration(image, target, 
												nix, niy, niz, rix, riy, riz, 
												ntx, nty, ntz, rtx, rty, rtz,
												smoothing, 1.0f, cropBackground, backThreshold, scale, Ni, Nt, 
												regType, forceType, fieldType, preType, null);
			
			sdemons.runBestRegistration();
		} else if (algoType==BEST_SCALED_PYRAMID) {
			float maxscale = scale;
			float minscale = 1.0f;
			float[][] prev = null;
			int nb = 0;
			while (scale >= minscale && nb < levels) {
				sdemons = new DemonsScaledRegistration(image, target, 
													nix, niy, niz, rix, riy, riz, 
													ntx, nty, ntz, rtx, rty, rtz,
													smoothing, 1.0f, cropBackground, backThreshold, scale, Ni, Nt, 
													regType, forceType, fieldType, preType, prev);
				
				sdemons.runBestRegistration();
				prev = sdemons.exportTransformMapping();
				scale = scale - 1.0f;
				nb++;
			}
		} else if (algoType==INVERSE_CONSISTENT) {
			idemons = new DemonsScaledConsistentRegistration(image, target, 
												nix, niy, niz, rix, riy, riz, 
												ntx, nty, ntz, rtx, rty, rtz,
												smoothing, 1.0f, cropBackground, backThreshold, scale, Ni, Nt, 
												regType, forceType, fieldType, preType, null);
			
			idemons.runRegistration();
		} else if (algoType==LEVELSET_IMG || algoType==LEVELSET_IC || algoType==LEVELSET_MAXIC) {
			ldemons = new DemonsLevelsetRegistration(image, target, 
												nix, niy, niz, rix, riy, riz, 
												ntx, nty, ntz, rtx, rty, rtz,
												smoothing, 1.0f, cropBackground, backThreshold, scale, Ni, Nt, 
												regType, forceType, fieldType, preType, null);
			
			if (algoType==LEVELSET_IMG) ldemons.runDecoupledRegistration();
			else if (algoType==LEVELSET_IC) ldemons.runRegistration();
			else if (algoType==LEVELSET_MAXIC) ldemons.runConsistentRegistration();
		}
		// output
		try {
			fireProgressStateChanged(srcImage.getImageName(), "Output ...");
			fireProgressStateChanged(80);
			
			if (algoType==DEMONS2) {
				if (output.equals("transformed_image")) {
					destImage = new ModelImage[1];
					int[] imgExtents;
					if (nc>1) imgExtents = tcrop.getOriginalExtents(nc);
					else imgExtents = tcrop.getOriginalExtents();
				
					destImage[0] = new ModelImage(ModelStorageBase.FLOAT, imgExtents,
														JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_reg"));				
					
					destImage[0].importData(0, tcrop.uncropAndBuffer(demons2.exportTransformedImage(),nc), true);
				} else if  (output.equals("transformation_field")) {
					destImage = new ModelImage[1];
					destImage[0] = new ModelImage(ModelStorageBase.FLOAT, tcrop.getOriginalExtents(3),
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_vec"));				
	
					destImage[0].importData(0, tcrop.uncropAndBuffer(demons2.exportTransformField(),3), true);
				} else if  (output.equals("all_images")) {
					destImage = new ModelImage[3];
					int[] imgExtents;
					if (nc>1) imgExtents = tcrop.getOriginalExtents(nc);
					else imgExtents = tcrop.getOriginalExtents();
					
					destImage[0] = new ModelImage(ModelStorageBase.FLOAT, imgExtents,
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_reg"));				
	
					destImage[0].importData(0, tcrop.uncropAndBuffer(demons2.exportTransformedImage(),nc), true);
					
					destImage[1] = new ModelImage(ModelStorageBase.FLOAT, tcrop.getOriginalExtents(3),
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_vec"));				
	
					destImage[1].importData(0, tcrop.uncropAndBuffer(demons2.exportTransformField(),3), true);
					
					destImage[2] = new ModelImage(ModelStorageBase.FLOAT, imgExtents,
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_err"));				
	
					destImage[2].importData(0, tcrop.uncropAndBuffer(demons2.exportResiduals(),nc), true);
				}
			} else if (algoType==SCALED || algoType==BEST_SCALED || algoType==BEST_SCALED_PYRAMID) {
				if (output.equals("transformed_image")) {
					destImage = new ModelImage[1];
					int[] imgExtents;
					if (nc>1) imgExtents = tcrop.getOriginalExtents(nc);
					else imgExtents = tcrop.getOriginalExtents();
				
					destImage[0] = new ModelImage(ModelStorageBase.FLOAT, imgExtents,
														JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_reg"));				
					
					destImage[0].importData(0, tcrop.uncropAndBuffer(sdemons.exportTransformedImage(),nc), true);
				} else if  (output.equals("transformation_field")) {
					destImage = new ModelImage[1];
					destImage[0] = new ModelImage(ModelStorageBase.FLOAT, tcrop.getOriginalExtents(3),
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_vec"));				
	
					destImage[0].importData(0, tcrop.uncropAndBuffer(sdemons.exportTransformField(),3), true);
				} else if  (output.equals("all_images")) {
					destImage = new ModelImage[3];
					int[] imgExtents;
					if (nc>1) imgExtents = tcrop.getOriginalExtents(nc);
					else imgExtents = tcrop.getOriginalExtents();
					
					destImage[0] = new ModelImage(ModelStorageBase.FLOAT, imgExtents,
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_reg"));				
	
					destImage[0].importData(0, tcrop.uncropAndBuffer(sdemons.exportTransformedImage(),nc), true);
					
					destImage[1] = new ModelImage(ModelStorageBase.FLOAT, tcrop.getOriginalExtents(3),
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_vec"));				
	
					destImage[1].importData(0, tcrop.uncropAndBuffer(sdemons.exportTransformField(),3), true);
					
					destImage[2] = new ModelImage(ModelStorageBase.FLOAT, imgExtents,
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_err"));				
	
					destImage[2].importData(0, tcrop.uncropAndBuffer(sdemons.exportResiduals(),nc), true);
				}
			} else if (algoType==INVERSE_CONSISTENT) {
				if (output.equals("transformed_image")) {
					destImage = new ModelImage[2];
					int[] imgExtents;
					if (nc>1) imgExtents = tcrop.getOriginalExtents(nc);
					else imgExtents = tcrop.getOriginalExtents();
				
					int[] trgExtents;
					if (nc>1) trgExtents = icrop.getOriginalExtents(nc);
					else trgExtents = icrop.getOriginalExtents();
				
					destImage[0] = new ModelImage(ModelStorageBase.FLOAT, imgExtents,
														JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_img_reg"));				
					
					destImage[0].importData(0, tcrop.uncropAndBuffer(idemons.exportTransformedImage(),nc), true);
				
					destImage[1] = new ModelImage(ModelStorageBase.FLOAT, trgExtents,
														JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_trg_reg"));				
					
					destImage[1].importData(0, icrop.uncropAndBuffer(idemons.exportInverseTransformedTarget(),nc), true);
				} else if  (output.equals("transformation_field")) {
					destImage = new ModelImage[2];
					
					destImage[0] = new ModelImage(ModelStorageBase.FLOAT, tcrop.getOriginalExtents(3),
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_img_vec"));				
	
					destImage[0].importData(0, tcrop.uncropAndBuffer(idemons.exportTransformField(),3), true);
					
					destImage[1] = new ModelImage(ModelStorageBase.FLOAT, icrop.getOriginalExtents(3),
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_trg_vec"));				
	
					destImage[1].importData(0, icrop.uncropAndBuffer(idemons.exportInverseTransformField(),3), true);
				} else if  (output.equals("all_images")) {
					destImage = new ModelImage[8];
					int[] imgExtents;
					if (nc>1) imgExtents = tcrop.getOriginalExtents(nc);
					else imgExtents = tcrop.getOriginalExtents();
					
					int[] trgExtents;
					if (nc>1) trgExtents = icrop.getOriginalExtents(nc);
					else trgExtents = icrop.getOriginalExtents();
					
					destImage[0] = new ModelImage(ModelStorageBase.FLOAT, imgExtents,
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_img_reg"));				
	
					destImage[0].importData(0, tcrop.uncropAndBuffer(idemons.exportTransformedImage(),nc), true);
					
					destImage[1] = new ModelImage(ModelStorageBase.FLOAT, trgExtents,
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_trg_reg"));				
	
					destImage[1].importData(0, icrop.uncropAndBuffer(idemons.exportInverseTransformedTarget(),nc), true);
					
					destImage[2] = new ModelImage(ModelStorageBase.FLOAT, tcrop.getOriginalExtents(3),
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_img_vec"));				
	
					destImage[2].importData(0, tcrop.uncropAndBuffer(idemons.exportTransformField(),3), true);
					
					destImage[3] = new ModelImage(ModelStorageBase.FLOAT, icrop.getOriginalExtents(3),
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_trg_vec"));				
	
					destImage[3].importData(0, icrop.uncropAndBuffer(idemons.exportInverseTransformField(),3), true);
					
					destImage[4] = new ModelImage(ModelStorageBase.FLOAT, imgExtents,
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_img_err"));				
	
					destImage[4].importData(0, tcrop.uncropAndBuffer(idemons.exportResiduals(),nc), true);
					
					destImage[5] = new ModelImage(ModelStorageBase.FLOAT, trgExtents,
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_trg_err"));				
	
					destImage[5].importData(0, icrop.uncropAndBuffer(idemons.exportInverseResiduals(),nc), true);

					destImage[6] = new ModelImage(ModelStorageBase.FLOAT, tcrop.getOriginalExtents(),
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_img_dst"));				
	
					destImage[6].importData(0, tcrop.uncropImage(idemons.exportForwardBackwardDistance()), true);
					
					destImage[7] = new ModelImage(ModelStorageBase.FLOAT, icrop.getOriginalExtents(),
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_trg_dst"));				
	
					destImage[7].importData(0, icrop.uncropImage(idemons.exportBackwardForwardDistance()), true);
				}
			} else if (algoType==LEVELSET_IMG || algoType==LEVELSET_IC || algoType==LEVELSET_MAXIC) {
				if (output.equals("transformed_image")) {
					destImage = new ModelImage[2];
					int[] imgExtents;
					if (nc>1) imgExtents = tcrop.getOriginalExtents(nc);
					else imgExtents = tcrop.getOriginalExtents();
				
					int[] trgExtents;
					if (nc>1) trgExtents = icrop.getOriginalExtents(nc);
					else trgExtents = icrop.getOriginalExtents();
				
					destImage[0] = new ModelImage(ModelStorageBase.FLOAT, imgExtents,
														JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_img_reg"));				
					
					destImage[0].importData(0, tcrop.uncropAndBuffer(ldemons.exportTransformedImage(),nc), true);
				
					destImage[1] = new ModelImage(ModelStorageBase.FLOAT, trgExtents,
														JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_trg_reg"));				
					
					destImage[1].importData(0, icrop.uncropAndBuffer(ldemons.exportInverseTransformedTarget(),nc), true);
				} else if  (output.equals("transformation_field")) {
					destImage = new ModelImage[4];
					
					destImage[0] = new ModelImage(ModelStorageBase.FLOAT, tcrop.getOriginalExtents(3),
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_img_vec"));				
	
					destImage[0].importData(0, tcrop.uncropAndBuffer(ldemons.exportTransformField(),3), true);
					
					destImage[1] = new ModelImage(ModelStorageBase.FLOAT, icrop.getOriginalExtents(3),
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_trg_vec"));				
	
					destImage[1].importData(0, icrop.uncropAndBuffer(ldemons.exportInverseTransformField(),3), true);
					
					destImage[2] = new ModelImage(ModelStorageBase.UBYTE, tcrop.getOriginalExtents(),
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_img_grid"));				
	
					destImage[2].importData(0, tcrop.uncropImage(ldemons.exportTransformGrid(20)), true);
					
					destImage[3] = new ModelImage(ModelStorageBase.UBYTE, icrop.getOriginalExtents(),
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_trg_grid"));				
	
					destImage[3].importData(0, icrop.uncropImage(ldemons.exportInverseTransformGrid(20)), true);
				} else if  (output.equals("all_images")) {
					destImage = new ModelImage[10];
					int[] imgExtents;
					if (nc>1) imgExtents = tcrop.getOriginalExtents(nc);
					else imgExtents = tcrop.getOriginalExtents();
					
					int[] trgExtents;
					if (nc>1) trgExtents = icrop.getOriginalExtents(nc);
					else trgExtents = icrop.getOriginalExtents();
					
					destImage[0] = new ModelImage(ModelStorageBase.FLOAT, imgExtents,
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_img_reg"));				
	
					destImage[0].importData(0, tcrop.uncropAndBuffer(ldemons.exportTransformedImage(),nc), true);
					
					destImage[1] = new ModelImage(ModelStorageBase.FLOAT, trgExtents,
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_trg_reg"));				
	
					destImage[1].importData(0, icrop.uncropAndBuffer(ldemons.exportInverseTransformedTarget(),nc), true);
					
					destImage[2] = new ModelImage(ModelStorageBase.FLOAT, tcrop.getOriginalExtents(3),
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_img_vec"));				
	
					destImage[2].importData(0, tcrop.uncropAndBuffer(ldemons.exportTransformField(),3), true);
					
					destImage[3] = new ModelImage(ModelStorageBase.FLOAT, icrop.getOriginalExtents(3),
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_trg_vec"));				
	
					destImage[3].importData(0, icrop.uncropAndBuffer(ldemons.exportInverseTransformField(),3), true);
					
					destImage[4] = new ModelImage(ModelStorageBase.FLOAT, imgExtents,
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_img_err"));				
	
					destImage[4].importData(0, tcrop.uncropAndBuffer(ldemons.exportResiduals(),nc), true);
					
					destImage[5] = new ModelImage(ModelStorageBase.FLOAT, trgExtents,
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_trg_err"));				
	
					destImage[5].importData(0, icrop.uncropAndBuffer(ldemons.exportInverseResiduals(),nc), true);

					destImage[6] = new ModelImage(ModelStorageBase.FLOAT, tcrop.getOriginalExtents(),
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_img_dst"));				
	
					destImage[6].importData(0, tcrop.uncropImage(ldemons.exportForwardBackwardDistance()), true);
					
					destImage[7] = new ModelImage(ModelStorageBase.FLOAT, icrop.getOriginalExtents(),
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_trg_dst"));				
	
					destImage[7].importData(0, icrop.uncropImage(ldemons.exportBackwardForwardDistance()), true);
					
					destImage[8] = new ModelImage(ModelStorageBase.UBYTE, tcrop.getOriginalExtents(),
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_img_grid"));				
	
					destImage[8].importData(0, tcrop.uncropImage(ldemons.exportTransformGrid(20)), true);
					
					destImage[9] = new ModelImage(ModelStorageBase.UBYTE, icrop.getOriginalExtents(),
															JDialogBase.makeImageName(srcImage.getImageName(), "_Demons2_trg_grid"));				
	
					destImage[9].importData(0, icrop.uncropImage(ldemons.exportInverseTransformGrid(20)), true);
				}
			}
		} catch (OutOfMemoryError e) {
			errorCleanUp("Demons2: Out of memory creating hard classification", true);
			System.err.println(e.getMessage());
			finalize();
			
			setCompleted(false);
			return;
		} catch (IOException error) {
			errorCleanUp("Demons2: export problem to destImage[]", true);
			System.err.println(error.getMessage());
			finalize();
			
			setCompleted(false);
			return;
		}
		if (algoType==DEMONS2) demons2.finalize();
		demons2 = null;
		if (algoType==SCALED) sdemons.finalize();
		sdemons = null;
		if (algoType==INVERSE_CONSISTENT) idemons.finalize();
		idemons = null;

		fireProgressStateChanged(90);
		
		setCompleted(true);
		threadStopped = false;
       // compute the elapsed time
        computeElapsedTime();
    } // end runAlgorithm()

}
