package edu.jhmi.rad.medic.algorithms;

import edu.jhmi.rad.medic.methods.*;
import edu.jhmi.rad.medic.utilities.*;

import gov.nih.mipav.model.algorithms.AlgorithmBase;
import gov.nih.mipav.model.structures.*;
import gov.nih.mipav.model.structures.jama.*;
import gov.nih.mipav.model.file.FileInfoBase;
import gov.nih.mipav.view.*;
import gov.nih.mipav.view.dialogs.*;

import java.io.*;
import java.util.*;
import java.lang.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;


/**
 *	TOADS: TOpology-preserving Anatomy-Driven Segmentation.
 *	<p>
 *	This is the main version of the segmentation. It includes inhomogeneity correction,
 *	template registration and adaptive coefficients.
 *
 *	@version    April 2006
 *	@author     Pierre-Louis Bazin
 *  @see 		JDialogToads
 *	@see 		Toads
 *	@see		RegisterTemplate
 *
*/
public class AlgorithmMultipleToads extends AlgorithmBase {
	public AlgorithmMultipleToads() {
	}

	private static final String cvsversion = "$Revision: 1.6 $";
	public static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");

	public String get_version() {
		return revnum;
	}

    // Fuzzy images require 1 image for each class
    // Hard images 1 image with assigned classes
    private ModelImage  		destImage[];
    private ModelImage  		srcImage[];
    private String      		resultOutput;
    	
    // image size
	private int 		classes; 		// number of classes
	private int 		nx,ny,nz,nc; 		// original image dimensions
	private int 		ntx,nty,ntz; 	// original template dimensions
	private	float		rx,ry,rz;		// image resolutions
	private	float		rtx,rty,rtz;	// template resolutions
	private int			orient;			// image orientation
	private int			orx,ory,orz;	// image axis orientation
	private	String[]	modality;		// image modality
	private	String[]	psfType;		// image psf type
	private	String		algorithmMode;		// variants of the algorithm
	private	String		connectivityType;		// choice of connectivity
	
    // segmentation parameters
    private	String				atlasName;
    private	MultipleToadDeformableAtlas		atlas=null;
    private byte[]				objType;

	private	boolean		segment = true;
	private float   	smoothing;
	private float		outlier;
	private int 		iterations;
	private float   	maxDistance;
	private float		bgthreshold;
	private	boolean		doGrowing;
	
	private	boolean		useTopologyPrior;
	private float		demonsSmoothing;
	private float		demonsScale;
	private float		marchingLimit;
	private float		volumeLimit;
	//private	int			steps;
	private float		lesion;
	
	private	boolean		useShapePrior	= false;
	private	float		atlasCoefficient, atlasScale, atlasCoupling;
	
	private boolean		useIntensityPrior;
	
	private	String		relationMode;
	private	String		centroidMode;
	private	float		centroidSmoothness = 0.5f;
	
	private	boolean		register = false;
	private	int			levels;
	private	int			firstIter;
	private	int			topIter;
	private	int			lastIter;
	private	String		transformType;
	
	private	boolean		alignShapes = true;
	private	int			alignIter = 1;
	//private	String		alignType = "fully_affine";
	private	String		alignType = "rigid";
	//private	String		alignType = "proportional_scale";
	//private	String		alignType = "single_scale";
	private	int			initAlignIter = 10;
	
    private	boolean		correctInhomogeneity = false;
	private	int			polynomialDegree;
	private	float		splineKernel;
	private	int			correctionMethod;
	private float   	lowEratio = 0.1f;
	
	private boolean		useApprox = false;
	private int			approx;
	
	private boolean outputMaxMem;
	private boolean outputField;
	private boolean		exactTopology = false;
	
	private static final	boolean		verbose = true;
	private static final	boolean		debug = false;
	
    /**
    *	Constructor for 3D images in which changes are placed in a predetermined destination image.
    *   @param destImg_      Image model where result image is to stored.
    *   @param srcImg_       Source image model.
    */
	public AlgorithmMultipleToads(ModelImage[] srcImg_, int nInput_,
								String[] imgModal_, String[] psf_,
								String aName_, MultipleToadDeformableAtlas atlas_, String segOutput_, 
								boolean segment_, 
								float smooth_, float out_, int nIterMax_, float distMax_, float bgth_,
								boolean useTopology_, 
								float ridge_, float lIncr_, float mLim_, float vLim_, int approx_, float lesion_,
								boolean useShape_,
								float aCoeff_, float aScale_, float aCoupling_,
								boolean useIntensity_, 
								String relMode_, 
								String centMode_, float smoothCentr_,
								String spcMode_,
								boolean register_, 
								int lev_, int first_, int top_, int last_,
								boolean correct_, 
								String correctType_,
								int poly_, float kernel_,
								String algo_,
								String connect_,
								boolean	exact_,boolean outputMaxMem_, boolean outputField_
								) {
        super(null, srcImg_[0]);
		srcImage = srcImg_;
        nc = nInput_;
		modality = imgModal_;
		psfType = psf_;
		
		atlasName = aName_;
		atlas = atlas_;
		classes = atlas.getNumber();

		resultOutput = segOutput_;
        
		segment = segment_;
		smoothing = smooth_;
		outlier = out_;
		iterations = Numerics.abs(nIterMax_);
		doGrowing = (nIterMax_>0);
        maxDistance = distMax_;
		lowEratio = (float)Math.sqrt(maxDistance);
		bgthreshold = bgth_;
		
		useTopologyPrior = useTopology_;
		demonsSmoothing = ridge_;
		demonsScale = lIncr_;
		marchingLimit = mLim_;
		volumeLimit = vLim_;
		lesion = lesion_;
		approx = approx_;
		useApprox = ( (approx<classes) && (approx>1) );
		
		useShapePrior = useShape_;
		atlasCoefficient = aCoeff_;
		atlasScale = aScale_;
		atlasCoupling = aCoupling_;

		useIntensityPrior = useIntensity_;
		
		centroidMode = centMode_;
		centroidSmoothness = smoothCentr_;
		
		algorithmMode = algo_;
		
		connectivityType = connect_;
		
		exactTopology = exact_;
		
		register = register_;
		levels = lev_;
		firstIter = first_;
		topIter = top_;
		lastIter = last_;
		
		alignIter = topIter;
		initAlignIter = firstIter;
		
		alignType = spcMode_;
		outputMaxMem = outputMaxMem_;
		correctInhomogeneity = correct_;
		if (correctInhomogeneity) {
			outputField = outputField_;
			polynomialDegree = poly_;
			splineKernel = kernel_;
			if (correctType_.equals("Splines")) correctionMethod = InhomogeneityCorrection2.SPLINES;
			else correctionMethod = InhomogeneityCorrection2.CHEBYSHEV;
		} else {
			outputField = false;
			polynomialDegree = 0;
			splineKernel = 0.0f;
			correctionMethod = InhomogeneityCorrection2.CHEBYSHEV;
		}
	}

	
    /**
    *	Constructor for 3D images in which changes are placed in a predetermined destination image.
    *   @param destImg_      Image model where result image is to stored.
    *   @param srcImg_       Source image model.
    */
	public AlgorithmMultipleToads(ModelImage[] srcImg_, int nInput_,
								String[] imgModal_, 
								String aName_, MultipleToadDeformableAtlas atlas_, String segOutput_, 
								float smooth_, int nIterMax_, float distMax_,
								float fLim_, float lLim_,
								float aCoeff_, float aScale_,
								String align_,
								int lev_, int first_, int top_, int last_,
								boolean correct_, 
								String correctType_,
								int poly_, float kernel_,
								String connect_,boolean outputMaxMem_, boolean outputField_
								) {
        super(null, srcImg_[0]);
		srcImage = srcImg_;
        nc = nInput_;
		modality = imgModal_;
		
		psfType = new String[nc];
		for (int i=0;i<nc;i++) psfType[i] = "1x1x1";
		
		atlasName = aName_;
		atlas = atlas_;
		classes = atlas.getNumber();

		resultOutput = segOutput_;
        
		segment = true;
		smoothing = smooth_;
		outlier = 2.0f/(float)classes;
		iterations = nIterMax_;
		doGrowing = true;
        maxDistance = distMax_;
		lowEratio = (float)Math.sqrt(maxDistance);
		bgthreshold = 0.0f;
		
		useTopologyPrior = true;
		demonsSmoothing = 1.0f;
		demonsScale = 2.0f;
		marchingLimit = fLim_;
		volumeLimit = lLim_;
		//steps = 0;
		lesion = 0.1f;
		
		useShapePrior = true;
		atlasCoefficient = aCoeff_;
		atlasScale = aScale_;
		atlasCoupling = 0.5f;

		useIntensityPrior = true;
		
		//centroidMode = "linked";
		centroidMode = "damped";
		centroidSmoothness = 0.1f;
		
		algorithmMode = "linear_fast_marching";
		
		connectivityType = connect_;
		
		register = true;
		levels = lev_;
		firstIter = first_;
		topIter = top_;
		lastIter = last_;
		
		alignIter = topIter;
		initAlignIter = firstIter;
		
		alignType = align_;
		
		outputMaxMem = outputMaxMem_;
		correctInhomogeneity = correct_;
		if (correctInhomogeneity) {
			outputField = outputField_;
			polynomialDegree = poly_;
			splineKernel = kernel_;
			if (correctType_.equals("Splines")) correctionMethod = InhomogeneityCorrection2.SPLINES;
			else correctionMethod = InhomogeneityCorrection2.CHEBYSHEV;
		} else {
			outputField = false;
			polynomialDegree = 0;
			splineKernel = 0.0f;
			correctionMethod = InhomogeneityCorrection2.CHEBYSHEV;
		}

		useApprox = false;
		approx = atlas.getNumber();
	}

    /**
    *	Prepares this class for destruction.
    */
	public void finalize(){
	    destImage   = null;
	    srcImage    = null;
        super.finalize();
        System.gc();
	}
	
    /**
    *   Starts the segmentation.
    */
	public void runAlgorithm() {

        if (srcImage  == null) {
            displayError("Source Image is null");
            //notifyListeners(this);
            return;
        }
        
		// start the timer to compute the elapsed time
        setStartTime();

		//constructLog(true);
		
		float[][] buffer = null;
		if (srcImage[0].getNDims() == 2){
			try {
				// image length is length in 2 dims
				int length = srcImage[0].getExtents()[0]  * srcImage[0].getExtents()[1];
				 buffer = new float[nc][];
				for (int t=0;t<nc;t++) {
					buffer[t] = new float[length];
					srcImage[t].exportData(0,length, buffer[t]); // locks and releases lock
				}
			}
			catch (IOException error) {
				buffer = null;
				errorCleanUp("Segmentation reports: source image locked", true);
				return;
			}
			catch (OutOfMemoryError e){
				buffer = null;
				errorCleanUp("Segmentation reports: out of memory", true);
				return;
			}
			// init dimensions
			nx = srcImage[0].getExtents()[0];
			ny = srcImage[0].getExtents()[1];
			nz = 1;
			
			ntx = atlas.getImageDim()[0];
			nty = atlas.getImageDim()[1];
			ntz = 1;
			
			rx = srcImage[0].getFileInfo()[0].getResolutions()[0];
			ry = srcImage[0].getFileInfo()[0].getResolutions()[1];
			rz = 1.0f;
			
			// check for non-mm units
			if (srcImage[0].getFileInfo()[0].getUnitsOfMeasure()[0]==FileInfoBase.CENTIMETERS) rx = 10.0f*rx;
			if (srcImage[0].getFileInfo()[0].getUnitsOfMeasure()[1]==FileInfoBase.CENTIMETERS) ry = 10.0f*ry;
			
			rtx = atlas.getImageRes()[0];
			rty = atlas.getImageRes()[1];
			rtz = 1.0f;
			
			orient = srcImage[0].getImageOrientation();
			orx = srcImage[0].getFileInfo()[0].getAxisOrientation(0);
			ory = srcImage[0].getFileInfo()[0].getAxisOrientation(1);
			orz = FileInfoBase.UNKNOWN_ORIENT;
			
			// main segmentation
			calcSegmentation(buffer);
			
		}
		else if (srcImage[0].getNDims() > 2) {
		   try {
				// image length is length in 3 dims
				int length = srcImage[0].getExtents()[0] *
							 srcImage[0].getExtents()[1] *
							 srcImage[0].getExtents()[2];
									 buffer = new float[nc][];
				for (int t=0;t<nc;t++) {
					buffer[t] = new float[length];
					srcImage[t].exportData(0,length, buffer[t]); // locks and releases lock
				}
				fireProgressStateChanged(srcImage[0].getImageName(), "Processing image ...");
			}
			catch (IOException error) {
				buffer = null;
				errorCleanUp("Segmentation: source image locked", true);
				return;
			}
			catch (OutOfMemoryError e){
				buffer = null;
				errorCleanUp("Segmentation: Out of memory creating process buffer", true);
				return;
			}
			// init dimensions
			nx = srcImage[0].getExtents()[0];
			ny = srcImage[0].getExtents()[1];
			nz = srcImage[0].getExtents()[2];
			
			ntx = atlas.getImageDim()[0];
			nty = atlas.getImageDim()[1];
			ntz = atlas.getImageDim()[2];
			
			rx = srcImage[0].getFileInfo()[0].getResolutions()[0];
			ry = srcImage[0].getFileInfo()[0].getResolutions()[1];
			rz = srcImage[0].getFileInfo()[0].getResolutions()[2];
			
			// check for non-mm units
			if (srcImage[0].getFileInfo()[0].getUnitsOfMeasure()[0]==FileInfoBase.CENTIMETERS) rx = 10.0f*rx;
			if (srcImage[0].getFileInfo()[0].getUnitsOfMeasure()[1]==FileInfoBase.CENTIMETERS) ry = 10.0f*ry;
			if (srcImage[0].getFileInfo()[0].getUnitsOfMeasure()[2]==FileInfoBase.CENTIMETERS) rz = 10.0f*rz;
			
			rtx = atlas.getImageRes()[0];
			rty = atlas.getImageRes()[1];
			rtz = atlas.getImageRes()[2];
			
			orient = srcImage[0].getImageOrientation();
			orx = srcImage[0].getFileInfo()[0].getAxisOrientation(0);
			ory = srcImage[0].getFileInfo()[0].getAxisOrientation(1);
			orz = srcImage[0].getFileInfo()[0].getAxisOrientation(2);
			
			classes = atlas.getNumber();
	
			// main segmentation
			calcSegmentation(buffer);
			
        }

        // compute the elapsed time
        computeElapsedTime();
    } // end run()

    /**
    *	produces the segmentation of the input image
    */
    private void calcSegmentation(float img[][]) {
		byte[][][]   	classification = null; 
		byte[][][]   	priorclassif = null; 
		float[][]		centroids = null;
		float[][][][] 	memberships = null;
        float[][][][] 	field = null;
        float[][][]   	ordering = null; 
		float[][][]   	lesions = null; 
		byte[][][]      relations = null;
		byte[][][][]    best = null;
		boolean[][][]	obj;
		float[] 		intensityPrior;
		float   	num,den;
		float   	ngb, neighbors;
		float[]     buffer;
		byte[]		bytebuffer;
		MultipleToadSegmentation 	segmentation=null;
		MultipleToadPreprocess		preprocess=null;
		InhomogeneityCorrection2[]	correction=null;
		String      info = "";
		float		energy, Eprev, Eratio;
		float		correctRate;
		float[]		misclass = new float[classes];
		float[]		transform = null;
		
		int mod;
		int progress;

		boolean stop;
		
		if (verbose) MedicUtilPublic.displayMessage("TOADS\n");
		
		if (debug) {
			MedicUtilPublic.displayMessage(getParameterText());
		}
		
		fireProgressStateChanged(srcImage[0].getImageName(), "Initialization");
		fireProgressStateChanged(10);
		
		//---------------------------------------/
		//-          MAIN SEGMENTATION          -/
		//---------------------------------------/
		
		// record time
		long start_time = System.currentTimeMillis();
		long inner_loop_time;

        memberships = null;
        classification = null;
		
		// pre-processing
        if (verbose) MedicUtilPublic.displayMessage("Initialization\n");
			
		preprocess = new MultipleToadPreprocess(img, nc, nx, ny, nz, rx, ry, rz,
												orient, orx, ory, orz,
												atlas.getTemplate(), ntx, nty, ntz, rtx, rty, rtz,
												classes, (byte)1, 0.0f, 10);
		
		if (debug) System.out.println(preprocess.displayTransform());
		// put all intensities in [0,1]										
		preprocess.normalizeImages();
		// find unused regions
		preprocess.findCroppingBoundaries();
		// crop images and atlas
		preprocess.cropImages();
		if (debug) System.out.println("cropped: "+preprocess.displayTransform());
		int[] dim = preprocess.getCurrentImageDimensions();
		nx = dim[0]; ny = dim[1]; nz = dim[2];
		float[] res = preprocess.getSignedImageResolutions();
		rx = res[0]; ry = res[1]; rz = res[2];
		
		// first centroids, memberships : tissue types
		centroids = preprocess.initialCentroids(atlas.getIntensityPriors(modality,nc),classes);		
			
		fireProgressStateChanged(20);
			
		// align the atlas
		if (alignShapes) {
			if (verbose) MedicUtilPublic.displayMessage("Atlas alignment\n");
			
			if (useApprox) {
				memberships = new float[nx][ny][nz][approx];
				best = new byte[nx][ny][nz][approx];
			} else {
				memberships = new float[nx][ny][nz][classes];
				best = null;
			}
			preprocess.initialMemberships(memberships, best, approx, centroids, classes);
			
			if (debug) System.out.println("approx: "+useApprox+", "+approx);
			
			// align both images centers ? not very efficient
			//preprocess.alignImagesAndTopolgyCenters(atlas.getLabels(), centroids, classes);
			//if (debug) System.out.println("IMAGE DIMENSIONS "+nx+" "+ny+" "+nz);
			atlas.setImageInfo(nx,ny,nz,rx,ry,rz,orient,orx,ory,orz);
			
			atlas.initShapeRegistration(memberships, best, approx, "rigid", initAlignIter, levels);				
			
			atlas.setTransform(preprocess.getTransform());
			
			if (debug) System.out.println("pre-process: "+preprocess.displayTransform());
			
			atlas.updateShapeRegistration(memberships, best, null, modality, "rigid", initAlignIter, levels, 1.0f, 1.0f);
			if (debug) System.out.println("atlas: "+atlas.displayTransform(atlas.getTransform()));
			
			fireProgressStateChanged(30);
			
			atlas.registerShapesPyramid();
			
			/* better remove this and use more iterations for registration instead
			// add a last step ? not needed, can be nice as an option !! not working for topology !!
			atlas.updateShapeRegistration(memberships, best, alignType, lastIter, 1, 0.0f);
			if (debug) System.out.println("atlas: "+atlas.displayTransform(atlas.getTransform()));
			
			fireProgressStateChanged(50);
			
			atlas.registerShapes();
			*/
			
			memberships = null;
			
			if (debug) System.out.println("aligned atlas: "+atlas.displayTransform(atlas.getTransform()));
			if (debug) System.out.println("aligned pre-process: "+preprocess.displayTransform());
			
			// re-compute the boundaries to include everything
			preprocess.uncropImages();
			preprocess.findCroppingBoundaries();
			preprocess.cropImages();
			
			if (debug) System.out.println("final pre-process: "+preprocess.displayTransform());
			if (debug) System.out.println("final aligned atlas: "+atlas.displayTransform(atlas.getTransform()));
			
			dim = preprocess.getCurrentImageDimensions();
			nx = dim[0]; ny = dim[1]; nz = dim[2];
			
			atlas.setImageInfo(nx,ny,nz,rx,ry,rz,orient,orx,ory,orz);
			
			atlas.computeTransformedTemplate();
			
			// re-compute the centroids to get better estimates: not needed
			preprocess.updateTransformedTemplate(atlas.getTemplate());
			//centroids = preprocess.initialCentroids(atlas.getLabels(),atlas.getIntensityPriors(modality,nc),classes);		
		}
		
		fireProgressStateChanged(60);
			
		// create the segmentation algorithm
		segmentation = new MultipleToadSegmentation(preprocess.getImages(), atlas, 
								nx, ny, nz, nc, 
								smoothing, outlier,
								0.0f, 0.0f, marchingLimit, volumeLimit, 
								0, lesion, 
								atlasCoefficient, atlasScale, atlasCoupling,
								modality,
								algorithmMode, connectivityType,
								centroidMode, centroidSmoothness,
								approx,
								getProgressChangeListener());

		// get the centroids from initialization
		segmentation.initCentroids(centroids);
		segmentation.setIntensityMax(preprocess.getIntensityMax());
		segmentation.setIntensityScale(preprocess.getIntensityScale());
		segmentation.setLesionCentroid(preprocess.initialLesionCentroid(atlas.getLesionPriors(modality,nc)));

		// also use the initial memberships (re-computed)
		preprocess.initialMemberships(segmentation.getMemberships(), segmentation.getBest(), atlas, approx, centroids, classes);
		
		// re-initialize the atlas alignment
		atlas.updateShapeRegistration(segmentation.getMemberships(), segmentation.getBest(), segmentation.getSegmentation(),
										modality, alignType, alignIter, 1, demonsSmoothing, demonsScale);

		// non-rigid shape alignment ?
		if (alignShapes) atlas.registerShapes();
		
		if (verbose) MedicUtilPublic.displayMessage("aligned atlas: "+atlas.displayTransform(atlas.getTransform()));
			
		if (verbose) MedicUtilPublic.displayMessage("Centroid, membership and field initialization\n");
			
		if (verbose) MedicUtilPublic.displayMessage(segmentation.displayCentroids()); 
		if (debug) MedicUtilPublic.displayMessage("initial lesion centroid: \n"+segmentation.displayLesionCentroid()); 
		
		fireProgressStateChanged(70);
			
		if (correctInhomogeneity) {
			correction = new InhomogeneityCorrection2[nc];
			for (int c=0;c<nc;c++) {
				// create the inhomogeneity correction segmentation
				if (approx<classes && approx>1) {
					correction[c] = new InhomogeneityCorrection2(preprocess.getImages()[c],
														segmentation.getMemberships(),
														segmentation.getBest(),
														segmentation.getCentroids(c),
														classes, approx,
														segmentation.getSegmentation(),atlas.getTopology(),
														polynomialDegree, splineKernel,
														InhomogeneityCorrection2.IMAGE,
														InhomogeneityCorrection2.GLOBAL,
														correctionMethod,
														InhomogeneityCorrection2.FIXED,
														InhomogeneityCorrection2.EQUAL_VARIANCE,
														2.0f, null, null,
														nx,ny,nz,rx,ry,rz,
														getProgressChangeListener());
				} else {
					correction[c] = new InhomogeneityCorrection2(preprocess.getImages()[c],
														segmentation.getMemberships(),
														segmentation.getCentroids(c),
														classes,
														segmentation.getSegmentation(),atlas.getTopology(),
														polynomialDegree, splineKernel,
														InhomogeneityCorrection2.IMAGE,
														InhomogeneityCorrection2.GLOBAL,
														correctionMethod,
														InhomogeneityCorrection2.FIXED,
														InhomogeneityCorrection2.EQUAL_VARIANCE,
														2.0f, null, null,
														nx,ny,nz,rx,ry,rz,
														getProgressChangeListener());
				}
				// integrate it into the segmentation
				segmentation.addInhomogeneityCorrection(correction[c].getField(),c);
			}
		} else correction = null;
		
		if (debug) System.out.print("pre-processing time: " + (System.currentTimeMillis()-start_time)+"\n"); 
		
		fireProgressStateChanged(80);
					
		// relations
		segmentation.computeRelations();
		segmentation.computeBoundaryDistances();
		segmentation.computeClosestLabels();
		//segmentation.displayRelationList();
		
		if (debug) System.out.print("relation initialization time: " + (System.currentTimeMillis()-start_time)+"\n"); 
		
		segmentation.computeVariances();
		if (debug) MedicUtilPublic.displayMessage(segmentation.displayVariances());
		fireProgressStateChanged(90);
			
		// memberships & energy
		//energy = 0.0f;
		energy = segmentation.computeMemberships();
		
		if (verbose) MedicUtilPublic.displayMessage("initial energy : "+energy+"\n");
		
		if (debug) System.out.print("membership initialization time: " + (System.currentTimeMillis()-start_time)+"\n"); 
			
		Eprev = energy;
		Eratio = 1.0f;
			
		if (debug) System.out.print("final initialization time: " + (System.currentTimeMillis()-start_time)+"\n"); 
			
		if (debug) System.out.print("algorithm mode: " +algorithmMode+"\n"); 
		
		// 3. main loop
		//--------------------------------------------------------------------
		if (verbose) MedicUtilPublic.displayMessage("Main computation loop\n");
		stop = false;
		int iter = 0;
		int deg = 0;
		if (iterations==0) stop=true;
		while ((!stop) && (!threadStopped) && segmentation.isWorking()) {
			long iterationTime=0;
			if (debug) iterationTime = System.currentTimeMillis();
				
			iter++;
			if (verbose) MedicUtilPublic.displayMessage("\n iteration "+iter+"\n");
			
			// a. thinning
			//----------------------------------------------------
			//fireProgressStateChanged("iteration " + iter + " (thinning)");
			fireProgressStateChanged(10);
			
			segmentation.propagateThinning();
			
			if (debug) System.out.print("thinning time: " + (System.currentTimeMillis()-iterationTime)+"\n"); 
			if (debug) iterationTime = System.currentTimeMillis();
			fireProgressStateChanged(20);
					
			// b. update parameter information
			//------------------------------------------------------------
					
			// inhomogeneity field
			if (correctInhomogeneity) {
				if ( (deg<polynomialDegree) && (Eratio < lowEratio) ) deg++;
				for (int c=0;c<nc;c++) correction[c].computeCorrectionField(deg);
				fireProgressStateChanged(30);

				if (debug) System.out.print("inhomogeneity time: " + (System.currentTimeMillis()-iterationTime)+"\n"); 
				if (debug) iterationTime = System.currentTimeMillis();
			}
				
			// centroids
			segmentation.computeCentroids();
			if (verbose) MedicUtilPublic.displayMessage(segmentation.displayCentroids());
			if (debug) MedicUtilPublic.displayMessage(segmentation.displayLesionCentroid());
			// cluster-specific variations
			segmentation.computeVariances();
			//if (verbose) MedicUtilPublic.displayMessage(segmentation.displayVariances());
			//if (debug) MedicUtilPublic.displayMessage(segmentation.displayLesionVariance());
			// variance weights: no estimation
			//segmentation.computeVarianceWeights();
			if (debug) MedicUtilPublic.displayMessage(segmentation.displayVarianceWeights());
			
			fireProgressStateChanged(40);
			if (debug) System.out.print("centroid time: " + (System.currentTimeMillis()-iterationTime)+"\n"); 
			if (debug) iterationTime = System.currentTimeMillis();
				
			// memberships & energy
			Eprev = energy;
			energy = segmentation.computeMemberships();
			//segmentation.propagateExactMemberships();
			Eratio = Numerics.abs( 2.0f*(Eprev-energy)/(Eprev+energy) );
			//Eratio = 2.0f*(Eprev-energy)/(Eprev+energy);
			fireProgressStateChanged(60);
			
			if (debug) System.out.print("membership time: " + (System.currentTimeMillis()-iterationTime)+"\n"); 
			if (debug) iterationTime = System.currentTimeMillis();
				
			// shape alignment
			if (alignShapes) atlas.registerShapes();
				
			if (debug) System.out.print("alignment time: " + (System.currentTimeMillis()-iterationTime)+"\n"); 
			if (debug) iterationTime = System.currentTimeMillis();
			fireProgressStateChanged(80);
			
			// check for convergence
			if ( Eratio < maxDistance) stop = true;
			if (verbose) MedicUtilPublic.displayMessage("energy: "+energy+", "+"ratio: "+Eratio+"\n");
			
			if (iter >= iterations) stop = true;
				
			// c. growing
			//---------------------------------------------------------------
			//fireProgressStateChanged("iteration " + iter + " (growing)");
			if ( (!stop) || doGrowing) segmentation.propagateGrowing();
			
			if (debug) System.out.print("growing time: " + (System.currentTimeMillis()-iterationTime)+"\n"); 
			if (debug) iterationTime = System.currentTimeMillis();
			
			fireProgressStateChanged(90);
					
				
			// d. update classification only at the end
			//--------------------------------------------------------------
			if (stop) segmentation.updateClassificationFromLabels();
			
			// update the relation list
			if (!stop) {
				segmentation.computeRelations();
				segmentation.computeBoundaryDistances();
				segmentation.computeClosestLabels();
			}
			//else segmentation.displayRelationList();
				
			// decrease the smoothing if the skeleton is good enough
			// (must be done after the growth: changes the scaling)
			fireProgressStateChanged(99);
				
			if (debug) System.out.print("update time: " + (System.currentTimeMillis()-iterationTime)+"\n"); 
			if (debug) iterationTime = System.currentTimeMillis();
		}
		segmentation.cleanUp();
		
		if (verbose) MedicUtilPublic.displayMessage("Preparing output images\n");
		
		// debug
		if (verbose) MedicUtilPublic.displayMessage("total processing time (milliseconds): " + (System.currentTimeMillis()-start_time)); 
		
		fireProgressStateChanged("creating result images...");
		fireProgressStateChanged(30);
				
		try {
			int nImages, cnt;
			if (resultOutput.equals("hard segmentation")) {
				if (verbose) MedicUtilPublic.displayMessage("hard segmentation..\n");
				// create the output array
				nImages =1;
				if (outputMaxMem) nImages++;
				if (outputField) nImages++;
				destImage = new ModelImage[nImages];
				cnt=0;
			
				// hard segmentation
				destImage[cnt] = new ModelImage(ModelStorageBase.UBYTE, preprocess.getOriginalImageDimensions(),
                                                        JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_seg"));				

				bytebuffer = preprocess.uncropAndBuffer(segmentation.exportClassification(),(byte)1);
				destImage[cnt].importData(0, bytebuffer, true);
				cnt++;
				
				//maximum membership classification
				if (outputMaxMem){
					segmentation.updateClassificationFromMemberships();
					destImage[cnt] = new ModelImage(ModelStorageBase.UBYTE, preprocess.getOriginalImageDimensions(),
                            JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_seg_max_mem"));				

					bytebuffer = preprocess.uncropAndBuffer(segmentation.getClassification(),(byte)1);
					destImage[cnt].importData(0, bytebuffer, true);
					cnt++;
				}
				
				// inhomogeneity fields
				if (outputField){
					if (nc>1) {
						destImage[cnt] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(nc),
															JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_fields"));
						for (int c=0;c<nc;c++) {
							buffer = preprocess.uncropAndBuffer(correction[c].exportField(),1.0f);
							destImage[cnt].importData(c*preprocess.getOriginalImageSize(), buffer, true);
						}
					} else {
						destImage[cnt] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
															JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_field"));
						buffer = preprocess.uncropAndBuffer(correction[0].exportField(),1.0f);
						destImage[cnt].importData(0, buffer, true);
					}		
				}
				bytebuffer = null;
				buffer = null;
			} else if  (resultOutput.equals("hard segmentation+memberships")) {
				if (verbose) MedicUtilPublic.displayMessage("hard segmentation+memberships...\n");
				// create the output array
				nImages =2;
				if (outputMaxMem) nImages++;
				if (outputField) nImages++;
				destImage = new ModelImage[nImages];
				cnt=0;
				// hard segmentation
				destImage[cnt]= new ModelImage(ModelStorageBase.UBYTE, preprocess.getOriginalImageDimensions(),
                                                        JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_seg"));				

				bytebuffer = preprocess.uncropAndBuffer(segmentation.exportClassification(),(byte)1);
				destImage[cnt].importData(0, bytebuffer, true);
				cnt++;
				
				// memberships
				destImage[cnt] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(classes),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_mems"));
				memberships = segmentation.exportMemberships();
				for (int k=0;k<classes;k++) {
					buffer = preprocess.uncropAndBuffer(memberships[k],0.0f);
					destImage[cnt].importData(k*preprocess.getOriginalImageSize(), buffer, true);
				}
				memberships = null;
				cnt++;
				
				//maximum membership classification
				if (outputMaxMem){
					segmentation.updateClassificationFromMemberships();
					destImage[cnt] = new ModelImage(ModelStorageBase.UBYTE, preprocess.getOriginalImageDimensions(),
                            JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_seg_max_mem"));				

					bytebuffer = preprocess.uncropAndBuffer(segmentation.getClassification(),(byte)1);
					destImage[cnt].importData(0, bytebuffer, true);
					cnt++;
				}
				
				// inhomogeneity fields
				if (outputField){
					if (nc>1) {
						destImage[cnt] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(nc),
															JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_fields"));
						for (int c=0;c<nc;c++) {
							buffer = preprocess.uncropAndBuffer(correction[c].exportField(),1.0f);
							destImage[cnt].importData(c*preprocess.getOriginalImageSize(), buffer, true);
						}
					} else {
						destImage[cnt] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
															JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_field"));
						buffer = preprocess.uncropAndBuffer(correction[0].exportField(),1.0f);
						destImage[cnt].importData(0, buffer, true);
					}		
				}
				bytebuffer = null;
				buffer = null;
			} else if  (resultOutput.equals("cruise inputs")||resultOutput.equals("dura removal inputs")) {
				if (verbose) MedicUtilPublic.displayMessage("Cruise input..\n");
				// create the output array
				nImages =5;
				if (outputMaxMem) nImages++;
				if (outputField) nImages++;
				destImage = new ModelImage[nImages];
				cnt=0;
			
				// hard segmentation
				destImage[cnt] = new ModelImage(ModelStorageBase.UBYTE, preprocess.getOriginalImageDimensions(),
                                                        JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_seg"));				

				bytebuffer = preprocess.uncropAndBuffer(segmentation.exportClassification(),(byte)1);
				destImage[cnt].importData(0, bytebuffer, true);
				cnt++;
				
				
				// Cruise images
				if (resultOutput.equals("cruise inputs")) 
					memberships = segmentation.generateCruiseOutputs();
				else
					memberships = segmentation.generateDuraRemovalOutputs();
				
				destImage[cnt] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_csf"));
				destImage[cnt+1] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_gm"));
				destImage[cnt+2] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_wmfill"));
				destImage[cnt+3] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_wmmask"));
				cnt +=4;
				for (int k=0;k<4;k++) {
					buffer = preprocess.uncropAndBuffer(memberships[k],0.0f);
					destImage[k+1].importData(0, buffer, true);
				}
				memberships = null;
				
				//maximum membership classification
				if (outputMaxMem){
					segmentation.updateClassificationFromMemberships();
					destImage[cnt] = new ModelImage(ModelStorageBase.UBYTE, preprocess.getOriginalImageDimensions(),
                            JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_seg_max_mem"));				

					bytebuffer = preprocess.uncropAndBuffer(segmentation.getClassification(),(byte)1);
					destImage[cnt].importData(0, bytebuffer, true);
					cnt++;
				}
				
				
				// inhomogeneity fields
				if (outputField){
					if (nc>1) {
						destImage[cnt] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(nc),
															JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_fields"));
						for (int c=0;c<nc;c++) {
							buffer = preprocess.uncropAndBuffer(correction[c].exportField(),1.0f);
							destImage[cnt].importData(c*preprocess.getOriginalImageSize(), buffer, true);
						}
					} else {
						destImage[cnt] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
															JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_field"));
						buffer = preprocess.uncropAndBuffer(correction[0].exportField(),1.0f);
						destImage[cnt].importData(0, buffer, true);
					}		
				}
				bytebuffer = null;
				buffer = null;
			} else if  (resultOutput.equals("thalamus segmentation")) {
				if (verbose) MedicUtilPublic.displayMessage("Thalamus segmentation input..\n");
				// create the output array
				nImages =2;
				if (outputMaxMem) nImages++;
				if (outputField) nImages++;
				destImage = new ModelImage[nImages];
				cnt=0;
			
				// hard segmentation
				destImage[cnt] = new ModelImage(ModelStorageBase.UBYTE, preprocess.getOriginalImageDimensions(),
                                                        JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_seg"));				

				bytebuffer = preprocess.uncropAndBuffer(segmentation.exportClassification(),(byte)1);
				destImage[cnt].importData(0, bytebuffer, true);
				cnt++;
				
				
				// Thalamus segmentation
				memberships = segmentation.generateThalamusOutput();
				
				destImage[cnt] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_thalamus"));
					
				buffer = preprocess.uncropAndBuffer(memberships[0],0.0f);
				destImage[cnt].importData(0, buffer, true);
				cnt++;
				
				//maximum membership classification
				if (outputMaxMem){
					segmentation.updateClassificationFromMemberships();
					destImage[cnt] = new ModelImage(ModelStorageBase.UBYTE, preprocess.getOriginalImageDimensions(),
                            JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_seg_max_mem"));				

					bytebuffer = preprocess.uncropAndBuffer(segmentation.getClassification(),(byte)1);
					destImage[cnt].importData(0, bytebuffer, true);
					cnt++;
				}
				
				
				// inhomogeneity fields
				if (outputField){
					if (nc>1) {
						destImage[cnt] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(nc),
															JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_fields"));
						for (int c=0;c<nc;c++) {
							buffer = preprocess.uncropAndBuffer(correction[c].exportField(),1.0f);
							destImage[cnt].importData(c*preprocess.getOriginalImageSize(), buffer, true);
						}
					} else {
						destImage[cnt] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
															JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_field"));
						buffer = preprocess.uncropAndBuffer(correction[0].exportField(),1.0f);
						destImage[cnt].importData(0, buffer, true);
					}		
				}
				bytebuffer = null;
				memberships = null;
				buffer = null;				
			} else if  (resultOutput.equals("all_images")) {
				if (verbose) MedicUtilPublic.displayMessage("All images..\n");
				// create the output array
				
				nImages =4;
				if (outputMaxMem) nImages++;
				if (outputField) nImages++;
				if (atlas.isDeformable()) nImages++;
				destImage = new ModelImage[nImages];
				cnt=0;
			
				// hard segmentation
				destImage[cnt] = new ModelImage(ModelStorageBase.UBYTE, preprocess.getOriginalImageDimensions(),
                                                        JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_seg"));				

				bytebuffer = preprocess.uncropAndBuffer(segmentation.exportClassification(),(byte)1);
				destImage[cnt].importData(0, bytebuffer, true);
				cnt++;
				
			
				// memberships
				destImage[cnt] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(classes),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_mems"));
				memberships = segmentation.exportMemberships();
				for (int k=0;k<classes;k++) {
					buffer = preprocess.uncropAndBuffer(memberships[k],0.0f);
					destImage[cnt].importData(k*preprocess.getOriginalImageSize(), buffer, true);
				}
				memberships = null;
				cnt++;
				
				// extra maps
				destImage[cnt] = new ModelImage(ModelStorageBase.UBYTE, preprocess.getOriginalImageArrayDimensions(8),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_maps"));
				
				bytebuffer = preprocess.uncropAndBuffer(segmentation.generateClassification(),(byte)0);
				destImage[cnt].importData(0, bytebuffer, true);
				
				bytebuffer = preprocess.uncropAndBuffer(atlas.generateTransformedClassification(),(byte)0);
				destImage[cnt].importData(1*preprocess.getOriginalImageSize(), bytebuffer, true);
				
				bytebuffer = preprocess.uncropAndBuffer(segmentation.exportRelationMap(),(byte)0);
				destImage[cnt].importData(2*preprocess.getOriginalImageSize(), bytebuffer, true);
				
				bytebuffer = preprocess.uncropAndBuffer(segmentation.exportSkeleton(),(byte)0);
				destImage[cnt].importData(3*preprocess.getOriginalImageSize(), bytebuffer, true);
				
				bytebuffer = preprocess.uncropAndBuffer(segmentation.exportClosestLabelMap(0),(byte)0);
				destImage[cnt].importData(4*preprocess.getOriginalImageSize(), bytebuffer, true);
				
				bytebuffer = preprocess.uncropAndBuffer(segmentation.exportClosestLabelMap(1),(byte)0);
				destImage[cnt].importData(5*preprocess.getOriginalImageSize(), bytebuffer, true);
				
				bytebuffer = preprocess.uncropAndBuffer(segmentation.exportClosestLabelMap(2),(byte)0);
				destImage[cnt].importData(6*preprocess.getOriginalImageSize(), bytebuffer, true);
				
				bytebuffer = preprocess.uncropAndBuffer(segmentation.exportClosestLabelMap(3),(byte)0);
				destImage[cnt].importData(7*preprocess.getOriginalImageSize(), bytebuffer, true);
				cnt++;
				
				// prior memberships
				destImage[cnt] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(classes),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_priormems"));
				memberships = atlas.generateTransformedShapes();
				for (int k=0;k<classes;k++) {
					buffer = preprocess.uncropAndBuffer(memberships[k],0.0f);
					destImage[cnt].importData(k*preprocess.getOriginalImageSize(), buffer, true);
				}
				cnt++;
								
				if (atlas.isDeformable()) {
					destImage[cnt] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(3),
														JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_vec"));
					buffer = preprocess.uncropAndBuffer(atlas.exportDeformationField(),0.0f,3);
					destImage[cnt].importData(0, buffer, true);
					cnt++;
				}
				
				//maximum membership classification
				if (outputMaxMem){
					segmentation.updateClassificationFromMemberships();
					destImage[cnt] = new ModelImage(ModelStorageBase.UBYTE, preprocess.getOriginalImageDimensions(),
                            JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_seg_max_mem"));				

					bytebuffer = preprocess.uncropAndBuffer(segmentation.getClassification(),(byte)1);
					destImage[cnt].importData(0, bytebuffer, true);
					cnt++;
				}
				
				
				// inhomogeneity fields
				if (outputField){
					if (nc>1) {
						destImage[cnt] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(nc),
															JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_fields"));
						for (int c=0;c<nc;c++) {
							buffer = preprocess.uncropAndBuffer(correction[c].exportField(),1.0f);
							destImage[cnt].importData(c*preprocess.getOriginalImageSize(), buffer, true);
						}
					} else {
						destImage[cnt] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
															JDialogBase.makeImageName(srcImage[0].getImageName(), "_Toads_field"));
						buffer = preprocess.uncropAndBuffer(correction[0].exportField(),1.0f);
						destImage[cnt].importData(0, buffer, true);
					}		
				}
				bytebuffer = null;
				memberships = null;
				buffer = null;
									
			} 
		} catch (OutOfMemoryError e) {
			buffer = null;
			errorCleanUp("Segmentation TOADS: Out of memory creating hard classification", true);
			System.err.println(e.getMessage());
			finalize();
			
			setCompleted(false);
			return;
		} catch (IOException error) {
			errorCleanUp("Segmentation TOADS: export problem to destImage[]", true);
			System.err.println(error.getMessage());
			finalize();
			
			setCompleted(false);
			return;
		}

		fireProgressStateChanged(90);
		
		// clean-up the algorithms
		if (atlas!=null) atlas.finalize();
		atlas = null;
		if (segmentation!=null) segmentation.finalize();
		segmentation = null;
		if (correction!=null) {
			for (int c=0;c<nc;c++) if (correction[c]!=null) correction[c].finalize();
			correction = null;
		}
		if (preprocess!=null) preprocess.finalize();
		preprocess = null;
		
        if (debug) if (verbose) MedicUtilPublic.displayMessage("output...\n");

		
		memberships = null;
		classification = null;
		relations = null;
		lesions = null;
		priorclassif = null;
		
		threadStopped = false; // to ensure we output a result even if 'cancel' was hit
		setCompleted(true);
		threadStopped = false;
    } // calcSegmentation
	
    public ModelImage[] getResultImages(){
    	return destImage;
    }

    /**
     * Construct a readable list of the parameters to this segmentation.
     * @return       the parameter string
     */
	public String getParameterText() {
		String delim = "\n";
        String str = new String();
		for (int c=0;c<nc;c++) {
			str += "image "+(c+1)+": "+  srcImage[c].getImageName() + " (modality: "+modality[c]+")"+ delim;
		}
        str += "atlas name: " + atlasName + delim;
        str += "results: "+ resultOutput + delim;
		
		str += "segment: "+ segment + delim;
		str += "smoothing: " + smoothing + delim;
        str += "outliers: "+ outlier + delim;
        str += "max difference: "+ maxDistance + delim;
        str += "iterations: "+ iterations + delim;
        str += "background threshold: "+ bgthreshold + delim;
		
		str += "use topology: "+ useTopologyPrior + delim;
		str += "demons smoothing: "+ demonsSmoothing + delim;
        str += "demons scale: "+ demonsScale + delim;
        str += "first limit: "+ marchingLimit + delim;
        str += "last limit: "+ volumeLimit + delim;
        //str += "steps: "+ steps + delim;
        str += "lesion: "+ lesion + delim;
		
		str += "use shape prior: "+ useShapePrior + delim;
		str += "atlas coefficient: "+ atlasCoefficient + delim;
		str += "atlas scale: "+ atlasScale + delim;
		
		str += "use intensity prior: "+ useIntensityPrior + delim;
		
		str += "relation mode: "+ relationMode + delim;
		str += "centroid mode: "+ centroidMode + delim;
		str += "centroid smoothing: "+ centroidSmoothness + delim;
		str += "alignment type: "+ alignType + delim;
		str += "connectivity: "+ connectivityType + delim;
		
		str += "register: "+ register + delim;
		str += "registration steps (first,top,last): " + firstIter +", "+ topIter +", "+ lastIter + delim;
        str += "registration levels: "+ levels + delim;
        
		str += "inhomogeneity correction: "+ correctInhomogeneity + delim;
		str += "correction method: "+ correctionMethod + delim;
        str += "degree: "+ polynomialDegree + delim;
        str += "kernel: "+ splineKernel + delim;
        
        return str;
    }

	/** access the atlas information */
	public final MultipleToadDeformableAtlas getAtlas() { return atlas; }
}
