package edu.jhmi.rad.medic.dialogs;

import edu.jhmi.rad.medic.algorithms.*;
import edu.jhmi.rad.medic.utilities.*;

import gov.nih.mipav.plugins.*;     
import gov.nih.mipav.view.*;
import gov.nih.mipav.view.dialogs.*;
import gov.nih.mipav.model.structures.*;
import gov.nih.mipav.model.file.*;
import gov.nih.mipav.model.algorithms.*;
import gov.nih.mipav.model.scripting.*;
import gov.nih.mipav.model.scripting.parameters.*;

import java.awt.event.*;
import java.awt.*;
import java.util.*;
import java.io.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;
 /** 
*   
*   Dialog class for the Fantasm plug-in, full version (with all possible options)
*
*   
*	@version    July 2002
*	@author     Pilou Bazin
*   @see        AlgorithmFantasm
*   @see        AlgorithmFantasmLongitudinal
*   @see        AlgorithmFantasmMultichannel
*
*
*/  
public class JDialogFantasms extends JDialogScriptableBase implements AlgorithmInterface, DialogDefaultsInterface {
    
    private     AlgorithmFantasm		algo = null;
    private     AlgorithmFantasmLongitudinal		algo2 = null;
    private     AlgorithmFantasmMultichannel		algo3 = null;
    private     ModelImage              image;                // reference image
    private     ModelImage              images[];             // source images
    private     ModelImage              resultImage[] = null; // result images
    private     int                     resultNumber;
    private     int                     imageNumber;
    private     int                     presentNumber;
    private     int                     destExtents[];
    private 	ViewUserInterface       userInterface;
    private     String                  titles[];
	private		int[]					order;
	private		int						nOrder;
	private		int						borders=0;

	// parameters
	// classification
	private 	int 		nClasses		= 3;
	private 	int 		nIterations		= 50;
	private 	float 		maxDiff			= 0.01f;
	private 	float 		smoothing		= 0.01f;
    private 	float 		fuzziness		= 2.0f;
    private		boolean		addOutliers		= false;
	private		float		outlierRatio	= 0.25f;
    private     String      algoType = "single_image";
	private     String[]    algoTypes = {"single_image","multichannel","longitudinal"};
    private     String      initMode = "range";
	private     String[]    initModes = {"range","modes","manual"};
    private     String      scalingMode = "intensity";
	private     String[]    scalingModes = {"intensity","energy","variance","centroids"};
    private     String      distanceMode = "Gaussian";
	private     String[]    distanceModes = {"Gaussian","Gauss-Rayleigh","Gauss-Rayleigh2","Rician"};

	// temporal / multi channel
	private 	float 		temporal		= 0.05f;

	// registration
	private		boolean		useRegistration = false;
	private		int			topIter	= 10;
	private		int			firstIter	= 2;
	private		int			lastIter	= 3;
	private		int			levels		= 3;
	
	// inhomogeneity
	private		boolean		correctField		= true;
	private		int			polynomialDegree		= 3;
	private		String[]	polynomialDegrees		= {"1","2","3","4"};
	
	// edges
	private		boolean		useEdges = false;
	private		float		edgeSmoothness = 0.05f;
	private		float		edgeContrast = 0.05f;
	private		float		edgePrior = 0.001f;
	
	// general
    private     float       backThreshold	= 0.0f;
    private     boolean     cropBackground	= true;
	private		boolean		useRelative		= true;
    private     String      outputType = "hard_segmentation";
	private     String[]    outputTypes = {"all_result_images", "hard_segmentation", "fuzzy_segmentation", "both_fuzzy_&_hard"};
	private		boolean		adaptiveSmoothing = false;     
	
	// logging
	private		boolean		verbose = true;
	
    // dialog elements
	private		JTabbedPane	tabPane;
	private		JPanel		mainPanel;
	private		JPanel		imagePanel;
	private		JPanel		parameterPanel;
    
	private 	JPanel  	classifPanel;
	private 	JTextField  textNClasses;
	private 	JLabel  	labelNClasses;
	private		JComboBox	comboType;
	private		JLabel		labelType;
	
	private		JPanel		classifParamPanel;
	private 	JTextField  textMaxIter;
	private 	JLabel  	labelMaxIter;
	private 	JTextField  textMaxDiff;
	private 	JLabel  	labelMaxDiff;
	private 	JTextField  textSmoothing;
	private 	JLabel  	labelSmoothing;
	private 	JTextField  textFuzzy;
	private 	JLabel  	labelFuzzy;
	private 	JTextField  textTemporal;
	private 	JLabel  	labelTemporal;
	private 	JComboBox   comboInit;
	private 	JLabel  	labelInit;
	private 	JComboBox   comboScaling;
	private 	JLabel  	labelScaling;
	private     JCheckBox   checkAdaptive;
	
    private		JPanel		generalPanel;
	private     JCheckBox   checkCrop;
	private		JComboBox	comboResult;
	private		JLabel		labelResult;
    
	private		JPanel		generalParamPanel;
	private 	JTextField  textSignal;
	private 	JLabel  	labelSignal;
    private     JCheckBox   checkRelative;
	
	private		JPanel		optionPanel;
	private     JCheckBox   checkInhomog;
	private     JCheckBox   checkOutliers;
	private     JCheckBox   checkEdges;
	private 	JComboBox   comboDistance;
	private 	JLabel  	labelDistance;
	
	private		JPanel		optionParamPanel;
	//private		JPanel		correctionPanel;
	private		JComboBox	comboPolyDeg;
	private		JLabel		labelPolyDeg;
    
	//private		JPanel		outlierPanel;
	private 	JTextField  textOutliers;
	private 	JLabel  	labelOutliers;

	private		JPanel		registrationPanel;
	private     JCheckBox   checkRegister;
	private 	JTextField  textTopIter;
	private 	JTextField  textFirstIter;
	private 	JTextField  textLastIter;
	private 	JLabel  	labelRegIter;
	private 	JTextField  textRegLevels;
	private 	JLabel  	labelRegLevels;
	
	//private		JPanel		edgesPanel;
	private 	JTextField  textEdgeSmoothing;
	private 	JLabel  	labelEdgeSmoothing;
	private 	JTextField  textEdgeContrast;
	private 	JLabel  	labelEdgeContrast;
	private 	JTextField  textEdgePrior;
	private 	JLabel  	labelEdgePrior;

	// label list
	private		JPanel			imageListPanel;
	private		JCheckBox[]		checkImage;
	private		JLabel[]		labelOrder;
	private		int				nImage = 0;
	private		JLabel			labelImage;
	private		JPanel			viewPanel;
	private		JScrollPane		scrollImage;

    /**
    *  Creates dialog for plugin.
    *  @param theParentFrame          Parent frame.
    *  @param im              Source image.
    */
    public JDialogFantasms(Frame theParentFrame, ModelImage im) {
		super(theParentFrame, false);
		if (im.getType() == ModelImage.BOOLEAN || im.isColorImage()) {
            MipavUtil.displayError("Source Image must NOT be Boolean or Color"); 
            dispose();
            return;
        }
        image = im;
		userInterface = ((ViewJFrameBase)(parentFrame)).getUserInterface();	    
		loadDefaults();
        init();
        setVisible(true);
	}
	
    /**
    *	Used primarily for the script to store variables and run the algorithm.  No
    *	actual dialog will appear but the set up info and result image will be stored here.
    *	@param UI   The user interface, needed to create the image frame.
    *	@param im	Source image.
    */
    public JDialogFantasms(ViewUserInterface UI, ModelImage im) {
        super();
    	userInterface = UI;
    	image = im;
        parentFrame = image.getParentFrame();
    }
    
    /**
     * Empty constructor needed for dynamic instantiation.
     */
    public JDialogFantasms() {};

    /**
     * Construct a delimited string that contains the parameters to this algorithm.
     * @param delim  the parameter delimiter (defaults to " " if empty)
     * @return       the parameter string
     */
	public String getParameterString( String delim ) {
        if ( delim.equals( "" ) ) {
            delim = " ";
        }

        String str = new String();
        str += nClasses + delim;
        str += nIterations + delim;
        str += maxDiff + delim;
        str += smoothing + delim;
        str += fuzziness + delim;
        str += addOutliers + delim;
        str += outlierRatio + delim;
        str += algoType + delim;
        str += initMode + delim;
        str += scalingMode + delim;
        str += temporal + delim;
        str += useRegistration + delim;
        str += topIter + delim;
        str += firstIter + delim;
        str += lastIter + delim;
        str += levels + delim;
        str += correctField + delim;
        str += polynomialDegree + delim;
        str += useEdges + delim;
        str += edgeSmoothness + delim;
        str += edgeContrast + delim;
        str += edgePrior + delim;
		str += distanceMode + delim;
        str += backThreshold + delim;
        str += cropBackground + delim;
        str += useRelative + delim;
        str += adaptiveSmoothing + delim;
        str += outputType;

        return str;
    }
	

    /**
     *  Loads the default settings from Preferences to set up the dialog
     */
	public void loadDefaults() {
        String defaultsString = Preferences.getDialogDefaults(getDialogName());

        if (defaultsString != null) {

            try {
                System.out.println(defaultsString);
                StringTokenizer st = new StringTokenizer(defaultsString, ",");
				nClasses = MipavUtil.getInt(st);
				nIterations = MipavUtil.getInt(st);
				maxDiff = MipavUtil.getFloat(st);
				smoothing = MipavUtil.getFloat(st);
				fuzziness = MipavUtil.getFloat(st);
				addOutliers = MipavUtil.getBoolean(st);
				outlierRatio = MipavUtil.getFloat(st);
				algoType = st.nextToken();
				initMode = st.nextToken();
				scalingMode = st.nextToken();
				temporal = MipavUtil.getFloat(st);
				useRegistration = MipavUtil.getBoolean(st);
				topIter = MipavUtil.getInt(st);
				firstIter = MipavUtil.getInt(st);
				lastIter = MipavUtil.getInt(st);
				levels = MipavUtil.getInt(st);
				correctField = MipavUtil.getBoolean(st);
				polynomialDegree = MipavUtil.getInt(st);
				useEdges = MipavUtil.getBoolean(st);
				edgeSmoothness = MipavUtil.getFloat(st);
				edgeContrast = MipavUtil.getFloat(st);
				edgePrior = MipavUtil.getFloat(st);
				distanceMode = st.nextToken();
				backThreshold = MipavUtil.getFloat(st);
				cropBackground = MipavUtil.getBoolean(st);
				useRelative = MipavUtil.getBoolean(st);
				adaptiveSmoothing = MipavUtil.getBoolean(st);
				outputType = st.nextToken();
            }
            catch (Exception ex) {
                // since there was a problem parsing the defaults string, start over with the original defaults
                System.out.println( "Resetting defaults for dialog: " + getDialogName() );
                Preferences.removeProperty( getDialogName() );
            }
        } else {
			System.out.println( "no saved dialogs for "+getDialogName() );
		}
    }
		
    /**
     * Saves the default settings into the Preferences file
     */
	
    public void saveDefaults() {
        String defaultsString = new String( getParameterString(",") );
        System.out.println(defaultsString);
        Preferences.saveDialogDefaults(getDialogName(),defaultsString);
    }
	 

    /**
    *	Sets up the GUI (panels, buttons, etc) and displays it on the screen.
    */
	private void init(){
        setForeground(Color.black);
		setTitle("Fantasm(s)");
				
        classifPanel = new JPanel(new GridBagLayout());
		classifPanel.setBorder(buildTitledBorder("Classification"));
		
 		labelNClasses = new JLabel("Number of classes ");
		labelNClasses.setForeground(Color.black);
		labelNClasses.setFont(serif12);

		textNClasses = new JTextField(5);
		textNClasses.setText(String.valueOf(nClasses));
		textNClasses.setFont(serif12);
		        		
 		labelType = new JLabel("Algorithm ");
		labelType.setForeground(Color.black);
		labelType.setFont(serif12);

		comboType = new JComboBox(algoTypes);
		comboType.setFont(serif12);
        comboType.setBackground(Color.white);
		comboType.setSelectedItem(algoType);

		classifParamPanel = new JPanel(new GridBagLayout());
		classifParamPanel.setBorder(buildTitledBorder("Classification"));
		
		labelMaxIter = new JLabel("Maximum iterations ");
		labelMaxIter.setForeground(Color.black);
		labelMaxIter.setFont(serif12);
	
		textMaxIter = new JTextField(5);
		textMaxIter.setText(String.valueOf(nIterations));
		textMaxIter.setFont(serif12);
	
		labelMaxDiff = new JLabel("Maximum difference ");
		labelMaxDiff.setForeground(Color.black);
		labelMaxDiff.setFont(serif12);
	
		textMaxDiff = new JTextField(5);
		textMaxDiff.setText(String.valueOf(maxDiff));
		textMaxDiff.setFont(serif12);
	
		labelSmoothing = new JLabel("Smoothing parameter ");
		labelSmoothing.setForeground(Color.black);
		labelSmoothing.setFont(serif12);
	
		textSmoothing = new JTextField(5);
		textSmoothing.setText(String.valueOf(smoothing));
		textSmoothing.setFont(serif12);
	
        labelTemporal = new JLabel("Temporal correlation ");
		labelTemporal.setForeground(Color.black);
		labelTemporal.setFont(serif12);
	
		textTemporal = new JTextField(5);
		textTemporal.setText(String.valueOf(temporal));
		textTemporal.setFont(serif12);
	
        labelFuzzy = new JLabel("Fuzziness coefficient ");
		labelFuzzy.setForeground(Color.black);
		labelFuzzy.setFont(serif12);
	
		textFuzzy = new JTextField(5);
		textFuzzy.setText(String.valueOf(fuzziness));
		textFuzzy.setFont(serif12);
	
		labelInit = new JLabel("Initialization ");
		labelInit.setForeground(Color.black);
		labelInit.setFont(serif12);
	
		comboInit = new JComboBox(initModes);
		comboInit.setFont(serif12);
        comboInit.setBackground(Color.white);
		comboInit.setSelectedItem(initMode);
	
        labelScaling = new JLabel("Scaling mode ");
		labelScaling.setForeground(Color.black);
		labelScaling.setFont(serif12);
	
		comboScaling = new JComboBox(scalingModes);
		comboScaling.setFont(serif12);
        comboScaling.setBackground(Color.white);
		comboScaling.setSelectedItem(scalingMode);
	
		checkAdaptive = new JCheckBox("Energy-adaptive smoothing");
        checkAdaptive.setFont(serif12);
        checkAdaptive.setSelected(adaptiveSmoothing);
        
        optionPanel = new JPanel(new GridBagLayout());
		optionPanel.setBorder(buildTitledBorder("Options"));
		
        checkOutliers = new JCheckBox("Use outlier class");
        checkOutliers.setFont(serif12);
        checkOutliers.setSelected(addOutliers);
        
        checkInhomog = new JCheckBox("Correct inhomogeneity");
        checkInhomog.setFont(serif12);
        checkInhomog.setSelected(correctField);

        checkEdges = new JCheckBox("Detect edges");
        checkEdges.setFont(serif12);
        checkEdges.setSelected(useEdges);

        labelDistance = new JLabel("Distance mode ");
		labelDistance.setForeground(Color.black);
		labelDistance.setFont(serif12);
	
		comboDistance = new JComboBox(distanceModes);
		comboDistance.setFont(serif12);
        comboDistance.setBackground(Color.white);
		comboDistance.setSelectedItem(distanceMode);
	
		optionParamPanel = new JPanel(new GridBagLayout());
		optionParamPanel.setBorder(buildTitledBorder("Options"));
		
		labelOutliers = new JLabel("Outlier level ");
        labelOutliers.setForeground(Color.black);
        labelOutliers.setFont(serif12);
        
        textOutliers = new JTextField(5);
        textOutliers.setText(String.valueOf(outlierRatio));
        textOutliers.setFont(serif12);
                
		labelPolyDeg = new JLabel("Inhomogeneity field degree ");
		labelPolyDeg.setForeground(Color.black);
		labelPolyDeg.setFont(serif12);
	
		comboPolyDeg = new JComboBox(polynomialDegrees);
		comboPolyDeg.setFont(serif12);
        comboPolyDeg.setBackground(Color.white);
		comboPolyDeg.setSelectedItem(""+polynomialDegree);
	
		labelEdgeSmoothing = new JLabel("Edge smoothing ");
		labelEdgeSmoothing.setForeground(Color.black);
		labelEdgeSmoothing.setFont(serif12);
	
		textEdgeSmoothing = new JTextField(5);
		textEdgeSmoothing.setText(String.valueOf(edgeSmoothness));
		textEdgeSmoothing.setFont(serif12);
	
		labelEdgeContrast = new JLabel("Edge contrast ");
		labelEdgeContrast.setForeground(Color.black);
		labelEdgeContrast.setFont(serif12);
	
		textEdgeContrast = new JTextField(5);
		textEdgeContrast.setText(String.valueOf(edgeContrast));
		textEdgeContrast.setFont(serif12);
	
		labelEdgePrior = new JLabel("Edge prior ");
		labelEdgePrior.setForeground(Color.black);
		labelEdgePrior.setFont(serif12);
	
		textEdgePrior = new JTextField(5);
		textEdgePrior.setText(String.valueOf(edgePrior));
		textEdgePrior.setFont(serif12);
					
        generalPanel = new JPanel(new GridBagLayout());
		generalPanel.setBorder(buildTitledBorder("General"));
		
        checkCrop = new JCheckBox("Background cropping");
        checkCrop.setFont(serif12);
        checkCrop.setSelected(cropBackground);
        
		labelResult = new JLabel("Output ");
		labelResult.setForeground(Color.black);
		labelResult.setFont(serif12);
	
		comboResult = new JComboBox(outputTypes);
		comboResult.setSelectedItem(outputType);
		comboResult.setFont(serif12);
        comboResult.setBackground(Color.white);
				
        generalParamPanel = new JPanel(new GridBagLayout());
		generalParamPanel.setBorder(buildTitledBorder("General"));
		
        labelSignal = new JLabel("Background threshold ");
        labelSignal.setForeground(Color.black);
        labelSignal.setFont(serif12);

        textSignal = new JTextField(5);
        textSignal.setText(String.valueOf(backThreshold));
        textSignal.setFont(serif12);
        
        checkRelative = new JCheckBox("Use relative values");
        checkRelative.setFont(serif12);
        checkRelative.setSelected(useRelative);

        imageListPanel = new JPanel(new GridBagLayout());
		imageListPanel.setBorder(buildTitledBorder("Images"));

		scrollImage = new JScrollPane();
		scrollImage.setMaximumSize(new Dimension(200,200));
		scrollImage.setPreferredSize(new Dimension(100,100));
		scrollImage.setMinimumSize(new Dimension(30,30));
		displayImageList();

		registrationPanel = new JPanel(new GridBagLayout());
        registrationPanel.setBorder(buildTitledBorder("Registration"));
		
        checkRegister = new JCheckBox("Register images");
        checkRegister.setFont(serif12);
        checkRegister.setSelected(useRegistration);

		labelRegIter = new JLabel("Iterations (first|top|last) ");
		labelRegIter.setForeground(Color.black);
		labelRegIter.setFont(serif12);
	
		textFirstIter = new JTextField(3);
		textFirstIter.setText(String.valueOf(firstIter));
		textFirstIter.setFont(serif12);
	
		textTopIter = new JTextField(3);
		textTopIter.setText(String.valueOf(topIter));
		textTopIter.setFont(serif12);
	
		textLastIter = new JTextField(3);
		textLastIter.setText(String.valueOf(lastIter));
		textLastIter.setFont(serif12);
	
		labelRegLevels = new JLabel("Multigrid levels ");
		labelRegLevels.setForeground(Color.black);
		labelRegLevels.setFont(serif12);
	
		textRegLevels = new JTextField(5);
		textRegLevels.setText(String.valueOf(levels));
		textRegLevels.setFont(serif12);
	
		// put all together
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridwidth = 1;
        gbc.gridheight = 1;
        gbc.anchor = gbc.WEST;
        gbc.insets = new Insets(0, 0, 0, 0);

		// classification panel
        gbc.gridx = 0; 
		gbc.gridy = 0;
        gbc.weightx = 1; 
		gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        classifPanel.add(labelNClasses, gbc);
        gbc.gridx = 1; 
		gbc.gridy = 0;
        gbc.weightx = 0; 
		gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        classifPanel.add(textNClasses, gbc);
        gbc.gridx = 0; 
		gbc.gridy = 1;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        classifPanel.add(labelType, gbc);
        gbc.gridx = 1;
        gbc.gridy = 1;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        classifPanel.add(comboType, gbc);
		
		// classif params
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        classifParamPanel.add(labelSmoothing, gbc);
        gbc.gridx = 1;
        gbc.gridy = 0;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        classifParamPanel.add(textSmoothing, gbc);
        gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        classifParamPanel.add(labelTemporal, gbc);
        gbc.gridx = 1;
        gbc.gridy = 1;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        classifParamPanel.add(textTemporal, gbc);
        gbc.gridx = 0;
        gbc.gridy = 2;
        gbc.weightx = 1;
        gbc.anchor = gbc.WEST;
        gbc.fill = gbc.HORIZONTAL;
        classifParamPanel.add(labelMaxDiff, gbc);
        gbc.gridx = 1;
        gbc.gridy = 2;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        classifParamPanel.add(textMaxDiff, gbc);
        gbc.gridx = 0;
        gbc.gridy = 3;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        gbc.anchor = gbc.EAST;
        classifParamPanel.add(labelMaxIter, gbc);
        gbc.gridx = 1;
        gbc.gridy = 3;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        classifParamPanel.add(textMaxIter, gbc);
        gbc.gridx = 0;
        gbc.gridy = 4;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        classifParamPanel.add(labelInit, gbc);
        gbc.gridx = 1;
        gbc.gridy = 4;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        classifParamPanel.add(comboInit, gbc);
        gbc.gridx = 0;
        gbc.gridy = 5;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        classifParamPanel.add(labelScaling, gbc);
        gbc.gridx = 1;
        gbc.gridy = 5;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        classifParamPanel.add(comboScaling, gbc);
		gbc.gridx = 0;
        gbc.gridy = 6;
        gbc.gridwidth = 2;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        classifParamPanel.add(checkAdaptive, gbc);
        gbc.gridx = 0;
        gbc.gridy = 9;
        gbc.gridwidth = 1;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        classifParamPanel.add(labelFuzzy, gbc);
        gbc.gridx = 1;
        gbc.gridy = 9;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        classifParamPanel.add(textFuzzy, gbc);
		
		// options
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.gridwidth = 1;
		gbc.weightx = 1;
		gbc.anchor = gbc.WEST;
        gbc.fill = gbc.HORIZONTAL;
        optionPanel.add(checkOutliers, gbc);
        gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.gridwidth = 1;
        gbc.fill = gbc.HORIZONTAL;
        optionPanel.add(checkInhomog, gbc);
        gbc.gridx = 0;
        gbc.gridy = 2;
        gbc.gridwidth = 1;
        gbc.fill = gbc.HORIZONTAL;
        optionPanel.add(checkEdges, gbc);
		gbc.gridx = 0;
        gbc.gridy = 3;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        optionPanel.add(labelDistance, gbc);
        gbc.gridx = 1;
        gbc.gridy = 3;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        optionPanel.add(comboDistance, gbc);
		
		// option param panel
        gbc.gridx = 0; 
		gbc.gridy = 0;
        gbc.gridwidth = 1;
        gbc.weightx = 1; 
		gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        optionParamPanel.add(labelOutliers, gbc);
        gbc.gridx = 1;
        gbc.gridy = 0;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        optionParamPanel.add(textOutliers, gbc);
		gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.gridwidth = 1;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        optionParamPanel.add(labelPolyDeg, gbc);
        gbc.gridx = 1;
        gbc.gridy = 1;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        optionParamPanel.add(comboPolyDeg, gbc);
		gbc.gridx = 0;
        gbc.gridy = 2;
        gbc.gridwidth = 1;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        optionParamPanel.add(labelEdgeSmoothing, gbc);
        gbc.gridx = 1;
        gbc.gridy = 2;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        optionParamPanel.add(textEdgeSmoothing, gbc);		
        gbc.gridx = 0;
        gbc.gridy = 3;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        optionParamPanel.add(labelEdgeContrast, gbc);
        gbc.gridx = 1;
        gbc.gridy = 3;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        optionParamPanel.add(textEdgeContrast, gbc);		
        gbc.gridx = 0;
        gbc.gridy = 4;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        optionParamPanel.add(labelEdgePrior, gbc);
        gbc.gridx = 1;
        gbc.gridy = 4;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        optionParamPanel.add(textEdgePrior, gbc);		
		
		// general panel
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.gridwidth = 2;
        gbc.anchor = gbc.WEST;
        generalPanel.add(checkCrop, gbc);
		gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.gridwidth = 1;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        generalPanel.add(labelResult, gbc);
        gbc.gridx = 1;
        gbc.gridy = 1;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        generalPanel.add(comboResult, gbc);

		// general params
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.gridwidth = 2;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.WEST;
        generalParamPanel.add(checkRelative, gbc);
        gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.gridwidth = 1;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        generalParamPanel.add(labelSignal, gbc);
        gbc.gridx = 1;
        gbc.gridy = 1;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        generalParamPanel.add(textSignal, gbc);
        
		// registration panel
		gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.weightx = 0;
        gbc.gridwidth = 2;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.WEST;
        registrationPanel.add(checkRegister, gbc);
        gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.gridwidth = 1;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        registrationPanel.add(labelRegLevels, gbc);
        gbc.gridx = 1;
        gbc.gridy = 1;
        gbc.gridwidth = 3;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        registrationPanel.add(textRegLevels, gbc);
		gbc.gridx = 0;
        gbc.gridy = 2;
        gbc.gridwidth = 1;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        registrationPanel.add(labelRegIter, gbc);
        gbc.gridx = 1;
        gbc.gridy = 2;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        registrationPanel.add(textFirstIter, gbc);
        gbc.gridx = 2;
        gbc.gridy = 2;
        registrationPanel.add(textTopIter, gbc);
        gbc.gridx = 3;
        gbc.gridy = 2;
        registrationPanel.add(textLastIter, gbc);
				
		// image list panel
		gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.gridwidth = 2;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        imageListPanel.add(scrollImage, gbc);

		// assemble everything
		mainPanel = new JPanel(new GridBagLayout());
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.gridx = 0;
        
		gbc.gridy = 0;
        mainPanel.add(classifPanel, gbc);
		gbc.gridy = 1;
        mainPanel.add(optionPanel, gbc);
		gbc.gridy = 2;
        mainPanel.add(generalPanel, gbc);
		
		parameterPanel = new JPanel(new GridBagLayout());
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.gridx = 0;
		
		gbc.gridy = 0;
        parameterPanel.add(classifParamPanel, gbc);
		gbc.gridy = 1;
        parameterPanel.add(optionParamPanel, gbc);

		imagePanel = new JPanel(new GridBagLayout());
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.gridx = 0;
		
		gbc.gridy = 0;
        imagePanel.add(generalParamPanel, gbc);
		gbc.gridy = 1;
        imagePanel.add(imageListPanel, gbc);
		gbc.gridy = 2;
        imagePanel.add(registrationPanel, gbc);
		
        tabPane = new JTabbedPane();
        tabPane.setForeground(Color.black);
		
        tabPane.add("main",mainPanel);
        tabPane.add("images",imagePanel);
        tabPane.add("parameters",parameterPanel);
        
        getContentPane().add(tabPane);
        getContentPane().add(buildButtons(), BorderLayout.SOUTH);

        pack();
        //setVisible(true); 
		setResizable(false);
    	System.gc();
		
	} // end init()
	
	void displayImageList() {
		int j;
        
        Enumeration names = userInterface.getRegisteredImageNames();

        // Add images from user interface that have the same exact dimensionality
        // ! note: if the images have different dimensions, problems occur in Algorithms
        
		// 1. count the images
		nImage = 0;
		while (names.hasMoreElements()) {
            String name = (String) names.nextElement();
            ModelImage img = userInterface.getRegisteredImageByName(name);
			if (userInterface.getFrameContainingImage(img) != null) {
				if ( (image.getNDims() == img.getNDims()) && (!img.isColorImage()) ) {
					nImage++;
				}
            }
        }
		// 2. create the checkboxes
		checkImage = new JCheckBox[nImage];
		labelOrder = new JLabel[nImage];
		order = new int[nImage];
		for (int n=0;n<nImage;n++) order[n] = -1;
		int n = 0;
		names = userInterface.getRegisteredImageNames();
		while (names.hasMoreElements()) {
            String name = (String) names.nextElement();
            ModelImage img = userInterface.getRegisteredImageByName(name);
			if (userInterface.getFrameContainingImage(img) != null) {
				if ( (image.getNDims() == img.getNDims()) && (!img.isColorImage()) ) {
					checkImage[n] = new JCheckBox(name);
					checkImage[n].setFont(serif12);
					checkImage[n].setSelected(false);
					checkImage[n].addActionListener(this);
					checkImage[n].setActionCommand("Image "+n);
					
					labelOrder[n] = new JLabel(".");
					labelOrder[n].setFont(serif12);
					labelOrder[n].setBackground(Color.white);
					n++;
				}
            }
        }
		// display
		viewPanel = new JPanel(new GridBagLayout());
		GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridwidth = 1;
        
		for (n=0;n<nImage;n++) {
			gbc.gridy = n;
			gbc.gridx = 0;
			gbc.fill = gbc.NONE;
			gbc.weightx = 0;
			viewPanel.add(labelOrder[n], gbc);
			gbc.gridx = 1;
			gbc.fill = gbc.HORIZONTAL;
			gbc.weightx = 1;
			viewPanel.add(checkImage[n], gbc);
		}
		scrollImage.setViewportView(viewPanel);
		scrollImage.revalidate();
		pack();
    }

	
    /**
    *  Accessor that returns the image.
    *  @return          The result image.
    */
    public ModelImage[] getResultImage() {
        return resultImage;
    }

    /**
    *	Accessor that sets the parameters (empty)
    */
    public void setParameters() {
    }
	
    //************************************************************************
    //************************** Event Processing ****************************
    //************************************************************************

	/**
	*  Closes dialog box when the OK button is pressed and calls the algorithm.
	*  @param event       Event that triggers function.
	*/
	public void actionPerformed(ActionEvent event) {
		String command = event.getActionCommand();
	
		if (command.equals("OK")) {
			if (setVariables()) { 
				callAlgorithm();
			}    
		} else if (command.equals("Cancel")) {
			dispose();
		} else if (command.equals("Help")) {
            //MedicUtilPublic.showHelp("Fantasm");
        } else {
			for (int n=0;n<nImage;n++) {
				if (command.equals("Image "+n)) {
					if (checkImage[n].isSelected()) {
						order[n] = nOrder;
						labelOrder[n].setText(""+(nOrder+1));
						nOrder++;
					} else {
						for (int m=0;m<nImage;m++) {
							if (order[m]>order[n]) {
								order[m]--;
								labelOrder[m].setText(""+(order[m]+1));
							}
						}
						order[n] = -1;
						labelOrder[n].setText(".");
						nOrder--;
					}
				}
			}
		}
		return;		
    }

    //************************************************************************
    //************************** Algorithm Events ****************************
    //************************************************************************
    
    /** 
    *	This method is required if the AlgorithmPerformed interface is implemented. 
    *   It is called by the algorithm when it has completed or failed to to complete, 
    *   so that the dialog can be display the result image and/or clean up.
    *   @param algorithm   Algorithm that caused the event.
    */
    public void algorithmPerformed(AlgorithmBase algorithm) {
        
       if (Preferences.isPreference(Preferences.PREF_SAVE_DEFAULTS) && this.getOwner() != null && !isScriptRunning()) {
          saveDefaults();
       }
	   
	   int i;
	   int Nimg;
        
		if (algoType.equals("longitudinal")) Nimg = imageNumber;
		else Nimg = 1;
        ViewJFrameImage imageFrame[] = new ViewJFrameImage[resultNumber*imageNumber];
        if ((algorithm instanceof AlgorithmFantasm) || (algorithm instanceof AlgorithmFantasmLongitudinal) || (algorithm instanceof AlgorithmFantasmMultichannel)) {
            image.clearMask();
			if(algorithm.isCompleted() == true && resultImage != null) {
                //The algorithm has completed and produced a new image to be displayed.
                for (i = 0; i < resultNumber*Nimg; i++) {
                    updateFileInfo(image, resultImage[i]);
                    resultImage[i].clearMask();

                    try {
                        imageFrame[i] = new ViewJFrameImage(resultImage[i], null,
                                           new Dimension(610, 200 + i * 20) );
                    } catch (OutOfMemoryError error) {
                        System.gc();
                        JOptionPane.showMessageDialog(null, 
                                                "Out of memory: unable to open new frame",
                                                "Error", JOptionPane.ERROR_MESSAGE);
                    }
                 }
            } else if (resultImage != null) {
                //algorithm failed but result image still has garbage
                for (i = 0; i < resultNumber*Nimg; i++) {
                    if (resultImage[i] != null) {
                        resultImage[i].disposeLocal(); // Clean up memory of result image
                        resultImage[i] = null;
                    }
                }
                resultImage = null;
                System.gc();
			}
			if (algorithm.isCompleted()) {
				insertScriptLine();
			}
       }
       algorithm.finalize();
       algorithm = null;
       dispose();
    }  // end AlgorithmPerformed()
    
  
    /**
    *	Use the GUI results to set up the variables needed to run the algorithm.
    *	@return		<code>true</code> if parameters set successfully, <code>false</code> otherwise.
    */
    /**
    *	Use the GUI results to set up the variables needed to run the algorithm.
    *	@return		<code>true</code> if parameters set successfully, <code>false</code> otherwise.
    */
    private boolean setVariables() {
    	String tmpStr;
             	        
       	tmpStr = textNClasses.getText();
        if ( testParameter(tmpStr, 1, 20) ){
            nClasses = Integer.valueOf(tmpStr).intValue();
        } else {
            textNClasses.requestFocus();
            textNClasses.selectAll();
            return false;
        }
       	tmpStr = textMaxIter.getText();
        if ( testParameter(tmpStr, 0, 1000) ){
            nIterations = Integer.valueOf(tmpStr).intValue();
        } else {
            textMaxIter.requestFocus();
            textMaxIter.selectAll();
            return false;
        }
       	tmpStr = textMaxDiff.getText();
        if ( testParameter(tmpStr, 0.0, 10.0) ){
            maxDiff = Float.valueOf(tmpStr).floatValue();
        } else {
            textMaxDiff.requestFocus();
            textMaxDiff.selectAll();
            return false;
        }
       	tmpStr = textSmoothing.getText();
        if ( testParameter(tmpStr, 0.0, 10000.0) ){
            smoothing = Float.valueOf(tmpStr).floatValue();
        } else {
            textSmoothing.requestFocus();
            textSmoothing.selectAll();
            return false;
        }
       	tmpStr = textTemporal.getText();
        if ( testParameter(tmpStr, 0.0, 10000.0) ){
            temporal = Float.valueOf(tmpStr).floatValue();
        } else {
            textTemporal.requestFocus();
            textTemporal.selectAll();
            return false;
        }
		tmpStr = textFuzzy.getText();
        if ( testParameter(tmpStr, 0.5, 10.0) ){
            fuzziness = Float.valueOf(tmpStr).floatValue();
        } else {
            textFuzzy.requestFocus();
            textFuzzy.selectAll();
            return false;
        }
		adaptiveSmoothing = checkAdaptive.isSelected();
		cropBackground = checkCrop.isSelected();
		
		tmpStr = textSignal.getText();
        if (testParameter(tmpStr, -100000.0, 100000.0)) {
          backThreshold = Float.valueOf(tmpStr).floatValue();
        } else {
          textSignal.requestFocus();
          textSignal.selectAll();
          return false;
        }
		addOutliers = checkOutliers.isSelected();
		
		tmpStr = textOutliers.getText();
        if (testParameter(tmpStr, 0.0, 100000.0)) {
          outlierRatio = Float.valueOf(tmpStr).floatValue();
        } else {
          textOutliers.requestFocus();
          textOutliers.selectAll();
          return false;
        }
		useEdges = checkEdges.isSelected();
		
		tmpStr = textEdgeSmoothing.getText();
        if (testParameter(tmpStr, 0.0, 100000.0)) {
          edgeSmoothness = Float.valueOf(tmpStr).floatValue();
        } else {
          textEdgeSmoothing.requestFocus();
          textEdgeSmoothing.selectAll();
          return false;
        }
        tmpStr = textEdgeContrast.getText();
        if (testParameter(tmpStr, 0.0, 100000.0)) {
          edgeContrast = Float.valueOf(tmpStr).floatValue();
        } else {
          textEdgeContrast.requestFocus();
          textEdgeContrast.selectAll();
          return false;
        }
        tmpStr = textEdgePrior.getText();
        if (testParameter(tmpStr, 0.0, 100000.0)) {
          edgePrior = Float.valueOf(tmpStr).floatValue();
        } else {
          textEdgePrior.requestFocus();
          textEdgePrior.selectAll();
          return false;
        }
		
		distanceMode = (String)comboDistance.getSelectedItem();
		
		useRegistration = checkRegister.isSelected();
		
		tmpStr = textRegLevels.getText();
        if (testParameter(tmpStr, 0, 10)) {
          levels = Integer.valueOf(tmpStr).intValue();
        } else {
          textRegLevels.requestFocus();
          textRegLevels.selectAll();
          return false;
        }
        tmpStr = textFirstIter.getText();
        if (testParameter(tmpStr, 0, 50)) {
          firstIter = Integer.valueOf(tmpStr).intValue();
        } else {
          textFirstIter.requestFocus();
          textFirstIter.selectAll();
          return false;
        }
		tmpStr = textTopIter.getText();
        if (testParameter(tmpStr, 0, 50)) {
          topIter = Integer.valueOf(tmpStr).intValue();
        } else {
          textTopIter.requestFocus();
          textTopIter.selectAll();
          return false;
        }
        tmpStr = textLastIter.getText();
        if (testParameter(tmpStr, 0, 50)) {
          lastIter = Integer.valueOf(tmpStr).intValue();
        } else {
          textLastIter.requestFocus();
          textLastIter.selectAll();
          return false;
        }
        if (checkRelative.isSelected()) {
			useRelative = true;
		} else {
			useRelative = false;
		}
        outputType = (String)comboResult.getSelectedItem();
		initMode = (String)comboInit.getSelectedItem();
		scalingMode = (String)comboScaling.getSelectedItem();
		correctField = checkInhomog.isSelected();
		
		polynomialDegree = Integer.valueOf((String)comboPolyDeg.getSelectedItem()).intValue();
    
		algoType = (String)comboType.getSelectedItem();
		
		imageNumber = 0;
		for (int i=0;i<nImage;i++) 
			if (checkImage[i].isSelected() ) imageNumber++;
		
		if (algoType.equals("single_image")) {
			if (imageNumber>1) {
				tabPane.setSelectedComponent(mainPanel);
				comboType.requestFocus();
				return false;
			}
			imageNumber = 1;
			images = new ModelImage[imageNumber];
			images[0] = image;
		} else {
			if (imageNumber==0) {
				tabPane.setSelectedComponent(imagePanel);
				imageListPanel.requestFocus();
				return false;
			}
			images = new ModelImage[imageNumber];
			// put images in proper order
			for (int i=0;i<nImage;i++) if (checkImage[i].isSelected() ) {
				images[order[i]] = userInterface.getRegisteredImageByName(checkImage[i].getText());
			}
		}
    	return true;  	
    }   // end setVariables()
    
    /**
    *	Once all the necessary variables are set, call the Gaussian Blur
    *	algorithm based on what type of image this is and whether or not there
    *	is a separate destination image.
    */
    protected void callAlgorithm() {
        int i, Nimg;
        String name = makeImageName(image.getImageName(), "");
		AlgorithmBase algorithm = null;
		String suffix = "";
        
        // Calculate the number of result images.
        resultNumber = 0;
        if (outputType.equals("hard_segmentation")) resultNumber=1; 
		else if (outputType.equals("fuzzy_segmentation")) {
			resultNumber = nClasses;
			if (addOutliers) resultNumber++;
        } else {
			resultNumber = nClasses+1;
			if (addOutliers) resultNumber++;
			if (outputType.equals("all_result_images")) {
				if (correctField) {
					if (algoType.equals("multichannel")) resultNumber+=imageNumber;
					else resultNumber++;
				}
				if (useEdges) resultNumber += 3;
			}
		}
		if (algoType.equals("longitudinal")) Nimg = imageNumber;
		else Nimg = 1;
		
		// add suffix for algorithm type
		if (algoType.equals("longitudinal")) suffix = "_long";
		else if (algoType.equals("multichannel")) suffix = "_multi";
		
        if (image.getNDims() == 2) { // source image is 2D
            destExtents = new int[2];
            destExtents[0] = image.getExtents()[0]; // X dim
            destExtents[1] = image.getExtents()[1]; // Y dim
        } else { // source image is 3D or more (?)
            destExtents = new int[3];
            destExtents[0] = image.getExtents()[0];
            destExtents[1] = image.getExtents()[1];
            destExtents[2] = image.getExtents()[2];
        }

        try {
            resultImage = new ModelImage[resultNumber*Nimg];
            presentNumber = 0;
            
			for (int n=0;n<Nimg;n++) {
				if (!outputType.equals("hard_segmentation")) {
					for (i = 0; i < nClasses; i++) {
						resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents,
															makeImageName(images[n].getImageName(), suffix+"_class" + (i + 1)));
					}
					if (addOutliers) {
						resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents, 
															makeImageName(images[n].getImageName(), suffix+"_out"));
					}
				}
				if (!outputType.equals("fuzzy_segmentation")) {
					resultImage[presentNumber++] = new ModelImage(ModelStorageBase.UBYTE, destExtents,
															makeImageName(images[n].getImageName(), suffix+"_seg"));
				}
				if (outputType.equals("all_result_images")) {
					if (correctField) {
						if (algoType.equals("multichannel")) {
							for (int t=0;t<imageNumber;t++) {
								resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents,
																	makeImageName(images[n].getImageName(), suffix+"_field"));
							}
						} else {
							resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents,
																makeImageName(images[n].getImageName(), suffix+"_field"));
						}
					}
					if (useEdges) {
						resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents,
															makeImageName(images[n].getImageName(), suffix+"_edgeX"));
						resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents,
															makeImageName(images[n].getImageName(), suffix+"_edgeY"));
						resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents,
															makeImageName(images[n].getImageName(), suffix+"_edgeZ"));
					}
				}
			}
            // Create algorithm
			if (algoType.equals("single_image")) {
				String correct;
				if (correctField) correct = "image";
				else correct = "none";
				
				algo = new AlgorithmFantasm(resultImage, images[0], resultNumber,
											initMode, outputType,
											nClasses, nIterations, maxDiff, smoothing,
											fuzziness,
											addOutliers, outlierRatio, 
											correct, polynomialDegree,
											useEdges, edgeSmoothness, edgeContrast, edgePrior,
											cropBackground, backThreshold, useRelative, 
											adaptiveSmoothing, distanceMode);
				algorithm = algo;
			} else if (algoType.equals("longitudinal")) {
				algo2 = new AlgorithmFantasmLongitudinal(resultImage, images, imageNumber, resultNumber,
												initMode, outputType,
												nClasses, nIterations, maxDiff, smoothing, temporal,
												fuzziness,
												addOutliers, outlierRatio, 
												correctField, polynomialDegree,
												useEdges, edgeSmoothness, edgeContrast, edgePrior,
												useRegistration, levels, firstIter, topIter, lastIter,
												cropBackground, backThreshold, scalingMode, useRelative);
				algorithm = algo2;
			} else if (algoType.equals("multichannel")) {
				algo3 = new AlgorithmFantasmMultichannel(resultImage, images, imageNumber, resultNumber,
												initMode, outputType,
												nClasses, nIterations, maxDiff, smoothing, 
												fuzziness,
												addOutliers, outlierRatio, 
												correctField, polynomialDegree,
												useEdges, edgeSmoothness, edgeContrast, edgePrior,
												useRegistration, levels, firstIter, topIter, lastIter,
												cropBackground, backThreshold, scalingMode);
				algorithm = algo3;
			}
			
            
            // This is very important. Adding this object as a listener allows the algorithm to
            // notify this object when it has completed or failed. See algorithm performed event.
            // This is made possible by implementing AlgorithmedPerformed interface
            algorithm.addListener(this);
                 
            setVisible(false);  // Hide dialog

			createProgressBar(image.getImageName(), algorithm);
                
			// bug fix: check if the linux version crashes in this case
			setSeparateThread(false);
			
			// for logging
			if (verbose) {
				MedicUtilPublic.displayMessage("Fantasm(s)\n");
				MedicUtilPublic.displayMessage("images:\n");
				for (i = 0; i < Nimg; i++) MedicUtilPublic.displayMessage(images[i].getImageName()+" ");
				MedicUtilPublic.displayMessage("\n parameters:\n"+getParameterString(", ")+"\n");
			}
			
            if (runInSeparateThread) {
                // Start the thread as a low priority because we wish to still have user interface work fast.
                if (algorithm.startMethod(Thread.MIN_PRIORITY) == false) {
                    MipavUtil.displayError("A thread is already running on this object");
                }
            } else {
                //algorithm.setActiveImage(isActiveImage);
                algorithm.run();
            }
        } catch (OutOfMemoryError x) {

            if (resultImage != null) {
                for (i = 0; i < resultNumber*Nimg; i++) {
                    if (resultImage[i] != null) {
                        resultImage[i].disposeLocal(); // Clean up memory of result image
                        resultImage[i] = null;
                    }
                }
                resultImage = null;
            }
            System.gc();
            MipavUtil.displayError( "Fantasms: unable to allocate enough memory");
            return;
        }
    } // end callAlgorithm()
    
    /**
     * Perform any actions required after the running of the algorithm is complete.
     */
    protected void doPostAlgorithmActions() {
		for (int n=0;n<resultNumber;n++)
			AlgorithmParameters.storeImageInRunner(getResultImage()[n]);
    }
    /**
     * Set up the dialog GUI based on the parameters before running the algorithm as part of a script.
     */
    protected void setGUIFromParams() {
		imageNumber = scriptParameters.getParams().getInt("input_images");
		images = new ModelImage[imageNumber];
		for (int n=0;n<imageNumber;n++) {
			images[n] = scriptParameters.retrieveInputImage(n+1);
		}
		image = images[0];
        userInterface = ViewUserInterface.getReference();
        parentFrame = image.getParentFrame();
		
		nClasses = scriptParameters.getParams().getInt("classes");
		nIterations = scriptParameters.getParams().getInt("iterations");
		maxDiff = scriptParameters.getParams().getFloat("max_difference");
        smoothing = scriptParameters.getParams().getFloat("smoothing");
		temporal = scriptParameters.getParams().getFloat("temporal_smoothing");
		fuzziness = scriptParameters.getParams().getFloat("fuzziness");
		
		addOutliers = scriptParameters.getParams().getBoolean("add_outliers");
		outlierRatio = scriptParameters.getParams().getFloat("outlier_ratio");
		
		algoType = scriptParameters.getParams().getString("algorithm_type");
		initMode = scriptParameters.getParams().getString("initialization");
		scalingMode = scriptParameters.getParams().getString("scaling_mode");
		
		useRegistration = scriptParameters.getParams().getBoolean("use_registration");
		topIter = scriptParameters.getParams().getInt("top_iteration");
		firstIter = scriptParameters.getParams().getInt("first_iteration");
		lastIter = scriptParameters.getParams().getInt("last_iteration");
		levels = scriptParameters.getParams().getInt("levels");
		
		correctField = scriptParameters.getParams().getBoolean("correct_inhomogeneity");
		polynomialDegree = scriptParameters.getParams().getInt("field_degree");
		
		useEdges = scriptParameters.getParams().getBoolean("use_edges");
		edgeSmoothness = scriptParameters.getParams().getFloat("edge_smoothness");
		edgeContrast = scriptParameters.getParams().getFloat("edge_contrast");
		edgePrior = scriptParameters.getParams().getFloat("edge_prior");
		
		cropBackground = scriptParameters.getParams().getBoolean("crop_background");
		backThreshold = scriptParameters.getParams().getFloat("background_threshold");
		useRelative = scriptParameters.getParams().getBoolean("scaled_coefficients");
		adaptiveSmoothing = scriptParameters.getParams().getBoolean("adaptive_smoothing");
		outputType = scriptParameters.getParams().getString("output_type");
		
		distanceMode = scriptParameters.getParams().getString("distance_mode");
    }

	/**
     * Store the parameters from the dialog to record the execution of this algorithm.
     * 
     * @throws  ParserException  If there is a problem creating one of the new parameters.
     */
    protected void storeParamsFromGUI() throws ParserException {
        scriptParameters.getParams().put(ParameterFactory.newParameter("input_images", imageNumber));
		for (int n=0;n<imageNumber;n++) {
			scriptParameters.storeInputImage(images[n]);
		}
        scriptParameters.getParams().put(ParameterFactory.newParameter("output_images", resultNumber));
		for (int n=0;n<resultNumber;n++) {
			scriptParameters.storeImageInRecorder(resultImage[n]);
		}
		
		scriptParameters.getParams().put(ParameterFactory.newParameter("classes", nClasses));
		scriptParameters.getParams().put(ParameterFactory.newParameter("iterations", nIterations));
		scriptParameters.getParams().put(ParameterFactory.newParameter("max_difference", maxDiff));
		scriptParameters.getParams().put(ParameterFactory.newParameter("smoothing", smoothing));
		scriptParameters.getParams().put(ParameterFactory.newParameter("temporal_smoothing", temporal));
		scriptParameters.getParams().put(ParameterFactory.newParameter("fuzziness", fuzziness));
		
		scriptParameters.getParams().put(ParameterFactory.newParameter("add_outliers", addOutliers));
		scriptParameters.getParams().put(ParameterFactory.newParameter("outlier_ratio", outlierRatio));
		
		scriptParameters.getParams().put(ParameterFactory.newParameter("algorithm_type", algoType));
		scriptParameters.getParams().put(ParameterFactory.newParameter("initialization", initMode));
		scriptParameters.getParams().put(ParameterFactory.newParameter("scaling_mode", scalingMode));
		
		scriptParameters.getParams().put(ParameterFactory.newParameter("use_registration", useRegistration));
		scriptParameters.getParams().put(ParameterFactory.newParameter("top_iteration", topIter));
		scriptParameters.getParams().put(ParameterFactory.newParameter("first_iteration", firstIter));
		scriptParameters.getParams().put(ParameterFactory.newParameter("last_iteration", lastIter));
		scriptParameters.getParams().put(ParameterFactory.newParameter("levels", levels));
		
		scriptParameters.getParams().put(ParameterFactory.newParameter("correct_inhomogeneity", correctField));
		scriptParameters.getParams().put(ParameterFactory.newParameter("field_degree", polynomialDegree));
		
		scriptParameters.getParams().put(ParameterFactory.newParameter("use_edges", useEdges));
		scriptParameters.getParams().put(ParameterFactory.newParameter("edge_smoothness", edgeSmoothness));
		scriptParameters.getParams().put(ParameterFactory.newParameter("edge_contrast", edgeContrast));
		scriptParameters.getParams().put(ParameterFactory.newParameter("edge_prior", edgePrior));
		
		scriptParameters.getParams().put(ParameterFactory.newParameter("crop_background", cropBackground));
		scriptParameters.getParams().put(ParameterFactory.newParameter("background_threshold", backThreshold));
		scriptParameters.getParams().put(ParameterFactory.newParameter("scaled_coefficients", useRelative));
		scriptParameters.getParams().put(ParameterFactory.newParameter("adaptive_smoothing", adaptiveSmoothing));
		scriptParameters.getParams().put(ParameterFactory.newParameter("output_type", outputType));
		
		scriptParameters.getParams().put(ParameterFactory.newParameter("distance_mode", distanceMode));
	}
}
