package edu.jhmi.rad.medic.dialogs;

import edu.jhmi.rad.medic.algorithms.AlgorithmTopologyCorrection;

import gov.nih.mipav.view.*;
import gov.nih.mipav.view.dialogs.*;
import gov.nih.mipav.view.components.*;

import gov.nih.mipav.model.structures.*;
import gov.nih.mipav.model.file.*;
import gov.nih.mipav.model.algorithms.*;
import gov.nih.mipav.model.scripting.*;
import gov.nih.mipav.model.scripting.parameters.*;

import java.awt.event.*;
import java.awt.*;
import java.util.*;
import java.io.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

/** 
*   
*   Dialog class for topology correction plug-in.
*
*
*	@version    April 2005
*	@author     Pilou Bazin
*   @see        AlgorithmTopologyCorrection
*
*
*/  
public class JDialogTopologyCorrection extends JDialogScriptableBase implements AlgorithmInterface, DialogDefaultsInterface {
    
    private     AlgorithmTopologyCorrection algo = null;
    private     ModelImage              image;                // source image
    private     ModelImage              resultImage = null;	// result image
    private     int                     destExtents[];
    private 	ViewUserInterface       userInterface;
    private     String                  titles[];            

	// parameters
	private		float		lowestLevel		=	0.0f;
	private		float		highestLevel	=	1.0f;
	private		boolean		useMinMax	=	true;
	private		boolean		thresholdStop	=	true;
	private		int			dimension	=	2;
	private		int			objConnect	=	4;
	private		int			bckConnect	=	8;
	private     String      type = "scalar_image";
    private     String[]    types = {"scalar_image", "binary_object"};
    private		String[]	connects2D = {"2D:8/4", "2D:4/8"};
	private		String[]	connects3D = {"3D:26/6", "3D:18/6", "3D:6/26", "3D:6/18", "3D:all"};
	private 	String		connect2D = "2D:8/4";
	private		String		connect3D = "3D:26/6";
	private		float		minDistance		=		0.0001f;
	private		String[]	inputSkels = {"intensity","paint_mask"};
	private		String		inputSkel = "intensity";
    private     BitSet      inputPaint = null;
	private		String[]	propagationTypes = {"object->background","background->object"};
	private		String		propagationType = "object->background";
    	    
    // dialog elements
	private 	JPanel  	mainPanel;
    
	private 	JPanel  	algoPanel;
	private 	JPanel  	paramPanel;
	private 	JTextField  textLowestLevel;
	private 	JLabel  	labelLowestLevel;
	private 	JTextField  textHighestLevel;
	private 	JLabel  	labelHighestLevel;
	private		JComboBox	comboConnect;
	private		JLabel		labelConnect;
	private		JCheckBox	checkMinMax;
	private		JCheckBox	checkStop;
	private		JLabel		labelType;
	private		JComboBox	comboType;
    private     JLabel      labelDist;
	private 	JTextField  textDist;
    private		JLabel		labelStart;
	private		JComboBox	comboStart;
	private		JLabel		labelPropag;
	private		JComboBox	comboPropag;
	    
  
    /**
    *  Creates dialog for plugin.
    *  @param theParentFrame          Parent frame.
    *  @param im              Source image.
    */
    public JDialogTopologyCorrection(Frame theParentFrame, ModelImage im) {
		super(theParentFrame, false);
		if (im.isColorImage()) {
            MipavUtil.displayError("Source Image must be scalar"); 
            dispose();
            return;
        }
        if (im.getNDims()>3) {
            MipavUtil.displayError("Source Image must be 2D or 3D"); 
            dispose();
            return;
        }
    	image = im;
		userInterface = ((ViewJFrameBase)(parentFrame)).getUserInterface();	    
		loadDefaults();
        init();
		setVisible(true);
	}
	
    /**
    *	Used primarily for the script to store variables and run the algorithm.  No
    *	actual dialog will appear but the set up info and result image will be stored here.
    *	@param UI   The user interface, needed to create the image frame.
    *	@param im	Source image.
    */
    public JDialogTopologyCorrection(ViewUserInterface UI, ModelImage im) {
        super();
    	userInterface = UI;
    	image = im;
        parentFrame = image.getParentFrame();
    }
    
    /**
     * Empty constructor needed for dynamic instantiation.
     */
    public JDialogTopologyCorrection() {};

    /**
    *	Sets up the GUI (panels, buttons, etc) and displays it on the screen.
    */
	private void init(){
        setForeground(Color.black);
        setTitle("Topology Correction");
				
		labelType = new JLabel("Image type:");
		labelType.setForeground(Color.black);
		labelType.setFont(serif12);

        comboType = new JComboBox(types);
		comboType.setFont(serif12);
        comboType.setBackground(Color.white);
		comboType.setSelectedItem(type);
		
        labelConnect = new JLabel("Connectivity");
		labelConnect.setForeground(Color.black);
		labelConnect.setFont(serif12);
	
		if (image.getNDims()==2) {
			comboConnect = new JComboBox(connects2D);
			comboConnect.setSelectedItem(connect2D);
		} else {
			comboConnect = new JComboBox(connects3D);
			comboConnect.setSelectedItem(connect3D);
		}
		comboConnect.setFont(serif12);
        comboConnect.setBackground(Color.white);

		labelStart = new JLabel("Starting point");
		labelStart.setForeground(Color.black);
		labelStart.setFont(serif12);
	
		comboStart = new JComboBox(inputSkels);
		comboStart.setFont(serif12);
        comboStart.setBackground(Color.white);
		comboStart.setSelectedItem(inputSkel);
		comboStart.setToolTipText("Let the algorithm use the highest intensity point or create a starting object with the paint tools.");       

        labelPropag = new JLabel("Propagation direction");
		labelPropag.setForeground(Color.black);
		labelPropag.setFont(serif12);
	
		comboPropag = new JComboBox(propagationTypes);
		comboPropag.setFont(serif12);
        comboPropag.setBackground(Color.white);
		comboPropag.setSelectedItem(propagationType);
		comboPropag.setToolTipText("Start the propagation from the highest intensity point or the image boundaries (only for the 'intensity' mode).");       

        labelDist = new JLabel("Regularization");
        labelDist.setForeground(Color.black);
        labelDist.setFont(serif12);
		
        textDist = new JTextField(5);
        textDist.setText(String.valueOf(minDistance));
        textDist.setFont(serif12);
		textDist.setToolTipText("Set the amount of geometric regularization (0 - 10e-3)");       
        
        labelLowestLevel = new JLabel("Lowest level");
        labelLowestLevel.setForeground(Color.black);
        labelLowestLevel.setFont(serif12);
		
        textLowestLevel = new JTextField(5);
        textLowestLevel.setText(String.valueOf(lowestLevel));
        textLowestLevel.setFont(serif12);
        textLowestLevel.setToolTipText("Set the lowest intensity value to process");       
        
        labelHighestLevel = new JLabel("Highest level");
        labelHighestLevel.setForeground(Color.black);
        labelHighestLevel.setFont(serif12);
		
        textHighestLevel = new JTextField(5);
        textHighestLevel.setText(String.valueOf(highestLevel));
        textHighestLevel.setFont(serif12);
        textHighestLevel.setToolTipText("Set the highest intensity value to process (higher values will be set as background)");       
        
        checkMinMax = new JCheckBox("Normalized values");
        checkMinMax.setFont(serif12);
        checkMinMax.setSelected(useMinMax);
		checkMinMax.setToolTipText("Normalize the image in [0,1]");       
        
        checkStop = new JCheckBox("Stop at intensity boundaries");
        checkStop.setFont(serif12);
        checkStop.setSelected(thresholdStop);
		checkStop.setToolTipText("Stop the correction when the intensity boundary (min or max, depending on the propagation direction) is reached");       
        
		algoPanel = new JPanel(new GridBagLayout());
		algoPanel.setBorder(buildTitledBorder("Algorithm"));

		paramPanel = new JPanel(new GridBagLayout());
		paramPanel.setBorder(buildTitledBorder("Parameters"));

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridwidth = 1;
        gbc.gridheight = 1;
        gbc.anchor = gbc.WEST;
        gbc.insets = new Insets(5, 5, 5, 5);

        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        algoPanel.add(labelType, gbc);
        gbc.gridx = 1;
        gbc.gridy = 0;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        algoPanel.add(comboType, gbc);
        gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        algoPanel.add(labelConnect, gbc);
        gbc.gridx = 1;
        gbc.gridy = 1;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        algoPanel.add(comboConnect, gbc);
		
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.gridwidth = 1;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        paramPanel.add(labelStart, gbc);
        gbc.gridx = 1;
        gbc.gridy = 0;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        paramPanel.add(comboStart, gbc);
        gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        paramPanel.add(labelDist, gbc);
        gbc.gridx = 1;
        gbc.gridy = 1;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        paramPanel.add(textDist, gbc);
        gbc.gridx = 0;
        gbc.gridy = 2;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        paramPanel.add(labelLowestLevel, gbc);
        gbc.gridx = 1;
        gbc.gridy = 2;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        paramPanel.add(textLowestLevel, gbc);
        gbc.gridx = 0;
        gbc.gridy = 3;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        paramPanel.add(labelHighestLevel, gbc);
        gbc.gridx = 1;
        gbc.gridy = 3;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        paramPanel.add(textHighestLevel, gbc);
		gbc.gridx = 0;
        gbc.gridy = 4;
        gbc.gridwidth = 2;
        gbc.fill = gbc.NONE;
        paramPanel.add(checkMinMax, gbc);
        gbc.gridx = 0;
        gbc.gridy = 5;
        gbc.gridwidth = 1;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        paramPanel.add(labelPropag, gbc);
        gbc.gridx = 1;
        gbc.gridy = 5;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        paramPanel.add(comboPropag, gbc);
        gbc.gridx = 0;
        gbc.gridy = 6;
        gbc.gridwidth = 2;
        gbc.fill = gbc.NONE;
        paramPanel.add(checkStop, gbc);
        
        mainPanel = new JPanel(new GridBagLayout());
        mainPanel.setForeground(Color.black);
        
        gbc.gridx = 0;
        gbc.gridwidth = gbc.REMAINDER;
        gbc.gridy = 0;
        gbc.fill = gbc.BOTH;
        gbc.weightx = 1;
        mainPanel.add(algoPanel, gbc);
		gbc.gridy = 1;
        gbc.fill = gbc.BOTH;
        gbc.weightx = 1;
        mainPanel.add(paramPanel, gbc);

        getContentPane().add(mainPanel);
        getContentPane().add(buildButtons(), BorderLayout.SOUTH);

        pack();
        setResizable(false);
    	System.gc();
		
	} // end init()
	
    /**
    *  Accessor that returns the image.
    *  @return          The result image.
    */
    public ModelImage getResultImage() {
        return resultImage;
    }

    /**
    *	Accessor that sets the parameters
    */
    public void setParameters(String type_, 
								int dimension_, int objConnect_, int bckConnect_,
                                float minDist_, String skel_,
								float lowLevel_, float highLevel_, 
								boolean useMinMax_, String propag_, boolean stop_) {
        dimension = dimension_;
        objConnect = objConnect_;
        bckConnect = bckConnect_;
        minDistance = minDist_;
		type = type_;
		inputSkel = skel_;
        lowestLevel = lowLevel_;
        highestLevel = highLevel_;
        useMinMax = useMinMax_;
		propagationType = propag_;
		thresholdStop = stop_;
    }
	
    /**
     * Construct a delimited string that contains the parameters to this algorithm.
     * @param delim  the parameter delimiter (defaults to " " if empty)
     * @return       the parameter string
     */
	public String getParameterString( String delim ) {
        if ( delim.equals( "" ) ) {
            delim = " ";
        }

        String str = new String();
        str += type + delim;
		str += connect2D + delim;
        str += connect3D + delim;
        str += inputSkel + delim;
        str += minDistance + delim;
		str += lowestLevel + delim;
        str += highestLevel + delim;
        str += useMinMax + delim;
		str += propagationType + delim;
		str += thresholdStop;
		
		return str;
     }
	
 	/**
     *  Loads the default settings from Preferences to set up the dialog
     */
	public void loadDefaults() {
        String defaultsString = Preferences.getDialogDefaults(getDialogName());

        if (defaultsString != null) {

            try {
                System.out.println(defaultsString);
                StringTokenizer st = new StringTokenizer(defaultsString, ",");
				type = st.nextToken();
				connect2D = st.nextToken();
				connect3D = st.nextToken();
				inputSkel = st.nextToken();
				minDistance = MipavUtil.getFloat(st);
				lowestLevel = MipavUtil.getFloat(st);
				highestLevel = MipavUtil.getFloat(st);
				useMinMax = MipavUtil.getBoolean(st);
				propagationType = st.nextToken();
				thresholdStop = MipavUtil.getBoolean(st);
            }
            catch (Exception ex) {
                // since there was a problem parsing the defaults string, start over with the original defaults
                System.out.println( "Resetting defaults for dialog: " + getDialogName() );
                Preferences.removeProperty( getDialogName() );
            }
        } else {
			System.out.println( "no saved dialogs for "+getDialogName() );
		}
    }
		
    /**
     * Saves the default settings into the Preferences file
     */
	
    public void saveDefaults() {
        String defaultsString = new String( getParameterString(",") );
        System.out.println(defaultsString);
        Preferences.saveDialogDefaults(getDialogName(),defaultsString);
    }
	 

    //************************************************************************
    //************************** Event Processing ****************************
    //************************************************************************

	/**
	*  Closes dialog box when the OK button is pressed and calls the algorithm.
	*  @param event       Event that triggers function.
	*/
	public void actionPerformed(ActionEvent event) {
		String command = event.getActionCommand();
	
		if (command.equals("OK")) {
			if (setVariables()) { 
				callAlgorithm();
			}    
		} else if (command.equals("Cancel")) {
			dispose();
		} else if (command.equals("Help")) {
            //MipavUtil.showHelp("10027");
        } 
    }

    //************************************************************************
    //************************** Algorithm Events ****************************
    //************************************************************************
    
    /** 
    *	This method is required if the AlgorithmPerformed interface is implemented. 
    *   It is called by the algorithm when it has completed or failed to to complete, 
    *   so that the dialog can be display the result image and/or clean up.
    *   @param algorithm   Algorithm that caused the event.
    */
    public void algorithmPerformed(AlgorithmBase algorithm) {
                
		if (Preferences.isPreference(Preferences.PREF_SAVE_DEFAULTS) && this.getOwner() != null && !isScriptRunning()) {
			saveDefaults();
		}
	   
		ViewJFrameImage imageFrame;
        ViewJFrameImage extraFrame;
        if ( algorithm instanceof AlgorithmTopologyCorrection) {
            if(algorithm.isCompleted() == true && resultImage != null) {
                //The algorithm has completed and produced a new image to be displayed.
            	updateFileInfo(image, resultImage);
                resultImage.clearMask();

                try {
                    imageFrame = new ViewJFrameImage(resultImage, null,
                                       new Dimension(610, 200) );
                } catch (OutOfMemoryError error) {
                    System.gc();
                    JOptionPane.showMessageDialog(null, 
                                            "Out of memory: unable to open new frame",
                                            "Error", JOptionPane.ERROR_MESSAGE);
                }
            } else if (resultImage != null) {
                //algorithm failed but result image still has garbage
                resultImage.disposeLocal(); // Clean up memory of result image
                resultImage = null;
                System.gc();
            }
	   		if (algorithm.isCompleted()) {
				insertScriptLine();
			}
       }
       algorithm.finalize();
       algorithm = null;
       dispose();
    }  // end AlgorithmPerformed()
    
    /**
    *	Use the GUI results to set up the variables needed to run the algorithm.
    *	@return		<code>true</code> if parameters set successfully, <code>false</code> otherwise.
    */
    private boolean setVariables() {
    	String tmpStr;
             	        
       	tmpStr = textDist.getText();
        if ( testParameter(tmpStr, -1e20, 1e20) ){
            minDistance = Float.valueOf(tmpStr).floatValue();
        } else {
            textDist.requestFocus();
            textDist.selectAll();
            return false;
        }
		
       	tmpStr = textLowestLevel.getText();
        if ( testParameter(tmpStr, -1e20, 1e20) ){
            lowestLevel = Float.valueOf(tmpStr).floatValue();
        } else {
            textLowestLevel.requestFocus();
            textLowestLevel.selectAll();
            return false;
        }
		
       	tmpStr = textHighestLevel.getText();
        if ( testParameter(tmpStr, -1e20, 1e20) ){
            highestLevel = Float.valueOf(tmpStr).floatValue();
        } else {
            textHighestLevel.requestFocus();
            textHighestLevel.selectAll();
            return false;
        }
		
		if (comboConnect.getSelectedItem().equals("3D:26/6")) {
            dimension = 3;
			objConnect = 26;
			bckConnect = 6;
			connect3D = "3D:26/6";
        } else if (comboConnect.getSelectedItem().equals("3D:18/6")) {
            dimension = 3;
			objConnect = 18;
			bckConnect = 6;
			connect3D = "3D:18/6";
		} else if (comboConnect.getSelectedItem().equals("3D:6/26")) {
            dimension = 3;
			objConnect = 6;
			bckConnect = 26;
			connect3D = "3D:6/26";
		} else if (comboConnect.getSelectedItem().equals("3D:6/18")) {
            dimension = 3;
			objConnect = 6;
			bckConnect = 18;
			connect3D = "3D:6/18";
		} else if (comboConnect.getSelectedItem().equals("3D:all")) {
            dimension = 3;
			objConnect = 6;
			bckConnect = 6;
			connect3D = "3D:all";
		} else if (comboConnect.getSelectedItem().equals("2D:8/4")) {
            dimension = 2;
			objConnect = 8;
			bckConnect = 4;
			connect2D = "2D:8/4";
		} else if (comboConnect.getSelectedItem().equals("2D:4/8")) {
            dimension = 2;
			objConnect = 4;
			bckConnect = 8;
			connect2D = "2D:4/8";
		} else {
			comboConnect.requestFocus();
			return false;
		}
		
		if (dimension != image.getNDims() ) {
			comboConnect.requestFocus();
			return false;
		}
        
		type = (String)comboType.getSelectedItem();
                
		inputSkel = (String)comboStart.getSelectedItem();
        if (inputSkel.equals("paint_mask")) {
            inputPaint = image.getParentFrame().getComponentImage().getPaintMask();
        }
		if ( (type.equals("binary_objects")) && (!inputSkel.equals("intensity")) ) {
			inputSkel = "intensity";
		}
		
		useMinMax = checkMinMax.isSelected();
		
        propagationType = (String)comboPropag.getSelectedItem();
        
		thresholdStop = checkStop.isSelected();

		return true;  	
    }   // end setVariables()
    
    /**
    *	Once all the necessary variables are set, call the Gaussian Blur
    *	algorithm based on what type of image this is and whether or not there
    *	is a separate destination image.
    */
    protected void callAlgorithm() {
        int i;
        String name = makeImageName(image.getImageName(), "_corrected");
        
        if (image.getNDims() == 2) { // source image is 2D
            destExtents = new int[2];
            destExtents[0] = image.getExtents()[0]; // X dim
            destExtents[1] = image.getExtents()[1]; // Y dim
        } else { // source image is 3D or more (?)
            destExtents = new int[3];
            destExtents[0] = image.getExtents()[0];
            destExtents[1] = image.getExtents()[1];
            destExtents[2] = image.getExtents()[2];
        }

        try {
            resultImage = new ModelImage(ModelStorageBase.FLOAT, destExtents,
                                          makeImageName(image.getImageName(), "_corrected"));
										   			
            // Create algorithm
            algo = new AlgorithmTopologyCorrection(resultImage, image, type,
												minDistance, dimension, 
                                                objConnect, bckConnect, 
												inputSkel, inputPaint, 
												lowestLevel, highestLevel, useMinMax,
												propagationType, thresholdStop);
            
            // This is very important. Adding this object as a listener allows the algorithm to
            // notify this object when it has completed or failed. See algorithm performed event.
            // This is made possible by implementing AlgorithmedPerformed interface
            algo.addListener(this);
                           
            setVisible(false);  // Hide dialog

            createProgressBar(image.getImageName(), algo);
                
			// for linux debugs: no threading
			//setSeparateThread(false);

			if (runInSeparateThread) {
                // Start the thread as a low priority because we wish to still have user interface work fast.
                if (algo.startMethod(Thread.MIN_PRIORITY) == false) {
                    MipavUtil.displayError("A thread is already running on this object");
                }
            } else {
                //algo.setActiveImage(isActiveImage);
                algo.run();
            }
        } catch (OutOfMemoryError x) {

            if (resultImage != null) {
                resultImage.disposeLocal(); // Clean up memory of result image
                resultImage = null;
            }
            System.gc();
            MipavUtil.displayError( "Dialog: unable to allocate enough memory");
            return;
        }
    } // end callAlgorithm()

    /**
     * Perform any actions required after the running of the algorithm is complete.
     */
    protected void doPostAlgorithmActions() {
        AlgorithmParameters.storeImageInRunner(getResultImage());
    }

    /**
     * Set up the dialog GUI based on the parameters before running the algorithm as part of a script.
     */
    protected void setGUIFromParams() {
        image = scriptParameters.retrieveInputImage();
        userInterface = ViewUserInterface.getReference();
        parentFrame = image.getParentFrame();

        type = scriptParameters.getParams().getString("algorithm_type");
		connect2D = scriptParameters.getParams().getString("2D_connectivity");
		connect3D = scriptParameters.getParams().getString("3D_connectivity");
		propagationType = scriptParameters.getParams().getString("propagation_type");
		inputSkel = scriptParameters.getParams().getString("initialization");
		lowestLevel = scriptParameters.getParams().getFloat("lowest_level");
		highestLevel = scriptParameters.getParams().getFloat("highest_level");
		minDistance = scriptParameters.getParams().getFloat("regularization");
		useMinMax = scriptParameters.getParams().getBoolean("normalize_image");
		thresholdStop = scriptParameters.getParams().getBoolean("stop_at_level");
		
		// set the final connectivity parameters
		if (image.getNDims()==3) {
			if (connect3D.equals("3D:26/6")) {
				dimension = 3;
				objConnect = 26;
				bckConnect = 6;
			} else if (connect3D.equals("3D:18/6")) {
				dimension = 3;
				objConnect = 18;
				bckConnect = 6;
			} else if (connect3D.equals("3D:6/26")) {
				dimension = 3;
				objConnect = 6;
				bckConnect = 26;
			} else if (connect3D.equals("3D:6/18")) {
				dimension = 3;
				objConnect = 6;
				bckConnect = 18;
			} 
		} else if (image.getNDims()==2) {
			if (connect2D.equals("2D:8/4")) {
				dimension = 2;
				objConnect = 8;
				bckConnect = 4;
			} else if (connect2D.equals("2D:4/8")) {
				dimension = 2;
				objConnect = 4;
				bckConnect = 8;
			}
		}
		// set the mask, check type compatibility
		if (inputSkel.equals("paint_mask")) {
            inputPaint = image.getParentFrame().getComponentImage().getPaintMask();
        }
		if ( (type.equals("binary_objects")) && (!inputSkel.equals("intensity")) ) {
			inputSkel = "intensity";
		}
		
    }

    /**
     * Store the parameters from the dialog to record the execution of this algorithm.
     * 
     * @throws  ParserException  If there is a problem creating one of the new parameters.
     */
    protected void storeParamsFromGUI() throws ParserException {
        scriptParameters.storeInputImage(image);
        scriptParameters.storeOutputImageParams(resultImage, true);  // false means computation in place

        scriptParameters.getParams().put(ParameterFactory.newParameter("algorithm_type", type));
		scriptParameters.getParams().put(ParameterFactory.newParameter("2D_connectivity", connect2D));
		scriptParameters.getParams().put(ParameterFactory.newParameter("3D_connectivity", connect3D));
		scriptParameters.getParams().put(ParameterFactory.newParameter("initialization", inputSkel));
		scriptParameters.getParams().put(ParameterFactory.newParameter("propagation_type", propagationType));
		scriptParameters.getParams().put(ParameterFactory.newParameter("lowest_level", lowestLevel));
		scriptParameters.getParams().put(ParameterFactory.newParameter("highest_level", highestLevel));
		scriptParameters.getParams().put(ParameterFactory.newParameter("regularization", minDistance));
		scriptParameters.getParams().put(ParameterFactory.newParameter("normalize_image", useMinMax));		
		scriptParameters.getParams().put(ParameterFactory.newParameter("stop_at_level", thresholdStop));		
    }
}
