package edu.jhmi.rad.medic.floatvoi;

import edu.jhmi.rad.medic.visualization.framework.loaders.IncompatibleFormatException;
import edu.jhmi.rad.medic.visualization.framework.loaders.VTKMeshLoader;
import edu.jhmi.rad.medic.visualization.models.TriangleMesh;
import edu.jhmi.rad.medic.visualization.primitives.*;
import edu.jhmi.rad.medic.visualization.ui.ProgressBarDialog;

import gov.nih.mipav.view.*;
import gov.nih.mipav.view.dialogs.*;
import gov.nih.mipav.model.structures.*;

import java.awt.event.*;
import java.awt.*;
import java.util.*;
import java.util.Timer;
import java.io.*;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;

import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.event.*;
import javax.swing.border.*;


/**
 * Float VOI Dialog - Implements basic 
 */
@SuppressWarnings("serial")
public class JDialogFloatVoi extends JDialogBase implements MouseListener, MouseWheelListener, MouseMotionListener, KeyListener, TreeSelectionListener, WindowListener{
	private	ModelImage image;							// Reference to image
	private ViewJComponentEditImage editImage;			// Reference to actual drawing component
	private ViewJComponentTriImage[] triImages = null;	// Reference to the images in the TriFrame	
	public Vector<FVOI> voiVector = new Vector<FVOI>(50);		// VOI vector
	private float oldXZoom, oldYZoom;					// Required : using zoom tools causes a click event that can place a voi	
	private int singleSelectedVOI;						// Selected voi
	private int singleHoverVOI;							// Cause display of VOI number upon hover
	public float num_slices = 0;
	private final double SELECT_THRESH = 4;				// Select threshold in pixel distance	
	public Vector<VTreeNode> groupVector = new Vector<VTreeNode>(50);
	private int GroupCounter = 0;
	private VTreeNode sel_GroupNode = null;
	private JDialogFloatVoi me;
	public int[] extents;
	
	private Timer timer1, timer2;
	
	private Vector<JSurfaceFrame> jsfVector;
	
	private boolean isPaused = false;
	
	private final int NODE_GROUP = 1;	// Represents a group in the tree view
	private final int NODE_VOI   = 2;	// Represents a voi in the tree view
	
	
	/** GUI Stuff ************************************************************************/
	private JMenuItem loadMenuItem;
	private JMenuItem saveAsMenuItem;
	private JMenuItem saveAllMenuItem;
	private JMenuItem loadSurfaceMenuItem;	
	private JTree voiTree;
	private DefaultMutableTreeNode root;
	private JColorChooser colorChooser	= new JColorChooser();
	private JFileChooser fc				= new JFileChooser();
	
	
	
	/** Options Panel ********************************************************************/
	private JToggleButton pauseButton;
	private JButton newGroupButton;
	private JButton deleteGroupButton;
	private JButton loadDefaultsButton;
	
	/** Group Properties *****************************************************************/
	private JButton renameButton;
	private JButton colorButton;
	private JTextField groupNameText;
		
	/** VOI Properties *******************************************************************/
	private JLabel voiGroupName		= new JLabel();
	private JLabel voiCoordinates	= new JLabel();
	private JLabel voiNumber		= new JLabel();
	
	private boolean redrawFlag = false;
	
	public Vector<Line> lines;	

	public JDialogFloatVoi(Frame parentFrame, ModelImage iImage) {
		// Store the model image reference.
		image = iImage;
		
		// Get the Edit Image object
		editImage = image.getParentFrame().getComponentImage();
		
		// Get the number of slices. This is required to maintain the same numbering scheme across tri-planar and primary views
		num_slices = image.getExtents()[2] - 1;
		
		extents = image.getExtents();
		
		jsfVector = new Vector<JSurfaceFrame>();
		
		// Get the initial zoom for the primary image.
		oldXZoom = editImage.getZoomX();
		oldYZoom = editImage.getZoomY();
		
		//  Set the selected voi index to negative 
		singleSelectedVOI = -1;
		
		// Check if the tri-planar frame is available. If it is get references to the edit images
		if(image.getTriImageFrame() != null) {					
			triImages = new ViewJComponentTriImage[3];			
			triImages[0] = image.getTriImageFrame().getTriImage	(0);
			triImages[1] = image.getTriImageFrame().getTriImage	(1);
			triImages[2] = image.getTriImageFrame().getTriImage	(2);
		}
		
		// Initialize the dialog
		init_dialog();	
			
		
		me = this;
		
		timer1 = new Timer();
		timer2 = new Timer();
		timer1.scheduleAtFixedRate(new GCTimer(), 10000, 10000);
	}

	class RedrawTimer extends TimerTask {
		public void run() {			
			drawVOI();
			redrawFlag = false;
		}
	}

	class GCTimer extends TimerTask {
		public void run() {
			System.gc();
		}
	}
	
	/*
	 * Initializes the Dialog. Mostly UI stuff.
	 */
	private void init_dialog() {
		Container contentPane = getContentPane();				
		setTitle("Float Point VOI (" + image.getImageName() + ")");			
		setSize(400,550);
		
		TitledBorder title;				
		
		//g.setHgap(10);
		JPanel uiPanel = new JPanel(new GridLayout(1,2));
		JPanel propPanel = new JPanel(new GridLayout(4,0,2,5));	
		uiPanel.setSize(400,400);
		
		// Create the voi tree
		root = new DefaultMutableTreeNode("VOI List");
		voiTree = new JTree(root);
		voiTree.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
		voiTree.addTreeSelectionListener(this);
		
		JScrollPane treeView = new JScrollPane(voiTree);
		uiPanel.add(treeView);
		
		// Create the options panel				
		title = BorderFactory.createTitledBorder("Options");
		JPanel optionsPanel = new JPanel(new GridLayout(0,1,5,5));
		optionsPanel.setBorder(title);
		
		newGroupButton = new JButton("New Group");		
		deleteGroupButton = new JButton("Delete Group");
		loadDefaultsButton = new JButton("Load Default Groups");
		pauseButton = new JToggleButton("Stop Recording");
				
		optionsPanel.add(pauseButton);
		optionsPanel.add(newGroupButton);
		optionsPanel.add(deleteGroupButton);
		optionsPanel.add(loadDefaultsButton);
		
		pauseButton.addActionListener(this);
		newGroupButton.addActionListener(this);
		deleteGroupButton.addActionListener(this);
		loadDefaultsButton.addActionListener(this);
				
		// Create Group Panel
		JPanel groupPanel = new JPanel(new GridLayout(0,1,5,5));
		groupNameText = new JTextField();
		renameButton = new JButton("Rename");
		colorButton = new JButton("Color");
		
		title = BorderFactory.createTitledBorder("Group Options");
		groupPanel.setBorder(title);		
		groupPanel.add(new Label("Group Name"));
		groupPanel.add(groupNameText);		
		groupPanel.add(renameButton);
		groupPanel.add(colorButton);
		
		renameButton.addActionListener(this);
		colorButton.addActionListener(this);
		
		// VOI Properties Panel	
		title = BorderFactory.createTitledBorder("VOI Parameters");		
		JPanel voiPanel = new JPanel(new GridLayout(0,2,5,5));
		
		voiPanel.add(new JLabel("Group Name :"));
		voiPanel.add(voiGroupName);
		voiPanel.add(new JLabel("Coordinates :"));
		voiPanel.add(voiCoordinates);
		voiPanel.add(new JLabel("Index :"));
		voiPanel.add(voiNumber);
				
		propPanel.add(optionsPanel);
		propPanel.add(groupPanel);		
		propPanel.add(voiPanel);		
				
		uiPanel.add(propPanel);		
				
		// Setup Menu Bars
		JMenuBar menuBar = new JMenuBar();
		JMenu menu = new JMenu("File");		
		menu.setMnemonic(KeyEvent.VK_F);
		
		// Add Open
		loadMenuItem = new JMenuItem("Load VOI Group", KeyEvent.VK_O);
		loadMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O, ActionEvent.ALT_MASK));
		loadMenuItem.addActionListener(this);
		menu.add(loadMenuItem);
					
		// Add Save as
		saveAsMenuItem = new JMenuItem("Save VOI Group as");
		saveAsMenuItem.addActionListener(this);
		menu.add(saveAsMenuItem);
		
		// Add Save All VOI as
		saveAllMenuItem = new JMenuItem("Save All Groups to");
		saveAllMenuItem.addActionListener(this);
		menu.add(saveAllMenuItem);		
		
		// Load Surface
		menu.addSeparator();
		loadSurfaceMenuItem = new JMenuItem("Load Surface");
		loadSurfaceMenuItem.addActionListener(this);		
		menu.add(loadSurfaceMenuItem);
		
		menuBar.add(menu);
		
		getContentPane().add(uiPanel);
		setJMenuBar(menuBar);
				
		// Add Listeners for primary image
		editImage.addMouseListener(this);
		editImage.addMouseWheelListener(this);
		editImage.addMouseMotionListener(this);
		image.getParentFrame().addWindowListener(this);
		image.getParentFrame().addKeyListener(this);
		
		// Add Listeners for tri images
		if(triImages != null) {	
			image.getTriImageFrame().addKeyListener(this);
			for(int i = 0; i < 3; ++i) {
				triImages[i].addKeyListener(this);
				triImages[i].addMouseListener(this);
				triImages[i].addMouseWheelListener(this);
				triImages[i].addMouseMotionListener(this);		
			}		
		}
		
		// Called by default
		//addWindowListener(this);
		
		// Trigger a group add
		ActionEvent e = new ActionEvent(newGroupButton, 0, "");		
		actionPerformed(e);
		
		// Set the dialog to be visible
		setVisible(true);
	}
	
	public void valueChanged(TreeSelectionEvent e) {
		sel_GroupNode = null;
		DefaultMutableTreeNode node = (DefaultMutableTreeNode) voiTree.getLastSelectedPathComponent();
		if(node == null) 
			return;
			
		Object nodeInfo = node.getUserObject();
		if(nodeInfo == null)
			return;
		int GroupID = ((VTreeNode) node).ID;
		int size = groupVector.size();
		for(int i = 0; i < size; ++i)
			if(groupVector.get(i).ID ==GroupID) {
				UpdateUIFromNode((VTreeNode) node);		
				break;
			}
	}
	
	/*
	 * Updates UI from specified VTreeNode object.
	 *	@param	vtn		VTreeNode object from where to retrieve VOIGroup object.
	 *	@see	VTreeNode
	 *	@see	VOIGroup
	 */
	private void UpdateUIFromNode(VTreeNode vtn) {
		if(vtn.Type == NODE_GROUP) {		
			sel_GroupNode = vtn;
			VOIGroup vg = (VOIGroup) vtn.getUserObject();
			colorButton.setBackground(vg.color);
			groupNameText.setText(vg.GroupName);			
		}
		voiTree.updateUI();
	}
	
	
	/*
	 * Adds a new empty group to the tree. Note : Must manually increment groupcounter
	 *	@param	groupName	Name of the string
	 *	@param	ID			integer index of group
	 *	@param	gc			Color used to draw VOI belonging in this group
	 *	@return				the index of this group in the groupVector object.	
	 *
	 */
	private int AddNewGroup(String groupName, int ID, Color gc) {
		VOIGroup vg = new VOIGroup(groupName, ID);
		vg.color = gc;
		VTreeNode vtn = new VTreeNode(vg, NODE_GROUP, ID);
		groupVector.add(vtn);
		root.add(vtn);
		UpdateUIFromNode(vtn);
		sel_GroupNode  = vtn;
		return groupVector.size() -1;
	}
	
	public void actionPerformed(ActionEvent event) {
		Object source = event.getSource();
		
		// Adds a new Group to the Tree
		if(source == newGroupButton) {	
			AddNewGroup("Group " + GroupCounter, GroupCounter, GetColor(GroupCounter));
			GroupCounter++;
			return;
		}
		
		// Deletes Group From Tree
		if(source == deleteGroupButton) {
			if(sel_GroupNode != null) {
				int GroupID = sel_GroupNode.ID;
				DefaultTreeModel model = (DefaultTreeModel)voiTree.getModel();    			    		    
			    model.removeNodeFromParent(sel_GroupNode);
				voiTree.updateUI();				
				for(int i = 0; i < voiVector.size(); ++i)
					if(voiVector.get(i).GroupID == GroupID) {					
						voiVector.remove(i);
						--i;
					}				
				drawVOI();
			} 
			return;
		}
		
		// Color Chooser Button
		if(source == colorButton) {
			if(sel_GroupNode == null)
				return;
			colorChooser.setColor(((VOIGroup) sel_GroupNode.getUserObject()).color);			
			
			Color newColor = colorChooser.showDialog(null, "Choose Group Color", colorChooser.getColor());										
			if(newColor == null)
				return;

			int GroupID = sel_GroupNode.ID;
			int size = groupVector.size();
			for(int i = 0; i < size; ++i) 
				if(groupVector.get(i).ID == GroupID) {
					VOIGroup vg = (VOIGroup) groupVector.get(i).getUserObject();
					vg.color = newColor;
					groupVector.get(i).setUserObject(vg);						
					UpdateUIFromNode(groupVector.get(i));
					break;
				}			
			drawVOI();			
			return;
		}
		
		// Pause Button
		if(source == pauseButton) {
			isPaused = !isPaused;
			if(isPaused)
				pauseButton.setText("Start Recording");
			else
				pauseButton.setText("Stop Recording");
			return;
		}
		
		// Rename Button
		if(source == renameButton) {
			if(sel_GroupNode == null)
				return;
			String newName = groupNameText.getText();
			int GroupID = sel_GroupNode.ID;
			int size = groupVector.size();
			for(int i = 0; i < size; ++i) 
				if(groupVector.get(i).ID == GroupID) {
					VOIGroup vg = (VOIGroup) groupVector.get(i).getUserObject();
					vg.GroupName = newName;
					groupVector.get(i).setUserObject(vg);
					UpdateUIFromNode(groupVector.get(i));
					break;
				}			
			return;
		}
		
		// Save As triggered
		if(source == saveAsMenuItem) {			
			fc.setFileSelectionMode(JFileChooser.FILES_ONLY );
			fc.setSelectedFile(new File(((VOIGroup) sel_GroupNode.getUserObject()).GroupName + ".xml"));
			int returnVal = fc.showSaveDialog(this);
			if(returnVal == JFileChooser.APPROVE_OPTION) {
				File file =fc.getSelectedFile();					
				Save(file, (VOIGroup) sel_GroupNode.getUserObject());			
			}
			return;
		}
		
		// Save to
		if(source == saveAllMenuItem) {
			fc.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
			int returnVal = fc.showSaveDialog(this);
			if(returnVal == JFileChooser.APPROVE_OPTION)
				saveAllGroup(fc.getSelectedFile());			
			return;
		}
		
		// Load VOI
		if(source == loadMenuItem) {
			fc.setFileSelectionMode(JFileChooser.FILES_ONLY );
			int returnVal = fc.showOpenDialog(this);
			if(returnVal == JFileChooser.APPROVE_OPTION)
				LoadVOI(fc.getSelectedFile());			
			return;
		}
		
		// Load Defaults Button
		if(source == loadDefaultsButton) {
			LoadDefaults();
			return;
		}
		
		// Load Surface
		if(source == loadSurfaceMenuItem) {
			fc.setFileSelectionMode(JFileChooser.FILES_ONLY );
			int returnVal = fc.showOpenDialog(this);
			if(returnVal == JFileChooser.APPROVE_OPTION) {					
				new Thread( new Runnable() {					
      				public void run() {          					        
      					File f = fc.getSelectedFile();	      					
      					ProgressBarDialog pb = new ProgressBarDialog("Loading ...", "Reading Surface : " + f.getName(), 0, 100); 					      					
      					try {									
							String extension = f.getName().substring(f.getName().lastIndexOf('.') + 1).toLowerCase();		
							String errorMessage = "Unable to load Surface File.";
							TriangleMesh m = null;
							if(extension.equals("vtk")){
								VTKMeshLoader loader = new VTKMeshLoader();
								try {
									m = (TriangleMesh) loader.Load(f.getAbsolutePath(), pb);
								} catch (IncompatibleFormatException e) {
									JOptionPane.showMessageDialog(null, "An error was encountered while loading surface file.", "Loading Error", JOptionPane.ERROR_MESSAGE); 
									e.printStackTrace();
								}
							}
							else
								errorMessage = "Unrecognized file format.";
            				if(m == null) 
								JOptionPane.showMessageDialog(null, errorMessage);
							else {							
								JSurfaceFrame jsf = new JSurfaceFrame(me, m,editImage, image.getExtents() );								
								
								jsfVector.addElement(jsf);
							}
      					}
						catch(IOException e) {
							JOptionPane.showMessageDialog(null, "Error loading surface file. Reason : " + e.getMessage());
						}
						pb.dispose();
          			}	
    			}).start();					
			}
			System.gc();
			return;
		}
	}
	

	public int getSelectedVOI() {
		return singleSelectedVOI;
	}

	public int getHoverVOI() {
		return singleHoverVOI;
	}

	/*
	 * Loads a VOI group from specified file object.
	 *	@param	file	FILE object from where to read.
	 */
	private void LoadVOI(File file) {
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		NodeList nlct, nlcl, nlpt;
		try {
			// Apologizes for poor XML parsing.
			DocumentBuilder db = dbf.newDocumentBuilder();
			Document dom = db.parse(file.getAbsolutePath());
			Element docEle = dom.getDocumentElement();
			
			if(docEle.getTagName().compareTo("VOI") == 0) {
				nlct = docEle.getElementsByTagName("Curve-type");
				nlcl = docEle.getElementsByTagName("Color");
				nlpt = docEle.getElementsByTagName("Pt");
				if(nlct == null)
					throw new Exception("Not a valid VOI file (Curve Type Tag Missing)");
				if(nlcl == null)
					throw new Exception("Not a valid VOI file (Color Tag missing)");
				if(nlct.getLength() == 0 || nlcl.getLength() == 0)
					throw new Exception("Not a valid VOI file (Color Tag or Curve- Tag missing)");
				// MIPAV curve-type for point voi is 3
				if(nlct.item(0).getFirstChild() != null && nlct.item(0).getFirstChild().getNodeValue().compareTo("3") == 0) {
					String colorString = "255,255,0,0";	// Default color is red 
					if(nlcl.item(0).getFirstChild() != null)
						colorString = nlcl.item(0).getFirstChild().getNodeValue();
					AddNewGroup(file.getName(), GroupCounter, GetColor(colorString));
					for(int i = 0; i < nlpt.getLength(); ++i) {						
						String [] coords =  nlpt.item(i).getFirstChild().getNodeValue().split(",");
						if(coords.length != 3)
							continue;
						FVOI voi = new FVOI(Float.parseFloat(coords[0]), Float.parseFloat(coords[1]), Float.parseFloat(coords[2]), GroupCounter);
						voi.VOINumber = sel_GroupNode.getChildCount() + 1;
						VTreeNode vtn = new VTreeNode("" + voi.VOINumber, NODE_VOI, voi.VOINumber);
						voiVector.addElement(voi);
						sel_GroupNode.add(vtn);							
					}
					voiTree.updateUI();	
					GroupCounter++;
				}
			}
			else
				throw new Exception("Not a valid VOI file (VOI Tag missing)");
		}
		catch(Exception e) {
			JOptionPane.showMessageDialog(null, "Unable to open file. Reason : " +   e.getMessage());
		}
		drawVOI();
	}
	
	/*
	 * Loads default group settings from accompanying "floatVOI.xml" file
	 */
	private void LoadDefaults() {
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		NodeList nl1, nl2;
		Node n;	
		try {
			DocumentBuilder db = dbf.newDocumentBuilder();			
			Document dom = db.parse(this.getClass().getResource("FloatVOI.xml").toString());
			Element docEle = dom.getDocumentElement();	
			nl1 = docEle.getElementsByTagName("Group");			
						
			if(nl1 != null && nl1.getLength() > 0)
				for(int i = 0 ; i < nl1.getLength();i++) {
					n = nl1.item(i);
				
					Element el = (Element)nl1.item(i);
					JOptionPane.showMessageDialog(null, el.getNodeName());
			}
			else
				JOptionPane.showMessageDialog(null, "No groups found within defaults xml.");
		}catch(ParserConfigurationException pce) {
			JOptionPane.showMessageDialog(null, "Unable to parse XML. Reason : " + pce.getMessage());
			pce.printStackTrace();
		}catch(SAXException se) {
			JOptionPane.showMessageDialog(null, "Unable to parse XML. Reason : " +  se.getMessage());
			se.printStackTrace();
		}catch(IOException ioe) {
			JOptionPane.showMessageDialog(null, "Unable to open file. Reason : " + ioe.getMessage());
			ioe.printStackTrace();
		}		
	}
	
	/*
	 * Saves all open groups into specified directories. Group names are used as file name.
	 *	@param	directory	Directory where to store files.
	 *	@see	VOIGroup
	 */
	private void saveAllGroup(File directory) {		
		String dir = directory.getAbsolutePath() + "/";
		int size = groupVector.size();
		VOIGroup vg;
		for(int i = 0; i < size; ++i) {
			vg = (VOIGroup) groupVector.get(i).getUserObject();
			File f = new File(dir + vg.GroupName + ".xml");
			Save(f, vg);
		}
	}
	
	/*
	 * Save a particular groups VOI into a specified file object. Note group names are not stored into XML file.
	 *	@param	file	File object where to write VOI into
	 *	@param	vg		VOI Group object
	 *	@see	VOIGroup
	 */
	private void Save(File file, VOIGroup vg) {
		try {
			BufferedWriter out = new BufferedWriter(new FileWriter(file));
			
			// Output XML Headers
			out.write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\r\n");
			out.write("<!-- MIPAV VOI file -->\r\n");
						
			// Now Write out MIPAV compatible XML.
			out.write("<VOI xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\">\r\n");
			out.write("<Unique-ID>0</Unique-ID>\r\n");
			out.write("<Curve-type>3</Curve-type>\r\n");
			
			// Write out color in ARGB format
			out.write("<Color>255," + vg.color.getRed() + "," + vg.color.getGreen() + "," + vg.color.getBlue() + "</Color>\r\n");
			
			// Write out thickness
			out.write("<Thickness>1</Thickness>\r\n");
			
			int size = voiVector.size();			
			for(int i = 0; i < size; ++i) {
				if(voiVector.get(i).GroupID == vg.GroupID) {
					out.write("<Contour>\r\n");
					out.write("\t<Pt>" + voiVector.get(i).x + "," + voiVector.get(i).y + "," + voiVector.get(i).z + "</Pt>\r\n");
					out.write("</Contour>\r\n");
				}
			}
			out.write("</VOI>\r\n");
			out.close();
		}
		catch(IOException e) {
			JOptionPane.showMessageDialog(null, "Error Writing File. Check Permissions and Disk space");
			e.printStackTrace();			
		}
	}
	
	private void updateList() {
		int voi_display_index = -1;
		if(singleHoverVOI != -1)
			voi_display_index = singleHoverVOI;
		if(singleSelectedVOI != -1)
			voi_display_index = singleSelectedVOI;
		if(voi_display_index == -1) {
			voiNumber.setText("---");
			voiCoordinates.setText("---");
			voiGroupName.setText("---");
			return;
		}
		
		FVOI voi = voiVector.get(voi_display_index);		
		voiNumber.setText("" + voi.VOINumber);
		voiCoordinates.setText("" + voi.x + ", " + voi.y + ", " + voi.z);			
		voiCoordinates.setAutoscrolls(true);
		if(voi != null) {		
			VTreeNode vtn = getGroupNode(voi.GroupID);
			if(vtn == null)
				return;
			voiGroupName.setText("" + ((VOIGroup) vtn.getUserObject()).GroupName);			
		}		
	}
	
	/* Returns the VTreeNode object based on the group id.
	 *	@param	groupID		integer id of group
	 *	@return	VTreeNode
	 *	@see	VTreeNode
	 */
	private VTreeNode getGroupNode(int groupID) {
		int size = groupVector.size();
		for(int i = 0; i < size; ++i) {
			if(groupVector.get(i).ID == groupID)
				return groupVector.get(i);
		}
		return null;
	}
	
	/*
	 * Initiates the Draw Routine for primary image + tri-planar images.
	 */
	private void drawVOI() {
		try {
			if(editImage == null)
				return;
			Graphics g = editImage.getGraphics();
			if(g == null)
				return;
			editImage.paintComponent(g);
			for(int i = 0; i < jsfVector.size(); ++i)			
				jsfVector.elementAt(i).drawOutline(editImage);		
			drawVOI(editImage, -1);
			if(triImages != null) {
				
				g = triImages[0].getGraphics();
				triImages[0].paintComponent(g);
				g = triImages[1].getGraphics();
				triImages[1].paintComponent(g);
				g = triImages[2].getGraphics();
				triImages[2].paintComponent(g);
				
				drawVOI(triImages[0], 0);		
				drawVOI(triImages[1], 1);
				drawVOI(triImages[2], 2);
			}				
		}
		catch(Exception e) {
			//Suppresed;
		}
	}
	
	/*
	 * Draws VOIs on the specified image frame.
	 * Image 0 : X, Y						
	 * Image 1 : Y, Z
	 * Image 2 : X, Z
	 *	@param	selImage	Image on which to draw voi on
	 *	@param	pos			Position index of selected voi in voi vector.	 *
	 */
	private void drawVOI(ViewJComponentEditImage selImage, int pos) {
		int size = voiVector.size();
			
		float xCor = selImage.getZoomX() * selImage.getResolutionX();
		float yCor = selImage.getZoomY() * selImage.getResolutionY();
		int x = 0, y = 0;
		double width = selImage.getSize().getWidth();
		long currentSlice = selImage.getSlice();	
				
		Graphics g = selImage.getGraphics();
		if(g == null)
			return;
		g.setColor(Color.RED);
		Color drawColor = Color.RED;
		
		//selImage.paintComponent(g);
		int groupID = 0;
		
		for(int i = 0; i < size; ++i) {
			switch(pos) {
				case -1:
				case  0:					
					if(Math.floor(voiVector.get(i).z) != currentSlice && Math.ceil(voiVector.get(i).z) != currentSlice) 			
						continue;
					x = Math.round(voiVector.get(i).x * xCor);
					y = Math.round(voiVector.get(i).y * yCor);
					break;					
				case  1:
					if(Math.floor(voiVector.get(i).x) != currentSlice && Math.ceil(voiVector.get(i).x) != currentSlice) 			
						continue;
					x = Math.round(voiVector.get(i).y * xCor);
					y = Math.round((num_slices - voiVector.get(i).z) * yCor);		
					break;
				case  2:
					if(Math.floor(voiVector.get(i).y) != currentSlice && Math.ceil(voiVector.get(i).y) != currentSlice) 			
						continue;
					x = Math.round(voiVector.get(i).x * xCor);
					y = Math.round((num_slices - voiVector.get(i).z) * yCor);		
					break;
						
			}
			
			groupID = voiVector.get(i).GroupID;
			VTreeNode vtn =  getGroupNode(groupID);
			if(vtn != null)	
				drawColor = ((VOIGroup) vtn.getUserObject()).color;
			else
				drawColor = Color.white;
			
			if(i == singleHoverVOI)
				g.setColor(Color.yellow);
			else
				g.setColor(drawColor);
			g.drawLine(x - 4,	 y,	x + 4,	  y);
			g.drawLine(x	,y - 4,	x	 ,y + 4);				
			if(i == singleSelectedVOI) {
				g.setColor(Color.WHITE);
				g.drawRect(x - 2, y - 2, 4, 4);
				g.setColor(Color.BLACK);
				g.fillRect(x - 1, y - 1, 3, 3);							
			}				
			if(i == singleHoverVOI || singleSelectedVOI == i) {			
				g.setColor(Color.YELLOW);	
				int sX = x;
				int sY = y;
				if(sX < 20)
					sX += 20;
				else {
					if(width - sX < 100) 
						sX -= 120;
					else
						sX -= 20;					
				}		
					
				if(sY < 30)
					sY += 25;	
				else
					sY -= 10;
					
				g.drawString("" + voiVector.get(i).VOINumber + " (" +
					voiVector.get(i).x + ", " +
					voiVector.get(i).y + ", " +
					voiVector.get(i).z + ")",
					sX, sY);		
			}			
		}	
	}
	
	/*
	 * Returns color for a group.
	 *	@param	groupID		Unique identifier of group
	 */
	public Color getGroupColor(int groupID) {		
		Color drawColor;
		VTreeNode vtn =  getGroupNode(groupID);
		if(vtn != null)	
			drawColor = ((VOIGroup) vtn.getUserObject()).color;
		else
			drawColor = Color.white;
		return drawColor;
	}

	/*
	 * Adds a voi from current mouse co-ordinates.	 
	 *	@param	mouseV1		first co-ordinate of pointer
	 *	@param	mouseV2		second co-ordinate of pointer	 
	 *	@param	selImage	image on which selected voi is being repositioned.
	 *	@param	image_num	index of image in tri-planar view. Primary view and triplanar view 0 are considered the same.
	 */
	private void addVOI(float mouseV1, float mouseV2, ViewJComponentEditImage selImage, int image_num) {
		float xZoom = selImage.getZoomX();
		float yZoom = selImage.getZoomY();
		float xRes  = selImage.getResolutionX();
		float yRes  = selImage.getResolutionY();

		float v1 = (mouseV1 / (xZoom * xRes));
		float v2 = (mouseV2 / (yZoom * yRes));	
		float v3 = selImage.getSlice();

		switch(image_num) {
			case -1:
			case  0:
				voiVector.addElement(new FVOI(v1,v2,v3, sel_GroupNode.ID));
				break;
			case  1:
				voiVector.addElement(new FVOI(v3,v1, num_slices - v2, sel_GroupNode.ID));
				break;
			case  2:
				voiVector.addElement(new FVOI(v1, v3, num_slices -v2, sel_GroupNode.ID));
				break;
		}		
		
		// Add VOI to UI Tree
		FVOI last = voiVector.get(voiVector.size() - 1);		
		last.VOINumber = sel_GroupNode.getChildCount() + 1;
		VTreeNode vtn = new VTreeNode("" + last.VOINumber, NODE_VOI, last.VOINumber);
		sel_GroupNode.add(vtn);	
		voiTree.updateUI();			
	}
	
	/*
	 * Updates the position of the selected voi.
	 *	@param	mouseV1		first co-ordinate of pointer
	 *	@param	mouseV2		second co-ordinate of pointer
	 *	@param	pos			index of selected voi in voi vector
	 *	@param	selImage	image on which selected voi is being repositioned.
	 *	@param	image_num	index of image in tri-planar view. Primary view and triplanar view 0 are considered the same.
	 *	@see	ViewJComponentEditImage
	 */
	private void updateVOI(float mouseV1, float mouseV2, int pos, ViewJComponentEditImage selImage, int image_num) {
		if(pos < 0 || pos >= voiVector.size())
			return;

		float xZoom = selImage.getZoomX();
		float yZoom = selImage.getZoomY();
		float xRes  = selImage.getResolutionX();
		float yRes  = selImage.getResolutionY();

		float v1 = (mouseV1 / (xZoom * xRes));
		float v2 = (mouseV2 / (yZoom * yRes));	

		switch(image_num) {
			case -1:
			case  0:
				voiVector.get(pos).x = v1;
				voiVector.get(pos).y = v2;
				break;
			case  1:
				voiVector.get(pos).y = v1;
				voiVector.get(pos).z = num_slices - v2;
				break;
			case  2:
				voiVector.get(pos).x = v1;
				voiVector.get(pos).z = num_slices - v2;
				break;
		}
		updateList();
	}
	
	/*
	 * Deletes a voi from the voi vector.
	 *	@param	pos		position index of voi to be deleted from the vector	 
	 */
	private void deleteVOI(int pos) {
		// Check if the voi index is valid in voi vector		
		if(pos < 0 || pos >= voiVector.size())
			return;
			
		// Get the voi object.
		FVOI fv = voiVector.get(pos);
		// Fetch group id and remove voi from the tree.
		int groupID = fv.GroupID;			
		VTreeNode vtn =  getGroupNode(groupID);
		if(vtn != null) {				
			VTreeNode child;
			for(int i = 0; i < vtn.getChildCount(); ++i) {
				child = (VTreeNode) vtn.getChildAt(i);
				if(child != null)
					if(child.ID == fv.VOINumber) {						
						vtn.remove(i);
						break;
					}
			}
			voiTree.updateUI();				
		}
		// Remove voi from the voi vector
		if(pos >= 0 && pos < voiVector.size())
			voiVector.remove(pos);		
	}
		
	/*
	 * Attempts to a select a voi irrespective of group based on the pointers current coordinates.
	 * Only selects one VOI at a time
	 *	@param	x			x co-ordinate of pointer
	 *	@param	y			y co-ordinate of pointer
	 *	@param	selImage	the image from where the voi will be selected from.
	 *	@param	image_num	a integer index ranging from 0-2 in case the call orginated from a tri-planar image
	 *	@return				a index into the voi vector for the selected voi. If no voi is selected returns -1.
	 *	@see	ViewJComponentEditImage
	 */
	private int AttemptSelect(double x, double y, ViewJComponentEditImage selImage, int image_num) {
		int size = voiVector.size();		
		int selected = -1;
		double xVOI, yVOI, distance;
		double xCor = selImage.getZoomX() * selImage.getResolutionX();
		double yCor = selImage.getZoomY() * selImage.getResolutionY();
		double currentSlice = selImage.getSlice();
		double v1 = 0, v2 = 0;

		for(int i = 0; i < size; ++i) {
			switch(image_num) {
				case -1:
				case 0:					
					if(Math.floor(voiVector.get(i).z) != currentSlice && Math.ceil(voiVector.get(i).z) != currentSlice) 			
						continue;
					v1 = voiVector.get(i).x * xCor;
					v2 = voiVector.get(i).y * yCor;
					break;					
				case 1:
					if(Math.floor(voiVector.get(i).x) != currentSlice && Math.ceil(voiVector.get(i).x) != currentSlice) 			
						continue;
					v1 = voiVector.get(i).y * xCor;
					v2 = (num_slices - voiVector.get(i).z) * yCor;		
					break;
				case 2:
					if(Math.floor(voiVector.get(i).y) != currentSlice && Math.ceil(voiVector.get(i).y) != currentSlice) 			
						continue;
					v1 = voiVector.get(i).x * xCor;
					v2 = (num_slices - voiVector.get(i).z) * yCor;		
					break;
			}
			distance = Math.sqrt(Math.pow(x - v1, 2.0) + Math.pow(y - v2, 2.0));
			if(distance < SELECT_THRESH) {
				selected = i;
				break;
			}		
		}
		return selected;
	}
		
	public void mouseClicked(MouseEvent mouseEvent) {		
		if(mouseEvent.getButton() != MouseEvent.BUTTON1 || isPaused)
			return;
		
		Object source = mouseEvent.getSource();
		float x = mouseEvent.getX();
		float y = mouseEvent.getY();

		singleSelectedVOI =  AttemptSelect(x, y, editImage, -1);								

		// Check if event originated from single or tri-image
		if(source == editImage) {
			float xZoom = editImage.getZoomX();
			float yZoom = editImage.getZoomY();
			float xS, yS, zS;					
			// Check if the left button has been clicked and the zoom levels have not changed		
			if(oldXZoom == xZoom && oldYZoom == yZoom && singleSelectedVOI == -1)
					addVOI(x, y, editImage, -1);
			oldXZoom = xZoom;
			oldYZoom = yZoom;
		}
		else if(source == triImages[0] && singleSelectedVOI == -1) 
				addVOI(x,y, triImages[0], 0);
		else if(source == triImages[1] && singleSelectedVOI == -1) 
				addVOI(x,y, triImages[1], 1);
		else if(source == triImages[2] && singleSelectedVOI == -1) 
				addVOI(x,y, triImages[2], 2);	
		drawVOI();
	}
	
	public void mouseDragged(MouseEvent mouseEvent) {
		if(!isPaused) {		
			Object source = mouseEvent.getSource();
			float x = mouseEvent.getX();
			float y = mouseEvent.getY();
		
			if(source == editImage) 
				updateVOI(x, y, singleSelectedVOI, editImage, -1);
			else if(source == triImages[0]) 
				updateVOI(x, y, singleSelectedVOI, triImages[0], 0);
			else if(source == triImages[1]) 
				updateVOI(x, y, singleSelectedVOI, triImages[1], 1);
			else if(source == triImages[2]) 
				updateVOI(x, y, singleSelectedVOI, triImages[2], 2);
				updateList();
		}
		drawVOI();		
	}
          
 	public void mouseMoved(MouseEvent mouseEvent) {
 		if(isPaused)
 			return;
		Object source =mouseEvent.getSource();
		double x = mouseEvent.getX();
		double y = mouseEvent.getY();

 		if(source == editImage)
 			singleHoverVOI = AttemptSelect(x, y, editImage	 , -1);
		else if(source == triImages[0]) 
	 		singleHoverVOI = AttemptSelect(x, y, triImages[0],  0);
		else if(source == triImages[1]) 
	 		singleHoverVOI = AttemptSelect(x, y, triImages[1],  1);
		else if(source == triImages[2]) 
	 		singleHoverVOI = AttemptSelect(x, y, triImages[2],  2);

 		if(singleHoverVOI != -1) {
 			drawVOI();
 			if(!redrawFlag)
 				timer2.schedule(new RedrawTimer(), 3000);
 		}
 		
 		updateList();
 	}  
          
 	public void	keyReleased(KeyEvent keyEvent) { 		
 		if(keyEvent.getSource() == image.getParentFrame()) {	// KeyEvent from primary image frame 		
 			if(singleSelectedVOI > -1) {
				if(keyEvent.getKeyCode() == KeyEvent.VK_DELETE) {
	 				deleteVOI(singleSelectedVOI);
	 				singleSelectedVOI = -1;	 				
				}
				else if(keyEvent.getKeyCode() == KeyEvent.VK_ESCAPE) 
					singleSelectedVOI = -1;
 			}
 		}
 		else {	// Keyevent from tri image frame
 			if(keyEvent.getKeyCode() == KeyEvent.VK_ESCAPE)
				singleSelectedVOI = -1;
			else if(keyEvent.getKeyCode() == KeyEvent.VK_DELETE) {
				deleteVOI(singleSelectedVOI);
	 			singleSelectedVOI = -1;	 			
			}
 		}
		drawVOI();
 	}

	/*
	 * Returns a java color object based on a numeric index
	 *
	 *	@params	index	a integer value indicating a color
	 *	@return			java color object
	 *	@see			Color
	 */
	private Color GetColor(int index) {
		index = index % 8;		
		switch(index) {
			case 0:
				return Color.red;
			case 1:
				return Color.orange;			
			case 2:
				return Color.green;					
			case 3:
				return Color.cyan;
			case 4:
				return Color.blue;
			case 5:
				return new Color(112,138,144);
			case 6:
				return new Color(160,32,240);				
			case 7:				
				return new Color(255,20,147);
		}
		return Color.white;
	}
	
	/*
	 * Returns a java color object based on a csv string in argb format
	 *
	 *	@params	colorString	String containing color
	 *	@return			java color object
	 *	@see			Color
	 */
	private Color GetColor(String colorString) {
		Color color = Color.RED;
		String [] components = colorString.split(",");
		if(components != null && components.length == 4)
			color = new Color(Integer.parseInt(components[1]), Integer.parseInt(components[2]), Integer.parseInt(components[3]), Integer.parseInt(components[0]));
		return color;
	}
	
	
	/*
	 * Remove the listeners that were attached and disposes this object
	 *	@todo	Should listen for parent image window events. Dont want hanging fVOI Frames.
	 */
	public void windowClosing(WindowEvent e) {				
		if(e.getSource() == this) {			
			editImage.removeMouseListener(this);
			editImage.removeMouseWheelListener(this);
			editImage.removeMouseMotionListener(this);
			/*
			if(image != null) {			
				image.getParentFrame().removeWindowListener(this);
				image.getParentFrame().removeKeyListener(this);
			}
			*/
			
			// Add Listeners for tri images
			if(triImages != null) {	
				if(image != null)
					if(image.getTriImageFrame() != null)
						image.getTriImageFrame().removeKeyListener(this);
				for(int i = 0; i < 3; ++i) {
					if(triImages[i] == null)
						continue;
					triImages[i].removeKeyListener(this);
					triImages[i].removeMouseListener(this);
					triImages[i].removeMouseWheelListener(this);
					triImages[i].removeMouseMotionListener(this);		
				}		
			}			
			voiVector.clear();
			drawVOI();						
		}
		else 
			// @bug		Close Surface windows in the event of modelimage closing. Works but is flaky.
			getToolkit().getSystemEventQueue().postEvent(new WindowEvent(this, WindowEvent.WINDOW_CLOSING)); 					
		timer1.cancel();
		timer2.cancel();
		dispose();
		System.gc();
	}
	
	

	
	public void mouseEntered(MouseEvent mouseEvent)		{	drawVOI();	}
	public void mouseExited(MouseEvent mouseEvent)		{	drawVOI();	}
	public void mouseReleased(MouseEvent mouseEvent) 	{	drawVOI();	}
	public void mousePressed(MouseEvent mouseEvent) 	{	drawVOI();	}
	public void mouseWheelMoved(MouseWheelEvent e)		{	drawVOI();	}
 	public void keyTyped(KeyEvent keyEvent) 			{}
	public void keyPressed(KeyEvent e) 					{}
	public void windowClosed(WindowEvent e)				{}	
	public void windowActivated(WindowEvent e)			{ drawVOI();} 	         	
 	public void windowDeactivated(WindowEvent e)		{}          
 	public void windowDeiconified(WindowEvent e)		{ drawVOI();}
	public void windowIconified(WindowEvent e)			{}          
 	public void windowOpened(WindowEvent e)				{}
}
