package edu.jhmi.rad.medic.methods;

import gov.nih.mipav.view.*;

import edu.jhmi.rad.medic.libraries.*;
import edu.jhmi.rad.medic.structures.BinaryHeap4D;
import edu.jhmi.rad.medic.structures.CriticalPointLUT;
import edu.jhmi.rad.medic.structures.RotationMatrix;
import edu.jhmi.rad.medic.utilities.*;
 
/**
 *
 *  This algorithm handles making a true digital
 *	Homeomorphism from a given transform and 
 *	segmentation pair
 *
 *
 *	@version    January 2007
 *	@author     Pierre-Louis Bazin
 *		
 *
 */
 
public class DigitalHomeomorphism {
		
	// numerical quantities
	private static final	float   INF=1e30f;
	private static final	float   ZERO=1e-30f;
	private static final	float   SQR3=(float)Math.sqrt(3.0)/2.0f;
	
	// data buffers
	private 	byte[][][]		image;  		// original image 
	private 	byte[][][]		transformed;  	// transformed image 
	private 	float[][][]		intensityImage;  		// original image 
	private 	float[][][]		intensityTransformed;  	// transformed image 
	private		float[][][]		ux;				// transform from original to transformed space
	private		float[][][]		uy;				// transform from original to transformed space
	private		float[][][]		uz;				// transform from original to transformed space
	private		float[][][]		vx;				// transform from original to transformed space
	private		float[][][]		vy;				// transform from original to transformed space
	private		float[][][]		vz;				// transform from original to transformed space
	//private		float[][][]		dvx;				// transform from original to transformed space
	//private		float[][][]		dvy;				// transform from original to transformed space
	//private		float[][][]		dvz;				// transform from original to transformed space
	private		float[][][]		order;			// ordering function
	private 	boolean[][][]	free;  			// list of points that have been transformed already, thus becoming free for further transformation 
	private 	boolean[][][]	nonsimple;		// list of points that have been transformed already, thus becoming free for further transformation 
	private 	boolean[][][]	finished;		// list of points that have been transformed already, thus becoming free for further transformation 
	private static	int			nx,ny,nz;   	// image dimensions 
	private static	float		rx,ry,rz;   	// image resolutions
	
	// parameters
	private		String				connectivity = "6/26";
	private		CriticalPointLUT	lut;				// the LUT for critical points
	private 	BinaryHeap4D  	tree;		   		// the binary tree used for the fast marching
	private		float			delta = 0.1f;
	private		boolean			checkComposed = false;
	
	// computation flags
	private 	boolean 		isWorking;
	private 	boolean 		isCompleted;
	
	// for debug and display
	ViewJProgressBar            progressBar;
	static final boolean		debug=true;
	static final boolean		verbose=true;
    
	/**
	 *  constructor
     */
	public DigitalHomeomorphism(byte[][][] image_, 
									float[][][] ux_,
									float[][][] uy_,
									float[][][] uz_,
									int nx_, int ny_, int nz_,
									float rx_, float ry_, float rz_,
									String connect_,
									ViewJProgressBar bar_) {
		progressBar = bar_;
		
		rx = rx_;
		ry = ry_;
		rz = rz_;
			
		nx = nx_;
		ny = ny_;
		nz = nz_;
			
		image = image_;
		ux = ux_;
		uy = uy_;
		uz = uz_;
		
		connectivity = connect_;
		
		try {
			tree = new BinaryHeap4D(nx*ny*nz, BinaryHeap4D.MINTREE);
			if (connectivity.equals("26/6")) lut = new CriticalPointLUT("critical266LUT.raw.gz",200);
			else if (connectivity.equals("18/6")) lut = new CriticalPointLUT("critical186LUT.raw.gz",200);
			else if (connectivity.equals("6/26")) lut = new CriticalPointLUT("critical626LUT.raw.gz",200);
			else if (connectivity.equals("6/18")) lut = new CriticalPointLUT("critical618LUT.raw.gz",200);
			else if (connectivity.equals("wcs")) {
				lut = new CriticalPointLUT("critical66LUT.raw.gz",200);
				checkComposed=true;
			}
			else if (connectivity.equals("wco")) {
				lut = new CriticalPointLUT("criticalNoLUT.raw.gz",200);
				checkComposed=true;
			}
			lut.loadCompressedPattern();
			
			vx = new float[nx][ny][nz];
			vy = new float[nx][ny][nz];
			vz = new float[nx][ny][nz];
			
			order = new float[nx][ny][nz];
			free = new boolean[nx][ny][nz];
			finished = new boolean[nx][ny][nz];
			nonsimple = new boolean[nx][ny][nz];
			transformed = new byte[nx][ny][nz];
			
		} catch (OutOfMemoryError e){
			isWorking = false;
            finalize();
			System.out.println(e.getMessage());
			return;
		}
			
		isWorking = true;

		
		if (debug) MedicUtilPublic.displayMessage("DDiff:initialisation\n");
	}

	/**
	 *  constructor
     */
	public DigitalHomeomorphism(byte[][][] image_, 
									float[][] trans_,
									int nx_, int ny_, int nz_,
									float rx_, float ry_, float rz_,
									String connect_,
									ViewJProgressBar bar_) {
		progressBar = bar_;
		
		rx = rx_;
		ry = ry_;
		rz = rz_;
			
		nx = nx_;
		ny = ny_;
		nz = nz_;
			
		image = image_;
		
		connectivity = connect_;
		
		try {
			createAffineField(trans_);
			tree = new BinaryHeap4D(nx*ny*nz, BinaryHeap4D.MINTREE);
			if (connectivity.equals("26/6")) lut = new CriticalPointLUT("critical266LUT.raw.gz",200);
			else if (connectivity.equals("18/6")) lut = new CriticalPointLUT("critical186LUT.raw.gz",200);
			else if (connectivity.equals("6/26")) lut = new CriticalPointLUT("critical626LUT.raw.gz",200);
			else if (connectivity.equals("6/18")) lut = new CriticalPointLUT("critical618LUT.raw.gz",200);
			else if (connectivity.equals("wcs")) {
				lut = new CriticalPointLUT("critical66LUT.raw.gz",200);
				checkComposed=true;
			}
			else if (connectivity.equals("wco")) {
				lut = new CriticalPointLUT("criticalNoLUT.raw.gz",200);
				checkComposed=true;
			}
			lut.loadCompressedPattern();
			
			vx = new float[nx][ny][nz];
			vy = new float[nx][ny][nz];
			vz = new float[nx][ny][nz];
			
			order = new float[nx][ny][nz];
			free = new boolean[nx][ny][nz];
			finished = new boolean[nx][ny][nz];
			nonsimple = new boolean[nx][ny][nz];
			transformed = new byte[nx][ny][nz];
			
		} catch (OutOfMemoryError e){
			isWorking = false;
            finalize();
			System.out.println(e.getMessage());
			return;
		}
			
		isWorking = true;

		
		if (debug) MedicUtilPublic.displayMessage("DDiff:initialisation\n");
	}

	/**
	 *  constructor when generating an artificial field
     */
	public DigitalHomeomorphism(byte[][][] image_, 
									String transformType_,
									int nx_, int ny_, int nz_,
									float rx_, float ry_, float rz_,
									String connect_,
									ViewJProgressBar bar_) {
		progressBar = bar_;
		
		rx = rx_;
		ry = ry_;
		rz = rz_;
			
		nx = nx_;
		ny = ny_;
		nz = nz_;
			
		image = image_;
		
		connectivity = connect_;
		
		try {
			tree = new BinaryHeap4D(nx*ny*nz, BinaryHeap4D.MINTREE);
			if (connectivity.equals("26/6")) lut = new CriticalPointLUT("critical266LUT.raw.gz",200);
			else if (connectivity.equals("18/6")) lut = new CriticalPointLUT("critical186LUT.raw.gz",200);
			else if (connectivity.equals("6/26")) lut = new CriticalPointLUT("critical626LUT.raw.gz",200);
			else if (connectivity.equals("6/18")) lut = new CriticalPointLUT("critical618LUT.raw.gz",200);
			else if (connectivity.equals("wcs")) {
				lut = new CriticalPointLUT("critical66LUT.raw.gz",200);
				checkComposed=true;
			}
			else if (connectivity.equals("wco")) {
				lut = new CriticalPointLUT("criticalNoLUT.raw.gz",200);
				checkComposed=true;
			}
			lut.loadCompressedPattern();
			
			createArtificialField(transformType_);
			
			order = new float[nx][ny][nz];
			free = new boolean[nx][ny][nz];
			finished = new boolean[nx][ny][nz];
			nonsimple = new boolean[nx][ny][nz];
			transformed = new byte[nx][ny][nz];
			
		} catch (OutOfMemoryError e){
			isWorking = false;
            finalize();
			System.out.println(e.getMessage());
			return;
		}
			
		isWorking = true;

		
		if (debug) MedicUtilPublic.displayMessage("DDiff:initialisation\n");
	}

	/**
	 *  constructor
     */
	public DigitalHomeomorphism(float[][][] image_, 
									float[][][] ux_,
									float[][][] uy_,
									float[][][] uz_,
									int nx_, int ny_, int nz_,
									float rx_, float ry_, float rz_,
									String connect_,
									ViewJProgressBar bar_) {
		progressBar = bar_;
		
		rx = rx_;
		ry = ry_;
		rz = rz_;
			
		nx = nx_;
		ny = ny_;
		nz = nz_;
			
		intensityImage = image_;
		ux = ux_;
		uy = uy_;
		uz = uz_;
		
		connectivity = connect_;
		
		try {
			tree = new BinaryHeap4D(nx*ny*nz, BinaryHeap4D.MINTREE);
			if (connectivity.equals("26/6")) lut = new CriticalPointLUT("critical266LUT.raw.gz",200);
			else if (connectivity.equals("18/6")) lut = new CriticalPointLUT("critical186LUT.raw.gz",200);
			else if (connectivity.equals("6/26")) lut = new CriticalPointLUT("critical626LUT.raw.gz",200);
			else if (connectivity.equals("6/18")) lut = new CriticalPointLUT("critical618LUT.raw.gz",200);
			else if (connectivity.equals("wcs")) {
				lut = new CriticalPointLUT("critical66LUT.raw.gz",200);
				checkComposed=true;
			}
			else if (connectivity.equals("wco")) {
				lut = new CriticalPointLUT("criticalNoLUT.raw.gz",200);
				checkComposed=true;
			}
			lut.loadCompressedPattern();
			
			vx = new float[nx][ny][nz];
			vy = new float[nx][ny][nz];
			vz = new float[nx][ny][nz];
			
			order = new float[nx][ny][nz];
			free = new boolean[nx][ny][nz];
			finished = new boolean[nx][ny][nz];
			nonsimple = new boolean[nx][ny][nz];
			intensityTransformed = new float[nx][ny][nz];
			
		} catch (OutOfMemoryError e){
			isWorking = false;
            finalize();
			System.out.println(e.getMessage());
			return;
		}
			
		isWorking = true;

		
		if (debug) MedicUtilPublic.displayMessage("DDiff:initialisation\n");
	}

	/**
	 *  constructor when generating an artificial field
     */
	public DigitalHomeomorphism(float[][][] image_, 
									String transformType_,
									int nx_, int ny_, int nz_,
									float rx_, float ry_, float rz_,
									String connect_,
									ViewJProgressBar bar_) {
		progressBar = bar_;
		
		rx = rx_;
		ry = ry_;
		rz = rz_;
			
		nx = nx_;
		ny = ny_;
		nz = nz_;
			
		intensityImage = image_;
		
		connectivity = connect_;
		
		try {
			tree = new BinaryHeap4D(nx*ny*nz, BinaryHeap4D.MINTREE);
			if (connectivity.equals("26/6")) lut = new CriticalPointLUT("critical266LUT.raw.gz",200);
			else if (connectivity.equals("18/6")) lut = new CriticalPointLUT("critical186LUT.raw.gz",200);
			else if (connectivity.equals("6/26")) lut = new CriticalPointLUT("critical626LUT.raw.gz",200);
			else if (connectivity.equals("6/18")) lut = new CriticalPointLUT("critical618LUT.raw.gz",200);
			else if (connectivity.equals("wcs")) {
				lut = new CriticalPointLUT("critical66LUT.raw.gz",200);
				checkComposed=true;
			}
			else if (connectivity.equals("wco")) {
				lut = new CriticalPointLUT("criticalNoLUT.raw.gz",200);
				checkComposed=true;
			}
			lut.loadCompressedPattern();
			
			createArtificialField(transformType_);
			
			order = new float[nx][ny][nz];
			free = new boolean[nx][ny][nz];
			finished = new boolean[nx][ny][nz];
			nonsimple = new boolean[nx][ny][nz];
			intensityTransformed = new float[nx][ny][nz];
			
		} catch (OutOfMemoryError e){
			isWorking = false;
            finalize();
			System.out.println(e.getMessage());
			return;
		}
			
		isWorking = true;

		
		if (debug) MedicUtilPublic.displayMessage("DDiff:initialisation\n");
	}

	/**
	 *  constructor when just using the criterion
     */
	public DigitalHomeomorphism(String connect_) {
		connectivity = connect_;
		
		try {
			if (connectivity.equals("26/6")) lut = new CriticalPointLUT("critical266LUT.raw.gz",200);
			else if (connectivity.equals("18/6")) lut = new CriticalPointLUT("critical186LUT.raw.gz",200);
			else if (connectivity.equals("6/26")) lut = new CriticalPointLUT("critical626LUT.raw.gz",200);
			else if (connectivity.equals("6/18")) lut = new CriticalPointLUT("critical618LUT.raw.gz",200);
			else if (connectivity.equals("wcs")) {
				lut = new CriticalPointLUT("critical66LUT.raw.gz",200);
				checkComposed=true;
			}
			else if (connectivity.equals("wco")) {
				lut = new CriticalPointLUT("criticalNoLUT.raw.gz",200);
				checkComposed=true;
			}
			lut.loadCompressedPattern();
		} catch (OutOfMemoryError e){
			isWorking = false;
            finalize();
			System.out.println(e.getMessage());
			return;
		}
		isWorking = true;
	}

	final public void cleanUp() {
		order = null;
		free = null;
		finished = null;
		nonsimple = null;
		System.gc();
	}
    
	final public void finalize() {
		vx = null; vy = null; vz = null;
		transformed = null;
		System.gc();
	}
    
    public final boolean isWorking() { return isWorking; }
	public final boolean isCompleted() { return isCompleted; }
    
	public final byte[][][] getTransformed() { return transformed; }
	public final float[][][] getJacobian() { return order; }
	public final float[][][] getIntensityTransformed() { return intensityTransformed; }
	
	public final float[][][] getTransformedFieldX() { return vx; }
	public final float[][][] getTransformedFieldY() { return vy; }
	public final float[][][] getTransformedFieldZ() { return vz; }
	
	public final void setFieldX(float[][][] f) { ux = f; }
	public final void setFieldY(float[][][] f) { uy = f; }
	public final void setFieldZ(float[][][] f) { uz = f; }
	
	public final byte[][][]	exportTransformed() {
		byte[][][] img = new byte[nx][ny][nz];
		
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			img[x][y][z] = transformed[x][y][z];
		}
		return img;
	}
	
	public final String computeFieldDistance() {
		float mean,max,num,maxnorm;
		float norm,dist;
		float valorder,meanorder,varorder,maxorder,numorder;
		float countorder,countdist;
		
		computeTransformResiduals();
		
		mean = 0; max = 0; num = 0; maxnorm = 0;
		meanorder = 0; maxorder = 0; numorder = 0;
		countorder = 0; countdist = 0;
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			norm = ux[x][y][z]*ux[x][y][z]+uy[x][y][z]*uy[x][y][z]+uz[x][y][z]*uz[x][y][z];
			
			if (norm>0) {
				dist = (float)Math.sqrt( (ux[x][y][z]-vx[x][y][z])*(ux[x][y][z]-vx[x][y][z])
								 		+(uy[x][y][z]-vy[x][y][z])*(uy[x][y][z]-vy[x][y][z])
										+(uz[x][y][z]-vz[x][y][z])*(uz[x][y][z]-vz[x][y][z]) );
				mean += dist;
				num++;
				if (dist>0) countdist++;
				if (dist>max) max = dist;
				if (norm>maxnorm) maxnorm = norm;
			}
			valorder = (float)Math.sqrt(order[x][y][z]);
			meanorder += valorder;
			if (valorder>0) countorder++;
			if (valorder>maxorder) maxorder = valorder;
			numorder++;
			
		}
		varorder = 0;
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			valorder = (float)Math.sqrt(order[x][y][z]);
			varorder += (valorder-meanorder/numorder)*(valorder-meanorder/numorder);
		}
		return "Field distance (mean>0|mean|max): "+(mean/num)+" | "+(mean/numorder)+" | "+max+", (max field: "+maxnorm+")\n"
				+"Boundary excluded (mean>0|mean|std|max): "+(meanorder/num)+" | "+(meanorder/numorder)+" | "+Math.sqrt(varorder/numorder)+" | "+maxorder+"\n"
				+"Count of changes (with boundary|without): "+(countdist)+" | "+(countorder)+" %: "+(countdist/numorder)+" | "+(countorder/numorder)+"\n";
	}
		
	public final void computeOriginalTransformedImage() {
		int x,y,z;
		
		for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
			// current transform: original
			vx[x][y][z] = +ux[x][y][z];
			vy[x][y][z] = +uy[x][y][z];
			vz[x][y][z] = +uz[x][y][z];
			// transformed image: no constraint
			transformed[x][y][z] = ImageFunctions.nearestNeighborInterpolation(image,
									x+vx[x][y][z],y+vy[x][y][z],z+vz[x][y][z],nx,ny,nz);
		}
	}
	
	public final void computeOriginalTransformedIntensityImage() {
		int x,y,z;
		
		for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
			// current transform: original
			vx[x][y][z] = +ux[x][y][z];
			vy[x][y][z] = +uy[x][y][z];
			vz[x][y][z] = +uz[x][y][z];
			// transformed image: no constraint
			intensityTransformed[x][y][z] = ImageFunctions.linearInterpolation(intensityImage,
									x+vx[x][y][z],y+vy[x][y][z],z+vz[x][y][z],nx,ny,nz);
		}
	}
	
	public final void computeOriginalFieldJacobian() {
		int x,y,z;
		float[][] du = new float[3][3];
		
		for (x=0;x<nx-1;x++) for (y=0;y<ny-1;y++) for (z=0;z<nz-1;z++) {
			// compute the derivatives
			du[0][0] = ux[x+1][y][z]-ux[x][y][z];
			du[0][1] = ux[x][y+1][z]-ux[x][y][z];
			du[0][2] = ux[x][y][z+1]-ux[x][y][z];
			du[1][0] = uy[x+1][y][z]-uy[x][y][z];
			du[1][1] = uy[x][y+1][z]-uy[x][y][z];
			du[1][2] = uy[x][y][z+1]-uy[x][y][z];
			du[2][0] = uz[x+1][y][z]-uz[x][y][z];
			du[2][1] = uz[x][y+1][z]-uz[x][y][z];
			du[2][2] = uz[x][y][z+1]-uz[x][y][z];
			
			// get the determinant
			order[x][y][z] = du[0][0]*du[1][1]*du[2][2]
								  +du[0][1]*du[1][2]*du[2][0]
								  +du[0][2]*du[1][0]*du[2][1]
								  -du[0][0]*du[1][2]*du[2][1]
								  -du[0][1]*du[1][0]*du[2][2]
								  -du[0][2]*du[1][1]*du[2][0];
								  
		}
	}
	
	public final void computeHomeomorphicTransformAndImage() {
		int x,y,z;
		float val;
		boolean candidates, finish;
		float maxorder,prevmax;
		int itermax = 1000;
		
		for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
			// current transform: zero
			vx[x][y][z] = 0.0f;
			vy[x][y][z] = 0.0f;
			vz[x][y][z] = 0.0f;
			// order by increasing sizes ?
			order[x][y][z] = Numerics.round(ux[x][y][z])*Numerics.round(ux[x][y][z])
							+Numerics.round(uy[x][y][z])*Numerics.round(uy[x][y][z])
							+Numerics.round(uz[x][y][z])*Numerics.round(uz[x][y][z]);
			
			finished[x][y][z] = false;
			nonsimple[x][y][z] = false;
			transformed[x][y][z] = image[x][y][z];
			free[x][y][z] = true;
		}
		
		int iter=0;
		candidates = true;
		finish = false;
		maxorder = 0.0f;
		prevmax = 0.0f;
		while (candidates) {
			candidates = false;
			iter++;
			prevmax = maxorder;
			maxorder = 0.0f;
			for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
				if (updateTransformAt(x,y,z)) {
					candidates = true;
					if (order[x][y][z]>maxorder) maxorder = order[x][y][z];
				}
			}
			
			for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=nz-1;z>=0;z--) {
				if (updateTransformAt(x,y,z)) {
					candidates = true;
					if (order[x][y][z]>maxorder) maxorder = order[x][y][z];
				}
			}
			for (x=0;x<nx;x++) for (y=ny-1;y>=0;y--) for (z=0;z<nz;z++) {
				if (updateTransformAt(x,y,z)) {
					candidates = true;
					if (order[x][y][z]>maxorder) maxorder = order[x][y][z];
				}
			}
			for (x=nx-1;x>=0;x--) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
				if (updateTransformAt(x,y,z)) {
					candidates = true;
					if (order[x][y][z]>maxorder) maxorder = order[x][y][z];
				}
			}
			
			for (x=0;x<nx;x++) for (y=ny-1;y>=0;y--) for (z=nz-1;z>=0;z--) {
				if (updateTransformAt(x,y,z)) {
					candidates = true;
					if (order[x][y][z]>maxorder) maxorder = order[x][y][z];
				}
			}
			for (x=nx-1;x>=0;x--) for (y=ny-1;y>=0;y--) for (z=0;z<nz;z++) {
				if (updateTransformAt(x,y,z)) {
					candidates = true;
					if (order[x][y][z]>maxorder) maxorder = order[x][y][z];
				}
			}
			for (x=nx-1;x>=0;x--) for (y=0;y<ny;y++) for (z=nz-1;z>=0;z--) {
				if (updateTransformAt(x,y,z)) {
					candidates = true;
					if (order[x][y][z]>maxorder) maxorder = order[x][y][z];
				}
			}
			
			for (x=nx-1;x>=0;x--) for (y=ny-1;y>=0;y--) for (z=nz-1;z>=0;z--) {
				if (updateTransformAt(x,y,z)) {
					candidates = true;
					if (order[x][y][z]>maxorder) maxorder = order[x][y][z];
				}
			}
			System.out.println("iteration "+iter+": "+maxorder);
			if (!candidates) System.out.println("no possible update left");
			
			//if (maxorder==prevmax) candidates = false;
			if (iter>itermax) candidates = false;
			if ( ((maxorder==prevmax) || (maxorder<2)) && !finish) {
				finish = true;
				itermax = iter + 5;
			}
			
		}// while there are changes
		
	}
	
	public final void computeFastHomeomorphicTransformAndImage() {
		int x,y,z;
		float val;
		boolean candidates, finish;
		float maxorder,prevmax;
		int itermax = 1000;
		
		for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
			// current transform: zero
			vx[x][y][z] = 0.0f;
			vy[x][y][z] = 0.0f;
			vz[x][y][z] = 0.0f;
			// order by increasing sizes ?
			order[x][y][z] = Numerics.round(ux[x][y][z])*Numerics.round(ux[x][y][z])
							+Numerics.round(uy[x][y][z])*Numerics.round(uy[x][y][z])
							+Numerics.round(uz[x][y][z])*Numerics.round(uz[x][y][z]);
			
			finished[x][y][z] = false;
			nonsimple[x][y][z] = false;
			transformed[x][y][z] = image[x][y][z];
			free[x][y][z] = true;
		}
		
		int iter=0;
		candidates = true;
		finish = false;
		maxorder = 0.0f;
		prevmax = 0.0f;
		while (candidates) {
			candidates = false;
			iter++;
			prevmax = maxorder;
			maxorder = 0.0f;
			for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
				if (updateTransformAt(x,y,z)) {
					candidates = true;
					if (order[x][y][z]>maxorder) maxorder = order[x][y][z];
				}
			}
						
			if (iter>itermax) candidates = false;
			if ( ((maxorder==prevmax) || (maxorder<SQR3)) && !finish) {
				finish = true;
				itermax = iter + 5;
			}
			
			System.out.println("iteration "+iter+": "+maxorder);
		}// while there are changes
		
	}
	
	
	public final void computeTransformResiduals() {
		int x,y,z;
		int xt,yt,zt;
		
		for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
			vx[x][y][z] += ux[x][y][z]-Numerics.round(ux[x][y][z]);
			vy[x][y][z] += uy[x][y][z]-Numerics.round(uy[x][y][z]);
			vz[x][y][z] += uz[x][y][z]-Numerics.round(uz[x][y][z]);
		}	
		
	}
	
	public final void computeOneHomeomorphicTransformAndImage() {
		int x,y,z;
		float val;
		boolean candidates;
		float maxorder,prevmax;
		int itermax = 1000;
		
		for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
			// current transform: zero
			vx[x][y][z] = 0.0f;
			vy[x][y][z] = 0.0f;
			vz[x][y][z] = 0.0f;
			// order by increasing sizes ?
			order[x][y][z] = ux[x][y][z]*ux[x][y][z]
							+uy[x][y][z]*uy[x][y][z]
							+uz[x][y][z]*uz[x][y][z];
			
			finished[x][y][z] = false;
			nonsimple[x][y][z] = false;
			transformed[x][y][z] = image[x][y][z];
			free[x][y][z] = true;
		}
		System.out.print("First iteration\n");
		for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
			updateTransformAt(x,y,z);
		}
		System.out.print("Second iteration\n");
		for (x=1;x<nx-1;x++) for (y=1;y<ny-1;y++) for (z=1;z<nz-1;z++) {
			updateTransformAt(x,y,z);
		}
		
	}
	
	public final void computeRankedHomeomorphicTransformAndImage() {
		int x,y,z;
		float val;
		boolean candidates;
		float maxorder,prevmax;
		int itermax = 1000;
		
		tree.reset();
		tree.setMinTree();
		
		for (x=1;x<nx-1;x++) for (y=1;y<ny-1;y++) for (z=1;z<nz-1;z++) {
			// current transform: zero
			vx[x][y][z] = 0.0f;
			vy[x][y][z] = 0.0f;
			vz[x][y][z] = 0.0f;
			// order by increasing sizes ?
			order[x][y][z] = ux[x][y][z]*ux[x][y][z]
							+uy[x][y][z]*uy[x][y][z]
							+uz[x][y][z]*uz[x][y][z];
			
			finished[x][y][z] = false;
			free[x][y][z] = true;
			nonsimple[x][y][z] = false;
			transformed[x][y][z] = image[x][y][z];
			tree.addValue(order[x][y][z],x,y,z,0);
		}
		
		int iter=0;
		candidates = true;
		maxorder = 0.0f;
		prevmax = 0.0f;
		while (tree.isNotEmpty()) {
			x = tree.getFirstX();
			y = tree.getFirstY();
			z = tree.getFirstZ();
			val = tree.getFirst();
			tree.removeFirst();
		
			//updateTransformAt(x,y,z);
			findTransformAt2(x,y,z);
		}
	}
		
	/**
	 *  following each point
	 */
	private final void updateFullTransformAt(int x, int y, int z) {
		boolean changed = false;
		int xt,yt,zt;
		float dx,dy,dz,norm;
		int xi,yi,zi;
		int i,j,l;
		
		if (finished[x][y][z]) return;
		
		// this point has already been visited: cycle!
		if (!free[x][y][z]) {
			nonsimple[x][y][z] = true;
			free[x][y][z] = true;
			return;
		}
		
		// set as not free; not critical a priori (allows for several trials)
		free[x][y][z] = false;
		nonsimple[x][y][z] = false;
			
		// find all voxels in path
		xt = x + Numerics.round(ux[x][y][z]);
		yt = y + Numerics.round(uy[x][y][z]);
		zt = z + Numerics.round(uz[x][y][z]);
				 
		norm = Numerics.max(Numerics.abs(xt-x),Numerics.abs(yt-y),Numerics.abs(zt-z));
		if (norm>0) {
			dx = (xt-x)/norm;
			dy = (yt-y)/norm;
			dz = (zt-z)/norm;
		} else {
			dx = 0; dy = 0; dz = 0;
		}
		vx[x][y][z] = Numerics.round(dx);
		vy[x][y][z] = Numerics.round(dy);
		vz[x][y][z] = Numerics.round(dz);	
		/*
		vx[x][y][z] = Numerics.round(norm*dx);
		vy[x][y][z] = Numerics.round(norm*dy);
		vz[x][y][z] = Numerics.round(norm*dz);	
		
		//get all voxels along the way
		xi = x; yi = y; zi = z;
		for (int n=1;n<=norm;n++) {
			// only 26-c neighbors ?
			xi = Numerics.round(x+n*dx);
			yi = Numerics.round(y+n*dy);
			zi = Numerics.round(z+n*dz);
			if  ( (xi<=0) || (xi>=nx-1) || (yi<=0) || (yi>=ny-1) || (zi<=0) || (zi>=nz-1) ) {
				vx[x][y][z] = Numerics.round((n-1)*dx);
				vy[x][y][z] = Numerics.round((n-1)*dy);
				vz[x][y][z] = Numerics.round((n-1)*dz);
				break;
			} else {
				updateFullTransformAt(xi,yi,zi);
				if (nonsimple[xi][yi][zi]) {
					vx[x][y][z] = Numerics.round((n-1)*dx);
					vy[x][y][z] = Numerics.round((n-1)*dy);
					vz[x][y][z] = Numerics.round((n-1)*dz);
					break;
				}
			}
		}
		*/
		xt = x + (int)vx[x][y][z];
		yt = y + (int)vy[x][y][z];
		zt = z + (int)vz[x][y][z];
		
		// check whether it's simple or not
		if (simplePoint(transformed, x,y,z, image[xt][yt][zt])) {
			finished[x][y][z] = true;
			// change the label
			transformed[x][y][z] = image[xt][yt][zt];
		} else {
			nonsimple[x][y][z] = true;
			vx[x][y][z] = 0;
			vy[x][y][z] = 0;
			vz[x][y][z] = 0;
		}
		free[x][y][z] = true;
		
		return;
	}
	
	/**
	 *  following each point
	 */
	private final boolean updateTransformAt(int x, int y, int z) {
		boolean changed = false;
		int xt,yt,zt;
		int i,j,l;
		float norm;
		
		//return if at boundary
		if  ((x <= 0) || (x >= nx-1) || (y<=0) || (y>=ny-1) || (z<=0) || (z>=nz-1) ) {
			return false;
		}

		if (!finished[x][y][z]) {
			// check size: if too small we're done
			if (order[x][y][z] == 0) {
				finished[x][y][z] = true;
			} else {
				// find the position in original image
				xt = x + Numerics.round(vx[x][y][z]);
				yt = y + Numerics.round(vy[x][y][z]);
				zt = z + Numerics.round(vz[x][y][z]);
				
				i=0;
				j=0;
				l=0;
				
				// find the next 6-c point to be reached
				if ( (Numerics.abs(ux[x][y][z]-vx[x][y][z])>=Numerics.abs(uy[x][y][z]-vy[x][y][z]) )
				  && (Numerics.abs(ux[x][y][z]-vx[x][y][z])>=Numerics.abs(uz[x][y][z]-vz[x][y][z]) ) ) {
					i = Numerics.sign(ux[x][y][z]-vx[x][y][z]);
				} else if ( (Numerics.abs(uy[x][y][z]-vy[x][y][z])>=Numerics.abs(ux[x][y][z]-vx[x][y][z]))
						 && (Numerics.abs(uy[x][y][z]-vy[x][y][z])>=Numerics.abs(uz[x][y][z]-vz[x][y][z])) ) {
					j = Numerics.sign(uy[x][y][z]-vy[x][y][z]);
				} else {
					l = Numerics.sign(uz[x][y][z]-vz[x][y][z]);
				}
					
				if  ( (xt+i<=0) || (xt+i>=nx-1) || (yt+j<=0) || (yt+j>=ny-1) || (zt+l<=0) || (zt+l>=nz-1) ) {
					// the origin point is out of the image: no need for constraint
					finished[x][y][z] = true;
					order[x][y][z] = 0;					
				} else if ( (i==0) && (j==0) && (l==0) ) {
					// not going any further
					finished[x][y][z] = true;
					order[x][y][z] = 0;
				} else {
					// first, check if the spot is taken by another object ?? 
					// nope, only one arrival point possible
					// check if the point can switch labels
					if (simplePoint(transformed, x,y,z, image[xt+i][yt+j][zt+l])) {
						// change the label
						transformed[x][y][z] = image[xt+i][yt+j][zt+l];
						// update the transform : composition of transforms ??
						vx[x][y][z] += i;
						vy[x][y][z] += j;
						vz[x][y][z] += l;
						// recompute the ordering on the new object
						order[x][y][z] = (Numerics.round(ux[x][y][z])-vx[x][y][z])*(Numerics.round(ux[x][y][z])-vx[x][y][z])
										+(Numerics.round(uy[x][y][z])-vy[x][y][z])*(Numerics.round(uy[x][y][z])-vy[x][y][z])
										+(Numerics.round(uz[x][y][z])-vz[x][y][z])*(Numerics.round(uz[x][y][z])-vz[x][y][z]);
										
						// changes: go for a new loop
						changed = true;
					} else {
						// not possible to change the point: wait until one 
						//of the neighbors is changed
						nonsimple[x][y][z] = true;
						
					}// simple point
				}// boundary
			}// transform size
		}// finished
		return changed;
	}
	
	private final void findTransformAt(int x, int y, int z) {
			
		// find size
		int size = Numerics.abs(Numerics.round(ux[x][y][z]-vx[x][y][z]))
					+Numerics.abs(Numerics.round(uy[x][y][z]-vy[x][y][z]))
					+Numerics.abs(Numerics.round(uz[x][y][z]-vz[x][y][z]));
		
		// not going any further
		if (size==0) {		
			finished[x][y][z] = true;
			return;
		}
					
		int[] i,j,l;
		i = new int[size];
		j = new int[size];
		l = new int[size];
		int dx = 0, dy = 0, dz = 0;
		for (int n=0;n<size;n++) {
			i[n] = 0; j[n] = 0; l[n] = 0;
			// find the next 6-c point to be reached
			if ( (Numerics.abs(ux[x][y][z]-vx[x][y][z]-dx)>=Numerics.abs(uy[x][y][z]-vy[x][y][z]-dy) )
				&& (Numerics.abs(ux[x][y][z]-vx[x][y][z]-dx)>=Numerics.abs(uz[x][y][z]-vz[x][y][z]-dz) ) ) {
					i[n] = +Numerics.sign(ux[x][y][z]-vx[x][y][z]-dx);
			} else if ( (Numerics.abs(uy[x][y][z]-vy[x][y][z]-dy)>=Numerics.abs(ux[x][y][z]-vx[x][y][z]-dx))
				&& (Numerics.abs(uy[x][y][z]-vy[x][y][z]-dy)>=Numerics.abs(uz[x][y][z]-vz[x][y][z]-dz)) ) {
					j[n] = +Numerics.sign(uy[x][y][z]-vy[x][y][z]-dy);
			} else {
					l[n] = +Numerics.sign(uz[x][y][z]-vz[x][y][z]-dz);
			}
			dx += i[n];
			dy += j[n];
			dz += l[n];
		}		
		// we have a list of all the points to be visited
		int xt,yt,zt;
		for (int n=0;n<size;n++) {
			xt = x+(int)vx[x][y][z];
			yt = x+(int)vx[x][y][z];
			zt = x+(int)vx[x][y][z];
			if  ( (xt+i[n]<=0) || (xt+i[n]>=nx-1) || (yt+j[n]<=0) || (yt+j[n]>=ny-1) || (zt+l[n]<=0) || (zt+l[n]>=nz-1) ) {
				// the origin point is out of the image: no constraint
				finished[x][y][z] = true;
				return;
			}
			// check if the current point is finished, otherwise compute it
			// check if the point can switch labels
			if (simplePoint(transformed, x,y,z, image[xt+i[n]][yt+j[n]][zt+l[n]])) {
				// change the label
				transformed[x][y][z] = image[xt+i[n]][yt+j[n]][zt+l[n]];
				// update the transform
				vx[x][y][z] += i[n];
				vy[x][y][z] += j[n];
				vz[x][y][z] += l[n];
				
			} else {
				// not possible to change the point: wait until one 
				//of the neighbors is changed
				nonsimple[x][y][z] = true;
				return;
			}
		}
		return;
	}
	
	private final void findTransformAt2(int x, int y, int z) {
		if  ( (x<0) || (x>nx-1) || (y<0) || (y>ny-1) || (z<0) || (z>nz-1) ) {
			return;
		}
		
		// tag to avoid cycles
		free[x][y][z] = false;
		if  ( (x<=0) || (x>=nx-1) || (y<=0) || (y>=ny-1) || (z<=0) || (z>=nz-1) ) {
			// the target point is out of the image: no constraint
			finished[x][y][z] = true;
			free[x][y][z] = true;
			return;
		}
		
		// find size
		int size = Numerics.abs(Numerics.round(ux[x][y][z]))
					+Numerics.abs(Numerics.round(uy[x][y][z]))
					+Numerics.abs(Numerics.round(uz[x][y][z]));
		
		// not going any further
		if (size==0) {		
			finished[x][y][z] = true;
			free[x][y][z] = true;
			return;
		}
					
		// find the point to be reached
		int xt, yt, zt;
		xt = x+Numerics.round(ux[x][y][z]);
		yt = y+Numerics.round(uy[x][y][z]);
		zt = z+Numerics.round(uz[x][y][z]);
		
		if  ( (xt<=0) || (xt>=nx-1) || (yt<=0) || (yt>=ny-1) || (zt<=0) || (zt>=nz-1) ) {
			// the origin point is out of the image: no constraint
			finished[x][y][z] = true;
			return;
		}
		
		// check if neighbors are set, wait otherwise
		for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
			if (i*i+j*j+l*l>0) {
				if ( (free[x+i][y+j][z+l]) && (!finished[x+i][y+j][z+l]) ) {
					findTransformAt2(x+i,y+j,z+l);
				}
			}
		}
		finished[x][y][z] = true;
		free[x][y][z] = true;
		
		// check if the point can switch labels
		if (simplePoint(transformed, x,y,z, image[xt][yt][zt])) {
			// change the label
			transformed[x][y][z] = image[xt][yt][zt];
			// update the transform
			vx[x][y][z] = xt-x;
			vy[x][y][z] = yt-y;
			vz[x][y][z] = zt-z;
			return;
		} else {
			// not possible to change the point: wait until one 
			//of the neighbors is changed
			vx[x][y][z] = 0;
			vy[x][y][z] = 0;
			vz[x][y][z] = 0;
			nonsimple[x][y][z] = true;
			return;
		}
	}
	
	/**
	 *  following each point
	 */
	private final boolean updateTransformAtOld(int x, int y, int z) {
		boolean changed = false;
		int xt,yt,zt;
		int i,j,l;
		float norm;
		
		if (!finished[x][y][z]) {
			// check size: if too small we're done
			if (order[x][y][z] <= 0.5f) {
				// add remainder to vector ? yes
				//vx[x][y][z] = ux[x][y][z];
				//vy[x][y][z] = uy[x][y][z];
				//vz[x][y][z] = uz[x][y][z];
				finished[x][y][z] = true;
				free[x][y][z] = false;
			} else {
				// find the current position
				xt = x - Numerics.round(vx[x][y][z]);
				yt = y - Numerics.round(vy[x][y][z]);
				zt = z - Numerics.round(vz[x][y][z]);
				
				i=0;j=0;l=0;
				if (connectivity.equals("6/26")) {
					// find the next 6-c point to be reached
					if ( (Numerics.abs(ux[x][y][z]-vx[x][y][z])>=Numerics.abs(uy[x][y][z]-vy[x][y][z]) )
					  && (Numerics.abs(ux[x][y][z]-vx[x][y][z])>=Numerics.abs(uz[x][y][z]-vz[x][y][z]) ) ) {
						i = -Numerics.sign(ux[x][y][z]-vx[x][y][z]);
					} else if ( (Numerics.abs(uy[x][y][z]-vy[x][y][z])>=Numerics.abs(ux[x][y][z]-vx[x][y][z]))
							 && (Numerics.abs(uy[x][y][z]-vy[x][y][z])>=Numerics.abs(uz[x][y][z]-vz[x][y][z])) ) {
						j = -Numerics.sign(uy[x][y][z]-vy[x][y][z]);
					} else {
						l = -Numerics.sign(uz[x][y][z]-vz[x][y][z]);
					}
				} else if (connectivity.equals("26/6")) {
					// find the next 26-c point to be reached
					norm = (float)Math.sqrt(order[x][y][z]);
					
					i = -Numerics.round((ux[x][y][z]-vx[x][y][z])/norm);
					j = -Numerics.round((uy[x][y][z]-vy[x][y][z])/norm);
					l = -Numerics.round((uz[x][y][z]-vz[x][y][z])/norm);
				}
				
				if  ( (xt+i<=0) || (xt+i>=nx-1) || (yt+j<=0) || (yt+j>=ny-1) || (zt+l<=0) || (zt+l>=nz-1) ) {
					// is the next point out of the image ? then it is free
					finished[x][y][z] = true;
				} else if ( (i==0) && (j==0) && (l==0) ) {
					// not going any further
					finished[x][y][z] = true;
					free[xt][yt][zt] = false;
				} else if ( (free[xt+i][yt+j][zt+l]) || (transformed[xt+i][yt+j][zt+l]==image[x][y][z]) ) {
					// first, check if the spot is taken by another object
					
					// check if the point can switch labels
					if (simplePoint(transformed, xt+i,yt+j,zt+l, image[x][y][z])) {
						// change the label
						transformed[xt+i][yt+j][zt+l] = image[x][y][z];
						// occupy next point, free current one
						free[xt+i][yt+j][zt+l] = false;
						free[xt][yt][zt] = true;
						// update the transform
						vx[x][y][z] = vx[x][y][z]-i;
						vy[x][y][z] = vy[x][y][z]-j;
						vz[x][y][z] = vz[x][y][z]-l;
						// recompute the ordering on the new object
						order[x][y][z] = (ux[x][y][z]-vx[x][y][z])*(ux[x][y][z]-vx[x][y][z])
										+(uy[x][y][z]-vy[x][y][z])*(uy[x][y][z]-vy[x][y][z])
										+(uz[x][y][z]-vz[x][y][z])*(uz[x][y][z]-vz[x][y][z]);
										
						// changes: go for a new loop
						nonsimple[x][y][z] = false;
						changed = true;
					} else {
						// not possible to change the point: wait until one 
						//of the neighbors is changed
						nonsimple[x][y][z] = true;
					}// simple point
				}// boundary
			}// transform size
		}// finished
		return changed;
	}
	
	/**
	 *  critical relation detection: groups objects with relations
	 */
    private final boolean simplePoint(byte[][][] img, int x, int y, int z, int k) {
		// is the new object regular ? not the growing object, the original one!!
		//if (!isRegular(x,y,z,k)) return false;
		
		// inside the original object ?
		if (img[x][y][z]==k) return true;
		
		boolean [][][] obj = new boolean[3][3][3];
		
		// does it change the topology of the new object ?
		for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
			
			if(x+i < 0 || x+i >= img.length || y+j < 0 || y+j >= img[0].length || z+l < 0 || z+l >= img[0][0].length){
				//if out of bounds then label is zero
				if (0==k) {
					obj[1+i][1+j][1+l] = true;
				} else {
					obj[1+i][1+j][1+l] = false;
				}
			}else{
				if (img[x+i][y+j][z+l]==k) {
					obj[1+i][1+j][1+l] = true;
				} else {
					obj[1+i][1+j][1+l] = false;
				}
			}
		}
		obj[1][1][1] = true;
		// check well-composedness for new object(s)
		if (checkComposed) if (!ObjectProcessing.isWellComposed(obj,1,1,1)) return false;
		
		if (!lut.get(lut.keyFromPattern(obj,1,1,1))) return false;
		
		// does it change the topology of the object it modifies ?
		for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
			if(x+i < 0 || x+i >= img.length || y+j < 0 || y+j >= img[0].length || z+l < 0 || z+l >= img[0][0].length){
				//if out of bounds then label is zero
				if (0==img[x][y][z]) {
					obj[1+i][1+j][1+l] = true;
				} else {
					obj[1+i][1+j][1+l] = false;
				}
			}else{
				if (img[x+i][y+j][z+l]==img[x][y][z]) {
					obj[1+i][1+j][1+l] = true;
				} else {
					obj[1+i][1+j][1+l] = false;
				}
			}
		}
		obj[1][1][1] = false;
		
		if (!lut.get(lut.keyFromPattern(obj,1,1,1))) return false;
		
		// does it change the topology of a relation between the modified object and its neighbors ?
		int  Nconfiguration = 0;
		byte[] lb = new byte[26];
		for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
			if ( (i*i+j*j+l*l>0) 
				&& (img[x+i][y+j][z+l]!=k) 
				&& (img[x+i][y+j][z+l]!=img[x][y][z]) ) {
				boolean found = false;
				for (int n=0;n<Nconfiguration;n++) 
					if (img[x+i][y+j][z+l]==lb[n]) { found = true; break; }
				
				if (!found) {
					lb[Nconfiguration] = img[x+i][y+j][z+l];
					Nconfiguration++;
				}
			}
		}
		
		// pairs
		for (int n=0;n<Nconfiguration;n++) {
			// in relation with previous object
			for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
				if ( (img[x+i][y+j][z+l]==img[x][y][z])
					|| (img[x+i][y+j][z+l]==lb[n]) ) {
					obj[1+i][1+j][1+l] = true;
				} else {
					obj[1+i][1+j][1+l] = false;
				}
			}
			obj[1][1][1] = false;
			if (!lut.get(lut.keyFromPattern(obj,1,1,1))) return false;
			// in relation with new object
			for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
				if ( (img[x+i][y+j][z+l]==k)
					|| (img[x+i][y+j][z+l]==lb[n]) ) {
					obj[1+i][1+j][1+l] = true;
				} else {
					obj[1+i][1+j][1+l] = false;
				}
			}
			obj[1][1][1] = true;
			if (checkComposed) if (!ObjectProcessing.isWellComposed(obj,1,1,1)) return false;
			if (!lut.get(lut.keyFromPattern(obj,1,1,1))) return false;
		}
		// triplets
		for (int n=0;n<Nconfiguration;n++) {
			for (int m=n+1;m<Nconfiguration;m++) {
				// in relation with previous object
				for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
					if ( (img[x+i][y+j][z+l]==img[x][y][z])
						|| (img[x+i][y+j][z+l]==lb[n])
						|| (img[x+i][y+j][z+l]==lb[m]) ) {
						obj[1+i][1+j][1+l] = true;
					} else {
						obj[1+i][1+j][1+l] = false;
					}
				}
				obj[1][1][1] = false;
				if (!lut.get(lut.keyFromPattern(obj,1,1,1))) return false;
				// in relation with new object
				for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
					if ( (img[x+i][y+j][z+l]==k)
						|| (img[x+i][y+j][z+l]==lb[n]) 
						|| (img[x+i][y+j][z+l]==lb[m]) ) {
						obj[1+i][1+j][1+l] = true;
					} else {
						obj[1+i][1+j][1+l] = false;
					}
				}
				obj[1][1][1] = true;
				if (checkComposed) if (!ObjectProcessing.isWellComposed(obj,1,1,1)) return false;
				if (!lut.get(lut.keyFromPattern(obj,1,1,1))) return false;
			}
		}
		// else, it works
		return true;
    }

	public final void createArtificialField(String type) {
		int x,y,z;
		
		ux = new float[nx][ny][nz];
		uy = new float[nx][ny][nz];
		uz = new float[nx][ny][nz];
		vx = new float[nx][ny][nz];
		vy = new float[nx][ny][nz];
		vz = new float[nx][ny][nz];
		
		if (type.equals("translation1")) {
			for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
				// current transform: zero
				ux[x][y][z] = -1.5f;
				uy[x][y][z] = 0.0f;
				uz[x][y][z] = 0.0f;
			}
		} else if (type.equals("translation2")) {
			for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
				// current transform: zero
				ux[x][y][z] = -3.5f;
				uy[x][y][z] = 2.1f;
				uz[x][y][z] = 1.7f;
			}
		} else if (type.equals("rotation1")) {
			RotationMatrix R = new RotationMatrix();
			R.setParameters(0,0,0);
			float[][] rot = R.getMatrix();
			rot[0][0] =  0.996194698f;
			rot[0][1] = -0.087155743f;
			rot[1][0] =  0.087155743f;
			rot[1][1] =  0.996194698f;
			for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
				// current transform: zero
				ux[x][y][z] = rot[0][0]*(x-nx/2)+rot[0][1]*(y-ny/2)+rot[0][2]*(z-nz/2) + nx/2 - x;
				uy[x][y][z] = rot[1][0]*(x-nx/2)+rot[1][1]*(y-ny/2)+rot[1][2]*(z-nz/2) + ny/2 - y;
				uz[x][y][z] = rot[2][0]*(x-nx/2)+rot[2][1]*(y-ny/2)+rot[2][2]*(z-nz/2) + nz/2 - z;
			}	
			R.setMatrix(rot);
			float[] angles = R.computeEulerAngles();
			System.out.print("Rotation angles (Euler): "+angles[0]+", "+angles[1]+", "+angles[2]+"\n");
			MedicUtilPublic.displayMessage("Rotation angles (Euler): "+angles[0]+", "+angles[1]+", "+angles[2]+"\n");
		} else if (type.equals("rotation2")) {
			RotationMatrix R = new RotationMatrix();
			R.setParameters(0,0,0);
			float[][] rot = R.getMatrix();
			rot[0][0] =  (float)Math.sqrt(2)/2.0f;
			rot[0][1] = -(float)Math.sqrt(2)/2.0f;
			rot[1][0] =  (float)Math.sqrt(2)/2.0f;
			rot[1][1] =  (float)Math.sqrt(2)/2.0f;
			for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
				// current transform: zero
				ux[x][y][z] = rot[0][0]*(x-nx/2)+rot[0][1]*(y-ny/2)+rot[0][2]*(z-nz/2) + nx/2 - x;
				uy[x][y][z] = rot[1][0]*(x-nx/2)+rot[1][1]*(y-ny/2)+rot[1][2]*(z-nz/2) + ny/2 - y;
				uz[x][y][z] = rot[2][0]*(x-nx/2)+rot[2][1]*(y-ny/2)+rot[2][2]*(z-nz/2) + nz/2 - z;
			}	
			R.setMatrix(rot);
			float[] angles = R.computeEulerAngles();
			System.out.print("Rotation angles (Euler): "+angles[0]+", "+angles[1]+", "+angles[2]+"\n");
			MedicUtilPublic.displayMessage("Rotation angles (Euler): "+angles[0]+", "+angles[1]+", "+angles[2]+"\n");
		} else if (type.equals("rotation3")) {
			RotationMatrix R = new RotationMatrix();
			R.setParameters(0,0,0);
			float[][] rot = R.getMatrix();
			rot[0][0] =  0.965925826f;
			rot[0][1] = -0.258819045f;
			rot[1][0] =  0.258819045f;
			rot[1][1] =  0.965925826f;
			for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
				// current transform: zero
				ux[x][y][z] = rot[0][0]*(x-nx/2)+rot[0][1]*(y-ny/2)+rot[0][2]*(z-nz/2) + nx/2 - x;
				uy[x][y][z] = rot[1][0]*(x-nx/2)+rot[1][1]*(y-ny/2)+rot[1][2]*(z-nz/2) + ny/2 - y;
				uz[x][y][z] = rot[2][0]*(x-nx/2)+rot[2][1]*(y-ny/2)+rot[2][2]*(z-nz/2) + nz/2 - z;
			}	
			R.setMatrix(rot);
			float[] angles = R.computeEulerAngles();
			System.out.print("Rotation angles (Euler): "+angles[0]+", "+angles[1]+", "+angles[2]+"\n");
			MedicUtilPublic.displayMessage("Rotation angles (Euler): "+angles[0]+", "+angles[1]+", "+angles[2]+"\n");
		} else if (type.equals("scaling1")) {
			float[] scale = new float[3];
			scale[0] = 0.95f; scale[1] = 0.95f; scale[2] = 1.0f;
			float[] trans = new float[3];
			trans[0] = 0.0f; trans[1] = 0.0f; trans[2] = 0.0f;
			RotationMatrix R = new RotationMatrix();
			R.setParameters(0,0,0);
			float[][] rot = R.getMatrix();
			
			for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
				// current transform: zero
				ux[x][y][z] = scale[0]*(rot[0][0]*(x-nx/2)+rot[0][1]*(y-ny/2)+rot[0][2]*(z-nz/2)) + trans[0] + nx/2 - x;
				uy[x][y][z] = scale[1]*(rot[1][0]*(x-nx/2)+rot[1][1]*(y-ny/2)+rot[1][2]*(z-nz/2)) + trans[1] + ny/2 - y;
				uz[x][y][z] = scale[2]*(rot[2][0]*(x-nx/2)+rot[2][1]*(y-ny/2)+rot[2][2]*(z-nz/2)) + trans[2] + nz/2 - z;
			}	
		} else if (type.equals("scaling2")) {
			float[] scale = new float[3];
			scale[0] = 0.8f; scale[1] = 0.8f; scale[2] = 1.0f;
			float[] trans = new float[3];
			trans[0] = 0.0f; trans[1] = 0.0f; trans[2] = 0.0f;
			RotationMatrix R = new RotationMatrix();
			R.setParameters(0,0,0);
			float[][] rot = R.getMatrix();
			
			for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
				// current transform: zero
				ux[x][y][z] = scale[0]*(rot[0][0]*(x-nx/2)+rot[0][1]*(y-ny/2)+rot[0][2]*(z-nz/2)) + trans[0] + nx/2 - x;
				uy[x][y][z] = scale[1]*(rot[1][0]*(x-nx/2)+rot[1][1]*(y-ny/2)+rot[1][2]*(z-nz/2)) + trans[1] + ny/2 - y;
				uz[x][y][z] = scale[2]*(rot[2][0]*(x-nx/2)+rot[2][1]*(y-ny/2)+rot[2][2]*(z-nz/2)) + trans[2] + nz/2 - z;
			}	
		} else if (type.equals("affine1")) {
			float[] scale = new float[3];
			scale[0] = 0.8f; scale[1] = 0.8f; scale[2] = 1.0f;
			float[] trans = new float[3];
			trans[0] = 0.0f; trans[1] = 0.0f; trans[2] = 0.0f;
			RotationMatrix R = new RotationMatrix();
			R.setParameters(0,0,0);
			float[][] rot = R.getMatrix();
			
			for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
				// current transform: zero
				ux[x][y][z] = scale[0]*(rot[0][0]*(x-nx/2)+rot[0][1]*(y-ny/2)+rot[0][2]*(z-nz/2)) + trans[0] + nx/2 - x;
				uy[x][y][z] = scale[1]*(rot[1][0]*(x-nx/2)+rot[1][1]*(y-ny/2)+rot[1][2]*(z-nz/2)) + trans[1] + ny/2 - y;
				uz[x][y][z] = scale[2]*(rot[2][0]*(x-nx/2)+rot[2][1]*(y-ny/2)+rot[2][2]*(z-nz/2)) + trans[2] + nz/2 - z;
			}	
		}
	}
	
	public final void createAffineField(float[][] trans) {
		int x,y,z;
		
		ux = new float[nx][ny][nz];
		uy = new float[nx][ny][nz];
		uz = new float[nx][ny][nz];
		
		for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
			ux[x][y][z] = trans[0][0]*x + trans[0][1]*y + trans[0][2]*z + trans[0][3];
			uy[x][y][z] = trans[1][0]*x + trans[1][1]*y + trans[1][2]*z + trans[1][3];
			uz[x][y][z] = trans[2][0]*x + trans[2][1]*y + trans[2][2]*z + trans[2][3];
		}
	
	}
}
