package edu.jhmi.rad.medic.methods;
 import java.io.*;
import java.util.*;
import gov.nih.mipav.view.*;
import gov.nih.mipav.model.structures.jama.*;

import edu.jhmi.rad.medic.libraries.*;
import edu.jhmi.rad.medic.utilities.*;

/**
 *
 *  This algorithm handles all sorts of Fuzzy C-Means operations
 *  for multichannel 3D data.
 *
 *	@version    November 2004
 *	@author     Pierre-Louis Bazin
 *		
 *
 */
 
public class MultichannelFCM {
		
	// numerical quantities
	private static final	float   INF=1e30f;
	private static final	float   ZERO=1e-30f;
	
	// data buffers
	private 	float[][][][]		image;  			// original image (4D)
	private 	float[][][][]	    mems;				// membership function (3D)
	private 	float[][]			centroids;			// class centroids (4D)
	private 	float[][][][]		field;  			// inhomogeneity field (4D)
	private 	float[][][][]		edges;  			// edge map (4D)
	private 	boolean[][][][]		mask;   			// image mask: true for data points (4d)
	//private		float[][]			transform;			// transform from common space to original space
	private static	int				nx,ny,nz,nt;   		// image dimensions
	private static	float			rx,ry,rz;   		// image resolutions
	
	// parameters
	private 	int 		clusters;
	private 	int 		classes;
	private 	int 		members;
	private 	float		smoothing;
    private 	float		fuzziness;
    private		PowerTable	power, invpower;
	private 	float[][] 	positiveScaling;
    private 	float[][] 	negativeScaling;
    private		float		outlier;
    private		boolean		useOutliers;
	private		boolean		useEdges;
	private		boolean[]	correctField;
	
	// options for masking multiple images
	private static final int	ONE=1;
	private static final int	ALL=2;
	private		int			maskingMode = ONE;
	
	// otpions for scaling the images / classes
	private static final int	INTENSITY=1;
	private static final int	ENERGY=2;
	private static final int	VARIANCE=3;
	private static final int	CENTROID=4;
	private		int			scalingMode = INTENSITY;
	
	// transformation
	private		boolean		useRegistration;
	private		float[]		maskVal;
	private		float[][][]	transforms;		// array of 3x4 transform matrices for each image
	// faster computations: offline registration
	
	// computation variables
	private		float[]         prev;
	private		float[][]		wt;			// polynomials for Cubic Lagrangian
	private		float[]			Imax, Imin; // image boundaries for interpolation
        
	// computation flags
	private 	boolean 		isWorking;
	private 	boolean 		isCompleted;
	
	// for debug and display
	ViewUserInterface			UI;
    ViewJProgressBar            progressBar;
	static final boolean		debug=true;
	static boolean		verbose=true;
	/**
	 *  constructor
	 */
	public MultichannelFCM(float[][][][] image_, boolean [][][][] mask_, 
					int nx_, int ny_, int nz_, int nt_,
					float rx_, float ry_, float rz_,
					int nclasses_, int nclusters_,
					float smoothing_, float fuzzy_,
					float[] scal_,
                    boolean useOut_, float outlierRatio_,
					boolean useReg_, float[] maskVal_, String scalingMode_,
					float[] min_, float[] max_,
					ViewUserInterface UI_, ViewJProgressBar bar_) {
						
		image = image_;
		mask = mask_;
		nx = nx_;
		ny = ny_;
		nz = nz_;
		nt = nt_;
		rx = rx_;
		ry = ry_;
		rz = rz_;
		classes = nclasses_;
		clusters = nclusters_;
		smoothing = smoothing_;
        fuzziness = fuzzy_;
        useOutliers = useOut_;
        outlier = outlierRatio_;
		
		useEdges = false;
		
		useRegistration = useReg_;
		maskVal = maskVal_;
		/*
		if (maskMode_.equals("all")) maskingMode = ALL;
		else maskingMode = ONE;
		*/
		if (scalingMode_.equals("energy")) scalingMode = ENERGY;
		else if (scalingMode_.equals("variance")) scalingMode = VARIANCE;
		else if (scalingMode_.equals("centroids")) scalingMode = CENTROID;
		else scalingMode = INTENSITY;
		
		if (useRegistration) members = classes+1;
		else members = classes;
		
		Imin = new float[nt];
		Imax = new float[nt];
		for (int t=0;t<nt;t++) Imin[t] = min_[t];
		for (int t=0;t<nt;t++) Imax[t] = max_[t];
		
		UI = UI_;
        progressBar = bar_;
		if(progressBar==null)verbose=false;
		// init all the arrays
		try {
			mems = new float[nx][ny][nz][members];
			transforms = new float[nt][3][4];
			centroids = new float[nt][clusters];
			prev = new float[members];
			correctField = new boolean[nt];
			field = new float[nt][][][];
			wt = ImageFunctions.setup3DCubicLagrangianInterpolation();
			positiveScaling = new float[nt][members];
			negativeScaling = new float[nt][members];
			if (fuzziness!=2) {
				power = new PowerTable(0.0f , 1.0f , 0.000001f , fuzziness );
				invpower = new PowerTable(0.0f , 1.0f , 0.000001f , 1.0f/(1.0f-fuzziness) );
			} else {
				power = null;
				invpower = null;
			}
		} catch (OutOfMemoryError e){
			isWorking = false;
            finalize();
			System.out.println(e.getMessage());
			return;
		}
		isWorking = true;

		// init values
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			for (int k=0;k<members;k++) {
				mems[x][y][z][k] = 0.0f;
			}
		}
		for (int t=0;t<nt;t++) {
			if (useRegistration) {
				for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
					if (!mask[t][x][y][z]) image[t][x][y][z] = maskVal[t];
				}
			}
			for (int k=0;k<clusters;k++) {
				centroids[t][k] = 0.0f;
			}
			for (int k=0;k<members;k++) {
				if (scalingMode==INTENSITY) {
					negativeScaling[t][k] = scal_[t];
					positiveScaling[t][k] = scal_[t];
				} else if (scalingMode==ENERGY) {
					negativeScaling[t][k] = 1.0f;
					positiveScaling[t][k] = 1.0f;
				} else if (scalingMode==VARIANCE) {
					negativeScaling[t][k] = scal_[t]/clusters;
					positiveScaling[t][k] = scal_[t]/clusters;
				} else if (scalingMode==CENTROID) {
					negativeScaling[t][k] = 0.5f*scal_[t]/clusters;
					positiveScaling[t][k] = 0.5f*scal_[t]/clusters;
				}
			}
			for (int i=0;i<3;i++) {
				for (int j=0;j<4;j++) transforms[t][i][j] = 0.0f;
				transforms[t][i][i] = 1.0f;
			}
			correctField[t] = false;
		}
		if (debug) MedicUtilPublic.displayMessage("initialisation\n");
	}

	final public void finalize() {
		mems = null;
		transforms = null;
		prev = null;
		System.gc();
	}
    
    public final void setImage(int t, float[][][] img) {
        image[t] = img;
    }

    public final void setMask(int t, boolean[][][] msk) {
        mask[t] = msk;
    }

    public final void setTransform(int t, float[][] trans) {
		transforms[t] = trans;
    }
    public final float[][] getTransform(int t) {
		return transforms[t];
    }
    
    public final void setMRF(float smooth) {
        smoothing = smooth;
    }

	public final boolean isWorking() { return isWorking; }
	public final boolean isCompleted() { return isCompleted; }
    
    /** accessor for computed data */ 
    public final float[][][][] getMemberships() { return mems; }
    /** accessor for computed data */ 
    public final float[] getCentroids(int t) { return centroids[t]; }
	/** accessor for computed data */ 
    public final void setCentroids(int t, float[] cent) { centroids[t] = cent; }

	final public void importCentroids(int t, float[] cent) {
		for (int k=0;k<clusters;k++) centroids[t][k] = cent[k];
	}// importCentroids
	
    /** add inhomogeneity correction */
    public final void addInhomogeneityCorrection(int t, float[][][] field_) {
        field[t] = field_;
        correctField[t] = true;
    }
    /** add edge parameter */
    public final void addEdgeMap(float[][][][] edges_) {
        edges = edges_;
        useEdges = true;
    }
		
    /** 
	 *  compute the RFCM membership functions given the centroids 
	 *  with multichannel images
	 *	the algorithm only uses images 1 through Nt
	 *
	 */
    final public float computeMemberships(int Nt) {
        float distance;
        int x,y,z,k,m,s,t;
        int progress, mod;
        long inner_loop_time;
        float den,num;
        float neighbors, ngb;
        float dist;
        boolean	isUsed;
		float val;
        
        if (fuzziness!=2) return computeGeneralMemberships(Nt);
		
		distance = 0.0f;
		progress = 0;
        mod = nx*ny*nz/20; // mod is 1 percent of length

        inner_loop_time = System.currentTimeMillis();
        for (x=1;x<nx-1;x++) for (y=1;y<ny-1;y++) for (z=1;z<nz-1;z++) {
			progress++;
			if ((verbose) && (progress%mod==0) )
                progressBar.updateValue((int)Math.round( (float)progress/(float)mod*5.0f),false);

			// check if real value first
			isUsed = true;
			if (!useRegistration) {
				if (maskingMode==ONE) {
					isUsed = false;
					for (t=0;t<Nt;t++) if (mask[t][x][y][z]) isUsed = true;
				} else if (maskingMode==ALL) {
					isUsed = true;
					for (t=0;t<Nt;t++) if (!mask[t][x][y][z]) isUsed = false;
				}	
			}
				
			if (isUsed) {
				den = 0;
				// remember the previous values
				for (k=0;k<members;k++) prev[k] = mems[x][y][z][k];
				
				for (k=0;k<members;k++) {
					// numerator
					num = 0; 
					
					for (t=0;t<Nt;t++) {
						if ( (useRegistration) || (mask[t][x][y][z]) ) {
							// data term
							if (k<clusters) {
								if (correctField[t]) val = (field[t][x][y][z]*image[t][x][y][z]-centroids[t][k]);
								else val = (image[t][x][y][z]-centroids[t][k]);
								if (val>0) num += (val*val)/(positiveScaling[t][k]*positiveScaling[t][k]);
								else num += (val*val)/(negativeScaling[t][k]*negativeScaling[t][k]);
							} else if (k<classes) {
								num += outlier*outlier;
							} else {
								if (correctField[t]) val = (field[t][x][y][z]*image[t][x][y][z]-maskVal[t]);
								else val = (image[t][x][y][z]-maskVal[t]);
								if (val>0) num += (val*val)/(positiveScaling[t][k]*positiveScaling[t][k]);
								else num += (val*val)/(negativeScaling[t][k]*negativeScaling[t][k]);
							}
						} else {
							num += INF;
						}
					}
					// normalize over the images
					num = num/(float)Nt;
					
					// spatial smoothing
					if (smoothing > 0.0f) { 
						ngb = 0.0f;  
						neighbors = 0.0f;
						// case by case	: X+
						for (m=0;m<members;m++) if (m!=k) {
							if (useEdges) ngb += edges[0][x][y][z]*mems[x+1][y][z][m]*mems[x+1][y][z][m];
							else ngb += mems[x+1][y][z][m]*mems[x+1][y][z][m];
							neighbors ++;
						}
						// case by case	: X-
						for (m=0;m<members;m++) if (m!=k) {
							if (useEdges) ngb += edges[0][x-1][y][z]*mems[x-1][y][z][m]*mems[x-1][y][z][m];
							else ngb += mems[x-1][y][z][m]*mems[x-1][y][z][m];
							neighbors ++;
						}
						// case by case	: Y+
						for (m=0;m<members;m++) if (m!=k) {
							if (useEdges) ngb += edges[1][x][y][z]*mems[x][y+1][z][m]*mems[x][y+1][z][m];
							else ngb += mems[x][y+1][z][m]*mems[x][y+1][z][m];
							neighbors ++;
						}
						// case by case	: Y-
						for (m=0;m<members;m++) if (m!=k) {
							if (useEdges) ngb += edges[1][x][y-1][z]*mems[x][y-1][z][m]*mems[x][y-1][z][m];
							else ngb += mems[x][y-1][z][m]*mems[x][y-1][z][m];
							neighbors ++;
						}
						// case by case	: Z+
						for (m=0;m<members;m++) if (m!=k) {
							if (useEdges) ngb += edges[2][x][y][z]*mems[x][y][z+1][m]*mems[x][y][z+1][m];
							else ngb += mems[x][y][z+1][m]*mems[x][y][z+1][m];
							neighbors ++;
						}
						// case by case	: Z-
						for (m=0;m<members;m++) if (m!=k) {
							if (useEdges) ngb += edges[2][x][y][z-1]*mems[x][y][z-1][m]*mems[x][y][z-1][m];
							else ngb += mems[x][y][z-1][m]*mems[x][y][z-1][m];
							neighbors ++;
						}
						if (neighbors>0.0) num = num + smoothing*ngb/neighbors;
					}
					
					// invert after adding all t
					if (num>ZERO) num = 1.0f/num;
					else num = INF;
	
					mems[x][y][z][k] = num;
					den += num;
				}
	
				// normalize
				for (k=0;k<members;k++) {
					mems[x][y][z][k] = mems[x][y][z][k]/den;
	
					// compute the maximum distance
					dist = Math.abs(mems[x][y][z][k]-prev[k]);
					if (dist > distance) distance = dist; 
				}
			} else {
				for (k=0;k<members;k++) {
					mems[x][y][z][k] = 0.0f;;
				}
			}
		}
        if (debug) System.out.print("inner loop time: (milliseconds): " + (System.currentTimeMillis()-inner_loop_time) +"\n"); 

        return distance;
    } // computeMemberships
    
    /** 
	 *  compute the RFCM membership functions given the centroids 
	 *  with multichannel images
	 *	the algorithm only uses images 1 through Nt
	 *
	 */
    final public float computeGeneralMemberships(int Nt) {
        float distance;
        int x,y,z,k,m,s,t;
        int progress, mod;
        long inner_loop_time;
        float den,num;
        float neighbors, ngb;
        float dist;
        boolean	isUsed;
		float val;
        
        distance = 0.0f;
		progress = 0;
        mod = nx*ny*nz/20; // mod is 1 percent of length

        inner_loop_time = System.currentTimeMillis();
        for (x=1;x<nx-1;x++) for (y=1;y<ny-1;y++) for (z=1;z<nz-1;z++) {
			progress++;
			if ((verbose) && (progress%mod==0) )
                progressBar.updateValue((int)Math.round( (float)progress/(float)mod*5.0f),false);

			// check if real value first
			isUsed = true;
			if (!useRegistration) {
				if (maskingMode==ONE) {
					isUsed = false;
					for (t=0;t<Nt;t++) if (mask[t][x][y][z]) isUsed = true;
				} else if (maskingMode==ALL) {
					isUsed = true;
					for (t=0;t<Nt;t++) if (!mask[t][x][y][z]) isUsed = false;
				}	
			}
				
			if (isUsed) {
				den = 0;
				// remember the previous values
				for (k=0;k<members;k++) prev[k] = mems[x][y][z][k];
				
				for (k=0;k<members;k++) {
					// numerator
					num = 0; 
					
					for (t=0;t<Nt;t++) {
						if ( (useRegistration) || (mask[t][x][y][z]) ) {
							// data term
							if (k<clusters) {
								if (correctField[t]) val = (field[t][x][y][z]*image[t][x][y][z]-centroids[t][k]);
								else val = (image[t][x][y][z]-centroids[t][k]);
								if (val>0) num += (val*val)/(positiveScaling[t][k]*positiveScaling[t][k]);
								else num += (val*val)/(negativeScaling[t][k]*negativeScaling[t][k]);
							} else if (k<classes) {
								num += outlier*outlier;
							} else {
								if (correctField[t]) val = (field[t][x][y][z]*image[t][x][y][z]-maskVal[t]);
								else val = (image[t][x][y][z]-maskVal[t]);
								if (val>0) num += (val*val)/(positiveScaling[t][k]*positiveScaling[t][k]);
								else num += (val*val)/(negativeScaling[t][k]*negativeScaling[t][k]);
							}
						} else {
							num += INF;
						}
					}
					// normalize over the images
					num = num/(float)Nt;
					
					// spatial smoothing
					if (smoothing > 0.0f) { 
						ngb = 0.0f;  
						neighbors = 0.0f;
						// case by case	: X+
						for (m=0;m<members;m++) if (m!=k) {
							if (useEdges) ngb += edges[0][x][y][z]*power.lookup(mems[x+1][y][z][m],fuzziness);
							else ngb += power.lookup(mems[x+1][y][z][m],fuzziness);
							neighbors ++;
						}
						// case by case	: X-
						for (m=0;m<members;m++) if (m!=k) {
							if (useEdges) ngb += edges[0][x-1][y][z]*power.lookup(mems[x-1][y][z][m],fuzziness);
							else ngb += power.lookup(mems[x-1][y][z][m],fuzziness);
							neighbors ++;
						}
						// case by case	: Y+
						for (m=0;m<members;m++) if (m!=k) {
							if (useEdges) ngb += edges[1][x][y][z]*power.lookup(mems[x][y+1][z][m],fuzziness);
							else ngb += power.lookup(mems[x][y+1][z][m],fuzziness);
							neighbors ++;
						}
						// case by case	: Y-
						for (m=0;m<members;m++) if (m!=k) {
							if (useEdges) ngb += edges[1][x][y-1][z]*power.lookup(mems[x][y-1][z][m],fuzziness);
							else ngb += power.lookup(mems[x][y-1][z][m],fuzziness);
							neighbors ++;
						}
						// case by case	: Z+
						for (m=0;m<members;m++) if (m!=k) {
							if (useEdges) ngb += edges[2][x][y][z]*power.lookup(mems[x][y][z+1][m],fuzziness);
							else ngb += power.lookup(mems[x][y][z+1][m],fuzziness);
							neighbors ++;
						}
						// case by case	: Z-
						for (m=0;m<members;m++) if (m!=k) {
							if (useEdges) ngb += edges[2][x][y][z-1]*power.lookup(mems[x][y][z-1][m],fuzziness);
							else ngb += power.lookup(mems[x][y][z-1][m],fuzziness);
							neighbors ++;
						}
						if (neighbors>0.0) num = num + smoothing*ngb/neighbors;
					}
					
					// invert after adding all t
					if (num>ZERO) num = (float)power.lookup(num,1.0f/(1.0f-fuzziness) );
					else num = INF;
	
					mems[x][y][z][k] = num;
					den += num;
				}
	
				// normalize
				for (k=0;k<members;k++) {
					mems[x][y][z][k] = mems[x][y][z][k]/den;
	
					// compute the maximum distance
					dist = Math.abs(mems[x][y][z][k]-prev[k]);
					if (dist > distance) distance = dist; 
				}
			} else {
				for (k=0;k<members;k++) {
					mems[x][y][z][k] = 0.0f;;
				}
			}
		}
        if (debug) System.out.print("inner loop time: (milliseconds): " + (System.currentTimeMillis()-inner_loop_time) +"\n"); 

        return distance;
    } // computeMemberships
    
    /** 
	 *  compute the RFCM membership functions given the centroids 
	 *  with multichannel images and transformed images
	 *	the algorithm only uses images 1 through Nt
	 */
    final public float computeTransformedMemberships(int Nt) {
        float distance;
        int x,y,z,k,m,s,t;
        int progress, mod;
        long inner_loop_time;
        float den,num;
        float neighbors, ngb;
        float dist,imgT;
        float xT,yT,zT;
        float xTp,yTp,zTp;
        float xTm,yTm,zTm;
		float val;
        
        if (fuzziness!=2) return computeGeneralTransformedMemberships(Nt);
		
		distance = 0.0f;
		progress = 0;
        mod = nx*ny*nz/20; // mod is 1 percent of length

        inner_loop_time = System.currentTimeMillis();
        for (x=1;x<nx-1;x++) for (y=1;y<ny-1;y++) for (z=1;z<nz-1;z++) {
			progress++;
			if ((verbose) && (progress%mod==0) )
                progressBar.updateValue((int)Math.round( (float)progress/(float)mod*5.0f),false);

			den = 0;
			// remember the previous values
			for (k=0;k<members;k++) prev[k] = mems[x][y][z][k];
			
			for (k=0;k<members;k++) {
				// numerator
				num = 0; 
				
				for (t=0;t<Nt;t++) {
					// compute the local position: X' = RX+T
					xT = (transforms[t][0][0]*x*rx + transforms[t][0][1]*y*ry + transforms[t][0][2]*z*rz + transforms[t][0][3])/rx;
					yT = (transforms[t][1][0]*x*rx + transforms[t][1][1]*y*ry + transforms[t][1][2]*z*rz + transforms[t][1][3])/ry;
					zT = (transforms[t][2][0]*x*rx + transforms[t][2][1]*y*ry + transforms[t][2][2]*z*rz + transforms[t][2][3])/rz;
					// neighbors
					xTp = (transforms[t][0][0]*(x+1)*rx + transforms[t][0][1]*y*ry     + transforms[t][0][2]*z*rz     + transforms[t][0][3])/rx;
					yTp = (transforms[t][1][0]*x*rx     + transforms[t][1][1]*(y+1)*ry + transforms[t][1][2]*z*rz     + transforms[t][1][3])/ry;
					zTp = (transforms[t][2][0]*x*rx     + transforms[t][2][1]*y*ry     + transforms[t][2][2]*(z+1)*rz + transforms[t][2][3])/rz;
					xTm = (transforms[t][0][0]*(x-1)*rx + transforms[t][0][1]*y*ry     + transforms[t][0][2]*z*rz     + transforms[t][0][3])/rx;
					yTm = (transforms[t][1][0]*x*rx     + transforms[t][1][1]*(y-1)*ry + transforms[t][1][2]*z*rz     + transforms[t][1][3])/ry;
					zTm = (transforms[t][2][0]*x*rx     + transforms[t][2][1]*y*ry     + transforms[t][2][2]*(z-1)*rz + transforms[t][2][3])/rz;

					if ((useRegistration) || (ImageFunctions.maskInterpolation(mask[t],xT,yT,zT,nx,ny,nz))) {
						// pre-compute interpolated values
						imgT = ImageFunctions.linearInterpolation(image[t],maskVal[t],xT,yT,zT,nx,ny,nz);

						// data term
						if (k<clusters) {
							if (correctField[t]) val = (field[t][x][y][z]*imgT-centroids[t][k]);
							else val = (imgT-centroids[t][k]);
							if (val>0) num += (val*val)/(positiveScaling[t][k]*positiveScaling[t][k]);
							else num += (val*val)/(negativeScaling[t][k]*negativeScaling[t][k]);
						} else if (k<classes) {
							num += outlier*outlier;
						} else {
							if (correctField[t]) val = (field[t][x][y][z]*imgT-maskVal[t]);
							else val = (imgT-maskVal[t]);
							if (val>0) num += (val*val)/(positiveScaling[t][k]*positiveScaling[t][k]);
							else num += (val*val)/(negativeScaling[t][k]*negativeScaling[t][k]);
						}
					} else {
						num += INF;
					}
				}
				
				// normalize over the images
				num = num/(float)Nt;
				
				// spatial smoothing
				if (smoothing > 0.0f) { 
					ngb = 0.0f;  
					neighbors = 0.0f;
					// case by case	: X+
					for (m=0;m<members;m++) if (m!=k) {
						if (useEdges) ngb += edges[0][x][y][z]*mems[x+1][y][z][m]*mems[x+1][y][z][m];
						else ngb += mems[x+1][y][z][m]*mems[x+1][y][z][m];
						neighbors ++;
					}
					// case by case	: X-
					for (m=0;m<members;m++) if (m!=k) {
						if (useEdges) ngb += edges[0][x-1][y][z]*mems[x-1][y][z][m]*mems[x-1][y][z][m];
						else ngb += mems[x-1][y][z][m]*mems[x-1][y][z][m];
						neighbors ++;
					}
					// case by case	: Y+
					for (m=0;m<members;m++) if (m!=k) {
						if (useEdges) ngb += edges[1][x][y][z]*mems[x][y+1][z][m]*mems[x][y+1][z][m];
						else ngb += mems[x][y+1][z][m]*mems[x][y+1][z][m];
						neighbors ++;
					}
					// case by case	: Y-
					for (m=0;m<members;m++) if (m!=k) {
						if (useEdges) ngb += edges[1][x][y-1][z]*mems[x][y-1][z][m]*mems[x][y-1][z][m];
						else ngb += mems[x][y-1][z][m]*mems[x][y-1][z][m];
						neighbors ++;
					}
					// case by case	: Z+
					for (m=0;m<members;m++) if (m!=k) {
						if (useEdges) ngb += edges[2][x][y][z]*mems[x][y][z+1][m]*mems[x][y][z+1][m];
						else ngb += mems[x][y][z+1][m]*mems[x][y][z+1][m];
						neighbors ++;
					}
					// case by case	: Z-
					for (m=0;m<members;m++) if (m!=k) {
						if (useEdges) ngb += edges[2][x][y][z-1]*mems[x][y][z-1][m]*mems[x][y][z-1][m];
						else ngb += mems[x][y][z-1][m]*mems[x][y][z-1][m];
						neighbors ++;
					}
					if (neighbors>0.0) num = num + smoothing*ngb/neighbors;
				}
				
				// invert after adding all t
				if (num>ZERO) num = 1.0f/num;
				else num = INF;

				mems[x][y][z][k] = num;
				den += num;
			}

			// normalize
			for (k=0;k<members;k++) {
				mems[x][y][z][k] = mems[x][y][z][k]/den;

				// compute the maximum distance
				dist = Math.abs(mems[x][y][z][k]-prev[k]);
				if (dist > distance) distance = dist; 
			}
		}
        if (debug) System.out.print("inner loop time: (milliseconds): " + (System.currentTimeMillis()-inner_loop_time) +"\n"); 

        return distance;
    } // computeTransformedMemberships
    
    /** 
	 *  compute the RFCM membership functions given the centroids 
	 *  with multichannel images and transformed images
	 *	the algorithm only uses images 1 through Nt
	 */
    final public float computeGeneralTransformedMemberships(int Nt) {
        float distance;
        int x,y,z,k,m,s,t;
        int progress, mod;
        long inner_loop_time;
        float den,num;
        float neighbors, ngb;
        float dist,imgT;
        float xT,yT,zT;
        float xTp,yTp,zTp;
        float xTm,yTm,zTm;
        float val;
        
        distance = 0.0f;
		progress = 0;
        mod = nx*ny*nz/20; // mod is 1 percent of length

        inner_loop_time = System.currentTimeMillis();
        for (x=1;x<nx-1;x++) for (y=1;y<ny-1;y++) for (z=1;z<nz-1;z++) {
			progress++;
			if ((verbose) && (progress%mod==0) )
                progressBar.updateValue((int)Math.round( (float)progress/(float)mod*5.0f),false);

			den = 0;
			// remember the previous values
			for (k=0;k<members;k++) prev[k] = mems[x][y][z][k];
			
			for (k=0;k<members;k++) {
				// numerator
				num = 0; 
				
				for (t=0;t<Nt;t++) {
					// compute the local position: X' = RX+T
					xT = (transforms[t][0][0]*x*rx + transforms[t][0][1]*y*ry + transforms[t][0][2]*z*rz + transforms[t][0][3])/rx;
					yT = (transforms[t][1][0]*x*rx + transforms[t][1][1]*y*ry + transforms[t][1][2]*z*rz + transforms[t][1][3])/ry;
					zT = (transforms[t][2][0]*x*rx + transforms[t][2][1]*y*ry + transforms[t][2][2]*z*rz + transforms[t][2][3])/rz;
					// neighbors
					xTp = (transforms[t][0][0]*(x+1)*rx + transforms[t][0][1]*y*ry     + transforms[t][0][2]*z*rz     + transforms[t][0][3])/rx;
					yTp = (transforms[t][1][0]*x*rx     + transforms[t][1][1]*(y+1)*ry + transforms[t][1][2]*z*rz     + transforms[t][1][3])/ry;
					zTp = (transforms[t][2][0]*x*rx     + transforms[t][2][1]*y*ry     + transforms[t][2][2]*(z+1)*rz + transforms[t][2][3])/rz;
					xTm = (transforms[t][0][0]*(x-1)*rx + transforms[t][0][1]*y*ry     + transforms[t][0][2]*z*rz     + transforms[t][0][3])/rx;
					yTm = (transforms[t][1][0]*x*rx     + transforms[t][1][1]*(y-1)*ry + transforms[t][1][2]*z*rz     + transforms[t][1][3])/ry;
					zTm = (transforms[t][2][0]*x*rx     + transforms[t][2][1]*y*ry     + transforms[t][2][2]*(z-1)*rz + transforms[t][2][3])/rz;

					if ((useRegistration) || (ImageFunctions.maskInterpolation(mask[t],xT,yT,zT,nx,ny,nz))) {
						// pre-compute interpolated values
						imgT = ImageFunctions.linearInterpolation(image[t],maskVal[t],xT,yT,zT,nx,ny,nz);

						// data term
						if (k<clusters) {
							if (correctField[t]) val = (field[t][x][y][z]*imgT-centroids[t][k]);
							else val = (imgT-centroids[t][k]);
							if (val>0) num += (val*val)/(positiveScaling[t][k]*positiveScaling[t][k]);
							else num += (val*val)/(negativeScaling[t][k]*negativeScaling[t][k]);
						} else if (k<classes) {
							num += outlier*outlier;
						} else {
							if (correctField[t]) val = (field[t][x][y][z]*imgT-maskVal[t]);
							else val = (imgT-maskVal[t]);
							if (val>0) num += (val*val)/(positiveScaling[t][k]*positiveScaling[t][k]);
							else num += (val*val)/(negativeScaling[t][k]*negativeScaling[t][k]);
						}
					} else {
						num += INF;
					}
				}
				
				// normalize over the images
				num = num/(float)Nt;
				
				// spatial smoothing
				if (smoothing > 0.0f) { 
					ngb = 0.0f;  
					neighbors = 0.0f;
					// case by case	: X+
					for (m=0;m<members;m++) if (m!=k) {
						if (useEdges) ngb += edges[0][x][y][z]*power.lookup(mems[x+1][y][z][m],fuzziness);
						else ngb += power.lookup(mems[x+1][y][z][m],fuzziness);
						neighbors ++;
					}
					// case by case	: X-
					for (m=0;m<members;m++) if (m!=k) {
						if (useEdges) ngb += edges[0][x-1][y][z]*power.lookup(mems[x-1][y][z][m],fuzziness);
						else ngb += power.lookup(mems[x-1][y][z][m],fuzziness);
						neighbors ++;
					}
					// case by case	: Y+
					for (m=0;m<members;m++) if (m!=k) {
						if (useEdges) ngb += edges[1][x][y][z]*power.lookup(mems[x][y+1][z][m],fuzziness);
						else ngb += power.lookup(mems[x][y+1][z][m],fuzziness);
						neighbors ++;
					}
					// case by case	: Y-
					for (m=0;m<members;m++) if (m!=k) {
						if (useEdges) ngb += edges[1][x][y-1][z]*power.lookup(mems[x][y-1][z][m],fuzziness);
						else ngb += power.lookup(mems[x][y-1][z][m],fuzziness);
						neighbors ++;
					}
					// case by case	: Z+
					for (m=0;m<members;m++) if (m!=k) {
						if (useEdges) ngb += edges[2][x][y][z]*power.lookup(mems[x][y][z+1][m],fuzziness);
						else ngb += power.lookup(mems[x][y][z+1][m],fuzziness);
						neighbors ++;
					}
					// case by case	: Z-
					for (m=0;m<members;m++) if (m!=k) {
						if (useEdges) ngb += edges[2][x][y][z-1]*power.lookup(mems[x][y][z-1][m],fuzziness);
						else ngb += power.lookup(mems[x][y][z-1][m],fuzziness);
						neighbors ++;
					}
					if (neighbors>0.0) num = num + smoothing*ngb/neighbors;
				}
				
				// invert after adding all t
				if (num>ZERO) num = (float)invpower.lookup(num,1.0f/(1.0f-fuzziness) );
				else num = INF;
	
				mems[x][y][z][k] = num;
				den += num;
			}

			// normalize
			for (k=0;k<members;k++) {
				mems[x][y][z][k] = mems[x][y][z][k]/den;

				// compute the maximum distance
				dist = Math.abs(mems[x][y][z][k]-prev[k]);
				if (dist > distance) distance = dist; 
			}
		}
        if (debug) System.out.print("inner loop time: (milliseconds): " + (System.currentTimeMillis()-inner_loop_time) +"\n"); 

        return distance;
    } // computeTransformedMemberships
    
     /**
	 * compute the centroids given the membership functions
	 */
    final public void computeCentroids(int Nt) {
        if (scalingMode==CENTROID) {
			computeScaledCentroids(Nt);
			return;
		}
			
		int x,y,z,k,t;
		float num,den;
        
		if (fuzziness!=2) {
			computeGeneralCentroids(Nt);
			return;
		}
        
		for (t=0;t<Nt;t++) {
			for (k=0;k<clusters;k++) {
				num = 0;
				den = 0;
				for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
					if ((useRegistration) || (mask[t][x][y][z]) ) {
						if (correctField[t]) num += mems[x][y][z][k]*mems[x][y][z][k]
													*field[t][x][y][z]*image[t][x][y][z];
						else num += mems[x][y][z][k]*mems[x][y][z][k]*image[t][x][y][z];
						den += mems[x][y][z][k]*mems[x][y][z][k];
					}
				}
				if (den>0.0) {
					centroids[t][k] = num/den;
				} else {
					centroids[t][k] = 0.0f;
				}
			}
			if (verbose) {
				MedicUtilPublic.displayMessage("centroids: ("+centroids[t][0]);
				for (k=1;k<clusters;k++) MedicUtilPublic.displayMessage(", "+centroids[t][k]);
				MedicUtilPublic.displayMessage(")\n");
			}
		}
        return;
    } // computeCentroids
    
    final public void computeScaledCentroids(int Nt) {
        int x,y,z,k,t;
		float num,den;
		float mem,dc,cy;
		float lower, higher;
        
		for (t=0;t<Nt;t++) {
			for (k=0;k<clusters;k++) {
				//find closest centroids
				int above=-1;
				int below=-1;
				for (int m=0;m<clusters;m++) {
					if (centroids[t][m]<centroids[t][k]) {
						if (below==-1) below = m;
						else if (centroids[t][below]<centroids[t][m]) below = m;
					} else if (centroids[t][m]>centroids[t][k]) {
						if (above==-1) above = m;
						else if (centroids[t][above]>centroids[t][m]) above = m;
					}
				}
				if (above==-1) higher = Imax[t];
				else higher = centroids[t][above];
				if (below==-1) lower = Imin[t];
				else lower = centroids[t][below];
				
				
				num = 0;
				den = 0;
				for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
					if ((useRegistration) || (mask[t][x][y][z]) ) {
						
						// new formula!
						if (correctField[t]) {
							if (field[t][x][y][z]*image[t][x][y][z]<centroids[t][k]) {
								if (below==-1) {
									mem = 0.0f;
									dc = centroids[t][k]-Imin[t];
									cy = field[t][x][y][z]*image[t][x][y][z]-Imin[t];
								} else {
									mem = mems[x][y][z][below];
									dc = centroids[t][k]-centroids[t][below];
									cy = field[t][x][y][z]*image[t][x][y][z]-centroids[t][below];
								}	
							} else {
								if (above==-1) {
									mem = 0.0f;
									dc = Imax[t]-centroids[t][k];
									cy = Imax[t]-field[t][x][y][z]*image[t][x][y][z];
								} else {
									mem = mems[x][y][z][above];
									dc = centroids[t][k]-centroids[t][above];
									cy = field[t][x][y][z]*image[t][x][y][z]-centroids[t][above];
								}	
							}
							mem = mem*mem;
							dc =  dc*dc*dc;
							num += mems[x][y][z][k]*mems[x][y][z][k]*field[t][x][y][z]*image[t][x][y][z]*cy/dc
							 	 - mem*cy*cy/dc;
							den += mems[x][y][z][k]*mems[x][y][z][k]*cy/dc;
							
						} else {
							if (image[t][x][y][z]<centroids[t][k]) {
								if (below==-1) {
									mem = 0.0f;
									dc = centroids[t][k]-Imin[t];
									cy = image[t][x][y][z]-Imin[t];
								} else {
									mem = mems[x][y][z][below];
									dc = centroids[t][k]-centroids[t][below];
									cy = image[t][x][y][z]-centroids[t][below];
								}	
							} else {
								if (above==-1) {
									mem = 0.0f;
									dc = Imax[t]-centroids[t][k];
									cy = Imax[t]-image[t][x][y][z];
								} else {
									mem = mems[x][y][z][above];
									dc = centroids[t][k]-centroids[t][above];
									cy = image[t][x][y][z]-centroids[t][above];
								}	
							}
							mem = mem*mem;
							dc =  dc*dc*dc;
							num += mems[x][y][z][k]*mems[x][y][z][k]*image[t][x][y][z]*cy/dc
							 	 - mem*cy*cy/dc;
							den += mems[x][y][z][k]*mems[x][y][z][k]*cy/dc;
							
						}
					}
				}
				if (den>0.0) {
					centroids[t][k] = num/den;
				} else {
					centroids[t][k] = 0.0f;
				}
			}
			if (verbose) {
				MedicUtilPublic.displayMessage("centroids: ("+centroids[t][0]);
				for (k=1;k<clusters;k++) MedicUtilPublic.displayMessage(", "+centroids[t][k]);
				MedicUtilPublic.displayMessage(")\n");
			}
		}
        return;
    } // computeCentroids
    
     /**
	 * compute the centroids given the membership functions
	 */
    final public void computeGeneralCentroids(int Nt) {
        int x,y,z,k,t;
		float num,den;
        float val;
		
		for (t=0;t<Nt;t++) {
			for (k=0;k<clusters;k++) {
				num = 0;
				den = 0;
				for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
					if ((useRegistration) || (mask[t][x][y][z]) ) {
						val = (float)power.lookup(mems[x][y][z][k],fuzziness);
						if (correctField[t]) num += val*field[t][x][y][z]*image[t][x][y][z];
						else num += val*image[t][x][y][z];
						den += val;
					}
				}
				if (den>0.0) {
					centroids[t][k] = num/den;
				} else {
					centroids[t][k] = 0.0f;
				}
			}
			if (verbose) {
				MedicUtilPublic.displayMessage("centroids: ("+centroids[t][0]);
				for (k=1;k<clusters;k++) MedicUtilPublic.displayMessage(", "+centroids[t][k]);
				MedicUtilPublic.displayMessage(")\n");
			}
		}
        return;
    } // computeGeneralCentroids
    
    /**
	 * compute the centroids given the membership functions
	 */
    final public void computeTransformedCentroids(int Nt) {
        int x,y,z,k,t;
		float xT,yT,zT,imgT;
		float num,den;
        
		if (fuzziness!=2) {
			computeGeneralTransformedCentroids(Nt);
			return;
		}
        
		for (t=0;t<Nt;t++) {
			for (k=0;k<clusters;k++) {
				num = 0;
				den = 0;
				for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
					// compute the local position: X' = RX+T
					xT = (transforms[t][0][0]*x*rx + transforms[t][0][1]*y*ry + transforms[t][0][2]*z*rz + transforms[t][0][3])/rx;
					yT = (transforms[t][1][0]*x*rx + transforms[t][1][1]*y*ry + transforms[t][1][2]*z*rz + transforms[t][1][3])/ry;
					zT = (transforms[t][2][0]*x*rx + transforms[t][2][1]*y*ry + transforms[t][2][2]*z*rz + transforms[t][2][3])/rz;

					if ((useRegistration) || (ImageFunctions.maskInterpolation(mask[t],xT,yT,zT,nx,ny,nz)) ) {
						// pre-compute interpolated values
						imgT = ImageFunctions.linearInterpolation(image[t],maskVal[t],xT,yT,zT,nx,ny,nz);
					
						if (correctField[t]) num += mems[x][y][z][k]*mems[x][y][z][k]
													*field[t][x][y][z]*imgT;
						else num += mems[x][y][z][k]*mems[x][y][z][k]*imgT;
						den += mems[x][y][z][k]*mems[x][y][z][k];
					}
				}
				if (den>0.0) {
					centroids[t][k] = num/den;
				} else {
					centroids[t][k] = 0.0f;
				}
			}
			if (verbose) {
				MedicUtilPublic.displayMessage("centroids: ("+centroids[t][0]);
				for (k=1;k<clusters;k++) MedicUtilPublic.displayMessage(", "+centroids[t][k]);
				MedicUtilPublic.displayMessage(")\n");
			}
		}
        return;
    } // computeTransformedCentroids
    
    /**
	 * compute the centroids given the membership functions
	 */
    final public void computeGeneralTransformedCentroids(int Nt) {
        int x,y,z,k,t;
		float xT,yT,zT,imgT;
		float num,den;
        float val;
		
		for (t=0;t<Nt;t++) {
			for (k=0;k<clusters;k++) {
				num = 0;
				den = 0;
				for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
					// compute the local position: X' = RX+T
					xT = (transforms[t][0][0]*x*rx + transforms[t][0][1]*y*ry + transforms[t][0][2]*z*rz + transforms[t][0][3])/rx;
					yT = (transforms[t][1][0]*x*rx + transforms[t][1][1]*y*ry + transforms[t][1][2]*z*rz + transforms[t][1][3])/ry;
					zT = (transforms[t][2][0]*x*rx + transforms[t][2][1]*y*ry + transforms[t][2][2]*z*rz + transforms[t][2][3])/rz;

					if ((useRegistration) || (ImageFunctions.maskInterpolation(mask[t],xT,yT,zT,nx,ny,nz)) ) {
						// pre-compute interpolated values
						imgT = ImageFunctions.linearInterpolation(image[t],maskVal[t],xT,yT,zT,nx,ny,nz);
						val = (float)power.lookup(mems[x][y][z][k],fuzziness);
					
						if (correctField[t]) num += val*field[t][x][y][z]*imgT;
						else num += val*imgT;
						den += val;
					}
				}
				if (den>0.0) {
					centroids[t][k] = num/den;
				} else {
					centroids[t][k] = 0.0f;
				}
			}
			if (verbose) {
				MedicUtilPublic.displayMessage("centroids: ("+centroids[t][0]);
				for (k=1;k<clusters;k++) MedicUtilPublic.displayMessage(", "+centroids[t][k]);
				MedicUtilPublic.displayMessage(")\n");
			}
		}
        return;
    } // computeTransformedCentroids
    
    /** 
	 *  compute the scaling factor given the memberships and centroids 
	 *  with multichannel images
	 *	the algorithm only uses images 1 through Nt
	 *
	 */
    final public void computeScaling(int Nt) {
		if (scalingMode==INTENSITY) return;
		else if (scalingMode==ENERGY) computeEnergyScaling(Nt);
		else if (scalingMode==VARIANCE) computeVarianceScaling(Nt);
		else if (scalingMode==CENTROID) return;
		//computeCentroidDistanceScaling(Nt);
	}
	
	/** 
	 *  compute the energy-based scaling factor given the memberships and centroids 
	 *  with multichannel images
	 *	the algorithm only uses images 1 through Nt
	 *
	 */
    final public void computeEnergyScaling(int Nt) {
		float[] energy = new float[Nt];
        float distance;
        int x,y,z,k,m,s,t;
        int progress, mod;
        long inner_loop_time;
        float den,num;
        float neighbors, ngb;
        float dist;
        boolean	isUsed;
        
        for (t=0;t<Nt;t++) energy[t] = 0.0f;
		
        for (x=1;x<nx-1;x++) for (y=1;y<ny-1;y++) for (z=1;z<nz-1;z++) {
			// check if real value first
			isUsed = true;
			if (!useRegistration) {
				if (maskingMode==ONE) {
					isUsed = false;
					for (t=0;t<Nt;t++) if (mask[t][x][y][z]) isUsed = true;
				} else if (maskingMode==ALL) {
					isUsed = true;
					for (t=0;t<Nt;t++) if (!mask[t][x][y][z]) isUsed = false;
				}	
			}
				
			if (isUsed) {
				den = 0;
				
				for (k=0;k<members;k++) {
					// numerator
					num = 0; 
					
					for (t=0;t<Nt;t++) {
						num = 0;
						if ( (useRegistration) || (mask[t][x][y][z]) ) {
							// data term
							if (k<clusters) {
								if (correctField[t]) num += (field[t][x][y][z]*image[t][x][y][z]-centroids[t][k])*(field[t][x][y][z]*image[t][x][y][z]-centroids[t][k]);
								else num += (image[t][x][y][z]-centroids[t][k])*(image[t][x][y][z]-centroids[t][k]);
							} else if (k<classes) {
								num += outlier*outlier;
							} else {
								if (correctField[t]) num += (field[t][x][y][z]*image[t][x][y][z]-maskVal[t])*(field[t][x][y][z]*image[t][x][y][z]-maskVal[t]);
								else num += (image[t][x][y][z]-maskVal[t])*(image[t][x][y][z]-maskVal[t]);
							}
						} else {
							num += INF;
						}
						energy[t] += mems[x][y][z][k]*mems[x][y][z][k]*num;
					}
				}
			}
		}
		for (t=0;t<Nt;t++) for (k=0;k<members;k++) {
			positiveScaling[t][k] = positiveScaling[t][k]*energy[t]/energy[0];
			negativeScaling[t][k] = negativeScaling[t][k]*energy[t]/energy[0];
		}
		if (debug) {
			System.out.print("scaling: ");
			for (t=0;t<Nt-1;t++) System.out.print((energy[t]/energy[0])+", ");
			System.out.print((energy[Nt-1]/energy[0])+"\n");
		}
        return;
    } // computeEnergyScaling
    
	/** 
	 *  compute the variance-based scaling factor given the memberships and centroids 
	 *  with multichannel images
	 *	the algorithm only uses images 1 through Nt
	 *
	 */
    final public void computeVarianceScaling(int Nt) {
		float[][] variance = new float[Nt][members];
		float[][] norm = new float[Nt][members];
        float distance;
        int x,y,z,k,m,s,t;
        int progress, mod;
        long inner_loop_time;
        float den,num;
        float neighbors, ngb;
        float dist;
        boolean	isUsed;
        
        for (t=0;t<Nt;t++) for (k=0;k<members;k++) {
			variance[t][k] = 0.0f;
			norm[t][k] = 0.0f;
		}
		
        for (x=1;x<nx-1;x++) for (y=1;y<ny-1;y++) for (z=1;z<nz-1;z++) {
			// check if real value first
			isUsed = true;
			if (!useRegistration) {
				if (maskingMode==ONE) {
					isUsed = false;
					for (t=0;t<Nt;t++) if (mask[t][x][y][z]) isUsed = true;
				} else if (maskingMode==ALL) {
					isUsed = true;
					for (t=0;t<Nt;t++) if (!mask[t][x][y][z]) isUsed = false;
				}	
			}
				
			if (isUsed) {
				for (k=0;k<members;k++) {
					for (t=0;t<Nt;t++) {
						num = 0;
						if ( (useRegistration) || (mask[t][x][y][z]) ) {
							// data term
							if (k<clusters) {
								if (correctField[t]) num += (field[t][x][y][z]*image[t][x][y][z]-centroids[t][k])*(field[t][x][y][z]*image[t][x][y][z]-centroids[t][k]);
								else num += (image[t][x][y][z]-centroids[t][k])*(image[t][x][y][z]-centroids[t][k]);
							} else if (k<classes) {
								num += outlier*outlier;
							} else {
								if (correctField[t]) num += (field[t][x][y][z]*image[t][x][y][z]-maskVal[t])*(field[t][x][y][z]*image[t][x][y][z]-maskVal[t]);
								else num += (image[t][x][y][z]-maskVal[t])*(image[t][x][y][z]-maskVal[t]);
							}
						} else {
							num += INF;
						}
						variance[t][k] += mems[x][y][z][k]*mems[x][y][z][k]*num;
						norm[t][k] += mems[x][y][z][k]*mems[x][y][z][k];
					}
				}
			}
		}
		for (t=0;t<Nt;t++) for (k=0;k<members;k++) {
			positiveScaling[t][k] = (float)Math.sqrt(variance[t][k]/norm[t][k]);
			negativeScaling[t][k] = (float)Math.sqrt(variance[t][k]/norm[t][k]);
		}
		if (debug) {
			System.out.print("scaling: \n");
			for (t=0;t<Nt;t++) {
				for (k=0;k<members-1;k++)
					System.out.print(positiveScaling[t][k]+", ");
				System.out.print(positiveScaling[t][members-1]+"\n");
			}
		}
        return;
    } // computeVarianceScaling
    
	/** 
	 *  compute the centroid-based scaling factor given the memberships and centroids 
	 *  with multichannel images
	 *	the algorithm only uses images 1 through Nt
	 *
	 */
    final public void computeCentroidDistanceScaling(int Nt) {
		for (int t=0;t<Nt;t++) for (int k=0;k<members;k++) {
			if (k<clusters) {
				//find closest centroids
				int above=-1;
				int below=-1;
				for (int m=0;m<clusters;m++) {
					if (centroids[t][m]<centroids[t][k]) {
						if (below==-1) below = m;
						else if (centroids[t][below]<centroids[t][m]) below = m;
					} else if (centroids[t][m]>centroids[t][k]) {
						if (above==-1) above = m;
						else if (centroids[t][above]>centroids[t][m]) above = m;
					}
				}
				if (above==-1) positiveScaling[t][k] = Imax[t]-centroids[t][k];
				else positiveScaling[t][k] = 0.5f*positiveScaling[t][k] + 0.25f*(centroids[t][above]-centroids[t][k]);
				if (below==-1) negativeScaling[t][k] = centroids[t][k]-Imin[t];
				else negativeScaling[t][k] = 0.5f*negativeScaling[t][k] + 0.25f*(centroids[t][k]-centroids[t][below]);
			} else {
				negativeScaling[t][k] = 1.0f;
				positiveScaling[t][k] = 1.0f;
			}
		}
		if (debug) {
			System.out.print("scaling: \n");
			for (int t=0;t<Nt;t++) {
				for (int k=0;k<members-1;k++)
					System.out.print(positiveScaling[t][k]+", ");
				System.out.print(positiveScaling[t][members-1]+"\n");
				for (int k=0;k<members-1;k++)
					System.out.print(negativeScaling[t][k]+", ");
				System.out.print(negativeScaling[t][members-1]+"\n");
			}
		}
        return;
    } // computeVarianceScaling
    
	/** 
	 *	returns the hard classification (max_{clusters}(Mems)) 
	 */
	public final byte[][][] exportHardClassification() {
		int 	x,y,z,t,k,best;
		byte[][][]	classification = new byte[nx][ny][nz];
		float bestmem;
        
        for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {                    
			best = -1;bestmem = 0.0f;
            for (k=0;k<members;k++) {
				if (mems[x][y][z][k] > bestmem) {
					best = k;
					bestmem = mems[x][y][z][k];
                }
            }   
			if ( (best==-1) || (best==classes) ) {
				classification[x][y][z] = 0;
			} else {
				classification[x][y][z] = (byte)(best+1);
			}
		}
		return classification;
	} // exportHardClassification

	/** 
	 *	returns the hard classification (max_{clusters}(Mems)) 
	 *  in the case with registration (doesn't change much, the mask is the last class)
	 */
	public final byte[][][] exportTransformedHardClassification() {
		int 	x, y, z, k,best;
		float		xT, yT, zT;
		byte[][][]	classification = new byte[nx][ny][nz];
        float bestmem;
        
		for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {                    
			best = -1;bestmem = 0.0f;
			for (k=0;k<members;k++) {
				if (mems[x][y][z][k] > bestmem) {
					best = k;
					bestmem = mems[x][y][z][k];
				}
			}   
			if ( (best==-1) || (best==classes) )
				classification[x][y][z] = 0;
			else
				classification[x][y][z] = (byte)(best+1);
		}
		return classification;
	} // exportHardClassification

	/** 
	 *	export membership functions 
	 */
	public final float[][][][] exportMembership() {
		int 	x,y,z,k,t;
		float[][][][]	Mems = new float[classes][nx][ny][nz];
		
        for (k=0;k<classes;k++) {
            for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
				Mems[k][x][y][z] = mems[x][y][z][k];
 			}
		}
		return Mems;
	} // exportMemberships

	/** 
	 *	returns the masked image 
	 */
	public final float[][][] exportMaskedImage(int t) {
		int 	x,y,z;
		float[][][]	img = new float[nx][ny][nz];
		
        for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {                    
			if (mask[t][x][y][z]) {
				img[x][y][z] = image[t][x][y][z];
			} else {
				img[x][y][z] = 0.0f;
			}
		}
		return img;
	} // exportMaskedImage

	/** 
	 *	returns the transformed image
	 *  in the case with registration
	 */
	public final float[][][] exportTransformedImage(int t) {
		int 	x, y, z;
		float		xT, yT, zT;
		float[][][]	img = new float[nx][ny][nz];
        
		for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {                    
			// compute the local position: X' = RX+T
			xT = (transforms[t][0][0]*x*rx + transforms[t][0][1]*y*ry + transforms[t][0][2]*z*rz + transforms[t][0][3])/rx;
			yT = (transforms[t][1][0]*x*rx + transforms[t][1][1]*y*ry + transforms[t][1][2]*z*rz + transforms[t][1][3])/ry;
			zT = (transforms[t][2][0]*x*rx + transforms[t][2][1]*y*ry + transforms[t][2][2]*z*rz + transforms[t][2][3])/rz;

			if (ImageFunctions.maskInterpolation(mask[t],xT,yT,zT,nx,ny,nz) ) {
				img[x][y][z] = ImageFunctions.cubicLagrangianInterpolation3D(image[t],wt,Imin[t],Imax[t],xT,yT,zT,nx,ny,nz);
			} else {
				img[x][y][z] = Imin[t];
			}
		}
		return img;
	} // exportTransformedImage

	/** 
	 *	makes a new mask from the image mask and the inverse
	 *	of the transform (bringing a mask from the aligned space 
	 *  to the original space)
	 */
	public final boolean[][][] createBackTransformedMask(int t, boolean[][][] msk) {
		int 	x, y, z;
		float		xT, yT, zT;
		boolean[][][]	transMask = new boolean[nx][ny][nz];
        
		for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {                    
			// compute the inverse local position: X' = (RX+T)^-1
			xT = (transforms[t][0][0]*(x*rx - transforms[t][0][3]) + transforms[t][1][0]*(y*ry - transforms[t][1][3]) + transforms[t][2][0]*(z*rz - transforms[t][2][3]) )/rx;
			yT = (transforms[t][0][1]*(x*rx - transforms[t][0][3]) + transforms[t][1][1]*(y*ry - transforms[t][1][3]) + transforms[t][2][1]*(z*rz - transforms[t][2][3]) )/ry;
			zT = (transforms[t][0][2]*(x*rx - transforms[t][0][3]) + transforms[t][1][2]*(y*ry - transforms[t][1][3]) + transforms[t][2][2]*(z*rz - transforms[t][2][3]) )/rz;

			if (ImageFunctions.maskInterpolation(msk,xT,yT,zT,nx,ny,nz) ) {
				transMask[x][y][z] = true;
			} else {
				transMask[x][y][z] = false;
			}
		}
		return transMask;
	} // createBackTransformedMask

	/** 
	 *	makes a mask from all image masks transformed to the common space
	 */
	public final boolean[][][] createCommonTransformedMask() {
		float		xT, yT, zT;
		boolean[][][]	common = new boolean[nx][ny][nz];
		float	val;
        
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {                    
			val = 0.0f;
			for (int t=0;t<nt;t++) {
				// compute the local position: X' = RX+T
				xT = (transforms[t][0][0]*x*rx + transforms[t][0][1]*y*ry + transforms[t][0][2]*z*rz + transforms[t][0][3])/rx;
				yT = (transforms[t][1][0]*x*rx + transforms[t][1][1]*y*ry + transforms[t][1][2]*z*rz + transforms[t][1][3])/ry;
				zT = (transforms[t][2][0]*x*rx + transforms[t][2][1]*y*ry + transforms[t][2][2]*z*rz + transforms[t][2][3])/rz;

				val += ImageFunctions.linearMaskInterpolationValue(mask[t],xT,yT,zT,nx,ny,nz);
				//if (ImageFunctions.linearMaskInterpolation(mask[t],xT,yT,zT,nx,ny,nz)) val++;
			}
			if (val >= 0.5f*nt) common[x][y][z] = true;
			else common[x][y][z] = false;
		}
		return common;
	} // createCommonTransformedMask

    /** 
	 *	returns the transform in the Mipav transformation matrix style
	 */
	public final double[][] convertTransform(int t) {
		int 	x,y,z,k,best;
		double[][]	mat = new double[4][4];
		       
		// compute the inverse transformation matrix (image tp classification)
		// R0 = R^t
		for (int i=0;i<3;i++) for (int j=0;j<3;j++) {
			mat[i][j] = (double)transforms[t][j][i];
		}
		// T0 = -R^t T
        // warning: TransMatrix uses an awful convention where 
		for (int i=0;i<3;i++) {
			mat[i][3] = 0.0;
            for (int j=0;j<3;j++) {
				mat[i][3] += -mat[i][j]*(double)transforms[t][j][3];
			}
		}
		// bottom line
		for (int i=0;i<3;i++) mat[3][i] = 0.0;
		mat[3][3] = 1.0;
		
		return mat;
	} // convertTransformMatrix

	/** 
	 *	create ids for the centroids based on ordering
	 */
	public final int[] computeCentroidOrder() {
		int 	k,l;
		int[]	id = new int[classes+1];
		int		lowest;
		float[] cent = new float[clusters];
		
		// copy the centroids
        for (k=0;k<clusters;k++) 
			cent[k] = centroids[0][k];
		
		// add the zero
		id[0] = 0;

		// order them from smallest to largest
		for (k=0;k<clusters;k++) {
			lowest = 0;
			for (l=1;l<clusters;l++) {
				if (cent[l] < cent[lowest]) {
					lowest = l;
				}
			}
			id[k+1] = lowest+1;
			cent[lowest] = INF;
		}
		// keep order for other class types (outliers, etc)
		for (k=clusters;k<classes;k++)
			id[k+1] = k+1;
		
        if (debug) {
			MedicUtilPublic.displayMessage("ordering: ("+id[0]);
			for (k=0;k<classes;k++) MedicUtilPublic.displayMessage(", "+id[k+1]);
			MedicUtilPublic.displayMessage(")\n");
		}   
 		return id;
	} // computeCentroidOrder

	/** 
	 *	associate the centroids from image to image based on the least distance 
	 */
	public final void computeCentroidPairing(int t1, int t2) {
		float[][] count = new float[clusters][clusters];
		float maxcount;
		float num1,num2;
		float den1,den2;
		float max1,max2;
		int	id1,id2;
		boolean isUsed;
		
		// copy the centroids
        for (int k=0;k<clusters;k++) 
			for (int m=0;m<clusters;m++)
				count[k][m] = 0.0f;
		
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {                    
			// check if real value first
			isUsed = true;
			if (!useRegistration) {
				if (maskingMode==ONE) {
					isUsed = false;
					if (mask[t1][x][y][z]) isUsed = true;
					if (mask[t2][x][y][z]) isUsed = true;
				} else if (maskingMode==ALL) {
					isUsed = true;
					if (!mask[t1][x][y][z]) isUsed = false;
					if (!mask[t2][x][y][z]) isUsed = false;
				}	
			}
				
			if (isUsed) {
				// find the class with highest membership
				id1 = 0;
				id2 = 0;
				den1 = 0;
				den2 = 0;
				max1 = 0;
				max2 = 0;
				for (int k=0;k<clusters;k++) {
					num1 = (image[t1][x][y][z] - centroids[t1][k])*(image[t1][x][y][z] - centroids[t1][k]);
					if (num1>ZERO) num1 = 1.0f/num1;
					else num1 = INF;
					den1 += num1;
					if (num1>max1) {
						max1 = num1;
						id1 = k;
					}
					num2 = (image[t2][x][y][z] - centroids[t2][k])*(image[t2][x][y][z] - centroids[t2][k]);
					if (num2>ZERO) num2 = 1.0f/num2;
					else num2 = INF;
					den2 += num2;
					if (num2>max2) {
						max2 = num2;
						id2 = k;
					}
				}
				count[id1][id2]+= (max1/den1)*(max2/den2);
			}
		}
		
		// find max count : make pair
		for (int l=0;l<clusters;l++) {
			maxcount = 0;
			id1 = 0; id2 = 0;
			for (int k=0;k<clusters;k++) {
				for (int m=0;m<clusters;m++) {
					if (count[k][m]>maxcount) {
						maxcount = count[k][m];
						id1 = k;
						id2 = m;
					}
				}
			}
			// swap id2 and k in image 2
			System.out.print("swap "+id1+"<-"+id2+"\n");
			float val = centroids[t2][id2];
			centroids[t2][id2] = centroids[t2][id1];
			centroids[t2][id1] = val;
			// reset the values for row and column
			for (int k=0;k<clusters;k++) {
				count[k][id2] = count[k][id1];
				count[id1][k] = 0;
				count[k][id1] = 0;
			}
		}
		if (verbose) {
			MedicUtilPublic.displayMessage("centroids "+(t1+1)+": ("+centroids[t1][0]);
			for (int k=1;k<clusters;k++) MedicUtilPublic.displayMessage(", "+centroids[t1][k]);
			MedicUtilPublic.displayMessage(")\n");
			MedicUtilPublic.displayMessage("centroids "+(t2+1)+": ("+centroids[t2][0]);
			for (int k=1;k<clusters;k++) MedicUtilPublic.displayMessage(", "+centroids[t2][k]);
			MedicUtilPublic.displayMessage(")\n");
		}
		return;
	} // computeCentroidPairing
	
}
