package edu.jhmi.rad.medic.methods;
 import java.io.*;
import java.util.*;

import gov.nih.mipav.view.*;
import gov.nih.mipav.model.structures.jama.*;

import edu.jhmi.rad.medic.structures.*;
import edu.jhmi.rad.medic.libraries.*;
import edu.jhmi.rad.medic.utilities.*;

/**
 *
 *  This algorithm handles registration algorithms
 *	of an image against a segmentation map (soft or hard)
 *	with a multigrid least squares technique
 *	(euclidean, not robust)
 *
 *	@version    November 2004
 *	@author     Pierre-Louis Bazin
 *		
 *
 */
 
public class RegisterSegmentation {
		
	// numerical quantities
	private static final	float   INF=1e30f;
	private static final	float   ZERO=1e-30f;
	
	// data buffers
	private 	float[][][][]		images;  		// original image (3D pyramid)
	private 	float[][][][][]	    mems;			// membership function (3D+class+mask pyramid)
	private 	float[]				centroids;		// class centroids (no pyramid)
	private		float[][]			transform;		// transform from common space to original space (pyramid)
	private		float[][]			transformMatrix; // transform from common space to original space (3x4 matrix)
	private static	int[]			nx,ny,nz;   	// image dimensions (pyramid)
	private static	float			rx,ry,rz;   	// image resolutions (no pyramid)
	
	// pyramid handling
	private		int			levels;		// number of levels in the pyramid
	private		int			Nfirst;		// iterations for the first pass at a level
	private		int			Nlast;		// iterations for the last pass at a level
	private		int			Ntop;		// iterations for the pass at top level
	
	// parameters
	private 	int 		clusters;   // number of clusters
	private 	int 		classes;    // number of classes in original membership: > clusters if outliers
	private 	int 		members;    // number of members in the pyramid membership, including the mask
	private		float		outlier;    // outlier distance (not squared!)
	private		float		maskVal;    // mask value (usually zero)
	private		float		maxResidual; // not used (??)
	private		String		transformType;
	
	// computation variables
	private		float[][][][]   dx,dy,dz;
        
	// computation flags
	private 	boolean 		isWorking;
	private 	boolean 		isCompleted;
	
	// for debug and display
	ViewUserInterface			UI;
    ViewJProgressBar            progressBar;
	static final boolean		debug=true;
	static final boolean		verbose=true;
    
	/**
	 *  constructor
     *  note: the membership function mems_ must have sum = 0 on the masked areas
	 */
	public RegisterSegmentation(float[][][] image_, float[][][][] mems_,
									int nx_, int ny_, int nz_,
									float rx_, float ry_, float rz_,
									int nclasses_, int nclusters_,
                                    float outlierRatio_,
									int lev_, int Nf_, int Nl_, int Nt_,
									float maxResidual_, float maskVal_,
									ViewUserInterface UI_, ViewJProgressBar bar_) {
		classes = nclasses_;
		clusters = nclusters_;
        members = classes+1;
		outlier = outlierRatio_;
		maxResidual = maxResidual_;
		maskVal = maskVal_;

		UI = UI_;
        progressBar = bar_;
		
		levels = lev_;
		Nfirst = Nf_;
		Nlast = Nl_;
		Ntop = Nt_;
		
		rx = rx_;
		ry = ry_;
		rz = rz_;
			
		
		// init all the arrays : allocate the pyramids
		try {
			nx = new int[levels];
			ny = new int[levels];
			nz = new int[levels];
            // compute the pyramid dimensions
			nx[0] = nx_;
			ny[0] = ny_;
			nz[0] = nz_;
			for (int l=1;l<levels;l++) {
				nx[l] = (int)Math.floor(nx[l-1]/2.0);
				ny[l] = (int)Math.floor(ny[l-1]/2.0);
				nz[l] = (int)Math.floor(nz[l-1]/2.0);
			}
			mems = new float[levels][][][][];
			images = new float[levels][][][];
			transform = new float[levels][6];
			transformMatrix = new float[3][4];
			centroids = new float[clusters];
			dx = new float[levels][][][];
			dy = new float[levels][][][];
			dz = new float[levels][][][];
			for (int l=0;l<levels;l++) {
				mems[l] = new float[nx[l]][ny[l]][nz[l]][members];
				images[l] = new float[nx[l]][ny[l]][nz[l]];
				dx[l] = new float[nx[l]][ny[l]][nz[l]];
				dy[l] = new float[nx[l]][ny[l]][nz[l]];
				dz[l] = new float[nx[l]][ny[l]][nz[l]];
			}
		} catch (OutOfMemoryError e){
			isWorking = false;
            finalize();
			System.out.println(e.getMessage());
			return;
		}
		isWorking = true;

		// init values
		buildImagePyramid(images, image_);
        buildMembershipPyramid(mems, mems_);        

        for (int k=0;k<clusters;k++) {
            centroids[k] = 0.0f;
        }
        for (int l=0;l<levels;l++) {
            for (int d=0;d<6;d++) {
                transform[l][d] = 0.0f;
            }
			for (int i=0;i<3;i++) {
				for (int j=0;j<4;j++)
					transformMatrix[i][j] = 0.0f;
				transformMatrix[i][i] = 1.0f;
			}
            // compute the gradients
            ImageFunctions.computeGradientHorn(images[l],dx[l],dy[l],dz[l],nx[l],ny[l],nz[l]);
		}
		if (debug) MedicUtilPublic.displayMessage("F4D:initialisation\n");
	}

	final public void finalize() {
		images = null;
		mems = null;
		transform = null;
		transformMatrix = null;
		System.gc();
	}
    
    public final void setImage(float[][][] img) {
        buildImagePyramid(images, img);
    }

    public final void setTransform(float rx, float ry, float rz, float tx, float ty, float tz) {
        transform[0][0] = rx;
        transform[0][1] = ry;
        transform[0][2] = rz;
        transform[0][3] = tx;
        transform[0][4] = ty;
        transform[0][5] = tz;
        propagateTransform(0,levels-1);
		transformToMatrix(0);
    }
    
    public final void setTransform(float[] trans) {
        transform[0] = trans;
        propagateTransform(0,levels-1);
		transformToMatrix(0);
    }
    
    public final float[] getTransform() {
        return transform[0];
    }

    public final float[][] getTransformMatrix() {
        return transformMatrix;
    }
    public final float[][] exportTransformMatrix() {
		float[][] trans = new float[3][4];
		for (int i=0;i<3;i++) for (int j=0;j<4;j++)
			trans[i][j] = transformMatrix[i][j];
        return trans;
    }
    public final void transformToMatrix(int t) {
        RotationMatrix rotation = new RotationMatrix();
        // compute the matrix coefficients
		rotation.setParameters(transform[t][0],transform[t][1],transform[t][2]);

		for (int i=0;i<3;i++)
			for (int j=0;j<3;j++)
				transformMatrix[i][j] = rotation.getMatrix(i,j);
			
		transformMatrix[0][3] = transform[t][3];
		transformMatrix[1][3] = transform[t][4];
		transformMatrix[2][3] = transform[t][5];
		rotation = null;
    }
	
	public final boolean isWorking() { return isWorking; }
	public final boolean isCompleted() { return isCompleted; }
    
	final public void setCentroids(float[] cent) {
		for (int k=0;k<clusters;k++)
			centroids[k] = cent[k];
	}// setCentroids
    
	final public float[] exportCentroids() {
		float[] cent  = new float[clusters];
		for (int k=0;k<clusters;k++)
			cent[k] = centroids[k];
		return cent;
	}// exportCentroids
    
	/** create an image pyramid from the original data */
	private final void buildImagePyramid(float[][][][] img, float [][][] src) {
		int xM,xP,yM,yP,zM,zP;
		// first level : copy image
		for (int x=0;x<nx[0];x++) for (int y=0;y<ny[0];y++) for (int z=0;z<nz[0];z++) {
			img[0][x][y][z] = src[x][y][z];
		}
		// higher levels : average
		for (int l=1;l<levels;l++) {
			for (int x=0;x<nx[l];x++) for (int y=0;y<ny[l];y++) for (int z=0;z<nz[l];z++) {
				xM = 2*x; xP = 2*x+1;
				yM = 2*y; yP = 2*y+1;
				zM = 2*z; zP = 2*z+1;
				
				img[l][x][y][z] = 0.125f*( img[l-1][xM][yM][zM]+img[l-1][xP][yM][zM]
										  +img[l-1][xM][yP][zM]+img[l-1][xP][yP][zM]
										  +img[l-1][xM][yM][zP]+img[l-1][xP][yM][zP]
										  +img[l-1][xM][yP][zP]+img[l-1][xP][yP][zP] );
			}
		}
	}//buildImagePyramid
	
    /** create a pyramid for the memberships, add a class for masking */
	private final void buildMembershipPyramid(float[][][][][] mem, float [][][][] src) {
		int xM,xP,yM,yP,zM,zP;
        float sum;
		// first level : copy image
		for (int x=0;x<nx[0];x++) for (int y=0;y<ny[0];y++) for (int z=0;z<nz[0];z++) {
            sum = 0.0f;
            for (int k=0;k<classes;k++) {
                mem[0][x][y][z][k] = src[x][y][z][k];
                sum += src[x][y][z][k];
            }
			mem[0][x][y][z][classes] = 1.0f-sum;
		}
		// higher levels : average
		for (int l=1;l<levels;l++) {
			for (int x=0;x<nx[l];x++) for (int y=0;y<ny[l];y++) for (int z=0;z<nz[l];z++) {
				xM = 2*x; xP = 2*x+1;
				yM = 2*y; yP = 2*y+1;
				zM = 2*z; zP = 2*z+1;
				
                for (int k=0;k<members;k++) {
                    mem[l][x][y][z][k] = 0.125f*( mem[l-1][xM][yM][zM][k]+mem[l-1][xP][yM][zM][k]
                                                 +mem[l-1][xM][yP][zM][k]+mem[l-1][xP][yP][zM][k]
                                                 +mem[l-1][xM][yM][zP][k]+mem[l-1][xP][yM][zP][k]
                                                 +mem[l-1][xM][yP][zP][k]+mem[l-1][xP][yP][zP][k] );
                }
            }
		}
	}//buildMembershipPyramid
    
    /** propagate the transform from level l1 to level l2 */
    private final void propagateTransform(int l1, int l2) {
        if (l2>l1) {
            // going up in scale
            for (int l=l1+1;l<=l2;l++) {
                for (int d=0;d<3;d++) {
                    // rotation : no scaling (same origin too)
                    transform[l][d] = transform[l-1][d];
                    // translation : scale change
                    transform[l][d+3] = 0.5f*transform[l-1][d+3];
                }
            }
        } else {
            // going down in scale
            for (int l=l2;l<=l1-1;l++) {
                for (int d=0;d<3;d++) {
                    // rotation : no scaling (same origin too)
                    transform[l][d] = transform[l+1][d];
                    // translation : scale change
                    transform[l][d+3] = 2.0f*transform[l+1][d+3];
                }
            }
        }
    }
    
    /**
	 * compute the centroids given the membership functions
	 * for a scale l (only update the clusters)
	 */
    final public void computeCentroids(int l) {
        int x,y,z,k;
        float num,den,imgT;
        float xT,yT,zT;
		RotationMatrix  rotation;  
		
        rotation = new RotationMatrix();
        rotation.setParameters(transform[l][0],transform[l][1],transform[l][2]);
        
        for (k=0;k<clusters;k++) {
            num = 0;
            den = 0;
            for (x=0;x<nx[l];x++) for (y=0;y<ny[l];y++) for (z=0;z<nz[l];z++) {
                // compute the local position: X' = RX+T
				// with scaling!
				xT = (rotation.getMatrix(0,0)*x*rx+rotation.getMatrix(0,1)*y*ry+rotation.getMatrix(0,2)*z*rz+transform[l][3])/rx;
				yT = (rotation.getMatrix(1,0)*x*rx+rotation.getMatrix(1,1)*y*ry+rotation.getMatrix(1,2)*z*rz+transform[l][4])/ry;
				zT = (rotation.getMatrix(2,0)*x*rx+rotation.getMatrix(2,1)*y*ry+rotation.getMatrix(2,2)*z*rz+transform[l][5])/rz;
				
                // interpolated values
                imgT = ImageFunctions.linearInterpolation(images[l],maskVal,xT,yT,zT,nx[l],ny[l],nz[l]);
                    
				num += mems[l][x][y][z][k]*mems[l][x][y][z][k]*imgT;
				den += mems[l][x][y][z][k]*mems[l][x][y][z][k];
            }
            if (den>0.0) {
                centroids[k] = num/den;
            } else {
                centroids[k] = 0.0f;
            }
        }
        if (debug) {
			MedicUtilPublic.displayMessage("centroids: ("+centroids[0]);
			for (k=1;k<clusters;k++) 
				MedicUtilPublic.displayMessage(", "+centroids[k]);
			MedicUtilPublic.displayMessage(")\n");
		}
		rotation = null;
        return;
    } // computeCentroids
    
    /**
	 * compute the cost function given the parameters
	 * for a scale l
	 */
    final public float computeCostFunction(int l) {
        int x,y,z,k;
        float cost,num,den,imgT;
        float xT,yT,zT;
		RotationMatrix  rotation;  
		
        rotation = new RotationMatrix();
        rotation.setParameters(transform[l][0],transform[l][1],transform[l][2]);
        
        num = 0;
		den = 0;
        for (k=0;k<clusters;k++) {
            for (x=0;x<nx[l];x++) for (y=0;y<ny[l];y++) for (z=0;z<nz[l];z++) {
                // compute the local position: X' = RX+T
				// with scaling!
				xT = (rotation.getMatrix(0,0)*x*rx+rotation.getMatrix(0,1)*y*ry+rotation.getMatrix(0,2)*z*rz+transform[l][3])/rx;
				yT = (rotation.getMatrix(1,0)*x*rx+rotation.getMatrix(1,1)*y*ry+rotation.getMatrix(1,2)*z*rz+transform[l][4])/ry;
				zT = (rotation.getMatrix(2,0)*x*rx+rotation.getMatrix(2,1)*y*ry+rotation.getMatrix(2,2)*z*rz+transform[l][5])/rz;
				
                // interpolated values
                imgT = ImageFunctions.linearInterpolation(images[l],maskVal,xT,yT,zT,nx[l],ny[l],nz[l]);
                    
				num += mems[l][x][y][z][k]*mems[l][x][y][z][k]*(imgT-centroids[k])*(imgT-centroids[k]);
				den ++;
            }
        }
		cost = num/den;
        if (debug) MedicUtilPublic.displayMessage("cost function: "+cost+"\n");
		
		rotation = null;
        return cost;
     } // computeCostFunction
    
    /**
	 * compute the image position given the membership functions
	 * for a given level l
     * performs only one iteration
	 */
    final public void registerImageToSegmentation(int l) {
        int x,y,z,k,i,j;
        float vec,mat,dp, res,val;
        float xT,yT,zT;
        float dxT,dyT,dzT,imgT;
        float[][]       dRa,dRb,dRc;
        float[]         dimg = new float[6];
        JamaMatrix          regMat,regVec,regParam;
		RotationMatrix  rotation;

        regMat = new JamaMatrix(6,6,0.0f);
        regVec = new JamaMatrix(6,1,0.0f);
        rotation = new RotationMatrix();
       
		res = 0.0f;
		
        // set up rotation parameters
        rotation.setParameters(transform[l][0],transform[l][1],transform[l][2]);
        dRa = rotation.derivatives(1.0f, 0.0f, 0.0f);
        dRb = rotation.derivatives(0.0f, 1.0f, 0.0f);
        dRc = rotation.derivatives(0.0f, 0.0f, 1.0f);
        
        for (x=0;x<nx[l];x++) for (y=0;y<ny[l];y++) for (z=0;z<nz[l];z++) {
            // compute the local position: X' = RX+T
            xT = (rotation.getMatrix(0,0)*x*rx+rotation.getMatrix(0,1)*y*ry+rotation.getMatrix(0,2)*z*rz+transform[l][3])/rx;
            yT = (rotation.getMatrix(1,0)*x*rx+rotation.getMatrix(1,1)*y*ry+rotation.getMatrix(1,2)*z*rz+transform[l][4])/ry;
            zT = (rotation.getMatrix(2,0)*x*rx+rotation.getMatrix(2,1)*y*ry+rotation.getMatrix(2,2)*z*rz+transform[l][5])/rz;

            // compute interpolated values
            imgT = ImageFunctions.linearInterpolation(images[l],maskVal,xT,yT,zT,nx[l],ny[l],nz[l]);
            dxT = ImageFunctions.linearInterpolation(dx[l],0.0f,xT,yT,zT,nx[l],ny[l],nz[l]);
            dyT = ImageFunctions.linearInterpolation(dy[l],0.0f,xT,yT,zT,nx[l],ny[l],nz[l]);
            dzT = ImageFunctions.linearInterpolation(dz[l],0.0f,xT,yT,zT,nx[l],ny[l],nz[l]);
                
            // factor : clusters
            vec = 0.0f; mat = 0.0f;
            for (k=0;k<clusters;k++) {
                vec += mems[l][x][y][z][k]*mems[l][x][y][z][k]*(imgT-centroids[k]);
                mat += mems[l][x][y][z][k]*mems[l][x][y][z][k];
            }
            // outliers
            for (k=clusters;k<classes;k++) {
                vec += mems[l][x][y][z][k]*mems[l][x][y][z][k]*outlier;
                mat += mems[l][x][y][z][k]*mems[l][x][y][z][k];
            }                
            // masking 
            vec += mems[l][x][y][z][classes]*mems[l][x][y][z][classes]*(imgT-maskVal);
            mat += mems[l][x][y][z][classes]*mems[l][x][y][z][classes];
									
            // compute the derivative vector:
            // dImg/dX (XT) dX/dParam (XT)
            dimg[0] = dxT*(dRa[0][0]*xT*rx + dRa[0][1]*yT*ry + dRa[0][2]*zT*rz)/rx
                     +dyT*(dRa[1][0]*xT*rx + dRa[1][1]*yT*ry + dRa[1][2]*zT*rz)/ry
                     +dzT*(dRa[2][0]*xT*rx + dRa[2][1]*yT*ry + dRa[2][2]*zT*rz)/rz;

            dimg[1] = dxT*(dRb[0][0]*xT*rx + dRb[0][1]*yT*ry + dRb[0][2]*zT*rz)/rx
                     +dyT*(dRb[1][0]*xT*rx + dRb[1][1]*yT*ry + dRb[1][2]*zT*rz)/ry
                     +dzT*(dRb[2][0]*xT*rx + dRb[2][1]*yT*ry + dRb[2][2]*zT*rz)/rz;

            dimg[2] = dxT*(dRc[0][0]*xT*rx + dRc[0][1]*yT*ry + dRc[0][2]*zT*rz)/rx
                     +dyT*(dRc[1][0]*xT*rx + dRc[1][1]*yT*ry + dRc[1][2]*zT*rz)/ry
                     +dzT*(dRc[2][0]*xT*rx + dRc[2][1]*yT*ry + dRc[2][2]*zT*rz)/rz;
                
            dimg[3] = dxT/rx;
            dimg[4] = dyT/ry;
            dimg[5] = dzT/rz;
                
            // assemble everything
            for (i=0;i<6;i++) for (j=0;j<6;j++)
                regMat.set(i,j, regMat.get(i,j) + mat*dimg[i]*dimg[j] );
            for (i=0;i<6;i++)
                regVec.set(i,0, regVec.get(i,0) + vec*dimg[i]);                
        }
        // compute the new transformation
        regParam = regMat.solve(regVec);
        
        // update the transform
        // note: one could use SOR-like techniques (faster, less stable)
		// transform <- transform - 1.95*regVec
        for (i=0;i<6;i++)
            transform[l][i] -= regParam.get(i, 0); 
		
        if (debug) MedicUtilPublic.displayMessage("transform: ("+transform[l][0]+", "+transform[l][1]+", "+
                                            transform[l][2]+", "+transform[l][3]+", "+
                                            transform[l][4]+", "+transform[l][5]+")\n");

		rotation = null;
		regParam = null;
		regMat = null;
		regVec = null;

        return;
    } // registerImageToClassification
    
    /** 
	 *	runs a multigrid step : starts at level l, go to the top, come back
	 */
	public final void runMultigridStep(int l0) {        
        // start from l0 : suppose the transform is properly set
        if (debug) MedicUtilPublic.displayMessage("up ");
        for (int l=l0;l<levels-1;l++) {
            computeCentroids(l);
            for (int n=0;n<Nfirst;n++) registerImageToSegmentation(l);
            propagateTransform(l,l+1);
        }
        // top level
        if (debug) MedicUtilPublic.displayMessage("top ");
        computeCentroids(levels-1);
        for (int n=0;n<Ntop;n++) registerImageToSegmentation(levels-1);
        // going down to l0
        if (debug) MedicUtilPublic.displayMessage("down ");
        for (int l=levels-2;l>=l0;l--) {
            propagateTransform(l+1,l);
            computeCentroids(l);
            for (int n=0;n<Nlast;n++) registerImageToSegmentation(l);
        }
        if (debug) MedicUtilPublic.displayMessage("\n");
    }//runMultigridStep
    
    /** 
	 *	runs the full multigrid algorithm 
	 */
	public final void runFullMultigrid() {        
        // start from the top
        if (debug) MedicUtilPublic.displayMessage("multigrid : start: ");
        runMultigridStep(levels-1);
        // progress down
        for (int l=levels-2;l>=0;l--) {
            propagateTransform(l+1,l);
            if (debug) MedicUtilPublic.displayMessage("level "+l+": ");
            runMultigridStep(l);
        }
		transformToMatrix(0);
    }//runFullMultigrid

    /** 
	 *	runs the gaussian pyramid algorithm
	 */
	public final void runGaussianPyramid() {        
        // top level
        computeCentroids(levels-1);
        for (int n=0;n<Ntop;n++) registerImageToSegmentation(levels-1);
        // going down to 0
        for (int l=levels-2;l>=0;l--) {
            propagateTransform(l+1,l);
            computeCentroids(l);
            for (int n=0;n<Nlast;n++) registerImageToSegmentation(l);
        }
		transformToMatrix(0);
    }//runGaussianPyramid
    
    /** 
	 *	returns the transformed image
	 */
	public final float[][][] exportTransformedImage() {
		int 	x,y,z,t,k,best;
		float[][][]	img = new float[nx[0]][ny[0]][nz[0]];
		
        RotationMatrix rotation = new RotationMatrix();
        float xT,yT,zT;
       
        // set up rotation parameters
        rotation.setParameters(transform[0][0],transform[0][1],transform[0][2]);
        for (x=0;x<nx[0];x++) for (y=0;y<ny[0];y++) for (z=0;z<nz[0];z++) {
            // compute the local position: X' = RX+T
            xT = (rotation.getMatrix(0,0)*x*rx+rotation.getMatrix(0,1)*y*ry+rotation.getMatrix(0,2)*z*rz+transform[0][3])/rx;
            yT = (rotation.getMatrix(1,0)*x*rx+rotation.getMatrix(1,1)*y*ry+rotation.getMatrix(1,2)*z*rz+transform[0][4])/ry;
            zT = (rotation.getMatrix(2,0)*x*rx+rotation.getMatrix(2,1)*y*ry+rotation.getMatrix(2,2)*z*rz+transform[0][5])/rz;

            // compute interpolated values
            img[x][y][z] = ImageFunctions.linearInterpolation(images[0],xT,yT,zT,nx[0],ny[0],nz[0]);
		}
		return img;
	} // exportTransformedImage

    /** 
	 *	returns the transformed image
	 */
	public final byte[][][] exportTransformedByteImage() {
		int 	x,y,z,t,k,best;
		byte[][][]	img = new byte[nx[0]][ny[0]][nz[0]];
		
        RotationMatrix rotation = new RotationMatrix();
        float xT,yT,zT;
       
        // set up rotation parameters
        rotation.setParameters(transform[0][0],transform[0][1],transform[0][2]);
        for (x=0;x<nx[0];x++) for (y=0;y<ny[0];y++) for (z=0;z<nz[0];z++) {
            // compute the local position: X' = RX+T
            xT = (rotation.getMatrix(0,0)*x*rx+rotation.getMatrix(0,1)*y*ry+rotation.getMatrix(0,2)*z*rz+transform[0][3])/rx;
            yT = (rotation.getMatrix(1,0)*x*rx+rotation.getMatrix(1,1)*y*ry+rotation.getMatrix(1,2)*z*rz+transform[0][4])/ry;
            zT = (rotation.getMatrix(2,0)*x*rx+rotation.getMatrix(2,1)*y*ry+rotation.getMatrix(2,2)*z*rz+transform[0][5])/rz;

            // compute interpolated values
            img[x][y][z] = (byte)ImageFunctions.linearInterpolation(images[0],xT,yT,zT,nx[0],ny[0],nz[0]);
		}
		return img;
	} // exportTransformedImage

    /** 
	 *	returns the transform in the Mipav transformation matrix style
	 */
	public final double[][] convertTransform(int t) {
		int 	x,y,z,k,best;
		double[][]	mat = new double[4][4];
		
        RotationMatrix rotation = new RotationMatrix();
        
        // set up rotation parameters
        rotation.setParameters(transform[t][0],transform[t][1],transform[t][2]);
        
		// compute the inverse transformation matrix (image tp classification)
		// R0 = R^t
		for (int i=0;i<3;i++) for (int j=0;j<3;j++) {
			mat[i][j] = (double)rotation.getMatrix(j,i);
		}
		// T0 = -R^t T
        // warning: TransMatrix uses an awful convention where 
		for (int i=0;i<3;i++) {
			mat[i][3] = 0.0;
            for (int j=0;j<3;j++) {
				mat[i][3] += -mat[i][j]*(double)transform[t][j+3];
			}
            //mat[i][3] = -(double)transform[t][i+3];
		}
		// bottom line
		for (int i=0;i<3;i++) mat[3][i] = 0.0;
		mat[3][3] = 1.0;
		
		return mat;
	} // exportTransform

    /** 
	 *	returns the transform in the Mipav transformation matrix style
	 */
	public final double[][] convertTransformMatrix() {
		int 	x,y,z,k,best;
		double[][]	mat = new double[4][4];
		       
		// compute the inverse transformation matrix (image tp classification)
		// R0 = R^t
		for (int i=0;i<3;i++) for (int j=0;j<3;j++) {
			mat[i][j] = (double)transformMatrix[j][i];
		}
		// T0 = -R^t T
        // warning: TransMatrix uses an awful convention where 
		for (int i=0;i<3;i++) {
			mat[i][3] = 0.0;
            for (int j=0;j<3;j++) {
				mat[i][3] += -mat[i][j]*(double)transformMatrix[j][3];
			}
		}
		// bottom line
		for (int i=0;i<3;i++) mat[3][i] = 0.0;
		mat[3][3] = 1.0;
		
		return mat;
	} // convertTransformMatrix
}
