package edu.jhu.ece.iacl.algorithms.graphics.edit;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.PriorityQueue;

import javax.vecmath.Point2d;
import javax.vecmath.Point3f;

import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.SurfaceIntersector;
import edu.jhu.ece.iacl.algorithms.graphics.locator.sphere.SphericalMapLocator;
import edu.jhu.ece.iacl.algorithms.graphics.surf.HierarchicalSurface;
import edu.jhu.ece.iacl.algorithms.graphics.surf.ProgressiveSurface;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;

/**
 * Remesh surfaces using spherical maps. The resulting surface will have a
 * vertex count corresponding to the sum of vertices on both surfaces.
 * 
 * @author Blake Lucas
 * 
 */
public class RemeshSurfacesUsingSphericalMap extends ProgressiveSurface {
	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.6 $");
	}

	protected EmbeddedSurface source;
	protected EmbeddedSurface target;
	protected EmbeddedSurface remeshedSource;
	protected EmbeddedSurface remeshedTarget;
	protected int offset = 0;

	/**
	 * Constructor
	 * 
	 * @param parent
	 *            parent calculation
	 * @param source
	 *            source surface
	 * @param target
	 *            target surface
	 * @param offset
	 *            offset into vertex data to find spherical coordinates
	 */
	public RemeshSurfacesUsingSphericalMap(AbstractCalculation parent,
			EmbeddedSurface source, EmbeddedSurface target, int offset) {
		super(parent);
		this.source = source;
		this.target = target;
		this.offset = offset;
		setLabel("Remesh Surfaces");
	}

	/**
	 * Constructor
	 * 
	 * @param source
	 *            source surface
	 * @param target
	 *            target surface
	 */
	public RemeshSurfacesUsingSphericalMap(EmbeddedSurface source,
			EmbeddedSurface target) {
		super();
		this.source = source;
		this.target = target;
		setLabel("Remesh Surfaces");
	}

	/**
	 * Heap metric based on the minimum of the max edge length on the source and
	 * target surface.
	 */
	public double heapMetric(int v1) {
		int[] nbrs = neighborVertexVertexTable[v1];
		// If this a degenerate point, promote to top of queue
		// Degenerate points should be removed first
		double minEdgeLength = 1E30;
		int v2;
		if (remeshedTarget.getVertexDataAtOffset(v1, 0) == -1) {
			return -2;
		}
		for (int i = 0; i < nbrs.length; i++) {
			v2 = nbrs[i];
			/*
			 * double emlen=remeshedTarget.getPointAtOffset(v1,
			 * offset).distance(remeshedTarget.getPointAtOffset(v2, offset));
			 * if(emlen<MIN_EDGE_LENGTH){ return -1; } else {
			 */
			minEdgeLength = Math.min(minEdgeLength, Math.max(remeshedSource
					.getVertex(v1).distance(remeshedSource.getVertex(v2)),
					remeshedTarget.getVertex(v1).distance(
							remeshedTarget.getVertex(v2))));
			// }
		}
		return minEdgeLength;
	}

	/**
	 * Return list of vertex neighbors sorted by the max of the distance between
	 * neighbors on the source and target surface.
	 */
	protected int[] getSortedNeighbors(int v1) {
		double d;
		int[] nbrs = neighborVertexVertexTable[v1];
		SimpleVertexIndexed[] vd = new SimpleVertexIndexed[nbrs.length];
		int v2;
		for (int i = 0; i < nbrs.length; i++) {
			v2 = nbrs[i];
			if (remeshedTarget.getVertexDataAtOffset(v2, 0) == -1) {
				d = -2;
			} else {
				// double emlen=remeshedTarget.getPointAtOffset(v1,
				// offset).distance(remeshedTarget.getPointAtOffset(v2,
				// offset));
				// if(emlen<MIN_EDGE_LENGTH){
				// d=-1;
				// } else {
				d = Math.max(remeshedSource.getVertex(v1).distance(
						remeshedSource.getVertex(v2)), remeshedTarget
						.getVertex(v1).distance(remeshedTarget.getVertex(v2)));
				// }
			}
			vd[i] = new SimpleVertexIndexed(nbrs[i], d);
		}
		Arrays.sort(vd);
		int newNbrs[] = new int[nbrs.length];
		for (int i = 0; i < newNbrs.length; i++) {
			newNbrs[i] = vd[i].v;
		}
		return newNbrs;
	}

	/**
	 * Remap surfaces using spherical map.
	 */
	public void remapSphereBackward() {
		SphericalMapLocator sourceLocator = new SphericalMapLocator(this,
				source, offset);
		remeshedSource = target.clone();
		remeshedSource.setName(source.getName() + "_mapped");
		int vertTargetCount = target.getVertexCount();
		setTotalUnits((vertTargetCount));
		Point3f sp;
		for (int i = 0; i < vertTargetCount; i++) {
			Point3f p = target.getPointAtOffset(i, offset);
			sp = sourceLocator.locatePoint(p);
			remeshedSource.setVertex(i, sp);
		}
	}

	/**
	 * Remap surfaces using point map.
	 */
	public void remapPointBackward() {
		SurfaceIntersector sourceIntersector = new SurfaceIntersector(this, 12,
				source);
		remeshedSource = target.clone();
		remeshedSource.setName(source.getName() + "_mapped");
		int vertTargetCount = target.getVertexCount();
		Point3f sp;
		for (int i = 0; i < vertTargetCount; i++) {
			Point3f p = target.getPointAtOffset(i, offset);
			double d = sourceIntersector.distance(p);
			sp = sourceIntersector.getLastIntersectionPoint();
			remeshedSource.setVertex(i, sp);
		}
	}

	public void remesh() {
		SphericalMapLocator targetLocator = new SphericalMapLocator(this,
				target, offset);
		int vertSourceCount = source.getVertexCount();
		int vertTargetCount = target.getVertexCount();
		Point3f tp, sp;
		HierarchicalSurface hierarchialTarget = new HierarchicalSurface(target);
		setTotalUnits((vertSourceCount + vertTargetCount));
		for (int i = 0; i < vertSourceCount; i++) {
			// System.out.println("TARGET "+i+"/"+vertSourceCount);
			// Get embedded point on target
			Point3f p = source.getPointAtOffset(i, offset);
			// locate point on target
			tp = targetLocator.locatePoint(p);
			sp = source.getVertex(i);
			hierarchialTarget.insert(tp, new double[] { p.x, p.y, p.z },
					targetLocator.getLastFaceId());
			incrementCompletedUnits();
		}
		remeshedTarget = hierarchialTarget.getSurface(true);
		markCompleted();
		remeshedTarget.setName(target.getName() + "_mapped");
		int remeshedTargetVertCount = remeshedTarget.getVertexCount();
		targetLocator = null;
		double[][] sphereCoords = new double[remeshedTargetVertCount][2];
		for (int id = 0; id < remeshedTargetVertCount; id++) {
			Point3f p = remeshedTarget.getPointAtOffset(id, 0);
			Point2d r = GeometricUtilities.toUnitSpherical(p);
			sphereCoords[id][0] = Math.abs(r.x / Math.PI);
			sphereCoords[id][1] = Math.abs(r.y / Math.PI);
		}
		remeshedTarget.setTextureCoordinates(sphereCoords);
		System.gc();
		SphericalMapLocator sourceLocator = new SphericalMapLocator(this,
				source, offset);
		HierarchicalSurface hierarchialSource = new HierarchicalSurface(source);
		for (int i = 0; i < vertTargetCount; i++) {
			// Get embedded point on target
			Point3f p = target.getPointAtOffset(i, offset);
			// locate point on target
			sp = sourceLocator.locatePoint(p);
			tp = target.getVertex(i);
			hierarchialSource.insert(sp, new double[] { p.x, p.y, p.z },
					sourceLocator.getLastFaceId());
			incrementCompletedUnits();
		}
		remeshedSource = hierarchialSource.getSurface(false);
		remeshedSource.setName(source.getName() + "_mapped");
		sphereCoords = new double[remeshedTargetVertCount][2];
		for (int id = 0; id < remeshedTargetVertCount; id++) {
			Point3f p = remeshedSource.getPointAtOffset(id, 0);
			Point2d r = GeometricUtilities.toUnitSpherical(p);
			sphereCoords[id][0] = Math.abs(r.x / Math.PI);
			sphereCoords[id][1] = Math.abs(r.y / Math.PI);
		}
		remeshedSource.setTextureCoordinates(sphereCoords);
		markCompleted();
		System.gc();
	}

	protected void locateNeighbors(int id, int endIndex) {
		PriorityQueue<Integer> queue = new PriorityQueue<Integer>();
		queue.add(id);
		LinkedList<Integer> boundary = new LinkedList<Integer>();
		byte[] visited = new byte[endIndex];
		visited[id] = 1;
		int next;
		for (int nbr : neighborVertexVertexTable[id]) {
			System.out.println("NBR " + nbr);
		}
		while (queue.size() > 0) {
			next = queue.remove();
			int[] nbrs = neighborVertexVertexTable[next];
			for (int nbr : nbrs) {
				if (nbr >= endIndex) {
					if (!boundary.contains(nbr))
						boundary.add(nbr);
				} else if (visited[nbr] == 0) {
					queue.add(nbr);

					visited[nbr] = 1;

				}
			}
		}
		System.out.println("NODE " + id + " NBRS " + boundary.size() + " "
				+ boundary.subList(0, Math.min(100, boundary.size())));
	}

	/*
	 * protected void insertSphericalPoint(EdgeCollapse vs) { Point3f p =
	 * surf.getVertex(vs.v1); Point3f oldp = new Point3f(p); Matrix3d R =
	 * GeomUtil.rotateInverseMatrix3d(Math.atan2(p.y, p.x), Math .acos(p.z) -
	 * Math.PI / 2); int[] nbrs2 = neighborVertexVertexTable[vs.v2]; int[] nbrs1
	 * = neighborVertexVertexTable[vs.v1]; Point2d[] pnbrs2 = new
	 * Point2d[nbrs2.length]; Point2d[] pnbrs1 = new Point2d[nbrs1.length]; for
	 * (int i = 0; i < nbrs1.length; i++) { if (nbrs1[i] == vs.v2) { pnbrs1[i] =
	 * null; continue; } p = surf.getVertex(nbrs1[i]); p =
	 * GeomUtil.multMatrix(R, p); pnbrs1[i] = GeomUtil.sphereToStereo(p); } for
	 * (int i = 0; i < nbrs2.length; i++) { if (nbrs2[i] == vs.v1) { pnbrs2[i] =
	 * null; continue; } p = surf.getVertex(nbrs2[i]); p =
	 * GeomUtil.multMatrix(R, p); pnbrs2[i] = GeomUtil.sphereToStereo(p); } //
	 * Find correct line direction Vector2d sedge1 = new
	 * Vector2d(pnbrs2[find(vs.v2, vs.nbr1)]); Vector2d sedge2 = new
	 * Vector2d(pnbrs2[find(vs.v2, vs.nbr2)]); if (sedge1.length() <=
	 * MIN_EDGE_LENGTH || sedge2.length() <= MIN_EDGE_LENGTH) { // Edge too
	 * small return; } sedge1.normalize(); sedge2.normalize(); Vector2d sedge3 =
	 * GeomUtil.slerp(sedge1, sedge2, 0.5); double crossprod =
	 * GeomUtil.crossProduct(sedge3, sedge1); if (crossprod == 0) { //
	 * Degenerate Edge return; } if (crossprod < 0) { // Negate if clockwise
	 * angle between sedge1 and sedge2 is greater // than 90 deg
	 * sedge3.negate(); } Point2d pt; double tmin = maxIntersectionTime(sedge3,
	 * pnbrs2); sedge3.negate(); double tmax = -maxIntersectionTime(sedge3,
	 * pnbrs1); sedge3.negate(); R.invert(); double bestExt = 0; double
	 * bestAngle = 0; if (tmin != 1E30) { for (double ext = 1E-10; ext < 1; ext
	 * += 0.005) { pt = new Point2d(sedge3); pt.scale(tmin * ext); p =
	 * GeomUtil.multMatrix(R, GeomUtil.stereoToSphere(pt));
	 * surf.setVertex(vs.v2, p); if (isNbhdWoundCorrectly(vs.v2)) { double
	 * minAngle = minAngle(vs.v2); if (minAngle >= bestAngle) { bestExt = ext;
	 * bestAngle = minAngle; } } } } // Point insertion did not work, revert to
	 * old point if (bestExt == 0) { surf.setVertex(vs.v2, oldp); } else { pt =
	 * new Point2d(sedge3); pt.scale(tmin * bestExt); p = GeomUtil.multMatrix(R,
	 * GeomUtil.stereoToSphere(pt)); surf.setVertex(vs.v2, p); } bestExt = 0;
	 * bestAngle = 0; if (tmax != -1E30) { for (double ext = 1E-10; ext < 1; ext
	 * += 0.005) { pt = new Point2d(sedge3); pt.scale(tmax * ext); p =
	 * GeomUtil.multMatrix(R, GeomUtil.stereoToSphere(pt));
	 * surf.setVertex(vs.v1, p); if (isNbhdWoundCorrectly(vs.v1)) { double
	 * minAngle = minAngle(vs.v1); if (minAngle >= bestAngle) { bestExt = ext;
	 * bestAngle = minAngle; } } } } // Point insertion did not work, revert to
	 * old point if (bestExt == 0) { surf.setVertex(vs.v1, oldp); } else { pt =
	 * new Point2d(sedge3); pt.scale(tmax * bestExt); p = GeomUtil.multMatrix(R,
	 * GeomUtil.stereoToSphere(pt)); surf.setVertex(vs.v1, p); } }
	 */

	public void remap() {
		SphericalMapLocator targetLocator = new SphericalMapLocator(this,
				target, offset);
		int vertSourceCount = source.getVertexCount();
		Point3f tp;
		Point3f[] pts = new Point3f[vertSourceCount];
		setTotalUnits(vertSourceCount);
		remeshedSource = source.clone();
		for (int i = 0; i < vertSourceCount; i++) {
			// Get embedded point on target
			Point3f p = source.getPointAtOffset(i, offset);
			// locate point on source
			tp = targetLocator.locatePoint(p);
			pts[i] = tp;
			incrementCompletedUnits();
		}
		remeshedTarget = new EmbeddedSurface(pts, source.getIndexCopy(),
				remeshedSource.getVertexData());
		remeshedTarget.setName(target.getName() + "_mapped");
		markCompleted();
		init(remeshedSource, false);
	}

	public boolean topologyCheck(int v1, int v2) {
		int[] nbrs1 = neighborVertexVertexTable[v1];
		int[] nbrs2 = neighborVertexVertexTable[v2];
		int nbrIndex1 = find(v1, v2);
		int nbrIndex2 = find(v2, v1);
		int v3;
		// Check that nodes do not have more than three vertexes
		for (int i = 2; i < nbrs1.length - 1; i++) {
			v3 = nbrs1[(i + nbrIndex1) % nbrs1.length];
			for (int j = 2; j < nbrs2.length - 1; j++) {
				if (v3 == nbrs2[(j + nbrIndex2) % nbrs2.length]) {
					return false;
				}
			}
		}
		return true;
	}

	protected void createSurfaces() {
		int vertCount = remeshedSource.getVertexCount();
		int vertIDs[] = new int[vertCount];
		int currentId = 0, n1, n2;
		for (int i = 0; i < vertCount; i++) {
			int len = neighborVertexVertexTable[i].length;
			if (len == 0) {
				vertIDs[i] = -1;
			} else {
				vertIDs[i] = currentId++;
			}
		}
		int ptsCount = currentId;
		ArrayList<Integer> indexes = new ArrayList<Integer>();
		for (int i = 0; i < vertCount; i++) {
			int[] nbrs = neighborVertexVertexTable[i];
			currentId = vertIDs[i];
			for (int j = 0; j < nbrs.length; j++) {

				n1 = vertIDs[nbrs[j]];
				n2 = vertIDs[nbrs[(j + 1) % nbrs.length]];
				/*
				 * if(remeshedSource.getPointAtOffset(i,
				 * offset).distance(remeshedSource.getPointAtOffset(nbrs[j],
				 * offset))<MIN_EDGE_LENGTH){ System.out.println("TOO SMALL
				 * "+i+" "+nbrs[j]+" "+remeshedSource.getPointAtOffset(i,
				 * offset).distance(remeshedSource.getPointAtOffset(nbrs[j],
				 * offset))); }
				 */
				// Only create new triangles if they include vertex not seen
				// before
				if (n1 > currentId && n2 > currentId) {
					// Add new triangle
					indexes.add(currentId);
					indexes.add(n2);
					indexes.add(n1);

				}
			}
		}
		int[] indexArray = new int[indexes.size()];
		Point3f[] sourcePts = new Point3f[ptsCount];
		Point3f[] targetPts = new Point3f[ptsCount];
		double[][] oldData = remeshedSource.getVertexData();
		double[][] newData = new double[ptsCount][oldData[0].length];
		// Copy indexes to array
		for (int i = 0; i < indexArray.length; i++) {
			indexArray[i] = indexes.get(i);
		}
		int id;
		// Copy subset of points to array
		for (int i = 0; i < vertCount; i++) {
			if (vertIDs[i] != -1) {
				id = vertIDs[i];
				sourcePts[id] = remeshedSource.getVertex(i);
				targetPts[id] = remeshedTarget.getVertex(i);
				newData[id] = oldData[i];
			}
		}
		remeshedSource = new EmbeddedSurface(sourcePts, indexArray, newData);
		remeshedSource.setName(source.getName() + "_mapped");
		remeshedTarget = new EmbeddedSurface(targetPts, indexArray, newData);
		remeshedTarget.setName(target.getName() + "_mapped");
	}

	public EmbeddedSurface getRemeshedSource() {
		return remeshedSource;
	}

	public EmbeddedSurface getRemeshedTarget() {
		return remeshedTarget;
	}
}
