package edu.jhu.ece.iacl.algorithms.graphics.edit;

import java.util.*;

import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

import com.sun.j3d.utils.geometry.GeometryInfo;
import com.sun.j3d.utils.geometry.Triangulator;

import edu.jhmi.rad.medic.libraries.NeighborConnectivity;
import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.IntersectorTriangle;
import edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree.KdSegment;
import edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree.KdTriangle;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.fiber.XYZ;
import edu.jhu.ece.iacl.jist.structures.geom.CurveCollection;
import edu.jhu.ece.iacl.jist.structures.geom.CurvePath;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedCurvePath;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.geom.MipavVOI;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface.Edge;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface.EdgeSplit;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface.Face;

import gov.nih.mipav.model.structures.VOI;
import gov.nih.mipav.model.structures.VOIContour;


// added by clara, 2012.4.15
class ITRIANGLE {
	int p1, p2, p3;
	ITRIANGLE() { ; }
}

class IEDGE {
	int p1, p2;
	IEDGE() { p1=-1; p2=-1; }
}


class QuicksortbyX  {
	private XYZ[] pXYZ;
	private int number;

	public void sort(XYZ[] p) {
		// Check for empty or null array
		if (p ==null || p.length==0){
			return;
		}
		this.pXYZ = p;
		number = p.length;
		quicksort(0, number - 1);
	}

	private void quicksort(int low, int high) {
		int i = low, j = high;
		// Get the pivot element from the middle of the list
		float pivot = pXYZ[low + (high-low)/2].x;

		// Divide into two lists
		while (i <= j) {
			// If the current value from the left list is smaller then the pivot
			// element then get the next element from the left list
			while (pXYZ[i].x < pivot) {
				i++;
			}
			// If the current value from the right list is larger then the pivot
			// element then get the next element from the right list
			while (pXYZ[j].x > pivot) {
				j--;
			}

			// If we have found a values in the left list which is larger then
			// the pivot element and if we have found a value in the right list
			// which is smaller then the pivot element then we exchange the
			// values.
			// As we are done we can increase i and j
			if (i <= j) {
				exchange(i, j);
				i++;
				j--;
			}
		}
		// Recursion
		if (low < j)
			quicksort(low, j);
		if (i < high)
			quicksort(i, high);
	}

	private void exchange(int i, int j) {
		XYZ temp = pXYZ[i];
		pXYZ[i] = pXYZ[j];
		pXYZ[j] = temp;
	}
}



	
/**
 * Cut surface along specified plane.
 * 
 * @author Blake Lucas
 * 
 */
public class SurfaceSlicer extends AbstractCalculation {
	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.5 $");
	}

	protected static double EPS = 1E-6;

	/**
	 * Cute surface in half along specified plane
	 * 
	 * @param surf
	 *            surface
	 * @param origin
	 *            point on plane
	 * @param normal
	 *            normal to plane
	 * @return left, right, and split contour
	 */
	public EmbeddedSurface[] cut(EmbeddedSurface surf, Point3f origin,
			Vector3f normal) {
		EmbeddedSurface split = labelCut(surf, null,origin, normal);
		split.setName(surf.getName() + "_cut");
		EmbeddedSurface[] surfs = cut(split, true);
		return new EmbeddedSurface[] { surfs[0], surfs[1], split };
	}
	public EmbeddedSurface[] cut(EmbeddedSurface surf, EmbeddedSurface refSurface,Point3f origin,
			Vector3f normal) {
		EmbeddedSurface split = labelCut(surf, refSurface,origin, normal);
		split.setName(refSurface.getName() + "_cut");

		EmbeddedSurface[] surfs = cut(split, true);
		return new EmbeddedSurface[] { surfs[0], surfs[1], split };
	}
	/**
	 * Label vertices as either above (1) below (-1) or on (0) the specified
	 * plane. This method assumes that the surface is an iso-surface and surface
	 * vertices were sampled exactly on the plane. Therefore, the surface can be
	 * split without insertion of additional vertices.
	 * 
	 * @param surf
	 *            Surface
	 * @param origin
	 *            point on surface
	 * @param normal
	 *            surface normal
	 */
	protected void labelIsoCut(EmbeddedSurface surf, Point3f origin,
			Vector3f normal) {
		int vertCount = surf.getVertexCount();
		Point3f p;
		Vector3f test = new Vector3f();
		double[][] signs = surf.getVertexData();
		// Points below plane are labeled -1, points above the plane are labeled
		// 1, and points on the plane are labeled 0
		for (int i = 0; i < vertCount; i++) {
			p = surf.getVertex(i);
			test.sub(p, origin);
			double val = test.dot(normal);
			int sgn = (int) ((Math.abs(val) <= EPS) ? 0 : Math.signum(val));
			signs[i][0] = sgn;
		}
	}

	/**
	 * Label vertices as either above (1) below (-1) or on (0) the specified
	 * plane and insert vertices so that there is a contour that lies exactly on
	 * the plane. There is some arbitrariness as to how vertices are inserted if
	 * there are already multiple vertices that lie exactly on the cut plane.
	 * 
	 * @param surf
	 *            surface
	 * @param refSurface
	 *            reference surface to relocate resulting split surface vertices
	 * @param origin
	 *            point on plane
	 * @param normal
	 *            normal to plane
	 * @return Labeled surface with vertices inserted along the cut plane.
	 */
	public static EmbeddedSurface labelCut(EmbeddedSurface surf,EmbeddedSurface refSurface,
			Point3f origin, Vector3f normal) {
		Edge[] edges = EmbeddedSurface.buildEdgeTable(surf);
		LinkedList<Point3f> splitPoints = new LinkedList<Point3f>();
		int vertCount = surf.getVertexCount();
		// Split edges where they intersect the cutting plane
		int[][] nbrTable = EmbeddedSurface.buildNeighborVertexVertexTable(surf,
				EmbeddedSurface.Direction.COUNTER_CLOCKWISE);
		for (int i = 0; i < edges.length; i++) {
			Edge e = edges[i];
			if (e.v1 < e.v2) {
				KdSegment s = new KdSegment(surf.getVertex(e.v1), surf
						.getVertex(e.v2));
				Point3f pint = s.intersectsPlane(origin, normal);
				if (pint != null) {
					e = new EdgeSplit(e, vertCount + splitPoints.size());
					edges[i] = e;
					splitPoints.add(pint);
				}
			}
		}
		Hashtable<Long, Edge> edgeHash = EmbeddedSurface.buildEdgeHash(edges);
		Face[] faces = EmbeddedSurface.buildFaceTable(surf, edges, edgeHash);
		EdgeSplit e1;
		Edge e2, e3;
		int v1, v2, v3;
		boolean isSplit;
		LinkedList<Face> newFaces = new LinkedList<Face>();
		// Build new surface that includes split edges
		for (Face f : faces) {
			isSplit = false;
			for (int i = 0; i < f.edges.length; i++) {
				if (f.edges[i] instanceof EdgeSplit) {
					e1 = (EdgeSplit) f.edges[i];
					e2 = f.edges[(i + 1) % 3];
					e3 = f.edges[(i + 2) % 3];
					int[] verts = f.getVertexes();
					v1 = verts[i];
					v2 = verts[(i + 1) % 3];
					v3 = verts[(i + 2) % 3];
					if ((e2 instanceof EdgeSplit) && !(e3 instanceof EdgeSplit)) {

						isSplit = true;

						newFaces.add(new Face(v1, e1.mid, v3, newFaces.size()));
						newFaces.add(new Face(v2, ((EdgeSplit) e2).mid, e1.mid,
								newFaces.size()));
						newFaces.add(new Face(v3, e1.mid, ((EdgeSplit) e2).mid,
								newFaces.size()));

					} else if ((e3 instanceof EdgeSplit)
							&& !(e2 instanceof EdgeSplit)) {

						isSplit = true;
						newFaces.add(new Face(v1, e1.mid, ((EdgeSplit) e3).mid,
								newFaces.size()));
						newFaces.add(new Face(v2, v3, e1.mid, newFaces.size()));
						newFaces.add(new Face(v3, ((EdgeSplit) e3).mid, e1.mid,
								newFaces.size()));

					} else if (!(e3 instanceof EdgeSplit)
							&& !(e2 instanceof EdgeSplit)) {
						isSplit = true;
						newFaces.add(new Face(v1, e1.mid, v3, newFaces.size()));
						newFaces.add(new Face(v2, v3, e1.mid, newFaces.size()));
					}
					break;
				}
			}
			if (!isSplit) {
				f.id = newFaces.size();
				newFaces.add(f);
			}
		}
		int[] indexes = new int[newFaces.size() * 3];
		double[][] signs = new double[vertCount + splitPoints.size()][1];
		Point3f[] points = new Point3f[signs.length];
		Point3f p;
		int sgn;
		Face f;
		Vector3f test = new Vector3f();
		// Points below plane are labeled -1, points above the plane are labeled
		// 1, and points on the plane are labeled 0
		int vid1 = -1;
		for (int i = 0; i < vertCount; i++) {
			p = surf.getVertex(i);
			test.sub(p, origin);
			double val = test.dot(normal);
			sgn = (int) Math.signum(val);
			signs[i][0] = sgn;
			points[i] = p;
		}
		int ii = 0;

		for (Point3f pt : splitPoints) {
			points[ii + vertCount] = pt;
			signs[ii + vertCount][0] = 0;
			ii++;
		}
		ii = 0;
		for (Face face : newFaces) {
			int[] vert = face.getVertexes();
			indexes[ii * 3] = vert[0];
			indexes[ii * 3 + 1] = vert[1];
			indexes[ii * 3 + 2] = vert[2];
			ii++;
		}
		EmbeddedSurface splitSurf = new EmbeddedSurface(points, indexes, signs);
		
		if(refSurface!=null){
			vertCount=splitSurf.getVertexCount();

			nbrTable=EmbeddedSurface.buildNeighborVertexVertexTable(splitSurf, EmbeddedSurface.Direction.COUNTER_CLOCKWISE);
			int origVertCount=refSurface.getVertexCount();
			System.out.println("SPLIT SURFACE "+splitSurf.getVertexCount()+" "+refSurface.getVertexCount());
			// Move inflated surface positions to reference surface positions
			for (int i = 0; i < vertCount; i++) {
				if (i >= origVertCount) {
					int count = 0;
					int[] nbrs=nbrTable[i];
					Point3f avgPt = new Point3f();
					for (int nbr : nbrs) {
						if (nbr < origVertCount) {
							count++;
							avgPt.add(refSurface.getVertex(nbr));
						}
					}
					avgPt.scale(1.0f / count);
					// For vertices that are along split, use average of neighbors
					splitSurf.setVertex(i, avgPt);
				} else {
					// Use reference surface positions
					splitSurf.setVertex(i, refSurface.getVertex(i));
				}
			}
			splitSurf.setName(refSurface.getName() + "_cut");
		} else {
			splitSurf.setName(surf.getName() + "_cut");
		}
		
		
		return splitSurf;
	}

	/**
	 * Find offset position of neighboring vertex around 1-ring vertex.
	 * 
	 * @param v1
	 *            pivot vertex
	 * @param v2
	 *            neighboring vertex
	 * @param vertexTable
	 *            vertex-vertex lookup table
	 * @return offset position of neighboring vertex
	 */
	protected static int find(int v1, int v2, int vertexTable[][]) {
		int[] nbrs = vertexTable[v1];
		for (int i = 0; i < nbrs.length; i++) {
			if (nbrs[i] == v2) {
				return i;
			}
		}
		return -1;
	}

	/**
	 * Cut labeled surface in half
	 * 
	 * @param surf
	 *            labeled surface with labels {-1,0,1}
	 * @param fillSurf
	 *            fill holes in surfaces after cutting the surface in half.
	 * @return return left surface, right surface, filled contour surface
	 */
	protected EmbeddedSurface[] cut(EmbeddedSurface surf, boolean fillSurf) {
		int totalCount = surf.getVertexCount();
		int faceCount = surf.getFaceCount();
		int[] lookupLeft = new int[totalCount];
		int[] lookupRight = new int[totalCount];
		double[][] signs = surf.getVertexData();
		ArrayList<Point3f> leftPoint = new ArrayList<Point3f>();
		ArrayList<Point3f> rightPoint = new ArrayList<Point3f>();
		ArrayList<Integer> leftIndex = new ArrayList<Integer>();
		ArrayList<Integer> rightIndex = new ArrayList<Integer>();
		ArrayList<Integer> leftLabel = new ArrayList<Integer>();
		ArrayList<Integer> rightLabel = new ArrayList<Integer>();
		// Split labeled surface into two surfaces
		Point3f p;
		int sgn;
		for (int i = 0; i < totalCount; i++) {
			p = surf.getVertex(i);
			sgn = (int) signs[i][0];
			if (sgn <= 0) {
				lookupLeft[i] = leftPoint.size();
				leftPoint.add(p);
				leftLabel.add(sgn);
			} else {
				lookupLeft[i] = -1;
			}
			if (sgn >= 0) {
				lookupRight[i] = rightPoint.size();
				rightPoint.add(p);
				rightLabel.add(sgn);
			} else {
				lookupRight[i] = -1;
			}
		}
		for (int i = 0; i < faceCount; i++) {
			int[] vert = new int[] { surf.getCoordinateIndex(i * 3),
					surf.getCoordinateIndex(i * 3 + 1),
					surf.getCoordinateIndex(i * 3 + 2) };
			if (lookupLeft[vert[0]] > -1 && lookupLeft[vert[1]] > -1
					&& lookupLeft[vert[2]] > -1) {
				leftIndex.add(lookupLeft[vert[0]]);
				leftIndex.add(lookupLeft[vert[1]]);
				leftIndex.add(lookupLeft[vert[2]]);
			}
			if (lookupRight[vert[0]] > -1 && lookupRight[vert[1]] > -1
					&& lookupRight[vert[2]] > -1) {
				rightIndex.add(lookupRight[vert[0]]);
				rightIndex.add(lookupRight[vert[1]]);
				rightIndex.add(lookupRight[vert[2]]);

			}
		}

		int[] rightIndexArray = new int[rightIndex.size()];
		for (int i = 0; i < rightIndexArray.length; i++)
			rightIndexArray[i] = rightIndex.get(i);

		int[] leftIndexArray = new int[leftIndex.size()];
		for (int i = 0; i < leftIndexArray.length; i++)
			leftIndexArray[i] = leftIndex.get(i);

		double[][] rightLabelArray = new double[rightLabel.size()][1];
		Point3f[] rightPointArray = new Point3f[rightPoint.size()];
		for (int i = 0; i < rightPointArray.length; i++) {
			rightPointArray[i] = rightPoint.get(i);
			rightLabelArray[i][0] = rightLabel.get(i);
		}

		double[][] leftLabelArray = new double[leftLabel.size()][1];
		Point3f[] leftPointArray = new Point3f[leftPoint.size()];
		for (int i = 0; i < leftPointArray.length; i++) {
			leftPointArray[i] = leftPoint.get(i);
			leftLabelArray[i][0] = leftLabel.get(i);
		}
		EmbeddedSurface rightSurf = new EmbeddedSurface(rightPointArray,
				rightIndexArray, rightLabelArray);
		rightSurf.setName(surf.getName() + "_right");
		if (fillSurf)
			rightSurf = fillHoles(rightSurf, -1);
		EmbeddedSurface leftSurf = new EmbeddedSurface(leftPointArray,
				leftIndexArray, leftLabelArray);
		leftSurf.setName(surf.getName() + "_left");
		if (fillSurf)
			leftSurf = fillHoles(leftSurf, 1);
		return new EmbeddedSurface[] { leftSurf, rightSurf, surf };
	}

	/**
	 * Locate closest point on surface along line directed in the z-axis that
	 * passes through point p.
	 * 
	 * @param mesh
	 *            surface
	 * @param p
	 *            point on line
	 * @return closest point on surface
	 */
	public static Point3f locateClosestPoint(EmbeddedSurface mesh, Point3f p) {
		int inCount = mesh.getIndexCount();
		IntersectorTriangle tri = null;
		int index = 0;

		Point3f pt;
		for (int i = 0; i < inCount / 3; i++) {
			tri = getTriangle(mesh, i);
			pt = tri.intersectionPoint(p, new Vector3f(0, 0, 1));
			if (pt != null)
				return pt;
			pt = tri.intersectionPoint(p, new Vector3f(0, 0, -1));
			if (pt != null)
				return pt;
		}
		if (index == 0) {
			System.err.println("Could not locate north pole from point " + p);
		}
		return null;
	}

	/**
	 * Locate closest vertex from two points.
	 * 
	 * @param mesh
	 *            surface
	 * @param p1
	 *            vertex 1
	 * @param p2
	 *            vertex 2
	 * @return vertex id
	 */
	public static int locateClosestVertex(EmbeddedSurface mesh, Point3f p1,
			Point3f p2) {
		Point3f pt = null;
		double minD = 1E10;
		double d;
		int index = -1;
		double[][] signs = mesh.getVertexData();
		for (int i = 0; i < mesh.getVertexCount(); i++) {
			if (signs[i][0] == 0) {
				pt = mesh.getVertex(i);
				d = pt.distance(p1) + pt.distance(p2);
				if (d < minD) {
					index = i;
					minD = d;
				}
			}
		}
		return index;
	}

	/**
	 * Triangle class for surface
	 * 
	 * @param surf
	 *            surface
	 * @param i
	 *            face id
	 * @return triangle class
	 */
	static public IntersectorTriangle getTriangle(EmbeddedSurface surf, int i) {
		IntersectorTriangle tri = new IntersectorTriangle(surf
				.getCoordinateIndex(3 * i), surf.getCoordinateIndex(3 * i + 1),
				surf.getCoordinateIndex(3 * i + 2), i, surf);
		return tri;
	}

	CurveCollection curves;
	MipavVOI voi;

	/**
	 * Cut contours for iso-surface along specifed normal direction
	 * 
	 * @param origSurf
	 *            surface
	 * @param normal
	 *            plane normal
	 * @param numSlices
	 *            number of slices
	 * @param sliceSpacing
	 *            slice spacing
	 * @param isIso
	 *            is iso-surface
	 */
	public void cutContours(EmbeddedSurface origSurf, Vector3f normal,
			int numSlices, int sliceSpacing, boolean isIso) {
		Vector3f v;
		curves = new CurveCollection();
		setTotalUnits(numSlices);
		int vertCount = origSurf.getVertexCount();
		double minV = Double.MAX_VALUE, maxV = Double.MIN_VALUE;
		voi = new MipavVOI(origSurf.getName(), numSlices);
		for (int i = 0; i < vertCount; i++) {
			double dot = GeometricUtilities.dot(normal, origSurf.getVertex(i));
			minV = Math.min(minV, dot);
			maxV = Math.max(maxV, dot);
		}
		origSurf.setVertexData(new double[origSurf.getVertexCount()][1]);
		EmbeddedSurface surf;
		for (int i = 0; i < numSlices; i++) {
			v = new Vector3f(normal);
			float slice = (float) (i * sliceSpacing);
			if (slice < minV || slice > maxV) {
				incrementCompletedUnits();
				continue;
			}
			v.scale(slice);

			Point3f origin = new Point3f(v);

			if (isIso) {
				labelIsoCut(origSurf, origin, normal);
				surf = origSurf;
			} else {
				surf = labelCut(origSurf, null,origin, normal);
			}
			contours(surf, curves, voi, slice);
			incrementCompletedUnits();
		}
		curves.setName(origSurf.getName() + "_contours");
		markCompleted();
	}

	/**
	 * Get contour curves
	 * 
	 * @return
	 */
	public CurveCollection getCurves() {
		return curves;
	}

	/**
	 * Get MIPAV VOIs
	 * 
	 * @return
	 */
	public MipavVOI getVOI() {
		return voi;
	}

	/**
	 * Cut cortical surface along corpus callosum
	 * 
	 * @param surf
	 *            surface
	 * @param pac
	 *            Anterior Commisure point
	 * @param ppc
	 *            Posterior Commisure point
	 * @param normal
	 *            plane normal
	 * @return left, right, and corpus callosum contour
	 */
	public EmbeddedSurface[] cutClosestContour(EmbeddedSurface surf,
			Point3f pac, Point3f ppc, Vector3f normal) {

		Point3f avg = new Point3f(0.5f * (pac.x + ppc.x),
				0.5f * (pac.y + ppc.y), 0.5f * (pac.z + ppc.z));
		surf = labelCut(surf, null,avg, normal);

		int startIndex = locateClosestVertex(surf, pac, ppc);
		// Only cut contour that is closest to origin point
		Edge[] edges = EmbeddedSurface.buildEdgeTable(surf);
		double[][] labels = surf.getVertexData();
		int vid1 = -1;
		int vid2 = -2;

		// Find edge that contains start point

		for (Edge e : edges) {
			if (labels[e.v1][0] == 0 || labels[e.v2][0] == 0) {
				if (e.v1 == startIndex || e.v2 == startIndex) {
					vid1 = e.v1;
					vid2 = e.v2;
					break;
				}
			}
		}
		if (vid1 == -1)
			return null;

		int[][] vertTable = EmbeddedSurface.buildNeighborVertexVertexTable(
				surf, EmbeddedSurface.Direction.COUNTER_CLOCKWISE);
		for (int i = 0; i < labels.length; i++) {
			if (labels[i][0] == 0)
				labels[i][0] = 3;
		}
		int current = vid1;
		boolean cont = false;
		boolean found = false;
		if (labels[current][0] == 3)
			labels[current][0] = 0;
		LinkedList<Integer> path = new LinkedList<Integer>();
		Vector3f test = new Vector3f();
		do {
			cont = true;
			path.add(current);
			found = false;
			for (int j = 0; j < vertTable[current].length; j++) {
				int nbr = vertTable[current][j];
				if (labels[nbr][0] == 3) {
					labels[nbr][0] = 0;
					current = nbr;
					found = true;
					break;
				}
			}

			if (!found) {
				for (int j = 0; j < vertTable[current].length; j++) {
					if (vertTable[current][j] == vid1) {
						cont = false;
						System.out.println("FOUND START " + vid1);
						break;
					}
				}
				if (cont) {
					double minPoint = 1E-4;
					int minIndex = -1;
					for (int j = 0; j < vertTable[current].length; j++) {
						int nbr = vertTable[current][j];
						if (labels[nbr][0] != 0) {
							Point3f p = surf.getVertex(nbr);
							test.sub(p, avg);
							double val = Math.abs(test.dot(normal));
							if (val < minPoint) {
								minIndex = nbr;
								minPoint = val;
							}
						}
					}
					if (minIndex == -1) {
						cont = false;
					} else {
						System.out
								.println("FOUND " + minPoint + " " + minIndex);
						current = minIndex;
						labels[current][0] = 0;
					}
				}
			}
		} while (cont);
		System.out.println("PATH LENGTH " + path.size());
		int vertCount = surf.getVertexCount();
		LinkedList<Integer> queue = new LinkedList<Integer>();
		int sgn = 1;
		// Look for neighbors around startPoint and breadth first label from
		// there

		int leftVid = -1, rightVid = -1;
		/*
		 * if (vid1 == -1) leftVid = vid1; if (vid1 == 1) rightVid = vid1; if
		 * (vid2 == -1) leftVid = vid2; if (vid2 == 1) rightVid = vid2;
		 */

		for (int vid : path) {
			leftVid = rightVid = -1;
			for (int i = 0; i < vertTable[vid].length; i++) {
				if (leftVid != -1 && rightVid != -1)
					break;
				if (labels[vertTable[vid][i]][0] == 1) {
					rightVid = vertTable[vid][i];
				}
				if (labels[vertTable[vid][i]][0] == -1) {
					leftVid = vertTable[vid][i];
				}
			}
			if (leftVid != -1 && rightVid != -1)
				break;
		}
		boolean allowChange = false;
		queue.add(rightVid);
		while (!queue.isEmpty()) {
			vid1 = queue.remove();
			labels[vid1][0] = 2;
			allowChange = true;
			for (int nid : vertTable[vid1]) {
				if (labels[nid][0] == 0 && vid1 != rightVid) {
					allowChange = false;
					break;
				}
			}
			if (allowChange) {
				for (int nid : vertTable[vid1]) {
					if (Math.abs(labels[nid][0]) == 1 || labels[nid][0] == 3) {
						labels[nid][0] = 2;
						queue.add(nid);
					}
				}
			}
		}
		queue.add(leftVid);
		while (!queue.isEmpty()) {
			vid1 = queue.remove();
			labels[vid1][0] = -2;
			allowChange = true;
			for (int nid : vertTable[vid1]) {
				if (labels[nid][0] == 0 && vid1 != leftVid) {
					allowChange = false;
					break;
				}
			}
			if (allowChange) {
				for (int nid : vertTable[vid1]) {
					if (Math.abs(labels[nid][0]) == 1 || labels[nid][0] == 3) {
						labels[nid][0] = -2;
						queue.add(nid);
					}
				}
			}
		}
		for (int i = 0; i < labels.length; i++) {
			if (labels[i][0] == 3)
				labels[i][0] = 0;
		}
		// return new EmbeddedSurface[]{surf,surf,surf};
		return cut(surf, true);

	}

	/**
	 * Cut contours for surface with labeled iso-contours and add them to a
	 * curve collection and MIPAV VOI
	 * 
	 * @param surf
	 *            labeled surface
	 * @param curves
	 *            curvature collection
	 * @param voi
	 *            MIPAV VOI
	 * @param label
	 *            contour label to identify contours that should be added to the
	 *            collection
	 */
	protected static void contours(EmbeddedSurface surf,
			CurveCollection curves, VOI voi, double label) {
		int[][] vertTable = EmbeddedSurface.buildNeighborVertexVertexTable(
				surf, EmbeddedSurface.Direction.COUNTER_CLOCKWISE);
		double[][] labels = surf.getVertexData();
		int current, count;
		boolean cont;
		Point3f pt;
		for (int i = 0; i < vertTable.length; i++) {
			// Check if vertexes are on the plane
			if (labels[i][0] == 0) {
				current = i;
				count = 1;
				labels[current][0] = 1;
				EmbeddedCurvePath curve = new EmbeddedCurvePath();
				VOIContour contour = new VOIContour(false);
				curve.add(surf.getVertex(current));
				do {
					cont = false;
					// find next vertex in contour
					for (int j = 0; j < vertTable[current].length; j++) {
						// Only examine labels that have not been traversed
						if (labels[vertTable[current][j]][0] == 0) {
							current = vertTable[current][j];
							labels[current][0] = 1;
							count++;
							cont = true;
							curve.add(pt = surf.getVertex(current));
							contour.addElement(pt.x, pt.y, pt.z);
							break;
						}
					}

				} while (cont);
				if (current == i)
					continue;
				curve.add(pt = surf.getVertex(current));
				contour.addElement(pt.x, pt.y, pt.z);
				curve.add(pt = surf.getVertex(i));
				contour.addElement(pt.x, pt.y, pt.z);
				curve.setValue(label);
				// voi.importCurve(contour, (short) label);
				voi.importCurve(contour);
				count++;
				curves.add(curve);

			} else {
				labels[i][0] /= Math.abs(labels[i][0]);
			}
		}
	}
	
	
	
	protected static EmbeddedSurface fillHoles(EmbeddedSurface surf, int bg) {
		int[][] vertTable = EmbeddedSurface.buildNeighborVertexVertexTable(
				surf, EmbeddedSurface.Direction.COUNTER_CLOCKWISE);
		double[][] labels = surf.getVertexData();
//		ArrayList<Integer> polySizes;

		ArrayList<Integer> finalIndexes = new ArrayList<Integer>();
//		ArrayList<Integer> indexes = new ArrayList<Integer>();
		for (int i = 0; i < surf.getIndexCount(); i++) {
			finalIndexes.add(surf.getCoordinateIndex(i));
		}
//		int current, count;
//		boolean cont;
		Point3f[] points = surf.getVertexCopy();
		
		// added by clara, 2012.4.15
		// find contour points
		ArrayList<Integer> contourVtxIds = new ArrayList<Integer>();
		int contourVtxNum = 0;
		for (int i = 0; i < vertTable.length; i++) {
			if (labels[i][0] == 0) {
				labels[i][0] = bg;
				contourVtxIds.add(i);
				contourVtxNum++;
			}
		}
		
		XYZ[] contourVtxCoords = new XYZ[contourVtxNum + 3];
		
        int c;
		for (int i=0; i<contourVtxNum; i++){
			c = contourVtxIds.get(i);
			contourVtxCoords[i] = new XYZ(points[c].x, points[c].y, points[c].z);
		}
		for (int i=0; i<3; i++){
			contourVtxCoords[contourVtxNum+i] = new XYZ(0,0,0);
		}
		
		QuicksortbyX sorter = new QuicksortbyX();
		sorter.sort(contourVtxCoords);
		
        // using Triangulate
		ITRIANGLE[]	contourTri 	= new ITRIANGLE[ contourVtxNum*3 ];

		for (int i=0; i<contourTri.length; i++)
			contourTri[i] = new ITRIANGLE();
		int ntri = Triangulate(contourVtxNum, contourVtxCoords, contourTri);
		for (int i = 0; i < ntri; i++) {
			finalIndexes.add(contourVtxIds.get(contourTri[i].p1));
			finalIndexes.add(contourVtxIds.get(contourTri[i].p2));
			finalIndexes.add(contourVtxIds.get(contourTri[i].p3));
		}
		

		
/*		// Blake's original
		for (int i = 0; i < vertTable.length; i++) {
			// Check if vertexes are on the plane
			if (labels[i][0] == 0) {
				polySizes = new ArrayList<Integer>();
				indexes = new ArrayList<Integer>();
				current = i;
				count = 1;
				labels[current][0] = bg;
				indexes.add(current);
				do {
					cont = false;
					// find next vertex in contour
					for (int j = 0; j < vertTable[current].length; j++) {
						// Only examine labels that have not been traversed
						if (labels[vertTable[current][j]][0] == 0) {
							current = vertTable[current][j];
							labels[current][0] = bg;
							count++;
							cont = true;
							indexes.add(current);
							break;
						}
					}

				} while (cont);
				// Triangulate contour
				if (current == i)
					continue;
				indexes.add(i);
				count++;
				polySizes.add(count);
				int[] indexArray = new int[indexes.size()];
				int[] polySizeArray = new int[polySizes.size()];
				for (int j = 0; j < polySizes.size(); j++)
					polySizeArray[j] = polySizes.get(j);
				for (int j = 0; j < indexes.size(); j++) {
					indexArray[indexes.size() - j - 1] = indexes.get(j);
					System.out.println("j = " + j + ", indexes.get(j) = " + indexes.get(j));
				}
				GeometryInfo gi = new GeometryInfo(GeometryInfo.POLYGON_ARRAY);

				gi.setCoordinates(points);
				gi.setStripCounts(polySizeArray);
				gi.setCoordinateIndices(indexArray);
				gi.setContourCounts(new int[] { polySizeArray.length });
				Triangulator tr = new Triangulator();
				tr.triangulate(gi);
				int m = 1;
				for (int index : gi.getCoordinateIndices()) {
					finalIndexes.add(index);
					System.out.println("m = " + m + ", index = " + index);
					m++;
				}
			} else {
				labels[i][0] /= Math.abs(labels[i][0]);
			}
		}
		int[] finalIndexArray = new int[finalIndexes.size()];
		for (int i = 0; i < finalIndexArray.length; i++) {
			finalIndexArray[i] = finalIndexes.get(i);
		}
*/

		int[] finalIndexArray = new int[finalIndexes.size()];
		for (int i = 0; i < finalIndexArray.length; i++) {
			finalIndexArray[i] = finalIndexes.get(i);
		}
		
		EmbeddedSurface surf2 = new EmbeddedSurface(points, finalIndexArray,
				surf.getVertexData());
		surf2.setName(surf.getName());
		return surf2;
	}

	public static float EPSILON = 0.000001f;

	/*
		Return TRUE if a point (xp,yp) is inside the circumcircle made up
		of the points (x1,y1), (x2,y2), (x3,y3)
		The circumcircle centre is returned in (xc,yc) and the radius r
		NOTE: A point on the edge is inside the circumcircle
	*/
	static boolean CircumCircle(
							float xp, float yp,
							float x1, float y1,
							float x2, float y2,
							float x3, float y3,
							/*float xc, float yc, float r*/
							XYZ circle
							)
	{
		float m1,m2,mx1,mx2,my1,my2;
		float dx,dy,rsqr,drsqr;
		float xc, yc, r;

		/* Check for coincident points */

		if ( Math.abs(y1-y2) < EPSILON && Math.abs(y2-y3) < EPSILON )
		{
			System.out.println("jist.plugins"+"\t"+"CircumCircle: Points are coincident.");
			return false;
		}

		if ( Math.abs(y2-y1) < EPSILON )
		{
			m2 = - (x3-x2) / (y3-y2);
			mx2 = (x2 + x3) / 2.0f;
			my2 = (y2 + y3) / 2.0f;
			xc = (x2 + x1) / 2.0f;
			yc = m2 * (xc - mx2) + my2;
		}
		else if ( Math.abs(y3-y2) < EPSILON )
		{
			m1 = - (x2-x1) / (y2-y1);
			mx1 = (x1 + x2) / 2.0f;
			my1 = (y1 + y2) / 2.0f;
			xc = (x3 + x2) / 2.0f;
			yc = m1 * (xc - mx1) + my1;
		}
		else
		{
			m1 = - (x2-x1) / (y2-y1);
			m2 = - (x3-x2) / (y3-y2);
			mx1 = (x1 + x2) / 2.0f;
			mx2 = (x2 + x3) / 2.0f;
			my1 = (y1 + y2) / 2.0f;
			my2 = (y2 + y3) / 2.0f;
			xc = (m1 * mx1 - m2 * mx2 + my2 - my1) / (m1 - m2);
			yc = m1 * (xc - mx1) + my1;
		}

		dx = x2 - xc;
		dy = y2 - yc;
		rsqr = dx*dx + dy*dy;
		r = (float)Math.sqrt(rsqr);

		dx = xp - xc;
		dy = yp - yc;
		drsqr = dx*dx + dy*dy;

		circle.x = xc;
		circle.y = yc;
		circle.z = r;

		return ( drsqr <= rsqr ? true : false );
	}

	/*
		Triangulation subroutine
		Takes as input NV vertices in array pxyz
		Returned is a list of ntri triangular faces in the array v
		These triangles are arranged in a consistent clockwise order.
		The triangle array 'v' should be malloced to 3 * nv
		The vertex array pxyz must be big enough to hold 3 more points
		The vertex array must be sorted in increasing x values say

		qsort(p,nv,sizeof(XYZ),XYZCompare);

		int XYZCompare(void *v1,void *v2)
		{
			XYZ *p1,*p2;
			p1 = v1;
			p2 = v2;
			if (p1->x < p2->x)
				return(-1);
			else if (p1->x > p2->x)
				return(1);
			else
				return(0);
		}
	*/

	static int Triangulate ( int nv, XYZ pxyz[], ITRIANGLE v[] )
	{
		boolean complete[] 		= null;
		IEDGE 	edges[] 		= null;
		int 	nedge 			= 0;
		int 	trimax, emax 	= 200;
		int 	status 			= 0;

		boolean	inside;
		//int 	i, j, k;
		float 	xp, yp, x1, y1, x2, y2, x3, y3, xc, yc, r;
		float 	xmin, xmax, ymin, ymax, xmid, ymid;
		float 	dx, dy, dmax;

		int		ntri			= 0;

		/* Allocate memory for the completeness list, flag for each triangle */
		trimax = 4*nv;
		complete = new boolean[trimax];
		for (int ic=0; ic<trimax; ic++) complete[ic] = false;

		/* Allocate memory for the edge list */
		edges = new IEDGE[emax];
		for (int ie=0; ie<emax; ie++) edges[ie] = new IEDGE();

		/*
		Find the maximum and minimum vertex bounds.
		This is to allow calculation of the bounding triangle
		*/
		xmin = pxyz[0].x;
		ymin = pxyz[0].y;
		xmax = xmin;
		ymax = ymin;
		for (int i=1;i<nv;i++)
		{
			if (pxyz[i].x < xmin) xmin = pxyz[i].x;
			if (pxyz[i].x > xmax) xmax = pxyz[i].x;
			if (pxyz[i].y < ymin) ymin = pxyz[i].y;
			if (pxyz[i].y > ymax) ymax = pxyz[i].y;
		}
		dx = xmax - xmin;
		dy = ymax - ymin;
		dmax = (dx > dy) ? dx : dy;
		xmid = (xmax + xmin) / 2.0f;
		ymid = (ymax + ymin) / 2.0f;

		/*
			Set up the supertriangle
			This is a triangle which encompasses all the sample points.
			The supertriangle coordinates are added to the end of the
			vertex list. The supertriangle is the first triangle in
			the triangle list.
		*/
		pxyz[nv+0].x = xmid - 2.0f * dmax;
		pxyz[nv+0].y = ymid - dmax;
		pxyz[nv+0].z = 0.0f;
		pxyz[nv+1].x = xmid;
		pxyz[nv+1].y = ymid + 2.0f * dmax;
		pxyz[nv+1].z = 0.0f;
		pxyz[nv+2].x = xmid + 2.0f * dmax;
		pxyz[nv+2].y = ymid - dmax;
		pxyz[nv+2].z = 0.0f;
		v[0].p1 = nv;
		v[0].p2 = nv+1;
		v[0].p3 = nv+2;
		complete[0] = false;
		ntri = 1;


		/*
			Include each point one at a time into the existing mesh
		*/
		for (int i=0;i<nv;i++) {

			xp = pxyz[i].x;
			yp = pxyz[i].y;
			nedge = 0;


			/*
				Set up the edge buffer.
				If the point (xp,yp) lies inside the circumcircle then the
				three edges of that triangle are added to the edge buffer
				and that triangle is removed.
			*/
			XYZ circle = new XYZ();
			for (int j=0;j<ntri;j++)
			{
				if (complete[j])
					continue;
				x1 = pxyz[v[j].p1].x;
				y1 = pxyz[v[j].p1].y;
				x2 = pxyz[v[j].p2].x;
				y2 = pxyz[v[j].p2].y;
				x3 = pxyz[v[j].p3].x;
				y3 = pxyz[v[j].p3].y;
				inside = CircumCircle( xp, yp,  x1, y1,  x2, y2,  x3, y3,  circle );
				xc = circle.x; yc = circle.y; r = circle.z;
				if (xc + r < xp) complete[j] = true;
				if (inside)
				{
					/* Check that we haven't exceeded the edge list size */
					if (nedge+3 >= emax)
					{
						emax += 100;
						IEDGE[] edges_n = new IEDGE[emax];
						for (int ie=0; ie<emax; ie++) edges_n[ie] = new IEDGE();
						System.arraycopy(edges, 0, edges_n, 0, edges.length);
						edges = edges_n;
					}
					edges[nedge+0].p1 = v[j].p1;
					edges[nedge+0].p2 = v[j].p2;
					edges[nedge+1].p1 = v[j].p2;
					edges[nedge+1].p2 = v[j].p3;
					edges[nedge+2].p1 = v[j].p3;
					edges[nedge+2].p2 = v[j].p1;
					nedge += 3;
					v[j].p1 = v[ntri-1].p1;
					v[j].p2 = v[ntri-1].p2;
					v[j].p3 = v[ntri-1].p3;
					complete[j] = complete[ntri-1];
					ntri--;
					j--;
				}
			}

			/*
				Tag multiple edges
				Note: if all triangles are specified anticlockwise then all
				interior edges are opposite pointing in direction.
			*/
			for (int j=0;j<nedge-1;j++)
			{
				//if ( !(edges[j].p1 < 0 && edges[j].p2 < 0) )
					for (int k=j+1;k<nedge;k++)
					{
						if ((edges[j].p1 == edges[k].p2) && (edges[j].p2 == edges[k].p1))
						{
							edges[j].p1 = -1;
							edges[j].p2 = -1;
							edges[k].p1 = -1;
							edges[k].p2 = -1;
						}
						/* Shouldn't need the following, see note above */
						if ((edges[j].p1 == edges[k].p1) && (edges[j].p2 == edges[k].p2))
						{
							edges[j].p1 = -1;
							edges[j].p2 = -1;
							edges[k].p1 = -1;
							edges[k].p2 = -1;
						}
					}
			}

			/*
				Form new triangles for the current point
				Skipping over any tagged edges.
				All edges are arranged in clockwise order.
			*/
			for (int j=0;j<nedge;j++)
			{
				if (edges[j].p1 == -1 || edges[j].p2 == -1)
					continue;
				if (ntri >= trimax) return -1;
				v[ntri].p1 = edges[j].p1;
				v[ntri].p2 = edges[j].p2;
				v[ntri].p3 = i;
				complete[ntri] = false;
				ntri++;
			}
		}


		/*
			Remove triangles with supertriangle vertices
			These are triangles which have a vertex number greater than nv
		*/
		for (int i=0;i<ntri;i++)
		{
			if (v[i].p1 >= nv || v[i].p2 >= nv || v[i].p3 >= nv)
			{
				v[i] = v[ntri-1];
				ntri--;
				i--;
			}
		}

		return ntri;
	}

}
