package edu.jhu.ece.iacl.algorithms.graphics.intersector;

import edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree.KdPoint3;
import edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree.KdTriangle;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;

import javax.vecmath.Matrix3d;
import javax.vecmath.Point2d;
import javax.vecmath.Point3d;
import javax.vecmath.Point3f;
import javax.vecmath.Vector2d;
import javax.vecmath.Vector3f;

/**
 * A mesh triangle object that is intended for barycentric interpolation of data
 * embedded in surface vertices.
 * 
 * @author Blake Lucas
 * 
 */
public class EmbeddedTriangle extends KdTriangle {
	public int id1, id2, id3;
	public int fid;

	/**
	 * Constructor
	 * 
	 * @param fid
	 *            face id
	 * @param id1
	 *            vertex id 1
	 * @param id2
	 *            vertex id 2
	 * @param id3
	 *            vertex id 3
	 * @param mesh
	 *            surface reference
	 */
	public EmbeddedTriangle(int fid, int id1, int id2, int id3,
			EmbeddedSurface mesh) {
		super();
		this.id1 = id1;
		this.id2 = id2;
		this.id3 = id3;
		this.fid = fid;
		pts = new KdPoint3[3];
		pts[0] = new KdPoint3(mesh.getVertex(id1));
		pts[1] = new KdPoint3(mesh.getVertex(id2));
		pts[2] = new KdPoint3(mesh.getVertex(id3));
		this.update();
	}

	/**
	 * Update bounding box
	 */
	public void update() {
		minPoint.x = Math.min(Math.min(pts[0].x, pts[1].x), pts[2].x)
				- (float) EPS;
		minPoint.y = Math.min(Math.min(pts[0].y, pts[1].y), pts[2].y)
				- (float) EPS;
		minPoint.z = Math.min(Math.min(pts[0].z, pts[1].z), pts[2].z)
				- (float) EPS;

		maxPoint.x = Math.max(Math.max(pts[0].x, pts[1].x), pts[2].x)
				+ (float) EPS;
		maxPoint.y = Math.max(Math.max(pts[0].y, pts[1].y), pts[2].y)
				+ (float) EPS;
		maxPoint.z = Math.max(Math.max(pts[0].z, pts[1].z), pts[2].z)
				+ (float) EPS;
	}

	/**
	 * Get interpolated scalar position
	 * 
	 * @param surf
	 *            reference surface
	 * @param b
	 *            barycentric coordinates
	 * @param offset
	 *            index into vertex data
	 * @return interpolated scalar
	 */
	public double getScalar(EmbeddedSurface surf, Point3d b, int offset) {
		double t1 = surf.getVertexDataAtOffset(id1, offset);
		double t2 = surf.getVertexDataAtOffset(id2, offset);
		double t3 = surf.getVertexDataAtOffset(id3, offset);
		return (t1 * b.x + t2 * b.y + t3 * b.z);
	}

	/**
	 * For a point on one surface, find a matching point on another surface with
	 * the same mesh connectivity.
	 * 
	 * @param surf
	 *            reference surface to interpolate location
	 * @param p
	 *            point on surface corresponding to this triangle
	 * @return interpolated location reference surface
	 */
	public Point3f mapToPoint(EmbeddedSurface surf, Point3f p) {
		Point3f b = getEmbeddedPointCoords(surf, getBaryCoords(p));
		return b;
	}

	/**
	 * Interpolate position on reference surface using barycentric coordinates
	 * 
	 * @param surf
	 *            reference surface
	 * @param b
	 *            barycentric coordinates
	 * @return position on reference surface
	 */
	public Point3f getEmbeddedPointCoords(EmbeddedSurface surf, Point3d b) {
		Point3f p1 = surf.getVertex(id1);
		Point3f p2 = surf.getVertex(id2);
		Point3f p3 = surf.getVertex(id3);
		return new Point3f((float) (p1.x * b.x + p2.x * b.y + p3.x * b.z),
				(float) (p1.y * b.x + p2.y * b.y + p3.y * b.z), (float) (p1.z
						* b.x + p2.z * b.y + p3.z * b.z));
	}
	/**
	 * Map point to vertex that is closest
	 * @param p point near surface
	 * @return vertex id
	 */
	public int mapToId(Point3f p) {
		double d1 = p.distance(pts[0]);
		double d2 = p.distance(pts[1]);
		double d3 = p.distance(pts[2]);
		if (d1 < d2) {
			if (d1 < d3) {
				return id1;
			} else {
				return id3;
			}
		} else {
			if (d2 < d3) {
				return id2;
			} else {
				return id3;
			}
		}

	}
	/**
	 * For a point on one surface, interpolate a scalar value on another surface with
	 * the same mesh connectivity.
	 * 
	 * @param surf reference surface to interpolate scalar
	 * @param p point on surface associated with this triangle
	 * @param offset offset into vertex data
	 * @return interpolated scalar
	 */
	public double mapToScalar(EmbeddedSurface surf, Point3f p, int offset) {
		return getScalar(surf, getBaryCoords(p), offset);
	}

}
