package edu.jhu.ece.iacl.algorithms.graphics.locator.balltree;

import java.util.LinkedList;
import java.util.List;
import java.util.Vector;

import javax.vecmath.Point3d;
import javax.vecmath.Point3f;
import javax.vecmath.Vector3d;
import javax.vecmath.Vector3f;

import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.algorithms.graphics.utilities.quickhull.QPoint3;
import edu.jhu.ece.iacl.algorithms.graphics.utilities.quickhull.QuickHull3D;

/**
 * A bounding ball used to store other balls. This is used for internal nodes.
 * 
 * @author Blake Lucas
 * 
 */
public class BBall extends Ball {
	protected double radius;
	protected LinkedList<Ball> children;
	protected int depth;
	/**
	 * Constructor
	 */
	public BBall() {
		this(0);
	}
	/**
	 * Constructor
	 * @param depth tree depth for ball
	 */
	public BBall(int depth) {
		super();
		radius = Double.MAX_VALUE;
		this.depth = depth;
		children = new LinkedList<Ball>();
	}
	public int getDepth() {
		return depth;
	}
	public void setDepth(int d) {
		depth = d;
	}

	public double getRadius() {
		return radius;
	}

	public List<Ball> getChildren() {
		return children;
	}
	
	public void updateDescendants() {
		for (Ball b : children) {
			b.parent = this;
			b.updateDescendants();
		}
		update();
	}

	public void updateAncestors() {
		update();
		if (parent != null) {
			parent.updateAncestors();
		}
	}

	protected void update() {
		int sz = children.size();
		if (sz <= 2) {
			if (sz == 0) {
				this.radius = 0;
			} else if (sz == 1) {
				Ball child = children.getFirst();
				this.radius = child.getRadius() + EPS;
				this.set(child);
			} else {
				// This is an internal node, compute left & right child exactly
				Ball leftChild = children.getFirst();
				Ball rightChild = children.getLast();
				if (leftChild.contains(rightChild)) {
					this.x = leftChild.x;
					this.y = leftChild.y;
					this.z = leftChild.z;
					radius = leftChild.getRadius() + EPS;
				} else if (rightChild.contains(leftChild)) {
					this.x = rightChild.x;
					this.y = rightChild.y;
					this.z = rightChild.z;
					radius = rightChild.getRadius() + EPS;
				} else {
					this.x = this.y = this.z = 0;
					this.sub(rightChild, leftChild);
					double len = GeometricUtilities.length(this);
					this
							.scale(((rightChild.getRadius() - leftChild
									.getRadius()) / len));
					this.add(leftChild);
					this.add(rightChild);
					this.scale(0.5);
					radius = 0.5
							* (leftChild.getRadius() + rightChild.getRadius() + len)
							+ EPS;
				}
			}
		} else {
			if (children.size() < 4) {
				this.x = this.y = this.z = 0;
				for (Ball b : children) {
					this.add(b);
				}
				this.scale(1 / (double) children.size());
			} else {
				// This is a leaf node, compute minimum bounding volume using a
				// convex hull
				try {
					QuickHull3D qhull = new QuickHull3D();
					qhull.build(children);
					this.set(qhull.getCenterOfMass());
				} catch (IllegalArgumentException e) {
					this.x = this.y = this.z = 0;
					System.err
							.println("POINTS ARE COPLANAR " + children.size());
					for (Ball b : children) {
						this.add(b);
						System.out.println(b);
					}
					this.scale(1 / (double) children.size());
				}
			}
			radius = 0;
			for (Point3d p : children) {
				radius = Math.max(radius, p.distance(this));
			}
			radius += EPS;
		}
	}

	public boolean isLeaf() {
		return (children.size() == 0);
	}

}
