package edu.jhu.ece.iacl.algorithms.graphics.locator.sphere;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.Vector;

import javax.vecmath.Point2d;
import javax.vecmath.Point2f;
import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree.BBox;
import edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree.KdPoint3;
import edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree.KdTree;
import edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree.KdTriangle;
import edu.jhu.ece.iacl.algorithms.thickness.grid.EmbeddedNode;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;

import gov.nih.mipav.model.structures.jama.*;

/**
 * Interpolate spherical coordinates by intersecting a ray against a spherical
 * parameterization of a surface.
 * 
 * @author Blake Lucas
 * 
 */
public class SphericalMapLocator extends AbstractCalculation {
	private EmbeddedSurface mesh;

	private ArrayList<KdTriangle> triangles;
	protected int lastFaceId = -1;
	protected int lastVertexId = -1;
	protected SphericalTriangle lastTriangle = null;
	private KdTree tree;
	private int offset;
	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.6 $");
	}
	/**
	 * Constructor
	 * @param parent parent calculation
	 */
	public SphericalMapLocator(AbstractCalculation parent) {
		super(parent);
		this.mesh = null;
	}

	public SphericalMapLocator() {
		super();
		this.mesh = null;
	}
	/**
	 * Constructor
	 * @param parent parent calculation
	 * @param mesh surface
	 * @param offset 
	 */
	public SphericalMapLocator(AbstractCalculation parent,
			EmbeddedSurface mesh, int offset) {
		super(parent);
		init(mesh, offset);
	}
	/**
	 * Initialize spherical locator
	 * @param mesh surface
	 * @param offset offset into surface to obtain spherical coordinates
	 */
	public void init(EmbeddedSurface mesh, int offset) {
		this.mesh = mesh;
		this.offset = offset;
		setLabel("Spherical Locator");
		buildTriangles(offset);
		tree = new KdTree(this, triangles, 18);
		markCompleted();
	}

	private static final double DISTANCE_TOLERANCE = 0;
	/**
	 * Get last intersection triangle
	 * @return intersection triangle
	 */
	public SphericalTriangle getLastIntersectionTriangle() {
		return lastTriangle;
	}

	public int getLastFaceId() {
		return lastFaceId;
	}

	public int getLastVertexId() {
		return lastVertexId;
	}
	/**
	 * Locate point on sphere. 
	 * 
	 * @param p2 point to locate
	 * @return
	 */
	public Point3f locatePoint(Point3f p2) {
		if (mesh == null)
			throw new RuntimeException(
					"Spherical Locator has not been initialized");

		KdPoint3 v = new KdPoint3(2 * p2.x, 2 * p2.y, 2 * p2.z);
		Vector3f dr = new Vector3f(p2.x, p2.y, p2.z);
		KdPoint3 org = new KdPoint3(0, 0, 0);
		LinkedList<BBox> boxes = new LinkedList<BBox>();
		boxes.add(tree.getRoot());
		BBox box = null;
		SphericalTriangle tri;
		Point3f pr, ret = null;
		double mind = 1E30, d;
		int hits = 0;
		lastVertexId = -1;
		while (!boxes.isEmpty()) {
			hits++;
			box = boxes.remove();
			if (box.intersects(org, dr)) {
				if (box instanceof SphericalTriangle) {
					tri = (SphericalTriangle) box;
					if (tri.pts[0].distance(p2) <= DISTANCE_TOLERANCE) {
						lastFaceId = tri.fid;
						lastTriangle = tri;
						return mesh.getVertex(tri.id1);
					}
					if (tri.pts[1].distance(p2) <= DISTANCE_TOLERANCE) {
						lastFaceId = tri.fid;
						lastTriangle = tri;
						return mesh.getVertex(tri.id2);
					}
					if (tri.pts[2].distance(p2) <= DISTANCE_TOLERANCE) {
						lastFaceId = tri.fid;
						lastTriangle = tri;
						return mesh.getVertex(tri.id3);
					}
					pr = tri.intersectionPoint(org, v);
					if (pr != null) {
						d = pr.distance(p2);
						if (d < mind) {
							mind = d;
							lastFaceId = tri.fid;
							lastTriangle = tri;
							ret = tri.mapToPoint(mesh, pr);
						}
					}
				}
				boxes.addAll(box.getChildren());
			}
		}
		// System.out.println("NODES EXAMINED "+hits+" "+lastFaceId);
		if (ret != null) {
			return ret;
		}
		System.out.println("Intersection not found, using nearest neighbor rule.");
		// USE NEAREST NEIGHBOR RULE IF INTERSECTION NOT FOUND
		Point3f p1 = new Point3f();
		int index = 0;
		double min = 1E10;
		int count = mesh.getVertexCount();
		for (int i = 0; i < count; i++) {
			p1 = mesh.getPointAtOffset(i, offset);
			if (p1.distance(p2) < min) {
				min = p1.distance(p2);
				index = i;
				if (min == 0)
					break;
			}
		}
		mesh.getCoordinate(index, p1);
		lastVertexId = index;
		for (KdTriangle tr : triangles) {
			int v1 = ((SphericalTriangle) tr).id1;
			int v2 = ((SphericalTriangle) tr).id2;
			int v3 = ((SphericalTriangle) tr).id3;
			if (v1 == index || v2 == index || v3 == index) {
				lastFaceId = ((SphericalTriangle) tr).fid;
				lastTriangle = (SphericalTriangle) tr;
				break;
			}
		}

		System.out.println("CANNOT LOCATE " + min + " " + p2 + " " + p1);
		return p1;
	}

	public double locateScalar(Point3f p, int offset) {
		Point3f pr = locatePoint(p);
		if (lastTriangle != null) {
			return lastTriangle.mapToScalar(mesh, pr, offset);
		} else {
			return Double.MIN_VALUE;
		}
	}

	public int locateId(Point3f p) {
		Point3f pr = locatePoint(p);
		if (lastTriangle != null) {
			return lastTriangle.mapToId(mesh, pr);
		} else {
			return -1;
		}
	}
	/**
	 * Build locator for spherical parameterization embedded in surface
	 * @param offset offset into vertex data to retrieve spherical coordinates.
	 */
	protected void buildTriangles(int offset) {
		int indexCount = mesh.getIndexCount();
		int id1, id2, id3;
		SphericalTriangle t;
		triangles = new ArrayList<KdTriangle>();
		setTotalUnits(indexCount / 3);
		for (int i = 0; i < indexCount; i += 3) {
			id1 = mesh.getCoordinateIndex(i);
			id2 = mesh.getCoordinateIndex(i + 1);
			id3 = mesh.getCoordinateIndex(i + 2);
			t = new SphericalTriangle(i / 3, id1, id2, id3, mesh, offset);
			triangles.add(t);
			incrementCompletedUnits();
		}
		markCompleted();
	}
}
