package edu.jhu.ece.iacl.algorithms.graphics.locator.sphere;

import edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree.KdPoint3;
import edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree.KdTriangle;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;

import javax.vecmath.Matrix3d;
import javax.vecmath.Point2d;
import javax.vecmath.Point3d;
import javax.vecmath.Point3f;
import javax.vecmath.Vector2d;
import javax.vecmath.Vector3f;

/**
 * Triangle class to be used for spherical map interpolation.
 * 
 * @author Blake Lucas
 * 
 */
public class SphericalTriangle extends KdTriangle {
	public int id1, id2, id3;
	public int fid;

	/**
	 * Constructor
	 * @param fid face id
	 * @param id1 vertex id 1
	 * @param id2 vertex id 2
	 * @param id3 vertex id 3
	 * @param mesh surface
	 * @param offset offset into vertex data to retrieve spherical coordinates 
	 */
	public SphericalTriangle(int fid, int id1, int id2, int id3,
			EmbeddedSurface mesh, int offset) {
		super();
		this.id1 = id1;
		this.id2 = id2;
		this.id3 = id3;
		this.fid = fid;
		double[] v1 = mesh.getVertexData(id1);
		double[] v2 = mesh.getVertexData(id2);
		double[] v3 = mesh.getVertexData(id3);
		pts = new KdPoint3[3];
		if (v1.length > offset + 2) {
			pts[0] = new KdPoint3((float) v1[offset], (float) v1[offset + 1],
					(float) v1[offset + 2]);
			pts[1] = new KdPoint3((float) v2[offset], (float) v2[offset + 1],
					(float) v2[offset + 2]);
			pts[2] = new KdPoint3((float) v3[offset], (float) v3[offset + 1],
					(float) v3[offset + 2]);
		} else {
			System.err.println("Could not extract vertex data " + v1.length);
		}
		this.update();
	}
	/**
	 * Constructor.
	 * @param fid face id
	 * @param id1 vertex id 1
	 * @param id2 vertex id 2
	 * @param id3 vertex id 3
	 * @param mesh spherical parameterization
	 */
	public SphericalTriangle(int fid, int id1, int id2, int id3,
			EmbeddedSurface mesh) {
		super();
		this.id1 = id1;
		this.id2 = id2;
		this.id3 = id3;
		this.fid = fid;
		pts = new KdPoint3[3];
		pts[0] = new KdPoint3(mesh.getVertex(id1));
		pts[1] = new KdPoint3(mesh.getVertex(id2));
		pts[2] = new KdPoint3(mesh.getVertex(id3));
		this.update();
	}
	/**
	 * Update bounding box
	 */
	public void update() {
		minPoint.x = Math.min(Math.min(pts[0].x, pts[1].x), pts[2].x)
				- (float) EPS;
		minPoint.y = Math.min(Math.min(pts[0].y, pts[1].y), pts[2].y)
				- (float) EPS;
		minPoint.z = Math.min(Math.min(pts[0].z, pts[1].z), pts[2].z)
				- (float) EPS;

		maxPoint.x = Math.max(Math.max(pts[0].x, pts[1].x), pts[2].x)
				+ (float) EPS;
		maxPoint.y = Math.max(Math.max(pts[0].y, pts[1].y), pts[2].y)
				+ (float) EPS;
		maxPoint.z = Math.max(Math.max(pts[0].z, pts[1].z), pts[2].z)
				+ (float) EPS;
	}
	/**
	 * Get barycentric coordinates
	 * @param surf reference surface
	 * @param p point on reference surface
	 * @return
	 */
	public Point3d getEmbeddedBaryCoords(EmbeddedSurface surf, Point3f p) {
		Point3f p1 = surf.getVertex(id1);
		Point3f p2 = surf.getVertex(id2);
		Point3f p3 = surf.getVertex(id3);

		double a = p1.x - p3.x;
		double b = p2.x - p3.x;
		double c = p3.x - p.x;

		double d = p1.y - p3.y;
		double e = p2.y - p3.y;
		double f = p3.y - p.y;

		double g = p1.z - p3.z;
		double h = p2.z - p3.z;
		double i = p3.z - p.z;

		double l1 = (b * (f + i) - c * (e + h)) / (a * (e + h) - b * (d + g));
		double l2 = (a * (f + i) - c * (d + g)) / (b * (d + g) - a * (e + h));
		if (Double.isNaN(l1) || Double.isInfinite(l1)) {
			l1 = 0;
		}
		if (Double.isNaN(l2) || Double.isInfinite(l2)) {
			l2 = 0;
		}
		if (l1 > 1 || l2 > 1 || l1 + l2 > 1 || l1 < 0 || l2 < 0) {
			// System.err.println("Barycentric Coordinate Invalid R1:"+r1+" R2:"+r2+" R3:"+r3+" P:"+p);
			l1 = Math.max(Math.min(l1, 1), 0);
			l2 = Math.max(Math.min(l2, 1), 0);
			if (l1 + l2 > 1) {
				double diff = 0.5 * (1 - l1 - l2);
				l1 += diff;
				l2 += diff;
			}
		}
		return new Point3d(l1, l2, 1 - l1 - l2);
	}
	/**
	 * Interpolate scalar embedded in surface vertex data
	 * @param surf reference surface
	 * @param b barycentric coordinate
	 * @param offset offset into vertex data
	 * @return scalar
	 */
	public double getScalar(EmbeddedSurface surf, Point3d b, int offset) {
		double t1 = surf.getVertexDataAtOffset(id1, offset);
		double t2 = surf.getVertexDataAtOffset(id2, offset);
		double t3 = surf.getVertexDataAtOffset(id3, offset);
		return (t1 * b.x + t2 * b.y + t3 * b.z);
	}
	/**
	 * Map point on sphere to point on surface
	 * @param surf surface
	 * @param p spherical coordinate
	 * @return interpolated point on surface
	 */
	public Point3f mapToPoint(EmbeddedSurface surf, Point3f p) {
		Point3f b = getEmbeddedPointCoords(surf, getBaryCoords(p));
		return b;
	}
	/**
	 * Get interpolated point on surface
	 * @param surf surface
	 * @param b barycentric coordinate
	 * @return interpolated point
	 */
	public Point3f getEmbeddedPointCoords(EmbeddedSurface surf, Point3d b) {
		Point3f p1 = surf.getVertex(id1);
		Point3f p2 = surf.getVertex(id2);
		Point3f p3 = surf.getVertex(id3);
		return new Point3f((float) (p1.x * b.x + p2.x * b.y + p3.x * b.z),
				(float) (p1.y * b.x + p2.y * b.y + p3.y * b.z), (float) (p1.z
						* b.x + p2.z * b.y + p3.z * b.z));
	}
	/**
	 * Map point on surface to vertex 
	 * @param surf reference surface
	 * @param p point on sphere
	 * @return closest vertex id on reference surface
	 */
	public int mapToId(EmbeddedSurface surf, Point3f p) {
		Point3f p1 = surf.getVertex(id1);
		Point3f p2 = surf.getVertex(id2);
		Point3f p3 = surf.getVertex(id3);
		Point3f b = getEmbeddedPointCoords(surf, getBaryCoords(p));
		double d1 = b.distance(p1);
		double d2 = b.distance(p2);
		double d3 = b.distance(p3);
		if (d1 < d2) {
			if (d1 < d3) {
				return id1;
			} else {
				return id3;
			}
		} else {
			if (d2 < d3) {
				return id2;
			} else {
				return id3;
			}
		}

	}
	/**
	 * Map point on sphere to scalar on reference surface
	 * @param surf reference surface
	 * @param p point on sphere
	 * @param offset offset into vertex data
	 * @return interpolated scalar
	 */
	public double mapToScalar(EmbeddedSurface surf, Point3f p, int offset) {
		return getScalar(surf, getBaryCoords(p), offset);
	}

}
