package edu.jhu.ece.iacl.algorithms.graphics.map;

import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

import no.uib.cipr.matrix.Matrix;
import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.algorithms.graphics.map.HemisphericalMap.WeightVectorFunc;
import edu.jhu.ece.iacl.algorithms.graphics.surf.ProgressiveSurface;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;

/**
 * Weighting function for conformal maps. Conformal weights are only valid for
 * acute triangulations. This weight function ignores negative weights since
 * they can prevent convergence. Conformal maps are unique up to a mobius
 * transformation of the spherical map. Although, mobius transformations applied
 * to discrete conformal maps will likely cause fold-overs of the triangle mesh.
 * 
 * @author Blake Lucas
 * 
 */
public class ConformalWeightFunc implements WeightVectorFunc {
	EmbeddedSurface surf;
	EmbeddedSurface labeledRefSurface;
	int[][] neighborVertexVertexTable;
	int[] labels;

	public ConformalWeightFunc(ProgressiveSurface ps, int[] labels,
			EmbeddedSurface labeledRefSurface) {
		this.surf = ps.surf;
		this.labels = labels;
		this.neighborVertexVertexTable = ps.neighborVertexVertexTable;
		this.labeledRefSurface = labeledRefSurface;
	}

	public double populate(Matrix Ax, Matrix Ay, int[] lookupTable,
			int[] revLookupTable, int id) {
		int origIndex = lookupTable[id];
		if (labels[origIndex] != 0) {
			int len = neighborVertexVertexTable[lookupTable[id]].length;
			Point3f pivot = labeledRefSurface.getVertex(lookupTable[id]);
			double weight;
			double cota, cotb;
			Point3f pnext, pcurr, plast;
			int nbr;
			double total = 0;
			for (int i = 0; i < len; i++) {
				pnext = labeledRefSurface
						.getVertex(neighborVertexVertexTable[origIndex][(i + 1)
								% len]);
				pcurr = labeledRefSurface
						.getVertex(nbr = neighborVertexVertexTable[origIndex][i]);
				plast = labeledRefSurface
						.getVertex(neighborVertexVertexTable[origIndex][(i
								+ len - 1)
								% len]);
				cota = GeometricUtilities.cotAngle(pivot, pcurr, pnext);
				cotb = GeometricUtilities.cotAngle(pivot, pcurr, plast);
				weight = (cota + cotb);
				total += weight;
				nbr = revLookupTable[nbr];
				Ax.set(id, nbr, -weight);
				Ay.set(id, nbr, -weight);
			}
			Ax.set(id, id, total);
			Ay.set(id, id, total);
		} else {
			Ax.set(id, id, 1);
			Ay.set(id, id, 1);
		}
		return 1;
	}

	public double populate(Matrix Ar, int[] lookupTable, int[] revLookupTable,
			int id) {
		int origIndex = lookupTable[id];
		if (labels[origIndex] != 0) {
			int len = neighborVertexVertexTable[lookupTable[id]].length;
			Point3f pivot = labeledRefSurface.getVertex(lookupTable[id]);
			double weight;
			double cota, cotb;
			Point3f pnext, pcurr, plast;
			int nbr;
			double total = 0;
			double curv = 1;
			for (int i = 0; i < len; i++) {
				pnext = labeledRefSurface
						.getVertex(neighborVertexVertexTable[origIndex][(i + 1)
								% len]);
				pcurr = labeledRefSurface
						.getVertex(nbr = neighborVertexVertexTable[origIndex][i]);
				plast = labeledRefSurface
						.getVertex(neighborVertexVertexTable[origIndex][(i
								+ len - 1)
								% len]);
				cota = GeometricUtilities.cotAngle(pivot, pcurr, pnext);
				cotb = GeometricUtilities.cotAngle(pivot, pcurr, plast);
				weight = (cota + cotb);
				total += weight;
				nbr = revLookupTable[nbr];
				Ar.set(id, nbr, -weight * GeometricUtilities.dot(pivot, pcurr));
			}
			Ar.set(id, id, total);
		} else {
			Ar.set(id, id, 1);
		}
		return 1;
	}

	public double populate(Matrix Ax, Matrix Ay, Matrix Az, int[] lookupTable,
			int[] revLookupTable, int id) {
		int origIndex = lookupTable[id];
		int len = neighborVertexVertexTable[lookupTable[id]].length;
		Point3f pivot = labeledRefSurface.getVertex(lookupTable[id]);
		double weight;
		double cota, cotb;
		Point3f pnext, pcurr, plast;
		int nbr;
		double total = 0;
		double curv = 1;
		if (labels[origIndex] != 0) {
			for (int i = 0; i < len; i++) {
				pnext = labeledRefSurface
						.getVertex(neighborVertexVertexTable[origIndex][(i + 1)
								% len]);
				pcurr = labeledRefSurface
						.getVertex(nbr = neighborVertexVertexTable[origIndex][i]);
				plast = labeledRefSurface
						.getVertex(neighborVertexVertexTable[origIndex][(i
								+ len - 1)
								% len]);
				cota = GeometricUtilities.cotAngle(pivot, pcurr, pnext);
				cotb = GeometricUtilities.cotAngle(pivot, pcurr, plast);
				weight = (cota + cotb);
				total += weight;
				nbr = revLookupTable[nbr];
				Ax.add(id, nbr, -weight);
				Ay.add(id, nbr, -weight);
				Az.add(id, nbr, -weight);
			}
			Ax.set(id, id, total);
			Ay.set(id, id, total);
			Az.set(id, id, total);
		} else {
			Ax.set(id, id, 1);
			Ay.set(id, id, 1);
			Az.set(id, id, 1);
		}
		return curv;
	}

	public Vector3f getResidualVector(int id) {
		int[] nbrs = neighborVertexVertexTable[id];
		int len = nbrs.length;
		Point3f pivot = labeledRefSurface.getVertex(id);
		double weight;
		double cota, cotb;
		Point3f pnext, pcurr, plast;
		int nbr;
		double total = 0;
		Vector3f errVec = new Vector3f();
		if (len == 0)
			return errVec;
		for (int i = 0; i < len; i++) {
			pnext = labeledRefSurface.getVertex(nbrs[(i + 1) % len]);
			pcurr = labeledRefSurface.getVertex(nbr = nbrs[i]);
			plast = labeledRefSurface.getVertex(nbrs[(i + len - 1) % len]);
			cota = GeometricUtilities.cotAngle(pivot, pcurr, pnext);
			cotb = GeometricUtilities.cotAngle(pivot, pcurr, plast);
			weight = (cota + cotb);
			Point3f pt = surf.getVertex(nbr);
			pt.scale((float) weight);
			errVec.add(pt);
			total += weight;
		}
		if (total > 0) {
			total = 1 / total;
		}
		errVec.scale((float) total);
		errVec.sub(surf.getVertex(id));
		return errVec;
	}
}
