package edu.jhu.ece.iacl.algorithms.graphics.map;

import java.util.Stack;

import javax.vecmath.Point3f;

import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.algorithms.graphics.surf.ProgressiveSurface;
import edu.jhu.ece.iacl.jist.structures.data.BinaryMinHeap;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.geom.VertexIndexed;

/**
 * This spherical mapping procedure is an incremental improvement over the
 * original spherical mapping procedure that didn't guarantee bijectivity of the
 * resulting map and would crash if the surface contained degenerate triangles.
 * 
 * @author Blake Lucas
 * 
 */
public class RobustSphericalMap extends ProgressiveSurface {
	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.5 $");
	}


	public RobustSphericalMap() {
	}

	protected Stack<MeshTopologyOperation> vertStack = new Stack<MeshTopologyOperation>();
	/**
	 * Decimate surface to specified maximum mean curvature
	 * @param curvThreshold curvature threshold
	 * @return number of decimated point
	 */
	protected int decimate(double curvThreshold) {
		int vertCount = surf.getVertexCount();
		BinaryMinHeap heap = new BinaryMinHeap(vertCount, vertCount, 1, 1);
		int count = 0;
		for (int i = 0; i < vertCount; i++) {

			if (neighborVertexVertexTable[i].length > 0) {
				// Insert un-decimated points into heap
				VertexIndexed vox = new VertexIndexed(heapMetric(i));
				vox.setPosition(i);
				heap.add(vox);
			} else {
				// Count removed points
				count++;
			}
		}
		int v1, v2;
		int step = 0;
		boolean removed = false;
		SphericalEdgeCollapse vs = null;
		while (!heap.isEmpty() && (vertCount - count) > 4) {
			VertexIndexed vox = (VertexIndexed) heap.remove();
			if (-vox.getValue() < curvThreshold)
				break;
			v2 = vox.getRow();
			int[] nbrs = neighborVertexVertexTable[v2];
			if (nbrs.length == 0) {
				// V2 has already been removed
				continue;
			}
			removed = false;
			vs = new SphericalEdgeCollapse();
			removed = vs.apply(v2);
			// Edge successfully removed
			if (removed) {
				vertStack.push(vs);
				for (int nbr : vs.getChangedVerts()) {
					vox = new VertexIndexed(heapMetric(nbr));
					vox.setPosition(nbr);
					heap.change(nbr, 0, 0, vox);
				}
				step++;
				count++;
			} else {
				// Haven't hit this statement yet, but it maybe possible
				System.err
						.println("COULD NOT REMOVE VERTEX BECAUSE OF TOPOLOGY "
								+ v2 + " " + vox.getValue());
			}
		}
		System.out.println("DECIMATION " + 100 * vertStack.size()
				/ (float) vertCount + " %");
		heap.makeEmpty();
		return vertCount - count;
	}
	/**
	 * Generate a robust spherical parameterization for a surface
	 * @param origSurf reference surface
	 * @param origin origin to initialize spherical mapping pole
	 * @param maxCurvatureThreshold maximum curvature threshold
	 * @return surface with embedded spherical map
	 */
	public EmbeddedSurface solve(EmbeddedSurface origSurf, Point3f origin,
			double maxCurvatureThreshold) {
		init(origSurf, false);
		// Decimate surface
		decimate(maxCurvatureThreshold);
		EmbeddedSurface smallSurf = createSurface();
		// Map decimated surface to sphere
		SurfaceToComplex stoc = new SurfaceToComplex(this, smallSurf);
		double[][] coords = stoc.mapToCartesianCoordinates(origin);
		smallSurf.setVertexData(coords);
		// Correct harmonic map of surfaces
		SphericalMapCorrection hcorrect = new SphericalMapCorrection(this);
		EmbeddedSurface smallSphere = hcorrect.solve(smallSurf
				.getEmbeddedSphere(0, false));
		hcorrect = null;
		int vertCount = surf.getVertexCount();
		// Replace current progressive surface with spherical map
		int currentId = 0;
		for (int i = 0; i < vertCount; i++) {
			if (neighborVertexVertexTable[i].length != 0) {
				surf.setVertex(i, smallSphere.getVertex(currentId++));
			} else {
				surf.setVertex(i, new Point3f(0, 0, 0));
			}
		}
		// Tessellate surface to insert missing vertices
		double vertData[][] = new double[vertCount][3];
		mask = new double[vertCount][1];
		tessellate();
		isHarmonic();
		/*
		 * if(!isHarmonic()){ HarmonicMapCorrection hmc=new
		 * HarmonicMapCorrection(this); surf=hmc.solve(surf); }
		 */
		// Assign coordinate data back to sphere
		for (int i = 0; i < vertCount; i++) {
			Point3f v = surf.getVertex(i);
			vertData[i] = new double[] { v.x, v.y, v.z };
		}
		origSurf.setVertexData(vertData);
		surf.setName(origSurf.getName() + "_sphere");
		surf.setVertexData(mask);
		return surf;
	}

	double[][] mask;
	/**
	 * Tessellate surface to restore original parameterization
	 */
	public void tessellate() {
		System.out.println("TESSELLATE " + vertStack.size());
		while (!vertStack.isEmpty()) {
			MeshTopologyOperation vs = vertStack.pop();
			vs.restore();
			mask[vs.getRemovedVertexId()][0] = 1;// maxEdge(vs.v2);
			// mask[vs.v1][0]=1;//maxEdge(vs.v1);
		}
	}
}
