package edu.jhu.ece.iacl.algorithms.graphics.map;

import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;

/**
 * Robust spherical map correction addresses the problem that if a fold-over
 * occurs in such a way that removing the fold will cause a topology change in
 * the surface, then the surface may require an unnecessary amount of decimation
 * in order to remove the fold. Robust spherical map correction will down-sample
 * the sphere if it is likely that such a scenario has occurred. The map is then
 * corrected at a lower resolution and then up-sampled using a mean-value
 * interpolator.
 * 
 * @author Blake Lucas
 * 
 */
public class RobustSphericalMapCorrection extends AbstractCalculation {
	public RobustSphericalMapCorrection(AbstractCalculation parent,
			double lambda, double decimationStep, int smoothStepIters) {
		super(parent);
		this.lambda = lambda;
		this.decimationStep = decimationStep;
		this.smoothStepIters = smoothStepIters;
		setLabel("Robust Spherical Map Correction");
	}
	/**
	 * Constructor
	 * @param lambda smoothing speed for spherical mapping
	 * @param decimationStep decimation amount step size
	 * @param smoothStepIters smoothing step iterations
	 */
	public RobustSphericalMapCorrection(double lambda, double decimationStep,
			int smoothStepIters) {
		super();
		this.lambda = lambda;
		this.decimationStep = decimationStep;
		this.smoothStepIters = smoothStepIters;
		setLabel("Robust Spherical Map Correction");
	}

	protected double hormannWeight = 0;
	protected double lambda;
	protected double decimationStep;
	protected int smoothStepIters;
	protected double maximumSphericalMapDecimation = 0.2;
	protected double downSampleAmount = 0.1;
	/**
	 * Set amount to decimate surface when changing resolutions
	 * @param downSampleAmount decimation amount
	 */
	public void setDownSampleAmount(double downSampleAmount) {
		this.downSampleAmount = downSampleAmount;
	}
	/**
	 * Set maximum spherical map decimation before changing resolutions
	 * @param maximumSphericalMapDecimation spherical map decimation threshold
	 */
	public void setMaximumSphericalMapDecimation(
			double maximumSphericalMapDecimation) {
		this.maximumSphericalMapDecimation = maximumSphericalMapDecimation;
	}
	/**
	 * Set hormann weight to balance area and angle preservation
	 * @param weight hormann weight
	 */
	public void setHormannWeight(double weight) {
		this.hormannWeight = weight;
	}
	/**
	 * Solve for spherical map
	 * @param refSurface surface used as a reference to minimize distortion
	 * @param origSphere original spherical map to correct
	 * @return corrected spherical map
	 */
	public EmbeddedSurface solve(EmbeddedSurface refSurface,
			EmbeddedSurface origSphere) {
		SphericalMapCorrection smc = new SphericalMapCorrection(this, lambda,
				decimationStep, smoothStepIters);
		smc.setMaxDecimation(maximumSphericalMapDecimation);
		smc.setUseHormannMetric(refSurface, hormannWeight);
		EmbeddedSurface correctedSphere = smc.solve(origSphere);
		// Could not correct sphere within permissible decimation amount
		if (correctedSphere == null) {
			DownSampleSphericalMap downSample = new DownSampleSphericalMap();
			// Down sample sphere and reference surface
			EmbeddedSurface[] tmp = downSample.solve(refSurface, origSphere,
					downSampleAmount);
			// Correct sphere at lower resolution
			EmbeddedSurface dwnCorrectedSphere = solve(tmp[0], tmp[1]);
			// Up sample corrected sphere
			MeanValueSphericalMapInterpolator upSample = new MeanValueSphericalMapInterpolator();
			origSphere = upSample.solve(refSurface, dwnCorrectedSphere);
			// Correct sphere at current resolution
			smc = new SphericalMapCorrection(this, lambda, decimationStep,
					smoothStepIters);
			smc.setUseHormannMetric(refSurface, hormannWeight);
			correctedSphere = smc.solve(origSphere);
		}
		return correctedSphere;
	}
}
